#ifndef IFT_IMAGE_H_
#define IFT_IMAGE_H_

#ifdef __cplusplus
extern "C" {
#endif

#include "iftCommon.h"
#include "iftColor.h"
#include "iftFIFO.h"
#include "iftAdjacency.h"

/**
@file
@brief iftImage definition and prototypes.
*/

typedef struct ift_image {
  int    *val;
  ushort *Cb,*Cr;
  int xsize,ysize,zsize;
  float dx,dy,dz;
  int *tby, *tbz;        // speed-up voxel access tables
  int maxval, minval, n; // minimum and maximum values, and number of voxels
} iftImage;

typedef struct {
  int hdrlen;
  int bpp;
  int dimensions;
  int W,H,D,T;
  float dx,dy,dz;
  int be_hint;
  int dt;
} AnalyzeHdr;

typedef struct {
  int        hdrlen;
  char       data_type[10];
  char       db_name[18];
  int        extents;
  short int  error;
  char       regular;
  char       hkey0;
} Ana_Hdr1;

typedef struct {
  short int  dim[8];
  short int  unused[7];
  short int  data_type;
  short int  bpp;
  short int  dim_un0;
  float      pixdim[8];
  float      zeroes[8];
  int        maxval;
  int        minval;
} Ana_Hdr2;

#define iftGetXCoord(s,p) (((p) % (((s)->xsize)*((s)->ysize))) % (s)->xsize)
#define iftGetYCoord(s,p) (((p) % (((s)->xsize)*((s)->ysize))) / (s)->xsize)
#define iftGetZCoord(s,p) ((p) / (((s)->xsize)*((s)->ysize)))
#define iftGetVoxelIndex(s,v) ((v.x)+(s)->tby[(v.y)]+(s)->tbz[(v.z)])
#define iftDiagonalSize(s) (ROUND(sqrtf(s->xsize*s->xsize + s->ysize*s->ysize + s->zsize*s->zsize)))

void      iftVerifyImageDomains(iftImage *img1, iftImage *img2, char *function);
char      iftIsColorImage(iftImage *img);
char      iftIs3DImage(iftImage *img);
int       iftXSize(iftImage *img);
int       iftYSize(iftImage *img);
int       iftZSize(iftImage *img);
void      iftDestroyPyImage(iftImage *img);
iftVoxel  iftGetVoxelCoord(iftImage *img, int p);

iftImage *iftCreateImage(int xsize,int ysize,int zsize);
iftImage *iftCreateColorImage(int xsize,int ysize,int zsize);
void      iftDestroyImage(iftImage **img);
void      iftUpdateMinMax(iftImage *img);
void      iftCopyVoxelSize(iftImage *img1, iftImage *img2);
void      iftCopyCbCr(iftImage *img1, iftImage *img2);
void      iftSetCbCr(iftImage *img, ushort value);
void      iftSetCb(iftImage *img, ushort value);
void      iftSetCr(iftImage *img, ushort value);
char      iftValidVoxel(iftImage *img, iftVoxel v);
int       iftMaximumValue(iftImage *img);
int       iftMaximumCb(iftImage *img);
int       iftMaximumCr(iftImage *img);
int       iftMinimumValue(iftImage *img);
iftImage *iftReadImage(char *filename);
iftImage *iftReadImageByExt(char *filename);
void      iftWriteImage(iftImage *img, char *filename);
iftImage *iftReadImageP5(char *filename);
void      iftWriteImageP5(iftImage *img, char *filename);
iftImage *iftReadImageAsP5(char *filename);
void      iftWriteImageExtFormat(iftImage *img, char *filename);
iftImage *iftReadImageAsP6(char *filename);
iftImage *iftReadImageP6(char *filename);
void      iftWriteImageP6(iftImage *img, char *filename);
iftImage *iftReadImageP2(char *filename);
void      iftWriteImageP2(iftImage *img, char *filename);
iftImage *iftExtractObject(iftImage *label, int obj_code, iftVoxel *pos);
void      iftInsertObject(iftImage *bin, iftImage *label, int obj_code, iftVoxel  pos);
iftImage  *iftCopyImage(iftImage *img);
iftImage  *iftCreateCuboid(int xsize, int ysize, int zsize);
iftImage  *iftCSVtoImage(char *filename);
char       iftAdjacentVoxels(iftImage *img, iftAdjRel *A, iftVoxel u, iftVoxel v);
iftImage  *iftImageGradientMagnitude(iftImage *img, iftAdjRel *A);
iftImage *iftAddFrame(iftImage *img, int sz, int value);
iftImage *iftRemFrame(iftImage *fimg, int sz);
void      iftSetImage(iftImage *img, int value);

iftImage *iftGetXYSlice(iftImage *img, int zcoord);
iftImage *iftGetZXSlice(iftImage *img, int ycoord);
iftImage *iftGetYZSlice(iftImage *img, int xcoord);
void      iftPutXYSlice(iftImage *img, iftImage *slice, int zcoord);
void      iftPutZXSlice(iftImage *img, iftImage *slice, int ycoord);
void      iftPutYZSlice(iftImage *img, iftImage *slice, int xcoord);

iftVoxel  iftGeometricCenterVoxel(iftImage *obj);
iftPoint  iftGeometricCenter(iftImage *obj);
int       iftObjectDiagonal(iftImage *obj);
iftImage *iftCropImage(iftImage *img, iftVoxel uo, iftVoxel uf);
void      iftGetDisplayRange(iftImage *img, int *lower, int *higher);
iftImage *iftImageCb(iftImage *img);
iftImage *iftImageCr(iftImage *img);
iftImage *iftImageRed(iftImage *img);
iftImage *iftImageGreen(iftImage *img);
iftImage *iftImageBlue(iftImage *img);
iftImage *iftImageGray(iftImage *img);
iftImage *iftCreateGaussian(int xsize, int ysize, int zsize, iftVoxel mean, float stdev, int maxval);
iftImage *iftRegionBorders(iftImage *label, int value);
iftImage *iftExtractROI(iftImage *img, iftVoxel uo, iftVoxel uf);
void      iftInsertROI(iftImage *roi, iftImage *img, iftVoxel pos);
float     iftObjectVolume(iftImage *label, int obj_code);
iftImage *iftCreateImageWithGaussianHistogram(int xsize, int ysize, int zsize, float mean, float stdev, int maxval);
iftImage *iftCreateImageWithTwoGaussiansHistogram(int xsize, int ysize, int zsize, float mean1, float stdev1, float mean2, float stdev2, int maxval);
iftImage *iftReadRawSlices(char *basename, int first, int last, int xsize, int ysize, int bits_per_voxel);
void      iftWriteRawSlices(iftImage *img, char *basename);
iftImage *iftExtractGreyObject(iftImage *image);
iftImage *iftExtractGreyObjectPos(iftImage *image, iftVoxel *pos);
iftImage *iftCrop2DImageByBorder(iftImage *img, int border);
int        iftImageValueAtPoint(iftImage *img, iftPoint P);
int        iftImageValueAtPoint2D(iftImage *img, iftPoint P);

  /* Simple voxel sampling methods. See iftMImage.h(c) for more
     complex ones. They return a binary mask with the selected
     voxels. */

  int       iftNumberOfElements(iftImage *mask);
  iftImage *iftSelectImageDomain(int xsize, int ysize, int zsize);
  iftImage *iftSelectRegionOfInterest(int xsize, int ysize, int zsize, iftVoxel uo, iftVoxel uf);

/** Inline functions **/
static inline void iftSetYCbCr(iftImage *img, int p, iftColor YCbCr)
{
	img->val[p] = YCbCr.val[0];
	img->Cb[p] = YCbCr.val[1];
	img->Cr[p] = YCbCr.val[2];
}

// Fast functions for accessing/putting RGB values into images.
// For efficiency, they do not check if @param img is colored.
static inline void iftSetRGB(iftImage *img, int p, uchar R, uchar G, uchar B)
{
	iftColor RGB;
	RGB.val[0] = R;
	RGB.val[1] = G;
	RGB.val[2] = B;

	iftSetYCbCr(img, p, iftRGBtoYCbCr(RGB));
}

// Fast functions for accessing/putting RGB values into images.
// For efficiency, they do not check if @param img is colored.
static inline void iftSetRGB2(iftImage *img, int p, iftColor rgb)
{
	iftSetRGB(img, p, rgb.val[0], rgb.val[1], rgb.val[2]);
}

static inline iftColor iftGetRGB(iftImage *img, int p)
{
	iftColor YCbCr;

	YCbCr.val[0] = img->val[p];
	YCbCr.val[1] = img->Cb[p];
	YCbCr.val[2] = img->Cr[p];

	return iftYCbCrtoRGB(YCbCr);
}

//Converts between color spaces. This function quantizes the LAB cspace between 0 and 255.
iftImage *iftConvertColorSpace(iftImage* image, char origin_cspace, char dest_cspace);
// iftDicomInfo *iftGetDicomInfo(iftNameMetricPair *orderedFiles, int size);
// iftDicomInfo *iftReadDicomInfo(char *file);
// void iftWriteDicomInfo(char *path, char *fileName, iftDicomInfo* dicomInfo);
// void  iftDestroyDicomInfo(iftDicomInfo **dicomInfo);
iftImage *iftReadDicom(iftNameMetricPair *orderedFiles, int size, int xsize, int ysize, float dx, float dy, float dz, int bits_per_voxel);
iftImage *iftLooseImage(iftImage *image, int xsize, int ysize, int zsize);
void iftCenterImages(iftImage *image1, iftImage *image2, iftImage **centeredImage1, iftImage **centeredImage2);
iftNameMetricPair *iftSortDicomDistance(char *source, int *size);

iftImage *iftColorTableToImage(iftColorTable *ct, int xsize, int ysize);
void iftTickColorTableImage(iftImage *img, float minval, float maxval, int nticks, char *filename);

// By TVS
char* iftGetImageType(char *filename);

// Scene to analyze file conversion
int   ana_fio_swab_16(int val);
int   ana_fio_swab_32(int val);
int   ana_fio_abs_read_8(FILE *f, long offset);
int   ana_fio_abs_read_16(FILE *f, long offset, int is_big_endian);
int   ana_fio_abs_read_32(FILE *f, long offset, int is_big_endian);
float ana_fio_abs_read_float32(FILE *f, long offset, int is_big_endian);
int   ana_fio_read_8(FILE *f);
int   ana_fio_read_16(FILE *f, int is_big_endian);
int   ana_fio_read_32(FILE *f, int is_big_endian);
float ana_fio_read_float32(FILE *f, int is_big_endian);
int   ana_fio_abs_write_8(FILE *f, long offset, int val);
int   ana_fio_abs_write_16(FILE *f, long offset, int is_big_endian, int val);
int   ana_fio_abs_write_32(FILE *f, long offset, int is_big_endian, int val);
int   ana_fio_abs_write_float32(FILE *f, long offset, int is_big_endian, float val);
int   ana_fio_abs_write_zeroes(FILE *f, long offset, int n);
int   ana_fio_write_8(FILE *f, int val);
int   ana_fio_write_16(FILE *f, int is_big_endian, int val);
int   ana_fio_write_32(FILE *f, int is_big_endian, int val);
int   ana_fio_write_float32(FILE *f, int is_big_endian, float val);
int   ana_fio_write_zeroes(FILE *f, int n);
void  *SwapEndian(void* Addr, const int Nb);
void  iftScn2Ana(char *imagePath, char *destinyBaseName);
iftImage* iftAna2Scn(char *imageBaseName);

#ifdef __cplusplus
}
#endif


#endif


