import numpy as np
import os
import subprocess
import tempfile

from sklearn.manifold import MDS
from sklearn.manifold import TSNE

from parsing import point_matrix


class Projection:

    def __init__(self, technique, technique_opts, distance, jarpath=None,
                 snepath=None):
        self.jarpath = jarpath
        if not jarpath:
            self.jarpath = 'externals/FeaturesProjection/FeaturesProjection.jar'
        self.snepath = snepath
        if not snepath:
            self.snepath = 'externals/bh_tsne/bhtsne.py'

        self.technique = technique
        self.technique_opts = technique_opts
        self.distance = distance

    def command_line(self, input_filename, output_filename, n_obs):
        cl = ['java', '-jar', self.jarpath, '-f', input_filename]

        cl.append('-dissimilarity')
        if self.distance == 'Cosine':
            cl.append('cosine')
        elif self.distance == 'Euclidean':
            cl.append('euclidean')
        else:
            raise Exception('Invalid distance selected')

        cl.append('-technique')
        if self.technique == 'LSP':
            cl.append('lsp')
            cl.append('-samplesize')
            cl.append(
                str(int(float(self.technique_opts['Control points']) * n_obs)))
            cl.append('-neighbors')
            cl.append(str(self.technique_opts['Neighbors']))
        elif self.technique == 'IDMAP':
            cl.append('idmap')
            cl.append('-fractiondelta')
            cl.append(str(self.technique_opts['Fraction of delta']))
            cl.append('-iterations')
            cl.append(str(self.technique_opts['Iterations']))
        elif self.technique == 'LAMP':
            cl.append('lamp')
            cl.append('-localneighbors')
            cl.append(str(self.technique_opts['Local Neighbors']))
            cl.append('-samplesize')
            cl.append(
                str(int(float(self.technique_opts['Control points']) * n_obs)))
            cl.append('-fractiondelta')
            cl.append(str(self.technique_opts['Fraction of delta']))
            cl.append('-iterations')
            cl.append(str(self.technique_opts['Iterations']))
        elif self.technique == 'PCA':
            cl.append('pca')
        elif self.technique == 'MDS':
            cl.append('mds')
        elif self.technique == 'Sammon':
            cl.append('sammon')
            cl.append('-iterations')
            cl.append(str(self.technique_opts['Iterations']))
        elif self.technique == 't-SNE':
            if self.distance != 'Euclidean':
                raise Exception(
                    't-SNE not available with distances other than Euclidean')

            cl = ['python2', self.snepath, '-i',
                  input_filename, '-o', output_filename]
            cl += ['-p', str(self.technique_opts['Perplexity'])]
            cl += ['-t', str(self.technique_opts['Theta'])]
            cl += ['-r', str(self.technique_opts['Seed'])]

        else:
            raise Exception('Invalid technique selected.')

        return cl


class ObservationProjection(Projection):

    def __init__(self, technique, technique_opts, distance, jarpath=None,
                 snepath=None):
        Projection.__init__(self, technique, technique_opts, distance, 
                            jarpath, snepath)

    def project(self, Xt, y=None, image_names=None, feature_names=None, 
                distance=None):
        if self.technique == 'Orthogonal':
            return Xt[:, 0:2]

        input_file_desc, input_filename = tempfile.mkstemp(suffix='.data')
        output_file_desc, output_filename = tempfile.mkstemp(suffix='.data')

        try:
            if self.technique != 't-SNE':
                input_file = open(input_filename, 'w')
                point_matrix.save(input_file, Xt, y, image_names,
                                  feature_names)

                input_file.close()

                subprocess.check_call(self.command_line(input_filename, output_filename, Xt.shape[0]),
                                      stderr=subprocess.PIPE)

                feature_2d_file = open(input_filename[:-4] + '2d')
                Xp, _, _, _ = point_matrix.load(feature_2d_file)
                feature_2d_file.close()

                os.remove(feature_2d_file.name)
            else:
                np.savetxt(input_filename, Xt, delimiter='\t')
                subprocess.check_call(self.command_line(input_filename, output_filename, Xt.shape[0]),
                                      )
                Xp = np.loadtxt(output_filename, delimiter='\t')
        finally:
            os.close(input_file_desc)
            os.remove(input_filename)

            os.close(output_file_desc)
            os.remove(output_filename)

        return Xp


class DistanceMatrixProjection:

    def __init__(self, technique, technique_opts, snepath=None):
        if not snepath:
            self.snepath = 'externals/bh_tsne/bhtsne.py'
            
        self.technique = technique
        self.technique_opts = technique_opts

    def project(self, dmatrix):
        if self.technique == 'MDS':        
            mds = MDS(n_components=2, dissimilarity='precomputed', random_state=0)
            mds.fit(dmatrix)
            return mds.embedding_
        if self.technique == 't-SNE':
            perplexity = float(self.technique_opts['Perplexity'])
            seed = int(self.technique_opts['Seed'])
            
            tsne = TSNE(n_components=2, metric='precomputed', random_state=seed,
                        perplexity=perplexity, verbose=1)       
            tsne.fit(dmatrix)
            return tsne.embedding_
            
        else:
            raise Exception('Invalid feature projection technique')

lsp_opts = {'name': 'LSP options', 'type': 'group',
                    'children': [{'name': 'Neighbors',
                                   'type': 'int', 'value': '15'},
                                 {'name': 'Control points',
                                  'type': 'float', 'value': '0.1', 'step': 0.1}]}

idmap_opts = {'name': 'IDMAP options', 'type': 'group',
              'children': [{'name': 'Fraction of delta',
                            'type': 'float', 'value': '8.'},
                           {'name': 'Iterations',
                            'type': 'int', 'value': '50'}]}

lamp_opts = {'name': 'LAMP options', 'type': 'group',
             'children': [{'name': 'Local Neighbors',
                           'type': 'float', 'value': '0.5'},
                          {'name': 'Control points',
                           'type': 'float', 'value': '0.1', 'step': 0.1},
                          {'name': 'Fraction of delta',
                           'type': 'float', 'value': '8.'},
                          {'name': 'Iterations',
                           'type': 'int', 'value': '50'}]}

pca_opts = {'name': 'PCA options', 'type': 'group', 'children': []}

mds_opts = {'name': 'MDS options', 'type': 'group', 'children': []}

orthogonal_opts = {'name': 'Orthogonal options', 'type': 'group',
                   'children': []}

sammon_opts = {'name': 'Sammon options', 'type': 'group',
               'children': [{'name': 'Iterations',
                             'type': 'int', 'value': '50'}]}

tsne_opts = {'name': 't-SNE options', 'type': 'group',
             'children': [{'name': 'Perplexity',
                           'type': 'float', 'value': '30'},
                          {'name': 'Theta',
                           'type': 'float', 'value': '0.5'},
                          {'name': 'Seed',
                           'type': 'int', 'value': '0'}]}
                           
none_opts = {'name': 'None options', 'type': 'group',
             'children': []}

available_observation_techniques = {
    'LSP': lsp_opts,
    'IDMAP': idmap_opts,
    'LAMP': lamp_opts,
    'PCA': pca_opts,
    'Orthogonal': orthogonal_opts,
    'Sammon': sammon_opts,
    't-SNE': tsne_opts
}

available_distance_matrix_techniques = {
    'MDS': mds_opts,
    't-SNE': tsne_opts,
    'None': none_opts
}
