import os
from collections import deque

from model.featured_model import FeaturedModel


class FeaturedController:

    def start(self):
        self.show()
        self.app.exec_()

    def open_path(self, path):
        self.model = FeaturedModel(path, self.settings)

        self.update_groups(self.model.group_map)
        self.update_features(self.model.hierarchy_root,
                             self.model.feat_hierarchy,
                             self.model.feature_names)

        self.feature_selection_history = deque(maxlen=100)
        self.push_feature_selection()

        self.update_images(os.path.join(self.model.directory, 'orig'),
                           self.model.image_names)
        self.update_image_list_order()

        self.run_projection()
        self.run_feature_projection()
        self.discard_labelling()

        self.update_status(len(self.model.selected_images),
                           len(self.model.selected_features))

    def run_projection(self):
        self.update_plot(self.model.Xp, self.model.Xt)
        self.update_plot_selection(self.model.selected_images)

    def run_feature_projection(self):
        self.update_feature_plot(self.model.fXp)
        self.update_feature_plot_selection(self.model.selected_features)

    def set_selected_points(self, selected_points):
        self.model.selected_images = set(selected_points)

        self.update_plot_selection(self.model.selected_images)
        self.update_point_selection(self.model.selected_images)
        self.update_group_selection()

        self.update_feature_point_coloring()

        self.update_status(len(self.model.selected_images),
                           len(self.model.selected_features))

    def set_selected_features(self, selected_features, push_selection=True):
        if push_selection:
            self.push_feature_selection()

        self.model.selected_features = selected_features
        self.update_feature_plot_selection(selected_features)
        self.update_feature_selection()

        self.run_projection()

        self.update_status(len(self.model.selected_images),
                           len(self.model.selected_features))

    def create_group(self, name):
        self.model.create_group(name)
        self.update_groups(self.model.group_map)

    def select_neighborhood(self, i, destination):
        ns = self.model.nearest_points(destination, i)

        self.model.selected_images = set(ns + [i])

        self.update_plot_selection(self.model.selected_images)
        self.update_point_selection(self.model.selected_images)
        self.update_group_selection()

        self.update_status(len(self.model.selected_images),
                           len(self.model.selected_features))

    def move_point_to(self, i, destination):
        self.model.move_point_to(i, destination)

        self.run_projection()

        self.update_feature_plot_selection(self.model.selected_features)

        self.update_feature_selection()
        self.update_plot_selection(self.model.selected_images)
        self.update_point_selection(self.model.selected_images)
        self.update_group_selection()
        self.update_feature_point_coloring()

        self.update_status(len(self.model.selected_images),
                           len(self.model.selected_features))

        self.highlight_points([i], size=1.5)

    def set_selected_groups(self, selected_groups):
        self.model.set_selected_groups(selected_groups)

        self.update_plot_selection(self.model.selected_images)
        self.update_point_selection(self.model.selected_images)
        self.update_feature_point_coloring()

        self.update_status(len(self.model.selected_images),
                           len(self.model.selected_features))

    def get_selected_image_paths(self):
        return {os.path.join(self.model.directory, 'orig',
                             self.model.image_names[i])
                for i in self.model.selected_images}

    def remove_groups(self, groups):
        self.model.remove_groups(groups)
        self.update_groups(self.model.group_map)

    def set_parameter_state(self, state, values):
        self.settings.set_parameter_state(state, values)

        if self.model:
            self.model.update_settings()
            self.run_projection()
            self.run_feature_projection()

    def hide_selected_images(self):
        self.model.hide_selected_images()
        self.discard_labelling()

        self.update_groups(self.model.group_map)
        self.update_images(
            os.path.join(self.model.directory, 'orig'), self.model.image_names)
        self.update_image_list_order()

        self.run_projection()
        self.run_feature_projection()

        self.update_status(len(self.model.selected_images),
                           len(self.model.selected_features))

    def create_unnamed_group(self):
        group_names = sorted([self.groups_list.item(i).text()
                              for i in range(self.groups_list.count())])
        i = 1
        while 'Group #{0}'.format(i) in group_names:
            i += 1
        group_name = 'Group #{0}'.format(i)
        self.create_group(group_name)

    def push_feature_selection(self):
        self.feature_selection_history.append(self.model.selected_features)

    def pop_feature_selection(self):
        if self.feature_selection_history:
            self.set_selected_features(
                self.feature_selection_history.pop(), False)

    def update_status(self, nsel_images, nsel_feats):
        if nsel_images == 0:
            msg_images = ''
        elif nsel_images == 1:
            msg_images = '{0} image selected. '.format(nsel_images)
        else:
            msg_images = '{0} images selected. '.format(nsel_images)

        if nsel_feats == 0:
            msg_feats = ''
        elif nsel_feats == 1:
            msg_feats = '{0} feature selected. '.format(nsel_feats)
        else:
            msg_feats = '{0} features selected. '.format(nsel_feats)

        self.update_message(msg_images + msg_feats)

    def update_labelling(self, d):
        if self.labelling is None:
            self.labelling = dict(d)
        else:
            self.labelling.update(d)

    def commit_labelling(self, d=None):
        self.model.active_learning_suggestions()

        if d is None:
            if self.labelling is None:
                return

            items = list(self.labelling.items())
            self.discard_labelling()
        else:
            items = list(d.items())

        indices = [it[0] for it in items]
        classes = [it[1] for it in items]

        self.model.opf_al.assign(indices, classes)

        self.run_projection()
        self.highlight_points(indices, size=1.5, symbol='d')

#        ratio_assg = float(self.model.opf_al.n_assignments)/self.model.n_observations
#        self.statusbar.showMessage('{0}% labeled.'.format(100*ratio_assg))

    def discard_labelling(self):
        self.labelling = None
