import os
import numpy as np
from functools import partial
from PyQt4 import QtCore, QtGui
import pyqtgraph as pg
import pylab

from model import featured_settings
from model.featured_settings import FeaturedSettings
from view.featured_handler import FeaturedHandler

from widgets import ui_base
from view.featured_controller import FeaturedController
from model import feature_scoring

class FeaturedView(QtGui.QMainWindow, ui_base.Ui_MainWindow, FeaturedHandler, 
                   FeaturedController):

    def __init__(self, app):
        QtGui.QMainWindow.__init__(self)
        ui_base.Ui_MainWindow.__init__(self)

        self.app = app

        self.settings = FeaturedSettings()
        self.model = None

        self.resize(1024, 768)
        self.setupUi(self)
#        self.showMaximized()

        self.set_options()
        self.set_icons()
        self.set_menu_options()

        self.bind_events()
        
        try:
            self.open_path('/home/paulorauber/projects/data/corel')
        except:
            pass

    def set_options(self):
        self.THUMBNAIL_SIZE = 150

        self.PEN_UNSELECTED = pg.mkPen(None)
        self.PEN_SELECTED = pg.mkPen('#000000', width=2)

        self.DOT_SIZE = 8
        self.FEATURE_DOT_SIZE = 11

        self.plot_widget.hideAxis('left')
        self.plot_widget.hideAxis('bottom')
        self.plot_widget.setAspectLocked(True)
        self.plot_widget.setBackground((255, 255, 255, 255))

        self.features_plot_widget.hideAxis('left')
        self.features_plot_widget.hideAxis('bottom')
        self.features_plot_widget.setAspectLocked(True)
        self.features_plot_widget.setBackground((255, 255, 255, 255))

        self.last_ordering = None
        self.order_policy_combobox.addItems(['Ascending', 'Descending'])
        self.color_by_combobox.addItems(featured_settings.obs_plot_color_by)
        self.feature_plot_color_by_combobox.addItems(
            featured_settings.feat_plot_color_by)

        self.scoring_combobox.setEnabled(False)
        self.scoring_combobox.addItems(
            sorted(feature_scoring.available_methods.keys()))

    def set_icons(self):
        self.view_button.setIcon(QtGui.QIcon.fromTheme('system-search'))
        self.view_button.setIconSize(QtCore.QSize(8, 8))
        self.run_projection_button.setIcon(
            QtGui.QIcon.fromTheme('media-playback-start'))
        self.run_projection_button.setIconSize(QtCore.QSize(8, 8))
        self.add_group_button.setIcon(QtGui.QIcon.fromTheme('list-add'))
        self.add_group_button.setIconSize(QtCore.QSize(8, 8))
        self.remove_group_button.setIcon(QtGui.QIcon.fromTheme('list-remove'))
        self.remove_group_button.setIconSize(QtCore.QSize(8, 8))

        self.button_lensing_features.setIcon(
            QtGui.QIcon.fromTheme('system-search'))
        self.button_lensing_features.setIconSize(QtCore.QSize(8, 8))
        self.button_lensing_observations.setIcon(
            QtGui.QIcon.fromTheme('system-search'))
        self.button_lensing_observations.setIconSize(QtCore.QSize(8, 8))

        self.features_tree.header().hide()

    def set_menu_options(self):
        self.action_open_dataset.setIcon(
            QtGui.QIcon.fromTheme('document-open'))
        self.action_open_dataset.setIconVisibleInMenu(True)

        self.action_open_features_file.setShortcut('Ctrl+O')
        self.action_open_features_file.setIcon(
            QtGui.QIcon.fromTheme('document-open'))
        self.action_open_features_file.setIconVisibleInMenu(True)

        self.action_save_groups.setIconVisibleInMenu(True)

        self.action_save_feature_state.setShortcut('Ctrl+S')
        self.action_save_feature_state.setIconVisibleInMenu(True)

        self.action_projection_options.setShortcut('Ctrl+P')
        self.action_projection_options.setIcon(
            QtGui.QIcon.fromTheme('preferences-system'))
        self.action_projection_options.setIconVisibleInMenu(True)

        self.action_auto_update_projection.setChecked(True)

        self.action_show_image_view.setChecked(True)
        self.action_show_observation_plot.setChecked(True)
        self.action_show_feature_plot.setChecked(True)

        self.action_last_feature_state.setShortcut('Ctrl+Z')
        self.action_save_feature_state.setShortcut('Ctrl+C')
        self.action_load_feature_state.setShortcut('Ctrl+v')
        self.action_invert_feature_state.setShortcut('Ctrl+I')

        self.action_select_suggestions.setShortcut('Ctrl+F')
        self.action_label_selection_auto.setShortcut('Ctrl+A')
        self.action_reset_active_learning.setShortcut('Ctrl+R')

    def bind_events(self):
        self.ignore_events = False
        self.last_hovered_obs = None
        self.last_hovered_feat = None

        self.plot_widget_vb = self.plot_widget.getViewBox()
        self.plot_widget_vb.sigPointsSelected.\
            connect(self.handle_points_selected)
        self.plot_widget_vb.sigPointDragged.\
            connect(self.handle_point_dragged)
        self.plot_widget_vb.sigPointReleased.\
            connect(self.handle_point_released)
        self.plot_widget.scene().sigMouseMoved.connect(
            self.plot_widget_vb.mouseMoveEvent)
        self.plot_widget_vb.sigPointHovered.\
            connect(self.handle_point_hovered)

        self.features_plot_widget_vb = self.features_plot_widget.getViewBox()
        self.features_plot_widget_vb.sigPointsSelected.\
            connect(self.handle_feature_points_selected)
        self.features_plot_widget.scene().sigMouseMoved.connect(
            self.features_plot_widget_vb.mouseMoveEvent)
        self.features_plot_widget_vb.sigPointHovered.\
            connect(self.handle_feature_point_hovered)

        self.action_open_dataset.triggered.connect(self.handle_open_folder)
        self.action_open_features_file.triggered.connect(self.handle_open_file)
        self.action_save_groups.triggered.connect(self.handle_save_groups)
        self.action_save_features.triggered.connect(self.handle_save_features)
        self.action_load_feature_selection.triggered.connect(
            self.handle_load_feature_selection)

        self.action_last_feature_state.triggered.connect(
            self.handle_last_feature_state_clicked)

        self.action_projection_options.triggered.connect(
            self.handle_projection_options)
        self.action_auto_update_projection.triggered.connect(
            self.handle_auto_update_projection)
        self.action_show_neighbors.triggered.connect(
            self.handle_run_projection_clicked)
        self.action_show_image_view.triggered.connect(
            self.handle_show_image_view_clicked)
        self.action_show_feature_plot.triggered.connect(
            self.handle_show_feature_plot_clicked)
        self.action_show_observation_plot.triggered.connect(
            self.handle_show_observation_plot_clicked)

        self.action_collapse_features.triggered.connect(
            self.handle_collapse_features)
        self.action_expand_features.triggered.connect(
            self.handle_expand_features)

        self.action_hide_selected.triggered.connect(
            self.handle_hide_selected)

        self.action_distance_histogram.triggered.connect(
            self.handle_distance_histogram)
        self.action_similarity_matrix.triggered.connect(
            self.handle_similarity_matrix)

        self.action_chi_squared_score.triggered.\
            connect(self.handle_chi_squared_stats)
        self.action_anova_score.triggered.\
            connect(self.handle_anova_stats)
        self.action_relief_score.triggered.\
            connect(self.handle_relief_stats)
        self.action_recursive_elimination_score.triggered.\
            connect(self.handle_recursive_elimination_stats)
        self.action_randomized_decision_trees_score.triggered.\
            connect(self.handle_decision_trees_stats)
        self.action_randomized_logistic_score.triggered.\
            connect(self.handle_randomized_logistic_score_stats)
        self.action_relative_variance.triggered.\
            connect(self.handle_variance_based_coherence)
        self.action_silhouette_coefficient.triggered.\
            connect(self.handle_silhouette_coefficient)

        self.action_correlation_x.triggered.\
            connect(partial(self.handle_correlation_axis, 'x'))
        self.action_correlation_y.triggered.\
            connect(partial(self.handle_correlation_axis, 'y'))

        self.run_projection_button.clicked.connect(
            self.handle_run_projection_clicked)
        self.view_button.clicked.connect(self.handle_view_clicked)

        self.add_group_button.clicked.connect(self.handle_add_group_clicked)
        self.remove_group_button.clicked.connect(
            self.handle_remove_group_clicked)

        self.color_by_combobox.currentIndexChanged.connect(
            self.handle_color_by_changed)
        self.feature_plot_color_by_combobox.currentIndexChanged.connect(
            self.handle_feature_color_by_changed)
        self.feature_combobox.currentIndexChanged.connect(
            self.handle_color_by_changed)
        self.scoring_combobox.currentIndexChanged.connect(
            self.handle_feature_color_by_changed)

        self.order_by_combobox.currentIndexChanged.connect(
            self.handle_order_by_changed)
        self.order_policy_combobox.currentIndexChanged.connect(
            self.handle_order_by_changed)

        self.action_save_feature_state.triggered.\
            connect(self.handle_save_feature_state_clicked)
        self.action_load_feature_state.triggered.\
            connect(self.handle_load_feature_state_clicked)
        self.action_invert_feature_state.triggered.\
            connect(self.handle_invert_feature_state_clicked)

        self.groups_list.selectionModel().selectionChanged.\
            connect(self.handle_group_selected)
        self.features_tree.itemClicked.connect(self.handle_feature_clicked)
        self.images_list.selectionModel().\
            selectionChanged.connect(self.handle_image_selected)

        self.button_lensing_observations.clicked.connect(
            self.handle_button_lensing_observations_clicked)
        self.button_lensing_features.clicked.connect(
            self.handle_button_lensing_features_clicked)

        self.action_select_suggestions.triggered.\
            connect(self.handle_select_suggestions_clicked)
        self.action_label_selection_auto.triggered.\
            connect(self.handle_auto_label_selection)
        self.action_label_selection.triggered.\
            connect(self.handle_label_suggestions_clicked)
        self.action_discard_labelling.triggered.\
            connect(self.handle_discard_labelling_clicked)
        self.action_commit_labelling.triggered.\
            connect(self.handle_commit_labelling_clicked)
        self.action_reset_active_learning.triggered.\
            connect(self.handle_reset_active_learning_clicked)
            
        self.action_compute_accuracy_loo.triggered.\
            connect(self.handle_compute_selection_accuracy_loo)
            
        self.action_neighborhood_hit.triggered.\
            connect(self.handle_compute_neighborhood_hit)

    def update_groups(self, group_map):
        self.ignore_events = True
        self.groups_list.clear()

        group_names = sorted(group_map.keys())
        for name in group_names:
            group_item = QtGui.QListWidgetItem()
            group_item.setText(name)
            self.groups_list.addItem(group_item)

        self.groups_list.setSelectionMode(
            QtGui.QAbstractItemView.ExtendedSelection)

        self.ignore_events = False

    def define_feature_hierarchy(self, hierarchy_root, feat_hierarchy,
                                 feature_names, node=None, item_map=None):
        if node == None:
            node = hierarchy_root
            item_map = {}

        if node == hierarchy_root:
            feature_item = QtGui.QTreeWidgetItem()
            feature_item.setText(0, 'all')
            feature_item.setData(0, QtCore.Qt.UserRole, -1)
            self.features_tree.addTopLevelItem(feature_item)
            feature_item.setExpanded(True)

            item_map[node] = feature_item

        for c in feat_hierarchy[node]:
            feature_item = QtGui.QTreeWidgetItem()
            feature_item.setText(0, c)

            item_map[c] = feature_item
            item_map[node].addChild(feature_item)

            if c in feat_hierarchy:
                feature_item.setData(0, QtCore.Qt.UserRole, -1)
                self.define_feature_hierarchy(hierarchy_root, feat_hierarchy,
                                              feature_names, c, item_map)
            else:
                feature_item.setData(0, QtCore.Qt.UserRole,
                                     feature_names.index(c))

        if node == hierarchy_root:
            self.hierarchy_root = node
            return item_map[node]

    def update_features(self, hierarchy_root, feat_hierarchy, feature_names):
        self.ignore_events = True
        self.features_tree.clear()

        self.define_feature_hierarchy(hierarchy_root, feat_hierarchy,
                                      feature_names)
        self.features_tree.sortItems(0, QtCore.Qt.AscendingOrder)

        self.features_tree.setSelectionMode(
            QtGui.QAbstractItemView.ExtendedSelection)

        self.features_tree.setDragDropMode(QtGui.QAbstractItemView.DragOnly)

        self.feature_combobox.setEnabled(False)
        self.feature_combobox.clear()
        self.feature_combobox.addItems(feature_names)

        self.order_by_combobox.clear()
        self.order_by_combobox.addItems(feature_names)

        self.ignore_events = False

    def update_images(self, directory, image_names):
        self.ignore_events = True

        self.images_list.clear()
        self.images_list.setViewMode(QtGui.QListView.IconMode)

        for i, image_name in enumerate(image_names):
            image_item = QtGui.QListWidgetItem()

            if os.path.isfile(os.path.join(directory, image_name)):
                image_icon = QtGui.QIcon()
                pix = QtGui.QPixmap(os.path.join(directory, image_name))
                image_icon.addPixmap(pix, QtGui.QIcon.Normal, QtGui.QIcon.Off)
                image_item.setIcon(image_icon)

            image_item.setText(image_name)

            image_item.setData(QtCore.Qt.UserRole, i)

            self.images_list.addItem(image_item)

        self.images_list.setSelectionMode(
            QtGui.QAbstractItemView.ExtendedSelection)
        self.images_list.setIconSize(
            QtCore.QSize(self.THUMBNAIL_SIZE, self.THUMBNAIL_SIZE))

        self.images_list.setWrapping(True)
        self.images_list.setResizeMode(QtGui.QListView.Adjust)

        self.images_list.setDragDropMode(QtGui.QAbstractItemView.DragOnly)
        self.images_list.setDragEnabled(False)

        self.ignore_events = False

        self.last_ordering = None

    def _get_colored_observation_points(self, Xp, Xt, criteria):
        criteria_map = \
            {
            'Feature':  self._color_points_by_feature,
            'Class': self._color_points_by_class,
            'Classification (k-fold)': self._color_points_by_classification_loo,
            'Classification error (k-fold)': 
                self._color_points_by_classification_error_loo,
            'Classification (AL)':
            self._color_points_by_classification_al,
            'Classification error (AL)':
            self._color_points_by_classification_error_al,
            'Projection Error': self._color_points_by_proj_error,
            'Relation Support': self._color_points_by_relation_support,
            'Prediction (knn)': partial(self._color_points_by_prediction, l_algorithm='kneighborsclassifier'),
            'Prediction (rfc)': partial(self._color_points_by_prediction, l_algorithm='randomforestclassifier'),
            'Prediction (svm)': partial(self._color_points_by_prediction, l_algorithm='svc'),
            'Prediction (ffnn)': partial(self._color_points_by_prediction, l_algorithm='ffnn')
            }

        return criteria_map[criteria](Xp, Xt)

    def update_plot(self, Xp, Xt):
        self.plot_widget.clear()

        self.primary_obs_points = self._get_colored_observation_points(Xp, Xt,
            str(self.color_by_combobox.currentText()))

        scatter_plot_item = pg.ScatterPlotItem(self.primary_obs_points,
                                               size=self.DOT_SIZE,  pxMode=True)

        if self.button_lensing_observations.isChecked():
            self.secondary_obs_points = self._get_colored_observation_points(Xp, 
                Xt, self.settings.parameter('observation_lensing_color'))

        self.plot_widget.addItem(scatter_plot_item)
        self.plot_widget.getViewBox().\
            set_item_names(self.model.get_image_names(set(range(Xp.shape[0]))))

        if self.action_show_neighbors.isChecked():
            pen = pg.mkPen((0, 0, 0, 30))
            for i in range(Xt.shape[0]):
                ns = self.model.get_nearest_neighbors(i)

                for n in ns:
                    pc = pg.PlotCurveItem([Xp[i][0], Xp[n][0]],
                                          [Xp[i][1], Xp[n][1]],
                                          pen=pen)

                    self.plot_widget.addItem(pc)

        scatter_plot_item.sigClicked.connect(self.handle_point_clicked)

    def _color_feature_points_by_scoring(self, fXp):
        try:
            images1, images2, _ = self._split_selection_for_scoring()

            Scorer = feature_scoring.available_methods[
                str(self.scoring_combobox.currentText())]

            feature_scorer = Scorer(
                self.model.Xt, images1, images2, range(self.model.n_features))
            contributions = feature_scorer.score()

            cmin = contributions.min()
            cmax = contributions.max()
            if not np.allclose(cmax, cmin):
                contributions = (contributions - cmin) / (cmax - cmin)
            else:
                return self._color_feature_points_default(fXp)

            brushes = self._brushes_from_attribute(contributions)

            points = [{'pos': fXp[i, :], 'data': i,
                       'brush': brushes[i],
                       'pen': self.PEN_UNSELECTED}
                      for i in range(fXp.shape[0])]

            return points
        except:
            return self._color_feature_points_default(fXp)

    def _color_feature_points_by_neighbor_rel(self, fXp):
        relations = self.model.compute_feature_relation_with_neighbors()

        brushes = self._brushes_from_attribute(relations)

        points = [{'pos': fXp[i, :], 'data': i,
                   'brush': brushes[i],
                   'pen': self.PEN_UNSELECTED}
                  for i in range(fXp.shape[0])]

        return points

    def _color_feature_points_default(self, fXp):
        points = [{'pos': fXp[i, :], 'data': i,
                   'brush': pg.mkBrush('#FF0000'),
                   'pen': self.PEN_UNSELECTED}
                  for i in range(fXp.shape[0])]

        return points

    def _get_colored_feature_points(self, fXp, criteria):
        criteria_map = \
            {
                'None': self._color_feature_points_default,
                'Score': self._color_feature_points_by_scoring,
                'Neighborhood Relation': self._color_feature_points_by_neighbor_rel
            }

        return criteria_map[criteria](fXp)

    def update_feature_plot(self, fXp):
        self.features_plot_widget.clear()

        self.primary_feat_points = self._get_colored_feature_points(fXp,
            str(self.feature_plot_color_by_combobox.currentText()))
        if self.button_lensing_features.isChecked():
            self.secondary_feat_points = self._get_colored_feature_points(fXp,
                self.settings.parameter('feature_lensing_color'))

        scatter_plot_item = pg.ScatterPlotItem(self.primary_feat_points,
                                               size=self.FEATURE_DOT_SIZE, pxMode=True)

        self.features_plot_widget.addItem(scatter_plot_item)
        self.features_plot_widget.getViewBox().\
            set_item_names(
                self.model.get_feature_names(set(range(fXp.shape[0]))))

        scatter_plot_item.sigClicked.connect(
            self.handle_feature_points_clicked)

    def _brushes_from_attribute(self, v):
        fmax, fmin = max(v), min(v)
        r = fmax - fmin
        if np.allclose(r, 0.0):
            r = 1

        def cm(a):
            a = 1 - (a*0.95 + 0.05)
            color = pylab.cm.gist_heat(a)  # @UndefinedVariable
            return int(color[0] * 255), int(color[1] * 255), int(color[2] * 255)

        return [pg.mkBrush(cm((v[i] - fmin) / r)) for i in range(len(v))]

    def _color_points_default(self, Xp, Xt, symbol='o'):
        points = [{'pos': Xp[i, :], 'data': i,
                   'brush': pg.mkBrush('#FF0000'),
                   'pen': self.PEN_UNSELECTED,
                   'symbol': symbol}
                  for i in range(Xp.shape[0])]

        return points

    def _color_points_by_feature(self, Xp, Xt):
        col = self.feature_combobox.currentIndex()
        brushes = self._brushes_from_attribute(Xt[:, col])

        points = [{'pos': Xp[i, :], 'data': i,
                   'brush': brushes[i],
                   'pen': self.PEN_UNSELECTED}
                  for i in range(Xp.shape[0])]

        return points

    def _color_points_by_proj_error(self, Xp, Xt=None):
        v = self.model.compute_mean_aggregated_projection_error()
        brushes = self._brushes_from_attribute(v)
        points = [{'pos': Xp[i, :], 'data': i,
                   'brush': brushes[i],
                   'pen': self.PEN_UNSELECTED}
                  for i in range(Xp.shape[0])]

        return points

    def _color_points_by_class(self, Xp, Xt=None):
        y = self.model.y
        classes, y = np.unique(y, return_inverse=True)

        points = [{'pos': Xp[i, :], 'data': i,
                   'brush': choose_color(y[i], len(classes)),
                   'pen': self.PEN_UNSELECTED}
                  for i in range(Xp.shape[0])]

        return points

    def _color_points_by_leave_one_out(self, Xp, Xt, get_points):
        ypred = self.model.compute_kfold_classification()

        self.update_message('{0:.3f}% accuracy.'.
                            format((100. * sum(ypred == self.model.y)) / len(ypred)))

        return get_points(ypred, Xp, Xt)

    def _color_points_by_active_learning(self, Xp, Xt, get_points):
        try:
            ypred = self.model.active_learning_y_pred()
        except:
            self.update_message('0% labelled.')
            return self._color_points_default(Xp, Xt, symbol='x')

        try:
            perc_assg = 100 * float(self.model.opf_al.n_assignments) / \
                self.model.n_observations

            self.update_message('{0:.3f}% accuracy. {1:.2f}% labelled.'.
                                format((100. * sum(ypred == self.model.y)) / len(ypred), perc_assg))

            return get_points(ypred, Xp, Xt)
        except Exception as e:
            QtGui.QMessageBox.warning(self, "Warning", str(e))
            return self._color_points_default(Xp, Xt, symbol='x')

    def _color_points_by_classification_loo(self, Xp, Xt=None):
        return self._color_points_by_leave_one_out(Xp, Xt,
                                                   self._color_points_by_classification)

    def _color_points_by_classification_error_loo(self, Xp, Xt=None):
        return self._color_points_by_leave_one_out(Xp, Xt,
                                                   self._color_points_by_classification_error)

    def _color_points_by_classification_al(self, Xp, Xt=None):
        return self._color_points_by_active_learning(Xp, Xt,
                                                     self._color_points_by_classification)

    def _color_points_by_classification_error_al(self, Xp, Xt):
        return self._color_points_by_active_learning(Xp, Xt,
                                                     self._color_points_by_classification_error)

    def _color_points_by_classification_error(self, ypred, Xp, Xt):
        y = self.model.y
        ycorrect = (y == ypred)

        brushes = [pg.mkBrush('#FF0000'),
                   pg.mkBrush('#0000FF')]

        points = [{'pos': Xp[i, :], 'data': i,
                   'brush': brushes[ycorrect[i]],
                   'pen': self.PEN_UNSELECTED}
                  for i in range(Xp.shape[0])]

        return points

    def _color_points_by_classification(self, ypred, Xp, Xt):
        y = self.model.y
        ycorrect = (y == ypred)

        unique_y = list(np.unique(y))
        unique_ypred = list(np.unique(ypred))
        if unique_y == unique_ypred:
            _, ypred = np.unique(ypred, return_inverse=True)
        elif set(unique_y).issuperset(unique_ypred):
            ypred = np.array([unique_y.index(ypred[i])
                              for i in range(len(ypred))])
        else:
            raise Exception('Inconsistent class labeling')

        points = [{'pos': Xp[i, :], 'data': i,
                   'brush': choose_color(ypred[i], len(unique_y)) if ycorrect[i] else choose_color_miss(ypred[i], len(unique_y)),
                   'pen': self.PEN_UNSELECTED,
                   'size': self.DOT_SIZE if ycorrect[i] else self.DOT_SIZE * 2,
                   'symbol': 'o' if ycorrect[i] else 't'}
                  for i in range(Xp.shape[0])]
                       
        return points

    def _color_points_by_relation_support(self, Xp, Xt):
        rel_support = self.model.compute_leave_one_out_relation_support()
        brushes = self._brushes_from_attribute(rel_support)

        points = [{'pos': Xp[i, :], 'data': i,
                   'brush': brushes[i],
                   'pen': self.PEN_UNSELECTED}
                  for i in range(Xp.shape[0])]

        return points
        
    def _color_points_by_prediction(self, Xp, Xt, l_algorithm):
        ypred = self.model.load_prediction(l_algorithm)
        
        points = self._color_points_by_classification(ypred, Xp, Xt)
        
        self.update_message('{0:.3f}% accuracy.'.
                            format((100. * sum(ypred == self.model.y)) / len(ypred)))

        return points

    def update_plot_selection(self, selected_images):
        points = self.plot_widget.listDataItems()[0].points()

        pens = []
        for p in points:
            if p.data() in selected_images:
                pens.append(self.PEN_SELECTED)
            else:
                pens.append(self.PEN_UNSELECTED)

        self.plot_widget.listDataItems()[0].setPen(pens)

    def update_plot_coloring(self, scatter, points, selected_points, dot_size):
        brushes = []
        pens = []
        sizes = []
        symbols = []

        plot_points = scatter.points()
        for p in plot_points:
            i = p.data()
            settings = points[i]

            brushes.append(settings['brush'])
            if i in selected_points:
                pens.append(self.PEN_SELECTED)
            else:
                pens.append(self.PEN_UNSELECTED)

            sizes.append(settings.get('size', dot_size))
            symbols.append(settings.get('symbol', 'o'))

        scatter.setBrush(brushes)
        scatter.setPen(pens)
        scatter.setSize(sizes)
        scatter.setSymbol(symbols)

    def update_point_coloring(self, points):
        self.update_plot_coloring(self.plot_widget.listDataItems()[0], points,
                                  self.model.selected_images, self.DOT_SIZE)

    def update_feature_point_coloring(self, points=None):
        if points is None:
            self.primary_feat_points = self._get_colored_feature_points(self.model.fXp,
                str(self.feature_plot_color_by_combobox.currentText()))

            if self.button_lensing_features.isChecked():
                self.secondary_feat_points = self._get_colored_feature_points(self.model.fXp,
                    self.settings.parameter('feature_lensing_color'))

            self.update_plot_coloring(self.features_plot_widget.listDataItems()[0],
                self.primary_feat_points, self.model.selected_features, self.FEATURE_DOT_SIZE)
        else:
            self.update_plot_coloring(self.features_plot_widget.listDataItems()[0],
                points, self.model.selected_features, self.FEATURE_DOT_SIZE)

    def update_feature_plot_selection(self, selected_features):
        points = self.features_plot_widget.listDataItems()[0].points()

        pens = []
        for p in points:
            if p.data() in selected_features:
                pens.append(self.PEN_SELECTED)
            else:
                pens.append(self.PEN_UNSELECTED)

        self.features_plot_widget.listDataItems()[0].setPen(pens)

    def highlight_points(self, l, size=None, brush=None, pen=None, symbol=None):
        points = self.plot_widget.listDataItems()[0].points()

        sizes = []
        brushes = []
        pens = []
        symbols = []

        for p in points:
            if p.data() in l:
                sizes.append(
                    self.DOT_SIZE if size is None else size * self.DOT_SIZE)
                brushes.append(p.brush() if brush is None else brush)
                pens.append(p.pen() if pen is None else pen)
                symbols.append(p.symbol() if symbol is None else symbol)
            else:
                sizes.append(p.size())
                brushes.append(p.brush())
                pens.append(p.pen())
                symbols.append(p.symbol())

        self.plot_widget.listDataItems()[0].setSize(sizes)
        self.plot_widget.listDataItems()[0].setPen(pens)
        self.plot_widget.listDataItems()[0].setBrush(brushes)
        self.plot_widget.listDataItems()[0].setSymbol(symbols)

    def update_message(self, message):
        self.statusbar.showMessage(message)

    def update_point_selection(self, selected_points):
        self.ignore_events = True

        for i in range(self.images_list.count()):
            item = self.images_list.item(i)
            data = item.data(QtCore.Qt.UserRole).toInt()[0]

            is_selected = (data in selected_points)
            if is_selected != item.isSelected():
                item.setSelected(is_selected)

        self.ignore_events = False

    def update_group_selection(self):
        self.ignore_events = True

        for i in range(self.groups_list.count()):
            item = self.groups_list.item(i)

            if item.isSelected():
                item.setSelected(False)

        self.ignore_events = False

    def _select_children_features(self, item, select):
        item.setSelected(select)
        for c in range(item.childCount()):
            self._select_children_features(item.child(c), select)

    def _select_parent_features(self, item):
        if item.childCount() > 0:
            all_selected = True

            for c in range(item.childCount()):
                self._select_parent_features(item.child(c))

                if not item.child(c).isSelected():
                    all_selected = False

            if all_selected:
                item.setSelected(True)

    def update_feature_selection(self):
        root = self.features_tree.findItems(self.hierarchy_root,
                                            QtCore.Qt.MatchRecursive)[0]
        self._select_children_features(root, False)

        names = self.model.get_selected_feature_names()
        items = []
        for name in names:
            items += self.features_tree.findItems(name,
                                                  QtCore.Qt.MatchRecursive)

        for item in items:
            item.setSelected(True)

        self._select_parent_features(root)

    def _get_selected_features_tree(self):
        selected_items = self.features_tree.selectedItems()
        return set([i.data(0, QtCore.Qt.UserRole).toInt()[0]
                    for i in selected_items]) - set([-1])

    def update_image_list_order(self):
        attributes = self.model.get_transformed_feature(
            self.order_by_combobox.currentIndex())
        if self.last_ordering == None:
            self.last_ordering = range(len(attributes))
        attributes = np.array(attributes)[self.last_ordering]

        # argsort(attributes) according to order policy
        is_descending = (self.order_policy_combobox.currentText() ==
                         'Descending')

        sorted_indices = sorted(range(len(attributes)),
                                key=attributes.__getitem__,
                                reverse=is_descending)

        items = []
        for i in sorted_indices:
            items.append(self.images_list.item(i))

        while self.images_list.count():
            self.images_list.takeItem(0)

        for i in items:
            self.images_list.addItem(i)

        new_order = [self.last_ordering[i] for i in sorted_indices]
        self.last_ordering = new_order

    def _split_selection_for_scoring(self):
        gs = [str(g.text()) for g in self.groups_list.selectedItems()]

        if len(gs) == 0:
            images1 = sorted(self.model.selected_images)
            images2 = [
                i for i in range(self.model.n_observations) if i not in images1]
            title = 'Feature scores (selection vs rest)'
        elif len(gs) == 1:
            title = 'Feature scores (Group \"{0}\" vs rest)'.format(gs[0])
            images1 = sorted(self.model.get_images_from_group(gs[0]))
            images2 = [
                i for i in range(self.model.n_observations) if i not in images1]
        elif len(gs) == 2:
            title = 'Feature scores (Group \"{0}\" vs Group \"{1}\")'.format(
                gs[0], gs[1])
            images1 = sorted(self.model.get_images_from_group(gs[0]))
            images2 = sorted(self.model.get_images_from_group(gs[1]))
        else:
            raise Exception('Either zero, one or two groups must be selected')

        return images1, images2, title
     

hsv_colors = [
             (0.56823266219239377, 0.82777777777777772, 0.70588235294117652),
             (0.078146611341632088, 0.94509803921568625, 1.0),
             (0.33333333333333331, 0.72499999999999998, 0.62745098039215685),
             (0.99904761904761907, 0.81775700934579443, 0.83921568627450982),
             (0.75387596899224807, 0.45502645502645506, 0.74117647058823533),
             (0.028205128205128216, 0.4642857142857143, 0.5490196078431373),
             (0.8842592592592593, 0.47577092511013214, 0.8901960784313725),
             (0.0, 0.0, 0.49803921568627452),
             (0.16774193548387095, 0.82010582010582012, 0.74117647058823533),
             (0.51539855072463769, 0.88888888888888884, 0.81176470588235294)
             ]
                
ten_colors = [pg.hsvColor(c[0],c[1],min(1, 1.2*c[2])) for c in hsv_colors]

darker_ten_colors = [pg.hsvColor(c[0], c[1], 0.9*c[2]) for c in hsv_colors]

def choose_color(c, n_colors):
    return ten_colors[c]

def choose_color_miss(c, n_colors):
    return darker_ten_colors[c]