#include "Image.h"
#include <string.h>
#include <sys/wait.h>
#include <math.h>
#include <assert.h>

const char* types[6]={" BYTE_2D", " SHORT_2D", " INT_2D", " FLOAT_2D", " DOUBLE_2D", " UNKNOWN"};

//  template Image<int> operator+< int >(const Image< int > &img1, const Image< int > &img2);
//  template Image<int> operator-< int >(const Image< int > &img1, const Image< int > &img2);
//  template Image<int> operator*< int >(const Image< int > &img1, const Image< int > &img2);

template< class T > Image< T >::Image(char *name)
{
  width = height = 0; 
  type = UNKNOWN;
  pixels = NULL;
  setName(name);
  pid = 0;    
}

template< class T > Image< T >::Image(int w, int h, char *name)
{
    pixels = (T **)calloc(h, sizeof(T *));
    assert(pixels);
    pixels[0] = (T *)calloc(w*h, sizeof(T));
    assert(pixels[0]);

  for(int i=1;i<h;i++)
    pixels[i] = pixels[i-1] + w;

  width = w; height = h;
  type = UNKNOWN;
  pid = 0;    
  setName(name);
}

template< class T > Image< T >::Image(const Image<T>& img)
{
  width  = img.width;
  height = img.height;
  type   = img.type;
  pixels = NULL;
  pid    = img.pid;
  setName(img.img_name);
  if (img.width*img.height != 0) 
    setPixels(img.width, img.height, img.pixels[0]);
}

template< class T > void Image< T >::destroyImage() 
{
  if(pixels != NULL) {
    free(pixels[0]);
    free(pixels);
    pixels = NULL;
    width=height=0;
  }
}

template< class T > Image< T >::~Image()
{ 
  type = UNKNOWN;
  destroyImage();
  setName("");
}

template< class T > void Image< T >::setName(const char *name) 
{ 
  strncpy(img_name, name, 128); 
}

template< class T > void Image< T >::makeImage(int w, int h)
{
  destroyImage(); 

    pixels = (T **)calloc(h, sizeof(T*));
    assert(pixels);
    pixels[0] = (T*)calloc(w*h, sizeof(T));
    assert(pixels[0]);


  for(int i=1;i<h;i++)
    pixels[i] = pixels[i-1] + w;

  width = w; height = h;
}

template< class T > void Image< T >::setPixels(int w, int h, T* data)
{
  makeImage(w, h);

    assert(data);
    memcpy(pixels[0], data, getSize()*sizeof(T));

}

template< class T > T Image<T> :: min()
{
  T res = pixels[0][0];
  for (int i = 0; i < height; i++)
    for (int j = 0; j < width; j++)
      res = (res < pixels[i][j]) ? res : pixels[i][j];
  return res;
}

template< class T > T Image<T> :: max()
{
  T res = pixels[0][0];
  for (int i = 0; i < height; i++)
    for (int j = 0; j < width; j++)
      res = (res > pixels[i][j]) ? res : pixels[i][j];
  return res;
}

template< class T > float Image<T> :: avg()
{
  float res = 0;
  for (int i = 0; i < height; i++)
    for (int j = 0; j < width; j++)
      res += (float)pixels[i][j];
  return (res/getSize());
}

template< class T > float Image<T> :: dev()
{
  float res = 0;
  float mean;
  mean = avg();
  for (int i = 0; i < height; i++)
    for (int j = 0; j < width; j++)
      res += ((float)pixels[i][j] - mean)*((float)pixels[i][j] - mean);
  return (sqrt(res/getSize()));
}

template< class T > Image< T >& Image< T >::operator=(const Image< T >& img)
{
  if(this != &img) {
    width = img.width;
    height = img.height;
    type = img.type;
    setName(img.img_name);
    if (img.width*img.height != 0)
      setPixels(img.width, img.height, img.pixels[0]);
    else
      pixels = NULL;
  }
  return *this;
}

template< class T > int Image< T >::operator==(const Image< T >&img)
{
  if (width==img.width && height==img.height)
    return (!memcmp(pixels[0], img.pixels[0], getSize()*sizeof(T)));
  else
    return (width==img.width && height==img.height); 
}

template< class T > int Image< T >::operator!=(const Image< T >&img)
{
  if (width==img.width && height==img.height)
    return (memcmp(pixels[0], img.pixels[0], getSize()*sizeof(T)));
  else
    return (width!=img.width && height!=img.height); 
}

template< class T > Image< T > &Image< T >::operator+=(const T val)
{
  for (int i=0;i<getSize();i++) 
    pixels[0][i] += val;
  return (*this);
}

template< class T > Image< T > &Image< T >::operator-=(const T val)
{
  for (int i=0;i<getSize();i++) 
    pixels[0][i] -= val;
  return (*this);
}

template< class T > Image< T > &Image< T >::operator*=(const T val)
{
  for (int i=0;i<getSize();i++) 
    pixels[0][i] = pixels[0][i] * val;
  return (*this);
}

template< class T > Image< T > &Image< T >::operator+=(const Image< T >&img)
{
  if(width!=img.width || height!=img.height)
    printf("The two images have different sizes");
  else
  {
    for(int i=0;i<getSize();i++)
      pixels[0][i] += img.pixels[0][i];
    return *this;
  }
}

template< class T > Image< T > &Image< T >::operator-=(const Image< T >&img)
{
  if(width!=img.width || height!=img.height)
  {
    printf("The two images have different sizes");
  }
  else
  {
    for(int i=0;i<getSize();i++)
      pixels[0][i] -= img.pixels[0][i];
    return *this;
  }
}

//  template< class T > Image< T > &Image< T >::operator+(const Image< T >&img)
//  {
//    Image< T > *temp = new Image< T >;
//    temp->setPixels(width,height,pixels[0]);
//    (*temp) += img;
//    return (*temp);
//  }

//  template< class T > Image<T> operator+(const Image<T>& img1, const Image<T>& img2)
//  {
//    Image< T > rez;
//    rez = img1;
//    rez += img2;
//    return rez;
//  }

template< class T > Image< T > &Image< T >::operator*=(const Image< T >&img)
{
  if(width!=img.width || height!=img.height)  
    printf("The two images have different sizes");
  else  {
    for(int i=0;i<getSize();i++)
      pixels[0][i] = pixels[0][i] * img.pixels[0][i];
    return *this;
  }
}

template< class T > void Image< T >::pad(int w, int h, PADDING_TYPE p)
{
  Image< T > temp = *this;
  if (w > 0)
  {
    makeImage(temp.width+2*w, temp.height+2*h);
    for (int i=h; i<temp.height+h; i++) 
      for (int j=w; j<temp.width+w; j++)
	(*this)[i][j] = temp[i-h][j-w]; 
    if (p == ZEROS)
    {
      for (int i = 0; i < h; i++)
	for (int j = w; j < getWidth()-w; j++)
	  (*this)[i][j] = 0;
      for (int i = getHeight()-h; i < getHeight(); i++)
	for (int j = w; j < getWidth()-w; j++)
	  (*this)[i][j] = 0;
      for (int i = 0; i < getHeight(); i++)
	for (int j = 0; j < w; j++)
	  (*this)[i][j] = 0;
      for (int i = 0; i < getHeight(); i++)
	for (int j = getWidth()-w; j < getWidth(); j++)
	  (*this)[i][j] = 0;
    }
    if (p == REFLECTION)
    {
      for (int i = 0; i < h; i++)
	for (int j = w; j < getWidth()-w; j++)
	  (*this)[i][j] = (*this)[2*h-1-i][j];
      for (int i = getHeight()-h; i < getHeight(); i++)
	for (int j = w; j < getWidth()-w; j++)
	  (*this)[i][j] = (*this)[2*getHeight()-2*h-1-i][j];
      for (int i = 0; i < getHeight(); i++)
	for (int j = 0; j < w; j++)
	  (*this)[i][j] = (*this)[i][2*w-1-j];
      for (int i = 0; i < getHeight(); i++)
	for (int j = getWidth()-w; j < getWidth(); j++)
	  (*this)[i][j] = (*this)[i][2*getWidth()-2*w-1-j];
    }
    if (p == WRAP)
    {
      for (int i = 0; i < h; i++)
	for (int j = w; j < getWidth()-w; j++)
	  (*this)[i][j] = (*this)[getHeight()-2*h+i][j];
      for (int i = getHeight()-h; i < getHeight(); i++)
	for (int j = w; j < getWidth()-w; j++)
	  (*this)[i][j] = (*this)[2*h-getHeight()+i][j];
      for (int i = 0; i < getHeight(); i++)
	for (int j = 0; j < w; j++)
	  (*this)[i][j] = (*this)[i][getWidth()-2*w+j];
      for (int i = 0; i < getHeight(); i++)
	for (int j = getWidth()-w; j < getWidth(); j++)
	  (*this)[i][j] = (*this)[i][2*w-getWidth()+j];
    }
  }
}

template< class T > void Image< T >::crop(int x1, int y1, int x2, int y2)
{
  Image< T > temp = *this;
  int offset_x1 = x2 > x1 ? x1 : x2;
  offset_x1 = offset_x1 > 0 ? offset_x1 : 0;
  int offset_y1 = y2 > y1 ? y1 : y2;
  offset_y1 = offset_y1 > 0 ? offset_y1 : 0;

  int offset_x2 = x2 < x1 ? x1 : x2;
  offset_x2 = offset_x2 < temp.width ? offset_x2 : temp.width;
  int offset_y2 = y2 < y1 ? y1 : y2;
  offset_y2 = offset_y2 < temp.height ? offset_y2 : temp.height;
  makeImage((offset_x2 - offset_x1), (offset_y2 - offset_y1));

  for (int i=0; i<(offset_y2 - offset_y1); i++) 
    for (int j=0; j<(offset_x2 - offset_x1); j++)
      pixels[i][j] = temp[i + offset_y1][j + offset_x1];
}

template< class T > void Image< T >::clearImage() 
{ 
  memset(pixels[0],0,sizeof(T)*getSize());
}

template< class T > void Image< T >::clearImage(const T val) 
{ 
  for (int i=0,sz=getSize(); i<sz; ++i)
    pixels[0][i] = val;
}


template< class T > void Image< T >::makeEvenSized(void) 
{
  Image< T > temp = *this;
  if ((width%2 == 1) || (height%2 == 1)) {
    makeImage(temp.width+temp.width%2,temp.height+temp.height%2);
    for(int i=0;i<temp.height;i++)
      for(int j=0;j<temp.width;j++)
	pixels[i][j] = temp[i][j];
    for(int i=0;i<temp.height;i++)
      pixels[i][width-1] = temp[i][temp.width-1];
    for(int j=0;j<temp.width;j++)
      pixels[height-1][j] = temp[temp.height-1][j];
    pixels[height-1][width-1] = pixels[height-1][width-2];
  }
}

template< class T > void Image< T >::closeWindow() 
{
  if(pid) {
    kill(pid, SIGUSR1); //sigusr
    waitpid(pid, NULL, 0);
    pid = 0;
  }
}      	   
  


/**************************************************************************
 *  line_fast                                                             *
 *    draws a line using Bresenham's line-drawing algorithm, which uses   *
 *    no multiplication or division.                                      *
 **************************************************************************/

template<class T>
void Image<T>::drawLine(int x1, int y1, int x2, int y2, T color)
{
  int i,dx,dy,sdx,sdy,dxabs,dyabs,x,y,px,py;

  dx=x2-x1;      /* the horizontal distance of the line */
  dy=y2-y1;      /* the vertical distance of the line */
  dxabs=abs(dx);
  dyabs=abs(dy);
  sdx=((dx<0)?-1:((dx>0)?1:0));
  sdy=((dy<0)?-1:((dy>0)?1:0));
  x=dyabs>>1;
  y=dxabs>>1;
  px=x1;
  py=y1;

  pixels[py][px]=color;

  if (dxabs>=dyabs) /* the line is more horizontal than vertical */
  {
    for(i=0;i<dxabs;i++)
    {
      y+=dyabs;
      if (y>=dxabs)
      {
        y-=dxabs;
        py+=sdy;
      }
      px+=sdx;
      pixels[py][px]=color;
    }
  }
  else /* the line is more vertical than horizontal */
  {
    for(i=0;i<dyabs;i++)
    {
      x+=dxabs;
      if (x>=dyabs)
      {
        x-=dyabs;
        px+=sdx;
      }
      py+=sdy;
      pixels[py][px]=color;
    }
  }
}


template class Image< char >;
template class Image< byte >;
template class Image< short >;
template class Image< int >;
template class Image< float >;
template class Image< double >;
