#ifndef IMAGE_H
#define IMAGE_H

#include <stdlib.h>
#include <string.h>

#ifdef PLATFORM_MACOSX
#include <sys/malloc.h>
#endif

#include <signal.h> //only for kill
//#include "Exceptions.h"
#include <stdio.h>

/** Implements generic operations for all types of images.
*/

typedef unsigned char byte;

template< class T > class Image 
{
 public:
  enum TYPE { BYTE_2D=0, SHORT_2D=1, INT_2D=2, FLOAT_2D=3, DOUBLE_2D=4, UNKNOWN=5};
  enum PADDING_TYPE { ZEROS=0, REFLECTION=1, WRAP=2};

  /*! \example test_image.cpp 
   */

 protected: 
  int width, height; 
  TYPE type; 
  T **pixels;
  char img_name[128]; 
  int pid; //only to close a window 
  
 public:
  /*! Generates an image with the specified name; does not allocate 
    internal buffer for the image data. */
  Image(char * name = "");
  /*! Generates an image of the specified size "w"x"h" and with the specified "name". */
  Image(int w, int h, char *name = "");
  /*! Copy constructor. */
  Image(const Image<T> &);
  /*! Destructor. */
  virtual ~Image();
  /*! Returns the image width. */
  int getWidth() const { return width; }
  /*! Returns the image height. */
  int getHeight() const { return height; }
  /*! Returns the image size - "w"x"h". */
  int getSize() { return width*height; }
  /*! Returns image type; currently defined types are:
    - 0: BYTE_2D;
    - 1: SHORT_2D
    - 2: INT_2D;
    - 3: FLOAT_2D;
    - 4: COMPLEX_2D;
    - 5: UNKNOWN.
  */
  TYPE& getType() { return type; }
  /*! Returns the image name. */
  char *getName() { return img_name; }
  /*! Sets the image name. */
  void setName(const char *name);
  /*! Allocates internal buffer data of size "w"x"h" for the image and
    initializes the pixel values to "0"; frees old internal buffers,
    if already allocated. */
  void makeImage(int w, int h);
  /*! Destroys all data structures associated with the image; frees
    internal buffers. */	
  void destroyImage();
  /*! Returns a pointer to the actual image buffer. */ 
  T *getPixels() { return pixels[0]; }
  /*! Creates an image buffer of size "w"x"h" and initializes
    the data with the buffer pointed by "data". */	
  void setPixels(int w, int h, T *data);
  /*! Returns the minimum gray level of the image. */
  T min();
  /*! Returns the maximum gray level of the image. */
  T max();
  /*! Returns the average value of the pixel values. */
  float avg();
  /*! Returns the standard deviation of the pixel values distribution. */
  float dev();
  
  /*! Overloaded access operator */
  T *operator[](int i) { return pixels[i]; }

  /*! Copy operator. */
  Image< T > &operator=(const Image< T > &img); 

  /*! Tests if two images are the same in both size and pixel values. */
  int operator==(const Image< T > &img);
  /*! Tests if two images are different in both sizes and pixel values. */
  int operator!=(const Image< T > &img);

  /*! \ingroup point_alg */
  /*! Adds a constant value to the value of image pixels. */
  Image< T >  &operator+=(const T val);
  /*! \ingroup point_alg */
  /*! Subtracts a constant value from the value of image pixels. */
  Image< T > &operator-=(const T val);
  /*! \ingroup point_alg */
  /*! Multiplies with a constant value the value of image pixels. */
  Image< T > &operator*=(const T val);

  /*! \ingroup point_alg */
  /*! Adds an image to the current one. */
  Image< T > &operator+=(const Image< T > & img);
  /*! \ingroup point_alg */
  /*! Multiplies an image with the current one. */
  Image< T > &operator*=(const Image< T > & img);
  /*! \ingroup point_alg */
  /*! Subtracts an image from the current one. */
  Image< T > &operator-=(const Image< T > & img);
  
  /*! \ingroup geom_alg */
  /*! Padds the current image with a border of size "w" for along
    the width and and "h" along the height. 
    The type of padding depends on the parameter "p". Currently defined padding types are: 
    - 0: ZEROS;
    - 1: REFLECTION;
    - 2: WRAP.
  */
  void pad(int w, int h, PADDING_TYPE p);
  /*! \ingroup geom_alg */
  /*! Crops part of the current image delimited by top-left corner (x1,y1) to
    bottom-right corner (x2, y2). */
  void crop(int x1, int y1, int x2, int y2);
  /*! Sets all the pixels in the current image to the specified value. */ 
  void clearImage(const T val);
  /*! Sets all the pixels in the current image to a zero-bit pattern. */
  void clearImage();
  /*! \ingroup geom_alg */
  /*! Rectifies the image size (width and height) to even value; if necessary,
    duplicates the last row and/or column */
  void makeEvenSized(void);
  
  /*! Closes the window's image, if the window exists. */
  void closeWindow();
  /*! \ingroup io */
  /*! Reads image from file; the image format is recognized by extension. */
  void readImage(char *name) { 
    printf("Image: warning does nothing.\n");
    /*
    T *d = read_image< T >(name, &width, &height);
    setPixels(width, height, d); 
    */
  }
  /*! \ingroup io */
  /*! Writes image to file; the image format is recognized by extension. */
  void writeImage()
  { 
    printf("Image: warning does nothing.\n");
    /*
    if (width * height != 0) 
      write_image(img_name, width, height, pixels[0]); 
    else
      ERROR("Empty image; nothing to write");
    */
  }
  /*! \ingroup io */
  /*! Shows the image. */
  int showImage()  { 
    printf("Image: warning does nothing.\n");
    /*
    if (width * height != 0) {
      char name[128];
      strncpy(name, img_name, 110);
      strcat(name, types[type]);
      closeWindow();  
      pid = show_image< T > (pixels[0], width, height, name);	  
    }
    else {
      pid = 0;
      ERROR("Empty image; cannot display it!");
    }    
    return(pid);
    */
    return 0;
  }

  void drawLine(int x1, int y1, int x2, int y2, T color);

  // Bellow are some functions intended to be used BUT
  // not used currently for various reasons:

  // Access operator for addressing the data buffer linearly 
  //  T &operator()(int i) { return pixels[0][i];}
  // - not used because returns T& and conflicts with passing arguments
  //   to the parser; reimplemented in the derived classes

  // Access operator 
  //  T &operator()(int i, int j) {return pixels[0][i*width+j];}
  // - not used because returns T& and conflicts with passing arguments
  //   to the parser; reimplemented in derived classes

  // "+","-", "*" operators: as friends functions 
  //  friend Image< T > operator+< T >(const Image< T >& img1, const Image< T >& img2);
  //  friend Image< T > operator*< T >(const Image< T >& img1, const Image< T >& img2);
  //  friend Image< T > operator-< T >(const Image< T >& img1, const Image< T >& img2);
  // - not used because conflict with "Image" container "+" and "-" operators
  //   of the parser AND because they generate memory without a garbage
  //   collection mechanism (see Stroustroup p. 282)

  //  "+", "-", "*" operators as members of the class 
  //  Image< T > &operator+(const Image< T > &img);
  //  Image< T > &operator-(const Image< T > &img);
  //  Image< T > &operator*(const Image< T > &img);
  // - not used because they generate memory without a garbage
  //   collection mechanism (see Stroustroup p. 282); can be adapted by
  //   using a pool of buffers of type "Image" (maybe later...)
};

/*  template< class T > Image<T> operator+(const Image<T>&, const Image<T>&); */
/*  template< class T > Image<T> operator-(const Image<T>&, const Image<T>&); */
/*  template< class T > Image<T> operator*(const Image<T>&, const Image<T>&); */


#endif








