#include "Volume.h"
#include <string.h>
#include <stdio.h>
#include <vector>
#include <math.h>
#include <ext/algorithm>
#include <assert.h>
#include <stdint.h>

static int FindRoot(int r, int *lab)
{
  if ( r != lab[r] )
    lab[r] = FindRoot(lab[r], lab);
  return lab[r];
}

static void Union(int x, int y, int *lab)
{
  int r0 = FindRoot(x, lab), r1 = FindRoot(y, lab);
  if (r0 > r1) lab[r0] = r1;
  else lab[r1] = r0;
}

const int xc27[27]={-1,0,1,-1,0,1,-1,0,1,-1,0,1,-1,0,1,-1,0,1,-1,0,1,-1,0,1,-1,0,1};
const int yc27[27]={-1,-1,-1,0,0,0,1,1,1,-1,-1,-1,0,0,0,1,1,1,-1,-1,-1,0,0,0,1,1,1};
const int zc27[27]={-1,-1,-1,-1,-1,-1,-1,-1,-1,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,1};

static int connComp(int x, int y, int z, Volume<byte> &thin_img) 
{
  int curlab = 0, idx = 0, visit[27];
  byte tval = 0;
  
  std::swap(tval, thin_img[z][y][x]);

  for(int i=-1;i<2;i++)
    for(int j=-1;j<2;j++)
      for(int k=-1;k<2;k++) {
	if(thin_img[z+i][y+j][x+k]>128) {	  
	  visit[idx] = idx;		  
	  for(int m=0;m<13;m++) {
	    int xx=k+xc27[m], yy=j+yc27[m], zz=i+zc27[m];	  	    
	    if(xx>=-1 && yy>=-1 && zz>=-1 && xx<2 && yy<2 && zz<2 && thin_img[z+zz][y+yy][x+xx]>128)
	      Union(idx, idx + m - 13, visit);	    
	  }	  
	}
	else visit[idx] = -1;
	idx++;
      }
	  
  for(int i=0;i<27;i++) {
    idx = visit[i];
    if(idx<0) continue;
    visit[i] = (idx==i) ? (curlab++) :  visit[idx];
  }

  thin_img[z][y][x] = tval;

  return curlab;
}

static int connCompB(int x, int y, int z, Volume<byte> &thin_img) 
{
  int curlab = 0, idx = 0, nn, visit[27];
  byte tval = 255;
  
  std::swap(tval, thin_img[z][y][x]);

  for(int i=-1;i<2;i++)
    for(int j=-1;j<2;j++)
      for(int k=-1;k<2;k++) {
	if(thin_img[z+i][y+j][x+k]==0) {	  
	  visit[idx] = idx;		  

	  nn=i-1;
	  if(nn>=-1 && thin_img[z+nn][y+j][x+k]==0) 
	    Union(idx, idx-9, visit);	    
	  nn=j-1;
	  if(nn>=-1 && thin_img[z+i][y+nn][x+k]==0)
	    Union(idx, idx-3, visit);	    
	  nn=k-1;
	  if(nn>=-1 && thin_img[z+i][y+j][x+nn]==0)
	    Union(idx, idx-1, visit);	    
	}
	else visit[idx] = -1;
	idx++;
      }
	  
  for(int i=0;i<27;i++) {
    idx = visit[i];
    if(idx<0) continue;
    visit[i] = (idx==i) ? (curlab++) :  visit[idx];
  }

  thin_img[z][y][x] = tval;

  return curlab;
}



template< class T >
void Volume<T>::getSphere(int cx, int cy, int cz, int r, float val)
{
  int tlx=cx-r, tly=cy-r, tlz=cz-r;
  int brx=cx+r, bry=cy+r, brz=cz+r;
  
  for(int z=tlz;z<=brz;z++)
    for(int y=tly;y<=bry;y++)
      for(int x=tlx;x<=brx;x++) {
	int xx=x-cx;
	int yy=y-cy;
	int zz=z-cz;
	if((float)(xx*xx+yy*yy+zz*zz)-r*r<=0.0f) 
	  slices[z][y][x]=val;
      }
}

template< class T >
void Volume<T>::showSlices()
{
  for(int i=0;i<getDepth();i++) {
    Image<T> &s=slices[i];
    s.showImage();
    printf("slice: %d.\n", i);
    getchar();
  }
}

template< class T >
void Volume<T>::erode(int r)
{
  int width=getWidth(), height=getHeight(), depth=getDepth();
  register int i, j, k, m;
  register T ming;
  Volume<T> tmp(*this);
  
  //x
  for(i=0;i<depth;i++)
    for(j=0;j<height;j++)
      for(k=0;k<width;k++) {
	ming=tmp[i][j][k];
	for(m=-r;m<=r;m++)
	  if(range(i, j, k+m, depth, height, width) && tmp[i][j][k+m]<ming) {
	    ming=tmp[i][j][k+m];
	  }
	slices[i][j][k]=ming;
      }

  //y
  for(i=0;i<depth;i++)
    for(j=0;j<height;j++)
      for(k=0;k<width;k++) {
	ming=slices[i][j][k];
	for(m=-r;m<=r;m++)
	  if(range(i, j+m, k, depth, height, width) && slices[i][j+m][k]<ming) {
	    ming=slices[i][j+m][k];
	  }
	tmp[i][j][k]=ming;
      }

  //z
  for(i=0;i<depth;i++)
    for(j=0;j<height;j++)
      for(k=0;k<width;k++) {
	ming=tmp[i][j][k];
	for(m=-r;m<=r;m++)
	  if(range(i+m, j, k, depth, height, width) && tmp[i+m][j][k]<ming) {
	    ming=tmp[i+m][j][k];
	  }
	slices[i][j][k]=ming;
      }
}

template< class T >
void Volume<T>::dilate(int r)
{
  int width=getWidth(), height=getHeight(), depth=getDepth();
  register int i, j, k, m;
  register T ming;
  Volume<T> tmp(*this);
  
  //x
  for(i=0;i<depth;i++)
    for(j=0;j<height;j++)
      for(k=0;k<width;k++) {
	ming=tmp[i][j][k];
	for(m=-r;m<=r;m++)
	  if(range(i, j, k+m, depth, height, width) && tmp[i][j][k+m]>ming) {
	    ming=tmp[i][j][k+m];
	  }
	slices[i][j][k]=ming;
      }

  //y
  for(i=0;i<depth;i++)
    for(j=0;j<height;j++)
      for(k=0;k<width;k++) {
	ming=slices[i][j][k];
	for(m=-r;m<=r;m++)
	  if(range(i, j+m, k, depth, height, width) && slices[i][j+m][k]>ming) {
	    ming=slices[i][j+m][k];
	  }
	tmp[i][j][k]=ming;
      }

  //z
  for(i=0;i<depth;i++)
    for(j=0;j<height;j++)
      for(k=0;k<width;k++) {
	ming=tmp[i][j][k];
	for(m=-r;m<=r;m++)
	  if(range(i+m, j, k, depth, height, width) && tmp[i+m][j][k]>ming) {
	    ming=tmp[i+m][j][k];
	  }
	slices[i][j][k]=ming;
      }
}

template< class T >
void Volume<T>::scale(T value)
{
  int size=getWidth()*getHeight(), depth=getDepth();
  register int i, j;

  for(i=0;i<depth;i++)
    for(j=0;j<size;j++)
      slices[i][0][j]*=value;
}

template< class T >
void Volume<T>::getMinMax(T &min, T &max)
{
  int width=getWidth(), height=getHeight(), depth=getDepth();
  register int i, j;
  register T gval;

  min=slices[0][0][0]; max=min;
  
  for(i=0;i<depth;i++)
    for(j=0;j<width*height;j++) {
      gval=slices[i][0][j];
      if(gval<min) min=gval;
      if(gval>max) max=gval;
    }
}

template< class T >
void Volume<T>::grad_order6(Volume<float> &gx, Volume<float> &gy, Volume<float> &gz)
{
  int width=getWidth(), height=getHeight(), depth=getDepth();
  
  gx.makeVolume(width, height, depth);
  gy.makeVolume(width, height, depth);
  gz.makeVolume(width, height, depth);


  static const float w1=(float)(4.0/45.0), w2=(float)(1.0/21.0), w3=(float)(2.0/105.0),
    w4=(float)(5.0/504.0), w5=(float)(1.0/315.0), w6=(float)(1.0/630.0), w8=(float)(1.0/5040.0);

  static const float k1[25] = {0.0f, 0.0f, 2.0f*w8, 0.0f, 0.0f,
			       0.0f, 2.0f*w6, 2.0f*w5, 2.0f*w6, 0.0f, 
			       2.0f*w8, 2.0f*w5, 2.0f*w4, 2.0f*w5, 2.0f*w8,
			       0.0f, 2.0f*w6, 2.0f*w5, 2.0f*w6, 0.0f,
			       0.0f, 0.0f, 2.0f*w8, 0.0f, 0.0f};

  static const float k2[25] = {0.0f, w6, w5, w6, 0.0f,
			       w6, w3, w2, w3, w6, 
			       w5, w2, w1, w2, w5,
			       w6, w3, w2, w3, w6,
			       0.0f, w6, w5, w6, 0.0f};

 for(int z=2;z<depth-2;z++)
    for(int y=2;y<height-2;y++)
      for(int x=2;x<width-2;x++) {
    
	int idx = 0; T gxx = 0, gyy = 0, gzz = 0; 

	for(int k=-2;k<=2;k++)
	  for(int m=-2;m<=2;m++) {

	    gxx += k1[idx]*(slices[z+k][y+m][x+2]-slices[z+k][y+m][x-2]);
	    gyy += k1[idx]*(slices[z+k][y+2][x+m]-slices[z+k][y-2][x+m]);
	    gzz += k1[idx]*(slices[z+2][y+k][x+m]-slices[z-2][y+k][x+m]);
	    
	    gxx += k2[idx]*(slices[z+k][y+m][x+1]-slices[z+k][y+m][x-1]);
	    gyy += k2[idx]*(slices[z+k][y+1][x+m]-slices[z+k][y-1][x+m]);
	    gzz += k2[idx]*(slices[z+1][y+k][x+m]-slices[z-1][y+k][x+m]);
	    
	    idx++;
	  }

	gx[z][y][x] = gxx; gy[z][y][x] = gyy; gz[z][y][x] = gzz;
      }
}





template< class T >
void Volume<T>::toVector(vector<T> &vec)
{
  int width=getWidth(), height=getHeight(), depth=getDepth();
  register int i, j, k, len=0;

  vec.resize(width*height*depth);
  
  for(i=0;i<depth;i++)
    for(j=0;j<height;j++)
      for(k=0;k<width;k++)
	vec[len++]=slices[i][j][k];
}


template< class T >
void Volume<T>::downsample(int down)
{
  int width=getWidth(), height=getHeight(), depth=getDepth();
  Volume tmp(width/down, height/down, depth/down);
  
  for(int i=0;i<depth/down;i++)
    for(int j=0;j<height/down;j++)
      for(int k=0;k<width/down;k++)
	tmp[i][j][k]=slices[down*i][down*j][down*k];
  
  (*this) = tmp;
}

template< class T >
void Volume<T>::readStanfordVolume(char *rootfname, bool endian)
{
  int depth=1;
  static char buffer[100];
  static short data[1024];
  FILE *pfile=NULL;
  
  do{
    //if(depth%2) depth++;
    sprintf(buffer, "%s.%d", rootfname, depth);
    pfile=fopen(buffer, "r");
    if(pfile) {
      vector<short> sdata;
      while(!feof(pfile)) {
	int read = fread(data, sizeof(short), 1024, pfile);
	/*	
	for(int i=0;i<read;i++)
	  if(data[i]>450) data[i]=255;
	  else data[i]=0;
	*/
	if(endian)
	  for(int i=0;i<read;i++) {
	    byte *bdata = (byte*)(&data[i]);
	    byte tmp = bdata[0];
	    bdata[0] = bdata[1];
	    bdata[1] =tmp;
	  }		    
	if(read>0)
	  copy(data, data+read, back_inserter( sdata ) );
      }
        
      int dim=(int)sqrt((float)sdata.size());
      Image<T> slice(dim, dim);
      for(int i=0;i<dim*dim;i++) 
	slice[0][i]=(T)sdata[i];
      slices.push_back(slice);      
      
      /*
      //bunny
      int i, j, dim=(int)sqrt((float)sdata.size())/2;
      Image<T> slice(dim, dim);
      for(i=0;i<dim;i++)
	for(j=0;j<dim;j++)
	  slice[i][j]=(T)sdata[i*(4*dim)+2*j];
      slices.push_back(slice);      
      */
      fclose(pfile);
    }
    depth++;
  }while(pfile);
}

template< class T >
void Volume<T>::readVolume8(char *rootfname)
{
  int depth=1;
  static char buffer[100];
  static byte data[100];
  FILE *pfile=NULL;
  
  do{
    sprintf(buffer, "%s-%d%d%d.raw", rootfname, depth/100, depth/10, depth%10);
    pfile=fopen(buffer, "r");
    if(pfile) {
      vector<byte> sdata;
      while(!feof(pfile)) {
	int read = fread(data, sizeof(byte), 100, pfile);
	if(read>0)
	  copy(data, data+read, back_inserter( sdata ) );
      }
      int dim=(int)sqrt((float)sdata.size());
      Image<T> slice(dim, dim);
      for(int i=0;i<dim*dim;i++) 
	slice[0][i]=(T)sdata[i];
      slices.push_back(slice);      
      fclose(pfile);
    }
    depth++;
  }while(pfile);
}

template< class T >
void Volume<T>::readOneFile8(char *fname, int dx, int dy, int dz)
{
  FILE *pfile=fopen(fname, "r");
  if(!pfile) return;

  makeVolume(dx, dy, dz);

  vector<byte> buf(dx*dy);
  
  for(int i=0;i<dz;i++) {
    int read=fread(&buf[0], sizeof(byte), dx*dy, pfile);
    assert(read==dx*dy);
    Image<T> slice(dx, dy);
    for(int j=0;j<read;j++)
      slice[0][j]=(T)buf[j];
    slices[i]=slice;
  }
  
  fclose(pfile);
}

template< class T >
void Volume<T>::readOneFile16(char *fname, int dx, int dy, int dz)
{
  FILE *pfile=fopen(fname, "r");
  if(!pfile) return;

  makeVolume(dx, dy, dz);

  vector<short> buf(dx*dy);

  for(int i=0;i<dz;i++) {
    int read=fread(&buf[0], sizeof(short), dx*dy, pfile);
    assert(read==dx*dy);
    Image<T> slice(dx, dy);
    for(int j=0;j<read;j++)
      slice[0][j]=(T)buf[j];
    slices[i]=slice;
  }
  
  fclose(pfile);
}

template< class T >
void Volume<T>::UniformSample(vector<point3> &points, float percent)
{
  int n=int((percent*(float)points.size())/100.0f+0.5f);
  
  vector<point3> tmp(n);
  random_sample(points.begin(), points.end(), tmp.begin(), tmp.end());

  printf("UniformSample: sampled %d points.\n", n);

  points=tmp;
}

template< class T >
vector<typename Volume<T>::point3> 
Volume<T>::readPoints(char *fname, int mindim, int pad, float &dx, float &dy, float &dz)
{
  FILE *pfile=fopen(fname, "r");
    
  vector<point3> points;

  if(!pfile) return points;

  while(!feof(pfile)) {
    point3 p;
    //fscanf(pfile, "p %f %f %f\n", &p.coord[0], &p.coord[1], &p.coord[2]);
    fscanf(pfile, "%f %f %f\n", &p.coord[0], &p.coord[1], &p.coord[2]);
    points.push_back(p);
  }

  //UniformSample(points, 30.0f);

  //get bbox
  int size=(int)points.size();
  assert(size>0);
  float minx=points[0].coord[0], maxx=minx;
  float miny=points[0].coord[1], maxy=miny;
  float minz=points[0].coord[2], maxz=minz;
  for(int i=0;i<size;i++) {
    if(points[i].coord[0]<minx) minx=points[i].coord[0];
    if(points[i].coord[0]>maxx) maxx=points[i].coord[0];
    if(points[i].coord[1]<miny) miny=points[i].coord[1];
    if(points[i].coord[1]>maxy) maxy=points[i].coord[1];
    if(points[i].coord[2]<minz) minz=points[i].coord[2];
    if(points[i].coord[2]>maxz) maxz=points[i].coord[2];    
  }
  printf("readPoints: dims = [w=%f, h=%f, d=%f].\n", 
	 maxx-minx, maxy-miny, maxz-minz);
  
  dx=maxx-minx;
  dy=maxy-miny;
  dz=maxz-minz;
  
  float dmin=max(max(maxx-minx, maxy-miny), maxz-minz);
  float scale=mindim/dmin;
  printf("readPoints: creating volume of size %d %d %d (scale=%f).\n", 
	 int((maxx-minx)*scale)+2*pad, 
	 int((maxy-miny)*scale)+2*pad, 
	 int((maxz-minz)*scale)+2*pad, scale);
  
  makeVolume(int((maxx-minx)*scale)+2*pad, 
	     int((maxy-miny)*scale)+2*pad, 
	     int((maxz-minz)*scale)+2*pad);

#if 1
  int x, y, z, k;
  int proj_points=0;
  for(k=0;k<size;k++) {
    x=int(pad+scale*(points[k].coord[0]-minx)+0.5);
    y=int(pad+scale*(points[k].coord[1]-miny)+0.5);
    z=int(pad+scale*(points[k].coord[2]-minz)+0.5);

    if(slices[z][y][x]<50) proj_points++;

    slices[z][y][x]=100;
    //slices[z][y][x]+=50.0;

    points[k].coord[0]=pad+scale*(points[k].coord[0]-minx);
    points[k].coord[1]=pad+scale*(points[k].coord[1]-miny);
    points[k].coord[2]=pad+scale*(points[k].coord[2]-minz);
  }
  printf("readPoints: %d projected points, out of %d input points.\n", proj_points, size);

#else
  
  //perform interpolation
  const int val=100;
  int intx, inty, intz, k;
  float x, y, z, fracx, fracy, fracz;
  int proj_points=0;
  for(k=0;k<size;k++) {
    point3 &p=points[k];
    x=pad+scale*(p.coord[0]-minx);
    y=pad+scale*(p.coord[1]-miny);
    z=pad+scale*(p.coord[2]-minz);
    intx=(int)floor(x);
    inty=(int)floor(y);
    intz=(int)floor(z);
    fracx=x-intx;
    fracy=y-inty;
    fracz=z-intz;
    if(slices[intz][inty][intx]<1e-2f) proj_points++;
    slices[intz][inty][intx]+=T(val*(1.0f-fracx)*(1.0f-fracy)*(1.0f-fracz));
    if(slices[intz][inty][intx+1]<1e-2f) proj_points++;
    slices[intz][inty][intx+1]+=T(val*fracx*(1.0f-fracy)*(1.0f-fracz));
    if(slices[intz][inty+1][intx]<1e-2f) proj_points++;
    slices[intz][inty+1][intx]+=T(val*fracy*(1.0f-fracx)*(1.0f-fracz));
    if(slices[intz+1][inty][intx]<1e-2f) proj_points++;
    slices[intz+1][inty][intx]+=T(val*fracz*(1.0f-fracx)*(1.0f-fracy));
    if(slices[intz+1][inty][intx+1]<1e-2f) proj_points++;
    slices[intz+1][inty][intx+1]+=T(val*fracx*fracz*(1.0f-fracy));
    if(slices[intz+1][inty+1][intx]<1e-2f) proj_points++;
    slices[intz+1][inty+1][intx]+=T(val*fracy*fracz*(1.0f-fracx));
    if(slices[intz][inty+1][intx+1]<1e-2f) proj_points++;
    slices[intz][inty+1][intx+1]+=T(val*fracx*fracy*(1.0f-fracz));
    if(slices[intz+1][inty+1][intx+1]<1e-2f) proj_points++;
    slices[intz+1][inty+1][intx+1]+=T(val*fracx*fracy*fracz);

    p.coord[0]=pad+scale*(p.coord[0]-minx);
    p.coord[1]=pad+scale*(p.coord[1]-miny);
    p.coord[2]=pad+scale*(p.coord[2]-minz);
  }
  printf("readPoints: %d projected points, out of %d input points.\n", proj_points, size);

#endif


  /*
  //add extra plane (manechine, distcap, vase, isis)
  for(k=0;k<getWidth()*getHeight();k++) {
    slices[5][0][k]=100;
    //slices[getDepth()-5][0][k]=100; //footA
  }
  */

  return points;
}

template< class T >
vector<typename Volume<T>::point3> 
Volume<T>::readRawPoints(char *fname, int mindim, int pad, float &dx, float &dy, float &dz)
{
  FILE *pfile=fopen(fname, "r");
    
  vector<point3> points;

  if(!pfile) return points;

  static char antet[256];
  fscanf(pfile, "%s\n", antet);

  while(!feof(pfile)) {
    point3 p;
    //fscanf(pfile, "p %f %f %f\n", &p.coord[0], &p.coord[1], &p.coord[2]);
    fscanf(pfile, "%f %f %f", &p.coord[0], &p.coord[1], &p.coord[2]);
    points.push_back(p);

    fscanf(pfile, "%f %f %f", &p.coord[0], &p.coord[1], &p.coord[2]);
    points.push_back(p);

    fscanf(pfile, "%f %f %f\n", &p.coord[0], &p.coord[1], &p.coord[2]);
    points.push_back(p);
  }

  //UniformSample(points, 30.0f);

  //get bbox
  int size=(int)points.size();
  assert(size>0);
  float minx=points[0].coord[0], maxx=minx;
  float miny=points[0].coord[1], maxy=miny;
  float minz=points[0].coord[2], maxz=minz;
  for(int i=0;i<size;i++) {
    if(points[i].coord[0]<minx) minx=points[i].coord[0];
    if(points[i].coord[0]>maxx) maxx=points[i].coord[0];
    if(points[i].coord[1]<miny) miny=points[i].coord[1];
    if(points[i].coord[1]>maxy) maxy=points[i].coord[1];
    if(points[i].coord[2]<minz) minz=points[i].coord[2];
    if(points[i].coord[2]>maxz) maxz=points[i].coord[2];    
  }
  printf("readPoints: dims = [w=%f, h=%f, d=%f].\n", 
	 maxx-minx, maxy-miny, maxz-minz);

  //printf("%f %f %f %f %f %f\n", minx, maxx, miny, maxy, minz, maxz);
  
  dx=maxx-minx;
  dy=maxy-miny;
  dz=maxz-minz;
  
  float dmin=max(max(maxx-minx, maxy-miny), maxz-minz);
  float scale=mindim/dmin;
  printf("readPoints: creating volume of size %d %d %d (scale=%f).\n", 
	 int((maxx-minx)*scale)+2*pad, 
	 int((maxy-miny)*scale)+2*pad, 
	 int((maxz-minz)*scale)+2*pad, scale);
  
  makeVolume(int((maxx-minx)*scale)+2*pad, 
	     int((maxy-miny)*scale)+2*pad, 
	     int((maxz-minz)*scale)+2*pad);

#if 1
  int x, y, z, k;
  int proj_points=0;
  for(k=0;k<size;k++) {
    x=int(pad+scale*(points[k].coord[0]-minx)+0.5);
    y=int(pad+scale*(points[k].coord[1]-miny)+0.5);
    z=int(pad+scale*(points[k].coord[2]-minz)+0.5);

    if(slices[z][y][x]<50) proj_points++;

    slices[z][y][x]=100;
    //slices[z][y][x]+=50.0;

    points[k].coord[0]=pad+scale*(points[k].coord[0]-minx);
    points[k].coord[1]=pad+scale*(points[k].coord[1]-miny);
    points[k].coord[2]=pad+scale*(points[k].coord[2]-minz);
  }
  printf("readPoints: %d projected points, out of %d input points.\n", proj_points, size);

#else
  
  //perform interpolation
  const int val=100;
  int intx, inty, intz, k;
  float x, y, z, fracx, fracy, fracz;
  int proj_points=0;
  for(k=0;k<size;k++) {
    point3 &p=points[k];
    x=pad+scale*(p.coord[0]-minx);
    y=pad+scale*(p.coord[1]-miny);
    z=pad+scale*(p.coord[2]-minz);
    intx=(int)floor(x);
    inty=(int)floor(y);
    intz=(int)floor(z);
    fracx=x-intx;
    fracy=y-inty;
    fracz=z-intz;
    if(slices[intz][inty][intx]<1e-2f) proj_points++;
    slices[intz][inty][intx]+=T(val*(1.0f-fracx)*(1.0f-fracy)*(1.0f-fracz));
    if(slices[intz][inty][intx+1]<1e-2f) proj_points++;
    slices[intz][inty][intx+1]+=T(val*fracx*(1.0f-fracy)*(1.0f-fracz));
    if(slices[intz][inty+1][intx]<1e-2f) proj_points++;
    slices[intz][inty+1][intx]+=T(val*fracy*(1.0f-fracx)*(1.0f-fracz));
    if(slices[intz+1][inty][intx]<1e-2f) proj_points++;
    slices[intz+1][inty][intx]+=T(val*fracz*(1.0f-fracx)*(1.0f-fracy));
    if(slices[intz+1][inty][intx+1]<1e-2f) proj_points++;
    slices[intz+1][inty][intx+1]+=T(val*fracx*fracz*(1.0f-fracy));
    if(slices[intz+1][inty+1][intx]<1e-2f) proj_points++;
    slices[intz+1][inty+1][intx]+=T(val*fracy*fracz*(1.0f-fracx));
    if(slices[intz][inty+1][intx+1]<1e-2f) proj_points++;
    slices[intz][inty+1][intx+1]+=T(val*fracx*fracy*(1.0f-fracz));
    if(slices[intz+1][inty+1][intx+1]<1e-2f) proj_points++;
    slices[intz+1][inty+1][intx+1]+=T(val*fracx*fracy*fracz);

    p.coord[0]=pad+scale*(p.coord[0]-minx);
    p.coord[1]=pad+scale*(p.coord[1]-miny);
    p.coord[2]=pad+scale*(p.coord[2]-minz);
  }
  printf("readPoints: %d projected points, out of %d input points.\n", proj_points, size);

#endif


  /*
  //add extra plane (manechine, distcap, vase, isis)
  for(k=0;k<getWidth()*getHeight();k++) {
    slices[5][0][k]=100;
    //slices[getDepth()-5][0][k]=100; //footA
  }
  */

  return points;
}

#define PUT_PIXEL(a) { \
point3 p; p.coord[0]=a[0]; p.coord[1]=a[1]; p.coord[2]=a[2]; points.push_back(p); }

template< class T>
void Volume<T>::bresenham_linie_3D(int x1, int y1, int z1, int x2, int y2, int z2,
				   vector<point3> &points)
{
    int i, dx, dy, dz, l, m, n, x_inc, y_inc, z_inc, 
        err_1, err_2, dx2, dy2, dz2;
    int pixel[3];

    pixel[0] = x1;
    pixel[1] = y1;
    pixel[2] = z1;
    dx = x2 - x1;
    dy = y2 - y1;
    dz = z2 - z1;
    x_inc = (dx < 0) ? -1 : 1;
    l = abs(dx);
    y_inc = (dy < 0) ? -1 : 1;
    m = abs(dy);
    z_inc = (dz < 0) ? -1 : 1;
    n = abs(dz);
    dx2 = l << 1;
    dy2 = m << 1;
    dz2 = n << 1;

    if ((l >= m) && (l >= n)) {
        err_1 = dy2 - l;
        err_2 = dz2 - l;
        for (i = 0; i < l; i++) {
            PUT_PIXEL(pixel);
            if (err_1 > 0) {
                pixel[1] += y_inc;
                err_1 -= dx2;
            }
            if (err_2 > 0) {
                pixel[2] += z_inc;
                err_2 -= dx2;
            }
            err_1 += dy2;
            err_2 += dz2;
            pixel[0] += x_inc;
        }
    } else if ((m >= l) && (m >= n)) {
        err_1 = dx2 - m;
        err_2 = dz2 - m;
        for (i = 0; i < m; i++) {
            PUT_PIXEL(pixel);
            if (err_1 > 0) {
                pixel[0] += x_inc;
                err_1 -= dy2;
            }
            if (err_2 > 0) {
                pixel[2] += z_inc;
                err_2 -= dy2;
            }
            err_1 += dx2;
            err_2 += dz2;
            pixel[1] += y_inc;
        }
    } else {
        err_1 = dy2 - n;
        err_2 = dx2 - n;
        for (i = 0; i < n; i++) {
            PUT_PIXEL(pixel);
            if (err_1 > 0) {
                pixel[1] += y_inc;
                err_1 -= dz2;
            }
            if (err_2 > 0) {
                pixel[0] += x_inc;
                err_2 -= dz2;
            }
            err_1 += dy2;
            err_2 += dx2;
            pixel[2] += z_inc;
        }
    }
    PUT_PIXEL(pixel);
}



template< class T >
vector<typename Volume<T>::point3> 
Volume<T>::readPoints_nuage_cnt(char *fname, int mindim, int pad)
{
  vector<point3> points;

  FILE *pfile=fopen(fname, "r");
  if(!pfile) return points;
  
  static char s1[132], s2[132];

  float zplane=0; char c;
  int planes=0, vertices=0, nb, end, draw=1;
  fscanf(pfile, "%s%d", s1, &planes);
  
  for(int i=0;i<planes;i++) {
    fscanf(pfile, "%s%d%s%f", s1, &vertices, s2, &zplane);
    nb=0;
    while(nb<vertices) {
      end=0;
      point3 pp;
      while((char)getc(pfile)!= '{') ;
      do {
	point3 p;
	fscanf(pfile, "%f%f", &p.coord[0], &p.coord[1]);
	p.coord[2]=zplane;
	points.push_back(p);
	nb++;
	if(nb>1 && draw) 
	  bresenham_linie_3D(pp.coord[0], pp.coord[1], pp.coord[2],
			     p.coord[0], p.coord[1], p.coord[2], points);
	else if(!draw) draw=!draw;

	pp.coord[0]=p.coord[0];
	pp.coord[1]=p.coord[1];
	pp.coord[2]=p.coord[2];	

	do {
	  c=(char)getc(pfile);
	}while(c==' ' || c=='\n');
	if(c=='}') {
	  end=1;
	  draw=0;
	}
	else 
	  ungetc(c, pfile);
      }while(!end);
    }
  }
	
  //get bbox
  int size=(int)points.size();
  assert(size>0);
  float minx=points[0].coord[0], maxx=minx;
  float miny=points[0].coord[1], maxy=miny;
  float minz=points[0].coord[2], maxz=minz;
  for(int i=0;i<size;i++) {
    if(points[i].coord[0]<minx) minx=points[i].coord[0];
    if(points[i].coord[0]>maxx) maxx=points[i].coord[0];
    if(points[i].coord[1]<miny) miny=points[i].coord[1];
    if(points[i].coord[1]>maxy) maxy=points[i].coord[1];
    if(points[i].coord[2]<minz) minz=points[i].coord[2];
    if(points[i].coord[2]>maxz) maxz=points[i].coord[2];    
  }
  printf("readPoints: dims = [w=%f, h=%f, d=%f].\n", 
	 maxx-minx, maxy-miny, maxz-minz);
  /*
  FILE *pfile1=fopen("n_pelvis1.pts", "w");
  for(int i=0;i<size;i++) {
    fprintf(pfile1, "%f %f %f\n", points[i].coord[0], points[i].coord[1], points[i].coord[2]);
  }
  fclose(pfile1);
  exit(0);
  */

  float dmin=max(max(maxx-minx, maxy-miny), maxz-minz);
  float scale=mindim/dmin;
  printf("readPoints: creating volume of size %d %d %d.\n", 
	 int((maxx-minx)*scale)+2*pad, 
	 int((maxy-miny)*scale)+2*pad, 
	 int((maxz-minz)*scale)+2*pad);
  
  makeVolume(int((maxx-minx)*scale)+2*pad, 
	     int((maxy-miny)*scale)+2*pad, 
	     int((maxz-minz)*scale)+2*pad);


#if 1    
  int x, y, z, k;
  for(k=0;k<size;k++) {
    x=int(pad+scale*(points[k].coord[0]-minx)+0.5);
    y=int(pad+scale*(points[k].coord[1]-miny)+0.5);
    z=int(pad+scale*(points[k].coord[2]-minz)+0.5);
    slices[z][y][x]=100;

    points[k].coord[0]=pad+scale*(points[k].coord[0]-minx);
    points[k].coord[1]=pad+scale*(points[k].coord[1]-miny);
    points[k].coord[2]=pad+scale*(points[k].coord[2]-minz);
  }

#else
  
  //perform interpolation
  const int val=100;
  int intx, inty, intz;
  float x, y, z, fracx, fracy, fracz;
  for(int k=0;k<size;k++) {
    point3 &p=points[k];
    x=pad+scale*(p.coord[0]-minx);
    y=pad+scale*(p.coord[1]-miny);
    z=pad+scale*(p.coord[2]-minz);
    intx=(int)floor(x);
    inty=(int)floor(y);
    intz=(int)floor(z);
    fracx=x-intx;
    fracy=y-inty;
    fracz=z-intz;
    slices[intz][inty][intx]+=T(val*(1.0f-fracx)*(1.0f-fracy)*(1.0f-fracz));
    slices[intz][inty][intx+1]+=T(val*fracx*(1.0f-fracy)*(1.0f-fracz));
    slices[intz][inty+1][intx]+=T(val*fracy*(1.0f-fracx)*(1.0f-fracz));
    slices[intz+1][inty][intx]+=T(val*fracz*(1.0f-fracx)*(1.0f-fracy));
    slices[intz+1][inty][intx+1]+=T(val*fracx*fracz*(1.0f-fracy));
    slices[intz+1][inty+1][intx]+=T(val*fracy*fracz*(1.0f-fracx));
    slices[intz][inty+1][intx+1]+=T(val*fracx*fracy*(1.0f-fracz));
    slices[intz+1][inty+1][intx+1]+=T(val*fracx*fracy*fracz);

    p.coord[0]=pad+scale*(p.coord[0]-minx);
    p.coord[1]=pad+scale*(p.coord[1]-miny);
    p.coord[2]=pad+scale*(p.coord[2]-minz);

  }

#endif

  //writePointsCrust("pelvis_high.pts", points);

  return points;
}

template< class T >
void Volume<T>::writePointsCrust(char *fname, vector<point3> &points)
{
  int i, size=(int)points.size();
  
  FILE *pfile=fopen(fname, "w");
  if(!pfile) return;

  for(i=0;i<size;i++)
    fprintf(pfile, "%f %f %f\n", 
	    points[i].coord[0], points[i].coord[1], points[i].coord[2]);
  
  fclose(pfile);
}

template< class T >
void Volume<T>::readAVSField(char *fname, int pad)
{
  FILE *pfile=fopen(fname, "r");
  
  char buffer[128];
  int dx=0, dy=0, dz=0, dummy, width;

  while(!feof(pfile)) {
    fgets(buffer, 128, pfile);
    if(buffer[0]=='#') 
      continue;
    sscanf(buffer, "ndim=%d\n", &dx);
    assert(dx==3);
    fscanf(pfile, "dim1=%d\n", &dx);
    assert(dx>0);
    fscanf(pfile, "dim2=%d\n", &dy);
    assert(dy>0);
    fscanf(pfile, "dim3=%d\n", &dz);
    assert(dz>0);
    /*
    width=dx; //need mod 4
    while((width%4)!=0) width--;
    */
    width = dx;

    printf("readAVSField: loading volume of dimensions: %d %d %d.\n", width, dy, dz);

    makeVolume(width+2*pad, dy+2*pad, dz+2*pad);
    fscanf(pfile, "nspace=%d\n", &dummy);
    fscanf(pfile, "veclen=%d\n", &dummy);
    fscanf(pfile, "data=%s\n", buffer);
    if(!strcmp(buffer, "byte")) {
     fgets(buffer, 128, pfile);
     fgets(buffer, 128, pfile);
     fgets(buffer, 128, pfile);     
      for(int i=0;i<dz;i++) {
	Image<byte> buf(dx, dy);
	int read=fread(buf[0], sizeof(byte), dx*dy, pfile);
	assert(read==dx*dy);
	Image<T> slice(width+2*pad, dy+2*pad);
	for(int k=0;k<dy;k++)
	  for(int j=0;j<width;j++)
	    slice[k+pad][j+pad]=(T)buf[k][j];
	slices[i+pad]=slice;
      }
    }
    
    return;
  }
}

template< class T >
void Volume<T>::readVTKField(char *fname, int pad)
{
  FILE *pfile=fopen(fname, "r");
  
  char buffer[128];
  int dx=0, dy=0, dz=0, dummy1, dummy2, dummy3;

  while(!feof(pfile)) {
    fgets(buffer, 128, pfile);
    if(buffer[0]=='#') continue;
    break;
  }
  
  if(strcmp(buffer, "BINARY\n")) {
    printf("readVTKField: cannot load asci files.\n");
    return;
  }

  fgets(buffer, 128, pfile);
  if(strcmp(buffer, "DATASET STRUCTURED_POINTS\n")) {
    printf("readVTKField: can load only VTK structured points.\n");
    return;
  }        
  fgets(buffer, 128, pfile);
  sscanf(buffer, "DIMENSIONS %d %d %d\n", &dz, &dx, &dy);
  assert(dx>0 && dy>0 && dz>0);

  fgets(buffer, 128, pfile);
  sscanf(buffer, "ORIGIN %d %d %d\n", &dummy1, &dummy2, &dummy3);
    
  fgets(buffer, 128, pfile);
  sscanf(buffer, "SPACING %d %d %d\n", &dummy1, &dummy2, &dummy3);
    
  fgets(buffer, 128, pfile);
  sscanf(buffer, "POINT_DATA %d\n", &dummy1);

  fgets(buffer, 128, pfile);
  fgets(buffer, 128, pfile);

  makeVolume(dx+2*pad, dy+2*pad, dz+2*pad);
  
  printf("readVTKField: loading volume of dimensions: %d %d %d.\n", dx, dy, dz);

  Image<byte> buf(dx, dy);

  for(int i=0;i<dz;i++) {
    int read=fread(buf[0], sizeof(byte), dx*dy, pfile);
    assert(read==dx*dy);

    Image<T> &slice = this->slices[i+pad];

    for(int k=0;k<dy;k++)
      for(int j=0;j<dx;j++) {
	int val = buf[k][j];
	if(val) val = (val<<1) - 1;
	slice[k+pad][j+pad]=(T)val;
      }
  }

  return;
}

template< class T >
void Volume<T>::readBinaryField(char *fname, int pad, int &np, int *&points, float *&normals)
{
  FILE *pfile=fopen(fname, "rb");
  
  char buffer[128];
  int dx=0, dy=0, dz=0, dummy1, dummy2, dummy3;
  
  unsigned char compressed = fgetc(pfile);

  assert(!compressed);

  uint64_t shape[3];
  fread(&shape, 3*sizeof(uint64_t), 1, pfile);

  dz = (int)shape[0];
  dy = (int)shape[1];
  dx = (int)shape[2];

  assert(dx>0 && dy>0 && dz>0);

  printf("readBinaryField: loading volume of dimensions: %d %d %d.\n", dx, dy, dz);

  makeVolume(dx+2*pad, dy+2*pad, dz+2*pad);
  
  Image<byte> buf(dx, dy);

  for(int i=0;i<dz;i++) {
    int read=fread(buf[0], sizeof(byte), dx*dy, pfile);
    assert(read==dx*dy);

    Image<T> &slice = this->slices[i+pad];

    for(int k=0;k<dy;k++)
      for(int j=0;j<dx;j++) {
	int val = buf[k][j];
	if(val) val = 255;
	slice[k+pad][j+pad]=(T)val;
      }
  }

  np = 0;
  fread(&np, sizeof(int), 1, pfile);
  
  points = new int[3*np];
  normals = new float[3*np];

  fread(points, 3*sizeof(int), np, pfile);
  fread(normals, 3*sizeof(float), np, pfile);

  fclose(pfile);

  return;
}

template< class T >
void Volume<T>::exportAVSField(char *fname)
{
  FILE *pfile=fopen(fname, "w");

  fprintf(pfile, "# AVS field file %s\n#\n", fname);
  fprintf(pfile, "%s", "ndim=3\n");
  fprintf(pfile, "dim1=%d\n", getWidth());
  fprintf(pfile, "dim2=%d\n", getHeight());
  fprintf(pfile, "dim3=%d\n", getDepth());
  fprintf(pfile, "%s", "nspace=3\n");
  fprintf(pfile, "%s", "veclen=1\n");
  //fprintf(pfile, "%s", "data=byte\n");
  fprintf(pfile, "%s", "data=float\n");
  fprintf(pfile, "%s", "field=uniform\n");
  fprintf(pfile, "min_ext=%f %f %f\n", 0.0f, 0.0f, 0.0f);
  fprintf(pfile, "max_ext=%f %f %f\n", 
	  (float)getWidth(), (float)getHeight(), (float)getDepth());
  
  fputc(0x0c, pfile);
  fputc(0x0c, pfile);

  //Image<byte> img(getWidth(), getHeight());

  Image<float> slice(getWidth(), getHeight());
  char *odata = (char*)slice.getPixels();

  for(int i=0;i<getDepth();i++) {
    /*  
    for(int j=0;j<getWidth()*getHeight();j++)  {
      img[0][j] = (byte)(255.0f*slices[i][0][j]);
    }

    fwrite(img.getPixels(), sizeof(byte), img.getSize(), pfile);
    */
    
    char *idata = (char*)slices[i].getPixels();

    //swap bytes (endianness issue)
    for(int j=0;j<getWidth()*getHeight();j++)  {
      odata[j*4+0] = idata[j*4+3];
      odata[j*4+1] = idata[j*4+2];
      odata[j*4+2] = idata[j*4+1];
      odata[j*4+3] = idata[j*4+0];
    }

    fwrite(slice.getPixels(), sizeof(float), slice.getSize(), pfile);
    
  }
  
  fclose(pfile);
}

template class Volume<float>;      
template class Volume<byte>;      
template class Volume<char>;






