#pragma once

#include "Volume.h"
#include "inverseFT.h"

#include <vector>
#include <deque>
#include <set>
#include <math.h>
#include <assert.h>

#include "hashwrap.h"

class collapseSkel3d {
 public:
  struct coord3 {
    short int x, y, z;

    bool operator==(const coord3 &a) const {
      return (a.x==x) && (a.y==y) && (a.z==z);
    }
    //lexicografic compare
    bool operator<(const coord3 &a) const {      
      if(z<a.z) return true;
      if(z>a.z) return false;
      if(y<a.y) return true;
      if(y>a.y) return false;
      if(x<a.x) return true;
      return false;
    }

    coord3(int xx=0, int yy=0, int zz=0) : x(xx), y(yy), z(zz) {};
  };

  struct coord3hasher {
    size_t operator() (const coord3& p) const {
      return p.z * 1000000 + p.y * 1000 + p.x;
      //distribute available bits over size_t
      //return (p.z << 20) ^ (p.y << 10 ) ^ (p.x);
    }
  };

  struct index_cmp_min {
    Volume<float> &data;

    index_cmp_min(Volume<float> &d) : data(d) {};

    bool operator()(const coord3 &a, const coord3 &b) const {
      return data[a.z][a.y][a.x] < data[b.z][b.y][b.x];
    }
  };

  struct index_cmp_max {
    Volume<float> &data;

    index_cmp_max(Volume<float> &d) : data(d) {};

    bool operator()(const coord3 &a, const coord3 &b) const {
      return data[a.z][a.y][a.x] > data[b.z][b.y][b.x];
    }
  };

  struct index_cmp_minb {
    Volume<byte> &thin_img;
    Volume<float> &edt;
    Volume<float> &dens;

  index_cmp_minb(Volume<byte> &_thin_img, Volume<float> &_edt, Volume<float> &_dens) : 
    thin_img(_thin_img), edt(_edt), dens(_dens) {};

    bool operator()(const coord3 &a, const coord3 &b) const {
      int na=0, nb=0;

      for(int z=a.z-1;z<=a.z+1;z++)
	for(int y=a.y-1;y<=a.y+1;y++)
	  for(int x=a.x-1;x<=a.x+1;x++)
	    if(thin_img[z][y][x]) na++;

      for(int z=b.z-1;z<=b.z+1;z++)
	for(int y=b.y-1;y<=b.y+1;y++)
	  for(int x=b.x-1;x<=b.x+1;x++)
	    if(thin_img[z][y][x]) nb++;

      return na > nb;

      
      float curv1 = thin_img[a.z][a.y][a.x+1]+thin_img[a.z][a.y][a.x-1]+ 
	thin_img[a.z][a.y+1][a.x]+thin_img[a.z][a.y-1][a.x]+
	thin_img[a.z+1][a.y][a.x]+thin_img[a.z-1][a.y][a.x];
	//- 6.0f*thin_img[a.z][a.y][a.x];

      float curv2 = thin_img[b.z][b.y][b.x+1]+thin_img[b.z][b.y][b.x-1]+ 
	thin_img[b.z][b.y+1][b.x]+thin_img[b.z][b.y-1][b.x]+
	thin_img[b.z+1][b.y][b.x]+thin_img[b.z-1][b.y][b.x];
	//-6.0f*thin_img[b.z][b.y][b.x];

      return curv1 > curv2;
      
      
      if(edt[a.z][a.y][a.x] < edt[b.z][b.y][b.x]) return true;
      else if(edt[a.z][a.y][a.x] > edt[b.z][b.y][b.x]) return false;

      return dens[a.z][a.y][a.x] < dens[b.z][b.y][b.x];

    }

  };

  struct index_cmp_min_div {
    Volume<float> &u, &v, &w;

  index_cmp_min_div(Volume<float> &_u, Volume<float> &_v, Volume<float> &_w) : 
    u(_u), v(_v), w(_w) {};

    bool operator()(const coord3 &a, const coord3 &b) const {
      
      float curv1 = u[a.z][a.y][a.x+1]-u[a.z][a.y][a.x-1]+ 
	v[a.z][a.y+1][a.x]-v[a.z][a.y-1][a.x]+ 
	w[a.z+1][a.y][a.x]-w[a.z-1][a.y][a.x]; 

      float curv2 = u[b.z][b.y][b.x+1]-u[b.z][b.y][b.x-1]+ 
	v[b.z][b.y+1][b.x]-v[b.z][b.y-1][b.x]+ 
	w[b.z+1][b.y][b.x]-w[b.z-1][b.y][b.x]; 

      return curv1 > curv2;

    }
  };

private:
  Volume<float> dens, dens_prev;
  Volume<byte> thin_img;
  Volume<float> max_dens, edt;

  Volume<float> u, v, w;

  Volume<int> ft;
  std::vector <InverseFT> ift;

  std::deque<coord3> queue, queue1;	
  index_cmp_minb icmp;

  //diffusion weights (precomputed)
  float *diff_weights;
  float (*diff_vecs)[3];
  int r;
  
  int sim_iter;
  float tdens, curr_dst, max_dst;

 protected:
  bool isSimple(int x, int y, int z,Volume<byte> &thin_img);
  bool isEndPoint(int x, int y, int z, bool curve, Volume<byte> &thin_img);

  bool isRemovable(int x, int y, int z, Volume<float> &dens);

  float transport_dens_diffusion(float weight, Volume<float> &dens, 
				 Volume<float> &dens_prev, 
				 std::vector<coord3> &ipoints);

  void reaction(std::vector<coord3> &ipoints, Volume<float> &dens);

  float transport_dens_advection(float weight, Volume<float> &dens, 
				 Volume<float> &dens_prev,
				 std::vector<coord3> &ipoints);

  float transport_dens_advection_unconstrained(float weight, std::vector<coord3> &out_coords, Volume<float> &tmp_dens, Volume<float> &dens, 
				 Volume<float> &dens_prev,
				 std::vector<coord3> &ipoints);

  float transport_dens_advection_project(float weight, std::vector<coord3> &out_coords, Volume<float> &tmp_dens, Volume<float> &dens, 
				 Volume<float> &dens_prev,
				 std::vector<coord3> &ipoints);

  void transport_vel_advection(float weight,Volume<float> &dens, 
			       Volume<float> &u,Volume<float> &v,Volume<float> &w, 
			       std::vector<coord3> &ipoints);

  void transport_vel_diffusion(float weight,Volume<float> &dens, 
			       Volume<float> &u,Volume<float> &v,Volume<float> &w, 
			       std::vector<coord3> &ipoints);


 public:
 collapseSkel3d() : icmp(thin_img, edt, dens_prev) { tdens = 0.0f; sim_iter = 0; curr_dst = 0.0f; max_dst = 0.0f; };
  
  virtual ~collapseSkel3d() { delete [] diff_weights; }
  void init(Volume<byte> &data, int thr, int np, int *points, float *normals);
  void init_iteration(float dt) { sim_iter = 0; curr_dst = 0.0f; }
  int  collapse_iteration(float dincr);
  void assignImpLoops();
  void simplify_skel(Volume<float> &lambda, float thr, float incr);
  void simplify_skel_filter(Volume<float> &lambda, float thr, float incr);
  void fillGaps(float thr);

  void get_corrected_flux(Volume<float> &logd, Volume<float> &nflux);
  void get_logdensity(Volume<float> &logd);
  void hj3d(bool use_corr, float thres);
  
  Volume<byte> &getThinImage() { return thin_img; }
  Volume<float> &getImportance() { return max_dens; }
  Volume<float> &getDensity(float &totdens) { totdens=tdens; return (sim_iter & 1) ? dens : dens_prev; }
  Volume<float> &getEDT(float  &max_edt) { max_edt = max_dst; return edt; }
  std::deque<coord3> &getIPoints() { return queue; }

  Volume<float> &getU() { return u; }
  Volume<float> &getV() { return v; }
  Volume<float> &getW() { return w; }

};

