#include <stdlib.h>
#include <stdio.h>
#include "glwrapper.h"
#include <math.h>
#include <unistd.h>
#include <algorithm>
#include <omp.h>

#include "Volume.h"
#include "collapseSkel3d.h"
#include "zpr.h"
#include "CIsoSurface.h"

int buttons[256];

static int win_id;
static int win_x, win_y;
static int mouse_down[3];
static int omx, omy, mx, my;
static float max_depth;

static collapseSkel3d skel;						//skeletonization engine
static float		  timer;	
static int			  iterations = 0;

const int threshold = 150;

//importance increment (for incrementing the threshold tau)
float imp_step = 1e-5f;
float imp_thr = imp_step;

CIsoSurface<byte> surf;
GLUquadric* quad = 0;

/*
  ----------------------------------------------------------------------
   OpenGL specific drawing routines
  ----------------------------------------------------------------------
*/

static void pre_display ( void )
{
  glPushMatrix();

  glViewport ( 0, 0, win_x, win_y );
  glMatrixMode ( GL_PROJECTION );
  glLoadIdentity ();
  //gluOrtho2D ( 0.0, 1.0, 0.0, 1.0 ); 	
  gluPerspective(60.0, double(win_x)/win_y, 1.0f, max_depth);
  
  glClearColor ( 1.0f, 1.0f, 1.0f, 1.0f );
  //glClearColor ( 0.0f, 0.0f, 0.0f, 1.0f );
  glClear ( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  glMatrixMode(GL_MODELVIEW);
  //glLoadIdentity ();
}

static void post_display ( void )
{
  glPopMatrix();

  glutSwapBuffers ();
}

static void float2rgb(float value,float *col)	//simple color-coding routine
{
   static const float dx = 0.8f;
   
   if (value<0.0f) value=0.0f;
   if (value>1.0f) value=1.0f;
   
   value = (6.0f-2.0f*dx)*value+dx;
   col[0] = std::max(0.0f,(3.0f-fabsf(value-4.0f)-fabsf(value-5.0f))/2.0f);
   col[1] = std::max(0.0f,(4.0f-fabsf(value-2.0f)-fabsf(value-4.0f))/2.0f);
   col[2] = std::max(0.0f,(3.0f-fabsf(value-1.0f)-fabsf(value-2.0f))/2.0f);
}

/*
  ----------------------------------------------------------------------
   GLUT callback routines
  ----------------------------------------------------------------------
*/


static void stop_collapse()										//End an earlier-started collapse process, display results/timings.
{
	glutIdleFunc(0);
	buttons['s'] = false;
	iterations = 0;
	
    glutSetWindow(win_id);
	glutPostRedisplay();
}

static void do_one_collapse_step()								//Do one collapse step
{
  static int prev_interface_size = -1;
  
  //glutSetWindow ( win_id );
  //glutPostRedisplay ();		
  int interface_size = skel.collapse_iteration(1.0f);
  ++iterations;
  
  printf("Iterations: %d in %f seconds\n",iterations,omp_get_wtime()-timer);	

  if(!interface_size)
  //if (interface_size<2 || prev_interface_size==interface_size)											//We can stop the collapse if the interface is one voxel..

  {
	stop_collapse();
  }
  
  prev_interface_size = interface_size;
}

static void start_collapse()									//Start the iterative collapse process
{
	timer = omp_get_wtime();
	iterations = 0;
	glutIdleFunc(do_one_collapse_step);
}



static void key_func ( unsigned char key, int x, int y )
{
  buttons[key] = !buttons[key];

  switch ( key )
    {
    case '+':
	case '=':
      imp_thr += imp_step;
      printf("Importance thr: %.7e\n", imp_thr);
      break;
    case '-':
	case '_':
      imp_thr -= imp_step;
      if(imp_thr<imp_step) imp_thr = imp_step;
      printf("Importance thr: %.7e\n", imp_thr);
      break;
    case 'e':
      printf("\nExporting importance to out.fld\n");
      skel.getImportance().exportAVSField("out.fld");
      break;

    case 'q':
    case 'Q':
      exit ( 0 );
    break;
    }

  glutPostRedisplay();
}


static void mouse_func ( int button, int state, int x, int y )
{
  mouse_down[button] = state == GLUT_DOWN;
  int keys = glutGetModifiers();
  if (keys & GLUT_ACTIVE_CTRL) button = GLUT_MIDDLE_BUTTON;
  if (keys & GLUT_ACTIVE_SHIFT) button = GLUT_RIGHT_BUTTON;

  zprMouse(button, state, x, y);

  mx = x;
  my = y;

}

static void motion_func ( int x, int y )
{
  zprMotion(x, y);

  mx = x;
  my = y;

  glutPostRedisplay();
}

static void reshape_func ( int width, int height )
{
  glutSetWindow ( win_id );
  glutReshapeWindow ( width, height );
  
  win_x = width;
  win_y = height;
}

static void draw_surface_transparent()
{
  static const float col[4]={1.0f, 1.0f, 1.0f, 0.3f};

  //glEnable(GL_LIGHTING);
  //glEnable(GL_LIGHT0);

  //glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, col);
  glColor4fv(col);

  glEnable(GL_BLEND); 
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glDepthMask (GL_FALSE);

  glBegin(GL_TRIANGLES);
  for(int i=0;i<surf.m_nTriangles;i++) {
    int v = surf.m_piTriangleIndices[i*3];
    glNormal3fv(surf.m_pvec3dNormals[v]);
    glVertex3fv(surf.m_ppt3dVertices[v]);

    v = surf.m_piTriangleIndices[i*3+1];
    glNormal3fv(surf.m_pvec3dNormals[v]);
    glVertex3fv(surf.m_ppt3dVertices[v]);

    v = surf.m_piTriangleIndices[i*3+2];
    glNormal3fv(surf.m_pvec3dNormals[v]);
    glVertex3fv(surf.m_ppt3dVertices[v]);
  }
  glEnd();

  glDepthMask (GL_TRUE);
  glDisable(GL_BLEND);

  //glDisable(GL_LIGHTING);
}

static void draw_thin_image()
{
  Volume<byte> &vol = skel.getThinImage();

  const int width = vol.getWidth(), height = vol.getHeight(), depth = vol.getDepth();

  glColor3f(0.0f, 0.0f, 1.0f);
  /*
  float tdens = 1.0f;
  Volume<float> &dens = skel.getDensity(tdens);
  float col[4];
  */
  for(int i=1;i<depth-1;i++)
    for(int j=1;j<height-1;j++)
      for(int k=1;k<width-1;k++) {

	if(!vol[i][j][k]) continue;

	bool draw = false;

	for(int m=-1;m<=1;m++)
	  for(int n=-1;n<=1;n++)
	    for(int p=-1;p<=1;p++) {

	      if(!vol[i+m][j+n][k+p]) {
		draw = true;
		break;
	      }	     
	    }
	
	if(!draw) continue;

	/*
	float2rgb(dens[i][j][k]/2.0f, col);	
	glColor3fv(col);
	*/

	glTranslatef(k, j, i);
	glutSolidCube(1.0f);
	glTranslatef(-k, -j, -i);
      }

}

static void draw_importance()
{
  Volume<float> &imp = skel.getImportance();
  
  float max_dst = 1.0f;
  Volume<float> &edt = skel.getEDT(max_dst);
  
  const int width = imp.getWidth(), height = imp.getHeight(), depth = imp.getDepth();

  float col[4];
  /*
  glEnable(GL_LIGHTING);
  glEnable(GL_LIGHT0);

  glEnable(GL_COLOR_MATERIAL);

  glShadeModel(GL_SMOOTH);
  glEnable(GL_AUTO_NORMAL);
  glEnable(GL_NORMALIZE);
  */
  for(int i=1;i<depth-1;i++)
    for(int j=1;j<height-1;j++)
      for(int k=1;k<width-1;k++) {
	
	if(imp[i][j][k]<imp_thr) continue;
	
	//printf("%f ", imp[i][j][k]/100);

	float2rgb(imp[i][j][k]*100, col); //100

	//float2rgb(edt[i][j][k]/max_dst, col);   
	glColor3fv(col);
	
	glTranslatef(k, j, i);
	glutSolidCube(1.0f);
	glTranslatef(-k, -j, -i);
      }
  /*
  glDisable(GL_COLOR_MATERIAL);
  glDisable(GL_LIGHTING);
  */
}

static void draw_velocity(int step, float len)
{
  Volume<float> &u = skel.getU(), &v = skel.getV(), &w = skel.getW();

  if(u.getDepth()==0) {
    printf("Velocity field not initialized !\n");
    return;
  }
    
  Volume<float> &imp = skel.getImportance();

  float tdens = 1.0f;
  Volume<float> &dens = skel.getDensity(tdens);

  const int width = imp.getWidth(), height = imp.getHeight(), depth = imp.getDepth();

  float mat[16], col[4];
  mat[3] = mat[7] = mat[11] = mat[12] = mat[13] = mat[14] = 0; mat[15] = 1;

  //const float len = 1.0f; //arrow length
  const float w_upx = 0.0f, w_upy = 1.0f, w_upz = 0.0f; //world up vector (0,1,0)

  float max_edt = 1.0f;
  Volume<float> &edt = skel.getEDT(max_edt);

  for(int i=1;i<depth-1;i+=step)
    for(int j=1;j<height-1;j+=step)
      for(int k=1;k<width-1;k+=step) {
	
	//if(imp[i][j][k]<imp_thr) continue;
	if(dens[i][j][k]/tdens<imp_thr) continue;

	//view vector 
	float vx = u[i][j][k], vy = v[i][j][k], vz = w[i][j][k];
	//vx = u[i][j][k]; vy = v[i][j][k]; vz = w[i][j][k];
	float mag = sqrtf(vx*vx + vy*vy + vz*vz);
	if(mag<1e-7f) continue;
	vx/=mag; vy/=mag; vz/=mag;


#if 0	
	float vx = 0.5f*(edt[i][j][k+1] - edt[i][j][k-1]);
	float vy = 0.5f*(edt[i][j+1][k] - edt[i][j-1][k]);
	float vz = 0.5f*(edt[i+1][j][k] - edt[i-1][j][k]);

	float dot = vx*u[i][j][k] + vy*v[i][j][k] +  vz*w[i][j][k];

	float2rgb(0.5f*(dot+1.0f), col);
	glColor3fv(col);	

#else
	
	
	float2rgb(dens[i][j][k]*mag/5.0f, col);
	glColor3fv(col);	


#endif



	//up vector
	float up_proj = vy; 	
	float v_upx = w_upx - up_proj*vx;
	float v_upy = w_upy - up_proj*vy;
	float v_upz = w_upz - up_proj*vz;
	mag = sqrtf(v_upx*v_upx + v_upy*v_upy + v_upz*v_upz);
	if(mag<1e-7f) continue;
	v_upx/=mag; v_upy/=mag; v_upz/=mag;
	
	//side vector
	float v_rx = -vy*v_upz + vz*v_upy;
	float v_ry = -vz*v_upx + vx*v_upz;
	float v_rz = -vx*v_upy + vy*v_upx;

	//fill-in rotation part
	mat[0] = v_rx; mat[1] = v_ry; mat[2] = v_rz;
	mat[4] = v_upx; mat[5] = v_upy; mat[6] = v_upz;
	mat[8] = vx; mat[9] = vy; mat[10] = vz;

	//arrow head
	glPushMatrix();
	float arrow_x = k + 0.6f*vx*len, arrow_y = j + 0.6f*vy*len, arrow_z =  i + 0.6f*vz*len;
	glTranslatef(arrow_x, arrow_y, arrow_z);

	glMultMatrixf(mat);

	gluCylinder(quad, 0.25f*len, 0.0f, 0.41f*len, 6, 3);
	glPopMatrix();       

	//arrow body
	glPushMatrix();
	glTranslatef(k, j, i);

	glMultMatrixf(mat);

	gluCylinder(quad, 0.12f*len, 0.12f*len, 0.6f*len, 6, 3);
	glPopMatrix();       

      }
}

static void draw_grad_thin(int step, float len)
{
   //spline kernel
   static const float kernel[10] = {1.0f, 4.0f, 1.0f, 4.0f, 16.0f, 4.0f, 1.0f, 4.0f, 1.0f, 72.0f};


  float tdens = 1.0f;
  Volume<float> &dens = skel.getDensity(tdens);

  const int width = dens.getWidth(), height = dens.getHeight(), depth = dens.getDepth();

  float mat[16], col[4];
  mat[3] = mat[7] = mat[11] = mat[12] = mat[13] = mat[14] = 0; mat[15] = 1;

  //const float len = 1.0f; //arrow length
  const float w_upx = 0.0f, w_upy = 1.0f, w_upz = 0.0f; //world up vector (0,1,0)

  float max_edt = 1.0f;
  Volume<float> &edt = skel.getEDT(max_edt);

  Volume<byte> &gimg = skel.getThinImage();


  for(int i=1;i<depth-1;i+=step)
    for(int j=1;j<height-1;j+=step)
      for(int k=1;k<width-1;k+=step) {
	
	//if(imp[i][j][k]<imp_thr) continue;
	//if(dens[i][j][k]/tdens<imp_thr) continue;

	float gx = 0.0f, gy = 0.0f, gz = 0.0f, kval; 
	  

	  if(gimg[i][j][k]) continue;
	  
	  int idx = 0; 
	  for(int n=-1;n<=1;n++)
	    for(int m=-1;m<=1;m++) {	
	      kval = kernel[idx++];
	      gx += kval*(gimg[i+n][j+m][k+1]-gimg[i+n][j+m][k-1]);
	      gy += kval*(gimg[i+n][j+1][k+m]-gimg[i+n][j-1][k+m]);
	      gz += kval*(gimg[i+1][j+n][k+m]-gimg[i-1][j+n][k+m]);	
	    }    

	//view vector 
	float vx = gx, vy = gy, vz = gz;
	//vx = u[i][j][k]; vy = v[i][j][k]; vz = w[i][j][k];
	float mag = sqrtf(vx*vx + vy*vy + vz*vz);
	if(mag<1e-7f) continue;
	vx/=mag; vy/=mag; vz/=mag;


#if 0	
	float vx = 0.5f*(edt[i][j][k+1] - edt[i][j][k-1]);
	float vy = 0.5f*(edt[i][j+1][k] - edt[i][j-1][k]);
	float vz = 0.5f*(edt[i+1][j][k] - edt[i-1][j][k]);

	float dot = vx*u[i][j][k] + vy*v[i][j][k] +  vz*w[i][j][k];

	float2rgb(0.5f*(dot+1.0f), col);
	glColor3fv(col);	

#else
	
	
	float2rgb(dens[i][j][k]*mag/5.0f, col);
	glColor3fv(col);	


#endif



	//up vector
	float up_proj = vy; 	
	float v_upx = w_upx - up_proj*vx;
	float v_upy = w_upy - up_proj*vy;
	float v_upz = w_upz - up_proj*vz;
	mag = sqrtf(v_upx*v_upx + v_upy*v_upy + v_upz*v_upz);
	if(mag<1e-7f) continue;
	v_upx/=mag; v_upy/=mag; v_upz/=mag;
	
	//side vector
	float v_rx = -vy*v_upz + vz*v_upy;
	float v_ry = -vz*v_upx + vx*v_upz;
	float v_rz = -vx*v_upy + vy*v_upx;

	//fill-in rotation part
	mat[0] = v_rx; mat[1] = v_ry; mat[2] = v_rz;
	mat[4] = v_upx; mat[5] = v_upy; mat[6] = v_upz;
	mat[8] = vx; mat[9] = vy; mat[10] = vz;

	//arrow head
	glPushMatrix();
	float arrow_x = k + 0.6f*vx*len, arrow_y = j + 0.6f*vy*len, arrow_z =  i + 0.6f*vz*len;
	glTranslatef(arrow_x, arrow_y, arrow_z);

	glMultMatrixf(mat);

	gluCylinder(quad, 0.25f*len, 0.0f, 0.41f*len, 6, 3);
	glPopMatrix();       

	//arrow body
	glPushMatrix();
	glTranslatef(k, j, i);

	glMultMatrixf(mat);

	gluCylinder(quad, 0.12f*len, 0.12f*len, 0.6f*len, 6, 3);
	glPopMatrix();       

      }
}

static void draw_density()
{
  float tdens = 1.0f;
  Volume<float> &dens = skel.getDensity(tdens);

  const int width = dens.getWidth(), height = dens.getHeight(), depth = dens.getDepth();

  float col[4];
 
  const float dens_scale = 20.0f;//0.5f*tdens; //2.0f

  for(int i=1;i<depth-1;i++)
    for(int j=1;j<height-1;j++)
      for(int k=1;k<width-1;k++) {
	
	//if(dens[i][j][k]/dens_scale<imp_thr) continue;
	if(dens[i][j][k]<1.0f) continue;
	
	float2rgb(dens[i][j][k]/dens_scale, col);
	glColor3fv(col);
	
	glTranslatef(k, j, i);
	glutSolidCube(1.0f);
	glTranslatef(-k, -j, -i);
      }
  
}

static void draw_ipoints()
{
  //std::vector<collapseSkel3d::coord3> &ipoints = skel.getIPoints();
  std::deque<collapseSkel3d::coord3> &ipoints = skel.getIPoints();
  //printf("ipoints: %d\n", ipoints.size());

  glColor3f(0.0f, 0.0f, 0.0f);

  for(int i=0;i<ipoints.size();i++) {
    
    collapseSkel3d::coord3 &c = ipoints[i];

    glTranslatef(c.x, c.y, c.z);
    glutSolidCube(1.0f);
    glTranslatef(-c.x, -c.y, -c.z);    
  }

}

static void display_func ( void )
{
  pre_display ();

  Volume<byte> &vol = skel.getThinImage();

  const int width = vol.getWidth(), height = vol.getHeight(), depth = vol.getDepth();

  glTranslatef(-width/2, -height/2, -1.7*depth); 

  //draw_surface_transparent();


  if(buttons['t']) 
  {
    //draw_ipoints();
    draw_thin_image();
  }
  else if(buttons['i'])
  {
    draw_importance();
    //draw_density();
  }
  else if(buttons['d'])
  {
    draw_density();
    //draw_ipoints();
  }
  else if(buttons['v']) 
  {
    draw_velocity(1, 1.2f);
    //draw_grad_thin(1, 1.2f);
  }

  post_display ();
}


/*
  ----------------------------------------------------------------------
   open_glut_window --- open a glut compatible window and set callbacks
  ----------------------------------------------------------------------
*/

static void open_glut_window ( void )
{
  glutInitDisplayMode ( GLUT_RGBA | GLUT_DOUBLE | GLUT_DEPTH);

  glutInitWindowPosition ( 1680, 0 );
  glutInitWindowSize ( win_x, win_y );
  win_id = glutCreateWindow ( "Alias | wavefront" );
  
  glClearColor ( 1.0f, 1.0f, 1.0f, 1.0f );
  glClear ( GL_COLOR_BUFFER_BIT );
  glutSwapBuffers ();
  glClear ( GL_COLOR_BUFFER_BIT );
  glutSwapBuffers ();
  
  glShadeModel(GL_SMOOTH);       
  //glCullFace(GL_FRONT_AND_BACK);
  glHint(GL_POLYGON_SMOOTH_HINT, GL_NICEST);

  //for shading
  glEnable(GL_LIGHTING);
  glEnable(GL_LIGHT0);

  glEnable(GL_COLOR_MATERIAL);

  glShadeModel(GL_SMOOTH);
  glEnable(GL_AUTO_NORMAL);
  glEnable(GL_NORMALIZE);
  //

  glEnable(GL_DEPTH_TEST);
  glDepthFunc(GL_LEQUAL);

  pre_display ();

  quad = gluNewQuadric();
  
  glutKeyboardFunc ( key_func );
  glutMouseFunc ( mouse_func );
  glutMotionFunc ( motion_func );
  glutReshapeFunc ( reshape_func );
  glutDisplayFunc ( display_func );
}


/*
  ----------------------------------------------------------------------
   main --- main routine
  ----------------------------------------------------------------------
*/

int main ( int argc, char ** argv )
{
  glutInit ( &argc, argv );

  for(int i=0;i<256;i++)
    buttons[i] = 0;
	
  win_x = 800;//1024;
  win_y = 800;//1024;
  open_glut_window ();

  buttons['t'] = true;
  buttons['i'] = true;

  {
    Volume<byte> data;  

    int np = 0;
    int *points;
    float *normals;

    bool use_corr = false;
    float thr = -0.3f;

    if(argc!=4) {
      printf("\nUsage: %s threshold method model.vtk\n", argv[0]);
      printf("\twith threshold: negative value to threshold flux (e.g. in range [-1e+3f, -1e-3f])\n");
      printf("\t        method: 0-original H-J or 1-curvature-corrected\n\n");    
      return 1;
    }

    thr = atof(argv[1]);
    use_corr = atoi(argv[2]) ? true : false;
    
    if(strstr(argv[3], ".fld"))
      data.readAVSField(argv[3], 2);
    else if (strstr(argv[3], ".vtk"))
      data.readVTKField(argv[3], 2);
    else if (strstr(argv[3], ".dat"))
      data.readBinaryField(argv[3], 2, np, points, normals);
    else {
      printf("Unknown file format.\n");
      return -1;
    }
	
    max_depth = 1.5*sqrt(data.getWidth()*data.getWidth() + data.getHeight()*data.getHeight() + data.getDepth()*data.getDepth());


    int width = data.getWidth();
    int height = data.getHeight();
    int depth = data.getDepth();
    
    std::vector<byte> vec;
    data.toVector(vec);
    surf.GenerateSurface(&vec[0], threshold, width-1, height-1, depth-1, 1.0, 1.0, 1.0);

    if(!surf.IsSurfaceValid()) {
      printf("\tCIsoSurface: Couldn't generate a valid surface. Exiting.\n");
      exit(0);
    }  

    skel.init(data, threshold, np, points, normals);

    skel.init_iteration(0.99990f);

    skel.hj3d(use_corr, thr);

    int mdim = std::max(depth, std::max(width, height));

    zprInit(0, 0, -(1.7f-0.5f)*depth);
  }
 
  glutMainLoop ();
  
  exit ( 0 );
}
