/************************************************************************************/
/* avs_io.c  - Reads and writes AVS field files (.fld)                              */
/* Usage   :   Implements Write functions for unsigned char and shorts              */
/* Author  :   Georgios K. Ouzounis & Andrei Jalba                                  */
/* Dep.    :   IWI - University of Groningen                                        */
/* Vs&Date :   vs 1.0 - 2nd Nov. 2004                                               */  
/************************************************************************************/
/*             ---    COPYRIGHT NOTICE FROM ORIGINAL CODE SOURCE   ---              */ 
/************************************************************************************|
|  Copyright (c) 2003, Division of Imaging Science and Biomedical Engineering,       |
|  University of Manchester, UK.  All rights reserved.                               |
|                                                                                    |
|  Redistribution and use in source and binary forms, with or without modification,  |
|  are permitted provided that the following conditions are met:                     |
|                                                                                    |
|    . Redistributions of source code must retain the above copyright notice,        |
|      this list of conditions and the following disclaimer.                         |
|                                                                                    |
|    . Redistributions in binary form must reproduce the above copyright notice,     |
|      this list of conditions and the following disclaimer in the documentation     |
|      and/or other materials provided with the distribution.                        |
|                                                                                    |
|    . Neither the name of the University of Manchester nor the names of its         |
|      contributors may be used to endorse or promote products derived from this     | 
|      software without specific prior written permission.                           | 
|                                                                                    |  
|  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"       |  
|  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE         |   
|  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE        |  
|  ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE         |  
|  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR               |     
|  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF              |  
|  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS          |     
|  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN           |  
|  CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)           |    
|  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE        |    
|  POSSIBILITY OF SUCH DAMAGE.                                                       |  
*************************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include "avs_io.h"

static char *avs_data_out[] = {"byte", "byte", "short", "short","integer","integer",
                               "float","float","double","double"};
static char *avs_data_in[]  = {"", "byte", "", "short", "integer", "", "float", 
                               "double"};
static char  avs_ndata = 8;
static char avs_end_head[3]   = "\f\f";

/*****************************************************************/
/* AVS Field WRITE Routines                                      */
/*****************************************************************/

int avs_write_header(FILE *fp, avs_header *header)
{
   fprintf(fp, "# AVS field file\n");
   fprintf(fp, "# Output generated by avs_io routine\n");
   fprintf(fp, "# by Georgios K Ouzounis - IWI, RuG, Oct.2004\n");
   fprintf(fp, "ndim=%d\n", header->ndim);
   fprintf(fp, "dim1=%d\n", header->dim1);
   fprintf(fp, "dim2=%d\n", header->dim2);
   fprintf(fp, "dim3=%d\n", header->dim3);
   fprintf(fp, "nspace=%d\n", header->ndim);
   fprintf(fp, "veclen=1\n");
   if(header->datatype >= avs_ndata || *avs_data_out[header->datatype] == '\0')
   {
     printf("AVS files don't support current datatype %d\n", header->datatype);
     return 0;
   }
   fprintf(fp, "data=%s\n", avs_data_out[header->datatype]);
   fprintf(fp, "field=uniform\n");
   fprintf(fp, "min_ext=%f %f %f\n",header->min_x, header->min_y, header->min_z);
   fprintf(fp, "max_ext=%f %f %f\n",header->max_x, header->max_y, header->max_z);
   /*fprintf(fp, "variable 1 file = %s filetype = binary\n", header->dataname);*/
   fprintf(fp, "\f\f");

   return 1;
}

/*****************************************************************/
/* AVS Field READ Routines                                       */
/*****************************************************************/

int _avs_read_header(FILE *fp, avs_header *header)
{
  int i,j,last_pos=0;
  char buffer[256], *tok, *tmp;
  int buffer_size = 255;
  
  char *avs_keys[] = {"ndim", "dim1", "dim2", "dim3", "nspace", "veclen", 
                      "data", "field","min_ext", "max_ext", "variable", "#", "label", "unit"};
  const int avs_nkeys = 13;
  
  char *avs_format_keys[] = {"file", "filetype", "skip", "offset", "stride"};
  char avs_format_nkeys = 2;
  
  int data=1; /*, offset=0,stride=1*/
  char *filetype_str="rb";
  
  /* First check if this is a valid AVS field file*/
  tmp = fgets(buffer,buffer_size,fp);
  if(!tmp || strncmp(buffer, "# AVS field file", 16))
  {
    printf("This is not a valid AVS field file\n");
    fclose(fp);
 /*	   return(0);*/
    exit(0);
  }

  /*Initialize to default header parameters*/
  header->dataname[0]='\0';
  header->skip=0;
  header->filetype=0;
  header->datatype=1;
  
  /*parse the rest of the header fields*/
  tmp = fgets(buffer, buffer_size, fp);

  while(tmp && strncmp(tmp, avs_end_head, 2))
  {
    /* find avs keyword */
    tok = strtok(buffer, " =\n\t");
    if(tok) /* If not a blank line */
    {
      for(i=0;i<avs_nkeys; i++)
      {
        if(!strcmp(avs_keys[i],tok))
	  break;
      }
      switch(i)
      {
        case 0:  /*ndims*/
	  header->ndim = atoi((const char *)strtok(NULL, " =\n\t"));
	  if(header->ndim < 2)
	  {  
	    printf("AVS file must have 2D or 3D data\n");
	    fclose(fp);
            return 0;
	  }
	  break;
	case 1: /* dim1 */
          header->dim1 = atoi((const char *)strtok(NULL, " =\n\t"));
          break;
        case 2: /* dim2 */
          header->dim2 = atoi((const char *)strtok(NULL, " =\n\t"));
          break;
        case 3: /* dim3 */
          header->dim3 = atoi((const char *)strtok(NULL, " =\n\t"));
          break;
        case 4: /* nspace */
          header->nspace = atoi((const char *)strtok(NULL, " =\n\t"));
          break;
        case 5: /* veclen */
          /* if not scalar only first value will be read */
          header->veclen = atoi((const char *)strtok(NULL, " =\n\t"));
          if(header->veclen!=1)
            printf("AVS file must should scalar data\n");
        
          break;
        case 6: /* data */
          tok = strtok(NULL, " =\n\t");
          
          for(data=0; data<avs_ndata; data++)
          {
            if(!strcmp(avs_data_in[data], tok))
              break;
          }
  
          if(data==avs_ndata)
          {
            printf("AVS file data type not recognised\n");
            fclose(fp);
            return 0;
          }
          
          header->datatype = data;
          break;
        case 7: /* field */
	  if(strcmp("uniform", (const char *)strtok(NULL, " =\n\t")) )
          {
            printf("AVS file must have uniform data\n");
            fclose(fp);
            return 0;
          }
          break;
        case 8: /* Minimum Extend*/
	  header->min_x = atof(strtok(NULL, " =\n\t"));
	  header->min_y = atof(strtok(NULL, " =\n\t"));
	  header->min_z = atof(strtok(NULL, " =\n\t"));
	  break;
        case 9: /* Maximum Extend*/
	  header->max_x = atof(strtok(NULL, " =\n\t"));
	  header->max_y = atof(strtok(NULL, " =\n\t"));
	  header->max_z = atof(strtok(NULL, " =\n\t"));
          break;
        case 10: /* variable */
	  if(atoi((const char *)strtok(NULL, " ="))!=1)
            printf("AVS file should contain scalar data\nignoring other values\n");
          else {
            while((tok = strtok(NULL, " =\n\t")))
            {
              for(j=0; j<avs_format_nkeys; j++)
              {
                if(!strcmp(avs_format_keys[j], tok))
                  break;
              }
          
              switch(j)
              {
                case 0: /* file */
                  strcpy(header->dataname, (const char *)strtok(NULL, " =\n\t"));
                  break;
                case 1: /* filetype */
                  if(!strcmp("binary",(const char *)strtok(NULL, " =\n\t")) )
                  {
                    header->filetype = 0;
                    filetype_str="rb";
                  } 
                  else 
                  {
                    header->filetype = 1;
                    filetype_str="r";
                    printf("AVS reader can only handle binary data at the moment\n");
                    fclose(fp);
                    return 0;
                  }
                  break;
              }   
            }/* while */
          
          }
          break;
        case 11:
	case 12:
        case 13: 
	  break;
        default:
          printf("Invalid keyword in AVS file: %s\n", tok);
        }
        
      } /* not a blank line */
      /* next line */
        last_pos=ftell(fp);
      	tmp = fgets(buffer, buffer_size, fp);
    } /* while */

    /* Put the file position indicator to the first byte after \f\f*/
    fseek(fp, last_pos+2, SEEK_SET);
    return 1;
}

int avs_read_header(char *fname, avs_header *header)
{
  FILE *fp;

  fp = fopen(fname,"r");
  if(!fp)
   {
      printf("avs_read_header: Cannot open the input AVS field file\n");
      return(0);
   }
   
  /* Read AVS Header */
  _avs_read_header(fp, header);
  fclose(fp);
  return 1;
}

