#pragma once

#include "stdafx.h"
#include <cmath>
#include <cassert>

class TColor3
{
public:
	union
	{
		float rgb[3];
		struct 
		{
			float r,g,b;
		};
	};
	
	TColor3() : r(0.0f), g(0.0f), b(0.0f) {}
	TColor3(float x, float y, float z) : r(x), g(y), b(z) {}
	TColor3(const float *v) : r(v[0]), g(v[1]), b(v[2]) {}
	
	void add(const TColor3 & v)
	{
		this->r += v.r;
		this->g += v.g;
		this->b += v.b;
	}
	
	void scale(float f)
	{
		this->r *= f;
		this->g *= f;
		this->b *= f;
	}
	
	void normalize()
	{
		float max = -1.0f;
		if(r > max) max = r; 
		if(g > max) max = g;
		if(b > max) max = b;
		r /= max;
		g /= max;
		b /= max;
	}

/*
	void print(const string & s) const
	{
		printf(string(s+": (%f,%f,%f)\n").c_str(),r,g,b);
	}
*/

};

class TColor4 : public TColor3
{
public:
	float a;
	TColor4() : TColor3(), a(0) {};
	TColor4(float x, float y, float z, float v) : TColor3(x,y,z), a(v) {}
};

class TPoint2 
{
public:
	union
	{
		float pos[2];
		struct 
		{
			float x,y;
		};
		struct 
		{
			float i,j;
		};
	};
	
	TPoint2() : x(0.0f), y(0.0f) {};
	TPoint2(float p_x, float p_y) : x(p_x), y(p_y) {};

	void subtract(const TPoint2 & v)
	{
		this->x -= v.x;
		this->y -= v.y;
	}
	
	float distance(const TPoint2 & v) const
	{
		const float dx = this->x - v.x;
		const float dy = this->y - v.y;
		return sqrtf(dx*dx + dy*dy);
	}


	float length() const
	{
		return sqrtf(this->x*this->x + this->y*this->y);
	}
			
	void normalize()
	{
		float l = length();
		assert(l != 0);
		this->x /= l;		
		this->y /= l;		
	}

	inline int squaredlength() const
	{
		return (int) (x*x + y*y);
	}

	void add(const TPoint2 & v)
	{
		this->x += v.x;
		this->y += v.y;
	}

	void print() const
	{
//		printf("(%f,%f)",x,y);
	}

/*
	void print(const string & s) const
	{
		printf(string(s+": (%f,%f)\n").c_str(),x,y);
	}
*/
};

class TPoint3
{
public:
	union
	{
		float pos[3];
		struct 
		{
			float x,y,z;
		};
	};

	TPoint3() : x(0.0f), y(0.0f), z(0.0f) {};
	TPoint3(float p_x, float p_y, float p_z) : x(p_x), y(p_y), z(p_z) {};
	TPoint3(const float * p) : x(p[0]), y(p[1]), z(p[2]) {};
	TPoint3(const TPoint3 & p_Point) : x(p_Point.x),y(p_Point.y),z(p_Point.z) {};
	TPoint3(const TPoint2 & p_Point) : x(p_Point.x),y(p_Point.y),z(0) {};

	void add(const TPoint3 & v)
	{
		this->x += v.x;
		this->y += v.y;
		this->z += v.z;
	}
	
	void subtract(const TPoint3 & v)
	{
		this->x -= v.x;
		this->y -= v.y;
		this->z -= v.z;
	}

	float distance(const TPoint3 & v) const
	{
		const float dx = this->x - v.x;
		const float dy = this->y - v.y;
		const float dz = this->z - v.z;
		return sqrtf(dx*dx + dy*dy + dz*dz);
	}
	
	inline float squareddistance(const TPoint3 & v) const
	{
		const float dx = this->x - v.x;
		const float dy = this->y - v.y;
		const float dz = this->z - v.z;
		return dx*dx + dy*dy + dz*dz;
	}

	float length() const
	{
		return sqrtf(this->x*this->x + this->y*this->y + this->z * this->z);
	}

};


class TVector3 : public TPoint3
{
public:
	// Constructors
	TVector3() : TPoint3(0.0f, 0.0f, 0.0f) {};
	TVector3(float p_x, float p_y, float p_z) : TPoint3(p_x,p_y,p_z) {};
	TVector3(const TPoint3 & p_Point) : TPoint3(p_Point.x,p_Point.y,p_Point.z) {};
	TVector3(const TPoint2 & p_Point) : TPoint3(p_Point.x,p_Point.y,0) {};
	TVector3(int) : TPoint3(0.0f, 0.0f, 0.0f) {};

	// Operators
 	float operator*(const TVector3& v) { return x*v.x+y*v.y+z*v.z; }
	
	// Methods
	float length() const
	{
		return sqrtf(this->x*this->x + this->y*this->y + this->z * this->z);
	}
	
	inline float  norm()  const { return sqrtf(x*x+y*y+z*z); }
	inline float  norm2() const { return x*x+y*y+z*z; }
	float  dist2(const TVector3& v) const { return (x-v.x)*(x-v.x)+(y-v.y)*(y-v.y)+(z-v.z)*(z-v.z); }

	void normalize()
	{
		float l = length();
		if(l < 0.0001f) l = 0.0001f;
		this->x /= l;		
		this->y /= l;		
		this->z /= l;		
	}
	
	float dot(const TVector3 & v) const
	{
		return this->x*v.x + this->y*v.y + this->z*v.z;
	}
	
	void cross(TVector3 & result, const TVector3 & b) const
	{
		TVector3 r;
		r.x = (this->y * b.z - this->z * b.y);
		r.y = (this->z * b.x - this->x * b.z);
		r.z = (this->x * b.y - this->y * b.x);
		result = r;
	}

	TVector3 cross(const TVector3 & b) const
	{
		TVector3 r;
		r.x = (this->y * b.z - this->z * b.y);
		r.y = (this->z * b.x - this->x * b.z);
		r.z = (this->x * b.y - this->y * b.x);
		return r;
	}

	void scale(float f)
	{
		this->x *= f;
		this->y *= f;
		this->z *= f;
	}
	
	void negate()
	{
		this->x *= -1;
		this->y *= -1;
		this->z *= -1;
	}
	
	bool isValid() const
	{
		// Check for NaN's
		return (this->x == this->x) && (this->y == this->y) && (this->z == this->z);
	}

};


class TVector2 : public TPoint2
{
public:
	TVector2() : TPoint2(0.0f, 0.0f) {};
	TVector2(float p_x, float p_y) : TPoint2(p_x,p_y) {};
	TVector2(const TPoint2 & p_Point) : TPoint2(p_Point.x,p_Point.y) {};

	float dot(const TVector2 & v) const
	{
		return this->x*v.x + this->y*v.y;
	}

	void scale(float f)
	{
		this->x *= f;
		this->y *= f;
	}

};


class TPlane
{
public:
	TPlane() {};
	TPlane(const TPoint3 & p_Point, const TVector3 & p_Normal) : m_Point(p_Point), m_Normal(p_Normal)
	{
		// Normal is stored normalized
		m_Normal.normalize();
		
		// Calculate distance to origin 
		m_D = distance2point( TPoint3(0.0f, 0.0f, 0.0f) );
	}
	
	const TPoint3 & getPoint() const { return m_Point; } 
	const TVector3 & getNormal() const { return m_Normal; } 
	
	float distance2point(const TPoint3 & p_Point) const
	// Calculate signed distance 
	{
		TVector3 diff = p_Point;
		diff.subtract(this->m_Point);
		return m_Normal.dot(diff); 
	}
	
	void project(TPoint3 & p_Target) const
	{
		float distance = distance2point(p_Target);
		TVector3 AddVector = m_Normal;
		AddVector.scale(-distance);
		p_Target.add(AddVector);
	}

	void vectorInPlane(TVector3 & p_Vector) const
	{
		// Generate vector Indep, that is independent from the normal
		TVector3 Indep = this->getNormal();	
		{
			float x = Indep.x;
			float y = Indep.y;
			float z = Indep.z;
			if( y != 0 ) 
			{
				Indep.x = -y; 
				Indep.y = x;	
			}
			else 
			{
				Indep.x = -z; 
				Indep.z = x;	
			}
		}
		
		// The cross product of Indep and the normal lies in the plane
		Indep.cross(p_Vector, this->getNormal());
		p_Vector.normalize();
	}
	
	float planeEquationXY2Z(float x, float y) const
	{
		if(m_Normal.z == 0.0f) 
		{
			assert(false && "Error: m_Normal.z == 0.0f\n");
			//exit(1);
		}
		return -( m_Normal.x * x + m_Normal.y * y + m_D) / m_Normal.z;
	}

	float planeEquationXZ2Y(float x, float z) const
	{
		if(m_Normal.y == 0.0f) 
		{
			assert(false && "Error: m_Normal.y == 0.0f\n");
			//exit(1);
		}
		return -( m_Normal.x * x + m_Normal.z * z + m_D) / m_Normal.y;
	}

	float planeEquationYZ2X(float y, float z) const
	{
		if(m_Normal.x == 0.0f) 
		{
			assert(false && "Error: m_Normal.z == 0.0f\n");
			//exit(1);
		}
		return -( m_Normal.y * y + m_Normal.z * z + m_D) / m_Normal.x;
	}
	void reverse()
	{
		m_Normal.scale(-1.0f);
	}

/*
	void print() const
	{
		printf("Plane:\n\tOrigin:");
		m_Point.print();
		printf("\n\tNormal:");
		m_Normal.print();
		printf("\n\tD: %f", m_D);
		printf("\n");
	}
*/	
		
	TPoint3 m_Point;
	TVector3 m_Normal;
	float m_D;
};


class TRay
{
public:
	TRay() : m_Point(TPoint3(0,0,0)), m_Vector(TVector3(0,0,0)), m_Color(TColor3(0,0,0)) {};

	TRay(const TPoint3 & p, const TVector3 & v, const TColor3 & c = TColor3(1.0f, 1.0f, 1.0f))
		: m_Point(p), m_Vector(v), m_Color(c)
	{
	}
/*
	void render() const
	{	
		TVector3 p = TVector3( m_Point );
		TVector3 v = m_Vector;
		v.scale(100.0f);

		glBegin(GL_LINE_STRIP);
			glColor3fv(m_Color.rgb);
			glVertex3f(p.x, p.y, p.z);
			p.add(v);
			glVertex3f(p.x, p.y, p.z);
		glEnd();
	}
*/
	inline const TPoint3 & getPoint() const { return m_Point; }
	inline const TPoint3 & getVector() const { return m_Vector; }
	
protected:
	TPoint3 m_Point;
	TVector3 m_Vector;
	TColor3 m_Color;
};

class TLine : public TRay
{
public:
	TLine(const TPoint3 & p, const TVector3 & v, const TColor3 & c)
		: TRay(p,v,c)
	{
	}

/*
	void render() const
	{	
		glBegin(GL_LINE_STRIP);
			glColor3fv(m_Color.rgb);
			glVertex3f(m_Point.x, m_Point.y, m_Point.z);
			glVertex3f(m_Vector.x, m_Vector.y, m_Vector.z);
		glEnd();
	}
*/
};
