//
// Copyright (c) 2005  Institute for Visualization and Interactive Systems,
// University of Stuttgart, Germany
//
// This source code is distributed as part of the paper "A Generic Software Framework for 
// the GPU Volume Rendering Pipeline", submitted to the 10th International Fall Workshop
// VISION, MODELING, AND VISUALIZATION 2005, November 16 - 18, 2005 Erlangen, Germany/
//
// Details about this project can be found on the project web page at
// http://wwwvis.informatik.uni-stuttgart.de/~vollrajm/
// 
// This file may be distributed, modified, and used free of charge as long as this copyright 
// notice is included in its original form. Commercial use is strictly prohibited.
//
// Filename: Volume.cpp
//
// parts of this source code are adapted from source code provided with friendly permission
// by Tobias Schafhitzel

#include "dxstdafx.h"

//------------------------------------------------------------------------------
#include "Volume.h"
#include "VolumeHelpers.h"
#include "VolumeRenderingObjectManager.h"
#include "VolumeRenderingSharedState.h"
#include "Filtering.h"
#include "StringHelpers.h"


//------------------------------------------------------------------------------
CVolume::CVolume(void)
: m_Width(0)
, m_Height(0)
, m_nSlices(0)
, m_nFillslices(0)
, m_nAlignedslices(0)
//, m_SliceThicknessX(0.0f)
//, m_SliceThicknessY(0.0f)
//, m_SliceThicknessZ(0.0f)
, m_CubeLength(2.0f) //default value magic number
//, m_MaxX(0)
//, m_MaxY(0)
//, m_MaxZ(0)
//, m_MinX(0)
//, m_MinY(0)
//, m_MinZ(0)
, m_pRawdata(NULL)
, m_VolumeFormat(VOLUME_FORMAT_UNDEFINED)
, m_pQuantizedGradientData(NULL)
, m_VolumeGradientMethod(VOLUME_GRADIENT_SOBEL)
, m_bVolumeLoaded(false)
, m_nVoxels(0)
, m_VoxelMaxVal(0)
, m_VoxelMinVal(0)
, m_VoxelDataRange(0)
, m_pHistogram(new CHistogram())
{
	m_anSamplingPoints[0] = m_anSamplingPoints[1] = m_anSamplingPoints[2] = 0;
	m_afSamplingDistances[0] = m_afSamplingDistances[1] = m_afSamplingDistances[2] = 0.0f;
}


//------------------------------------------------------------------------------
CVolume::~CVolume(void)
{
	SAFE_DELETE(m_pHistogram);
	SAFE_FREE(m_pRawdata);
	SAFE_FREE(m_pQuantizedGradientData);
}


//------------------------------------------------------------------------------
bool CVolume::LoadVolumeFromFile(std::string sFilename)
{
	if ( !DoLoadVolumeFromFile(sFilename) ) return false;

	m_pHistogram->Reset();

	//// gradients
	{
		float*	pFloatGradients=NULL;
		//allocate memory for quantized gradients
		SAFE_FREE(m_pQuantizedGradientData);
		m_pQuantizedGradientData	= (BYTE*) malloc(sizeof(BYTE)*m_nVoxels*3);

		std::string sGradientFilename = StringHelpers::Basedir(sFilename).append( 
											std::string("\\").append(
												StringHelpers::Basename(sFilename).append(std::string(".grd"))));

		if (!LoadGradientsFromFile( sGradientFilename ) )
		{		
			int sizes[3] = {m_Width, m_Height, m_nFillslices};
			float* newGradients = NULL;

			//allocate memory for floating point gradients
			pFloatGradients	= (float*) malloc(sizeof(float)*m_nVoxels*3);

			ComputeGradients(pFloatGradients, sizes);

			////filter gradients
			{
				const int dim = 5;

				float kernel[dim*dim*dim];
				filtering::Build3dGaussianKernel(kernel, dim, 0.9f);

				newGradients = (float*) malloc(sizeof(float)*m_nVoxels*3);
				FilterGradients(pFloatGradients, newGradients, kernel, dim);
				SAFE_FREE(pFloatGradients);
			}

			QuantizeGradients(newGradients, m_pQuantizedGradientData);
			SAFE_FREE(newGradients);
			SAFE_FREE(pFloatGradients);

			SaveGradientsToFile(sGradientFilename);
		}
	}

	FindMinMaxVoxelVals();

	ComputeHistogram();

	//normalize histogram
	m_pHistogram->Normalize();

	// FIXME: vollrajm, 20050126
	// store pointer to state in local variable
	CVolumeRenderingObjectManager::GetPInstance()->GetPSharedState()->SetState ( VRSS_AF_VOLUME_SAMPLING_DISTANCES, static_cast<void*>(m_afSamplingDistances));

	CVolumeRenderingObjectManager::GetPInstance()->GetPSharedState()->SetState ( VRSS_AN_VOLUME_SAMPLING_POINTS, static_cast<void*>(m_anSamplingPoints));

	const float volDim[3]	=	{	m_afSamplingDistances[0]*(m_anSamplingPoints[0]-1),
									m_afSamplingDistances[1]*(m_anSamplingPoints[1]-1),
									m_afSamplingDistances[2]*(m_anSamplingPoints[2]-1)	};
	CVolumeRenderingObjectManager::GetPInstance()->GetPSharedState()->SetState ( VRSS_AF_VOLUME_DIMENSIONS_WORLDSPACE, static_cast<void*>(const_cast<float*>(volDim)));

	//	calculate reciprocal of volume dimensions with 
	//	volDimRecip = 1.0f / (samplingDistance * ceiling_to_next_power_of_two(#samplingPoints))
	
	const float volDimRecip[3]	=	{	1.0f /	(	m_afSamplingDistances[0] *
													(float)(VolumeHelpers::CeilingToNextPowerOfTwo(m_anSamplingPoints[0])-1)),
										1.0f /	(	m_afSamplingDistances[1] *
													(VolumeHelpers::CeilingToNextPowerOfTwo(m_anSamplingPoints[1])-1)),
										1.0f /	(	m_afSamplingDistances[2] *
													(VolumeHelpers::CeilingToNextPowerOfTwo(m_anSamplingPoints[2])-1))	};
	D3DXMATRIXA16	mVolVertex2TexCoord;
	D3DXMatrixScaling(	&mVolVertex2TexCoord, volDimRecip[0], volDimRecip[1], volDimRecip[2]);
	CVolumeRenderingObjectManager::GetPInstance()->GetPSharedState()->SetState ( VRSS_M_VOLVERTEX2TEXCOORD, static_cast<void*>(&mVolVertex2TexCoord));
	
	return true;
}


//------------------------------------------------------------------------------
bool CVolume::DoLoadVolumeFromFile(std::string sFilename)
{
	m_VoxelMaxVal = m_VoxelMinVal = 0;

	FILE * pFile; // pointer to FILE structure
	char sTmpRawfilename[256];
	char sTmpLine[256];
	char sTmp1[256];
	char sTmp2[256];
	char sTmp3[256];

	// open datfile
	if ((pFile = fopen(sFilename.c_str(), "r")) == NULL)
		//FIXME: Messagebox here
		return false;
	
	// read all lines
	while (!feof(pFile)) {
		memset(sTmpRawfilename, 0, 256);
		memset(sTmpLine, 0, 256);
		memset(sTmp1, 0, 256);
		memset(sTmp2, 0, 256);
		memset(sTmp3, 0, 256);

		fgets(sTmpLine, 255, pFile);
        if (ferror(pFile))
		{
			//FIXME: messagebox!
			return false; // an error occured
		}

		sscanf(sTmpLine, "%s", sTmp1);

		if ( !std::string(sTmp1).compare( std::string(OBJECTFILENAME) ) ) {
			sscanf(sTmpLine, "%s %s", sTmp2, sTmpRawfilename);
			m_sRawFileName.clear();
			size_t pos = sFilename.rfind("\\");
			sFilename.copy(sTmp3, pos);
			m_sRawFileName.append(std::string(sTmp3));
			m_sRawFileName.append(std::string("\\"));
			m_sRawFileName.append(sTmpRawfilename);
        }

		if ( !std::string(sTmp1).compare( std::string(RESOLUTION) ) ) {
			//	20050503			
			sscanf(sTmpLine, "%s %d %d %d", sTmp2, &m_anSamplingPoints[0], &m_anSamplingPoints[1], &m_anSamplingPoints[2]);

			//	FIXME	20050508	fix this differently!!!!
			//	ceiling to next power of two
			//	on Geforce 6800 non-pow-2 will reduce performace significantly 
			//m_anSamplingPoints[0] = 1 << (int)ceil(log((float)m_anSamplingPoints[0])/log(2.0));
			//m_anSamplingPoints[1] = 1 << (int)ceil(log((float)m_anSamplingPoints[1])/log(2.0));
			//m_anSamplingPoints[2] = 1 << (int)ceil(log((float)m_anSamplingPoints[2])/log(2.0));

			//	20050503	old (for compatibility)
			m_Width		=	m_anSamplingPoints[0];
			m_Height	=	m_anSamplingPoints[1];
			m_nSlices	=	m_anSamplingPoints[2];
			// round up to the next power of two (2^x)
            m_nFillslices = 1 << (int)ceil(log((float)m_nSlices)/log(2.0));
			m_nAlignedslices = m_nFillslices;
		}
		
		if ( !std::string(sTmp1).compare( std::string(SLICETHICKNESS) ) ) {
			//	have to read in doubles to prevent compiler warning
			//	old
			double x,y,z;
			sscanf(sTmpLine, "%s %lf %lf %lf", sTmp2, &x, &y, &z);

			m_afSamplingDistances[0]	=	(float) x;
			m_afSamplingDistances[1]	=	(float) y;
			m_afSamplingDistances[2]	=	(float) z;
		}

		if ( !std::string(sTmp1).compare( std::string(FORMAT) ) ) {
			sscanf(sTmpLine, "%s %s", sTmp2, sTmp3);
			if ( !std::string(sTmp3).compare( std::string("UCHAR") ) ) m_VolumeFormat = VOLUME_FORMAT_UNSIGNED_CHAR;
			if ( !std::string(sTmp3).compare( std::string("USHORT") ) ) m_VolumeFormat = VOLUME_FORMAT_UNSIGNED_SHORT;
			if ( !std::string(sTmp3).compare( std::string("FLOAT") ) ) m_VolumeFormat = VOLUME_FORMAT_FLOAT;
			if ( !std::string(sTmp3).compare( std::string("TWOFLOAT") ) ) m_VolumeFormat = VOLUME_FORMAT_TWOFLOAT;
		}

    } //while

	// close datfile
    fclose(pFile);

	//	FIXME	20050508 original
	//m_nVoxels = (m_anSamplingPoints[0] * m_anSamplingPoints[1] * m_anSamplingPoints[2]);

	//	FIXME	20050508 test
	m_nVoxels = (m_anSamplingPoints[0] * m_anSamplingPoints[1] * m_nFillslices);

	// get memory for data
	SAFE_FREE(m_pRawdata);

	switch(m_VolumeFormat) {
		case VOLUME_FORMAT_UNSIGNED_CHAR:		
			m_pRawdata = (char *) calloc(m_nVoxels, sizeof(char) ); 
			break;

		case VOLUME_FORMAT_UNSIGNED_SHORT:
			m_pRawdata = (char *) calloc(m_nVoxels, sizeof(unsigned short) ); 
			break;

		case VOLUME_FORMAT_FLOAT:
			m_pRawdata = (char *) calloc(m_nVoxels, sizeof(float) ); 
			break;

		case VOLUME_FORMAT_TWOFLOAT:
			//FIXME: get test data
			break;

		default:
			break;
	}

		// open rawfile
	if (NULL == (pFile = fopen(m_sRawFileName.c_str(), "r"))) {
		//FIXME: messagebox
		m_VolumeFormat = VOLUME_FORMAT_UNDEFINED;	
		return false;
	}

	// read raw data into memory, start adress is incl. offset
	switch(m_VolumeFormat) {
		case VOLUME_FORMAT_UNSIGNED_CHAR:		
			fread(m_pRawdata, sizeof(char), m_nVoxels, pFile);
			break;

		case VOLUME_FORMAT_UNSIGNED_SHORT:
			fread(m_pRawdata, sizeof(unsigned short), m_nVoxels, pFile);
			break;

		case VOLUME_FORMAT_FLOAT:
			fread(m_pRawdata, sizeof(float), m_nVoxels, pFile);
			break;

		case VOLUME_FORMAT_TWOFLOAT:
			//FIXME: get test data
			break;

		default:
			break;
	}	

	// close rawfile
    fclose(pFile);

	// remember .dat file name
	m_sDatafileName = sFilename;

	//success
	m_bVolumeLoaded = true;
	return true;
}


//------------------------------------------------------------------------------
void CVolume::FindMinMaxVoxelVals(void)
{
	if (!m_bVolumeLoaded) return;

	unsigned char*		pBufUChar	= (unsigned char*) m_pRawdata;
	unsigned short*		pBufUShort	= (unsigned short*) m_pRawdata;
	float*				pBufFloat	= (float*) m_pRawdata;

	switch(m_VolumeFormat) {
		case VOLUME_FORMAT_UNSIGNED_CHAR:		
			for (unsigned long i=0; i<m_nVoxels; i++) {
				if (pBufUChar[i] >= m_VoxelMaxVal)
					m_VoxelMaxVal = pBufUChar[i];					

				if (pBufUChar[i] <= m_VoxelMinVal)
					m_VoxelMinVal = pBufUChar[i];					
			}
			break;

		case VOLUME_FORMAT_UNSIGNED_SHORT:
			for (unsigned long i=0; i<m_nVoxels; i++) {
				if (pBufUShort[i] >= m_VoxelMaxVal)
					m_VoxelMaxVal = pBufUShort[i];

				if (pBufUShort[i] <= m_VoxelMinVal)
					m_VoxelMinVal = pBufUShort[i];
			}
			break;

		case VOLUME_FORMAT_FLOAT:
			for (unsigned long i=0; i<m_nVoxels; i++) {
				if (pBufFloat[i] >= m_VoxelMaxVal)
					m_VoxelMaxVal = pBufFloat[i];

				if (pBufFloat[i] <= m_VoxelMinVal)
					m_VoxelMinVal = pBufFloat[i];
			}
			break;

		case VOLUME_FORMAT_TWOFLOAT:
			//FIXME: get test data
			//for (unsigned long i=0; i<m_nVoxels; i+=2) {
			//	if (pBufFloat[i] >= m_VoxelMaxVal)
			//		m_VoxelMaxVal = pBufFloat[i];

			//	if (pBufFloat[i] <= m_VoxelMinVal)
			//		m_VoxelMinVal = pBufFloat[i];
			//}
			break;

		default:
			break;
	}

	m_VoxelDataRange = m_VoxelMaxVal - m_VoxelMinVal;
}


//------------------------------------------------------------------------------
void CVolume::ComputeHistogram(void)
{
	if (!m_bVolumeLoaded) return;

	unsigned char*		pBufUChar	= (unsigned char*) m_pRawdata;
	unsigned short*		pBufUShort	= (unsigned short*) m_pRawdata;
	float*				pBufFloat	= (float*) m_pRawdata;

	switch(m_VolumeFormat) {
		case VOLUME_FORMAT_UNSIGNED_CHAR:		
			for (unsigned long i=0; i<m_nVoxels; i++) {
				m_pHistogram->IncreaseCount( MapVoxelVal01((double)pBufUChar[i]) );
			}
			break;

		case VOLUME_FORMAT_UNSIGNED_SHORT:
			for (unsigned long i=0; i<m_nVoxels; i++) {
				m_pHistogram->IncreaseCount( MapVoxelVal01((double)pBufUShort[i]) );
			}
			break;

		case VOLUME_FORMAT_FLOAT:
			for (unsigned long i=0; i<m_nVoxels; i++) {
				m_pHistogram->IncreaseCount( MapVoxelVal01((double)pBufFloat[i]) );
			}
			break;

		case VOLUME_FORMAT_TWOFLOAT:
			//FIXME: get test data
			//for (unsigned long i=0; i<m_nVoxels; i+=2) {
			//	m_pHistogram->IncreaseCount( MapVoxelVal01((double)pBufFloat[i]) );
			//}
			break;

		default:
			break;
	}
}


//------------------------------------------------------------------------------
VolumeFormat CVolume::GetVolumeFormat(void)
{
	return m_VolumeFormat;
}



//-----------------------------------------------------------------------------
bool CVolume::VolumeLoaded(void)
{
	return m_bVolumeLoaded;
}


//-----------------------------------------------------------------------------
void CVolume::GetVoxelValInfo(double& minVal, double& maxVal)
{
	minVal = m_VoxelMinVal;
	maxVal = m_VoxelMaxVal;
}


//-----------------------------------------------------------------------------
void CVolume::FilterGradients(float* oldGradients, float* newGradients, const float* kernel, const unsigned int dim)
{	
	float			tGrad[3],
					nGrad[3];
	int				t			=	dim/2;
	int				iG			=	0,
					iK			=	0;
	//unsigned int vi;

	//// loop through entire volume
	for (int vz = 0; vz<m_nFillslices; vz++) {
		for (int vy = 0; vy<m_Height; vy++) {
			for (int vx = 0; vx<m_Width; vx++) {

				nGrad[0]=0.0f; nGrad[1]=0.0f; nGrad[2]=0.0f;

				for (int gz = -t; gz<=t; gz++) {
					for (int gy = -t; gy<=t; gy++) {
						for (int gx = -t; gx<=t; gx++) {

							GetFloatGradientAt( oldGradients, vx+gx, vy+gy, vz+gz, tGrad );

							iK			=	dim*(dim*(gz+t) + (gy+t)) + (gx+t);

							nGrad[0]	+=	tGrad[0] * kernel[iK];
							nGrad[1]	+=	tGrad[1] * kernel[iK];
							nGrad[2]	+=	tGrad[2] * kernel[iK];
						}
					}
				}

				newGradients[iG++] = nGrad[0];
				newGradients[iG++] = nGrad[1];
				newGradients[iG++] = nGrad[2];
			}
		}
	}
}


//-----------------------------------------------------------------------------
// The following source code for gradient computation is adapted from the project:
// "A Simple and Flexible Volume Rendering Framework for Graphics-Hardware-based Raycasting"
// with friendly permission of Simon Stegmaier, Magnus Strengert and Thomas Klein.
/*
 * Copyright (c) 2005  Institute for Visualization and Interactive
 * Systems, University of Stuttgart, Germany
 *
 * This source code is distributed as part of the single-pass volume
 * rendering project. Details about this project can be found on the
 * project web page at http://www.vis.uni-stuttgart.de/eng/research/
 * fields/current/spvolren. This file may be distributed, modified,
 * and used free of charge as long as this copyright notice is
 * included in its original form. Commercial use is strictly
 * prohibited.
 *
 * Filename: main.c
 * 
 */

#define SOBEL 1

void CVolume::ComputeGradients(float* pGradients, int* sizes)
{
    int i, j, k, dir, di, vdi, idz, idy, idx;
    float* gp	=	pGradients;	
	float invLength = 0.0f;
     
    static int weights[][3][3][3] = {
        {{{-1, -3, -1},
          {-3, -6, -3},
          {-1, -3, -1}},
         {{ 0,  0,  0},
          { 0,  0,  0},
          { 0,  0,  0}},
         {{ 1,  3,  1},
          { 3,  6,  3},
          { 1,  3,  1}}},
        {{{-1, -3, -1},
          { 0,  0,  0},
          { 1,  3,  1}},
         {{-3, -6, -3},
          { 0,  0,  0},
          { 3,  6,  3}},
         {{-1, -3, -1},
          { 0,  0,  0},
          { 1,  3,  1}}},
        {{{-1,  0,  1},
          {-3,  0,  3},
          {-1,  0,  1}},
         {{-3,  0,  3},
          {-6,  0,  6},
          {-3,  0,  3}},
         {{-1,  0,  1},
          {-3,  0,  3},
          {-1,  0,  1}}}
    };

    di = 0;
    vdi = 0;
	gp = pGradients;
    for (idz = 0; idz < m_nFillslices; idz++) {
        for (idy = 0; idy < m_Height; idy++) {
            for (idx = 0; idx < m_Width; idx++) {
#if SOBEL == 1
                if (idx > 0 && idx < m_Width - 1 &&
                    idy > 0 && idy < m_Height - 1 &&
                    idz > 0 && idz < m_nFillslices - 1) {

                    for (dir = 0; dir < 3; dir++) {
                        gp[dir] = 0.0;
                        for (i = -1; i < 2; i++) {
                            for (j = -1; j < 2; j++) {
                                for (k = -1; k < 2; k++) {
                                    gp[dir] += weights[dir][i + 1]
                                                           [j + 1]
                                                           [k + 1] *
                                               GetVolumeAt(idx + i,
															idy + j,
															idz + k);
                                }
                            }
                        }

						gp[dir] /= 2.0f * m_afSamplingDistances[dir];
                    }
                } else {
                    // X-direction
                    if (idx < 1) {
                        gp[0] = (GetVolumeAt(idx + 1, idy, idz) -
                                 GetVolumeAt(idx, idy, idz))/
                                (m_afSamplingDistances[0]);
                    } else {
                        gp[0] = (GetVolumeAt(idx, idy, idz) -
                                 GetVolumeAt(idx - 1, idy, idz))/
                                (m_afSamplingDistances[0]);
                    }

                    // Y-direction
                    if (idy < 1) {
                        gp[1] = (GetVolumeAt(idx, idy + 1, idz) -
                                   GetVolumeAt(idx, idy, idz))/
                                   (m_afSamplingDistances[1]);
                    } else {
                        gp[1] = (GetVolumeAt(idx, idy, idz) -
                                   GetVolumeAt(idx, idy - 1, idz))/
                                   (m_afSamplingDistances[1]);
                    }

                    // Z-direction
                    if (idz < 1) {
                        gp[2] = (GetVolumeAt(idx, idy, idz + 1) -
                                 GetVolumeAt(idx, idy, idz))/
                                (m_afSamplingDistances[2]);
                    } else {
                        gp[2] = (GetVolumeAt(idx, idy, idz) -
                                 GetVolumeAt(idx, idy, idz - 1))/
                                (m_afSamplingDistances[2]);
                    }
                }
#else
                // X-direction
                if (idx < 1) {
                    gp[0] = (GetVolumeAt(idx + 1, idy, idz) -
                             GetVolumeAt(idx, idy, idz))/
                            (m_afSamplingDistances[0]);
                } else if (idx > m_Width - 1) {
                    gp[0] = (GetVolumeAt(idx, idy, idz) -
                             GetVolumeAt(idx - 1, idy, idz))/
                            (m_afSamplingDistances[0]);
                } else {
                    gp[0] = (GetVolumeAt(idx + 1, idy, idz) -
                             GetVolumeAt(idx - 1, idy, idz))/
                            (2.0f * m_afSamplingDistances[0]);
                }

                // Y-direction
                if (idy < 1) {
                    gp[1] = (GetVolumeAt(idx, idy + 1, idz) -
                             GetVolumeAt(idx, idy, idz))/
                            (m_afSamplingDistances[1]);
                } else if (idy > m_Height - 1) {
                    gp[1] = (GetVolumeAt(idx, idy, idz) -
                             GetVolumeAt(idx, idy - 1, idz))/
                            (m_afSamplingDistances[1]);
                } else {
                    gp[1] = (GetVolumeAt(idx, idy + 1, idz) -
                             GetVolumeAt(idx, idy - 1, idz))/
                            (2.0f * m_afSamplingDistances[1]);
                }

                // Z-direction
                if (idz < 1) {
                    gp[2] = (GetVolumeAt(idx, idy, idz + 1) -
                             GetVolumeAt(idx, idy, idz))/
                            (m_afSamplingDistances[2]);
                } else if (idz > m_nFillslices - 1) {
                    gp[2] = (GetVolumeAt(idx, idy, idz) -
                             GetVolumeAt(idx, idy, idz - 1))/
                            (m_afSamplingDistances[2]);
                } else {
                    gp[2] = (GetVolumeAt(idx, idy, idz + 1) -
                             GetVolumeAt(idx, idy, idz - 1))/
                            (2.0f * m_afSamplingDistances[2]);
                }
#endif
                gp += 3;
            }
        }
    }
}


//-----------------------------------------------------------------------------
bool CVolume::LoadGradientsFromFile(std::string sFilename)
{
	USES_CONVERSION;

    FILE *fp;

	if (! (fp = fopen(sFilename.c_str(), "rb"))) {
		//return silently, will calculate gradients instead
        return false;
    }

	if (fread(m_pQuantizedGradientData, sizeof(char)*m_nVoxels*3, 1, fp) != 1) {

		std::string sMsg("Error reading ");
        sMsg.append(sFilename);
		sMsg.append( std::string(".\nGradients will be computed now. This may take a while...") );
		MessageBox( DXUTGetHWND(), A2W(sMsg.c_str()), DXUTGetWindowTitle(), MB_ICONEXCLAMATION|MB_OK );

		fclose(fp);
		return false;
    }

    fclose(fp);

	return true;
}


//-----------------------------------------------------------------------------
bool CVolume::SaveGradientsToFile(std::string sFilename)
{
	USES_CONVERSION;

    FILE *fp;

	if (! (fp = fopen(sFilename.c_str(), "wb"))) {

		std::string sMsg("Cannot open ");
		sMsg.append(sFilename);
		sMsg.append(std::string(" for writing!"));
		MessageBox( DXUTGetHWND(), A2W(sMsg.c_str()), DXUTGetWindowTitle(), MB_ICONEXCLAMATION|MB_OK );
    }

    if (fwrite(m_pQuantizedGradientData, 3*m_nVoxels*sizeof(char), 1, fp) != 1) {
		std::string sMsg("Writing gradients failed!");
		MessageBox( DXUTGetHWND(), A2W(sMsg.c_str()), DXUTGetWindowTitle(), MB_ICONEXCLAMATION|MB_OK );
		fclose(fp);
		return false;
    }

    fclose(fp);

	return true;
}


//-----------------------------------------------------------------------------
void CVolume::QuantizeGradients(float* pFloatGradients, BYTE* pQuantizedGradients)
{
	float*	gp			=	pFloatGradients;
	float	ngp[3];
	BYTE*	qgp			=	pQuantizedGradients;
	float	invLength;

	//	each component of gradients is mapped
	//	from [-1.0 .. 1.0] to [0 .. 255]

	for (unsigned i=0; i<m_nVoxels; i++) {
		// normalize
		invLength	=	1.0f / sqrt(gp[0]*gp[0] + gp[1]*gp[1] + gp[2]*gp[2]);
		ngp[0]		=	gp[0] * invLength;
		ngp[1]		=	gp[1] * invLength;
		ngp[2]		=	gp[2] * invLength;

		// quantize
		qgp[0]	= (BYTE) (127.95f * (ngp[0] + 1.0f)) ;
		qgp[1]	= (BYTE) (127.95f * (ngp[1] + 1.0f)) ;
		qgp[2]	= (BYTE) (127.95f * (ngp[2] + 1.0f)) ;


		// special case: no gradient
		if (	(qgp[0] == 0) && (qgp[1] == 0) && (qgp[2] == 0) ) {
			qgp[0] = qgp[1] = qgp[2] = (BYTE) 127.95f;
		}

		gp	+=	3;
		qgp	+=	3;
	}
}


//-----------------------------------------------------------------------------
 float CVolume::GetVolumeAt(const int x, const int y, const int z)
{
	using namespace clamping;

	if (!m_bVolumeLoaded) return 0.0;

	const int _x = CLAMP3(x, 0, m_Width-1);
	const int _y = CLAMP3(y, 0, m_Height-1);
	const int _z = CLAMP3(z, 0, m_nFillslices-1);

	unsigned char*		pBufUChar	= (unsigned char*)  m_pRawdata;
	unsigned short*		pBufUShort	= (unsigned short*) m_pRawdata;
	float*				pBufFloat	= (float*) m_pRawdata;

	switch(m_VolumeFormat) {
		case VOLUME_FORMAT_UNSIGNED_CHAR:
			return (float) pBufUChar[_x + _y*m_Width + _z*m_Width*m_Height];
			break;

		case VOLUME_FORMAT_UNSIGNED_SHORT:
			return (float) pBufUShort[_x + _y*m_Width + _z*m_Width*m_Height];
			break;

		case VOLUME_FORMAT_FLOAT:
			return (float) pBufFloat[_x + _y*m_Width + _z*m_Width*m_Height];
			break;

		case VOLUME_FORMAT_TWOFLOAT:
			return 0;
			break;

		default:
			break;
	}

	return 0.0;
}


//-----------------------------------------------------------------------------
 double CVolume::GetVolumeAt(const int x, const int y, const int z, const double mapMin, const double mapMax)
{
	using namespace clamping;

	if (!m_bVolumeLoaded) return 0;

	const int _x = CLAMP3(x, 0, m_Width-1);
	const int _y = CLAMP3(y, 0, m_Height-1);
	const int _z = CLAMP3(z, 0, m_nFillslices-1);

	unsigned char*		pBufUChar	= (unsigned char*)  m_pRawdata;
	unsigned short*		pBufUShort	= (unsigned short*) m_pRawdata;
	float*				pBufFloat	= (float*) m_pRawdata;
	double				mapRange	= fabs(mapMax - mapMin);
	double				mapScale	= 1.0 / mapRange;

	switch(m_VolumeFormat) {
		case VOLUME_FORMAT_UNSIGNED_CHAR:
			return LERP( mapMin, mapMax, 1.0 - MapVoxelVal01( (double) pBufUChar[_x + _y*m_Width + _z*m_Width*m_Height])) ;
			break;

		case VOLUME_FORMAT_UNSIGNED_SHORT:
			return LERP( mapMin, mapMax, 1.0 - MapVoxelVal01( (double) pBufUShort[_x + _y*m_Width + _z*m_Width*m_Height])) ;
			break;

		case VOLUME_FORMAT_FLOAT:
			return LERP( mapMin, mapMax, 1.0 - MapVoxelVal01( (double) pBufFloat[_x + _y*m_Width + _z*m_Width*m_Height])) ;
			break;

		case VOLUME_FORMAT_TWOFLOAT:
			return 0.0;
			break;

		default:
			break;
	}

	return 0.0;
}


//-----------------------------------------------------------------------------
 void CVolume::GetQuantizedGradientAt(const int x, const int y, const int z, BYTE* grad) const
{
	using namespace clamping;

	if (!m_pQuantizedGradientData)
	{
		grad[0] = grad[1] = grad[2] = 0;
		return;
	}

	const int _x = CLAMP3(x, 0, (m_Width-1));
	const int _y = CLAMP3(y, 0, (m_Height-1));
	const int _z = CLAMP3(z, 0, (m_nFillslices-1));

	const unsigned int index = (_x + _y*m_Width + _z*m_Width*m_Height)*3;
	grad[0] = m_pQuantizedGradientData[index];
	grad[1] = m_pQuantizedGradientData[index + 1];
	grad[2] = m_pQuantizedGradientData[index + 2];
}


//-----------------------------------------------------------------------------
 void CVolume::GetFloatGradientAt(const float* pGradients, const int x, const int y, const int z, float* grad)
{
	using namespace clamping;

	if (!pGradients) {
		grad[0] = grad[1] = grad[2] = 0.0f;
		return;
	}

	const int _x = CLAMP3(x, 0, m_Width-1);
	const int _y = CLAMP3(y, 0, m_Height-1);
	const int _z = CLAMP3(z, 0, m_nFillslices-1);

	const unsigned int index = (_x + _y*m_Width + _z*m_Width*m_Height)*3;
	grad[0] = pGradients[index];
	grad[1] = pGradients[index + 1];
	grad[2] = pGradients[index + 2];
}
