//
// Copyright (c) 2005  Institute for Visualization and Interactive Systems,
// University of Stuttgart, Germany
//
// This source code is distributed as part of the paper "A Generic Software Framework for 
// the GPU Volume Rendering Pipeline", submitted to the 10th International Fall Workshop
// VISION, MODELING, AND VISUALIZATION 2005, November 16 - 18, 2005 Erlangen, Germany/
//
// Details about this project can be found on the project web page at
// http://wwwvis.informatik.uni-stuttgart.de/~vollrajm/
// 
// This file may be distributed, modified, and used free of charge as long as this copyright 
// notice is included in its original form. Commercial use is strictly prohibited.
//
// Filename: Volume.h
//
// parts of this source code are adapted from source code provided with friendly permission
// by Tobias Schafhitzel

//-----------------------------------------------------------------------------
#pragma once
#ifndef VOLUME_H
#define VOLUME_H

//-----------------------------------------------------------------------------
#include <string>
#include "Globals.h"
#include "Histogram.h"

//-----------------------------------------------------------------------------
//forward declarations
class CHistogram;

//-----------------------------------------------------------------------------
#define SAFE_DELETE(p)		{ if(p) { delete (p);     (p)=NULL; } }
#define LERP(x,y,z)			((x) * (z) + (y) * (1.0 - (z)))

//string definitions (for .dat-file)
#define OBJECTFILENAME "ObjectFileName:"
#define RESOLUTION "Resolution:"
#define SLICETHICKNESS "SliceThickness:"
#define FORMAT "Format:"
#define USHORT "USHORT"
#define UCHAR "UCHAR"
#define OBJECTMODEL "ObjectModel:"
#define RGBA "RGBA"
#define DENSITY "DENSITY"
#define GRIDTYPE "GridType:"
#define EQUIDISTANT "EQUIDISTANT"

// raw data format of the volume
typedef enum { 
	VOLUME_FORMAT_UNDEFINED, 
	VOLUME_FORMAT_UNSIGNED_CHAR, 
	VOLUME_FORMAT_UNSIGNED_SHORT, 
	VOLUME_FORMAT_FLOAT, 
	VOLUME_FORMAT_TWOFLOAT }  VolumeFormat ;

typedef enum {
	VOLUME_GRADIENT_UNDEFINED,
	VOLUME_GRADIENT_CENTRAL, 
	VOLUME_GRADIENT_SOBEL }  VolumeGradientMethod ;


//-----------------------------------------------------------------------------
// reads the volume from a file, stores it in memory, computes gradients, filters
// gradients ...
class CVolume
{
public:
	CVolume(void);
	~CVolume(void);

	//public interface
public:
	bool					LoadVolumeFromFile(std::string sFilename);
	bool					VolumeLoaded (void);
	VolumeFormat			GetVolumeFormat (void);
	VolumeGradientMethod	GetGradientMethod() const {return m_VolumeGradientMethod;};
	int						GetNSamplingPoints(int* pDest) const {	pDest[0] = m_anSamplingPoints[0];
																	pDest[1] = m_anSamplingPoints[1];
																	pDest[2] = m_anSamplingPoints[2]; };
	int						GetNSamplingPointsX(void) const { return m_anSamplingPoints[0]; };
	int						GetNSamplingPointsY(void) const { return m_anSamplingPoints[1]; };
	int						GetNSamplingPointsZ(void) const { return m_anSamplingPoints[2]; };	
	CHistogram*				GetPHistogram(void) const { return m_pHistogram; }
	double					GetVolumeAt(const int x, const int y, const int z, const double mapMin, const double mapMax);	
	float					GetVolumeAt(const int x, const int y, const int z) ;
	void					GetQuantizedGradientAt(const int x, const int y, const int z, BYTE* grad) const;

	//member functions
protected:
	bool					DoLoadVolumeFromFile(std::string sFilename);
	void					FilterGradients(float* oldGradients, float* newGradients, const float* kernel, const unsigned int dim);
	bool					LoadGradientsFromFile(std::string sFilename);
	void					GetFloatGradientAt(const float* pGradients, const int x, const int y, const int z, float* grad);
	bool					SaveGradientsToFile(std::string sFilename);
	void					QuantizeGradients(float* pFloatGradients, BYTE* pQuantizedGradients);
	void					FindMinMaxVoxelVals(void);
	void					ComputeHistogram(void);
	void					ComputeGradients(float* pGradients, int* sizes);
	double					MapVoxelVal01(double val)
	{
		return ((val - m_VoxelMinVal) / m_VoxelDataRange);
	}
	void					GetVoxelValInfo(double& minVal, double& maxVal);


	//member variables
protected:
	std::string				m_sDatafileName;  // name of the .dat file
	std::string				m_sRawFileName;	// name of the .raw file
	int						m_Width;
	int						m_Height;
	int						m_nSlices;
	int						m_nFillslices; //FIXME: what is this??
	int						m_nAlignedslices;
	int						m_anSamplingPoints[3];
	VolumeFormat			m_VolumeFormat;
	float					m_afSamplingDistances[3];
	float					m_CubeLength;
	char*					m_pRawdata;
	BYTE*					m_pQuantizedGradientData;
	VolumeGradientMethod	m_VolumeGradientMethod;
	bool					m_bVolumeLoaded;
	unsigned long			m_nVoxels;
	double					m_VoxelMaxVal;
	double					m_VoxelMinVal;
	CHistogram*				m_pHistogram;
	double					m_VoxelDataRange;
};

#endif
