#include "stdafx.h"

#include "action.h"

#include "globals.h"

#include "skeletonizer.h"
#include "field.h"
#include "layer.h"
#include "parameters.h"
#include "deltaomega.h"
#include "utils.h"
#include "indexedorigins_cells.h"
#include "component.h"
#include "danielsson3d.h"
#include "colormap.h"
#include "histogram.h"
#include "logwriter.h"
#include "settings.h"

#include "boundarysegmentation.h"

using namespace std;



// ------------------------------------------------------------------------------------------

bool TAction_EftSymmetrical::isAvailable()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	if(!skel) return false;
	return skel->m_FtField.get() != 0;
}

void TAction_EftSymmetrical::perform_main()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	TTypedFieldInterface<TIndexedOrigins*>* Output = static_cast<TTypedFieldInterface<TIndexedOrigins*>*>( g_Mediator.getCurrentLayerSet()->newIoField( skel->m_SskelIndexField, skel->Omega->m_BoundaryIndexField, getName() ) );
	Output->clear();

	
	const vector<TIndexedOrigins*> & FT = skel->m_FtField->m_Values;
	static TIndexedOrigins_Set ios;
	for(unsigned int y=0; y<Output->getIndexField()->getMaxIndex(); y++)
	{
		const TCoord3 & p = Output->getIndexField()->vidx2coord(y);

		ios.clear();
		if(!skel->m_FtField->getIndexField()->vinside(p)) continue;

		const TIndexedOrigins * v = FT[skel->m_FtField->getIndexField()->vcoord2idx(p)];
		if(v == 0) continue;
		ios.merge( v->castConstVector() );
		TCoord3 np;
		for(np.x=p.x-1; np.x<=p.x+1; np.x++) // neighbors
		for(np.y=p.y-1; np.y<=p.y+1; np.y++)
		for(np.z=p.z-1; np.z<=p.z+1; np.z++)
		{
			if( skel->Omega->m_ForegroundIndexField->vinside(np) )
			{
				const unsigned int npIndex = skel->Omega->m_ForegroundIndexField->valuep(np);
				if( FT[npIndex] != 0  )
				{
					ios.merge( FT[npIndex]->castVector() );
				}
			}
		}

		TIndexedOrigins_Vector * efts = new TIndexedOrigins_Vector();
		efts->merge(&ios);
		Output->wvaluep(p) = efts;
	}

	Output->getLayer()->onFieldChanged();
}










// ------------------------------------------------------------------------------------------


bool TAction_ExtendCollapseToLoops::isAvailable()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	if(!skel) return false;
	return true		;
}


void TAction_ExtendCollapseToLoops::perform_main()
{
	TTypedFieldInterface<unsigned char>* Cskel = static_cast<TTypedFieldInterface<unsigned char>*>( g_Mediator.getCurrentLayerSet()->getField( "C-skel detector" ) );
	if(!Cskel) throw string("!Cskel");

	TTypedFieldInterface<float>* CskelMeasure = static_cast<TTypedFieldInterface<float>*>( g_Mediator.getCurrentLayerSet()->getField( "C-skel measure" ) );
	if(!CskelMeasure) throw string("!CskelMeasure");

	TTypedFieldInterface<unsigned char> * Loops = static_cast<TTypedFieldInterface<unsigned char>*>( g_Mediator.getCurrentLayerSet()->newField( TType::TYPE_UCHAR, Cskel->getIndexField(), "C-skel loops" ) );
	if(!Loops) throw string("!Loops");
	Loops->clear();


	unsigned int LoopId = 1;
	for(unsigned int x=0; x<Cskel->getMaxIndex(); x++)
	{
		if(Loops->vvaluex(x) == 0 && (Cskel->vvaluex(x) == 1 || Cskel->vvaluex(x) >= 3))
		{
			stack<unsigned int> Q;
			Q.push(x);
			float MaxLoopBranch = 0.0f;
			while(!Q.empty())
			{
				unsigned int idx = Q.top();
				Loops->wvaluex(idx) = LoopId;
				Q.pop();
				if( Cskel->vvaluex(idx) >= 3 )
				{
					*g_Log << "Loop: " << LoopId << ", branch: " << CskelMeasure->vvaluex(idx) << "\n";
					if(CskelMeasure->vvaluex(idx) > MaxLoopBranch)
					{ 
						MaxLoopBranch = CskelMeasure->vvaluex(idx);
					}
				}
				const TCoord3 p = Loops->getIndexField()->vidx2coord(idx);
				
				TCoord3 np;
				for(np.x=p.x-1; np.x<=p.x+1; np.x++) // neighbors
				for(np.y=p.y-1; np.y<=p.y+1; np.y++)
				for(np.z=p.z-1; np.z<=p.z+1; np.z++)
				{
					if(np == p) continue;
					const unsigned int npidx = Loops->getIndexField()->vcoord2idx(np);
					if(Loops->vvaluex(npidx) == 0 && (Cskel->vvaluex(npidx) == 1 || Cskel->vvaluex(npidx) >= 3))
					{
						Loops->wvaluex(npidx) = LoopId;
						Q.push( npidx );
					}
				}
			}

			// Assign max to loop
			*g_Log << "Assign value " << MaxLoopBranch << " to loop " << LoopId << "\n";
			for(unsigned int y=0; y<Cskel->getMaxIndex(); y++)
			{
				if(Loops->vvaluex(y) == LoopId && Cskel->vvaluex(y) == 1)
					CskelMeasure->wvaluex(y) = MaxLoopBranch;
			}
			LoopId++;
		}
	}
	Loops->getLayer()->onFieldChanged();
	CskelMeasure->getLayer()->onFieldChanged();


/*
	TTypedFieldInterface<unsigned char> * DetectLoops = static_cast<TTypedFieldInterface<unsigned char>*>( g_Mediator.getCurrentLayerSet()->getField( TAction_DetectLoops::getStaticName() ) );
	if(!DetectLoops) throw string("!DetectLoops");
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	unsigned int x;
	unsigned char LoopNumber = 1;
	unsigned int VoxelsWithLoopNumber;

	while(1) // Iterate over C-loops
	{
		
		vector< pair<TCoord3, shared_ptr<TIndexedOrigins_Set> > > ForegroundComponents;

		*g_Log << "- Determine foreground components for loop: " << LoopNumber << "\n";
		VoxelsWithLoopNumber = 0;
		for(x=0; x<DetectLoops->getMaxIndex(); x++)
		{
			if(
				DetectLoops->vvaluex(x) == LoopNumber

				&& skel->m_FilteredJunctions->vvaluex(x) == 3 
				)
			{
				const TCoord3 c = DetectLoops->getIndexField()->vidx2coord(x);
				
				
				shared_ptr<TIndexedOrigins_Vector> dilatedsps( new TIndexedOrigins_Vector() );
				skel->Omega->m_DeltaOmega->dilate(
					//c, 
					1.0f,
					skel->m_SpSetField->vvaluep(c)->m_PathSet->castConstVector(), 
					dilatedsps.get(), 
					false);

				TComponentSet Collapses(c);
				skel->Omega->m_DeltaOmega->computeComponentSetUsingCells(dilatedsps.get(), &Collapses);

				shared_ptr<TIndexedOrigins_Set> CombinedForegroundComponents( new TIndexedOrigins_Set() );
				skel->getCombinedForegroundComponents(Collapses, CombinedForegroundComponents);
				CombinedForegroundComponents->merge( dilatedsps.get() );
				
				ForegroundComponents.push_back( pair<TCoord3,shared_ptr<TIndexedOrigins_Set> >(c, CombinedForegroundComponents) );

				VoxelsWithLoopNumber++;	
			}
		}
		if(VoxelsWithLoopNumber == 0) break;

		// Look for largest foreground component
		unsigned int maxid = 0;
		for(x=0; x<ForegroundComponents.size(); x++)
		{
			if( ForegroundComponents[x].second->vertexcount() > ForegroundComponents[maxid].second->vertexcount() )
			{
				maxid = x;
			}
		}

		*g_Log << "Maximum component: " << maxid << "\n";

		// See if largest foreground component is disjunct with all other components
		shared_ptr<TIndexedOrigins_Set> maxfg = ForegroundComponents[maxid].second;
		bool OverlapsAllOthers = true;
		for(x=0; x<ForegroundComponents.size(); x++)
		{
			if(x == maxid) continue;

			if( maxfg->disjunct( ForegroundComponents[x].second.get() ) )
			{
				//*g_Log << "Component " << x << " overlaps max fg comp\n";
				OverlapsAllOthers = false;
				break;
			}
		}

		*g_Log << "Determine collapse for loop\n";
		shared_ptr<TIndexedOrigins_Vector> AssignThis;
		if(!OverlapsAllOthers)
		{
			// If not overlap, use reverse of max. fg comp. for whole of loop
			*g_Log << "Max fg comp is disjunct with all other fg components\n";
			TIndexedOrigins_Vector * v = new TIndexedOrigins_Vector();
			v->reserve( skel->Omega->m_BoundaryIndexField->getMaxIndex() - maxfg->vertexcount() );
			for(x=0; x<skel->Omega->m_BoundaryIndexField->getMaxIndex(); x++)
			{
				if(maxfg->find(x) == maxfg->end())
				{
					v->push_back(x);
				}
			}
			AssignThis.reset( v );
		}
		else
		{
			TIndexedOrigins_Vector * v = new TIndexedOrigins_Vector();
			v->merge( maxfg.get() );
			// If overlap, use max. fg comp. as collapse for whole loop
			AssignThis.reset( v );
		}
	
		

		*g_Log << "Assign largest collapse to all loop voxels\n";
		for(x=0; x<DetectLoops->getMaxIndex(); x++)
		{
			if(DetectLoops->vvaluex(x) == LoopNumber)
			{
				const TCoord3 c = DetectLoops->getIndexField()->vidx2coord(x);
				if(!skel->m_TotalCollapseField->valuep(c)) skel->m_TotalCollapseField->valuep(c) = new TIndexedOrigins_Vector();
				skel->m_TotalCollapseField->valuep(c)->castVector()->clear();

				skel->m_TotalCollapseField->valuep(c)->castVector()->merge( AssignThis.get() );

				skel->m_CurveSkeletonField->valuep(c) = 
					(g_Parameters.m_Equalize ? sqrtf((float)skel->m_TotalCollapseField->valuep(c)->vertexcount()) : (float)skel->m_TotalCollapseField->valuep(c)->vertexcount())
					/ skel->m_NormalizeBy;
			}
		}
		
		LoopNumber++;
	}
	skel->m_CurveSkeletonField->getLayer()->onFieldChanged();
	skel->m_TotalCollapseField->getLayer()->onFieldChanged();
*/

}




// ------------------------------------------------------------------------------------------


bool TAction_Reconstruction::isAvailable()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	if(!skel) return false;
	return 
	g_Mediator.getCurrentLayerSet()->exists( "spset fg radius" )
	&& g_Mediator.getCurrentLayerSet()->exists( "spset fg maximum length" )
	;
}

void TAction_Reconstruction::perform_main()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();

	const TTypedFieldInterface<float>* RadiusField = static_cast<TTypedFieldInterface<float>*>( g_Mediator.getCurrentLayerSet()->getField( "spset fg radius" ) );
	if(!RadiusField) throw string("!radius");

	const TTypedFieldInterface<float>* SskelMeasure = 0;
	{
		TTypedFieldInterface<float> * field = static_cast<TTypedFieldInterface<float>*>( g_Mediator.getCurrentLayerSet()->getField( "spset fg maximum length" ) );
		if(field) SskelMeasure = field;
	}
	{
		TTypedFieldInterface<float> * field = static_cast<TTypedFieldInterface<float>*>( skel->m_SkeletonField.get() );
		if(field) SskelMeasure = field;
	}
	if(!SskelMeasure) throw string("!SskelMeasure");


	TTypedFieldInterface<unsigned char>* Output = static_cast<TTypedFieldInterface<unsigned char>*>( g_Mediator.getCurrentLayerSet()->newField( TType::TYPE_UCHAR, skel->Omega->m_ForegroundIndexField, getName() ) );
	Output->clear();


	*g_Log << "Reconstruction using layer and filter \"" << SskelMeasure->getLayer()->getDisplayName() << "\"\n";
	// Using S-skel measure " << SskelMeasure->getLayer()->m_Filter->m_LowerValue << "\n";
	
	unsigned int k;
	*g_Log << "["; for(k=0; k<SskelMeasure->getIndexField()->getMaxIndex()/5000+1; k++) *g_Log << "."; *g_Log << "]\n"; 
	*g_Log << "[";

	for(k=0; k<SskelMeasure->getIndexField()->getMaxIndex(); k++)
	{
		const TCoord3 p = SskelMeasure->getIndexField()->vidx2coord(k);

		if(k % 5000 == 0) 
		{
			*g_Log << ".";
			g_Mediator.yield();
		}

		if( SskelMeasure->getLayer()->m_Filter->test(p) )
		{
			const float value = SskelMeasure->vvaluep(p);
			const float r = ceil( RadiusField->vvaluep(p) );
			const unsigned int rd = ceil(r);
			const float r2 = r*r;
			
			for(unsigned int a=p.x-rd; a<=p.x+rd; a++)
			{
				for(unsigned int b=p.y-rd; b<=p.y+rd; b++)
				{
					for(unsigned int c=p.z-rd; c<=p.z+rd; c++)
					{
						TCoord3 q(a,b,c);

						if( p.distance2(q) <= r2 )
						{
							Output->wvaluep(q) = 1;
							//Output->wvaluep(q) = 
							//	value > Output->vvaluep(q) ? 
							//		value : Output->vvaluep(q);
						}
					}
				}
			}
				
		}
	}
	*g_Log << "]\n";

	// Leave only boundary of reconstruction
	{
		for(k=0; k<Output->getIndexField()->getMaxIndex(); k++)
		{
			const TCoord3 p = Output->getIndexField()->vidx2coord(k);

			bool HasOutsideNeighbor = false;

			TCoord3 np;
			for(np.x=p.x-1; np.x<=p.x+1; np.x++) // neighbors
			for(np.y=p.y-1; np.y<=p.y+1; np.y++)
			for(np.z=p.z-1; np.z<=p.z+1; np.z++)
			{
				if(np == p) continue;

				if(Output->vvaluep(np) == 0)
				{
					HasOutsideNeighbor = true;
				}
			}		
			if(!HasOutsideNeighbor)
			{
				Output->wvaluep(p) = 2;
			}
		}

		for(k=0; k<Output->getIndexField()->getMaxIndex(); k++)
		{
			if(Output->wvaluex(k) == 2) Output->wvaluex(k) = 0;
		}
	}

	Output->getLayer()->onFieldChanged();
}












bool TAction_TFT::isAvailable()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	if(!skel) return false;
	return true;
}


void TAction_TFT::perform_main()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();

	double Tolerance;
	wxString s;
	do
	{
		s = ::wxGetTextFromUser("Tolerance", "Tolerance", "1.0");
	} 
	while( !s.ToDouble(&Tolerance) );
	
	/*
	if(Tolerance > 0.0f)
	{
		TDanielsson3dTolerance danielsson(skel->Omega->m_ForegroundIndexField.get(), skel->Omega->m_DeltaOmega->m_Boundary.get(), Tolerance);
		danielsson.perform();

		
		shared_ptr<TSparseIOField> NewField( new TSparseIOField(skel->Omega->m_ForegroundIndexField, skel->Omega->m_BoundaryIndexField) );
		for(unsigned int x=0; x<danielsson.m_Origins.size(); x++)
		{	
			if((danielsson.m_Origins)[x].size() == 0) continue;

			NewField->m_Values[x] = new TIndexedOrigins_Vector( (danielsson.m_Origins)[x] );
		}

		shared_ptr<TLayer> NewLayer( new TLayer(NewField, getName() + wxString::Format("%.3f", Tolerance).c_str() + " field") );
		g_Mediator.addLayer( NewLayer ); 
	}
	*/
}






// ------------------------------------------------------------------------------------------


bool TAction_ReverseTft::isAvailable()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	if(!skel) return false;
	return g_Mediator.getCurrentLayer()->m_Field->getType() == TType::TYPE_UCHAR;
//	return g_Mediator.getCurrentLayer()->m_Field->getIndexField() == skel->m_SskelIndexField;
}

void TAction_ReverseTft::perform_main()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();

	double Tolerance;
	wxString s;
	do
	{
		s = wxGetTextFromUser("Tolerance", "Tolerance", "1.0");
	} 
	while( !s.ToDouble(&Tolerance) );

//	const TLayer * FilterLayer = g_Mediator.getCurrentLayer().get();
//	*g_Log << "Using filter of layer \"" << FilterLayer->m_Name << "\"\n";
//	const TFloatFilter * Filter = static_cast<TFloatFilter*>( FilterLayer->m_Filter.get() );

	TTypedFieldInterface<unsigned char> * MaskField = static_cast<TTypedFieldInterface<unsigned char> *>( g_Mediator.getCurrentLayer()->m_Field.get() );
	TTypedFieldInterface<TShortestPathSet*> * SpSet = skel->m_SpSetField.get();
//	const TIndexMapper * SkelIndexField = skel->m_SskelIndexField.get();

	TTypedFieldInterface<TIndexedOrigins*>* Output = static_cast<TTypedFieldInterface<TIndexedOrigins*>*>( g_Mediator.getCurrentLayerSet()->newIoField( skel->Omega->m_BoundaryIndexField, skel->m_SskelIndexField, getName() ) );
	Output->clear();

	TDanielsson3d2 Danielsson(skel->Omega->m_ForegroundIndexField.get(), skel->m_SskelIndexField.get(), Tolerance);
	Danielsson.useMask(MaskField);
	Danielsson.perform();

	for(unsigned int x=0; x<Output->getIndexField()->getMaxIndex(); x++)
	{
		const TCoord3 & px = Output->getIndexField()->vidx2coord(x);
		const unsigned int idx = skel->Omega->m_ForegroundIndexField->vcoord2idx(px);
		TIndexedOrigins_Vector & vec = Danielsson.m_Origins[idx];
		Output->wvaluep(px) = new TIndexedOrigins_Vector( vec );
	}
	Output->getLayer()->m_RenderLines = true;
	Output->getLayer()->m_AllVoxels->setCheckedForRendering(false);
	Output->getLayer()->onFieldChanged();

/*
	map<unsigned int, unsigned int> DistMap;
	unsigned int Tolerance2 = 1;
	for(unsigned int x=0; x<Output->getIndexField()->getMaxIndex(); x++)
	{
		const TCoord3 & px = Output->getIndexField()->vidx2coord(x);
		Output->wvaluex(x) = new TIndexedOrigins_Vector(4);

//		const unsigned int idxnearest = SpSet->vvaluep(px)->m_Eft->at(0);
//		const unsigned int MaxDist = skel->Omega->m_BoundaryIndexField->vidx2coord(idxnearest).distance2(px) + 2;
		
//		DistMap.clear();

		unsigned int minidx = 0;
		float mindist = (numeric_limits<float>::max)();
		
		// For all the input voxels
		unsigned int y;
		for(y=0; y<SkelIndexField->getMaxIndex(); y++)
		{
			const TCoord3 & py = SkelIndexField->vidx2coord(y);
			if(MaskField->vvaluep(py) == 0) continue;

			const float d = px.distance(py);

			if(d < mindist) 
			{
				minidx = y;
				mindist = d;
			}
		}

		for(y=0; y<SkelIndexField->getMaxIndex(); y++)
		{
			const TCoord3 & py = SkelIndexField->vidx2coord(y);
			if(MaskField->vvaluep(py) == 0) continue;
			const float d = px.distance(py);
			
			if(d <= mindist + 1.0f)
			{
				Output->vvaluex(x)->castVector()->push_back(y);
			}
		}
	}
	Output->getLayer()->m_RenderLines = true;
	Output->getLayer()->onFieldChanged();
*/
}


// ------------------------------------------------------------------------------------------

bool TAction_DiscardDisconnectedComponents::isAvailable()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	if(!skel) return false;
	return true;
}


void TAction_DiscardDisconnectedComponents::perform_main()
{
	unsigned int x;
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();

	TField * Field = g_Mediator.getCurrentLayer()->m_Field.get();
	if(!Field) throw string("!Field");
	const TFilter * Filter = Field->getLayer()->m_Filter.get();
	const TMeasure * Measure = Field->getLayer()->m_FilterMeasure.get();
	if(!Filter) throw string("!Filter");

	TTypedFieldInterface<float>* Output = static_cast<TTypedFieldInterface<float>*>( g_Mediator.getCurrentLayerSet()->newField( TType::TYPE_FLOAT, Field->getIndexField(), "Discarded components") );
	Output->clear();
	vector<unsigned int> ClusterSizes;
	ClusterSizes.push_back(0);
	for(x=0; x<Output->getMaxIndex(); x++)
	{
		if( Output->vvaluex(x) != 0 ) continue;

		const TCoord3 p = Output->getIndexField()->vidx2coord(x);
		if( Filter->test(p) )
		{
			queue<TCoord3> S;
			S.push(p);
			ClusterSizes.push_back(0);
			while(!S.empty())
			{
				const TCoord3 p = S.front();
				S.pop();
				Output->wvaluep(p) = ClusterSizes.size()-1;
				ClusterSizes.back()++;

				TCoord3 np;
				for(np.x=p.x-1; np.x<=p.x+1; np.x++) // neighbors
				for(np.y=p.y-1; np.y<=p.y+1; np.y++)
				for(np.z=p.z-1; np.z<=p.z+1; np.z++)
				{
					if(p == np) continue;

					if( Output->getIndexField()->vinside(np) 
						&& Measure->toFloat(np) != 0.0f
						&& Filter->test(np) 
						&& Output->vvaluep(np) == 0 
						)
					{
						S.push(np);
						Output->wvaluep(np) = ClusterSizes.size()-1;
					}
				}
			}
		}
	}

	unsigned int LargestCluster = 0;
	for(x=0; x<ClusterSizes.size(); x++)
	{
		if(ClusterSizes[x] > ClusterSizes[LargestCluster]) LargestCluster = x; 
	}
	

	// Now remove all but the largest cluster, and assign final value to Output
	{
		for(x=0; x<Output->getMaxIndex(); x++)
		{
			if(Output->vvaluex(x) == LargestCluster)
			{
				Output->wvaluex(x) = Measure->toFloat( Output->getIndexField()->vidx2coord(x) );
			}
			else
			{
				Output->wvaluex(x) = 0;
			}
		}
	}


	Output->getLayer()->onFieldChanged();

}




// ------------------------------------------------------------------------------------------


bool TAction_ComputeSimplifiedEft::isAvailable()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	if(!skel) return false;
	return 
	skel->m_ForegroundSpSetField
	;
}

void TAction_ComputeSimplifiedEft::perform_main()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();

	TShortestPathSetField * SPSF = skel->m_ForegroundSpSetField.get();

	{
		double Tolerance = SPSF->m_SeftTau;
		wxString s;
		do
		{
			s = ::wxGetTextFromUser("Simplified EFT threshold", "Simplified EFT threshold", wxString::Format("%f",Tolerance).c_str() );
			if(s == "") return;
		} 
		while( !s.ToDouble(&Tolerance) );
		g_Parameters.m_SimplifiedEftThreshold = Tolerance;
		SPSF->m_SeftTau = Tolerance;
	}	

	for(unsigned int x=0; x<SPSF->getMaxIndex(); x++)
	{
		TShortestPathSet * sps = SPSF->vvaluex(x);
		if(sps == 0) continue;
		process(sps, g_Parameters.m_SimplifiedEftThreshold);
	}
	SPSF->getLayer()->onFieldChanged();
	
	// Update layer name
	{
		shared_ptr<TLayer> Layer = g_Mediator.getCurrentLayerSet()->getLayer("spset fg seft");
		Layer->m_Name = Layer->m_Field->getAdaptorName() + wxString::Format(" tau=%.2f", SPSF->m_SeftTau).ToStdString();
		Layer->onFieldChanged();
	}

	g_Mediator.updateUI();
}


void TAction_ComputeSimplifiedEft::process(TShortestPathSet * sps, float p_LowerValue)
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();

	unsigned int y,a;
	// Detect equivalence sets
	typedef set< set<unsigned int> > TEqSet;
	TEqSet eqset;
	{
		for(y=0; y<sps->m_Eft->size(); y++) 
		{
			set<unsigned int> s;
			unsigned int idx = (*sps->m_Eft)[y];
			//if(skel && idx >= skel->m_BoundaryIndexField->getMaxIndex()) continue;

			s.insert(idx);
			eqset.insert(s);
		}

		TEqSet::iterator it;
		TEqSet::iterator jt;
		for(y=0; y<sps->m_Paths.size(); y++) 
		{
			TShortestPath * sp = sps->m_Paths[y].get();
			if( sp->m_Length < p_LowerValue )
			{
				unsigned int idx1 = sp->m_Begin;
				unsigned int idx2 = sp->m_End;		 	
				for(it = eqset.begin(); it != eqset.end(); it++) if(it->find(idx1) != it->end()) break;
				for(jt = eqset.begin(); jt != eqset.end(); jt++) if(jt->find(idx2) != jt->end()) break;

				// Now merge jt to it
				if(it != jt)
				{
					set<unsigned int>::iterator kt;
					for(kt=jt->begin(); kt != jt->end(); kt++) 
					{
						unsigned int idx = *kt;
						set<unsigned int> & s = const_cast<set<unsigned int> & >(*it);
						s.insert(idx);
					}
		
					eqset.erase(jt);
				}
			}
		}
	}

	// Now for each pair of items in eqset a path should remain
	TShortestPathSet::TPaths NewPaths;
	{
		TEqSet::iterator it;
		TEqSet::iterator jt;

		// For each two equivalence sets 
		for(it = eqset.begin(); it != eqset.end(); it++)
		for(jt = it; jt != eqset.end(); jt++)
		{
			if(it == jt) continue;
		
			unsigned int idx1 = *it->begin(); 
			unsigned int idx2 = *jt->begin();
			for(a=0; a<sps->m_Paths.size(); a++)
			{
				shared_ptr<TShortestPath> sp = sps->m_Paths[a];
				if( 
					(sp->m_Begin == idx1 && sp->m_End == idx2)
					||
					(sp->m_Begin == idx2 && sp->m_End == idx1)
					)
				{
					NewPaths.push_back( sp );
				}
			}
			
		}
	}

	{
		shared_ptr<TIndexedOrigins_Vector> NewEft( new TIndexedOrigins_Vector() );
		set<unsigned int> EftSet;
		sps->m_Paths.clear();
		for(a=0; a<NewPaths.size(); a++)
		{
			EftSet.insert( NewPaths[a]->m_Begin  );
			EftSet.insert( NewPaths[a]->m_End );
			sps->m_Paths.push_back( NewPaths[a] );
		}
		set<unsigned int>::iterator it;
		for(it = EftSet.begin(); it != EftSet.end(); it++)
		{
			NewEft->push_back( *it );
		}
		sps->m_Seft = NewEft;
	}
}

// ------------------------------------------------------------------------------------------


bool TAction_ReverseEft::isAvailable()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	if(!skel) return false;
	return 
	skel->m_ForegroundSpSetField
	;
}

void TAction_ReverseEft::perform_main()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();

	// Parameter 1: boundary
	// Parameter 2: surface skeleton

	// Build mask first, consisting out of simplified skeleton
	shared_ptr<TSubIndexMapper> Mask( new TSubIndexMapper(skel->Omega->m_ForegroundIndexField.get()) );
	Mask->m_SubIndex.reserve( skel->m_ForegroundSurfaceSkeletonField->getMaxIndex() );
	unsigned int maskidx = 0;
	for(unsigned int x=0; x<skel->Omega->m_ForegroundIndexField->getMaxIndex(); x++)
	{
		const TCoord3 p = skel->Omega->m_ForegroundIndexField->vidx2coord(x);
		if(skel->m_ForegroundSurfaceSkeletonField->getLayer()->m_Filter->test(p))
		{
			Mask->m_SubIndex.push_back( x );
			Mask->m_Coord2IdxMap.insert( pair<TCoord3,unsigned int>(p,maskidx) );	
			maskidx++;
		}
	}
		


	shared_ptr<class TDanielsson3d2> danielsson( new TDanielsson3d2(skel->Omega->m_ForegroundIndexField.get(), Mask.get(), 0.1f ) );
	//TTypedFieldInterface<unsigned char>* Mask = static_cast<TTypedFieldInterface<unsigned char>*>( g_Mediator.getCurrentLayerSet()->newField( TType::TYPE_UCHAR, skel->Omega->m_BoundaryIndexField, skel->m_ForegroundSskelIndexField, getStaticName() ) );
	//Mask->clear();

	danielsson->perform();

	g_Mediator.getCurrentLayerSet()->deleteLayer( getStaticName() );
	TTypedFieldInterface<TIndexedOrigins_Vector*>* Output = static_cast<TTypedFieldInterface<TIndexedOrigins_Vector*>*>( g_Mediator.getCurrentLayerSet()->newIoField( skel->Omega->m_BoundaryIndexField, Mask, getStaticName() ) );
	Output->clear();
	for(unsigned int x=0; x<danielsson->m_Origins.size(); x++)
	{	
		const TCoord3 p = skel->Omega->m_ForegroundIndexField->vidx2coord(x);
		if( skel->Omega->m_BoundaryIndexField->vinside(p) )
		{
			Output->wvaluep(p) = new TIndexedOrigins_Vector( danielsson->m_Origins[x] );
		}
		//if((Omega->m_ForegroundFt->m_Origins)[x].size() == 0) continue;
	}
	Output->getLayer()->onFieldChanged();

}


// ------------------------------------------------------------------------------------------


bool TAction_ReverseEftAngle::isAvailable()
{
	return true;
}

void TAction_ReverseEftAngle::perform_main()
{
	// Old method: use reverse eft obtained using danielsson starting from skeleton
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	const TTypedFieldInterface<TShortestPathSet*> * SPSF = skel->m_ForegroundSpSetField.get();
	if(!SPSF) throw string("!SPSF");
	unsigned int x,y;

	TTypedFieldInterface<TIndexedOrigins_Vector*>* ReverseEft = static_cast<TTypedFieldInterface<TIndexedOrigins_Vector*>*>( g_Mediator.getCurrentLayerSet()->getField(TAction_ReverseEft::getStaticName() ) );
	if(!ReverseEft) throw string("!ReverseEft");

	TTypedFieldInterface<float>* Output = static_cast<TTypedFieldInterface<float>*>( g_Mediator.getCurrentLayerSet()->newField( TType::TYPE_FLOAT, skel->Omega->m_BoundaryIndexField, getStaticName() ) );
	Output->clear();

	for(x=0; x<ReverseEft->getMaxIndex(); x++)
	{
		const TCoord3 q = ReverseEft->getIndexField()->vidx2coord(x);
		const TIndexedOrigins_Vector* vec = ReverseEft->vvaluep(q);	

		float mindot = 1.0f;
		for(y=0; y<vec->size(); y++)
		{
			const TCoord3 p = ReverseEft->getIndexToField()->vidx2coord( (*vec)[y] );


			int a,b,c;
			const int W = 3;
			for(a=p.x-W; a<=p.x+W; a++)
			for(b=p.y-W; b<=p.y+W; b++)
			for(c=p.z-W; c<=p.z+W; c++)
			{
				const TCoord3 p2(a,b,c);
				if( skel->m_ForegroundSurfaceSkeletonField->getLayer()->m_Filter->test(p2)
					&& (p.distance2(p2) <= W*W)
					)
				{
					const TShortestPathSet * sps = SPSF->vvaluep(p2);
					if(sps == 0) continue;
					shared_ptr<TPlane> plane = sps->getThreeLongestPaths()[0]->m_LocalSheet;
					
					const float dist = fabs( plane->distance2point(q.toPoint()) );
					if(dist < mindot) mindot = dist;
				}
			}
			
				
				/*
				if(!plane) throw string("!plane");
				TVector3 v = q.toVector();
				v.subtract(sps->m_SourceVoxel.toVector());
				v.normalize();
				const float dot = fabs( plane->m_Normal.dot(v) );
				if(dot < mindot) mindot = dot;
				*/
			//}
			//const TIndexedOrigins_Vector* vec2 = sps->m_Seft;
		}
		if(mindot == 0.0f) mindot = 0.001f;
		Output->wvaluep(q) = mindot;
	}

	Output->getLayer()->onFieldChanged();
}




// ------------------------------------------------------------------------------------------


bool TAction_ReverseEftAngle2::isAvailable()
{
	return true;
}

void TAction_ReverseEftAngle2::perform_main()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();

	const TTypedFieldInterface<TShortestPathSet*> * SPSF = skel->m_ForegroundSpSetField.get();
	if(!SPSF) throw string("!SPSF");

	TTypedFieldInterface<TIndexedOrigins_Vector*>* Reft = static_cast<TTypedFieldInterface<TIndexedOrigins_Vector*>*>( g_Mediator.getCurrentLayerSet()->getField( "Inverse Eft unfilled" ) );
	if(!Reft) throw string("!Reft");

	// New method: for each boundary voxel, search in neighborhood 
//	TTypedFieldInterface<float>* Output = static_cast<TTypedFieldInterface<float>*>( g_Mediator.getCurrentLayerSet()->newField( TType::TYPE_FLOAT, skel->Omega->m_BoundaryIndexField, getStaticName() ) );
//	Output->clear();

	TTypedFieldInterface<TIndexedOrigins_Vector*>* Reftc = static_cast<TTypedFieldInterface<TIndexedOrigins_Vector*>*>( g_Mediator.getCurrentLayerSet()->newIoField( skel->Omega->m_BoundaryIndexField, skel->m_ForegroundSskelIndexField, "Reft complete" ) );
	Reftc->clear();

	unsigned int x,y;
	for(x=0; x<Reftc->getMaxIndex(); x++)
	{
		TDeltaOmega::TDistancePropagationParameters par;
		par.m_Start = x;
		set<unsigned int> Seeds;
		par.m_StopDistance = 5.0f;
		par.m_OutputTargets = &Seeds;
		par.m_TargetFilter = Reft->getLayer()->m_Filter.get();
		skel->Omega->m_DeltaOmega->distancePropagation( par );
		set<unsigned int>::iterator it;

		if( Reftc->wvaluex(x) == 0 ) Reftc->wvaluex(x) = new TIndexedOrigins_Vector();
		set<unsigned int> Gather;
		for(it = Seeds.begin(); it != Seeds.end(); it++)
		{
			const TIndexedOrigins_Vector * vec = Reft->vvaluex(*it);
			for(y=0; y<vec->size(); y++)
			{
				Gather.insert( (*vec)[y] );
			}
		}
		for(it = Gather.begin(); it != Gather.end(); it++)
		{
			Reftc->wvaluex(x)->push_back(*it);
		}

	}
	Reftc->getLayer()->onFieldChanged();



	TTypedFieldInterface<float>* Output = static_cast<TTypedFieldInterface<float>*>( g_Mediator.getCurrentLayerSet()->newField( TType::TYPE_FLOAT, skel->Omega->m_BoundaryIndexField, getStaticName() ) );
	Output->clear();

	for(x=0; x<Reftc->getMaxIndex(); x++)
	{
		const TCoord3 q = Reftc->getIndexField()->vidx2coord(x);
		const TIndexedOrigins_Vector* vec = Reftc->vvaluep(q);	
		if(vec == 0) continue;

		float mindot = 1.0f;
		float maxdot = 0.0001f;
		for(unsigned int y=0; y<vec->size(); y++)
		{
			const TCoord3 p = Reftc->getIndexToField()->vidx2coord( (*vec)[y] );

			const TShortestPathSet * sps = SPSF->vvaluep(p);
			if(!sps) throw string("!sps");
			
			shared_ptr<TShortestPath> sp = sps->getThreeLongestPaths()[0];
			TVector3 va = skel->Omega->m_BoundaryIndexField->vidx2coord( sp->m_Begin ).toVector();
			va.subtract(sps->m_SourceVoxel.toVector());
			va.normalize();
			TVector3 vb = skel->Omega->m_BoundaryIndexField->vidx2coord( sp->m_End ).toVector();
			vb.subtract(sps->m_SourceVoxel.toVector());
			vb.normalize();
			if( va.dot(vb) < -0.8f ) continue;

			TVector3 bisector = va;
			bisector.add(vb);
			bisector.scale(0.5f);

			//if(bisector.norm2() < 0.00001f) continue;
			bisector.normalize();

			TVector3 v = q.toVector();
			v.subtract(sps->m_SourceVoxel.toVector());
			v.normalize();
			const float dot = bisector.dot(v);

			if(dot > maxdot) maxdot = dot;
		
			/*
			shared_ptr<TPlane> plane = sp->m_LocalSheet;
			if(!plane) throw string("!plane");

			TVector3 v = q.toVector();
			v.subtract(sps->m_SourceVoxel.toVector());
			v.normalize();
			const float dot = fabs( plane->m_Normal.dot(v) );
			if(dot < mindot) mindot = dot;
			*/
			//const TIndexedOrigins_Vector* vec2 = sps->m_Seft;
		}
		//if(mindot == 0.0f) mindot = 0.001f;
		Output->wvaluep(q) = maxdot;
	}

	Output->getLayer()->onFieldChanged();
}



// ------------------------------------------------------------------------------------------


bool TAction_DistanceToFeatureCollection::isAvailable()
{
	return true;
}

void TAction_DistanceToFeatureCollection::computeDistanceField(bool p_FgBg)
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();

	/*
	double FgThreshold  = max( (skel->m_ForegroundSurfaceSkeletonField->getLayer()->m_Filter->m_LowerValue - 5.0f) / 2.0f, 4.0f );
	{
		{
			wxString s = "";
			do
			{
				s = ::wxGetTextFromUser("FG Threshold", "FG Threshold", wxString::Format("%f",FgThreshold).c_str() );
				if(s == "") return;
			} 
			while( !s.ToDouble(&FgThreshold) );
		}
	}
	*/

	TTypedFieldInterface<TShortestPathSet*> * SPSF = 0;
	// Simplify the skeleton
	{
		TAction_PruneSpSetForBoundarySegmentation SimplifiedEft;
		SimplifiedEft.m_FgBg = p_FgBg;
		SimplifiedEft.m_ShortName = string("spset pruned ") + (p_FgBg  ? "fg" : "bg");
		const bool JustAdded = ! g_Mediator.getCurrentLayerSet()->exists( SimplifiedEft.m_ShortName ); 
		SimplifiedEft.perform();
		SPSF =  static_cast<TTypedFieldInterface<TShortestPathSet*> *>(g_Mediator.getCurrentLayerSet()->getField( SimplifiedEft.m_ShortName ) );
		if(! SPSF) throw string("!SPSF");
		//if(JustAdded) skel->addShortestPathSetAdaptorField(SPSF, SimplifiedEft.m_ShortName);
	}

	TTrueFilter Truefilter;

	
	const bool EXTENDSEEDS = true;

	// Compute seeds
	TTypedFieldInterface<unsigned int>* Seeds = static_cast<TTypedFieldInterface<unsigned int>*>( g_Mediator.getCurrentLayerSet()->newField( TType::TYPE_UINT, skel->Omega->m_BoundaryIndexField, string("Seeds ") + (p_FgBg  ? "fg" : "bg")) );
	set<unsigned int> SeedSet;
	{
		const TFilter * Filter = p_FgBg ? 
								skel->m_ForegroundSurfaceSkeletonField->getLayer()->m_Filter.get()
								: skel->m_BackgroundSurfaceSkeletonField->getLayer()->m_Filter.get();
		Seeds->clear();
		for(unsigned int x=0; x<SPSF->getMaxIndex(); x++)
		{
			TShortestPathSet * s = SPSF->vvaluex(x);
			if(s == 0) continue;

			if(!Filter->test(s->getMaxLength())) continue;

			if(s->m_Paths.size() != 1) continue;
			for(unsigned int y=0; y<s->m_Paths.size(); y++)
			{
				if(!Filter->test(s->m_Paths[y]->m_Length)) continue;

				const unsigned int begin = s->m_Paths[y]->m_Begin;
				const unsigned int end = s->m_Paths[y]->m_End;
				if(begin < skel->m_BoundaryIndexField->getMaxIndex()) 
				{
					if(EXTENDSEEDS)
					{
						set<unsigned int> Output;
						TDeltaOmega::TDistancePropagationParameters par;
						par.m_Start = begin;
						par.m_StopDistance = s->getRadius() / 3.0f;
						par.m_TargetFilter = &Truefilter;
						par.m_OutputTargets = &Output;
						skel->Omega->m_DeltaOmega->distancePropagation(par);
						
						for(set<unsigned int>::iterator it = Output.begin(); it != Output.end(); it++)
						{
							Seeds->wvaluex(*it) = 1;
							SeedSet.insert(*it);
						}
					}
					else
					{
						Seeds->wvaluex(begin) = 1;
						SeedSet.insert(begin);
					}
				}
				
				if(end < skel->m_BoundaryIndexField->getMaxIndex()) 
				{
					if(EXTENDSEEDS)
					{
						set<unsigned int> Output;
						TDeltaOmega::TDistancePropagationParameters par;
						par.m_Start = end;
						par.m_StopDistance = s->getRadius() / 3.0f;
						par.m_TargetFilter = &Truefilter;
						par.m_OutputTargets = &Output;
						skel->Omega->m_DeltaOmega->distancePropagation(par);
						
						for(set<unsigned int>::iterator it = Output.begin(); it != Output.end(); it++)
						{
							Seeds->wvaluex(*it) = 1;
							SeedSet.insert(*it);
						}
					}
					else
					{
						Seeds->wvaluex(end) = 1;
						SeedSet.insert(end);
					}
				}
			}
		}	
		Seeds->getLayer()->onFieldChanged();
	}

	TTypedFieldInterface<float>* FgDistanceField = static_cast<TTypedFieldInterface<float>*>( g_Mediator.getCurrentLayerSet()->newField( TType::TYPE_FLOAT, skel->Omega->m_BoundaryIndexField, string("Distance to seeds (") + (p_FgBg ? "fg" : "bg") + ")" ) );
	FgDistanceField->clear();

	TDeltaOmega::TDistancePropagationParameters par;
	par.m_OutputDistance = FgDistanceField;
	par.m_StartSet = &SeedSet;
	skel->Omega->m_DeltaOmega->distancePropagation(par);
	FgDistanceField->getLayer()->onFieldChanged();
	//FgDistanceField->getLayer()->m_Filter->m_LowerValue = FgThreshold;
	FgDistanceField->getLayer()->m_Filter->m_LowerValue = 0.0f;
//	FgDistanceField->getLayer()->m_ColorMap->m_UpperValue = 6.0f;

}




void TAction_DistanceToFeatureCollection::computeDistanceFieldSelectedLayer()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();

	bool p_FgBg;
	if(g_Mediator.getCurrentLayer()->m_Field->getIndexField() == skel->m_ForegroundSskelIndexField)
	{
		p_FgBg = true;
	}
	else if(g_Mediator.getCurrentLayer()->m_Field->getIndexField() == skel->m_BackgroundSskelIndexField)
	{
		p_FgBg = false;
	}
	else
	{
		throw string("Invalid indexfield");
	}

	TTypedFieldInterface<TShortestPathSet*> * SPSF = 0;
	// Simplify the skeleton
	{
		TAction_PruneSpSetForBoundarySegmentation SimplifiedEft;
		SimplifiedEft.m_FgBg = p_FgBg;
		SimplifiedEft.m_ShortName = string("spset pruned ") + (p_FgBg  ? "fg" : "bg");
		const bool JustAdded = ! g_Mediator.getCurrentLayerSet()->exists( SimplifiedEft.m_ShortName ); 
		SimplifiedEft.perform();
		SPSF =  static_cast<TTypedFieldInterface<TShortestPathSet*> *>(g_Mediator.getCurrentLayerSet()->getField( SimplifiedEft.m_ShortName ) );
		if(! SPSF) throw string("!SPSF");
		//if(JustAdded) skel->addShortestPathSetAdaptorField(SPSF, SimplifiedEft.m_ShortName);
	}

	TTrueFilter Truefilter;

	
	const bool EXTENDSEEDS = true;

	// Compute seeds
	TTypedFieldInterface<unsigned int>* Seeds = static_cast<TTypedFieldInterface<unsigned int>*>( g_Mediator.getCurrentLayerSet()->newField( TType::TYPE_UINT, skel->Omega->m_BoundaryIndexField, string("Seeds ") + (p_FgBg  ? "fg" : "bg")) );
	set<unsigned int> SeedSet;
	{
//		const TFilter * Filter = p_FgBg ? 
//								skel->m_ForegroundSurfaceSkeletonField->getLayer()->m_Filter.get()
///								: skel->m_BackgroundSurfaceSkeletonField->getLayer()->m_Filter.get();
		const TFilter * Filter = g_Mediator.getCurrentLayer()->m_Filter.get();

		Seeds->clear();
		for(unsigned int x=0; x<SPSF->getMaxIndex(); x++)
		{
			if(! Filter->testx(x)) continue;
			TShortestPathSet * s = SPSF->vvaluex(x);
			if(s == 0) continue;
			if(s->m_Paths.size() == 0) continue;
			shared_ptr<TShortestPath> sp = s->getThreeLongestPaths()[0];

			//if(!Filter->test(s->getMaxLength())) continue;
			//if(s->m_Paths.size() != 1) continue;
			//for(unsigned int y=0; y<s->m_Paths.size(); y++)
			{
			//	if(!Filter->test(s->m_Paths[y]->m_Length)) continue;

				//const unsigned int begin = s->m_Paths[y]->m_Begin;
				//const unsigned int end = s->m_Paths[y]->m_End;
				const unsigned int begin = sp->m_Begin;
				const unsigned int end = sp->m_End;

				if(begin < skel->m_BoundaryIndexField->getMaxIndex()) 
				{
					if(EXTENDSEEDS)
					{
						set<unsigned int> Output;
						TDeltaOmega::TDistancePropagationParameters par;
						par.m_Start = begin;
						par.m_StopDistance = s->getRadius() / 3.0f;
						par.m_TargetFilter = &Truefilter;
						par.m_OutputTargets = &Output;
						skel->Omega->m_DeltaOmega->distancePropagation(par);
						
						for(set<unsigned int>::iterator it = Output.begin(); it != Output.end(); it++)
						{
							Seeds->wvaluex(*it) = 1;
							SeedSet.insert(*it);
						}
					}
					else
					{
						Seeds->wvaluex(begin) = 1;
						SeedSet.insert(begin);
					}
				}
				
				if(end < skel->m_BoundaryIndexField->getMaxIndex()) 
				{
					if(EXTENDSEEDS)
					{
						set<unsigned int> Output;
						TDeltaOmega::TDistancePropagationParameters par;
						par.m_Start = end;
						par.m_StopDistance = s->getRadius() / 3.0f;
						par.m_TargetFilter = &Truefilter;
						par.m_OutputTargets = &Output;
						skel->Omega->m_DeltaOmega->distancePropagation(par);
						
						for(set<unsigned int>::iterator it = Output.begin(); it != Output.end(); it++)
						{
							Seeds->wvaluex(*it) = 1;
							SeedSet.insert(*it);
						}
					}
					else
					{
						Seeds->wvaluex(end) = 1;
						SeedSet.insert(end);
					}
				}
			}
		}	
		Seeds->getLayer()->onFieldChanged();
	}

	TTypedFieldInterface<float>* FgDistanceField = static_cast<TTypedFieldInterface<float>*>( g_Mediator.getCurrentLayerSet()->newField( TType::TYPE_FLOAT, skel->Omega->m_BoundaryIndexField, string("Distance to seeds (") + (p_FgBg ? "fg" : "bg") + ")" ) );
	FgDistanceField->clear();

	TDeltaOmega::TDistancePropagationParameters par;
	par.m_OutputDistance = FgDistanceField;
	par.m_StartSet = &SeedSet;
	skel->Omega->m_DeltaOmega->distancePropagation(par);
	FgDistanceField->getLayer()->onFieldChanged();
	//FgDistanceField->getLayer()->m_Filter->m_LowerValue = FgThreshold;
	FgDistanceField->getLayer()->m_Filter->m_LowerValue = 0.0f;
//	FgDistanceField->getLayer()->m_ColorMap->m_UpperValue = 6.0f;

}



void TAction_DistanceToFeatureCollection::perform_main()
{
	computeDistanceFieldSelectedLayer();
//	computeDistanceField(true);
//	if( g_Mediator.getSkeletonizer()->m_BackgroundSurfaceSkeletonField )
//	{
//		computeDistanceField(false);
//	}
}



void TAction_Thin::perform_main()
{
	shared_ptr<TLayer> Layer = g_Mediator.getCurrentLayer();
	shared_ptr<TFilter> Filter = Layer->m_Filter;;
	shared_ptr<TField> Field = Layer->m_Field;
	
	char tmp[3][3] = { 0, 1, 0,
					   0, 1, 1,
					   0, 0, 0 };

	for(unsigned int x=0; x<Field->getMaxIndex(); x++)
	{
		const TCoord3 p = Field->getIndexField()->vidx2coord(x);
		
		TCoord3 np;
		for(np.x=p.x-1; np.x<=p.x+1; np.x++) // neighbors
		for(np.y=p.y-1; np.y<=p.y+1; np.y++)
		for(np.z=p.z-1; np.z<=p.z+1; np.z++)
		{
//			if( 
//			Field->getIndexField()->vinside(
		}
	}
}

