#include "stdafx.h"

#include "camera2.h"

using namespace std;


TVector3 TObservingCamera::getPos()
{
	TVector3 pos;
	pos.x = cos(m_Pitch) * cos(m_Heading);
	pos.y = sin(m_Pitch);
	pos.z = cos(m_Pitch) * sin(m_Heading);
	pos.scale(m_Zoom);
	pos.add(m_Middle);
	return pos;
}

void TObservingCamera::setMatrices()
{
	TVector3 pos;
	pos.x = cos(m_Pitch) * cos(m_Heading);
	pos.y = sin(m_Pitch);
	pos.z = cos(m_Pitch) * sin(m_Heading);
	pos.scale(m_Zoom);
	pos.add(m_Middle);

	gluLookAt( 
		pos.x,
		pos.y,
		pos.z,
		m_Middle.x,
		m_Middle.y,
		m_Middle.z,
		0.0f,
		1.0f,
		0.0f
		);

	if(m_ProjectionMode == PROJECTIONMODE_ORTHOGONAL) // zoom by scaling scene
	{
//		glScalef(1.0f/m_Zoom, 1.0f/m_Zoom, 1.0f/m_Zoom);
	}
}


void TObservingCamera::move(float x, float y, bool p_Left, bool p_Middle, bool p_Right)
{
	if(p_Left)
	{
		m_Heading += x;
	
		if(m_Heading > 2*PI) m_Heading -= (float)(2*PI);
		else if(m_Heading < 0.0f ) m_Heading += (float)(2*PI);

		m_Pitch += y;

		if(m_Pitch < -0.49f * PI ) m_Pitch = (float)(-0.49f * PI);
		else if(m_Pitch > 0.49f * PI ) m_Pitch = (float)(0.49f * PI);
	
	}
	else if(p_Middle)
	{
		m_Zoom -= y;
		if(m_Zoom < 0.0f) 
		{
			m_Zoom = 0.0f;
		}
	}
}

void TObservingCamera::snapToAxis(int p_Axis)
{
	if(p_Axis == 0)
	{
		if(m_Heading < 0.5f*PI || m_Heading >= 1.5f*PI)
			m_Heading = 0.0f;
		else
			m_Heading = (float)PI;
	}
	else if(p_Axis == 1)
	{
		if(m_Pitch <= 0)
			m_Pitch = (float)(-0.49f*PI);
		else
			m_Pitch = (float)(0.49f*PI);
	}
	else if(p_Axis == 2)
	{
		if(m_Heading < 1.0f*PI && m_Heading >= 0.0f*PI)
			m_Heading = (float)(0.5f*PI);
		else
			m_Heading = (float)(1.5f*PI);
	}
}

void TObservingCamera::setProjectionMatrix(int w,int h)
{
	if(m_ProjectionMode == TCamera::PROJECTIONMODE_PERSPECTIVE)
	{
		gluPerspective(40, (float)w/(float)h, 0.2f, 100.0f);
//		gluPerspective(60, 1.0f, 0.2f, 100.0f);
	}
	else if(m_ProjectionMode == TCamera::PROJECTIONMODE_ORTHOGONAL)
	{
		float ratio = (float)w/(float)h;
		float zoom = m_Zoom * 0.1f;
		glOrtho(-1.0f*zoom, 1.0f*zoom, -1.0f*zoom/ratio, 1.0f*zoom/ratio, 0.1f, 100.0f);
//		glOrtho(-1.0f, 1.0f, -1.0f, 1.0f, 0.1f, 100.0f);
	}
	else throw std::string("Unknown projection mode");
}

void TObservingCamera::reset()
{
	m_Heading = m_InitialHeading;
	m_Pitch = m_InitialPitch;
	m_Zoom = m_InitialZoom;
}









TRoamingCamera::TRoamingCamera(TPoint3 p_InitialPosition, float p_InitialHeading, float p_InitialPitch)
: m_Position(p_InitialPosition)
, m_InitialPosition(p_InitialPosition)
, m_InitialHeading(p_InitialHeading)
, m_InitialPitch(p_InitialPitch)
{
	reset();
	calculateDirection();
};


void TRoamingCamera::setMatrices()
{
	TVector3 ref = getLookAt();

	gluLookAt( 
		m_Position.x,
		m_Position.y,
		m_Position.z,
		ref.x,
		ref.y,
		ref.z,
		0.0f,
		1.0f,
		0.0f
		);
	
	if(m_ProjectionMode == PROJECTIONMODE_ORTHOGONAL) // zoom by scaling scene
	{
		glScalef(1.0f/m_Zoom, 1.0f/m_Zoom, 1.0f/m_Zoom);
	}
}

TVector3 TRoamingCamera::getPos()
{
	return m_Position;
}

TVector3 TRoamingCamera::getLookAt()
{
	TVector3 ref = m_Position;
	ref.add(m_Direction);
	return ref;
}

void TRoamingCamera::calculateDirection()
{
	m_Direction.x = cos(m_Pitch) * cos(m_Heading);
	m_Direction.y = sin(m_Pitch);
	m_Direction.z = cos(m_Pitch) * sin(m_Heading);
	m_Direction.normalize();
}

void TRoamingCamera::move(float p_x, float p_y, bool p_Left, bool p_Middle, bool p_Right)
{
	const float x = p_x;
	const float y = p_y;
	const float PanFactor = 2.0f;

	if(p_Left) 
	{
		// Change orientation
		m_Heading += x;
		if(m_Heading > 2*PI) m_Heading -= (float)(2*PI);
		else if(m_Heading < 0.0f ) m_Heading += (float)(2*PI);

		m_Pitch += -y;
		if(m_Pitch < -0.49f * PI ) m_Pitch = (float)(-0.49f * PI);
		else if(m_Pitch > 0.49f * PI ) m_Pitch = (float)(0.49f * PI);

	}
	else if(p_Middle)
	{
		// Move forward
		TVector3 v = m_Direction; 
		v.scale(y);
		m_Position.add(v);

		// Maintain m_Zoom for orthogonal projection
		m_Zoom += y;
		if(m_Zoom < 0.0f) 
		{
			m_Zoom = 0.0f;
		}
	}
	else if(p_Right)
	{
		// Pan in y direction
		{
			TVector3 v = m_Direction;
			v = v.cross( TVector3( 0.0f, 1.0f, 0.0f ) );
			v = v.cross( m_Direction );
			v.normalize();
			v.scale(y * PanFactor);
			m_Position.add(v);
		}
		
		// Pan in x direction
		{
			TVector3 v = m_Direction;
			v = v.cross( TVector3( 0.0f, 1.0f, 0.0f ) );
			v.scale(-x * PanFactor);
			m_Position.add(v);
		}
	}

	calculateDirection();
}

void TRoamingCamera::setPosition(float x,float y,float z)
{
	m_Position.x = x;
	m_Position.y = y;
	m_Position.z = z;
}

void TRoamingCamera::setHeading(float v)
{
	m_Heading = v;
	calculateDirection();
}


void TRoamingCamera::setPitch(float v)
{
	m_Pitch = v;
	calculateDirection();
}

void TRoamingCamera::snapToAxis(int p_Axis)
{
	throw std::string("Not implemented");
}

void TRoamingCamera::setProjectionMatrix(int w,int h)
{
	if(m_ProjectionMode == TCamera::PROJECTIONMODE_PERSPECTIVE)
	{
		gluPerspective(40, (float)w/(float)h, 0.2f, 100.0f);
//		gluPerspective(60, 1.0f, 0.2f, 100.0f);
	}
	else if(m_ProjectionMode == TCamera::PROJECTIONMODE_ORTHOGONAL)
	{
		float ratio = (float)w/(float)h;
		float zoom = m_Zoom * 0.1f;
		glOrtho(-1.0f*zoom, 1.0f*zoom, -1.0f*zoom/ratio, 1.0f*zoom/ratio, 1.0f, 100.0f);
		// glOrtho(-1.0f, 1.0f, -1.0f, 1.0f, 1.0f, 100.0f);
	}
	else throw string("Unknown projection mode");
}

void TRoamingCamera::reset()
{
	m_Position = m_InitialPosition;
	m_Heading = m_InitialHeading;
	m_Pitch = m_InitialPitch;

	m_Zoom = 1.0f;
	calculateDirection();
}


void TRoamingCamera::writeToStream(std::ofstream * s) const
{
	*s << /*float*/ m_Heading << " ";
	*s << /*float*/ m_Pitch << " ";
	//TPoint3 m_Position;
	*s << m_Position.x << " ";
	*s << m_Position.y << " ";
	*s << m_Position.z << " ";
	//TVector3 m_Direction;
	*s << m_Direction.x << " ";
	*s << m_Direction.y << " ";
	*s << m_Direction.z << " ";
	*s << /*float*/ m_Zoom << " ";
	//TPoint3 m_InitialPosition;
	*s << m_InitialPosition.x << " ";
	*s << m_InitialPosition.y << " ";
	*s << m_InitialPosition.z << " ";
	*s << /*float*/ m_InitialHeading << " ";
	*s << /*float*/ m_InitialPitch << " ";
}

TRoamingCamera * TRoamingCamera::readFromStream(std::ifstream * s)
{
	TRoamingCamera * c = new TRoamingCamera();
	 
	*s >> /*float*/ c->m_Heading;
	*s >> /*float*/ c->m_Pitch;		
	//TPoint3 m_Position;
	*s >> c->m_Position.x;
	*s >> c->m_Position.y;
	*s >> c->m_Position.z;
	//TVector3 m_Direction;
	*s >> c->m_Direction.x;
	*s >> c->m_Direction.y;
	*s >> c->m_Direction.z;
	*s >> /*float*/ c->m_Zoom;
	//TPoint3 m_InitialPosition;
	*s >> c->m_InitialPosition.x;
	*s >> c->m_InitialPosition.y;
	*s >> c->m_InitialPosition.z;
	*s >> /*float*/ c->m_InitialHeading;
	*s >> /*float*/ c->m_InitialPitch;
	return c;
}



