#include "stdafx.h"

#include "colormap.h"

#include "abstractfilter.h"
#include "abstractmeasure.h"
#include "Geometry.h"

#include "controls.h"
#include "globals.h"
#include "layer.h"

TColorMap::TColorMap(float p_Lower, float p_Upper)
	: m_Lower(p_Lower)
	, m_Upper(p_Upper)
{
	if(m_Upper < m_Lower) m_Upper = m_Lower;
};




float TColorMap::interpolate(const float &v0, const float &v1, const float &t) const
{
    return ( 1.0f-t )*v0 + t*v1;
}

// ----------------------------------
void TColorMap::hlsToRgb( 
    float H,  float L,  float S,
//    float &R, float &G, float &B )
	TColor3 * col ) const
// ----------------------------------
// Adapted with permission from Hannes Pretorius
// ------------------------------------------------------------------
// Adapted from Foley et al., 1996.
// In:  h       in [0.0, 360.0]
//      l, s    in [0, 1]
// Out: r, g, b in [0, 1]
// ------------------------------------------------------------------
{
    float  var1, var2;

    if ( S == 0 )
    {
        col->r = L;
        col->g = L;
        col->b = L;
    }
    else
    {
        if ( L < 0.5 )
            var2 = L * ( L+S );
        else
            var2 = ( L+S ) - ( S*L );

        var1 = 2 * L - var2;

        col->r = hlsValue( var1, var2, H + 120.0 );
        col->g = hlsValue( var1, var2, H );
        col->b = hlsValue( var1, var2, H - 120.0 );
    }
}

// --------------------------------------
float TColorMap::hlsValue(float var1, float var2, float hue ) const
// --------------------------------------
// Adapted with permission from Hannes Pretorius
// ------------------------------------------------------------------
// Adapted from Foley et al., 1996.
// ------------------------------------------------------------------
{
    float result = 0.0; 
    
    if ( hue < 0.0 )
        hue += 360.0;
    if ( hue > 360.0 )
        hue -= 360.0;

    if ( hue < 60.0 )
        result = var1 + ( var2 - var1 ) * hue/60.0;
    else if ( hue < 180.0 )
        result = var2;
    else if ( hue < 240.0 )
        result = var1 + ( var2 - var1 ) * ( 240.0 - hue )/60.0;
    else
        result = var1;
    return result;
}


void TColorMapUsingMeasure::DrawGui(wxWindow * p_Window)
{
	new TColorMapUsingMeasurePanel(this, p_Window, -1, wxPoint(5,5), wxSize(200,200));
}


TColorMapUsingMeasure::TColorMapUsingMeasure(shared_ptr<TMeasure> * const p_Measure) 
: m_Measure(p_Measure)
{
	m_Lower = (*this->m_Measure)->getMinValue();
	m_Upper = (*this->m_Measure)->getMaxValue();
};

void TColorMapUsingMeasure::init()
{
	m_Lower = (*this->m_Measure)->getMinValue();
	m_Upper = (*this->m_Measure)->getMaxValue();
}


TNormalizedColorMapUsingMeasure::TNormalizedColorMapUsingMeasure(shared_ptr<TMeasure> * const p_Measure)
	: TColorMapUsingMeasure(p_Measure) 
{
};


void TNormalizedColorMap::getColor(float p_Value, TColor3 * c) const
{
	const float & l = m_Lower;
	const float & u = m_Upper;
	float value = p_Value;
//	value = (value-l) / (u-l);
//	if(value < 0.0f) value = 0.0f;
//	else if(value > 1.0f) value = 1.0f;

	float rr = u-l; 
	if (rr>1.0e-6) rr = 1.0/rr; 
	value = (value-l)*rr;	

	// value is normalized
//	value = value + 0.125f; // make darkest blue lighter, ERROR: roodtinten niet meer te onderscheiden
	const float begin = 0.125f;
	value = value * (1.0f - begin) + begin;
	
	if(value < 0.0f) value = 0.0f;
	else if(value > 1.0f) value = 1.0f;


	const float dx=0.8f; 
	value = (6.0f-2.0f*dx)*value+dx;
	c->rgb[0] = (3.0f-fabs(value-4.0f)-fabs(value-5.0f))/2; if (c->rgb[0]<0.0f) c->rgb[0] = 0.0f; 
	c->rgb[1] = (4.0f-fabs(value-2.0f)-fabs(value-4.0f))/2; if (c->rgb[1]<0.0f) c->rgb[1] = 0.0f; 
	c->rgb[2] = (3.0f-fabs(value-1.0f)-fabs(value-2.0f))/2; if (c->rgb[2]<0.0f) c->rgb[2] = 0.0f;

}


void TNormalizedColorMapUsingMeasure::getColor(const TCoord3 & p_Coord, TColor3 * c) const
{
	TNormalizedColorMap::getColor( (*m_Measure)->toFloat(p_Coord), c);
}

void TNormalizedColorMapUsingMeasure::getColor(unsigned int idx, TColor3 * c) const
{
	TNormalizedColorMap::getColor( (*m_Measure)->vvaluex(idx), c);
}



TColorMapUsingMeasurePanel::TColorMapUsingMeasurePanel(TColorMapUsingMeasure * p_ColorMap, wxWindow* parent, wxWindowID id, const wxPoint& pos, const wxSize& size, long style)
	: wxPanel(parent,id,pos,size,style)
	, m_ColorMap(p_ColorMap)
{
	int x = 5;
	int y = 5;

	new wxStaticText(this, -1, "Colormap settings", wxPoint(x,y) );
	y += 15;
	
	x += 20;
	y += 15;

	const float l = m_ColorMap->m_Lower;
	const float u = m_ColorMap->m_Upper;
	const float Rate = (u-l)/1000.0f; 
	
	new wxStaticText(this, -1, "Lower", wxPoint(x,y) );
	assert(m_ColorMap);
	m_SpinCtrlLower = new wxSpinCtrlForFloats(this, ID_LOWERVALUE, -10, 10000, Rate, l, wxPoint(x+40,y), wxSize(70,20) );
	y += 35;
	new wxStaticText(this, -1, "Upper", wxPoint(x,y) );
	m_SpinCtrlUpper = new wxSpinCtrlForFloats(this, ID_UPPERVALUE, -10, 10000, Rate, u, wxPoint(x+40,y), wxSize(70,20) );
	y += 35;
	x -= 20;

	m_UpdateButton = new wxButton(this, ID_UPDATE, "Reset", wxPoint(x,y));

//	this->Connect( ID_MANUAL, wxEVT_COMMAND_CHECKBOX_CLICKED, (wxObjectEventFunction) &TColorMapUsingMeasurePanel::OnChangeManual );
	this->Connect( ID_LOWERVALUE, wxEVT_COMMAND_SPINCTRL_UPDATED, (wxObjectEventFunction) &TColorMapUsingMeasurePanel::OnChangeLower );
	this->Connect( ID_UPPERVALUE, wxEVT_COMMAND_SPINCTRL_UPDATED, (wxObjectEventFunction) &TColorMapUsingMeasurePanel::OnChangeUpper );
	this->Connect( ID_UPDATE, wxEVT_COMMAND_BUTTON_CLICKED, (wxObjectEventFunction) &TColorMapUsingMeasurePanel::OnUpdateButton );
}

void TColorMapUsingMeasurePanel::OnChangeLower(wxCommandEvent & event)
{
	m_ColorMap->m_Lower =  m_SpinCtrlLower->GetFloat();
	(*m_ColorMap->m_Measure)->m_AbstractField->getLayer()->setNeedRedraw();
	g_Mediator.redrawCanvas();
}

void TColorMapUsingMeasurePanel::OnChangeUpper(wxCommandEvent & event)
{
	m_ColorMap->m_Upper = m_SpinCtrlUpper->GetFloat();
	(*m_ColorMap->m_Measure)->m_AbstractField->getLayer()->setNeedRedraw();
	g_Mediator.redrawCanvas();
}

void TColorMapUsingMeasurePanel::OnUpdateButton(wxCommandEvent & event)
{
	m_ColorMap->init();
	(*m_ColorMap->m_Measure)->m_AbstractField->getLayer()->setNeedRedraw();
	g_Mediator.redrawCanvas();
}




TMonochromeColorMapUsingMeasure::TMonochromeColorMapUsingMeasure(shared_ptr<TMeasure> * const p_Measure)
	: TColorMapUsingMeasure(p_Measure)
{
}

void TMonochromeColorMap::getColor(float p_Value, TColor3 * c) const
{
	const float & l = m_Lower;
	const float & u = m_Upper;
	float value = (p_Value - l);
	value /= (u-l);
	if(value < 0.0f) value = 0.0f;
	if(value > 1.0f) value = 1.0f;
	c->r = c->g = c->b = value;
	
}

void TMonochromeColorMapUsingMeasure::getColor(const TCoord3 & p_Coord, TColor3 * c) const
{
	TMonochromeColorMap::getColor( (*m_Measure)->toFloat(p_Coord), c);
}

void TMonochromeColorMapUsingMeasure::getColor(unsigned int idx, TColor3 * c) const
{
	TMonochromeColorMap::getColor( (*m_Measure)->vvaluex(idx), c);
}



TInverseMonochromeColorMapUsingMeasure::TInverseMonochromeColorMapUsingMeasure(shared_ptr<TMeasure> * const p_Measure)
	: TColorMapUsingMeasure(p_Measure)
{
}

void TInverseMonochromeColorMap::getColor(float p_Value, TColor3 * c) const
{
	const float & l = m_Lower;
	const float & u = m_Upper;
	float value = (p_Value - l);
	value /= (u-l);
	if(value < 0.0f) value = 0.0f;
	if(value > 1.0f) value = 1.0f;
	c->r = c->g = c->b = 1.0f - value;
	
}

void TInverseMonochromeColorMapUsingMeasure::getColor(const TCoord3 & p_Coord, TColor3 * c) const
{
	TInverseMonochromeColorMap::getColor( (*m_Measure)->toFloat(p_Coord), c);
}

void TInverseMonochromeColorMapUsingMeasure::getColor(unsigned int idx, TColor3 * c) const
{
	TInverseMonochromeColorMap::getColor( (*m_Measure)->vvaluex(idx), c);
}





THeatMapUsingMeasure::THeatMapUsingMeasure(shared_ptr<TMeasure> * const p_Measure)
	: TColorMapUsingMeasure(p_Measure) 
{
};

void THeatMap::getColor(float p_Value, class TColor3 * c) const
{
	const float hueBeg = 0.0f;
	const float hueEnd = 60.0f;

	float frac = (m_Upper-m_Lower > 1.0e-6) ? ((p_Value-m_Lower) / (m_Upper-m_Lower)) : p_Value;
	if(frac > 1.0f) frac = 1.0f;
	else if(frac < 0.0f) frac = 0.0f;

    const float h = hueBeg + frac*( hueEnd-hueBeg );
	const float l = interpolate( 0.4f, 0.9f, frac );
	const float s = 1.0f; 
	hlsToRgb(h,l,s,c);		
}

void THeatMapUsingMeasure::getColor(const TCoord3 & p_Coord, TColor3 * c) const
{
	THeatMap::getColor( (*m_Measure)->toFloat(p_Coord), c);
}

void THeatMapUsingMeasure::getColor(unsigned int idx, TColor3 * c) const
{
	THeatMap::getColor( (*m_Measure)->vvaluex(idx), c);
}




TBlueYellowMapUsingMeasure::TBlueYellowMapUsingMeasure(shared_ptr<TMeasure> * const p_Measure)
	: TColorMapUsingMeasure(p_Measure) 
{
};

void TBlueYellowMap::getColor(float p_Value, class TColor3 * c) const
{
	float frac = (m_Upper-m_Lower > 1.0e-6) ? ((p_Value-m_Lower) / (m_Upper-m_Lower)) : p_Value;
	if(frac > 1.0f) frac = 1.0f;
	else if(frac < 0.0f) frac = 0.0f;

	float h,s;
    if ( frac < 0.5 )
    {
        h = 240.0;
        s = 1.0 - frac*2.0;
    }
    else
    {
        h = 60.0;
        s = frac*2.0 - 1.0;
    }
	hlsToRgb(h,0.5f,s,c);		
}

void TBlueYellowMapUsingMeasure::getColor(const TCoord3 & p_Coord, TColor3 * c) const
{
	TBlueYellowMap::getColor( (*m_Measure)->toFloat(p_Coord), c);
}

void TBlueYellowMapUsingMeasure::getColor(unsigned int idx, TColor3 * c) const
{
	TBlueYellowMap::getColor( (*m_Measure)->vvaluex(idx), c);
}




void TDecompositionColorMap::getColor(float p_Value, class TColor3 * c) const
{
	/*
	static float colors[] = 
		{
			0.0f, 0.5f, 1.0f,  // blue
			1.0f, 1.0f, 0.0f, // yellow
			0.0f, 0.8f, 0.0f, // green
			1.0f, 0.3f, 0.0f, // red
		};
	*/

	// Fully saturated
	static float colors[] = 
		{
			1.0f, 0.0f, 0.0f, // red
			0.0f, 0.3f, 1.0f,  // blue
			1.0f, 1.0f, 0.0f, // yellow
			0.0f, 1.0f, 0.0f, // green
		};

	int v = ((int)(p_Value - m_Lower + 0.001f)) % ( sizeof(colors)/((3*sizeof(float) ) ) );

	c->r = colors[v*3+0];
	c->g = colors[v*3+1];
	c->b = colors[v*3+2];
};



TDecompositionColorMapUsingMeasure::TDecompositionColorMapUsingMeasure(shared_ptr<TMeasure> * const p_Measure)
	: TColorMapUsingMeasure(p_Measure) 
{
};

void TDecompositionColorMapUsingMeasure::getColor(const TCoord3 & p_Coord, TColor3 * c) const
{
	TDecompositionColorMap::getColor( (*m_Measure)->toFloat(p_Coord), c);
}

void TDecompositionColorMapUsingMeasure::getColor(unsigned int idx, TColor3 * c) const
{
	TDecompositionColorMap::getColor( (*m_Measure)->vvaluex(idx), c);
}





void TDecomposition2ColorMap::getColor(float p_Value, class TColor3 * c) const
{
	const float low = 0.2f;

	/*
	static float colors[] = 
		{
			1.0f,  low,  low, // 1 red 
			 low, 1.0f,  low, // 2 green
			 low,  low, 1.0f, // 3 blue
			1.0f, 1.0f,  low, // 4 yellow
			1.0f,  low, 1.0f, // 5 magenta
			 low, 1.0f, 1.0f, // 6 cyan
			0.5f, 0.5f, 0.5f, // 0 gray
		};
	*/
	static float colors[] = 
		{
			1.0f,  123.0f/255.0f,  0.0f, // 1 orange
			//1.0f,  147.0f/255.0f,  46.0f/255.0f, // 1 orange
			//166.0f/255.0f, 92.0f/255.0f,  23.0f/255.0f, // 2 brown
			246.0f/255.0f, 222.0f/255.0f, 124.0f/255.0f // yellow
		};

	// Orange cover colors

	int v = ((int)(p_Value - m_Lower + 0.001f)) % ( sizeof(colors)/((3*sizeof(float)) ) );

	c->r = colors[v*3+0];
	c->g = colors[v*3+1];
	c->b = colors[v*3+2];
};


TDecomposition2ColorMapUsingMeasure::TDecomposition2ColorMapUsingMeasure(shared_ptr<TMeasure> * const p_Measure)
	: TColorMapUsingMeasure(p_Measure) 
{
};

void TDecomposition2ColorMapUsingMeasure::getColor(const TCoord3 & p_Coord, TColor3 * c) const
{
	TDecomposition2ColorMap::getColor( (*m_Measure)->toFloat(p_Coord), c);
}

void TDecomposition2ColorMapUsingMeasure::getColor(unsigned int idx, TColor3 * c) const
{
	TDecomposition2ColorMap::getColor( (*m_Measure)->vvaluex(idx), c);
}





TColorizeColorMap::TColorizeColorMap()
	: m_Color(0.1f, 0.1f, 0.1f)
{
}

void TColorizeColorMap::getColor(float p_Value, class TColor3 * c) const
{
	*c = m_Color;
};


TColorizeColorMapUsingMeasure::TColorizeColorMapUsingMeasure(shared_ptr<TMeasure> * const p_Measure)
	: TColorMapUsingMeasure(p_Measure) 
{
};

void TColorizeColorMapUsingMeasure::getColor(const TCoord3 & p_Coord, TColor3 * c) const
{
	TColorizeColorMap::getColor( (*m_Measure)->toFloat(p_Coord), c);
}

void TColorizeColorMapUsingMeasure::getColor(unsigned int idx, TColor3 * c) const
{
	TColorizeColorMap::getColor( (*m_Measure)->vvaluex(idx), c);
}


TColorizeColorMapPanel::TColorizeColorMapPanel(TColorizeColorMapUsingMeasure * p_ColorMap, wxWindow* parent, wxWindowID id, const wxPoint& pos, const wxSize& size, long style)
	: wxPanel(parent,id,pos,size,style)
	, m_ColorMap(p_ColorMap)
{
	int x = 5;
	int y = 5;

	new wxStaticText(this, -1, "Colorize settings", wxPoint(x,y) );

	assert(m_ColorMap);
	new wxStaticText(this, -1, "Red", wxPoint(x,y) );
	m_SpinCtrlRed = new wxSpinCtrlForFloats(this, ID_RED, 0.0f, 1.0f, 0.1f, p_ColorMap->m_Color.r, wxPoint(x+40,y), wxSize(70,20) );
	y += 35;
	new wxStaticText(this, -1, "Green", wxPoint(x,y) );
	m_SpinCtrlGreen = new wxSpinCtrlForFloats(this, ID_GREEN, 0.0f, 1.0f, 0.1f, p_ColorMap->m_Color.g, wxPoint(x+40,y), wxSize(70,20) );
	y += 35;

	new wxStaticText(this, -1, "Blue", wxPoint(x,y) );
	m_SpinCtrlBlue = new wxSpinCtrlForFloats(this, ID_BLUE, 0.0f, 1.0f, 0.1f, p_ColorMap->m_Color.b, wxPoint(x+40,y), wxSize(70,20) );
	y += 35;

	x -= 20;

	m_UpdateButton = new wxButton(this, ID_UPDATE, "Reset", wxPoint(x,y));

//	this->Connect( ID_MANUAL, wxEVT_COMMAND_CHECKBOX_CLICKED, (wxObjectEventFunction) &TColorMapUsingMeasurePanel::OnChangeManual );
	this->Connect( ID_RED, wxEVT_COMMAND_SPINCTRL_UPDATED, (wxObjectEventFunction) &TColorizeColorMapPanel::OnChangeRed );
	this->Connect( ID_GREEN, wxEVT_COMMAND_SPINCTRL_UPDATED, (wxObjectEventFunction) &TColorizeColorMapPanel::OnChangeGreen );
	this->Connect( ID_BLUE, wxEVT_COMMAND_SPINCTRL_UPDATED, (wxObjectEventFunction) &TColorizeColorMapPanel::OnChangeBlue );

	this->Connect( ID_UPDATE, wxEVT_COMMAND_BUTTON_CLICKED, (wxObjectEventFunction) &TColorizeColorMapPanel::OnUpdateButton );

}

void TColorizeColorMapPanel::OnChangeRed(class wxCommandEvent & event)
{
	m_ColorMap->m_Color.r = m_SpinCtrlRed->GetFloat();
	(*m_ColorMap->m_Measure)->m_AbstractField->getLayer()->setNeedRedraw();
	g_Mediator.redrawCanvas();
}
void TColorizeColorMapPanel::OnChangeGreen(class wxCommandEvent & event)
{
	m_ColorMap->m_Color.g = m_SpinCtrlGreen->GetFloat();
	(*m_ColorMap->m_Measure)->m_AbstractField->getLayer()->setNeedRedraw();
	g_Mediator.redrawCanvas();
}
void TColorizeColorMapPanel::OnChangeBlue(class wxCommandEvent & event)
{
	m_ColorMap->m_Color.b = m_SpinCtrlBlue->GetFloat();
	(*m_ColorMap->m_Measure)->m_AbstractField->getLayer()->setNeedRedraw();
	g_Mediator.redrawCanvas();
}
void TColorizeColorMapPanel::OnUpdateButton(wxCommandEvent & event)
{
	m_ColorMap->init();
	g_Mediator.redrawCanvas();
}


void TColorizeColorMapUsingMeasure::DrawGui(wxWindow * p_Window)
{
	new TColorizeColorMapPanel(this, p_Window, -1, wxPoint(5,5), wxSize(200,200));
}
