#include "stdafx.h"

#include "deltaomega.h"

#include "globals.h"
#include "settings.h"

#include "indexedoriginsfield.h"
#include "indexedorigins.h"
#include "indexedorigins_cells.h"
#include "indexedorigins_pool.h"
#include "danielsson3d.h"
#include "parameters.h"
#include "layer.h"
#include "component.h"
#include "logwriter.h"

using namespace std;

const unsigned int TDeltaOmega::TCell::INVALIDCOMPONENT = (numeric_limits<unsigned int>::max)();

TDeltaOmega::TAuxiliaryStruct::TAuxiliaryStruct()
{
	m_Distance = std::numeric_limits<float>::infinity();
	m_Previous = TIndexedOrigins::INVALIDINDEX;
	m_IsKnown = false;
}

TDeltaOmega::TFloodfillStruct::TFloodfillStruct()
{
	m_Component = TCell::INVALIDCOMPONENT;
	m_NeighborsGeodesic = false;
}

void TOmega::initIndexFields(const class TUnsignedCharField3 * p_Image)
{
	m_ForegroundIndexField.reset( new TIndexField3(p_Image->getMaxX(), p_Image->getMaxY(), p_Image->getMaxZ()) );

	if(g_Parameters.m_ComputeBackgroundSskel)
	{
		m_BackgroundIndexField.reset( new TIndexField3(p_Image->getMaxX(), p_Image->getMaxY(), p_Image->getMaxZ()) );
	}

	unsigned int BackgroundVoxelCount = 0;
	unsigned int ObjectVoxelCount = 0;
	{
		for(unsigned int k=0; k<p_Image->getMaxZ(); k++)
		{
			for(unsigned int j=0; j<p_Image->getMaxY(); j++)
			{
				for(unsigned int i=0; i<p_Image->getMaxX(); i++)
				{
					const TCoord3 p(i,j,k);

					const float v = p_Image->valuep(p);
					if( v < 0.0001 )
					{ 
						if(m_BackgroundIndexField)
						{
							m_BackgroundIndexField->valuep(p) = 0;
							BackgroundVoxelCount++;
						}
					}
					else 
					{
						m_ForegroundIndexField->valuep(p) = 0;
						ObjectVoxelCount++;
					}
				}
			}
		}
	}


	const unsigned int BOUNDARY = TIndexMapper::OUTSIDE-1;
	unsigned int BoundaryVoxelCount = 0;
	{
		for(unsigned int k=0; k<p_Image->getMaxZ(); k++)
		{
			for(unsigned int j=0; j<p_Image->getMaxY(); j++)
			{
				for(unsigned int i=0; i<p_Image->getMaxX(); i++)
				{
					const TCoord3 p(i,j,k);

					if (m_ForegroundIndexField->valuep(p) == 0)
					{
						if (   m_ForegroundIndexField->value(i-1,j,k) == TIndexMapper::OUTSIDE
							|| m_ForegroundIndexField->value(i+1,j,k) == TIndexMapper::OUTSIDE 
							|| m_ForegroundIndexField->value(i,j-1,k) == TIndexMapper::OUTSIDE 
							|| m_ForegroundIndexField->value(i,j+1,k) == TIndexMapper::OUTSIDE
							|| m_ForegroundIndexField->value(i,j,k-1) == TIndexMapper::OUTSIDE
							|| m_ForegroundIndexField->value(i,j,k+1) == TIndexMapper::OUTSIDE
							)
						{ 
							m_ForegroundIndexField->value(i,j,k) = BOUNDARY;
							BoundaryVoxelCount++;
						}
					}
				}
			}
		}
	}

	// Do check whether boundary voxel is completely surrounded by other boundary voxels or outside voxels
	{
		vector<TCoord3> Remove;
		for(unsigned int k=0; k<p_Image->getMaxZ(); k++)
		{
			for(unsigned int j=0; j<p_Image->getMaxY(); j++)
			{
				for(unsigned int i=0; i<p_Image->getMaxX(); i++)
				{
					const TCoord3 p(i,j,k);

					if (   
						 m_ForegroundIndexField->value(i,j,k) >= BOUNDARY
						&& m_ForegroundIndexField->value(i-1,j,k) >= BOUNDARY
						&& m_ForegroundIndexField->value(i+1,j,k) >= BOUNDARY
						&& m_ForegroundIndexField->value(i,j-1,k) >= BOUNDARY
						&& m_ForegroundIndexField->value(i,j+1,k) >= BOUNDARY
						&& m_ForegroundIndexField->value(i,j,k-1) >= BOUNDARY
						&& m_ForegroundIndexField->value(i,j,k+1) >= BOUNDARY
						)
					{ 
						Remove.push_back(p);
					}
				}
			}
		}
		for(unsigned int x=0; x<Remove.size(); x++)
		{
			m_ForegroundIndexField->valuep(Remove[x]) = TIndexMapper::OUTSIDE;
		}
	}

	m_ForegroundIndexField->m_Voxels.reserve( ObjectVoxelCount );
	if(m_BackgroundIndexField) m_BackgroundIndexField->m_Voxels.reserve( BackgroundVoxelCount );
	m_BoundaryIndexField.reset( new TSubIndexMapper(m_ForegroundIndexField.get()) );
	m_BoundaryIndexField->m_SubIndex.reserve( BoundaryVoxelCount );
	
	if(m_BackgroundIndexField)
	{
		m_BackgroundBoundaryIndexField.reset( new TSubIndexMapper(m_BackgroundIndexField.get()) );
		m_BackgroundBoundaryIndexField->m_SubIndex.reserve( BoundaryVoxelCount );
	}

	unsigned int boundaryidx = 0;
	unsigned int backgroundidx = 0;
	// Index inside voxels
	{
		unsigned int baseidx = 0;
		for(unsigned int k=0; k<p_Image->getMaxZ(); k++)
		{
			for(unsigned int j=0; j<p_Image->getMaxY(); j++)
			{
				for(unsigned int i=0; i<p_Image->getMaxX(); i++)
				{
					const TCoord3 p(i,j,k);
					
					if( m_ForegroundIndexField->valuep(p) == 0 )
					{
						m_ForegroundIndexField->valuep(p) = baseidx;
						m_ForegroundIndexField->m_Voxels.push_back(p);
						baseidx++;
					}
					else if(m_ForegroundIndexField->valuep(p) == BOUNDARY )
					{
						// Fg
						m_ForegroundIndexField->valuep(p) = baseidx;
						m_ForegroundIndexField->m_Voxels.push_back(p);
						
						// Boundary
						m_BoundaryIndexField->m_SubIndex.push_back( baseidx );
						m_BoundaryIndexField->m_Coord2IdxMap.insert( pair<TCoord3,unsigned int>(p,boundaryidx) );
						if(m_BackgroundBoundaryIndexField)
						{
							m_BackgroundIndexField->valuep(p) = backgroundidx;
							m_BackgroundIndexField->m_Voxels.push_back(p);

							m_BackgroundBoundaryIndexField->m_SubIndex.push_back( backgroundidx );
							m_BackgroundBoundaryIndexField->m_Coord2IdxMap.insert( pair<TCoord3,unsigned int>(p,boundaryidx) );
							backgroundidx++;
						}

						baseidx++;
						boundaryidx++;
					}
					else if (m_ForegroundIndexField->valuep(p) == TIndexField3::OUTSIDE)
					{
						// outside voxel	
						if(m_BackgroundIndexField)
						{
							m_BackgroundIndexField->valuep(p) = backgroundidx;
							m_BackgroundIndexField->m_Voxels.push_back(p);
							backgroundidx++;
						}
					}


				}
			}
		}
	}

	// Test: compute distance field from boundary index field inside background index field 
	if(m_BackgroundIndexField)
	{
		TSparseFloatField3 BgDistance( m_BackgroundIndexField );
		BgDistance.setEveryValue( (numeric_limits<float>::max)() );

		TSparseUintField3 Origin( m_BackgroundIndexField );
		Origin.setEveryValue( (numeric_limits<unsigned int>::max)() );
		const unsigned int OriginMaxIndex = Origin.getMaxIndex();

		typedef multimap<float, TCoord3, less<float> > TQueue;
		TQueue S;
		for(unsigned int x=0; x<m_BoundaryIndexField->getMaxIndex(); x++)
		{
			const TCoord3 p = m_BoundaryIndexField->vidx2coord(x);
			S.insert( TQueue::value_type(0.0f, p) );
			
			if( !Origin.getIndexField()->vinside(p) ) throw string("!Origin.getMaxIndex()->vinside(p)");
			Origin.wvaluep(p) = x;
		}
		while(!S.empty())
		{
			const float pdist = S.begin()->first;
			const TCoord3 p = S.begin()->second;
			S.erase(S.begin());
			if(Origin.vvaluep(p) == (numeric_limits<unsigned int>::max)()) throw string("Origin.vvaluep(p) == (numeric_limits<unsigned int>::max)()");
			const unsigned int poriginidx = Origin.vvaluep(p);
			const TCoord3 porigin = m_BoundaryIndexField->vidx2coord( poriginidx );
			const float dist = porigin.distance( p );
			BgDistance.wvaluep(p) = dist;
			if(dist > 0.0f)
			{
			int test = 5;
			}

			TCoord3 np;
			for(np.x=p.x-1; np.x<=p.x+1; np.x++) // neighbors
			for(np.y=p.y-1; np.y<=p.y+1; np.y++)
			for(np.z=p.z-1; np.z<=p.z+1; np.z++)
			{
				if(p == np) continue;
				if(BgDistance.vvaluep(np) != (numeric_limits<float>::max)()) continue;

				if( m_ForegroundIndexField->valuep(np) != TIndexField3::OUTSIDE ) continue;

				const unsigned int nporiginidx = Origin.vvaluep(np);
				if( nporiginidx == (numeric_limits<unsigned int>::max)() )
				{
					const float dist = porigin.distance(np);
					S.insert( TQueue::value_type( dist, np) );
					Origin.wvaluep(np) = Origin.vvaluep(p);
					if( !Origin.getIndexField()->vinside(np) ) throw string("!Origin.getMaxIndex()->vinside(np)");
				}  
				else
				{
					const TCoord3 nporigin = m_BoundaryIndexField->vidx2coord( Origin.vvaluep(np) );
					const float newdist = np.distance(porigin);
					const float olddist = np.distance(nporigin);
					if( newdist < olddist )
					{
						S.insert( TQueue::value_type( newdist, np) );
						Origin.wvaluep(np) = Origin.vvaluep(p);
						if( !Origin.getIndexField()->vinside(np) ) throw string("!Origin.getMaxIndex()->vinside(np)");
					}
				}
			}
		}
		

		// Add background bounding box as boundary voxels
		for(unsigned int k=0; k<p_Image->getMaxZ(); k++)
		{
			for(unsigned int j=0; j<p_Image->getMaxY(); j++)
			{
				for(unsigned int i=0; i<p_Image->getMaxX(); i++)
				{
					const TCoord3 p(i,j,k);

					if( g_Parameters.m_UseBoundingBoxForBgFt )
					{
						/*
						if( BgDistance.vvaluep(p) > 15.0f )
						{
							m_BackgroundBoundaryIndexField->m_SubIndex.push_back( m_BackgroundIndexField->vcoord2idx(p) );
							m_BackgroundBoundaryIndexField->m_Coord2IdxMap.insert( pair<TCoord3,unsigned int>(p,boundaryidx) );
							boundaryidx++;
						}
						*/

						// /*
						if(	k == 0 || k == p_Image->getMaxZ()-1 ||
							j == 0 || j == p_Image->getMaxY()-1 ||
							i == 0 || i == p_Image->getMaxX()-1
						)
						{
							m_BackgroundBoundaryIndexField->m_SubIndex.push_back( backgroundidx );
							m_BackgroundBoundaryIndexField->m_Coord2IdxMap.insert( pair<TCoord3,unsigned int>(p,boundaryidx) );
							boundaryidx++;
						}
						// */
					}
					// /*
					if(m_ForegroundIndexField->valuep(p) == TIndexField3::OUTSIDE)
					{
						// outside voxel	
						m_BackgroundIndexField->valuep(p) = backgroundidx;
						m_BackgroundIndexField->m_Voxels.push_back(p);
						backgroundidx++;
					}
					// */

				}
			}
		}

	}

	if(!m_BackgroundBoundaryIndexField) m_BackgroundBoundaryIndexField = m_BoundaryIndexField;
}

void TOmega::initDeltaOmega()
{
	m_DeltaOmega.reset( new TDeltaOmega(this) );
}

void TOmega::initFT()
{
	TPrecisionTimer PrecisionTimer;
	PrecisionTimer.start();
	if(!g_Settings.m_ReleaseMode) *g_Log << "Compute foreground FT... ";
	{
		//TDanielsson3d danielsson(m_ForegroundIndexField.get(), m_DeltaOmega->m_Boundary.get(), 4);
		m_ForegroundFt.reset( new TDanielsson3d(m_ForegroundIndexField.get(), m_BoundaryIndexField.get(), g_Parameters.m_MaxFeaturePointCount) );
		m_ForegroundFt->perform();
		//m_ForegroundFt = danielsson.m_Origins;
	}
	if(!g_Settings.m_ReleaseMode) *g_Log << "done in " << PrecisionTimer.stop() << "s. \n";

	PrecisionTimer.start();
	if(!g_Settings.m_ReleaseMode) *g_Log << "Compute background FT... ";
	if(g_Parameters.m_ComputeBackgroundSskel)
	{
//		shared_ptr<TBoundary> BB( new TBoundary(m_BackgroundBoundaryIndexField.get()) );
		m_BackgroundFt.reset( new TDanielsson3d(m_BackgroundIndexField.get(), m_BackgroundBoundaryIndexField.get(), g_Parameters.m_MaxFeaturePointCount) );
		m_BackgroundFt->perform();

		//TDanielsson3d danielsson(m_BackgroundIndexField.get(), m_DeltaOmega->m_Boundary.get(), 4);
		//danielsson.perform();
		//m_BackgroundFt = danielsson.m_Origins;
	}
	if(!g_Settings.m_ReleaseMode) *g_Log << "done in " << PrecisionTimer.stop() << "s. \n";
}






TDeltaOmega::TDeltaOmega(TOmega * p_Omega)
	: m_Omega(p_Omega)
	, CELLCOUNT(g_Parameters.m_CellCount)
	, m_CacheSize(0)
	, m_ComputedGeodesics(0)
{
	WEIGHTS[0] = 0.9016f; WEIGHTS[1] = 1.289f; WEIGHTS[2] = 1.615f;
//	WEIGHTS[0] = 1.0f; WEIGHTS[1] = sqrtf(2.0f); WEIGHTS[2] = sqrtf(3.0f);

	TIndexMapper * Boundary = p_Omega->m_BoundaryIndexField.get();

	//m_Boundary.reset(new TBoundary(p_Omega->m_BoundaryIndexField.get()));
	//if(m_Boundary->size() >= TIndexedOrigins::INVALIDINDEX) throw string("m_Boundary->size() >= TIndexedOrigins::INVALIDINDEX");
	//if(m_Boundary->size() >= TCell::INVALIDCOMPONENT) throw string("m_Boundary->size() >= TCell::INVALIDCOMPONENT");

	wxStopWatch sw;
	sw.Start(0);
	{
		unsigned int WarningAdjacencySize = 0;
		m_AdjacencyList.resize(Boundary->getMaxIndex());
		for(unsigned int x=0; x<m_AdjacencyList.size(); x++)
		{
			int AdjacencySize = 0;

			const TCoord3 & p = Boundary->vidx2coord(x);
			for(int i=p.x-1; i<=p.x+1; i++)
			{
				for(int j=p.y-1; j<=p.y+1; j++)
				{
					for(int k=p.z-1; k<=p.z+1; k++)
					{
						const TCoord3 np(i,j,k);
						const unsigned int npidx = Boundary->vcoord2idx(np);
						if(npidx == TIndexMapper::OUTSIDE) continue;

						// New: check whether two neighboring voxels have common voxel that is INSIDE the shape
						// and not on the boundary
						// We know that p and np are both on the boundary
						// We should find a common neighbor that is inside the shape
						// 
						// /*
						bool commonoutside = false;
					 	for(int nx=min(p.x,np.x)-1; nx<=max(p.x,np.x)+1 && !commonoutside; nx++)
						for(int ny=min(p.y,np.y)-1; ny<=max(p.y,np.y)+1 && !commonoutside; ny++)
						for(int nz=min(p.z,np.z)-1; nz<=max(p.z,np.z)+1 && !commonoutside; nz++)
						{
							const TCoord3 q(nx,ny,nz);
							// q should be a neighbor of both p and np
							if(		abs(q.x-p.x)<=1 && abs(q.y-p.y)<=1 && abs(q.z-p.z)<=1 
								&&	abs(q.x-np.x)<=1 && abs(q.y-np.y)<=1 && abs(q.z-np.z)<=1
								&&  (!Boundary->vinside(q)) 
								&&  (p_Omega->m_ForegroundIndexField->vinside(q) )
							)

							{
								commonoutside = true;
							}
						}
						if(!commonoutside) continue;
						// */
						
						char weight = 3 - (np.x == p.x) - (np.y == p.y) - (np.z == p.z);
						if(weight == 0) continue; // Allow all 3 directions
						//if(weight <= 1) continue; // Allow only 2 directions

						if(AdjacencySize >= MAXNSIZE) throw string("TDeltaOmega(): MAXNSIZE too small\n");
						m_AdjacencyList[x].m_Edges[AdjacencySize].m_ToVertex = npidx;
						m_AdjacencyList[x].m_Edges[AdjacencySize].m_Weight = weight-1;
						AdjacencySize++;
					}
				}
			}
			if(AdjacencySize == 0) WarningAdjacencySize++; 
			m_AdjacencyList[x].m_EdgeCount = AdjacencySize;
		}
		if(WarningAdjacencySize > 0) *g_Log << "Warning: TDeltaOmega(): AdjacencySize == 0 for " << WarningAdjacencySize << " voxels.\n";
	}

	// Construct initial values of helper structures. These will be used for fast initialization using memcopying.
	{
		{
			m_InitialAuxiliaryList.m_Distance = (numeric_limits<float>::max)();
			m_InitialAuxiliaryList.m_Previous = TIndexedOrigins::INVALIDINDEX;
			m_InitialAuxiliaryList.m_IsKnown= false;
			m_AuxiliaryList.resize( m_AdjacencyList.size(), m_InitialAuxiliaryList );
		}

		{
			m_InitialComponentList.m_Component = TCell::INVALIDCOMPONENT;
			m_InitialComponentList.m_NeighborsGeodesic = false;
			m_ComponentList.resize( m_AdjacencyList.size(), m_InitialComponentList );
		}

		/*
		{
			TAllPathsStruct s;
			s.m_Dist = (numeric_limits<float>::max)();
			s.m_IsKnown = false;
			m_InitialAllPathDistances.resize( m_AdjacencyList.size(), s );
			m_AllPathDistances1 = m_InitialAllPathDistances;
			m_AllPathDistances2 = m_InitialAllPathDistances;
		}
		*/
	}


	Visited.resize( m_AdjacencyList.size() * 4 );
};


void TDeltaOmega::createSpatialSubdivision()
{
	// Create spatial subdivision
	{
		unsigned int x;
		m_Cells.resize( log((double)m_AdjacencyList.size())/log((double)CELLCOUNT) ); 
		m_Vertex2Cells.resize(m_AdjacencyList.size());
	
		// Reserve m_Cells sizes
		{
			unsigned int size = CELLCOUNT;
			for(x=0; x<m_Cells.size(); x++)
			{
				m_Cells[x].reserve(size);
				size *= CELLCOUNT;
			}
		}

		// Let io be all vertices
		TIndexedOrigins_Vector io;
		io.resize(m_AdjacencyList.size());
		for(x=0; x<m_AdjacencyList.size(); x++) io[x] = x;
		
		// Create the subdivision recursively
		createSpatialSubdivision(&io, 0, 0);
	}

	// Set m_Neighbors, m_Neighbors are neigbhor on same level
	{
		unsigned int l;
		unsigned int x,y,n; 
		TIndex pidx, npidx;
		set<TCell*> Neighbors;
		set<TCell*>::iterator it;
		for(l=0; l<m_Cells.size(); l++)
		{
			g_Mediator.yield();

			for(x=0; x<m_Cells[l].size(); x++)
			{
				TCell * cellx = m_Cells[l][x];

				Neighbors.clear();

				// Loop through cellx vertices...
				for(y=0; y < cellx->m_Vertices.size(); y++)
				{
					pidx = cellx->m_Vertices[y];
					for(n=0; n<m_AdjacencyList[pidx].m_EdgeCount; n++)
					{
						npidx = m_AdjacencyList[pidx].m_Edges[n].m_ToVertex;
						
						if( l < m_Vertex2Cells[npidx].size() ) 
						{
							TCell * cellnp = m_Vertex2Cells[npidx][l];
							if( cellnp != cellx ) // if cell of neighboring vertex is different than cellx, we have found a neighboring cell 
							{
								Neighbors.insert( cellnp );
							}
						}
					}
				}

				// Assign neighbor set
				{
					unsigned int c=0;
					cellx->m_Neighbors.resize(Neighbors.size());
					for(it=Neighbors.begin(); it != Neighbors.end(); it++)
					{
						cellx->m_Neighbors[c++] = *it;
					}
				}
			}
		}
	}

	// Reset all colors to white
	{
		unsigned int l,j;
		for(l=0; l<m_Cells.size(); l++)
		{
			for(j=0; j<m_Cells[l].size(); j++)
			{
				TCell * c = m_Cells[l][j];
				c->m_Color = TCell::COLOR_WHITE; 
				c->m_FillComponent = TCell::INVALIDCOMPONENT;
			}
		}
	}

	// Reset 
	{
		unsigned int i;
		for(i=0; i<m_AuxiliaryList.size(); i++)
		{
			m_AuxiliaryList[i] = m_InitialAuxiliaryList;
		}
		for(i=0; i<m_ComponentList.size(); i++)
		{
			m_ComponentList[i] = m_InitialComponentList;
		}
	}
}


void TDeltaOmega::createSpatialSubdivision(const TIndexedOrigins_Vector * p_Io, unsigned int p_Level, TCell * p_Parent)
{
	if(p_Io->size() == 0) return;


	TIndexedOrigins_Vector::const_iterator it;

	// Get component vector for specified level
	if(p_Level == m_Cells.size() ) return;
	else if(p_Level == m_Cells.size()-1 || p_Io->size() <= CELLCOUNT) 
	{
		// If at finest level, do the following...

		unsigned int x = 0;
		if(p_Parent) p_Parent->m_Children.resize( p_Io->size() );
//		m_Cells[p_Level].reserve( m_Cells[p_Level].size() + p_Io->size() );
		for(it = p_Io->begin(); it != p_Io->end(); it++)
		{
			assert(m_Vertex2Cells[*it].size() == p_Level);
			TCell * newcomponent = new TDeltaOmega::TCell(p_Parent, m_Cells[p_Level].size());
			m_Vertex2Cells[*it].push_back( newcomponent );

			//if(p_Parent) p_Parent->m_Children.push_back( newcomponent );
			if(p_Parent) p_Parent->m_Children[x] = newcomponent;
			m_Cells[p_Level].push_back( newcomponent );

			newcomponent->m_Vertices.add( *it );
			x++;
		}

		return; // quit recursion if deeper than specified depth
	}

	if( p_Io->size() <= CELLCOUNT) throw string("io->size() <= CELLCOUNT");


	unsigned int i, n, x;


	// Encode in m_ComponentList which vertices should be filled
	// m_ComponentList[i].m_Component == 0: do not fill
	// m_ComponentList[i].m_Component == 1: may be filled
	// m_ComponentList[i].m_Component == 2: is already filled
	for(i=0; i<m_AuxiliaryList.size(); i++)
	{
		m_AuxiliaryList[i] = m_InitialAuxiliaryList;
	}
	for(i=0; i<m_ComponentList.size(); i++)
	{
		m_ComponentList[i] = m_InitialComponentList;
	}
	
	for(it = p_Io->begin(); it != p_Io->end(); it++) 
	{
		m_ComponentList[*it].m_Component = 1;
	}


	unsigned int FirstCell = m_Cells[p_Level].size();
	TQueue S;
	if(p_Parent) p_Parent->m_Children.resize(CELLCOUNT);
	for(i=0; i<CELLCOUNT; i++)
	{
		TIndex pos = (*p_Io)[ (p_Io->size()/CELLCOUNT)*i ];
		
		TCell * newcomponent = new TDeltaOmega::TCell(p_Parent, m_Cells[p_Level].size());
		m_Vertex2Cells[pos].push_back( newcomponent );

		if(p_Parent) p_Parent->m_Children[i] = newcomponent;
		m_Cells[p_Level].push_back( newcomponent );

		newcomponent->m_Vertices.add(pos); // Add initial vertex
		m_ComponentList[pos].m_Component = 2;

		S.push( pair<float,TIndex>(0.0f, pos) );
		m_AuxiliaryList[pos].m_Distance = 0.0f;
	}

	// Label components by spreading a flood from components start positions
	while(!S.empty())
	{
		const TIndex pidx = S.top().second;
		S.pop();

		const TAdjacencyStruct & padj = m_AdjacencyList[pidx];
		const float & pdist = m_AuxiliaryList[pidx].m_Distance;
		for(n=0; n<padj.m_EdgeCount; n++)
		{
			const TIndex npidx = padj.m_Edges[n].m_ToVertex;

			if( m_ComponentList[npidx].m_Component == 2 )  // may this be filled?
			{
				//if(vertex2compidx[npidx] != v2c[pidx])
				//	m_Cells[p_Level][ v2c[pidx] ]->m_InnerBorder.add(pidx);
			}
			else if( m_ComponentList[npidx].m_Component == 1 )  // may this be filled?
			{
				// Compute distance from p to np and see if distance is less than current distance
				const float & weight = WEIGHTS[ padj.m_Edges[n].m_Weight ];

				const float t = pdist + weight; // Dijkstra
				float & npdist = m_AuxiliaryList[npidx].m_Distance;
				if( t < npdist )
				{
					npdist = t;
					
					if( m_Vertex2Cells[npidx].size() <= p_Level ) m_Vertex2Cells[npidx].push_back(0);
					m_Vertex2Cells[npidx][p_Level] = m_Vertex2Cells[pidx][p_Level];

					m_Vertex2Cells[npidx][p_Level]->m_Vertices.add(npidx);
					
					m_ComponentList[npidx].m_Component = 2;
					S.push( TQueue::value_type(npdist,npidx) );
				}
			}
		}
	}

	// Recursively subdivide components
	{
		for(x=0; x<CELLCOUNT; x++)
		{
			unsigned int cellid = FirstCell + x;
			createSpatialSubdivision(&m_Cells[p_Level][cellid]->m_Vertices, p_Level+1, m_Cells[p_Level][cellid] );
		}
	}
}










float TDeltaOmega::dijkstraSinglePathWithObstructions(TIndex p_Start, TIndex p_End, const TIndexedOrigins_Vector * p_Obstructions, TIndexedOrigins_Vector * p_Io, bool & p_Found)
{
	const bool ASTAR = g_Parameters.m_GeodesicType == TParameters::GEODESICTYPE_ASTAR ? true : false;
	unsigned int n;

	//static vector<TIndex> Visited( m_AdjacencyList.size() );
	//Visited.resize(0);
	vc = 0;

	// Add obstructions
	{
		TIndexedOrigins_Vector::const_iterator it;
		for(it = p_Obstructions->begin(); it != p_Obstructions->end(); it++)
		{
			m_AuxiliaryList[*it].m_IsKnown = true;
			Visited[vc] = *it;
			vc++;
		}
	}

	m_AuxiliaryList[p_Start].m_IsKnown = false;
	m_AuxiliaryList[p_End].m_IsKnown = false;

// Initialize m_AdjacencyList variables for dijkstra path
	//m_AuxiliaryList = m_InitialAuxiliaryList;
	const TCoord3 StartCoord = m_Omega->m_BoundaryIndexField->vidx2coord(p_Start);
	const TCoord3 EndCoord = (p_End != TIndexedOrigins::INVALIDINDEX) ? m_Omega->m_BoundaryIndexField->vidx2coord(p_End) : TCoord3(-1,-1,-1);

	if(!ASTAR) m_AuxiliaryList[p_Start].m_Distance = 0; // Dijkstra
	else m_AuxiliaryList[p_Start].m_Distance = 0; // A*


	static TQueue S;
	S = TQueue();
	S.push( TQueue::value_type(0.0f, p_Start) );
	Visited[vc] = p_Start;
	vc++;

	bool foundEnd = false;
	while(!S.empty())
	{
		const TIndex pidx = S.top().second;
		S.pop();

		if(m_AuxiliaryList[pidx].m_IsKnown) continue;
		if(pidx == p_End) { foundEnd = true; break; }
		m_AuxiliaryList[pidx].m_IsKnown = true;

		const TAdjacencyStruct & padj = m_AdjacencyList[pidx];
		const float & pdist = m_AuxiliaryList[pidx].m_Distance;

//		if(pdist > p_MaximumDistance) { foundEnd = false; break; }

		for(n=0; n<padj.m_EdgeCount; n++)
		{
			const TIndex npidx = padj.m_Edges[n].m_ToVertex;
			if(!m_AuxiliaryList[npidx].m_IsKnown) 
			{
				// Compute distance from p to np and see if distance is less than current distance
				const float & weight = WEIGHTS[ padj.m_Edges[n].m_Weight ];

				const float t = pdist + weight; // Dijkstra
				float & npdist = m_AuxiliaryList[npidx].m_Distance;
				if( t < npdist )
				{
					npdist = t;
					m_AuxiliaryList[npidx].m_Previous = pidx;

					if(!ASTAR)
					{
						S.push( TQueue::value_type(npdist,npidx) );
					}
					else
					{
						const TCoord3 & a = m_Omega->m_BoundaryIndexField->vidx2coord(npidx);
						S.push( TQueue::value_type( npdist + abs(a.x-EndCoord.x) + abs(a.y-EndCoord.y) + abs(a.z-EndCoord.z), npidx) );
					}
					Visited[vc] = npidx;
					vc++;
				}
			}
		}
	}
	const float distance = m_AuxiliaryList[p_End].m_Distance;

	// Return the path
	if(foundEnd && p_Io)
	{
		TIndex pidx = p_End;
		p_Io->add(pidx);
		while(pidx != p_Start)
		{
			pidx = m_AuxiliaryList[pidx].m_Previous;
			if(pidx == TIndexedOrigins::INVALIDINDEX) break;
			p_Io->add(pidx);
		}
	}

	// Reset 
	{
		for(unsigned int i=0; i<vc; i++)
		{
			m_AuxiliaryList[Visited[i]] = m_InitialAuxiliaryList;
		}
	}

	p_Found = foundEnd;

	return distance;
};



unsigned int  TDeltaOmega::floodfill(TIndex p_Index, unsigned int p_FillComponent)
{
	unsigned int n;
	TIndex pidx, npidx;
	static std::queue<TIndex> S;

	unsigned int filled = 0;
	S.push(p_Index);
	m_ComponentList[p_Index].m_Component = p_FillComponent;
	filled++;
	
	while(! S.empty())
	{
		pidx = S.front();
		S.pop();

		const TAdjacencyStruct & padj = m_AdjacencyList[pidx];
		for(n=0; n<padj.m_EdgeCount; n++)
		{	
			npidx = padj.m_Edges[n].m_ToVertex;
			
			if( m_ComponentList[npidx].m_Component == 0 )
			{
				if(!m_ComponentList[npidx].m_NeighborsGeodesic) S.push(npidx);
				m_ComponentList[npidx].m_Component = p_FillComponent;
				filled++;
			}
		}
	}
	return filled;
}







void TDeltaOmega::erode(const TIndexedOrigins_Vector * p_Input, TIndexedOrigins_Vector * p_Output)
{
	// WARNING
	// Should always be called after dilate() because it expects
	// m_AuxiliaryList and m_ComponentList to be filled with stuff
	// Using that:
	// m_AuxiliaryList[npidx].m_Distance != (numeric_limits<float>::max)() for all voxels in p_Input


	// erode works by first doing a dilation, enumerating the different boundary components, and then
	// remove all voxels that do not neighbor a component with a different number

	unsigned int n;

	// Find edge of p_Input, assign 0 to edge voxels, other voxels are of TCell::INVALIDCOMPONENT
	TIndexedOrigins_Vector::const_iterator it;
	TIndexedOrigins_Vector EdgeVoxels;
	EdgeVoxels.reserve(p_Input->size());
	for(it = p_Input->begin(); it != p_Input->end(); it++)
	{
		const TIndex & pidx = *it;
		bool hasNeighbor = false;
		const TAdjacencyStruct & padj = m_AdjacencyList[pidx];
		for(n=0; n<padj.m_EdgeCount; n++)
		{
			const TIndex npidx = padj.m_Edges[n].m_ToVertex;
			if(m_AuxiliaryList[npidx].m_Distance == (numeric_limits<float>::max)() )
			{
				hasNeighbor = true;
				break;
			}
		}
		if(hasNeighbor)
		{
			m_ComponentList[pidx].m_Component = 0;

			EdgeVoxels.push_back(pidx);
		}

		m_AuxiliaryList[pidx].m_Distance = (numeric_limits<float>::max)()/2.0f;			
		m_AuxiliaryList[pidx].m_IsKnown = false;
	}

	// Determine connected components
	// Assign to each voxel at edge a component id and put in narrowband T
	TQueue T;
	{
		unsigned int Component = 1;
		for(int i=0; i<EdgeVoxels.size(); i++)
		{
			if( m_ComponentList[EdgeVoxels[i]].m_Component > 0 ) continue;

			queue<TIndex> S;
			S.push( EdgeVoxels[i] );
			//ComponentSizes.push_back(1);
			m_ComponentList[EdgeVoxels[i]].m_Component = Component;

			while(! S.empty())
			{
				const TIndex pidx = S.front();
				S.pop();

				const TAdjacencyStruct & padj = m_AdjacencyList[pidx];
				for(n=0; n<padj.m_EdgeCount; n++)
				{
					const TIndex npidx = padj.m_Edges[n].m_ToVertex;
					
					if(m_ComponentList[npidx].m_Component == 0)
					{
						m_ComponentList[npidx].m_Component = Component;

						m_AuxiliaryList[npidx].m_Distance = 0.0f;			
						T.push( TQueue::value_type(0.0f, npidx) );

						S.push(npidx);
					}
				}			
			}

			Component++;
		}
	}

	// Propagate front in T inwards
	{
		while(!T.empty())
		{
			const TIndex pidx = T.top().second;
			T.pop();

			if(m_AuxiliaryList[pidx].m_IsKnown) continue;
			m_AuxiliaryList[pidx].m_IsKnown = true;

			const TAdjacencyStruct & padj = m_AdjacencyList[pidx];
			const float & pdist = m_AuxiliaryList[pidx].m_Distance;
			for(n=0; n<padj.m_EdgeCount; n++)
			{
				const TIndex npidx = padj.m_Edges[n].m_ToVertex;
				if(m_AuxiliaryList[npidx].m_Distance != (numeric_limits<float>::max)() )  
				{
					// Compute distance from p to np and see if distance is less than current distance
					const float & weight = WEIGHTS[ padj.m_Edges[n].m_Weight ];

					const float t = pdist + weight; // Dijkstra
					float & npdist = m_AuxiliaryList[npidx].m_Distance;
					if( t < npdist )
					{
						npdist = t;
						m_ComponentList[npidx].m_Component = m_ComponentList[pidx].m_Component;

						T.push( TQueue::value_type(npdist,npidx) );
					}
				}
			}
		}
	}

	// Detect border between different components
	{
		for(it = p_Input->begin(); it != p_Input->end(); it++)
		{
			const TIndex & pidx = *it;
			unsigned int pidxc = m_ComponentList[pidx].m_Component;

			bool hasNeighbor = false;
			const TAdjacencyStruct & padj = m_AdjacencyList[pidx];
			for(n=0; n<padj.m_EdgeCount; n++)
			{
				const TIndex npidx = padj.m_Edges[n].m_ToVertex;
				unsigned int npidxc = m_ComponentList[npidx].m_Component;

				if(npidxc != TCell::INVALIDCOMPONENT && pidxc < npidxc)
				{
					hasNeighbor = true;
					break;
				}
			}
			if(hasNeighbor)
			{
				p_Output->add(pidx);
			}
		}
	}


	// Reset 
	{
		for(unsigned int i=0; i<vc; i++)
		{
			m_ComponentList[Visited[i]] = m_InitialComponentList;
			m_AuxiliaryList[Visited[i]] = m_InitialAuxiliaryList;
		}
	}	
}


void TDeltaOmega::dilate(float p_Amount, const TIndexedOrigins_Vector * p_Input, TIndexedOrigins_Vector * p_Output, bool p_Reset)
{
	unsigned int n;

	static TQueue S;
	S = TQueue();
	vc = 0;

	TIndexedOrigins_Vector::const_iterator it;
	for(it = p_Input->begin(); it != p_Input->end(); it++)
	{
		m_AuxiliaryList[*it].m_Distance = 0.0f;
		S.push( TQueue::value_type(0.0f, *it) );
		Visited[vc] = *it; 
		vc++;
	}

	while(!S.empty())
	{
		const TIndex pidx = S.top().second;
		S.pop();

		if(m_AuxiliaryList[pidx].m_IsKnown) continue;
		m_AuxiliaryList[pidx].m_IsKnown = true;
		p_Output->push_back(pidx);

		const TAdjacencyStruct & padj = m_AdjacencyList[pidx];
		const float & pdist = m_AuxiliaryList[pidx].m_Distance;
		for(n=0; n<padj.m_EdgeCount; n++)
		{
			const TIndex npidx = padj.m_Edges[n].m_ToVertex;
			if(!m_AuxiliaryList[npidx].m_IsKnown) 
			{
				// Compute distance from p to np and see if distance is less than current distance
				const float & weight = WEIGHTS[ padj.m_Edges[n].m_Weight ];

				const float t = pdist + weight; // Dijkstra
				float & npdist = m_AuxiliaryList[npidx].m_Distance;
				if( t < npdist && t < p_Amount )
				{
					npdist = t;
					Visited[vc] = npidx; 
					vc++;

					S.push( TQueue::value_type(npdist,npidx) );
				}
			}
		}
	}

	// Reset 
	if(p_Reset)
	{
		for(unsigned int i=0; i<vc; i++)
		{
			m_AuxiliaryList[Visited[i]] = m_InitialAuxiliaryList;
		}
	}
}

bool TDeltaOmega::isLoop(const TIndexedOrigins_Vector * p_Io, int & p_Components, vector<shared_ptr<TIndexedOrigins_Vector> > * p_Boundaries)
{
	// WARNING
	// Should always be called after dilate() because it expects
	// m_AuxiliaryList and m_ComponentList to be filled
	unsigned int n;

	const TIndexedOrigins_Vector * Dilated = p_Io;

	// Find edge of dillated path-set
	TIndexedOrigins_Vector::const_iterator it;
	TIndexedOrigins_Vector EdgeVoxels;
	EdgeVoxels.reserve(p_Io->size());
	for(it = Dilated->begin(); it != Dilated->end(); it++)
	{
		const TIndex & pidx = *it;
		bool hasNeighbor = false;
		const TAdjacencyStruct & padj = m_AdjacencyList[pidx];
		for(n=0; n<padj.m_EdgeCount; n++)
		{
			const TIndex npidx = padj.m_Edges[n].m_ToVertex;
			if(m_AuxiliaryList[npidx].m_Distance == (numeric_limits<float>::max)() )
			{
				hasNeighbor = true;
				break;
			}
		}
		if(hasNeighbor)
		{
			m_ComponentList[pidx].m_Component = 0;

			EdgeVoxels.push_back(pidx);
		}
	}

	// Count connected boundary components
	vector<int> ComponentSizes;
	int Component = 1;
	{
		ComponentSizes.reserve(16);
		ComponentSizes.push_back(0);
		if(p_Boundaries) p_Boundaries->push_back( shared_ptr<TIndexedOrigins_Vector>(static_cast<TIndexedOrigins_Vector*>(0)) );
		for(unsigned int i=0; i<EdgeVoxels.size(); i++)
		{
			if( m_ComponentList[EdgeVoxels[i]].m_Component > 0 ) continue;

			queue<TIndex> S;
			S.push( EdgeVoxels[i] );
			ComponentSizes.push_back(1);
			if(p_Boundaries) p_Boundaries->push_back( shared_ptr<TIndexedOrigins_Vector>( new TIndexedOrigins_Vector() ) );
			m_ComponentList[EdgeVoxels[i]].m_Component = Component;
 
			while(! S.empty())
			{
				const TIndex pidx = S.front();
				if(p_Boundaries) (*p_Boundaries)[Component]->add(pidx);
				S.pop();

				const TAdjacencyStruct & padj = m_AdjacencyList[pidx];
				for(n=0; n<padj.m_EdgeCount; n++)
				{
					const TIndex npidx = padj.m_Edges[n].m_ToVertex;
					
					if(m_ComponentList[npidx].m_Component == 0)
					{
						m_ComponentList[npidx].m_Component = Component;
						ComponentSizes[Component] = ComponentSizes[Component] + 1;
						if(p_Boundaries) (*p_Boundaries)[Component]->add(npidx);

						S.push(npidx);
					}
				}			
			}

			Component++;
		}
	}

	Component -= 1;

	const bool Debug = g_Settings.m_DebugOutput;
	if(Debug)
	{
		*g_Log << "Performing TDeltaOmega::isLoop\n";
	}

	if(g_Parameters.m_DiscardSmallComponentBoundaries)
	{
		for(unsigned int i=1; i<ComponentSizes.size(); i++)
		{
			if(Debug) *g_Log << "\tBoundary component: " << i << ", size: " << ComponentSizes[i];

			if(ComponentSizes[i] < g_Parameters.m_DiscardSmallComponentBoundaries)
			{
				Component -= 1;
				if(Debug) *g_Log << ", removed because of heuristic";
			}

			if(Debug) *g_Log << "\n";
		}

		if(p_Boundaries) 
		{
			vector<shared_ptr<TIndexedOrigins_Vector> > NewBoundaries;
			NewBoundaries.push_back( (*p_Boundaries)[0] );
			for(unsigned int i=1; i<ComponentSizes.size(); i++)
			{
				if(ComponentSizes[i] < g_Parameters.m_DiscardSmallComponentBoundaries)
				{
				}
				else
				{
					NewBoundaries.push_back( (*p_Boundaries)[i] );
				}
			}
			*p_Boundaries = NewBoundaries;
		}
	}

	p_Components = Component;


	// Reset 
	{
		for(unsigned int i=0; i<vc; i++)
		{
			m_ComponentList[Visited[i]] = m_InitialComponentList;
			m_AuxiliaryList[Visited[i]] = m_InitialAuxiliaryList;
		}
	}


	// See if loop
	return Component >= 2;
}



bool TDeltaOmega::isLoop2(const TIndexedOrigins_Vector * p_Eft, const TIndexedOrigins_Vector * p_Path, int & p_Components)
{
	TTypedFieldInterface<unsigned char> * debug = static_cast<TTypedFieldInterface<unsigned char>*>(  g_Mediator.getCurrentLayerSet()->newField(TType::TYPE_UCHAR, m_Omega->m_BoundaryIndexField, "debug1" ) );
	debug->clear();
	unsigned int x;
	for(x=0; x<p_Path->size(); x++)
	{
		debug->wvaluex( (*p_Path)[x] ) = 1;
	}
	debug->getLayer()->onFieldChanged();
	
	unsigned int CompId = 2;
	for(x=0; x<debug->getMaxIndex(); x++)
	{
		if(debug->vvaluex(x) != 0) continue;

		TDistancePropagationParameters par;	
		TFloatFilter* domainfilter = static_cast<TFloatFilter*>(debug->getLayer()->m_Filter.get());
		domainfilter->m_LowerValue = -0.1f;
		domainfilter->m_UpperValue = +0.1f;

		par.m_DomainFilter = debug->getLayer()->m_Filter.get();
		set<unsigned int> targets;
		TTrueFilter filter;
		par.m_TargetFilter = &filter;
		par.m_OutputTargets = &targets;
		par.m_Start = x;
		distancePropagation(par);
		std::set<unsigned int>::iterator it;
		for(it = targets.begin(); it != targets.end(); it++)
		{
			debug->wvaluex( *it ) = CompId;
		}
		CompId++;
	}
	debug->getLayer()->onFieldChanged();

	return true;

/*
	TIndexedOrigins_Vector DilatedPathSet;
	dilate(g_Parameters.m_DilationDistance, p_Path, &DilatedPathSet);

	TTypedFieldInterface<unsigned char> * debug = static_cast<TTypedFieldInterface<unsigned char>*>(  g_Mediator.getCurrentLayerSet()->newField(TType::TYPE_UCHAR, m_Omega->m_BoundaryIndexField, "debug1" ) );
//	TTypedFieldInterface<unsigned char> * debug = static_cast<TTypedFieldInterface<unsigned char>*>(  g_Mediator.getCurrentLayerSet()->getField("debug1" ) );
	if(g_Settings.m_DebugOutput) debug->clear();
	debug->getLayer()->m_Filter->m_LowerValue = 0.5f;
	debug->getLayer()->m_Filter->m_UpperValue = 1.5f;

//	TTypedFieldInterface<float> * debug2 = static_cast<TTypedFieldInterface<float>*>(  g_Mediator.getCurrentLayerSet()->getField("debug2" ) );
	TTypedFieldInterface<float> * debug2 = static_cast<TTypedFieldInterface<float>*>(  g_Mediator.getCurrentLayerSet()->newField(TType::TYPE_FLOAT, m_Omega->m_BoundaryIndexField, "debug2" ) );
	if(g_Settings.m_DebugOutput) debug2->clear();
	
	unsigned int x;
	for(x=0; x<DilatedPathSet.size(); x++) debug->wvaluex(DilatedPathSet[x]) = 1;

	TDistancePropagationParameters par;
	par.m_DomainFilter = debug->getLayer()->m_Filter.get();
	par.m_Start = p_Eft->at(0);
	par.m_OutputDistance = debug2;
	distancePropagation(par);

	// Check #connected components of several level sets
	const float Min = g_Parameters.m_DilationDistance * 2.0f;
	const float Max = par.m_OutputEndDistance - g_Parameters.m_DilationDistance * 2.0f;
	if(Max < Min) return false;

	const float stepsize = g_Parameters.m_DilationDistance * 2.0f;
	unsigned int steps = (Max-Min) / stepsize;
	vector<unsigned int> Components;
	for(unsigned int y=0; y<steps+2; y++)
	{
		float l,u;
		if(y == 0) { l = 0.0f; u = Min; }
		else
		{
			l = Min + stepsize * (y-1);
			u = Min + stepsize * (y-0);
			if(y==steps+1) u = par.m_OutputEndDistance;
		}

		TIndexedOrigins_Vector LevelSet;
		for(x=0; x<DilatedPathSet.size(); x++) 
		{
			const float dist = debug2->vvaluex(DilatedPathSet[x]);
			if( dist >= l && dist <= u ) 
			{
				LevelSet.push_back( DilatedPathSet[x] );
				if(g_Settings.m_DebugOutput) debug->wvaluex(DilatedPathSet[x]) = y+1;
			}
		}

		vector<set<unsigned int> > Sizes;
		findConnectedComponents(&LevelSet, &Sizes);
		Components.push_back(Sizes.size()-1);
	}

	if(g_Settings.m_DebugOutput) *g_Log << "Components: ";
	unsigned int stage = 0;
	unsigned int maxcomp = 0;
	for(x=0; x<Components.size(); x++)
	{
		if(g_Settings.m_DebugOutput) *g_Log << Components[x] << ", ";
		if(stage == 0 && Components[x] == 1) stage++;
		else if(stage == 1 && Components[x] >= 2) stage++;
		else if(stage == 2 && Components[x] == 1) stage++;

		if(stage == 2) maxcomp = max(maxcomp, Components[x]);
	}
	if(g_Settings.m_DebugOutput) *g_Log << "\n";

	// Reset 
	if(!g_Settings.m_DebugOutput) 
	{
		for(x=0; x<DilatedPathSet.size(); x++) 
		{
			debug->wvaluex(DilatedPathSet[x]) = 0;
			debug2->wvaluex(DilatedPathSet[x]) = 0.0f;
		}
	}


	if(g_Settings.m_DebugOutput) 
	{	
		*g_Log << "maxcomp: " << maxcomp << "\n";
		debug->getLayer()->onFieldChanged();
		debug2->getLayer()->onFieldChanged();
	}

	p_Components = stage == 3 ? maxcomp : 1;
	return stage == 3;
*/
}

/*
	static TQueue2 S;
	S = TQueue2();
	S.insert( TQueue::value_type(0.0f, p_Eft->at(0) ) );
	m_AuxiliaryList[p_Eft->at(0)].m_Distance = 0.0f;
	
	unsigned int MaxComponents = 0;
	float olddistance = 10.0f;
	while(!S.empty())
	{
		const float distance = S.begin()->first;
		const TIndex pidx = S.begin()->second;
		S.erase( S.begin() );

		if(m_AuxiliaryList[pidx].m_IsKnown) continue;
		m_AuxiliaryList[pidx].m_IsKnown = true;

		// We check the front of the propagation: how many connected components does it have?
		if(
			distance > olddistance + 2.0f 
			&& MaxComponents == 0
			)
		{
			TTypedFieldInterface<unsigned char> * debug = static_cast<TTypedFieldInterface<unsigned char>*>(  g_Mediator.getCurrentLayerSet()->getField("debug3" ) );
			olddistance = distance;
			TIndexedOrigins_Vector Front;
			TQueue2::iterator it;
			for(it = S.begin(); it != S.end(); it++)
			{
				Front.push_back( it->second );
				if(DoOnce && debug) debug->wvaluex( it->second ) = 1;
			}
			vector<set<unsigned int> > Sizes;
			findConnectedComponents(&Front, &Sizes, false);
			if( Sizes.size()-1 > MaxComponents )
			{
				MaxComponents = Sizes.size()-1;
			}
		}

		const TAdjacencyStruct & padj = m_AdjacencyList[pidx];
		const float & pdist = m_AuxiliaryList[pidx].m_Distance;
		for(unsigned int n=0; n<padj.m_EdgeCount; n++)
		{
			const TIndex npidx = padj.m_Edges[n].m_ToVertex;
			if(m_AuxiliaryList[npidx].m_Previous != (numeric_limits<unsigned int>::max)() ) // only walk on path
			{
				const float & weight = WEIGHTS[ padj.m_Edges[n].m_Weight ];

				const float t = pdist + weight; // Dijkstra
				float & npdist = m_AuxiliaryList[npidx].m_Distance;
				if( t < npdist )
				{
					npdist = t;
					Visited[vc] = npidx; 
					vc++;

					S.insert( TQueue2::value_type(npdist,npidx) );
				}
			}
		}
	}

	// Reset 
	{
		unsigned int i;
		for(i=0; i<vc; i++)
		{
			m_AuxiliaryList[Visited[i]] = m_InitialAuxiliaryList;
		}
		for(i=0; i<vc; i++)
		{
			m_ComponentList[Visited[i]] = m_InitialComponentList[Visited[i]];
		}
	}
	if( MaxComponents >= 2 ) DoOnce = false;

	p_Components = MaxComponents;
	return MaxComponents >= 2;
}
*/


























void TDeltaOmega::floodfillUsingCells(TCell * p_Cell, unsigned int p_FillComponent)
{
	static vector<TCell*>::iterator it, jt, kt;
	static std::queue<TCell*> S;
	static std::queue<TCell*> T;
	S.push(p_Cell);

	TCell * t;
	while(!S.empty())
	{
		TCell * const c = S.front();
		S.pop();

		// See if c or ancestor has not yet been processed
		unsigned int fc = c->getFillComponent();
		if(fc != TCell::INVALIDCOMPONENT) continue; // ancestor already processed => c already processed

		// Fill it
		c->m_FillComponent = p_FillComponent;

		// Now go to neighbors
		if( c->m_Color == TCell::COLOR_WHITE ) 
		{
			for(it = c->m_Neighbors.begin(); it != c->m_Neighbors.end(); it++)
			{
				if( (*it)->m_Color == TCell::COLOR_WHITE) // if white neighbor...
				{
					// See if we can traverse up to another white node
					t = *it;
//					while(t->m_Parent != 0 && t->m_Parent->m_Color == TCell::COLOR_WHITE) 
					while(t->m_Parent != 0 && t->m_Parent->m_Color == TCell::COLOR_WHITE && t->m_FillComponent == TCell::INVALIDCOMPONENT) 
					{
						t = t->m_Parent;
					}

					// Push on stack S
					S.push(t);									
				}
				else if( (*it)->m_Color == TCell::COLOR_NEIGHBOR) // if mergedgeodesic neighbor 
				{
					S.push(*it);									
				}
				else if( (*it)->m_Color == TCell::COLOR_GRAY) // if gray neighbor...
				{
					// Now push on S all children of *it that neighbor cell c

					T.push(*it); // *it is a neighbor of c, we know that *it is gray
					while(!T.empty())
					{
						TCell * d = T.front();
						T.pop();

//						if(d->m_Color == TCell::COLOR_WHITE || d->m_Color == TCell::COLOR_NEIGHBOR)
						if(d->m_Color != TCell::COLOR_GRAY)
						{
							S.push(d);
						}
						else
						{
							// Push children of d on stack, but only those that neighbor cell c
							for(jt = d->m_Children.begin(); jt != d->m_Children.end(); jt++)
							{
								// See if *jt neighbors d, do this by considering *jt neigbhors and seeing if 
								// one of them is a child of c
								for(kt = (*jt)->m_Neighbors.begin(); kt != (*jt)->m_Neighbors.end(); kt++)
								{
									const TIndex & idx = (*kt)->m_Vertices[0];
									if(c->m_Level < m_Vertex2Cells[idx].size())
									{
										if( m_Vertex2Cells[idx][c->m_Level] == c )
										{
											T.push(*jt);
										}
									}

								}
							}
						} 
					}
				}
				else
				{
				}
			}
		}	
		else if( c->m_Color == TCell::COLOR_NEIGHBOR ) 
		{
			// neighbor may not be chosen

		}	
	}
}


unsigned int TDeltaOmega::computeCollapseUsingCells(const TIndexedOrigins_Vector * p_Geodesic, TIndexedOrigins_Cells * p_Result)
{
	unsigned int i,j,l;
	const bool DEBUG = g_Parameters.m_Debugging;

	{
		// Debugging: check of alle kleuren white zijn en fillcomponent 0 zijn
		if(DEBUG)
		{
			// Reset all colors to white
			for(l=0; l<m_Cells.size(); l++)
			{
				for(j=0; j<m_Cells[l].size(); j++)
				{
					TCell * const c = m_Cells[l][j];
					if( c->m_Color != TCell::COLOR_WHITE ) 
					{
						assert(false); throw string("c->m_Color != TCell::COLOR_WHITE");
					}
					if( c->m_FillComponent != TCell::INVALIDCOMPONENT ) 
					{
						*g_Log << "Error\n";
						*g_Log << "m_Level = " << c->m_Level << "\n";
						*g_Log << "m_Index = " << c->m_Index << "\n";
						*g_Log << "m_FillComponent = " << c->m_FillComponent << "\n";
						*g_Log << "m_Color = " << c->m_Color << "\n";
						assert(false); 
						throw string("c->m_FillComponent != TCell::INVALIDCOMPONENT");
					}
				}
			}
		}
	}
		

	TIndexedOrigins_Vector::const_iterator it;

	// Set neighbors of geodesic
	for(it = p_Geodesic->begin(); it != p_Geodesic->end(); it++) 
	{
		const TAdjacencyStruct & padj = m_AdjacencyList[*it];
		for(i=0; i<padj.m_EdgeCount; i++)
		{
			const TIndex & idx = padj.m_Edges[i].m_ToVertex;

			for(l=0; l<m_Vertex2Cells[idx].size()-1; l++)
			{
				m_Vertex2Cells[idx][l]->m_Color = TCell::COLOR_GRAY;
			}

			m_Vertex2Cells[idx][m_Vertex2Cells[idx].size()-1]->m_Color = TCell::COLOR_NEIGHBOR;
		}
	}

			

	// Make geodesic gray (overwrite) and fillcomponent = 0
	for(it = p_Geodesic->begin(); it != p_Geodesic->end(); it++) 
	{
		for(l=0; l<m_Vertex2Cells[*it].size(); l++)
		{
			m_Vertex2Cells[*it][l]->m_Color = TCell::COLOR_GRAY;
		}
		m_Vertex2Cells[*it][m_Vertex2Cells[*it].size()-1]->m_FillComponent = 0;
	}



	unsigned int FillComponent = 1; // 0 is reserved by geodesic
	// Start filling
	{
		std::queue<TCell*> S;
		vector<TCell*>::iterator it;
		for(j=0; j<m_Cells[0].size(); j++) S.push(m_Cells[0][j]);
		while( !S.empty() )
		{
			TCell * const c = S.front(); 
			S.pop();
			if( c->m_Color == TCell::COLOR_GRAY && c->m_Children.size() != 0 ) // recurse
			{
				for(it = c->m_Children.begin(); it != c->m_Children.end(); it++) 
				{
					S.push(*it);
				}
			}
			else
			{
				//if(c->m_FillComponent == TCell::INVALIDCOMPONENT)
				if(c->getFillComponent() == TCell::INVALIDCOMPONENT) // sneller?
				{
					if(c->m_Color == TCell::COLOR_NEIGHBOR)
					{
						// skip
					}
					else
					{
						floodfillUsingCells(c, FillComponent++);
					}
				}
			}
		}
	}


	// Determine sizes of components
	vector<unsigned int> ComponentSize(FillComponent, 0);
	{
		std::queue<const TCell*> S;
		vector<TCell*>::const_iterator it;
		for(j=0; j<m_Cells[0].size(); j++) S.push(m_Cells[0][j]);
		while( !S.empty() )
		{
			const TCell * const c = S.front(); 
			S.pop();
			if( c->m_Color == TCell::COLOR_GRAY && c->m_Children.size() != 0 )
			//if( c->m_FillComponent == TCell::INVALIDCOMPONENT ) // recurse
			{
				for(it = c->m_Children.begin(); it != c->m_Children.end(); it++) 
				{
					S.push(*it);
				}
			}
			else
			{
				// Sometimes FillComponent can be TCell::INVALIDCOMPONENT
				//if(c->getFillComponent() == TCell::INVALIDCOMPONENT) 
				//	throw string("c->getFillComponent() == TCell::INVALIDCOMPONENT");

				if(c->getFillComponent() != TCell::INVALIDCOMPONENT) 
				{
					ComponentSize[c->m_FillComponent] += c->m_Vertices.size();
				}

			}
		}
	}

	{
		// Determine component of maximum size
		unsigned int maxid = 0;
		for(i=0; i<ComponentSize.size(); i++)
		{
			if( ComponentSize[i] > ComponentSize[maxid] ) maxid = i;
		}

		p_Result->clear();
		std::queue<TCell*> S;
		vector<TCell*>::iterator it;
		for(j=0; j<m_Cells[0].size(); j++) S.push(m_Cells[0][j]);
		while( !S.empty() )
		{
			TCell * const c = S.front(); 
			S.pop();

			if( c->m_Color == TCell::COLOR_GRAY && c->m_Children.size() != 0 ) // recurse
			{
				for(it = c->m_Children.begin(); it != c->m_Children.end(); it++) 
				{
					S.push(*it);
				}
			}
			else
			{
				if(DEBUG && c->m_Color != TCell::COLOR_NEIGHBOR &&  c->m_FillComponent == TCell::INVALIDCOMPONENT) 
					throw string("c->m_FillComponent != TCell::INVALIDCOMPONENT (2)");

				if(	c->m_FillComponent != maxid 
					&& c->m_FillComponent != 0
					)
				{
					p_Result->add(c);
				}
			}

			c->m_Color = TCell::COLOR_WHITE;
			c->m_FillComponent = TCell::INVALIDCOMPONENT;
		}
	}

	return FillComponent;
}



void TDeltaOmega::computeComponentSetUsingCells(const TIndexedOrigins_Vector * p_Geodesic, TComponentSet * p_Result, bool p_FillShortestPathSet)
{
	unsigned int j,l;
	const bool DEBUG = g_Parameters.m_Debugging;

	{
		// Debugging: check of alle kleuren white zijn en fillcomponent 0 zijn
		if(DEBUG)
		{
			// Reset all colors to white
			for(l=0; l<m_Cells.size(); l++)
			{
				for(j=0; j<m_Cells[l].size(); j++)
				{
					TCell * const c = m_Cells[l][j];
					if( c->m_Color != TCell::COLOR_WHITE ) 
					{
						assert(false); throw string("c->m_Color != TCell::COLOR_WHITE");
					}
					if( c->m_FillComponent != TCell::INVALIDCOMPONENT ) 
					{
						*g_Log << "Error\n";
						*g_Log << "m_Level = " << c->m_Level << "\n";
						*g_Log << "m_Index = " << c->m_Index << "\n";
						*g_Log << "m_FillComponent = " << c->m_FillComponent << "\n";
						*g_Log << "m_Color = " << c->m_Color << "\n";
						assert(false); 
						throw string("c->m_FillComponent != TCell::INVALIDCOMPONENT");
					}
				}
			}
		}
	}
		

	TIndexedOrigins_Vector::const_iterator it;

			

	// Make geodesic gray (overwrite) and fillcomponent = 0
	for(it = p_Geodesic->begin(); it != p_Geodesic->end(); it++) 
	{
		for(l=0; l<m_Vertex2Cells[*it].size(); l++)
		{
			m_Vertex2Cells[*it][l]->m_Color = TCell::COLOR_GRAY;
		}
		m_Vertex2Cells[*it][m_Vertex2Cells[*it].size()-1]->m_FillComponent = 0;
	}



	unsigned int FillComponent = 1; // 0 is reserved by geodesic
	// Start filling
	{
		std::queue<TCell*> S;
		vector<TCell*>::iterator it;
		for(j=0; j<m_Cells[0].size(); j++) S.push(m_Cells[0][j]);
		while( !S.empty() )
		{
			TCell * const c = S.front(); 
			S.pop();
			if( c->m_Color == TCell::COLOR_GRAY && c->m_Children.size() != 0 ) // recurse
			{
				for(it = c->m_Children.begin(); it != c->m_Children.end(); it++) 
				{
					S.push(*it);
				}
			}
			else
			{
				if(c->getFillComponent() == TCell::INVALIDCOMPONENT) // kan dit sneller: getfillcomponent doet recursie
				{
					if(c->m_Color == TCell::COLOR_NEIGHBOR)
					{
						// skip
						// Todo: test
					}
					else
					{
						floodfillUsingCells(c, FillComponent++);
					}
				}
			}
		}
	}

	// Create collapse array
	TComponentSet & Collapses = *p_Result;
	Collapses.m_Cells.clear();
	{
		for(int i=0; i<FillComponent; i++)
		{
			shared_ptr<TIndexedOrigins_Cells> io( new TIndexedOrigins_Cells() );
			Collapses.m_Cells.push_back( io );
		}
	}

	// Determine sizes of components
	vector<unsigned int> ComponentSize(FillComponent, 0);
	{
		std::queue<const TCell*> S;
		vector<TCell*>::const_iterator it;
		for(j=0; j<m_Cells[0].size(); j++) S.push(m_Cells[0][j]);
		while( !S.empty() )
		{
			const TCell * const c = S.front(); 
			S.pop();
			if( c->m_Color == TCell::COLOR_GRAY && c->m_Children.size() != 0 )
			//if( c->m_FillComponent == TCell::INVALIDCOMPONENT ) // recurse
			{
				for(it = c->m_Children.begin(); it != c->m_Children.end(); it++) 
				{
					S.push(*it);
				}
			}
			else
			{
				// Sometimes FillComponent can be TCell::INVALIDCOMPONENT
				//if(c->getFillComponent() == TCell::INVALIDCOMPONENT) 
				//	throw string("c->getFillComponent() == TCell::INVALIDCOMPONENT");

				if(c->getFillComponent() != TCell::INVALIDCOMPONENT) 
				{
					ComponentSize[c->m_FillComponent] += c->m_Vertices.size();
				}

			}
		}
	}



	// Fill shortest path set (spset) and its neighbors with other component id than 0 if not already done so
	// that spset does not form a separate component.
	if(p_FillShortestPathSet)
	{
		unsigned int maxid = 0;
		for(unsigned int i=0; i<ComponentSize.size(); i++)
		{
			if( ComponentSize[i] > ComponentSize[maxid] ) maxid = i;
		}

		static TQueue S;
		S = TQueue();
		vc = 0;

		// Find border of spset and assign componentid != 0 of neighboring component
		unsigned int n;
		TIndexedOrigins_Vector::const_iterator jt;
		for(jt = p_Geodesic->begin(); jt != p_Geodesic->end(); jt++) 
		{
			const TIndex & idx = *jt;
			const TAdjacencyStruct & padj = m_AdjacencyList[idx];
			unsigned int Component = 0;
			for(n=0; n<padj.m_EdgeCount; n++)
			{
				const TIndex & npidx = padj.m_Edges[n].m_ToVertex;
				Component = m_Vertex2Cells[npidx][m_Vertex2Cells[npidx].size()-1]->getFillComponent();
				if(Component != 0 && Component != TCell::INVALIDCOMPONENT) 
				{
					break;
				}
			}

			if(Component != 0)
			{
				float d = 0.0f;
				m_AuxiliaryList[idx].m_Distance = d;			
				S.push( TQueue::value_type(d, idx) );
			}
			m_ComponentList[idx].m_Component = Component;
			Visited[vc] = idx; 
			vc++;
		}


		bool foundEnd = false;
		while(!S.empty())
		{
			const TIndex pidx = S.top().second;
			S.pop();

			if(m_AuxiliaryList[pidx].m_IsKnown) continue;
			m_AuxiliaryList[pidx].m_IsKnown = true;

			const TAdjacencyStruct & padj = m_AdjacencyList[pidx];
			const float & pdist = m_AuxiliaryList[pidx].m_Distance;
			for(n=0; n<padj.m_EdgeCount; n++)
			{
				const TIndex npidx = padj.m_Edges[n].m_ToVertex;
				if(m_ComponentList[npidx].m_Component != TCell::INVALIDCOMPONENT)  
				{
					// Compute distance from p to np and see if distance is less than current distance
					const float & weight = WEIGHTS[ padj.m_Edges[n].m_Weight ];

					const float t = pdist + weight; // Dijkstra
					float & npdist = m_AuxiliaryList[npidx].m_Distance;
					if( t < npdist )
					{
						npdist = t;
						m_ComponentList[npidx].m_Component = m_ComponentList[pidx].m_Component;

						S.push( TQueue::value_type(npdist,npidx) );
					}
				}
			}
		}

		{
			const TFloodfillStruct s = m_InitialComponentList;
			const TAuxiliaryStruct t = m_InitialAuxiliaryList;
			for(unsigned int i=0; i<vc; i++)
			{
				const TIndex & idx = Visited[i];
				m_Vertex2Cells[idx][m_Vertex2Cells[idx].size()-1]->m_FillComponent = m_ComponentList[idx].m_Component;		

				m_ComponentList[idx] = s;
				m_AuxiliaryList[idx] = t;
			}
		}


	}


	// Fill collapse array
	{
		std::queue<TCell*> S;
		vector<TCell*>::iterator it;
		for(j=0; j<m_Cells[0].size(); j++) S.push(m_Cells[0][j]);
		while( !S.empty() )
		{
			TCell * const c = S.front(); 
			S.pop();

			if( c->m_Color == TCell::COLOR_GRAY && c->m_Children.size() != 0 ) // recurse
			{
				for(it = c->m_Children.begin(); it != c->m_Children.end(); it++) 
				{
					S.push(*it);
				}
			}
			else
			{
				if(DEBUG && c->m_Color != TCell::COLOR_NEIGHBOR &&  c->m_FillComponent == TCell::INVALIDCOMPONENT) 
					throw string("c->m_FillComponent != TCell::INVALIDCOMPONENT (2)");

				if(	c->m_FillComponent != TCell::INVALIDCOMPONENT)
				{
					Collapses.m_Cells[c->m_FillComponent]->add(c);
				}
			}

			c->m_Color = TCell::COLOR_WHITE;
			c->m_FillComponent = TCell::INVALIDCOMPONENT;
		}
	}

}


void TDeltaOmega::computeComponentSetUsingCells2(const TIndexedOrigins_Vector * p_Geodesic, TComponentSet * p_Result)
{
// Difference to version 1: also assigns CellBorders to Cells

	unsigned int j,l;
	const bool DEBUG = g_Parameters.m_Debugging;


	TIndexedOrigins_Vector::const_iterator it;

	// Make geodesic gray (overwrite) and fillcomponent = 0
	for(it = p_Geodesic->begin(); it != p_Geodesic->end(); it++) 
	{
		const unsigned int SIZE = m_Vertex2Cells[*it].size();
		for(l=0; l<SIZE; l++)
		{
			m_Vertex2Cells[*it][l]->m_Color = TCell::COLOR_GRAY;
		}
		
		assert(m_Vertex2Cells[*it][SIZE-1] != 0);
		m_Vertex2Cells[*it][SIZE-1]->m_FillComponent = 0;
	}



	unsigned int FillComponent = 1; // 0 is reserved by geodesic
	// Start filling
	{
		std::queue<TCell*> S;
		vector<TCell*>::iterator it;
		for(j=0; j<m_Cells[0].size(); j++) S.push(m_Cells[0][j]);
		while( !S.empty() )
		{
			TCell * const c = S.front(); 
			S.pop();
			if( c->m_Color == TCell::COLOR_GRAY && c->m_Children.size() != 0 ) // recurse
			{
				for(it = c->m_Children.begin(); it != c->m_Children.end(); it++) 
				{
					S.push(*it);
				}
			}
			else
			{
				if(c->getFillComponent() == TCell::INVALIDCOMPONENT) // kan dit sneller: getfillcomponent doet recursie
				{
					if(c->m_Color == TCell::COLOR_NEIGHBOR)
					{
						// skip
						// Todo: test
					}
					else
					{
						floodfillUsingCells(c, FillComponent++);
					}
				}
			}
		}
	}

	// Create collapse array
	TComponentSet & Collapses = *p_Result;
	Collapses.m_Cells.clear();
	{
		for(int i=0; i<FillComponent; i++)
		{
			shared_ptr<TIndexedOrigins_Cells> io( new TIndexedOrigins_Cells() );
			Collapses.m_Cells.push_back( io );
		}
	}



	// Determine sizes of components
	vector<unsigned int> ComponentSize(FillComponent, 0);
	{
		std::queue<const TCell*> S;
		vector<TCell*>::const_iterator it;
		for(j=0; j<m_Cells[0].size(); j++) S.push(m_Cells[0][j]);
		while( !S.empty() )
		{
			const TCell * const c = S.front(); 
			S.pop();
			if( c->m_Color == TCell::COLOR_GRAY && c->m_Children.size() != 0 )
			//if( c->m_FillComponent == TCell::INVALIDCOMPONENT ) // recurse
			{
				for(it = c->m_Children.begin(); it != c->m_Children.end(); it++) 
				{
					S.push(*it);
				}
			}
			else
			{
				// Sometimes FillComponent can be TCell::INVALIDCOMPONENT
				//if(c->getFillComponent() == TCell::INVALIDCOMPONENT) 
				//	throw string("c->getFillComponent() == TCell::INVALIDCOMPONENT");

				if(c->getFillComponent() != TCell::INVALIDCOMPONENT) 
				{
					ComponentSize[c->m_FillComponent] += c->m_Vertices.size();
				}

			}
		}
	}

	/*
	unsigned int maxid = 0;
	for(unsigned int i=0; i<ComponentSize.size(); i++)
	{
		if( ComponentSize[i] > ComponentSize[maxid] ) maxid = i;
	}
	*/




	// Assign borders
	{
		TIndexedOrigins_Vector Boundary;
		Boundary.resize(0);
		determineBoundary(p_Geodesic, &Boundary);

		vector<set<unsigned int> > Boundaries;
		findConnectedComponents(&Boundary, &Boundaries);
		set<unsigned int>::iterator it;

		//Collapses.m_Borders.push_back( 
		for(unsigned int x=1; x<Boundaries.size(); x++)
		{
			unsigned int NeighborComp = 0;
			for(it = Boundaries[x].begin(); it != Boundaries[x].end(); it++)
			{
				unsigned int idx = *it;
				
				// See what component it neighbors
				// It is possible that a border neighbors multiple components
				// In that case, take the largest component
				const TAdjacencyStruct & padj = m_AdjacencyList[idx];
				bool foundneighbor = false;
				for(unsigned int n=0; n<padj.m_EdgeCount; n++)
				{
					const TIndex & npidx = padj.m_Edges[n].m_ToVertex;

					unsigned int fillcomp = m_Vertex2Cells[npidx][m_Vertex2Cells[npidx].size()-1]->m_FillComponent;
					if(fillcomp != 0 && fillcomp != (numeric_limits<unsigned int>::max)() ) 
					{
						if( NeighborComp == 0
							|| ComponentSize[fillcomp] > ComponentSize[NeighborComp] )
						{
							NeighborComp = fillcomp;
						}
						//foundneighbor = true;
						//break;
					}
				}
				//if(foundneighbor) break;
			}

			if(NeighborComp != 0) 
			{
				shared_ptr<TIndexedOrigins_Vector> b( new TIndexedOrigins_Vector( Boundaries[x] ) );
				Collapses.m_Borders.push_back( b );
				Collapses.m_Border2Cell.push_back( Collapses.m_Cells[NeighborComp] );
			}
			else
			{
				*g_Log << "Warning: couldn't find component for boundary\n";
			}
		}
	}


	// Fill collapse array
	{
		std::queue<TCell*> S;
		vector<TCell*>::iterator it;
		for(j=0; j<m_Cells[0].size(); j++) S.push(m_Cells[0][j]);
		while( !S.empty() )
		{
			TCell * const c = S.front(); 
			S.pop();

			if( c->m_Color == TCell::COLOR_GRAY && c->m_Children.size() != 0 ) // recurse
			{
				for(it = c->m_Children.begin(); it != c->m_Children.end(); it++) 
				{
					S.push(*it);
				}
			}
			else
			{
				if(DEBUG && c->m_Color != TCell::COLOR_NEIGHBOR &&  c->m_FillComponent == TCell::INVALIDCOMPONENT) 
					throw string("c->m_FillComponent != TCell::INVALIDCOMPONENT (2)");

				if(	c->m_FillComponent != TCell::INVALIDCOMPONENT)
				{
					Collapses.m_Cells[c->m_FillComponent]->add(c);
				}
			}

			c->m_Color = TCell::COLOR_WHITE;
			c->m_FillComponent = TCell::INVALIDCOMPONENT;
		}
	}



}



shared_ptr<TShortestPath> TDeltaOmega::getShortestPath(unsigned int i, unsigned int j)
//const TIndexedOrigins * TDeltaOmega::getGeodesic(unsigned int i, unsigned int j, float & length)
{
	const unsigned int a = i < j ? i : j;
	const unsigned int b = i < j ? j : i;

	const unsigned int m_MaximumCacheSize = 40 * 1000 * 1000;
//	const unsigned int m_MaximumCacheSize = 120 * 1000 * 1000;
	const unsigned int m_NormalCacheSize = 20 * 1000 * 1000;

	if(g_Parameters.m_CacheGeodesics)
	{
		// See if cache is too big
		if(m_CacheSize > m_MaximumCacheSize)
		{
			m_MutexDijkstra.Lock();
			*g_Log << "c";
			while(m_CacheSize > m_NormalCacheSize)
			{
				pair<unsigned int, unsigned int> & geo = m_CacheQueue.front();
				TGeodesicCache::iterator it = m_Geodesics.find(geo);
				if(it == m_Geodesics.end()) throw string("Could not find geodesic in cache");
				m_CacheSize -= it->second->m_Path->capacity();
				if(m_CacheSize < 0) throw string("Cache size < 0");
				it->second.reset();
				m_Geodesics.erase(it);
				m_CacheQueue.pop();
			}
			m_MutexDijkstra.Unlock();
		}

		TGeodesicCache::iterator it = m_Geodesics.find(pair<unsigned int,unsigned int>(a,b));
		if( it == m_Geodesics.end() ) 
		{
			shared_ptr<TIndexedOrigins_Vector> iop( new TIndexedOrigins_Vector(g_Parameters.m_GeodesicVectorReserveSize) );  // Reserve capacity
			TIndexedOrigins_Vector * io = iop.get();

			float length;
			unsigned int middle = TIndexedOrigins::INVALIDINDEX;
			switch(g_Parameters.m_GeodesicType)
			{
				case TParameters::GEODESICTYPE_DIJKSTRA:
					length = dijkstraSinglePath(a,b,io , &middle);
					break;
				case TParameters::GEODESICTYPE_ASTAR:
					length = dijkstraSinglePath(a,b,io , &middle);
					break;
//				case TParameters::GEODESICTYPE_ALL:
//					length = dijkstraAllPaths(a,b,io );
//					break;
				default:
					throw string("Unknown geodesic type");
					break;
			};

			m_ComputedGeodesics++;
			shared_ptr<TShortestPath> spinfo( new TShortestPath(a,b, length, iop) );
//			spinfo->m_Middle = middle;


			// TODO, Test
			m_MutexDijkstra.Lock();
			if(spinfo->m_Length > 10.0f) // Only store sufficiently large paths
			{
				m_Geodesics[ pair<unsigned int,unsigned int>(a,b) ] = spinfo;
				m_CacheSize += iop->capacity();
				m_CacheQueue.push( pair<unsigned int,unsigned int>(a,b) );
			}
			m_MutexDijkstra.Unlock();
			//shared_ptr<TShortestPath> NewSp( new TShortestPath( *spinfo.get() ) );
			//return NewSp;
			return spinfo;
		}
		else
		{
			return it->second;
		}
	}
	else
	{	
		shared_ptr<TIndexedOrigins_Vector> iop( new TIndexedOrigins_Vector(g_Parameters.m_GeodesicVectorReserveSize) );  // Reserve capacity
		TIndexedOrigins_Vector * io = iop.get();
		float length;
		unsigned int middle = TIndexedOrigins::INVALIDINDEX;
		switch(g_Parameters.m_GeodesicType)
		{
			case TParameters::GEODESICTYPE_DIJKSTRA:
				length = dijkstraSinglePath(a,b,io );
				break;
			case TParameters::GEODESICTYPE_ASTAR:
				length = dijkstraSinglePath(a,b,io , &middle);
				break;
//			case TParameters::GEODESICTYPE_ALL:
//				length = dijkstraAllPaths(a,b,io );
//				break;
			default:
				throw string("Unknown geodesic type");
				break;
		};

		m_ComputedGeodesics++;
		shared_ptr<TShortestPath> spinfo( new TShortestPath(a,b, length, iop) );
		return spinfo;
	}
}

void TDeltaOmega::clearGeodesicCache()
{
	TGeodesicCache::iterator it;
	for(it = m_Geodesics.begin(); it != m_Geodesics.end(); it++)
	{
		it->second.reset();
	}
	m_Geodesics.swap( TGeodesicCache() );
	while(!m_CacheQueue.empty()) m_CacheQueue.pop();
}

TDeltaOmega::~TDeltaOmega()
{
	{
		for(unsigned int i=0; i<m_Cells.size(); i++)
			for(unsigned int j=0; j<m_Cells[i].size(); j++)
			{
				delete m_Cells[i][j];
			}
	}

	clearGeodesicCache();
}


shared_ptr< vector<TDeltaOmega::TAuxiliaryStruct> > TDeltaOmega::getLockedAuxiliaryList()
{
	shared_ptr< vector<TDeltaOmega::TAuxiliaryStruct> > AuxiliaryList;
	m_MutexDijkstra.Lock();
	if( m_AuxiliaryListForDijkstra.size() == 0 ) 
	{
		AuxiliaryList.reset( new vector<TAuxiliaryStruct>() );
		AuxiliaryList->resize( m_AdjacencyList.size(), m_InitialAuxiliaryList );
		m_AuxiliaryListForDijkstra.insert( AuxiliaryList );
	}
	else
	{
		AuxiliaryList = *m_AuxiliaryListForDijkstra.begin();
		m_AuxiliaryListForDijkstra.erase(m_AuxiliaryListForDijkstra.begin());
	}
	m_MutexDijkstra.Unlock();
	return AuxiliaryList;
}

void TDeltaOmega::releaseLockedAuxiliaryList(shared_ptr< vector<TDeltaOmega::TAuxiliaryStruct> > AuxiliaryList)
{
	m_MutexDijkstra.Lock();
	m_AuxiliaryListForDijkstra.insert( AuxiliaryList );
	m_MutexDijkstra.Unlock();
}


shared_ptr< vector<TDeltaOmega::TFloodfillStruct> > TDeltaOmega::getLockedComponentList()
{
	shared_ptr< vector<TDeltaOmega::TFloodfillStruct> > ComponentList;
	m_MutexDijkstra.Lock();
	if( m_ComponentLists.size() == 0 ) 
	{
		ComponentList.reset( new vector<TFloodfillStruct>() );
		ComponentList->resize( m_AdjacencyList.size(), m_InitialComponentList );
		m_ComponentLists.insert( ComponentList );
	}
	else
	{
		ComponentList = *m_ComponentLists.begin();
		m_ComponentLists.erase(m_ComponentLists.begin());
	}
	m_MutexDijkstra.Unlock();
	return ComponentList;
}

void TDeltaOmega::releaseLockedComponentList(shared_ptr< vector<TDeltaOmega::TFloodfillStruct> > ComponentList)
{
	m_MutexDijkstra.Lock();
	m_ComponentLists.insert( ComponentList );
	m_MutexDijkstra.Unlock();
}

template<class T> 
float TDeltaOmega::dijkstraSinglePath(TIndex p_Start, TIndex p_End, T * io, unsigned int * p_Middle)
{
	// Get an auxilarylist
	shared_ptr< vector<TAuxiliaryStruct> > AuxiliaryList = getLockedAuxiliaryList();


	const bool ASTAR = g_Parameters.m_GeodesicType == TParameters::GEODESICTYPE_ASTAR ? true : false;
	unsigned int n;

	// Initialize m_AdjacencyList variables for dijkstra path
	const TCoord3 StartCoord = m_Omega->m_BoundaryIndexField->vidx2coord(p_Start);
	const TCoord3 EndCoord = (p_End != TIndexedOrigins::INVALIDINDEX) ? m_Omega->m_BoundaryIndexField->vidx2coord(p_End) : TCoord3(-1,-1,-1);

	if(!ASTAR) (*AuxiliaryList)[p_Start].m_Distance = 0; // Dijkstra
	else (*AuxiliaryList)[p_Start].m_Distance = 0; // A*

	static TQueue S;
	S = TQueue();
	S.push( TQueue::value_type(0.0f, p_Start) );
	unsigned int vc = 0;
	static vector<TIndex> Visited;
	Visited.resize( m_AdjacencyList.size() * 4 );
	Visited[vc] = p_Start; 
	vc++;

	bool foundEnd = false;
	while(!S.empty())
	{
		const TIndex pidx = S.top().second;
		S.pop();

		if(pidx == p_End) { foundEnd = true; break; }
		if((*AuxiliaryList)[pidx].m_IsKnown) continue;
		(*AuxiliaryList)[pidx].m_IsKnown = true;

		const TAdjacencyStruct & padj = m_AdjacencyList[pidx];
		const float & pdist = (*AuxiliaryList)[pidx].m_Distance;
		for(n=0; n<padj.m_EdgeCount; n++)
		{
			const TIndex npidx = padj.m_Edges[n].m_ToVertex;
			if(!(*AuxiliaryList)[npidx].m_IsKnown
//				&& padj.m_Edges[n].m_Weight != 2 // TODO: TEST: sta schuine paden niet toe
				) 
			{
				// Compute distance from p to np and see if distance is less than current distance
				const float & weight = WEIGHTS[ padj.m_Edges[n].m_Weight ];

				const float t = pdist + weight; // Dijkstra
				float & npdist = (*AuxiliaryList)[npidx].m_Distance;
				if( t < npdist )
				{
					npdist = t;
					(*AuxiliaryList)[npidx].m_Previous = pidx;
					Visited[vc] = npidx; vc++;

					if(!ASTAR)
					{
						S.push( TQueue::value_type(npdist,npidx) );
					}
					else
					{
						const TCoord3 & a = m_Omega->m_BoundaryIndexField->vidx2coord(npidx);
						S.push( TQueue::value_type( npdist + EndCoord.distance( m_Omega->m_BoundaryIndexField->vidx2coord(npidx) ) ,npidx) );
					}
				}
			}
		}
	}

	float distance = 0.0f;
	if(! foundEnd) 
	{
		//*g_Log << string( wxString::Format("*** Warning: dijkstraSinglePath: didn't find path between idx %i (%i,%i,%i) and idx %i (%i,%i,%i)\n", p_Start, StartCoord.x, StartCoord.y, StartCoord.z, p_End, EndCoord.x, EndCoord.y, EndCoord.z) );
		*g_Log << "!";
	}
	else
	{
		distance = (*AuxiliaryList)[p_End].m_Distance;

		// Return the path
		if(p_Middle)
		{
			const float halfdist = distance / 2.0f;
			float mindiff = (numeric_limits<float>::max)();
			if(io)
			{
				TIndex pidx = p_End;
				io->add(pidx);
				while(pidx != p_Start)
				{
					if( fabs( (*AuxiliaryList)[pidx].m_Distance - halfdist) < mindiff )
					{
						*p_Middle = pidx;
						mindiff = (*AuxiliaryList)[pidx].m_Distance - halfdist;
					}

					pidx = (*AuxiliaryList)[pidx].m_Previous;
					if(pidx == TIndexedOrigins::INVALIDINDEX) break;
					io->add(pidx);
				}
			}
		}
		else
		{
			if(io)
			{
				TIndex pidx = p_End;
				io->add(pidx);
				while(pidx != p_Start)
				{
					pidx = (*AuxiliaryList)[pidx].m_Previous;
					if(pidx == TIndexedOrigins::INVALIDINDEX) break;
					io->add(pidx);
				}
			}
		}
	}

	// Reset 
	{
		for(unsigned int i=0; i<vc; i++)
		{
			(*AuxiliaryList)[Visited[i]] = m_InitialAuxiliaryList;
		}
	}

	releaseLockedAuxiliaryList(AuxiliaryList);

	return distance;
};


/*
template<class T>
float TDeltaOmega::dijkstraAllPaths(TIndex p_Start, TIndex p_End, T * p_Io)
{
	unsigned int n;
	static TQueue S;
	vc = 0;

	// Forward search
	{
		const TIndex start = p_Start;
		const TIndex end = p_End;
		m_AllPathDistances1[start].m_Dist = 0.0f;
		//Visited.push_back(start);
		Visited[vc++] = start;
		S = TQueue();
		S.push( TQueue::value_type(m_AllPathDistances1[start].m_Dist, start) );
		while(!S.empty())
		{
			const TIndex pidx = S.top().second;
			S.pop();

			if(m_AllPathDistances1[pidx].m_IsKnown) continue;
			m_AllPathDistances1[pidx].m_IsKnown = true;

			if(pidx == end) { break; }

			const TAdjacencyStruct & padj = m_AdjacencyList[pidx];
			for(n=0; n<padj.m_EdgeCount; n++)
			{
				const TIndex npidx = padj.m_Edges[n].m_ToVertex;
				if( ! m_AllPathDistances1[npidx].m_IsKnown ) 
				{
					// Compute distance from p to np and see if distance is less than current distance
					const float weight = WEIGHTS[ padj.m_Edges[n].m_Weight ];
					const float t = m_AllPathDistances1[pidx].m_Dist + weight; // Dijkstra
					float & npdist = m_AllPathDistances1[npidx].m_Dist;
					if( t < npdist )
					{
						npdist = t;
						S.push( TQueue::value_type(npdist,npidx) );

						//Visited.push_back(npidx);
						Visited[vc++] = npidx;
					}
				}
			}
		}
	}
	const float MinDistance = m_AllPathDistances1[p_End].m_Dist;

	float PathTolerance; // parameter
	switch(g_Parameters.m_AllPathsToleranceType)
	{
		case TParameters::ALLPATHSTOLERANCETYPE_ABSOLUTE:
			PathTolerance = g_Parameters.m_AllPathsToleranceAbsolute;
			break;
		case TParameters::ALLPATHSTOLERANCETYPE_RELATIVE:
			PathTolerance = MinDistance / g_Parameters.m_AllPathsToleranceRelative;
			break;
	}


	// Backward search
	{
		const TIndex start = p_End;
		const TIndex end = p_Start;
		m_AllPathDistances2[start].m_Dist = 0.0f;
		//Visited.push_back(start);
		Visited[vc++] = start;
		S = TQueue();
		S.push( TQueue::value_type(m_AllPathDistances2[start].m_Dist, start) );
		while(!S.empty())
		{
			const TIndex pidx = S.top().second;
			S.pop();

			if(! m_AllPathDistances1[pidx].m_IsKnown ) continue; // only enter voxels visited by forward search

			if(m_AllPathDistances2[pidx].m_IsKnown) continue;
			m_AllPathDistances2[pidx].m_IsKnown = true;

			{
				const float d = m_AllPathDistances1[pidx].m_Dist + m_AllPathDistances2[pidx].m_Dist - MinDistance;
				if( d < PathTolerance ) 
				{
					p_Io->add( pidx );
				}
			}

			if(pidx == end) { break; }

			const TAdjacencyStruct & padj = m_AdjacencyList[pidx];
			for(n=0; n<padj.m_EdgeCount; n++)
			{
				const TIndex npidx = padj.m_Edges[n].m_ToVertex;
				if( ! m_AllPathDistances2[npidx].m_IsKnown ) 
				{
					// Compute distance from p to np and see if distance is less than current distance
					const float weight = WEIGHTS[ padj.m_Edges[n].m_Weight ];
					const float t = m_AllPathDistances2[pidx].m_Dist + weight; // Dijkstra
					float & npdist = m_AllPathDistances2[npidx].m_Dist;
					if( t < npdist )
					{
						npdist = t;
						S.push( TQueue::value_type(npdist,npidx) );

						//Visited.push_back(npidx);
						Visited[vc++] = npidx;
					}
				}
			}
		}
	}

	// Reset m_AllPathDistances1 & 2
	const TAllPathsStruct s = { (numeric_limits<float>::max)(), false };
	for(unsigned int i=0; i<vc; i++)
	{
		m_AllPathDistances1[Visited[i]] = s;
		m_AllPathDistances2[Visited[i]] = s;
	}

	return MinDistance;
};
*/





void TDeltaOmega::mergeCollapseWithOthers(const TIndexedOrigins_Set * p_Collapse, vector<TIndexedOrigins_Set*> & p_Others)
// Merges contents of p_Collapse with p_Others, based on distance.
// p_Collapse should neighbor the components in p_Others.
{

	static TQueue S;
	S = TQueue();
	vc = 0;

	// Detect vertices in p_Collapse that neighbor p_Others
	TIndexedOrigins_Set::const_iterator it;
	for(it = p_Collapse->begin(); it != p_Collapse->end(); it++)
	{
		const TIndex & idx = *it;

		m_ComponentList[idx].m_NeighborsGeodesic = true; // abuse as flag for 'allowed to go here'
		Visited[vc] = idx;
		vc++;

		const TAdjacencyStruct & padj = m_AdjacencyList[idx];
		for(unsigned int n=0; n<padj.m_EdgeCount; n++)
		{
			const TIndex & npidx = padj.m_Edges[n].m_ToVertex;
			
			// See if npidx is in p_Others
			bool found = false;
			unsigned int x;
			for(x=0; x<p_Others.size(); x++)
			{
				if( p_Others[x]->find(npidx) != p_Others[x]->end() )
				{
					found = true;
					break;
				}
			}

			if(found)
			{
				m_AuxiliaryList[idx].m_Distance = 0.0f;
				m_ComponentList[idx].m_Component = x;
				S.push( pair<float,TIndex>(0.0f, idx) );
			}
		}
	}

	// Do floodfill from those vertices
	bool foundEnd = false;
	while(!S.empty())
	{
		const TIndex pidx = S.top().second;
		S.pop();

		if(m_AuxiliaryList[pidx].m_IsKnown) continue;
		m_AuxiliaryList[pidx].m_IsKnown = true;
		p_Others[m_ComponentList[pidx].m_Component]->insert( pidx );

		const TAdjacencyStruct & padj = m_AdjacencyList[pidx];
		const float & pdist = m_AuxiliaryList[pidx].m_Distance;
		for(unsigned int n=0; n<padj.m_EdgeCount; n++)
		{
			const TIndex npidx = padj.m_Edges[n].m_ToVertex;
			if( m_ComponentList[npidx].m_NeighborsGeodesic
				&&
				!m_AuxiliaryList[npidx].m_IsKnown)
			{
				// Compute distance from p to np and see if distance is less than current distance
				const float & weight = WEIGHTS[ padj.m_Edges[n].m_Weight ];

				const float t = pdist + weight; // Dijkstra
				float & npdist = m_AuxiliaryList[npidx].m_Distance;
				if( t < npdist )
				{
					npdist = t;
					//m_AuxiliaryList[npidx].m_Previous = pidx;
					m_ComponentList[npidx].m_Component = m_ComponentList[pidx].m_Component;
					Visited[vc] = npidx; 
					vc++;

					S.push( TQueue::value_type(npdist,npidx) );
				}
			}
		}
	}

	// Reset 
	{
		const TFloodfillStruct s = m_InitialComponentList;
		for(unsigned int i=0; i<vc; i++)
		{
			m_ComponentList[Visited[i]] = s;
			m_AuxiliaryList[Visited[i]] = m_InitialAuxiliaryList;
		}
	}

}




void TDeltaOmega::determineNeighborhoods(const TTypedFieldInterface<unsigned int> * p_Field, vector<set<unsigned int> > & p_Neighbors)
{
	if( p_Neighbors.size() == 0 ) throw string("p_Neighbors.size() == 0");

	const TIndexMapper * IndexField = p_Field->getIndexField().get();
	for(unsigned int x=0; x<IndexField->getMaxIndex(); x++)
	{
		const TIndex & pidx = m_Omega->m_BoundaryIndexField->vcoord2idx( IndexField->vidx2coord(x) );
		if(pidx == TIndexMapper::OUTSIDE) continue; 
		unsigned int pidxc = p_Field->vvaluex(x);
		if( pidxc == 0 ) continue;
		pidxc -= 1;

		// Check neighbors of component idxc
		const TAdjacencyStruct & padj = m_AdjacencyList[pidx];
		for(unsigned int n=0; n<padj.m_EdgeCount; n++)
		{
			const TIndex & npidx = padj.m_Edges[n].m_ToVertex;
			unsigned int npidxc = p_Field->vvaluep( m_Omega->m_BoundaryIndexField->vidx2coord(npidx) );
			if( npidxc == 0 ) continue;
			npidxc -= 1;

			if( npidxc != pidxc )
			{
				p_Neighbors[npidxc].insert( pidxc );
				p_Neighbors[pidxc].insert( npidxc );
			}
		}			
	}
}

void TDeltaOmega::mergeCollapseWithOthers(const TIndexedOrigins_Set * p_MergeWhat, TTypedFieldInterface<unsigned int> * p_Field, set<unsigned int> & p_MergeWithComponents)
// Merges the voxels in p_Field indicated by the voxels id's in p_Collapse with the components indicated by p_MergeWithComponents
{
	static TQueue S;
	S = TQueue();
	vc = 0;

	// Detect vertices in p_Collapse that neighbor p_Others
	TIndexedOrigins_Set::const_iterator it;
	for(it = p_MergeWhat->begin(); it != p_MergeWhat->end(); it++)
	{
		const TIndex & idx = *it;

		m_ComponentList[idx].m_NeighborsGeodesic = true; // abuse as flag for 'allowed to go here'
		Visited[vc] = idx;
		vc++;

		const TAdjacencyStruct & padj = m_AdjacencyList[idx];
		for(unsigned int n=0; n<padj.m_EdgeCount; n++)
		{
			const TIndex & npidx = padj.m_Edges[n].m_ToVertex;
			const unsigned int npidxc = p_Field->vvaluep( m_Omega->m_BoundaryIndexField->vidx2coord(npidx) );
			
			// See if npidx is in another components
			if( p_MergeWithComponents.find(npidxc) != p_MergeWithComponents.end() )
			{
				m_AuxiliaryList[idx].m_Distance = 0.0f;
				m_ComponentList[idx].m_Component = *p_MergeWithComponents.find(npidxc);
				S.push( pair<float,TIndex>(0.0f, idx) );
			}
		}
	}

	// Do floodfill from those vertices
	bool foundEnd = false;
	while(!S.empty())
	{
		const TIndex pidx = S.top().second;
		S.pop();

		if(m_AuxiliaryList[pidx].m_IsKnown) continue;
		m_AuxiliaryList[pidx].m_IsKnown = true;
		//p_Others[m_ComponentList[pidx].m_Component]->insert( pidx );
		p_Field->wvaluep( m_Omega->m_BoundaryIndexField->vidx2coord(pidx) ) = m_ComponentList[pidx].m_Component;

		const TAdjacencyStruct & padj = m_AdjacencyList[pidx];
		const float & pdist = m_AuxiliaryList[pidx].m_Distance;
		for(unsigned int n=0; n<padj.m_EdgeCount; n++)
		{
			const TIndex npidx = padj.m_Edges[n].m_ToVertex;
			if( m_ComponentList[npidx].m_NeighborsGeodesic
				&&
				!m_AuxiliaryList[npidx].m_IsKnown)
			{
				// Compute distance from p to np and see if distance is less than current distance
				const float & weight = WEIGHTS[ padj.m_Edges[n].m_Weight ];

				const float t = pdist + weight; // Dijkstra
				float & npdist = m_AuxiliaryList[npidx].m_Distance;
				if( t < npdist )
				{
					npdist = t;
					m_ComponentList[npidx].m_Component = m_ComponentList[pidx].m_Component;
					Visited[vc] = npidx; 
					vc++;

					S.push( TQueue::value_type(npdist,npidx) );
				}
			}
		}
	}
	
	// Reset 
	{
		const TFloodfillStruct s = m_InitialComponentList;
		for(unsigned int i=0; i<vc; i++)
		{
			m_ComponentList[Visited[i]] = s;
			m_AuxiliaryList[Visited[i]] = m_InitialAuxiliaryList;
		}
	}
}







void TDeltaOmega::determineBoundary(const TIndexedOrigins_Vector * p_Collapse, TIndexedOrigins_Vector * p_Boundary)
{
	unsigned int x,n;

	TTypedFieldInterface<unsigned char> * debug = 0;
	if(g_Settings.m_DebugOutput)
	{
		debug = static_cast<TTypedFieldInterface<unsigned char>*>(  g_Mediator.getCurrentLayerSet()->newField(TType::TYPE_UCHAR, m_Omega->m_BoundaryIndexField, "determineBoundary debug" ) );
		debug->clear();
	}

	for(x=0; x<p_Collapse->vertexcount(); x++)
	{
		const TIndexedOrigins::TIndex idx = (*p_Collapse)[x];
		const unsigned int CurrentValue = m_ComponentList[idx].m_Component;
//		assert( CurrentValue == TCell::INVALIDCOMPONENT);
		m_ComponentList[idx].m_Component = 3;
	}
	
	for(x=0; x<p_Collapse->vertexcount(); x++)
	{
		const TIndexedOrigins::TIndex idx = (*p_Collapse)[x];
			
		const TAdjacencyStruct & padj = m_AdjacencyList[idx];
		for(n=0; n<padj.m_EdgeCount; n++)
		{
			const TIndexedOrigins::TIndex idx2 = padj.m_Edges[n].m_ToVertex;
			if(m_ComponentList[idx2].m_Component != 3)
			{
				p_Boundary->add(idx);
				if(debug) debug->wvaluex(idx) = 1;
				break;
			}
		}
	}

	if(g_Settings.m_DebugOutput)
	{
		debug->getLayer()->onFieldChanged();
	}

	// Reset 
	for(x=0; x<p_Collapse->vertexcount(); x++)
	{
		const TIndexedOrigins::TIndex idx = (*p_Collapse)[x];
//		assert( m_ComponentList[idx].m_Component == 3);
		m_ComponentList[idx].m_Component = TCell::INVALIDCOMPONENT;
	}
}




void TDeltaOmega::computeDistanceFieldWithSeeds(const TTypedFieldInterface<float>* p_Input, TTypedFieldInterface<float>* p_Output, bool p_UseInputValue, const TTypedFieldInterface<float>* p_CostFunction)
{
	vc = 0;
	static TQueue S;
	S = TQueue();
	
	if(p_Input->getMaxIndex() != m_AuxiliaryList.size()) throw string("p_Input->getMaxIndex() != m_AuxiliaryList.size()");

	unsigned int x;
	for(x=0; x<p_Input->getMaxIndex(); x++)
	{
		if(p_Input->vvaluex(x) != 0.0f)
		{
			const float dist = p_UseInputValue  ? p_Input->vvaluex(x) : 0.001f;
			S.push( TQueue::value_type(dist, x) );
			Visited[vc] = x; vc++;
			m_AuxiliaryList[x].m_Distance = dist;
		}
	}

	TIndex pidx;
	while(!S.empty())
	{
		pidx = S.top().second;
		S.pop();

		if(m_AuxiliaryList[pidx].m_IsKnown) continue;
		m_AuxiliaryList[pidx].m_IsKnown = true;
		p_Output->wvaluex(pidx) = m_AuxiliaryList[pidx].m_Distance;

		const TAdjacencyStruct & padj = m_AdjacencyList[pidx];
		const float & pdist = m_AuxiliaryList[pidx].m_Distance;
		for(unsigned int n=0; n<padj.m_EdgeCount; n++)
		{
			const TIndex npidx = padj.m_Edges[n].m_ToVertex;
			if( !m_AuxiliaryList[npidx].m_IsKnown ) 
			{
				float weight = WEIGHTS[ padj.m_Edges[n].m_Weight ];
				if(p_CostFunction) weight *= powf( p_CostFunction->vvaluex(npidx), 3.0f );

				const float t = pdist + weight; // Dijkstra
				float & npdist = m_AuxiliaryList[npidx].m_Distance;
				if( t < npdist )
				{
					npdist = t;
					Visited[vc] = npidx; vc++;
					m_AuxiliaryList[npidx].m_Previous = m_AuxiliaryList[pidx].m_Previous;

					S.push( TQueue::value_type(npdist,npidx) );
				}
			}
		}
	}

	// Reset 
	{
		for(unsigned int i=0; i<vc; i++)
		{
			m_AuxiliaryList[Visited[i]] = m_InitialAuxiliaryList;
		}
	}
}

void TDeltaOmega::computeDistanceFieldWithSeedsAndDomain(TTypedFieldInterface<float>* p_Input, TTypedFieldInterface<unsigned int>* p_Domain, unsigned int p_DomainIdx)
{
	vc = 0;
	static TQueue S;
	S = TQueue();
	
	if(p_Input->getMaxIndex() != m_AuxiliaryList.size()) throw string("p_Input->getMaxIndex() != m_AuxiliaryList.size()");

	unsigned int x;
	for(x=0; x<p_Input->getMaxIndex(); x++)
	{
		if(p_Input->vvaluex(x) != 0.0f)
		{
			const float dist = p_Input->vvaluex(x);
			S.push( TQueue::value_type(dist, x) );
			Visited[vc] = x; vc++;
			m_AuxiliaryList[x].m_Distance = dist;
		}
	}

	TIndex pidx;
	while(!S.empty())
	{
		pidx = S.top().second;
		S.pop();

		if(m_AuxiliaryList[pidx].m_IsKnown) continue;
		m_AuxiliaryList[pidx].m_IsKnown = true;
		p_Input->wvaluex(pidx) = m_AuxiliaryList[pidx].m_Distance;

		const TAdjacencyStruct & padj = m_AdjacencyList[pidx];
		const float & pdist = m_AuxiliaryList[pidx].m_Distance;
		for(unsigned int n=0; n<padj.m_EdgeCount; n++)
		{
			const TIndex npidx = padj.m_Edges[n].m_ToVertex;
			if( !m_AuxiliaryList[npidx].m_IsKnown && p_Domain->vvaluex(npidx) == p_DomainIdx ) 
			{
				const float & weight = WEIGHTS[ padj.m_Edges[n].m_Weight ];

				const float t = pdist + weight; // Dijkstra
				float & npdist = m_AuxiliaryList[npidx].m_Distance;
				if( t < npdist )
				{
					npdist = t;
					Visited[vc] = npidx; vc++;
					m_AuxiliaryList[npidx].m_Previous = m_AuxiliaryList[pidx].m_Previous;

					S.push( TQueue::value_type(npdist,npidx) );
				}
			}
		}
	}

	// Reset 
	{
		for(unsigned int i=0; i<vc; i++)
		{
			m_AuxiliaryList[Visited[i]] = m_InitialAuxiliaryList;
		}
	}
}

void TDeltaOmega::computeDistanceFieldWithSeedDistancesAndDomainFilter(const TTypedFieldInterface<float> *p_Input, const TAbstractFilter * p_DomainFilter, TTypedFieldInterface<float>* p_Output)
{
	vc = 0;
	static TQueue S;
	S = TQueue();
	
	unsigned int x;
	
	for(x=0; x<p_Input->getMaxIndex(); x++)
	{
		if(p_Input->vvaluex(x) != 0.0f)
		{
			const float dist = p_Input->vvaluex(x);
			S.push( TQueue::value_type(dist, x) );
			Visited[vc] = x; vc++;
			m_AuxiliaryList[x].m_Distance = dist;
		}
	}

	TIndex pidx;
	while(!S.empty())
	{
		pidx = S.top().second;
		S.pop();

		if(m_AuxiliaryList[pidx].m_IsKnown) continue;
		m_AuxiliaryList[pidx].m_IsKnown = true;
		p_Output->wvaluex(pidx) = m_AuxiliaryList[pidx].m_Distance;

		const TAdjacencyStruct & padj = m_AdjacencyList[pidx];
		const float & pdist = m_AuxiliaryList[pidx].m_Distance;
		for(unsigned int n=0; n<padj.m_EdgeCount; n++)
		{
			const TIndex npidx = padj.m_Edges[n].m_ToVertex;
			if( !m_AuxiliaryList[npidx].m_IsKnown && p_DomainFilter->testx(npidx) ) 
			{
				const float & weight = WEIGHTS[ padj.m_Edges[n].m_Weight ];

				const float t = pdist + weight; // Dijkstra
				float & npdist = m_AuxiliaryList[npidx].m_Distance;
				if( t < npdist )
				{
					npdist = t;
					Visited[vc] = npidx; vc++;
					m_AuxiliaryList[npidx].m_Previous = m_AuxiliaryList[pidx].m_Previous;

					S.push( TQueue::value_type(npdist,npidx) );
				}
			}
		}
	}

	// Reset 
	{
		for(unsigned int i=0; i<vc; i++)
		{
			m_AuxiliaryList[Visited[i]] = m_InitialAuxiliaryList;
		}
	}
}


void TDeltaOmega::computeDistanceFieldWithSeedsAndDomainFilter(const set<unsigned int> * p_Seeds, const TAbstractFilter * p_DomainFilter, TTypedFieldInterface<float>* p_Output, const TTypedFieldInterface<float>* p_Cost, TTypedFieldInterface<unsigned int>* p_OutputSeeds)
{
	vc = 0;
	static TQueue S;
	S = TQueue();
	
	for(set<unsigned int>::iterator it = p_Seeds->begin(); it != p_Seeds->end(); it++)
	{
		unsigned int x = *it;
		const float dist = 0.0f;
		S.push( TQueue::value_type(dist, x) );
		Visited[vc] = x; vc++;
		m_AuxiliaryList[x].m_Distance = dist;
		m_AuxiliaryList[x].m_Previous = *it;
	}

	TIndex pidx;
	while(!S.empty())
	{
		pidx = S.top().second;
		S.pop();

		if(m_AuxiliaryList[pidx].m_IsKnown) continue;
		m_AuxiliaryList[pidx].m_IsKnown = true;
		if(p_Output) p_Output->wvaluex(pidx) = m_AuxiliaryList[pidx].m_Distance;
		if(p_OutputSeeds) p_OutputSeeds->wvaluex(pidx) = m_AuxiliaryList[pidx].m_Previous;

		const TAdjacencyStruct & padj = m_AdjacencyList[pidx];
		const float & pdist = m_AuxiliaryList[pidx].m_Distance;
		for(unsigned int n=0; n<padj.m_EdgeCount; n++)
		{
			const TIndex npidx = padj.m_Edges[n].m_ToVertex;
			if( !m_AuxiliaryList[npidx].m_IsKnown && p_DomainFilter->testx(npidx) ) 
			{
				float weight = WEIGHTS[ padj.m_Edges[n].m_Weight ];
				if(p_Cost) weight *= powf( p_Cost->vvaluex(npidx), 2.0f );

				const float t = pdist + weight; // Dijkstra
				float & npdist = m_AuxiliaryList[npidx].m_Distance;
				if( t < npdist )
				{
					npdist = t;
					Visited[vc] = npidx; vc++;
					m_AuxiliaryList[npidx].m_Previous = m_AuxiliaryList[pidx].m_Previous;

					S.push( TQueue::value_type(npdist,npidx) );
				}
			}
		}
	}

	// Reset 
	{
		for(unsigned int i=0; i<vc; i++)
		{
			m_AuxiliaryList[Visited[i]] = m_InitialAuxiliaryList;
		}
	}
}






void TDeltaOmega::distancePropagation(TDistancePropagationParameters & p_Par)
{
	vc = 0;
	static TQueue S;
	S = TQueue();
	
	if(p_Par.m_Start != (numeric_limits<unsigned int>::max)())
	{
		unsigned int x = p_Par.m_Start;
		const float dist = 0.0f;
		S.push( TQueue::value_type(dist, x) );
		Visited[vc] = x; vc++;
		m_AuxiliaryList[x].m_Distance = dist;
		m_AuxiliaryList[x].m_Previous = x;
	}
	if(p_Par.m_StartSet) 
	{
		for(set<unsigned int>::iterator it = p_Par.m_StartSet->begin(); it != p_Par.m_StartSet->end(); it++)
		{
			unsigned int idx = *it;
			const float dist = 0.0f;
			S.push( TQueue::value_type(dist, idx) );
			Visited[vc] = idx; vc++;
			m_AuxiliaryList[idx].m_Distance = dist;
			m_AuxiliaryList[idx].m_Previous = idx;
		}
	}
	if(p_Par.m_StartVector) 
	{
		for(unsigned int x = 0; x < p_Par.m_StartVector->size(); x++)
		{
			unsigned int idx = (*p_Par.m_StartVector)[x];
			const float dist = 0.0f;
			S.push( TQueue::value_type(dist, idx) );
			Visited[vc] = idx; vc++;
			m_AuxiliaryList[idx].m_Distance = dist;
			m_AuxiliaryList[idx].m_Previous = idx;
		}
	}
	

	TIndex pidx = (std::numeric_limits<unsigned int>::max)();
	while(!S.empty())
	{
		pidx = S.top().second;
		S.pop();

		if(m_AuxiliaryList[pidx].m_IsKnown) continue;
		m_AuxiliaryList[pidx].m_IsKnown = true;
		if(pidx == p_Par.m_End) 
		{ 
			p_Par.m_OutputEndDistance = m_AuxiliaryList[pidx].m_Distance; 
			break; 
		}
		if(p_Par.m_EndSet != 0 && p_Par.m_EndSet->find( pidx ) != p_Par.m_EndSet->end() )
		{
			break; 
		}
		if(m_AuxiliaryList[pidx].m_Distance > p_Par.m_StopDistance) 
		{
			break;
		}
		if(p_Par.m_OutputDistance) p_Par.m_OutputDistance->wvaluex(pidx) = m_AuxiliaryList[pidx].m_Distance;
		if(p_Par.m_TargetFilter && p_Par.m_TargetFilter->testx(pidx) && p_Par.m_OutputTargets)
		{
			p_Par.m_OutputTargets->insert( pidx );
		}
		//if(p_Par.m_OutputClosestStartPoint) p_Par.m_OutputClosestStartPoint->wvaluex(pidx) = m_AuxiliaryList[pidx].m_Previous;

		const TAdjacencyStruct & padj = m_AdjacencyList[pidx];
		const float & pdist = m_AuxiliaryList[pidx].m_Distance;
		for(unsigned int n=0; n<padj.m_EdgeCount; n++)
		{
			const TIndex npidx = padj.m_Edges[n].m_ToVertex;
			if( !m_AuxiliaryList[npidx].m_IsKnown 
				&& (!p_Par.m_DomainFilter || p_Par.m_DomainFilter->testx(npidx)) 
//				&& padj.m_Edges[n].m_Weight != 2 // TODO: TEST: sta schuine paden niet toe
				) 
			{
				float weight = WEIGHTS[ padj.m_Edges[n].m_Weight ];
				if(p_Par.m_CostFunction) weight *= p_Par.m_CostFunction->vvaluex(npidx);

				const float t = pdist + weight; // Dijkstra
				float & npdist = m_AuxiliaryList[npidx].m_Distance;
				if( t < npdist )
				{
					npdist = t;
					Visited[vc] = npidx; vc++;
					m_AuxiliaryList[npidx].m_Previous = pidx;

					S.push( TQueue::value_type(npdist,npidx) );
				}
			}
		}
	}
	if(pidx != (std::numeric_limits<unsigned int>::max)())
	{
		p_Par.m_OutputEndDistance = m_AuxiliaryList[pidx].m_Distance; 
		p_Par.m_OutputEndIdx = pidx;
	}

	if(p_Par.m_OutputPath)
	{
		TIndex pidx = p_Par.m_End;
		p_Par.m_OutputPath->add(pidx);
		while(pidx != p_Par.m_Start)
		{
			pidx = m_AuxiliaryList[pidx].m_Previous;
			if(pidx == TIndexedOrigins::INVALIDINDEX) break;
			p_Par.m_OutputPath->add(pidx);
		}
	}

	// Reset 
	{
		for(unsigned int i=0; i<vc; i++)
		{
			m_AuxiliaryList[Visited[i]] = m_InitialAuxiliaryList;
		}
	}
}




void TDeltaOmega::computeDistancesToNonZeros(const TTypedFieldInterface<unsigned int>* p_Input, TTypedFieldInterface<float>* p_Output, TTypedFieldInterface<unsigned int>* p_ClosestNonZero)
{
	vc = 0;
	static TQueue S;
	S = TQueue();
	
	if(p_Input->getMaxIndex() != m_AuxiliaryList.size()) throw string("p_Input->getMaxIndex() != m_AuxiliaryList.size()");

	unsigned int x;
	for(x=0; x<p_Input->getMaxIndex(); x++)
	{
		if(p_Input->vvaluex(x) != 0)
		{
			S.push( TQueue::value_type(0.0f, x) );
			Visited[vc] = x; vc++;
			m_AuxiliaryList[x].m_Distance = 0;
			if(p_ClosestNonZero) m_AuxiliaryList[x].m_Previous = x;
		}
	}

	TIndex pidx;
	while(!S.empty())
	{
		pidx = S.top().second;
		S.pop();

		if(m_AuxiliaryList[pidx].m_IsKnown) continue;
		m_AuxiliaryList[pidx].m_IsKnown = true;
		p_Output->wvaluex(pidx) = m_AuxiliaryList[pidx].m_Distance;
		if(p_ClosestNonZero) p_ClosestNonZero->wvaluex(pidx) = m_AuxiliaryList[pidx].m_Previous;

		const TAdjacencyStruct & padj = m_AdjacencyList[pidx];
		const float & pdist = m_AuxiliaryList[pidx].m_Distance;
		for(unsigned int n=0; n<padj.m_EdgeCount; n++)
		{
			const TIndex npidx = padj.m_Edges[n].m_ToVertex;
			if( !m_AuxiliaryList[npidx].m_IsKnown ) 
			{
				const float & weight = WEIGHTS[ padj.m_Edges[n].m_Weight ];

				const float t = pdist + weight; // Dijkstra
				float & npdist = m_AuxiliaryList[npidx].m_Distance;
				if( t < npdist )
				{
					npdist = t;
					Visited[vc] = npidx; vc++;
					m_AuxiliaryList[npidx].m_Previous = m_AuxiliaryList[pidx].m_Previous;

					S.push( TQueue::value_type(npdist,npidx) );
				}
			}
		}
	}

	// Reset 
	{
		for(unsigned int i=0; i<vc; i++)
		{
			m_AuxiliaryList[Visited[i]] = m_InitialAuxiliaryList;
		}
	}
}


void TDeltaOmega::computeDistancesOnFilterUsingSeed(unsigned int idx, const TFilter * p_Filter, TTypedFieldInterface<float>* p_Output)
{
	vc = 0;
	static TQueue S;
	S = TQueue();

	S.push( TQueue::value_type(0.0f, idx) );
	Visited[vc] = idx; vc++;
	m_AuxiliaryList[idx].m_Distance = 0;

	TIndex pidx;
	while(!S.empty())
	{
		pidx = S.top().second;
		S.pop();

		if(m_AuxiliaryList[pidx].m_IsKnown) continue;
		m_AuxiliaryList[pidx].m_IsKnown = true;
		p_Output->wvaluex(pidx) = m_AuxiliaryList[pidx].m_Distance;

		const TAdjacencyStruct & padj = m_AdjacencyList[pidx];
		const float & pdist = m_AuxiliaryList[pidx].m_Distance;
		for(unsigned int n=0; n<padj.m_EdgeCount; n++)
		{
			const TIndex npidx = padj.m_Edges[n].m_ToVertex;
			if( !m_AuxiliaryList[npidx].m_IsKnown && p_Filter->testx(npidx) ) 
			{
				const float & weight = WEIGHTS[ padj.m_Edges[n].m_Weight ];

				const float t = pdist + weight; // Dijkstra
				float & npdist = m_AuxiliaryList[npidx].m_Distance;
				if( t < npdist )
				{
					npdist = t;
					Visited[vc] = npidx; vc++;
					m_AuxiliaryList[npidx].m_Previous = m_AuxiliaryList[pidx].m_Previous;

					S.push( TQueue::value_type(npdist,npidx) );
				}
			}
		}
	}

	// Reset 
	{
		for(unsigned int i=0; i<vc; i++)
		{
			m_AuxiliaryList[Visited[i]] = m_InitialAuxiliaryList;
		}
	}
}

float TDeltaOmega::computeDistancesOnIndexes(unsigned int idx, const TIndexedOrigins_Vector * p_Domain, vector<float>* p_Output)
{
	if(p_Domain->size() != p_Output->size()) throw string("computeDistancesOnIndexes(): p_Domain != p_Output->size()");
	vc = 0;
	static TQueue S;
	S = TQueue();

	// Abuse m_Previous member for domain 
	unsigned int x;
	for(x=0; x<p_Domain->size(); x++)
	{
		const unsigned int idx = (*p_Domain)[x];
		Visited[vc] = idx; vc++;
		m_AuxiliaryList[idx].m_Previous = x;
	}

	if(m_AuxiliaryList[idx].m_Previous == (numeric_limits<unsigned int>::max)() )
	{
		throw string("computeDistancesOnIndexes(): start not in domain");
	}

	S.push( TQueue::value_type(0.0f, idx) );
	m_AuxiliaryList[idx].m_Distance = 0;

	TIndex pidx;
	while(!S.empty())
	{
		pidx = S.top().second;
		S.pop();

		if(m_AuxiliaryList[pidx].m_IsKnown) continue;
		m_AuxiliaryList[pidx].m_IsKnown = true;
		(*p_Output)[m_AuxiliaryList[pidx].m_Previous] = m_AuxiliaryList[pidx].m_Distance;

		const TAdjacencyStruct & padj = m_AdjacencyList[pidx];
		const float & pdist = m_AuxiliaryList[pidx].m_Distance;
		for(unsigned int n=0; n<padj.m_EdgeCount; n++)
		{
			const TIndex npidx = padj.m_Edges[n].m_ToVertex;
			if( !m_AuxiliaryList[npidx].m_IsKnown 
				&& m_AuxiliaryList[npidx].m_Previous != (numeric_limits<unsigned int>::max)()
				) 
			{
				const float & weight = WEIGHTS[ padj.m_Edges[n].m_Weight ];

				const float t = pdist + weight; // Dijkstra
				float & npdist = m_AuxiliaryList[npidx].m_Distance;
				if( t < npdist )
				{
					npdist = t;
					S.push( TQueue::value_type(npdist,npidx) );
				}
			}
		}
	}
	const float maxdist = m_AuxiliaryList[pidx].m_Distance;

	// Reset 
	{
		for(unsigned int i=0; i<vc; i++)
		{
			m_AuxiliaryList[Visited[i]] = m_InitialAuxiliaryList;
		}
	}

	return maxdist;
}



float TDeltaOmega::findShortestPath(unsigned int p_Start, unsigned int p_End, TAbstractFilter * p_DomainFilter)
{
	unsigned int n;
	static TQueue S;
	S = TQueue();
	vc = 0;
	S.push( TQueue::value_type(0.0f, p_Start));
	m_AuxiliaryList[p_Start].m_Distance = 0.0f;
	Visited[vc] = p_Start; 
	vc++;


	bool foundEnd = false;
	TIndex pidx = (numeric_limits<unsigned int>::max)();
	float p_Distance = (numeric_limits<float>::max)();
	while(!S.empty())
	{
		pidx = S.top().second;
		S.pop();

		if(pidx == p_End)
		{ 
			foundEnd = true; 
			p_Distance = m_AuxiliaryList[pidx].m_Distance;
			break; 
		}

		if(m_AuxiliaryList[pidx].m_IsKnown) continue;
		m_AuxiliaryList[pidx].m_IsKnown = true;

		const TAdjacencyStruct & padj = m_AdjacencyList[pidx];
		const float & pdist = m_AuxiliaryList[pidx].m_Distance;
		for(n=0; n<padj.m_EdgeCount; n++)
		{
			const TIndex npidx = padj.m_Edges[n].m_ToVertex;
			if(!m_AuxiliaryList[npidx].m_IsKnown 
				&& (!p_DomainFilter || p_DomainFilter->testx(npidx) )
				) 
			{
				// Compute distance from p to np and see if distance is less than current distance
				const float & weight = WEIGHTS[ padj.m_Edges[n].m_Weight ];

				const float t = pdist + weight; // Dijkstra
				float & npdist = m_AuxiliaryList[npidx].m_Distance;
				if( t < npdist )
				{
					npdist = t;
					m_AuxiliaryList[npidx].m_Previous = pidx;
					Visited[vc] = npidx; vc++;

					S.push( TQueue::value_type(npdist,npidx) );
				}
			}
		}
	}

	// Reset 
	{
		for(unsigned int i=0; i<vc; i++)
		{
			m_AuxiliaryList[Visited[i]] = m_InitialAuxiliaryList;
		}
	}

	return p_Distance;
}


unsigned int TDeltaOmega::findNearestNonZero(unsigned int p_Start, TTypedFieldInterface<unsigned int>* p_Field, float & p_Distance)
{
	vector<unsigned int> starts;
	starts.push_back(p_Start);
	return findNearestNonZero(&starts, p_Field, p_Distance);
}

unsigned int TDeltaOmega::findNearestNonZero(set<unsigned int> * p_Start, TTypedFieldInterface<unsigned int>* p_Field, float & p_Distance)
{
	if(p_Start->size() == 0)
	{
		throw string("findNearestNonZero():: p_Start->size() == 0");
	}
	set<unsigned int>::iterator it; 
	vector<unsigned int> starts(p_Start->size());
	unsigned int c = 0;
	for(it=p_Start->begin(); it != p_Start->end(); it++)
	{
		starts[c++] = *it;
	}
	return findNearestNonZero(&starts, p_Field, p_Distance);
}

unsigned int TDeltaOmega::findNearestNonZero(vector<unsigned int> * p_Start, TTypedFieldInterface<unsigned int>* p_Field, float & p_Distance)
{
	if(p_Start->size() == 0) 
	{
		throw string("findNearestNonZero():: p_Start->size() == 0");
	}
	
	unsigned int n;
	static TQueue S;
	S = TQueue();
	vc = 0;
	for(n=0; n<p_Start->size(); n++)
	{
		const unsigned int idx = (*p_Start)[n];
		S.push( TQueue::value_type(0.0f, idx));
		m_AuxiliaryList[idx].m_Distance = 0.0f;
		Visited[vc] = idx; 
		vc++;
	}

	bool foundEnd = false;
	TIndex pidx = (numeric_limits<unsigned int>::max)();
	p_Distance = (numeric_limits<float>::max)();
	while(!S.empty())
	{
		pidx = S.top().second;
		S.pop();

		if(p_Field->vvaluex(pidx) != 0 && m_AuxiliaryList[pidx].m_Distance != 0.0f)
		{ 
			foundEnd = true; 
			p_Distance = m_AuxiliaryList[pidx].m_Distance;
			break; 
		}

		if(m_AuxiliaryList[pidx].m_IsKnown) continue;
		m_AuxiliaryList[pidx].m_IsKnown = true;

		const TAdjacencyStruct & padj = m_AdjacencyList[pidx];
		const float & pdist = m_AuxiliaryList[pidx].m_Distance;
		for(n=0; n<padj.m_EdgeCount; n++)
		{
			const TIndex npidx = padj.m_Edges[n].m_ToVertex;
			if(!m_AuxiliaryList[npidx].m_IsKnown ) 
			{
				// Compute distance from p to np and see if distance is less than current distance
				const float & weight = WEIGHTS[ padj.m_Edges[n].m_Weight ];

				const float t = pdist + weight; // Dijkstra
				float & npdist = m_AuxiliaryList[npidx].m_Distance;
				if( t < npdist )
				{
					npdist = t;
					m_AuxiliaryList[npidx].m_Previous = pidx;
					Visited[vc] = npidx; vc++;

					S.push( TQueue::value_type(npdist,npidx) );
				}
			}
		}
	}

	// Reset 
	{
		for(unsigned int i=0; i<vc; i++)
		{
			m_AuxiliaryList[Visited[i]] = m_InitialAuxiliaryList;
		}
	}


	return foundEnd ? pidx : TIndexedOrigins::INVALIDINDEX;
}

//unsigned int TDeltaOmega::findNearestWithFilter(unsigned int p_Start, float & p_Distance, TTypedFieldInterface<float> * p_Cost, const TAbstractFilter * p_Filter, TIndexedOrigins_Vector * p_Output)
unsigned int TDeltaOmega::findNearestWithTargetFilterAndDomainFilter(unsigned int p_Start, float & p_Distance, TTypedFieldInterface<float> * p_Cost, const TAbstractFilter * p_TargetFilter, const TAbstractFilter * p_DomainFilter, TIndexedOrigins_Vector * p_Output)
{
	
	unsigned int n;
	static TQueue S;
	S = TQueue();
	vc = 0;
	const unsigned int idx = p_Start;
	S.push( TQueue::value_type(0.0f, idx));
	m_AuxiliaryList[idx].m_Distance = 0.0f;
	Visited[vc] = idx; 
	vc++;

	/*
	for(n=0; n<p_Start->size(); n++)
	{
		const unsigned int idx = (*p_Start)[n];
		S.push( TQueue::value_type(0.0f, idx));
		m_AuxiliaryList[idx].m_Distance = 0.0f;
		Visited[vc] = idx; 
		vc++;
	}
	*/

	bool foundEnd = false;
	TIndex pidx = (numeric_limits<unsigned int>::max)(); 
	p_Distance = (numeric_limits<float>::max)();
	while(!S.empty())
	{
		pidx = S.top().second;
		S.pop();

		if( p_TargetFilter->testx(pidx) && pidx != p_Start )
		{ 
			foundEnd = true; 
			p_Distance = m_AuxiliaryList[pidx].m_Distance;
			break; 
		}

		if(m_AuxiliaryList[pidx].m_IsKnown) continue;
		m_AuxiliaryList[pidx].m_IsKnown = true;

		const TAdjacencyStruct & padj = m_AdjacencyList[pidx];
		const float & pdist = m_AuxiliaryList[pidx].m_Distance;
		for(n=0; n<padj.m_EdgeCount; n++)
		{
			const TIndex npidx = padj.m_Edges[n].m_ToVertex;
			if(!m_AuxiliaryList[npidx].m_IsKnown && (p_DomainFilter->testx(npidx) || p_TargetFilter->testx(npidx)) ) 
			{
				// Compute distance from p to np and see if distance is less than current distance
	
				// Normal path
				const float & weight = WEIGHTS[ padj.m_Edges[n].m_Weight ];

				const float t = pdist + powf( p_Cost->vvaluex(npidx) * weight, 1.2f );
				float & npdist = m_AuxiliaryList[npidx].m_Distance;
				if( t < npdist )
				{
					npdist = t;
					m_AuxiliaryList[npidx].m_Previous = pidx;
					Visited[vc] = npidx; vc++;

					S.push( TQueue::value_type(npdist,npidx) );
				}

				/*
				// MIN MAX PATH 
				const float t = max(pdist, p_Cost->vvaluex(npidx) );
				float & npdist = m_AuxiliaryList[npidx].m_Distance;
				if( t < npdist )
				{
					npdist = t;
					m_AuxiliaryList[npidx].m_Previous = pidx;
				}
				
				Visited[vc] = npidx; vc++;
				S.push( TQueue::value_type(npdist,npidx) );
				*/
			}
		}
	}
	TIndex end = pidx;

	if(p_Output)
	{
		p_Output->add(pidx);
		while(pidx != p_Start)
		{
			pidx = m_AuxiliaryList[pidx].m_Previous;
			if(pidx == TIndexedOrigins::INVALIDINDEX) break;
			p_Output->add(pidx);
		}
	}

	// Reset 
	{
		for(unsigned int i=0; i<vc; i++)
		{
			m_AuxiliaryList[Visited[i]] = m_InitialAuxiliaryList;
		}
	}


	return foundEnd ? end : TIndexedOrigins::INVALIDINDEX;
}


void TDeltaOmega::findAllNonZerosWithinDistance(const unsigned int p_Start, const TTypedFieldInterface<unsigned int>* p_Field, const float p_Distance, vector<unsigned int> * p_Result, vector<float> * p_Distances, TFilter * p_Filter)
{
	set<unsigned int> starts;
	starts.insert(p_Start);
	findAllNonZerosWithinDistance(&starts, p_Field, p_Distance, p_Result, p_Distances, p_Filter);
}

void TDeltaOmega::findAllNonZerosWithinDistance(set<unsigned int> * p_Start, const TTypedFieldInterface<unsigned int>* p_Field, const float p_Distance, vector<unsigned int> * p_Result, vector<float> * p_Distances, TFilter * p_Filter)
{
	if(p_Field->getMaxIndex() != m_AuxiliaryList.size()) throw string("fillZerosByPropagation(): can only be applied to boundary indexed field");

	static TQueue S;
	S = TQueue();
	vc = 0;
	unsigned int n;
	set<unsigned int>::iterator it;
	for(it=p_Start->begin(); it!=p_Start->end(); it++)
	{
		const unsigned int idx = *it;
		S.push( TQueue::value_type(0.0f, idx));
		m_AuxiliaryList[idx].m_Distance = 0.0f;
		Visited[vc] = idx; 
		vc++;
	}

	while(!S.empty())
	{
		TIndex pidx = S.top().second;
		S.pop();

		if(m_AuxiliaryList[pidx].m_IsKnown) continue;
		if(m_AuxiliaryList[pidx].m_Distance > p_Distance) continue;
		m_AuxiliaryList[pidx].m_IsKnown = true;
		
		if(p_Field->vvaluex(pidx) != 0 && m_AuxiliaryList[pidx].m_Distance != 0.0f) 
		{
			p_Result->push_back(pidx);
			if(p_Distances) p_Distances->push_back(m_AuxiliaryList[pidx].m_Distance);
		}

		const TAdjacencyStruct & padj = m_AdjacencyList[pidx];
		const float & pdist = m_AuxiliaryList[pidx].m_Distance;
		for(n=0; n<padj.m_EdgeCount; n++)
		{
			const TIndex npidx = padj.m_Edges[n].m_ToVertex;
			if(!m_AuxiliaryList[npidx].m_IsKnown && (!p_Filter || p_Filter->testx(npidx)) ) 
			{
				// Compute distance from p to np and see if distance is less than current distance
				const float & weight = WEIGHTS[ padj.m_Edges[n].m_Weight ];

				const float t = pdist + weight; // Dijkstra
				float & npdist = m_AuxiliaryList[npidx].m_Distance;
				if( t < npdist )
				{
					npdist = t;
//					m_AuxiliaryList[npidx].m_Previous = pidx;
					Visited[vc] = npidx; vc++;

					S.push( TQueue::value_type(npdist,npidx) );
				}
			}
		}
	}


	// Reset 
	{
		for(unsigned int i=0; i<vc; i++)
		{
			m_AuxiliaryList[Visited[i]] = m_InitialAuxiliaryList;
		}
	}
}



void TDeltaOmega::fillZerosByPropagation(TTypedFieldInterface<TVector3>* p_Field)
{
	if(p_Field->getMaxIndex() != m_AuxiliaryList.size()) throw string("fillZerosByPropagation(): can only be applied to boundary indexed field");

	static TQueue S;
	S = TQueue();

	vc = 0;
	unsigned int x, n;
	for(x=0; x<p_Field->getMaxIndex(); x++)
	{
		if(p_Field->vvaluex(x).norm2() > 0.01f)
		{
			S.push( TQueue::value_type(0.0f, x) );
			m_AuxiliaryList[x].m_Distance = 0.0f;
			Visited[vc] = x; 
			vc++;
		}
	}

	while(!S.empty())
	{
		TIndex pidx = S.top().second;
		S.pop();

		if(m_AuxiliaryList[pidx].m_IsKnown) continue;
		m_AuxiliaryList[pidx].m_IsKnown = true;

		const TAdjacencyStruct & padj = m_AdjacencyList[pidx];
		const float & pdist = m_AuxiliaryList[pidx].m_Distance;
		for(n=0; n<padj.m_EdgeCount; n++)
		{
			const TIndex npidx = padj.m_Edges[n].m_ToVertex;
			if(!m_AuxiliaryList[npidx].m_IsKnown ) 
			{
				// Compute distance from p to np and see if distance is less than current distance
				const float & weight = WEIGHTS[ padj.m_Edges[n].m_Weight ];

				const float t = pdist + weight; // Dijkstra
				float & npdist = m_AuxiliaryList[npidx].m_Distance;
				if( t < npdist )
				{
					npdist = t;
					m_AuxiliaryList[npidx].m_Previous = pidx;
					p_Field->wvaluex(npidx) = p_Field->vvaluex(pidx);
					Visited[vc] = npidx; vc++;

					S.push( TQueue::value_type(npdist,npidx) );
				}
			}
		}
	}


	// Reset 
	{
		for(unsigned int i=0; i<vc; i++)
		{
			m_AuxiliaryList[Visited[i]] = m_InitialAuxiliaryList;
		}
	}
}



void TDeltaOmega::fillZerosByPropagation(TTypedFieldInterface<unsigned int>* p_Field)
{
	if(p_Field->getMaxIndex() != m_AuxiliaryList.size()) throw string("fillZerosByPropagation(): can only be applied to boundary indexed field");

	static TQueue S;
	S = TQueue();

	vc = 0;
	unsigned int x, n;
	for(x=0; x<p_Field->getMaxIndex(); x++)
	{
		if(p_Field->vvaluex(x) != 0)
		{
			S.push( TQueue::value_type(0.0f, x) );
			m_AuxiliaryList[x].m_Distance = 0.0f;
			Visited[vc] = x; 
			vc++;
		}
	}

	while(!S.empty())
	{
		TIndex pidx = S.top().second;
		S.pop();

		if(m_AuxiliaryList[pidx].m_IsKnown) continue;
		m_AuxiliaryList[pidx].m_IsKnown = true;

		const TAdjacencyStruct & padj = m_AdjacencyList[pidx];
		const float & pdist = m_AuxiliaryList[pidx].m_Distance;
		for(n=0; n<padj.m_EdgeCount; n++)
		{
			const TIndex npidx = padj.m_Edges[n].m_ToVertex;
			if(!m_AuxiliaryList[npidx].m_IsKnown ) 
			{
				// Compute distance from p to np and see if distance is less than current distance
				const float & weight = WEIGHTS[ padj.m_Edges[n].m_Weight ];

				const float t = pdist + weight; // Dijkstra
				float & npdist = m_AuxiliaryList[npidx].m_Distance;
				if( t < npdist )
				{
					npdist = t;
					m_AuxiliaryList[npidx].m_Previous = pidx;
					p_Field->wvaluex(npidx) = p_Field->vvaluex(pidx);
					Visited[vc] = npidx; vc++;

					S.push( TQueue::value_type(npdist,npidx) );
				}
			}
		}
	}


	// Reset 
	{
		for(unsigned int i=0; i<vc; i++)
		{
			m_AuxiliaryList[Visited[i]] = m_InitialAuxiliaryList;
		}
	}
}

void TDeltaOmega::fillZerosByPropagationAndDomain(TTypedFieldInterface<unsigned int>* p_Field, TTypedFieldInterface<unsigned int>* p_Domain, unsigned int p_DomainIdx)
{
	if(p_Field->getMaxIndex() != m_AuxiliaryList.size()) throw string("fillZerosByPropagation(): can only be applied to boundary indexed field");

	static TQueue S;
	S = TQueue();

	vc = 0;
	unsigned int x, n;
	for(x=0; x<p_Field->getMaxIndex(); x++)
	{
		if(p_Field->vvaluex(x) != 0)
		{
			S.push( TQueue::value_type(0.0f, x) );
			m_AuxiliaryList[x].m_Distance = 0.0f;
			Visited[vc] = x; 
			vc++;
		}
	}

	while(!S.empty())
	{
		TIndex pidx = S.top().second;
		S.pop();

		if(m_AuxiliaryList[pidx].m_IsKnown) continue;
		m_AuxiliaryList[pidx].m_IsKnown = true;

		const TAdjacencyStruct & padj = m_AdjacencyList[pidx];
		const float & pdist = m_AuxiliaryList[pidx].m_Distance;
		for(n=0; n<padj.m_EdgeCount; n++)
		{
			const TIndex npidx = padj.m_Edges[n].m_ToVertex;
			if(!m_AuxiliaryList[npidx].m_IsKnown && p_Domain->vvaluex(npidx) == p_DomainIdx ) 
			{
				// Compute distance from p to np and see if distance is less than current distance
				const float & weight = WEIGHTS[ padj.m_Edges[n].m_Weight ];

				const float t = pdist + weight; // Dijkstra
				float & npdist = m_AuxiliaryList[npidx].m_Distance;
				if( t < npdist )
				{
					npdist = t;
					m_AuxiliaryList[npidx].m_Previous = pidx;
					p_Field->wvaluex(npidx) = p_Field->vvaluex(pidx);
					Visited[vc] = npidx; vc++;

					S.push( TQueue::value_type(npdist,npidx) );
				}
			}
		}
	}


	// Reset 
	{
		for(unsigned int i=0; i<vc; i++)
		{
			m_AuxiliaryList[Visited[i]] = m_InitialAuxiliaryList;
		}
	}
}


void TDeltaOmega::fillZerosByPropagation(TTypedFieldInterface<float>* p_Field)
{
	if(p_Field->getMaxIndex() != m_AuxiliaryList.size()) throw string("fillZerosByPropagation(): can only be applied to boundary indexed field");

	static TQueue S;
	S = TQueue();

	vc = 0;
	unsigned int x, n;
	for(x=0; x<p_Field->getMaxIndex(); x++)
	{
		if(p_Field->vvaluex(x) != 0)
		{
			const float dist = 0.0f;
			S.push( TQueue::value_type(dist, x) );
			m_AuxiliaryList[x].m_Distance = dist;
			Visited[vc] = x; 
			vc++;
		}
	}

	while(!S.empty())
	{
		TIndex pidx = S.top().second;
		S.pop();

		if(m_AuxiliaryList[pidx].m_IsKnown) continue;
		m_AuxiliaryList[pidx].m_IsKnown = true;

		const TAdjacencyStruct & padj = m_AdjacencyList[pidx];
		const float & pdist = m_AuxiliaryList[pidx].m_Distance;
		for(n=0; n<padj.m_EdgeCount; n++)
		{
			const TIndex npidx = padj.m_Edges[n].m_ToVertex;
			if(!m_AuxiliaryList[npidx].m_IsKnown ) 
			{
				// Compute distance from p to np and see if distance is less than current distance
				const float & weight = WEIGHTS[ padj.m_Edges[n].m_Weight ];

				const float t = pdist + weight; // Dijkstra
				float & npdist = m_AuxiliaryList[npidx].m_Distance;
				if( t < npdist )
				{
					npdist = t;
					m_AuxiliaryList[npidx].m_Previous = pidx;
					p_Field->wvaluex(npidx) = p_Field->vvaluex(pidx);
					Visited[vc] = npidx; vc++;

					S.push( TQueue::value_type(npdist,npidx) );
				}
			}
		}
	}


	// Reset 
	{
		for(unsigned int i=0; i<vc; i++)
		{
			m_AuxiliaryList[Visited[i]] = m_InitialAuxiliaryList;
		}
	}
}



void TDeltaOmega::fillZerosByPropagationUsingNormals(TTypedFieldInterface<unsigned int>* p_Field, const TTypedFieldInterface<TVector3>* p_BoundaryNormals, TTypedFieldInterface<TIndexedOrigins*>* p_DebugOrigins, float p_MaximumDot)
{
	if(p_Field->getMaxIndex() != m_AuxiliaryList.size()) throw string("fillZerosByPropagation(): can only be applied to boundary indexed field");

	static TQueue2 S;
	S = TQueue2();

	vc = 0;
	unsigned int n;
	for(unsigned int x=0; x<p_Field->getMaxIndex(); x++)
	{
		if(p_Field->vvaluex(x) != 0)
		{
			m_AuxiliaryList[x].m_IsKnown = true;
			m_AuxiliaryList[x].m_Previous = x;
			Visited[vc] = x; 
			vc++;

			const TAdjacencyStruct & padj = m_AdjacencyList[x];
			for(n=0; n<padj.m_EdgeCount; n++)
			{
				const TIndex npidx = padj.m_Edges[n].m_ToVertex;
				if(p_Field->vvaluex(npidx) != 0) continue;
				
				const float dot = 1.0f - fabs( p_BoundaryNormals->vvaluex(npidx).dot( p_BoundaryNormals->vvaluex(x) ) );
				S.insert( TQueue2::value_type(dot, npidx) );
				m_AuxiliaryList[npidx].m_Previous = x;
				m_AuxiliaryList[npidx].m_Distance = dot;
				Visited[vc] = npidx; 
				vc++;
			}
		}
	}

	while(!S.empty())
	{
		const float olddot = S.begin()->first;
		TIndex pidx = S.begin()->second;
		S.erase(S.begin());

		if(m_AuxiliaryList[pidx].m_IsKnown) continue;
		m_AuxiliaryList[pidx].m_IsKnown = true;

		const TAdjacencyStruct & padj = m_AdjacencyList[pidx];

		for(n=0; n<padj.m_EdgeCount; n++)
		{
			const TIndex npidx = padj.m_Edges[n].m_ToVertex;
			if( m_AuxiliaryList[npidx].m_IsKnown ) continue;
			
			const float dot = 1.0f - fabs( p_BoundaryNormals->vvaluex(npidx).dot( p_BoundaryNormals->vvaluex(m_AuxiliaryList[pidx].m_Previous) ) );
			//if(1.0f-dot < p_MaximumDot) continue;
			const float newdot = olddot + dot;

			if(newdot < m_AuxiliaryList[npidx].m_Distance )
			{
				m_AuxiliaryList[npidx].m_Distance = newdot;
				m_AuxiliaryList[npidx].m_Previous = m_AuxiliaryList[pidx].m_Previous;
				S.insert( TQueue2::value_type(newdot, npidx) );
				Visited[vc] = npidx; 
				vc++;
			}
		}
	}

	// For each voxel, set cluster of source voxel
	{
		for(unsigned int i=0; i<vc; i++)
		{
			unsigned int pidx = Visited[i];
			p_Field->wvaluex(pidx) = p_Field->vvaluex( m_AuxiliaryList[pidx].m_Previous );
			if(p_DebugOrigins) 
			{
				if(!p_DebugOrigins->vvaluex(pidx)) p_DebugOrigins->wvaluex(pidx) = new TIndexedOrigins_Vector();
				p_DebugOrigins->wvaluex(pidx)->castVector()->clear();
				p_DebugOrigins->wvaluex(pidx)->castVector()->push_back( m_AuxiliaryList[pidx].m_Previous );
			}
		}
	}

	// Reset 
	{
		for(unsigned int i=0; i<vc; i++)
		{
			m_AuxiliaryList[Visited[i]] = m_InitialAuxiliaryList;
		}
	}

}





void TDeltaOmega::fillZerosByPropagation(TTypedFieldInterface<TIndexedOrigins*>* p_Field)
{
	if(p_Field->getMaxIndex() != m_AuxiliaryList.size()) throw string("fillZerosByPropagation(): can only be applied to boundary indexed field");

	static TQueue S;
	S = TQueue();

	vc = 0;
	unsigned int x, n;
	for(x=0; x<p_Field->getMaxIndex(); x++)
	{
		if(p_Field->vvaluex(x) != 0)
		{
			S.push( TQueue::value_type(0.0f, x) );
			m_AuxiliaryList[x].m_Distance = 0.0f;
			Visited[vc] = x; 
			vc++;
		}
	}

	while(!S.empty())
	{
		TIndex pidx = S.top().second;
		S.pop();

		if(m_AuxiliaryList[pidx].m_IsKnown) continue;
		m_AuxiliaryList[pidx].m_IsKnown = true;

		const TAdjacencyStruct & padj = m_AdjacencyList[pidx];
		const float & pdist = m_AuxiliaryList[pidx].m_Distance;
		for(n=0; n<padj.m_EdgeCount; n++)
		{
			const TIndex npidx = padj.m_Edges[n].m_ToVertex;
			if(!m_AuxiliaryList[npidx].m_IsKnown ) 
			{
				// Compute distance from p to np and see if distance is less than current distance
				const float & weight = WEIGHTS[ padj.m_Edges[n].m_Weight ];

				const float t = pdist + weight; // Dijkstra
				float & npdist = m_AuxiliaryList[npidx].m_Distance;
				if( t < npdist )
				{
					npdist = t;
					m_AuxiliaryList[npidx].m_Previous = pidx;
					if(p_Field->vvaluex(npidx)) { delete p_Field->vvaluex(npidx); p_Field->wvaluex(npidx) = 0; }
					p_Field->wvaluex(npidx) = new TIndexedOrigins_Vector( *p_Field->vvaluex(pidx)->castConstVector() );
					Visited[vc] = npidx; vc++;

					S.push( TQueue::value_type(npdist,npidx) );
				}
			}
		}
	}


	// Reset 
	{
		for(unsigned int i=0; i<vc; i++)
		{
			m_AuxiliaryList[Visited[i]] = m_InitialAuxiliaryList;
		}
	}
}


void TDeltaOmega::computeDtAndFt(const TTypedFieldInterface<unsigned int>* p_Seeds, unsigned int p_SeedId, TTypedFieldInterface<float> * p_Dt, TTypedFieldInterface<TIndexedOrigins*>* p_Ft)
{
	static TQueue S;
	S = TQueue();

	vc = 0;
	unsigned int x, n;
	for(x=0; x<p_Seeds->getMaxIndex(); x++)
	{
		if(p_Seeds->vvaluex(x) == p_SeedId)
		{
			S.push( TQueue::value_type(0.0f, x) );
			m_AuxiliaryList[x].m_Distance = 0.0f;
			m_AuxiliaryList[x].m_Previous = x;
			Visited[vc] = x; 
			vc++;
		}
	}

	while(!S.empty())
	{
		TIndex pidx = S.top().second;
		S.pop();
		if(m_AuxiliaryList[pidx].m_IsKnown) continue;
		m_AuxiliaryList[pidx].m_IsKnown = true;
		if(p_Dt) p_Dt->wvaluex(pidx) = m_AuxiliaryList[pidx].m_Distance;

		const float & pdist = m_AuxiliaryList[pidx].m_Distance;
		const TAdjacencyStruct & padj = m_AdjacencyList[pidx];
		for(n=0; n<padj.m_EdgeCount; n++)
		{
			const TIndex npidx = padj.m_Edges[n].m_ToVertex;
			if(!m_AuxiliaryList[npidx].m_IsKnown 
				&& p_Seeds->vvaluex(npidx) == 0 
				) 
			{
				// Compute distance from p to np and see if distance is less than current distance
				const float & weight = WEIGHTS[ padj.m_Edges[n].m_Weight ];

				const float t = pdist + weight; // Dijkstra
				float & npdist = m_AuxiliaryList[npidx].m_Distance;
				if( t < npdist )
				{
					npdist = t;
					m_AuxiliaryList[npidx].m_Previous = m_AuxiliaryList[pidx].m_Previous;

					if(p_Ft)
					{
						if(!p_Ft->vvaluex(npidx)) p_Ft->wvaluex(npidx) = new TIndexedOrigins_Vector();
						p_Ft->vvaluex(npidx)->castVector()->clear(); 
						p_Ft->wvaluex(npidx)->castVector()->push_back( m_AuxiliaryList[npidx].m_Previous );
					}
					Visited[vc] = npidx; 
					vc++;

					S.push( TQueue::value_type(npdist,npidx) );
				}
			}
		}
	}

	// Reset 
	{
		for(unsigned int i=0; i<vc; i++)
		{
			m_AuxiliaryList[Visited[i]] = m_InitialAuxiliaryList;
		}
	}
}


void TDeltaOmega::getEquidistantNeighbors(const unsigned int p_Idx, const TCoord3 p, const float p_Tolerance, vector<unsigned int> * p_Neighbors)
{
	const float pdist = p.distance( m_Omega->m_BoundaryIndexField->vidx2coord(p_Idx) ) * p_Tolerance;

	queue<unsigned int> S;
	S.push(p_Idx);
//	m_AuxiliaryList[x].m_Distance = 0.0f;
	vc = 0;


	while(!S.empty())
	{
		TIndex idx = S.front();
		S.pop();
		m_AuxiliaryList[idx].m_IsKnown = true;
		Visited[vc] = idx;
		vc++;
		p_Neighbors->push_back(idx); 

		const TAdjacencyStruct & padj = m_AdjacencyList[idx];
		for(unsigned int n=0; n<padj.m_EdgeCount; n++)
		{
			const TIndex npidx = padj.m_Edges[n].m_ToVertex;
			if( m_AuxiliaryList[npidx].m_IsKnown ) continue;

			const float npdist = p.distance( m_Omega->m_BoundaryIndexField->vidx2coord(npidx) );
			if(npdist < pdist)
			{
				S.push(npidx);
				m_AuxiliaryList[npidx].m_IsKnown = true;
			}
		}
	}

	// Reset 
	{
		for(unsigned int i=0; i<vc; i++)
		{
			m_AuxiliaryList[Visited[i]] = m_InitialAuxiliaryList;
		}
	}
}


void TDeltaOmega::makeMinColoring(TTypedFieldInterface<unsigned int> * p_Field)
{
	if(p_Field->getMaxIndex() != m_AuxiliaryList.size()) throw string("fillZerosByPropagation(): can only be applied to boundary indexed field");

	typedef unsigned int TCount;
	typedef map<unsigned int, map<unsigned int, TCount> > TNeighborMap;
	TNeighborMap NeighborMap;

	// Determine neighborhood relations
	for(unsigned int pidx=0; pidx<p_Field->getMaxIndex(); pidx++)
	{
		const unsigned int segid = p_Field->vvaluex(pidx);
		if(segid == 0) continue;

		const TAdjacencyStruct & padj = m_AdjacencyList[pidx];
		for(unsigned n=0; n<padj.m_EdgeCount; n++)
		{
			const TIndex npidx = padj.m_Edges[n].m_ToVertex;
			if(p_Field->vvaluex(npidx) != segid)
			{
				NeighborMap[segid][ p_Field->vvaluex(npidx) ]++; // count neighboring voxels
			}
		}
	}
	
	vector<unsigned int> Colors;
	if(NeighborMap.size() == 0) throw string("TDeltaOmega::makeMinColoring(): NeighborMap.size() == 0");
	Colors.resize( NeighborMap.rbegin()->first + 1, 0 );

	// Assign colors to segid
	map<unsigned int, TCount>::iterator it;
	set<unsigned int> NonFreeColors;
	TNeighborMap::iterator jt;
	for(jt = NeighborMap.begin(); jt != NeighborMap.end(); jt++)
	{
		NonFreeColors.clear();
		unsigned int segid = jt->first;
		if(segid == 0) continue;
		for(it = jt->second.begin(); it != jt->second.end(); it++)
		{
			if(it->second < 10) continue;

			unsigned int colorit = Colors[ it->first ];
			if(colorit != 0)
			{
				NonFreeColors.insert( colorit );
			}
		}

		unsigned int lowestfree = 1;
		while( NonFreeColors.find( lowestfree ) != NonFreeColors.end() )
		{
			lowestfree++;
		}
		
		Colors[segid] = lowestfree;
	}
	
	for(unsigned int x=0; x<p_Field->getMaxIndex(); x++)
	{
		p_Field->wvaluex(x) = Colors[p_Field->vvaluex(x)];
	}
	p_Field->getLayer()->onFieldChanged();
	p_Field->getLayer()->m_ColorMap = p_Field->getLayer()->m_AllColorMaps[5];
}





unsigned int TDeltaOmega::findConnectedComponents(const TFilter * p_Filter, TTypedFieldInterface<unsigned int>* p_Output, bool p_InverseFilter, vector<set<unsigned int> >  * p_Sizes)
{
	unsigned int ClusterId = 1;
	if(p_Sizes) p_Sizes->push_back( set<unsigned int>() );
	if(p_Sizes) p_Sizes->push_back( set<unsigned int>() );
	unsigned int x;
	for(x=0; x<p_Output->getMaxIndex(); x++)
	{
		const TCoord3 p = p_Output->getIndexField()->vidx2coord(x);
		if( p_Output->vvaluex(x) != 0) continue;
		if( (p_InverseFilter && p_Filter->test(p))  
			||(!p_InverseFilter && !p_Filter->test(p) ) )
		{
			continue;
		}

		queue<unsigned int> S;
		S.push(x);

		while(!S.empty())
		{
			const TIndex idx = S.front();
			S.pop();
			const TCoord3 p = m_Omega->m_BoundaryIndexField->vidx2coord(idx);
			p_Output->wvaluep(p) = ClusterId;
			if(p_Sizes) (*p_Sizes)[ClusterId].insert( idx );
		
			const TAdjacencyStruct & padj = m_AdjacencyList[idx];
			for(unsigned int n=0; n<padj.m_EdgeCount; n++)
			{
				const TIndex npidx = padj.m_Edges[n].m_ToVertex;
				const TCoord3 np = p_Output->getIndexField()->vidx2coord(npidx);

				if( p_Output->vvaluep(np) != 0 ) continue;
				if( (!p_InverseFilter && p_Filter->test(np)) 
					||
					(p_InverseFilter && !p_Filter->test(np)) 
					)
				{
					p_Output->wvaluep(np) = ClusterId;
					S.push(npidx);
				}
			}
		}

		ClusterId++;
		if(p_Sizes) p_Sizes->push_back( set<unsigned int>() );
	}

	return ClusterId;
}


void TDeltaOmega::findConnectedComponents(const TIndexedOrigins_Vector * p_Input, vector<set<unsigned int> > * p_Sizes, bool p_Reset)
{
	unsigned int x;
	if(p_Sizes) p_Sizes->push_back( set<unsigned int>() );

	const unsigned int NOTASSIGNED = 5;
	const unsigned int ASSIGNED = 6;
	for(x=0; x<p_Input->size(); x++)
	{
		const unsigned int idx = (*p_Input)[x];
		m_ComponentList[idx].m_Component = NOTASSIGNED;
	}

	for(x=0; x<p_Input->size(); x++)
	{
		const unsigned int idx = (*p_Input)[x];
		if( m_ComponentList[idx].m_Component == ASSIGNED ) continue;
		m_ComponentList[idx].m_Component = ASSIGNED;

		p_Sizes->push_back( set<unsigned int>() );
		p_Sizes->back().insert( idx );

		queue<unsigned int> Q;
		Q.push( idx );

		while(!Q.empty())
		{
			const unsigned int idx = Q.front();
			Q.pop();

			const TAdjacencyStruct & padj = m_AdjacencyList[idx];
			for(unsigned int n=0; n<padj.m_EdgeCount; n++)
			{
				const TIndex & npidx = padj.m_Edges[n].m_ToVertex;
				if( m_ComponentList[npidx].m_Component == NOTASSIGNED ) 
				{
					m_ComponentList[npidx].m_Component = ASSIGNED;
					p_Sizes->back().insert( npidx );
					Q.push( npidx );
				}
			}
		}
	}

	if(g_Settings.m_DebugOutput)
	{
		TTypedFieldInterface<unsigned char> * debug = static_cast<TTypedFieldInterface<unsigned char>*>(  g_Mediator.getCurrentLayerSet()->newField(TType::TYPE_UCHAR, m_Omega->m_BoundaryIndexField, "findConnectedComponents debug" ) );
		debug->clear();
		for(x=1; x<p_Sizes->size(); x++)
		{
			std::set<unsigned int>::iterator it;
			for(it = (*p_Sizes)[x].begin(); it != (*p_Sizes)[x].end(); it++)
			{
				debug->wvaluex(*it) = x;
			}
		}
		debug->getLayer()->onFieldChanged();
	}


	// Reset 
	for(x=0; x<p_Input->size(); x++)
	{
		const unsigned int idx = (*p_Input)[x];
		m_ComponentList[idx] = m_InitialComponentList;
	}
}


void TOmega::writeToStream(std::ofstream * s) const
{
	this->m_ForegroundIndexField->writeToStream(s);
	this->m_BoundaryIndexField->writeToStream(s);
	char bg = this->m_BackgroundIndexField.get() == 0 ? 0 : 1;
	s->write( &bg, 1 );
	if(bg) 
	{
		this->m_BackgroundIndexField->writeToStream(s);
		this->m_BackgroundBoundaryIndexField->writeToStream(s);
	}
	
}

TOmega * TOmega::readFromStream(std::ifstream * s)
{
	TOmega * Omega = new TOmega();
	Omega->m_ForegroundIndexField.reset( TIndexField3::readFromStream(s) );
	const TIndexField3 * IndexField = static_cast<TIndexField3*>( Omega->m_ForegroundIndexField.get() );
	Omega->m_BoundaryIndexField.reset( TSubIndexMapper::readFromStream(s, IndexField) );
	Omega->m_BackgroundBoundaryIndexField = Omega->m_BoundaryIndexField;
	char bg = 0; 
	s->read( &bg, 1 );
	assert(bg == 0 || bg == 1);
	if(bg) 
	{
		Omega->m_BackgroundIndexField.reset( TIndexField3::readFromStream(s) ); 
		Omega->m_BackgroundBoundaryIndexField.reset( TSubIndexMapper::readFromStream(s, Omega->m_BackgroundIndexField.get()) );
	}
	
	Omega->initDeltaOmega();

	return Omega;
}



