#include "dt.h"


#include "field.h"
#include "layer.h"
#include "abstractfilter.h"
#include "cdt.h"
#include "cdt3.h"
#include "exactdt.h"
#include "lotufo.h"
#include "mullikin.h"
#include "edt3.h"

#include "main.h"

const float TDt::INFINITY = 1.0e8;

TDt::TDt()
	: m_AlreadyPerformed(false)
{
}

void TDt::getNeighbors(const TCoord2 & p, int s, vector<TCoord2> & N)
{
	N.clear();
	for(int i=p.x-1; i<=p.x+1; i++)
	{
		for(int j=p.y-1; j<=p.y+1; j++)
		{
			if(i == p.x && j == p.y) continue; 
			if(s == 4 && i!=p.x && j!=p.y) continue;		
			
			N.push_back( TCoord2(i,j) ); 
		}
	}
}

TDtComparison::TDtComparison(shared_ptr<TDt> p_Dt1,shared_ptr<TDt> p_Dt2, bool p_StoreResult)
	: m_Dt1(p_Dt1)
	, m_Dt2(p_Dt2)
{
	m_DtType = m_Dt1->getDtType();
	m_Tolerance = m_Dt1->getTolerance();
	m_NeighborCount = m_Dt1->getNeighborCount();
	m_LatexName = m_Dt1->getLatexName();
	m_ShortName = m_Dt1->getShortName();
	m_Time = m_Dt1->getTime();

	float DistDiffMax = -1;
	float DistDiffSum = 0;
	float DistDiffMin = 1.0e+8; 
	int DistDiffCount = 0;

	int OrigDiffMax = -1;
	int OrigDiffSum = 0;
	int OrigDiffMin = 1.0e+8;
	int OrigDiffCount = 0;

	float RelOrigDiffMax = -1.0f;
	float RelOrigDiffSum = 0.0f;

	int ObjectPixelCount=0;

	TFloatField3 * fd1 = p_Dt1->getDistance().get();
	TFloatField3 * fd2 = p_Dt2->getDistance().get();
	TOrigSetField2 * fo1 = p_Dt1->getOrigins().get();
	TOrigSetField2 * fo2 = p_Dt2->getOrigins().get();
	if(fd1->getMaxX() != fd2->getMaxX() && fd1->getMaxY() != fd2->getMaxY()) throw string("comparison: size does not match");

	if(p_StoreResult) 
	{
		m_DistanceDiff.reset( new TFloatField3(fd1->getMaxX(), fd1->getMaxY(), fd1->getMaxZ()) );
		m_DistanceDiff->setEveryValue(0);
		m_OriginsDiff.reset( new TFloatField3(fd1->getMaxX(), fd1->getMaxY(), fd1->getMaxZ()) );
		m_OriginsDiff->setEveryValue(0);
	}

	for(int i=0;i<fd1->getMaxX();i++)
	{
		for(int j=0;j<fd1->getMaxY();j++)
		{
			if(fo2->value(i,j).size() == 0) 
			{
				if(p_StoreResult) m_DistanceDiff->value(i,j) = 0;
				continue; // Do not process non-object pixels
			}

			// Distance
			{
				const float d1 = fd1->value(i,j);
				const float d2 = fd2->value(i,j);
				float v = fabs(d1 - d2);

				if(v != 0) { DistDiffCount++; }
				if (DistDiffMax < v) DistDiffMax = v;
				if (DistDiffMin > v) DistDiffMin = v;
				DistDiffSum += v; 
				if(p_StoreResult) m_DistanceDiff->value(i,j) = v;
			}

			// Origins
			{
				const TOrigSet2 & os1 = fo1->value(i,j);
				const TOrigSet2 & os2 = fo2->value(i,j);

				// Only origin counts are compared!
				// Test/todo/debug
				if(true)
				{
					int v = abs( os1.size() - os2.size() );
					if( os1.size() > os2.size() ) 
					{
						*g_Log << "DtComparison warning in " << p_Dt1->getShortName() << ": os1.size() > os2.size() for (" << i << "," << j << ")\n";
					}

					if(v != 0) { OrigDiffCount++; }
					if (OrigDiffMax < v) OrigDiffMax = v;
					if (OrigDiffMin > v) OrigDiffMin = v;
					OrigDiffSum += v; 
					if(p_StoreResult) m_OriginsDiff->value(i,j) = v;
				}
				// Compare all origin locations, not only count
				else 
				{
					TOrigSet2::const_iterator it, jt;
					bool difference = false;

					for(it = os1.begin(); it != os1.end(); it++)
					{
						for(jt = os2.begin(); jt != os2.end(); jt++)
							if(jt->second == it->second) break;
						
						if(jt == os2.end()) // it not found in os2 not found
						{
							difference = true;
							break;
						}
					}

					if(!difference)
					{
						for(it = os2.begin(); it != os2.end(); it++)
						{
							for(jt = os1.begin(); jt != os1.end(); jt++)
								if(jt->second == it->second) break;
							
							if(jt == os1.end()) // it not found in os2 not found
							{
								difference = true;
								break;
							}
						}
					}

					if(difference)
					{
						m_OriginsDiff->value(i,j) = 1;
						OrigDiffCount++;
						OrigDiffSum += 1;
					}
					else
					{
						m_OriginsDiff->value(i,j) = 0;
					}

				}
				
								
				{
					float v1 = fo1->value(i,j).size();
					float v2 = fo2->value(i,j).size();
					float rdiff = fabs(v1-v2)/v2;
					if(rdiff > RelOrigDiffMax)
					{
						RelOrigDiffMax = rdiff;
					}
					RelOrigDiffSum += rdiff;
				}

	
				ObjectPixelCount++;
			}
		}
	}

	m_ObjectPixelCount = ObjectPixelCount;

	m_DistErrorCount = DistDiffCount;
	m_DistPercError = ObjectPixelCount != 0 ? 100.0f*(float)DistDiffCount/(float)ObjectPixelCount : 0;
	m_DistMaxError = DistDiffMax;
	m_DistMinError = DistDiffMin;
	m_DistAvgError = ObjectPixelCount != 0 ? (float)DistDiffSum/(float)ObjectPixelCount : 0;

	m_OrigErrorCount = OrigDiffCount;
	m_OrigPercError = ObjectPixelCount != 0 ? 100.0f*(float)OrigDiffCount/(float)ObjectPixelCount : 0;
	m_OrigMaxError = OrigDiffMax;
	m_OrigMinError = OrigDiffMin;
	m_OrigAvgError = ObjectPixelCount != 0 ? (float)OrigDiffSum/(float)ObjectPixelCount : 0;

	m_RelOrigMaxError = RelOrigDiffMax;
	m_RelOrigAvgError = ObjectPixelCount != 0 ? (float)RelOrigDiffSum/(float)ObjectPixelCount : 0; 

}


string TDtComparison::toText()
{
	string r = "";
	r += "=== Compare " + m_Dt1->getShortName() + " and " + m_Dt2->getShortName() + "\n";
	r += wxString::Format("Distance: %.0f perc, max: %f, min: %f, avg: %f\n", m_DistPercError, m_DistMaxError, m_DistMinError, m_DistAvgError);
	r += wxString::Format("Origins: %.0f perc, max: %f, min: %f, avg: %f\n", m_OrigPercError, m_OrigMaxError, m_OrigMinError, m_OrigAvgError);
	return r;
}


string TDtComparison::convertDistancesToLatex()
{
	string r = "";
	r += m_Dt1->getLatexName() + "&" + wxString::Format("%f & %f & %f", m_DistPercError, m_DistMaxError, m_DistMinError, m_DistAvgError).c_str() + " \\";
	return r;
}



void TDtComparison::writeToStream(std::ofstream & p_Stream)
{
	wxString str = m_LatexName.c_str();
	str.Replace(" ","_");
	p_Stream << str.c_str() << "\n";
	
	str = m_ShortName.c_str();
	str.Replace(" ","_");
	p_Stream << str.c_str() << "\n";

	p_Stream << m_Dt1->getDtType() << " " << m_Dt1->getTolerance() << " " << m_Dt1->getNeighborCount() << " ";
	p_Stream << m_Time << " ";
	p_Stream << m_ObjectPixelCount << " ";
	p_Stream << m_DistErrorCount << " ";
	p_Stream << m_DistPercError << " " << m_DistMaxError << " " << m_DistMinError << " " << m_DistAvgError << " ";
	p_Stream << m_OrigErrorCount << " ";
	p_Stream << m_OrigPercError << " " << m_OrigMaxError << " " << m_OrigMinError << " " << m_OrigAvgError << " ";
	p_Stream << m_RelOrigMaxError << " " << m_RelOrigAvgError << " ";
	p_Stream << "\n";
}

TDtComparison * TDtComparison::readFromStream(std::ifstream & p_Stream)
{
	string input;
	wxString wxstr;
	TDtComparison * dtc = new TDtComparison();

	p_Stream >> input;
	wxstr = wxString(input.c_str());
	wxstr.Replace("_"," ");
	dtc->m_LatexName = wxstr.c_str();
	
	p_Stream >> input;
	wxstr = wxString(input.c_str());
	wxstr.Replace("_"," ");
	dtc->m_ShortName= wxstr.c_str();

	int dttype;
	p_Stream >> dttype;
	dtc->m_DtType = static_cast<TDt::DT_TYPE>(dttype);
	p_Stream >> dtc->m_Tolerance >> dtc->m_NeighborCount;
	p_Stream >> dtc->m_Time;
	p_Stream >> dtc->m_ObjectPixelCount;
	p_Stream >> dtc->m_DistErrorCount;
	p_Stream >> dtc->m_DistPercError >> dtc->m_DistMaxError >> dtc->m_DistMinError >> dtc->m_DistAvgError;
	p_Stream >> dtc->m_OrigErrorCount;
	p_Stream >> dtc->m_OrigPercError >> dtc->m_OrigMaxError >> dtc->m_OrigMinError >> dtc->m_OrigAvgError;
	p_Stream >> dtc->m_RelOrigMaxError >> dtc->m_RelOrigAvgError;
	return dtc;
}


void TDtProcessor::performComparison(const string p_Caption, const TFloatField3 * p_Image, bool p_AddLayer, vector<shared_ptr<TDtComparison> > & p_Comparisons, vector<float> & p_Tolerances, bool p_DoAll)
{
	const TFloatField3 * Image = p_Image;
	shared_ptr<TFlagField3> FlagField( TFlagField3::construct(p_Image, 0, 0) );

	// Add boundary layer
	/*
	{
		shared_ptr<TLayer> Layer( new TLayer(FlagField) );
		shared_ptr<TFilter> Filter = shared_ptr<TFilter>(new TFloatFilter(&Layer->m_FilterMeasure));
		shared_ptr<TVoxelSet> vs = TVoxelSubset::constructByFilter( FlagField.get(), Filter  );
		Layer->m_VoxelSets[0]->setCheckedForRendering(false);
		vs->setName("Boundary voxels");
		vs->setCheckedForRendering(true);
		vs->setApplyColorMap(false);
		if(!MyApp::m_RenderBlackAndWhite)
		{
			vs->setRenderAsOverlay(true);
			vs->setDefaultColor( TColor3(1.0f,0.3f,1.0f) );
		}
		else
		{
			vs->setRenderAsOverlay(true);
			vs->setDefaultColor( TColor3(0.0f,0.0f,0.0f) );
		}
		Layer->m_VoxelSets.push_back(vs);
		Layer->setName( "Boundary" );
		if(p_AddLayer) addLayer(Layer);
	}
	*/



	// Compare fmm and exact
	{
		shared_ptr<TFmm> fmm( new TFmm(Image, 0, 4) );
		fmm->perform();
		if(p_AddLayer) MyApp::instance()->addLayer( shared_ptr<TLayer>( new TLayer(fmm->getDistance(), fmm->getShortName()) ));
		shared_ptr<TExactDt> exactdt( new TExactDt(Image, 0) );
		exactdt->perform();

		// Copy/paste of beneath
		{
			shared_ptr<TDt> dt1 = fmm;
			shared_ptr<TDt> dt2 = exactdt;				

			shared_ptr<TDtComparison> comp(new TDtComparison(dt1, dt2, p_AddLayer));
			p_Comparisons.push_back(comp);

			if(p_AddLayer)
			{
				shared_ptr<TLayer> DistLayer( new TLayer(comp->m_DistanceDiff, dt1->getShortName() + " - " + dt2->getShortName() + " dist") );
				//DistLayer->m_DtComparison = comp;
				MyApp::instance()->addLayer( DistLayer );

				shared_ptr<TLayer> OriginLayer( new TLayer(comp->m_OriginsDiff, dt1->getShortName() + " - " + dt2->getShortName() + " origs") );
				//OriginLayer->m_DtComparison = comp;
				MyApp::instance()->addLayer( OriginLayer );
			}
		}
	}

	// Now compare different methods, different tolerances

	
	for(int j=0; j<p_Tolerances.size(); j++)
	{
		float Tolerance = p_Tolerances[j];

		// Normale DT selectie:
		vector<shared_ptr<TDt> > dts;

#define FTFTCLASS TCdt3
#define EDTCLASS TEdt3

		
		// Normale DT selectie:
		shared_ptr<FTFTCLASS> ftft_4( new FTFTCLASS(Image, Tolerance, 4) );
		if(p_DoAll) dts.push_back( ftft_4 );
		shared_ptr<TMullikinDt> mull_4( new TMullikinDt(Image, Tolerance, 4) );
		if(p_DoAll) dts.push_back( mull_4 );
		shared_ptr<TLotufo> lotufo_4( new TLotufo(Image, Tolerance, 4) );
		if(p_DoAll) dts.push_back( lotufo_4 );
		shared_ptr<EDTCLASS> etft_4( new EDTCLASS(Image, Tolerance, 4) );
		dts.push_back( etft_4 );

		shared_ptr<FTFTCLASS> ftft_8( new FTFTCLASS(Image, Tolerance, 8) );
		dts.push_back( ftft_8 );
		shared_ptr<TMullikinDt> mull_8( new TMullikinDt(Image, Tolerance, 8) );
		dts.push_back( mull_8 );
		shared_ptr<TLotufo> lotufo_8( new TLotufo(Image, Tolerance, 8) );
		dts.push_back( lotufo_8 );
		shared_ptr<EDTCLASS> etft_8( new EDTCLASS(Image, Tolerance, 8) );
		dts.push_back( etft_8 );


		// Perform exactdt beforehand
		shared_ptr<TExactDt> exactdt( new TExactDt(Image, Tolerance) );
		exactdt->perform();
		if(p_AddLayer) MyApp::instance()->addLayer( shared_ptr<TLayer>( new TLayer(exactdt->getDistance(), exactdt->getShortName()
			//, exactdt
			) ));
		if(p_AddLayer) MyApp::instance()->addLayer( shared_ptr<TLayer>( new TLayer(exactdt->getOrigins(), exactdt->getShortName()
			//, exactdt
			) ));

		{
			for(int i=0; i<dts.size(); i++) 
			{
				//*g_Log << dts[i]->getShortName() << "\n";

				{
					dts[i]->perform();
				}

				// Todo / debug
				//*g_Log << wxString::Format("%f\n",dts[i]->getTime());

				if(p_AddLayer) MyApp::instance()->addLayer( shared_ptr<TLayer>( new TLayer(dts[i]->getDistance(), dts[i]->getShortName()
					// , dts[i]) 
					)));
				if(p_AddLayer) MyApp::instance()->addLayer( shared_ptr<TLayer>( new TLayer(dts[i]->getOrigins(), dts[i]->getShortName()
					// ,dts[i]
					)));

				// Do comparison with exactdt
				{
					shared_ptr<TDt> dt1 = dts[i];
					shared_ptr<TDt> dt2 = exactdt;				

					shared_ptr<TDtComparison> comp(new TDtComparison(dt1, dt2, p_AddLayer));
					p_Comparisons.push_back(comp);
					
					if(p_AddLayer) 
					{
						shared_ptr<TLayer> DistLayer( new TLayer(comp->m_DistanceDiff, dt1->getShortName() + " - " + dt2->getShortName() + " dist") );
						//DistLayer->m_DtComparison = comp;
						MyApp::instance()->addLayer( DistLayer );
					
						shared_ptr<TLayer> OriginLayer( new TLayer(comp->m_OriginsDiff, dt1->getShortName() + " - " + dt2->getShortName() + " origs") );
						//OriginLayer->m_DtComparison = comp;
						MyApp::instance()->addLayer( OriginLayer );
					}
				}

				if(!p_AddLayer)
				{
//					*g_Log << "deleteFields() disabled because it has a bad influence on running times, probably because of memory fragmentation \n";
//					dts[i]->deleteFields();
				}

				MyApp::instance()->Yield();
			}


		}



	} // end for
}



void TDtProcessor::processComparisons(vector<shared_ptr<TDtComparison> > & Comparisons, vector<float> & Tolerances)
{
	if(1) // 1 for latex code, 0 for comma separated excel
	{
		// Print ALL distance and origin information in Latex code
		*g_Log << "\n\n\n% Comparison for file \n";
		
		*g_Log << "\\begin{table*}\n";
		*g_Log << "\\centering\n";
		*g_Log << "\\begin{small}\n";

		*g_Log << "\\begin{tabular}{r|rrr|rr|r} \n";
		*g_Log << "& \\multicolumn{3}{c}{distance $D^m$}  & \\multicolumn{2}{c}{origins $S^m$}  \\\\ \n";
		//*g_Log << "method $m$ & $\\#e$ & $\\%e$ & $\\max{e}$ & $\\avg{e}$ &          $\\# e$ & $\\% e$ & $\\max\\ e_r$  &  $\\avg\\ e_r$  &  $\\max\\ e_a$  &  $\\avg\\ e_a$ &  time (s) \\\\ \n"; 
		*g_Log << "method $m$ & $\\#e$ & $\\max{e}$ & $\\avg{e}$ &          $\\# e$ & $\\avg{e_r}$  &   time (s) \\\\ \n"; 

		*g_Log << "\\hline\n";

		// Print
		{
			vector< pair<int,int> > Methods;
			Methods.push_back( pair<int,int>(4,TDt::DT_FMM) );
			Methods.push_back( pair<int,int>(4,TDt::DT_FCDT) );
			Methods.push_back( pair<int,int>(4,TDt::DT_MCDT) );
			Methods.push_back( pair<int,int>(4,TDt::DT_GCDT) );
			Methods.push_back( pair<int,int>(4,TDt::DT_ECDT) );
			Methods.push_back( pair<int,int>(8,TDt::DT_FCDT) );
			Methods.push_back( pair<int,int>(8,TDt::DT_MCDT) );
			Methods.push_back( pair<int,int>(8,TDt::DT_GCDT) );
			Methods.push_back( pair<int,int>(8,TDt::DT_ECDT) );
			
			for(int k=0; k<Tolerances.size(); k++)
			{
				float Tolerance = Tolerances[k];
				for(int i=0; i<Methods.size(); i++)
				{
					pair<int,int> Method = Methods[i];
					int DistErrorCount = 0;
					float DistErrorMax = 0;
					float DistErrorAvg = 0;
					float Time = 0;
					int OrigErrorCount = 0;
					float RelOrigMaxError = 0;
					float RelOrigAvgError = 0;
					int Count = 0;
					string LatexName = "";

					for(int j=0; j<Comparisons.size(); j++)
					{
						shared_ptr<TDtComparison> comp = Comparisons[j];
						if(comp->m_NeighborCount == Method.first &&
							comp->m_DtType == Method.second  &&
							fabs(comp->m_Tolerance-Tolerance) < 0.001f)
						{
							LatexName = comp->m_LatexName;
							DistErrorCount += comp->m_DistErrorCount;
							DistErrorMax += comp->m_DistMaxError;;
							DistErrorAvg += comp->m_DistAvgError;

							OrigErrorCount += comp->m_OrigErrorCount;
							RelOrigMaxError += comp->m_RelOrigMaxError;
							RelOrigAvgError += comp->m_RelOrigAvgError;

							Time += comp->m_Time;
							Count++;
						}
					}

					// DistErrorCount /= (float)Count;
					DistErrorMax /= (float)Count;
					DistErrorAvg /= (float)Count;

					// OrigErrorCount /= (float)Count;
					RelOrigAvgError /= (float)Count;
					//Time /= (float)Count;


					string sDistErrorCount = DistErrorCount == 0 ? "0" : 
											DistErrorCount < 1 ? "$<$1" : 
											wxString::Format("%i", DistErrorCount).c_str();
					string sDistErrorMax =	wxString::Format("%.2f", DistErrorMax).c_str();
					string sDistErrorAvg =	wxString::Format("%.2f", DistErrorAvg).c_str();
				


					string sOrigErrorCount = OrigErrorCount == 0 ? "0" : 
											OrigErrorCount < 1 ? "$<$1" : 
											wxString::Format("%i", OrigErrorCount).c_str();
					string sRelOrigMaxError = wxString::Format("%.3f\\%%", 100*RelOrigMaxError).c_str();  
					string sRelOrigAvgError = wxString::Format("%.3f\\%%", 100*RelOrigAvgError).c_str();  
							
					if(Time > 0.00001f)				
					{
						*g_Log << wxString::Format(
							"%s &    %s & %s & %s &       %s & %s &      %.2f \\\\ \n"
							,LatexName.c_str()

							,sDistErrorCount.c_str()
							,sDistErrorMax.c_str()
							,sDistErrorAvg.c_str()

							,sOrigErrorCount.c_str()
							,sRelOrigAvgError.c_str()
							
							,Time
							);
					}

					if( Method.second == TDt::DT_ECDT )
					{
						*g_Log << "\\hline" << endl;
					}
				}
			}
		}

		*g_Log << "\\end{tabular}\n";
		*g_Log << "\\caption{test}\n";
		*g_Log << "\\end{small}\n";
		*g_Log << "\\end{table*}\n";
	}
	else
	{
		*g_Log << "=== Excel\n";

		vector< pair<int,int> > Methods;
		Methods.push_back( pair<int,int>(4,TDt::DT_FCDT) );
		Methods.push_back( pair<int,int>(4,TDt::DT_MCDT) );
		Methods.push_back( pair<int,int>(4,TDt::DT_GCDT) );
		Methods.push_back( pair<int,int>(4,TDt::DT_ECDT) );
		
		Methods.push_back( pair<int,int>(8,TDt::DT_FCDT) );
		Methods.push_back( pair<int,int>(8,TDt::DT_MCDT) );
		Methods.push_back( pair<int,int>(8,TDt::DT_GCDT) );
		Methods.push_back( pair<int,int>(8,TDt::DT_ECDT) );


		string sTimings = "";
		string sAccuracy = "";
		for(int k=0; k<Tolerances.size(); k++)
		{
			float Tolerance = Tolerances[k];
			
			sTimings += wxString::Format("\n%.3f, ",Tolerance);
			sAccuracy += wxString::Format("\n%.3f, ",Tolerance);
			
			for(int i=0; i<Methods.size(); i++)
			{
				pair<int,int> Method = Methods[i];
				int DistErrorCount = 0;
				float DistErrorMax = 0;
				float DistErrorAvg = 0;
				float Time = 0;
				int OrigErrorCount = 0;
				float RelOrigMaxError = 0;
				float RelOrigAvgError = 0;
				int Count = 0;
				string LatexName = "";

				for(int j=0; j<Comparisons.size(); j++)
				{
					shared_ptr<TDtComparison> comp = Comparisons[j];
					if(comp->m_NeighborCount == Method.first &&
						comp->m_DtType == Method.second  &&
						fabs(comp->m_Tolerance-Tolerance) < 0.001f)
					{
						LatexName = comp->m_LatexName;
						DistErrorCount += comp->m_DistErrorCount;
						DistErrorMax += comp->m_DistMaxError;;
						DistErrorAvg += comp->m_DistAvgError;

						OrigErrorCount += comp->m_OrigErrorCount;
						RelOrigMaxError += comp->m_RelOrigMaxError;
						RelOrigAvgError += comp->m_RelOrigAvgError;

						Time += comp->m_Time;
						Count++;
					}
				}

				// DistErrorCount /= (float)Count;
				DistErrorMax /= (float)Count;
				DistErrorAvg /= (float)Count;

				// OrigErrorCount /= (float)Count;
				RelOrigAvgError /= (float)Count;
				//Time /= (float)Count;


				if(Time > 0.00001f)				
				{
					sTimings += wxString::Format("%.3f, ", Time);
					sAccuracy += wxString::Format("%.3f, ", RelOrigAvgError);
				}
			}
		}

		*g_Log << "Timings:\n\n" << sTimings << "\n\n";
		*g_Log << "Accuracy: \n\n" << sAccuracy << "\n\n";
	}
}




