#include "stdafx.h"

#include "field.h"

#include "abstractfilter.h"
#include "layer.h"
#include "parameters.h"
#include "filter.h"
#include "shortestpath.h"
#include "globals.h"
#include "logwriter.h"
#include "utils.h"
#include "settings.h"
#include "IM.h"

using namespace std;

const unsigned int TIndexMapper::OUTSIDE = (numeric_limits<unsigned int>::max)();
const unsigned int TIndexMapper::BOUNDARY = (numeric_limits<unsigned int>::max)() - 1;


shared_ptr<TMeasure> TFloatField3::getDefaultMeasure() { return shared_ptr<TMeasure>(new TFloatFieldMeasure_Identity(this)); }
shared_ptr<TMeasure> TSparseFloatField3::getDefaultMeasure() { return shared_ptr<TMeasure>(new TFloatFieldMeasure_Identity(this)); }
shared_ptr<TMeasure> TVirtualSparseFloatField_Maximize::getDefaultMeasure() { return shared_ptr<TMeasure>(new TFloatFieldMeasure_Identity(this)); }
shared_ptr<TMeasure> TUintegerField3::getDefaultMeasure() { return shared_ptr<TMeasure>(new TUintFieldMeasure_Identity(this)); }
shared_ptr<TMeasure> TSubIndexMapper::getDefaultMeasure() { return shared_ptr<TMeasure>(new TUintFieldMeasure_Identity(this)); }
shared_ptr<TMeasure> TSparseUintField3::getDefaultMeasure() { return shared_ptr<TMeasure>(new TUintFieldMeasure_Identity(this)); }
shared_ptr<TMeasure> TUnsignedCharField3::getDefaultMeasure() { return shared_ptr<TMeasure>(new TUnsignedCharFieldMeasure_Identity(this)); }
shared_ptr<TMeasure> TSparseUcharField3::getDefaultMeasure() { return shared_ptr<TMeasure>(new TUnsignedCharFieldMeasure_Identity(this)); }
shared_ptr<TMeasure> TCoord3SetField::getDefaultMeasure() { return shared_ptr<TMeasure>(new TCoord3SetField_CoordCount(this)); }
shared_ptr<TMeasure> TSparseCoord3SetField::getDefaultMeasure() { return shared_ptr<TMeasure>(new TCoord3SetField_CoordCount(this)); }
shared_ptr<TMeasure> TSparseCoord3WithFloatSetField::getDefaultMeasure() { return shared_ptr<TMeasure>(new TCoord3WithFloatSetField_Minimum(this)); }

shared_ptr<TMeasure> TVector3Field3::getDefaultMeasure() { return shared_ptr<TMeasure>(new TVector3FieldMeasure_Length(this)); }
shared_ptr<TMeasure> TSparseVector3Field::getDefaultMeasure() { return shared_ptr<TMeasure>(new TVector3FieldMeasure_Length(this)); }




TFlagField3 * TFlagField3::construct(const TUnsignedCharField3 * p_Image, TFloatField3 * p_Distance, float p_Value)
{
	const bool Is3D = p_Image->getMaxZ() > 1;

	TFlagField3 * FlagField = new TFlagField3(p_Image->dimX(), p_Image->dimY(), p_Image->dimZ());

	FlagField->m_NarrowBandCount = 0;
	FlagField->m_ObjectPixelCount = 0;

	{
		for(int i=0; i<p_Image->getMaxX(); i++)
		{
			for(int j=0; j<p_Image->getMaxY(); j++)
			{
				for(int k=0; k<p_Image->getMaxZ(); k++)
				{
					const float v = p_Image->value(i,j,k);
					if( fabs( v - p_Value ) > 0.0001 )
					{ 
						if(p_Distance) p_Distance->value(i,j,k) = -1; 
						FlagField->value(i,j,k) = TFlagField3::OUTSIDE;
					}
					else
					{
						if(p_Distance) p_Distance->value(i,j,k) = (numeric_limits<float>::max)(); 
						FlagField->value(i,j,k) = TFlagField3::INSIDE;
						FlagField->m_ObjectPixelCount++;
					} 
				}
			}
		}
	}

	{
		for(int i=0; i<p_Image->getMaxX(); i++)
		{
			for(int j=0; j<p_Image->getMaxY(); j++)
			{
				for(int k=0; k<p_Image->getMaxZ(); k++)
				{
					if (FlagField->inside(i,j,k))
					{
						if (FlagField->outside(i-1,j,k) 
							|| FlagField->outside(i+1,j,k) 
							|| FlagField->outside(i,j-1,k) 
							|| FlagField->outside(i,j+1,k)
							
							|| (Is3D && FlagField->inside(i,j,k-1)) 
							|| (Is3D && FlagField->inside(i,j,k+1))
							)
						{ 
							if(p_Distance) p_Distance->value(i,j,k) = 0;
							FlagField->value(i,j,k) = TFlagField3::NARROW_BAND;
							FlagField->m_NarrowBandCount++; 
							FlagField->m_ObjectPixelCount--;
						}
					}
				}
			}
		}
	}
	return FlagField;
}




TFlagField3 * TFlagField3::construct(const TFloatField3 * p_Image, TFloatField3 * p_Distance, float p_Value)
{
	const bool Is3D = p_Image->getMaxZ() > 1;

	TFlagField3 * FlagField = new TFlagField3(p_Image->dimX(), p_Image->dimY(), p_Image->dimZ());

	FlagField->m_NarrowBandCount = 0;

	{
		for(int i=0; i<p_Image->getMaxX(); i++)
		{
			for(int j=0; j<p_Image->getMaxY(); j++)
			{
				for(int k=0; k<p_Image->getMaxZ(); k++)
				{
					const float v = p_Image->value(i,j,k);
					if( fabs( v - p_Value ) > 0.0001 )
					{ 
						if(p_Distance) p_Distance->value(i,j,k) = -1; 
						FlagField->value(i,j,k) = TFlagField3::OUTSIDE;
					}
					else
					{
						if(p_Distance) p_Distance->value(i,j,k) = (numeric_limits<float>::max)(); 
						FlagField->value(i,j,k) = TFlagField3::INSIDE;
					}
				}
			}
		}
	}

	{
		for(int i=0; i<p_Image->getMaxX(); i++)
		{
			for(int j=0; j<p_Image->getMaxY(); j++)
			{
				for(int k=0; k<p_Image->getMaxZ(); k++)
				{
					if (FlagField->unknown(i,j,k))
					{
						for(int x=i-1; x<=i+1; x++)
							for(int y=j-1; y<=j+1; y++)
								for(int z=k-1; z<=k+1; z++)
								{
									if (x==i && y==j && z==k) continue;
									if (Is3D || z != 0) continue;

									if ( FlagField->known(x,y,z)  ) 
									{ 
										if(p_Distance) p_Distance->value(i,j,k) = 0;
										FlagField->value(i,j,k) = TFlagField3::NARROW_BAND;
										FlagField->m_NarrowBandCount++; 
									}
								}
					}
				}
			}
		}
	}
	return FlagField;
}


void TFlagField3::constructNarrowBand()
{
	for(int i=0; i<getMaxX(); i++)
	{
		for(int j=0; j<getMaxY(); j++)
		{
			for(int k=0; k<getMaxZ(); k++)
			{
				if (unknown(i,j,k))
				{
					for(int x=i-1; x<=i+1; x++)
						for(int y=j-1; y<=j+1; y++)
							for(int z=k-1; z<=k+1; z++)
							{
								if (x==i && y==j && z==k) continue;

								if ( known(x,y,z)  ) 
								{ 
									value(i,j,k) = TFlagField3::NARROW_BAND;
									m_NarrowBandCount++; 
								}
							}
				}
			}
		}
	}
}


unsigned long getLong(fstream& inf)
{
   unsigned long ip; char ic;
   unsigned char uc;
   inf.get(ic); uc = ic; ip = uc;
   inf.get(ic); uc = ic; ip |= ((unsigned long)uc <<8);
   inf.get(ic); uc = ic; ip |= ((unsigned long)uc <<16);
   inf.get(ic); uc = ic; ip |= ((unsigned long)uc <<24);
   return ip;
}

unsigned short getShort(fstream& inf)
{
   char ic; unsigned short ip;
   inf.get(ic); ip = ic;
   inf.get(ic); ip |= ((unsigned short)ic << 8);
   return ip;
}

TField * TFieldFactory::constructFromFile(const string & p_File)
{
	string Filename = p_File;

	// Decompress if necessary
	{
		if (Filename.substr(Filename.length() - 3,3) == ".gz")
		{
			string newName = Filename.substr(0, Filename.length() - 3);
			decompressZlib(Filename, newName);
			Filename = newName;
		}
	}
	
	std::ifstream File(Filename);

	// Check if file exists
	if (!File) throw string("File '"+Filename+"' does not seem to exist ");


	TField * Field = 0;

	if( Filename.substr(Filename.length() - 6,6) == ".field" )
	{
//		std::ifstream File(Filename.c_str(), ios::binary);
//		TField * f =  TShortestPathSetField::readFromStream(&File);
//		File.close();
//		Field = f;
	}
	else if( Filename.substr(Filename.length() - 4,4) == ".bmp" )
	{
		fstream inf;
		inf.open(Filename.c_str(), ios::in|ios::binary);
		if (!inf) return 0;
		char ch1,ch2;
		inf.get(ch1); inf.get(ch2);                             //read BMP header
		unsigned long  fileSize = getLong(inf);      
		unsigned short res1     = getShort(inf);
		unsigned short res2     = getShort(inf);
		unsigned long  offBits  = getLong(inf);
		unsigned long  hdrSize  = getLong(inf);
		unsigned long  numCols  = getLong(inf);     
		unsigned long  numRows  = getLong(inf);     
		unsigned short planes   = getShort(inf);
		unsigned short bitsPix  = getShort(inf);                //8 or 24 bits per pixel
		unsigned long  compr    = getLong(inf); 
		unsigned long  imgSize  = getLong(inf);
		unsigned long  xPels    = getLong(inf);
		unsigned long  yPels    = getLong(inf);
		unsigned long  lut      = getLong(inf);
		unsigned long  impCols  = getLong(inf);
		int bpp = bitsPix/8;                                    //1 or 3 bytes per pixel                       
		unsigned int nBytesInRow  = ((bpp*numCols+3)/4)*4;
		unsigned int numPadBytes  = nBytesInRow - bpp*numCols; 
		//FIELD<float>* f = new FIELD<float>(numCols,numRows);
		TFloatField3 * f = new TFloatField3(numCols,numRows,1);
		float * data = f->data();
		assert(data != 0);
		//unsigned char ch;
		char ch;
		for(unsigned int row=0;row<numRows;row++)               //for every row     
		{
		  for(unsigned int col=0;col<numCols;col++)
		  {
			 if (bpp==3)                                       //read data as RGB 'luminance'
			 {  
				//unsigned char r,g,b; 
				char r,g,b;
				inf.get(b); 
				inf.get(g); 
				inf.get(r);
				*data++ = (int(r)+int(g)+int(b))/3;
			 }
			 else                                              //read data as 8-bit luminance
			 {  
				 inf.get(ch); 
				 *data++ = ch; 
			 }
		  }
		  for(unsigned int k=0;k<numPadBytes;k++) inf>>ch;     //skip pad bytes at end of row
		}
		inf.close();
		Field = f;
	}
	else if( Filename.substr(Filename.length() - 4,4) == ".pgm" )
	{
		FILE* fp = fopen(Filename.c_str(),"r"); if (!fp) return 0;

		const int SIZE = 1024;
		char buf[SIZE]; int dimX,dimY,range;
		fscanf(fp,"%*s");				//skip "P5" header

		for(;;)
		{
		 fscanf(fp,"%s",buf);			//get dimX or #comment
		 if (buf[0]=='#') fgets(buf,SIZE,fp); 
			else { dimX = atoi(buf); break; }
		}
		for(;;)
		{
		 fscanf(fp,"%s",buf);			//get dimY or #comment
		 if (buf[0]=='#') fgets(buf,SIZE,fp); 
			else { dimY = atoi(buf); break; }
		}
		for(;;)
		{
		 fscanf(fp,"%s",buf);			//get range or #comment
		 if (buf[0]=='#') fgets(buf,SIZE,fp); 
			else { range = atoi(buf); break; }
		}


		TFloatField3 * f = new TFloatField3(dimX,dimY,1);
		int bb = SIZE; fgets(buf,SIZE,fp);

		for(float *d = f->data(),*end=d+dimX*dimY;d<end;d++)		//read the binary data into the field
		{								//be careful: buf is a char, we first need
		if (bb==SIZE) { fread(buf,SIZE,1,fp); bb=0; }		//to convert the read bytes to unsigned char and then assign
		*d = (unsigned char)buf[bb++];				//to the field!
		}

		fclose(fp);  				
		Field = f;
	}
	else if( Filename.substr(Filename.length() - 4,4) == ".binvox" )
	{
		//ifstream is(Filename);
		//string << "header
		//is.close();

	}
	else if( Filename.substr(Filename.length() - 4,4) == ".txt" )
	{
		int x,y,z;
		int maxx = 0;
		int maxy = 0;
		int maxz = 0;
		string s;

		// *g_Log << "Reading file ";
		{
			std::ifstream file(Filename.c_str());
			while(file >> s)
			{
				sscanf(s.c_str(), "0,%i,%i,%i,1,undefined",&x,&y,&z);
				maxx = max(x,maxx);
				maxy = max(y,maxy);
				maxz = max(z,maxz);
			}
			file.close();
		}
		// *g_Log << "(" << maxx << "," << maxy << "," << maxz << ") ";
		{
			std::ifstream file(Filename.c_str());
			TFloatField3 * f = new TFloatField3(maxx,maxy,maxz);
			f->setEveryValue(1.0f);
			while(file >> s)
			{
				sscanf(s.c_str(), "0,%i,%i,%i,1,undefined",&x,&y,&z);
				f->value(x,y,z) = 0.0f;
			}
			Field = f;
			file.close();
		}
		// *g_Log << "done\n";
	}

	else if( Filename.substr(Filename.length() - 4,4) == ".vol" )
	{
		FILE* fvol;
		size_t read;

		int M=0;
		int L=0;
		int N=0;
		wxString file( Filename.c_str() );
		int start = file.Find(".")+1;
		int end = file.Find(".vol");
		wxString tmp = file.Mid(start,end-start);
		if(! sscanf(tmp.c_str(), "%dx%dx%d",&M,&L,&N) )
		{
			throw string("Cannot determine dimensions.\n");
		}
		else if ((fvol = fopen(file.c_str(),"rb")) == NULL) 
		{
			throw string("Error while opening file.\n");
		}
		else
		{
			TUnsignedCharField3 * newf = new TUnsignedCharField3(M, L, N);
			read = fread(newf->data(), sizeof(unsigned char), L*M*N, fvol );

			if ( read < ((size_t) L*M*N)) 
			{
				throw string("Read error.\n");
				delete newf;
			}
			else
			{
				Field = newf;
			}

			for(unsigned int x=0; x<Field->getMaxX(); x++)
				for(unsigned int y=0; y<Field->getMaxY(); y++)
					for(unsigned int z=0; z<Field->getMaxZ(); z++)
					{
						if( newf->value(x,y,z) > 1 ) newf->value(x,y,z) = 1;
					}
		}

		fclose(fvol);
/*
		shared_ptr<
	//bool ReadVolume(char *filename, int L, int M, int N, unsigned char **vol) {
		FILE* fvol;
		size_t read;

		if ((fvol = fopen(filename,"rb")) == NULL) {
			printf("\nCannot open input file %s\n", filename);
		(*vol) = NULL;
			return false;
		}

		if(((*vol) = new unsigned char[L*M*N]) == NULL) {
			printf("\nError allocating memory for the volume. Not enough memory ?\n");
			return false;
		}

		read = fread((*vol), sizeof(unsigned char), L*M*N, fvol);
		if ( read < ((size_t) L*M*N)) {
		printf("\n\
		Read only %ld values before the end of input file. Expected: %ld.\n", 
		read, L*M*N);
		delete [] (*vol);
		(*vol) = NULL;
		return false;
		}

		fclose(fvol);
		return true;
		*/
	}
	else if( Filename.substr(Filename.length() - 3,3) == ".im" )
	{
		int PixType, Xdim, Ydim, Zdim, DimCnt;

		//char Name1[50];
		IM_TYPE *Image1;
		FLOAT_TYPE ***Data1;
		Image1 = im_open((char*) Filename.c_str(), &PixType, &Xdim, &Ydim, &Zdim, &DimCnt);
		if (DimCnt != 3) throw string("Can not process 1D or 2D images");
		

		Data1 = (FLOAT_TYPE ***)im_alloc3D(Image1, FLOAT);

		TUnsignedCharField3 * newf = new TUnsignedCharField3(Xdim, Ydim, Zdim);		
		im_read(Image1, BYTE, (char*) newf->data() );
		TUnsignedCharField3* cropped = crop(newf);
		Field = cropped;
		delete newf;

		{
			TUnsignedCharField3 * doublef = new TUnsignedCharField3(cropped->getMaxX()*2, cropped->getMaxY()*2, cropped->getMaxZ()*2);		
			// Test: double each voxel
			unsigned int x,y,z;
			for(x=0; x<cropped->getMaxX(); x++)
			for(y=0; y<cropped->getMaxY(); y++)
			for(z=0; z<cropped->getMaxZ(); z++)
			{
				doublef->value(x*2+0,y*2+0,z*2+0) = cropped->value(x,y,z);
				doublef->value(x*2+0,y*2+0,z*2+1) = cropped->value(x,y,z);
				doublef->value(x*2+0,y*2+1,z*2+0) = cropped->value(x,y,z);
				doublef->value(x*2+0,y*2+1,z*2+1) = cropped->value(x,y,z);
				doublef->value(x*2+1,y*2+0,z*2+0) = cropped->value(x,y,z);
				doublef->value(x*2+1,y*2+0,z*2+1) = cropped->value(x,y,z);
				doublef->value(x*2+1,y*2+1,z*2+0) = cropped->value(x,y,z);
				doublef->value(x*2+1,y*2+1,z*2+1) = cropped->value(x,y,z);
			}
			Field = doublef;
			delete cropped;
		}

		//*g_Log << "NOTICE: setting dilation distance to 3.0 because of .im files are typically smaller than normal\n";
		//g_Parameters.m_DilationDistance = 3.5f;
		//im_read(Image1, FLOAT, (char*) &(Data1[0][0][0]));
	}
	else if( Filename.substr(Filename.length() - 4,4) == ".vtk" )
	{
		// Read first string of file into s
		string s;
		File >> s;
		if(s == "ORIGSETFIELD_ASCII") 
		{
			throw string("Unsupported");
		}
//		else if(s == "ORIGSETFIELD_BINARY")
//		{
//			Field = TOrigSetField2::constructFromBinaryFile(Filename);
//		}
		else
		{
			// Assume vtk file: loop until SCALARS token is reached
			bool stop = false;
			unsigned int n = 0;
			while(!stop && n < 100)
			{
				File >> s;
				if (s == string("SCALARS"))
				{
					stop = true;
				}
				n++;
			}

			File >> s;
			File >> s;

			if(s == string("float")) 
			{
				typedef float T;
				FIELD3<T> * f = FIELD3<T>::read(Filename.c_str());
				TFloatField3 * TempField = new TFloatField3(f->dimX(), f->dimY(), f->dimZ());
				FIELD3<T> * Part = TempField;
				*Part = *f;
				delete f;
				Field = TempField;
			}
			else if(s == string("int")) 
			{
/*
				typedef int T;
				FIELD3<T> * f = FIELD3<T>::read(Filename.c_str());
				TIntegerField3 * TempField = new TIntegerField3(f->dimX(), f->dimY(), f->dimZ());
				FIELD3<T> * Part = TempField;
				*Part = *f;
				delete f;
				Field = TempField;
*/
			}
			else if(s == string("unsigned_char")) 
			{
				typedef unsigned char T;
				FIELD3<T> * f = FIELD3<T>::read(Filename.c_str());
				if(!f) throw string("Error while reading file");

				TUnsignedCharField3* newf = crop(f);

				/*
				// Do crop
				if(! g_Parameters.m_ComputeBackgroundSskel)
				{
					newf = new TUnsignedCharField3(2+d1, 2+d2, 2+d3);
					for(int i=0; i<d1; i++)
					{
						for(int j=0; j<d2; j++)
						{
							for(int k=0; k<d3; k++)
							{
								const T v = f->value(minx+i, miny+j, minz+k);

								newf->value(1+i,1+j,1+k) = v == 0 ? 0 : 1;
							}
						}
					}
				}
				else // Add border because of background S-skel computation
				{
					const unsigned int brd = g_Parameters.m_InvertBorder;
					newf = new TUnsignedCharField3(2+d1+brd*2, 2+d2+brd*2, 2+d3+brd*2);
					for(int i=0; i<d1; i++)
					{
						for(int j=0; j<d2; j++)
						{
							for(int k=0; k<d3; k++)
							{
								const T v = f->value(minx+i, miny+j, minz+k);

								newf->value(1+i+brd,1+j+brd,1+k+brd) = v == 0 ? 0 : 1;
							}
						}
					}
				}
				*/

				//if(		(2+d1 != X)
				//	||  (2+d2 != Y)
				//	||	(2+d3 != Z)
				//)
				//{
					// *g_Log << ", resized from (" << f->dimX() << "," << f->dimY() << "," << f->dimZ() << ") to ";
					//newf->writeToFile(Filename);
				//}
				// *g_Log << "(" << newf->dimX() << "," << newf->dimY() << "," << newf->dimZ() << ") ";

				Field = newf;
				delete f;

			}
			else throw string("Unknown vtk format " + Filename );
		}
	}
	else if (Filename.substr(Filename.length() - 4, 4) == ".scn")
	{
		std::ifstream stream(Filename, ifstream::in | ifstream::binary);

		std::string scn;
		stream >> scn;
		if (scn != "SCN") throw std::string("Invalid SCN file");

		unsigned int width, height, depth;
		stream >> width >> height >> depth;

		double voxelWidth, voxelHeight, voxelDepth;
		stream >> voxelWidth >> voxelHeight >> voxelDepth;

		int bits;
		stream >> bits;
		if (bits != 8) throw std::string("Only 8 bit SCN file supported");

		if (stream.get() != '\n') throw std::string("Expected line feed before binary data");

		std::unique_ptr<TUnsignedCharField3> field(new TUnsignedCharField3(width, height, depth));
		stream.read((char*)field->data(), width * height * depth);

		Field = crop(field.get());
	}
	else throw string("Unknown filetype for " + Filename);

	File.close();

	if(!Field) throw string("Couldn't load field");

	return Field;
}




int TVirtualSparseFloatField_Maximize::getMaxX() const { return (*m_Measure1)->m_AbstractField->getMaxX(); }	
int TVirtualSparseFloatField_Maximize::getMaxY() const { return (*m_Measure1)->m_AbstractField->getMaxY(); }	
int TVirtualSparseFloatField_Maximize::getMaxZ() const { return (*m_Measure1)->m_AbstractField->getMaxZ(); }	

float & TVirtualSparseFloatField_Maximize::wvaluep(const TCoord3 & p)
{
	assert("TVirtualSparseFloatField_Maximize::wvaluep not supported" && false);
	throw string("TVirtualSparseFloatField_Maximize::vvaluep not supported");
	static float v = 0;
	return v;
}

const float TVirtualSparseFloatField_Maximize::vvaluep(const TCoord3 & p) const
{
	const float v1 = (*m_Measure1)->toFloat(p);
	const float v2 = (*m_Measure2)->toFloat(p);
	return v1 > v2 ? v1 : v2;
}

float & TVirtualSparseFloatField_Maximize::wvaluex(unsigned int idx)
{
	assert("TVirtualSparseFloatField_Maximize::wvaluep not supported" && false);
	throw string("TVirtualSparseFloatField_Maximize::vvaluep not supported");
	static float v = 0;
	return v;
}

const float TVirtualSparseFloatField_Maximize::vvaluex(unsigned int idx) const
{
	return vvaluep( m_IndexField->vidx2coord(idx) );
}



const TCoord3 & TSubIndexMapper::vidx2coord(unsigned int idx) const
{ 
	const unsigned int subidx = m_SubIndex[idx];
	const TCoord3 & coord = m_BaseIndexField->m_Voxels[subidx];
	return coord; 
}

unsigned int TSubIndexMapper::vcoord2idx(const TCoord3 & p) const
{ 
	map<TCoord3, unsigned int>::const_iterator it = m_Coord2IdxMap.find(p); if(it != m_Coord2IdxMap.end()) return it->second; else return OUTSIDE; 
}

void TSubIndexMapper::writeToStream(std::ofstream * s) const
{
	//	map<TCoord3, unsigned int> m_Coord2IdxMap;
	{
		unsigned int count = m_Coord2IdxMap.size();
		s->write( (char*) &count, sizeof(unsigned int) );
		map<TCoord3, unsigned int>::const_iterator it;
		for(it = m_Coord2IdxMap.begin(); it != m_Coord2IdxMap.end(); it++)
		{
			TCoord3 c = it->first;
			unsigned int idx = it->second;
			s->write( (char*) &(c.x), sizeof(float)*3 );
			s->write( (char*) &idx, sizeof(unsigned int) );
		}
	}
	
	//	vector<unsigned int> m_SubIndex; 
	{
		unsigned int count = m_SubIndex.size();
		s->write( (char*) &count, sizeof(unsigned int) );
		s->write( (char*) &(m_SubIndex[0]), sizeof(unsigned int)*count );
	}
}

TSubIndexMapper * TSubIndexMapper::readFromStream(std::ifstream * s, const TIndexField3 * p_BaseIndexField)
{
	TSubIndexMapper * IF = new TSubIndexMapper(p_BaseIndexField);

	//	map<TCoord3, unsigned int> m_Coord2IdxMap;
	{
		unsigned int count = 0;
		s->read( (char*) &count, sizeof(unsigned int) );
		for(unsigned int x=0; x<count; x++)
		{
			TCoord3 c;
			unsigned int idx;
			s->read( (char*) &c, sizeof(float)*3 );
			s->read( (char*) &idx, sizeof(unsigned int) );
			IF->m_Coord2IdxMap.insert( pair<TCoord3,unsigned int>(c,idx) );
		}
	}
	
	//	vector<unsigned int> m_SubIndex; 
	{
		unsigned int count = 0;
		s->read( (char*) &count, sizeof(unsigned int) );
		IF->m_SubIndex.resize(count);
		s->read( (char*) &(IF->m_SubIndex[0]), sizeof(unsigned int)*count );
	}
	return IF;
}


void TIndexField3::writeToStream(std::ofstream * s) const
{
	unsigned int maxx = this->getMaxX();
	unsigned int maxy = this->getMaxY();
	unsigned int maxz = this->getMaxZ();
	s->write( (char*) &maxx, sizeof(unsigned int) );
	s->write( (char*) &maxy, sizeof(unsigned int) );
	s->write( (char*) &maxz, sizeof(unsigned int) );
	s->write( (char*) this->data(), maxx*maxy*maxz*sizeof(unsigned int));

	unsigned int count = m_Voxels.size();
	s->write( (char*) &count, sizeof(unsigned int) );
	s->write( (char*) &(m_Voxels[0]), sizeof(TCoord3)*count );
}

TIndexField3 * TIndexField3::readFromStream(std::ifstream * s)
{
	unsigned int maxx = 0;
	unsigned int maxy = 0;
	unsigned int maxz = 0;
	s->read( (char*) &maxx, sizeof(unsigned int) );
	s->read( (char*) &maxy, sizeof(unsigned int) );
	s->read( (char*) &maxz, sizeof(unsigned int) );
	assert(maxx < 1024 && maxy < 1024 && maxz < 1024);
	TIndexField3 * f = new TIndexField3(maxx,maxy,maxz);
	s->read( (char*) f->data(), maxx*maxy*maxz*sizeof(unsigned int));

	unsigned int count = 0;
	s->read( (char*) &count, sizeof(unsigned int) );
	f->m_Voxels.resize(count);
	s->read( (char*) &(f->m_Voxels[0]), sizeof(TCoord3)*count );
	return f;
}

TUnsignedCharField3* crop(const FIELD3<unsigned char>* field)
{
	int minX = field->dimX();
	int minY = field->dimY();
	int minZ = field->dimZ();
	
	int maxX = 0;
	int maxY = 0;
	int maxZ = 0;

	for (int i = 0; i < field->dimX(); i++)
	{
		for (int j = 0; j < field->dimY(); j++)
		{
			for (int k = 0; k < field->dimZ(); k++)
			{
				if (field->value(i, j, k) > 0)
				{
					maxX = max(i, maxX);
					maxY = max(j, maxY);
					maxZ = max(k, maxZ);

					minX = min(i, minX);
					minY = min(j, minY);
					minZ = min(k, minZ);
				}
			}
		}
	}

	const int d1 = maxX - minX + 1;
	const int d2 = maxY - minY + 1;
	const int d3 = maxZ - minZ + 1;

	const unsigned int brd = g_Parameters.m_InvertBorder;

	TUnsignedCharField3 * newField = new TUnsignedCharField3(2 + d1 + brd * 2, 2 + d2 + brd * 2, 2 + d3 + brd * 2);
	for (int i = 0; i < d1; i++)
	{
		for (int j = 0; j < d2; j++)
		{
			for (int k = 0; k < d3; k++)
			{
				const unsigned char v = field->value(minX + i, minY + j, minZ + k);
				newField->value(1 + i + brd, 1 + j + brd, 1 + k + brd) = v == 0 ? 0 : 1;
			}
		}
	}

	return newField;
}

static int round(double d)
{
	return (int)(d + .5);
}

TUnsignedCharField3* scale(const FIELD3<unsigned char>* input, double factor)
{
	assert(input != nullptr);
	assert(factor > 0);

	TUnsignedCharField3* output = new TUnsignedCharField3((int)(input->dimX() * factor),
		(int)(input->dimY() * factor), (int)(input->dimZ() * factor));

	for (int k = 0; k < output->dimZ(); k++)
	{
		for (int j = 0; j < output->dimY(); j++)
		{
			for (int i = 0; i < output->dimX(); i++)
			{
				output->value(i, j, k) = input->value(round(i / factor), round(j / factor), round(k / factor));
			}
		}
	}

	return output;
}
