#pragma once

#include "stdafx.h"

#include "abstractfield.h"
#include "indexmapper.h"
#include "field3.h"
#include "coord.h"
#include "Geometry.h"

#include "types.h"

using namespace std;


template<class T>
class TTypedField : public FIELD3<T>, public TTypedFieldInterface<T>
{
public:
	TTypedField(int x,int y,int z) 
		: FIELD3<T>(x,y,z)
		, TTypedFieldInterface<T>()
		, m_Layer(0)
	{
	};
	virtual ~TTypedField() {};
	virtual class TLayer * getLayer() const { return m_Layer; }
	virtual void setLayer(class TLayer * p_Layer) { m_Layer = p_Layer; }

	virtual shared_ptr<class TMeasure> getDefaultMeasure() { throw string("pure virtual called"); return shared_ptr<class TMeasure>(); };
	virtual void accept(TFieldVisitor * v) { throw string("pure virtual called"); }

	virtual int getMaxX() const { return dimX(); }	
	virtual int getMaxY() const { return dimY(); }	
	virtual int getMaxZ() const { return dimZ(); }	
	T & valuep(const TCoord3 & p) { return value(p.x,p.y,p.z); }
	const T & valuep(const TCoord3 & p) const { return value(p.x,p.y,p.z); }
	virtual T & wvaluep(const TCoord3 & p) { return valuep(p); }
	virtual const T vvaluep(const TCoord3 & p) const { return valuep(p); }

	void setEveryValue(const T & value)
	{
		T * v = data();
		const unsigned int N = dimX()*dimY()*dimZ();
		for(unsigned int i=0; i<N; i++)
		{
			*v = value;
			v++;
		}
	}

	// Read/write
	static TTypedField<T> * readFromStream(std::ifstream * s)
	{
		size_t X,Y,Z;
		*s >> X >> Y >> Z;
		TTypedField<T> * f = new TTypedField<T>(X,Y,Z);
		for(size_t i=0; i<X; i++)
			for(size_t j=0; j<Y; j++)
				for(size_t k=0; k<Z; k++)
				{
					f->readValueFromStream(i,j,k,s);
				}
		return f;
	}
	virtual void writeToStream(std::ofstream * s) const
	{
		*s << getMaxX() << " " << getMaxY() << " " << getMaxZ() << " ";
		for(size_t i=0; i<getMaxX(); i++)
			for(size_t j=0; j<getMaxY(); j++)
				for(size_t k=0; k<getMaxZ(); k++)
				{
					this->writeValueToStream((int)i, (int)j, (int)k, s);
					*s << " ";
				}
		*s << "\n";	
	}
	virtual void writeValueToStream(int i, int j, int k, std::ofstream * s) const { throw string("writeValueToStream not implemented"); }
	virtual void readValueFromStream(int i, int j, int k, std::ifstream * s) { throw string("writeValueToStream not implemented"); }

protected:
	mutable class TLayer * m_Layer;
};

template<class T>
class TBasicTypedField : public TTypedField<T>
{
public:
	TBasicTypedField(int x,int y,int z) 
		: TTypedField<T>(x,y,z) 
	{
		clear();
	};
	virtual void clear() { setEveryValue(0); }
	virtual ~TBasicTypedField() {};

	virtual void writeValueToStream(int i, int j, int k, std::ofstream * s) const { (*s) << value(i,j,k); }
	virtual void readValueFromStream(int i, int j, int k, std::ifstream * s) { T v; *s >> v; value(i,j,k) = v; }
};

template<class T>
class TPointerTypedField : public TTypedField<T>
{
public:
	TPointerTypedField(int x,int y,int z) 
		: TTypedField<T>(x,y,z) 
	{
		setEveryValue(0);
	};

	virtual void clear()
	{
		for(unsigned int i=0; i<dimX(); i++)
			for(unsigned int j=0; j<dimY(); j++)
				for(unsigned int k=0; k<dimZ(); k++)
				{
					if(value(i,j,k))
					{
						delete value(i,j,k);
						value(i,j,k) = 0;
					}
				}
	}

	virtual ~TPointerTypedField() 
	{
		clear();
	};
};











class TFloatField3 : public TBasicTypedField<float>
{
public:
	TFloatField3(int x,int y,int z) : TBasicTypedField<float>(x,y,z) {};
	virtual void writeToFile(const string & p_File)
	{
		this->setVTKCellData(0);
		FIELD3<float>::write(p_File.c_str());
		}
	virtual shared_ptr<class TMeasure> getDefaultMeasure();
	virtual void accept(TFieldVisitor * v) { v->visitFloatField(this); }
	virtual const string getTypeName() { return "float"; }
	virtual TType::TYPE getType() const { return TType::TYPE_FLOAT; }

protected:
	void readFromVTKFile(const string & p_File);
};


class TUintegerField3 : public TBasicTypedField<unsigned int>
{
public:
	TUintegerField3(int x,int y,int z) 
		: TBasicTypedField<unsigned int>(x,y,z) 
	{};
	virtual shared_ptr<class TMeasure> getDefaultMeasure();
	virtual void accept(TFieldVisitor * v) { v->visitUintegerField(this); } 
	virtual const string getTypeName() { return "uint"; }
	virtual TType::TYPE getType() const { return TType::TYPE_UINT; }
};


class TIndexField3 : public TIndexMapper, public TUintegerField3
{
public:
	friend class TSubIndex;
	TIndexField3(int x,int y,int z) 
		: TUintegerField3(x,y,z)
	{
		setEveryValue(OUTSIDE);
	}	

	virtual const TCoord3 & vidx2coord(unsigned int idx) const { return m_Voxels[idx]; }
	virtual unsigned int vcoord2idx(const TCoord3 & p) const { return valuep(p); }
	virtual unsigned int getMaxIndex() const { return (unsigned int)m_Voxels.size(); }
	virtual bool vinside(const TCoord3 & p) const { return vcoord2idx(p)!=OUTSIDE; }
	virtual const TIndexField3 * getBaseIndexField() const { return this; }
	
	static TIndexField3 * readFromStream(std::ifstream * s);
	virtual void writeToStream(std::ofstream * s) const;

	// Data
	vector<TCoord3> m_Voxels; 
protected:
};

class TSubIndexMapper : public TIndexMapper, public TTypedFieldInterface<unsigned int>
{
public:
	TSubIndexMapper(const TIndexField3 * p_BaseIndexField) : m_BaseIndexField(p_BaseIndexField) {}

	virtual class TLayer * getLayer() const { return m_Layer; }
	virtual void setLayer(class TLayer * p_Layer) { m_Layer = p_Layer; }


	// TField
	virtual shared_ptr<class TMeasure> getDefaultMeasure();
	virtual void accept(TFieldVisitor * v) { v->visitUintegerField(this); } 
	virtual const string getTypeName() { return "uint"; }
	virtual TType::TYPE getType() const { return TType::TYPE_UINT; }

	virtual int getMaxX() const { return m_BaseIndexField->getMaxX(); }	
	virtual int getMaxY() const { return m_BaseIndexField->getMaxY();; }	
	virtual int getMaxZ() const { return m_BaseIndexField->getMaxZ(); }	
	virtual unsigned int & wvaluep(const TCoord3 & p) { throw string("Cannot write to '") + getTypeName() + "' field."; }
	virtual const unsigned int vvaluep(const TCoord3 & p) const { return vcoord2idx(p); }

	// TIndexMapper 
	virtual const TCoord3 & vidx2coord(unsigned int idx) const;
	virtual unsigned int vcoord2idx(const TCoord3 & p) const;
	virtual unsigned int getMaxIndex() const { return (unsigned int)m_SubIndex.size(); }
	virtual const TIndexField3 * getBaseIndexField() const { return m_BaseIndexField; }
	virtual void writeToStream(std::ofstream * s) const;
	static TSubIndexMapper * readFromStream(std::ifstream * s, const TIndexField3 * p_BaseIndexField);

	// Data
	map<TCoord3, unsigned int> m_Coord2IdxMap;
	vector<unsigned int> m_SubIndex; 
protected:
	const TIndexField3 * m_BaseIndexField;
	mutable class TLayer * m_Layer;
};




class TUnsignedCharField3 : public TBasicTypedField<unsigned char>
{
public:
	TUnsignedCharField3(int x,int y,int z) : TBasicTypedField<unsigned char>(x,y,z) {};
	virtual shared_ptr<class TMeasure> getDefaultMeasure();
	virtual void accept(TFieldVisitor * v) { v->visitUnsignedCharField(this); }
	virtual const string getTypeName() { return "uchar"; }
	virtual TType::TYPE getType() const { return TType::TYPE_UCHAR; }

	virtual void writeToFile(const string & p_File) 
	{
		this->setVTKCellData(0);
		FIELD3<unsigned char>::write(p_File.c_str());
	}
};


class TFlagField3 : public TUnsignedCharField3
{
public:
	// Type
	enum FLAG_TYPE { OUTSIDE=0, KNOWN=0, NARROW_BAND=1, INSIDE=2, UNKNOWN=2, EXTREMUM=3 };

	// Constructors
	TFlagField3(int x,int y,int z) : TUnsignedCharField3(x,y,z) {};
	static TFlagField3 * construct(const TUnsignedCharField3 * p_Image, TFloatField3 * p_Distance, float p_Value);
	static TFlagField3 * TFlagField3::construct(const TFloatField3 * p_Image, TFloatField3 * p_Distance, float p_Value);
	void constructNarrowBand();

	// Accessors
	bool outside(int i,int j,int k=0) const	{ return value(i,j,k)==OUTSIDE; }
	bool known(int i,int j,int k=0) const	{ return value(i,j,k)==OUTSIDE; }
	bool narrowband(int i,int j,int k=0) const { return value(i,j,k)==NARROW_BAND; }
	bool inside(int i,int j,int k=0) const { return value(i,j,k)==INSIDE;  }
	bool unknown(int i,int j,int k=0) const { return value(i,j,k)==INSIDE;  }

	// Read/write
	unsigned int m_ObjectPixelCount;
	unsigned int m_NarrowBandCount;
};



class TVector3Field3 : public TTypedField<TVector3>  
{
public:
	TVector3Field3(int x,int y,int z) : TTypedField<TVector3>(x,y,z) {};
	virtual shared_ptr<class TMeasure> getDefaultMeasure();
	virtual void accept(TFieldVisitor * v) { v->visitVector3Field(this); }
	virtual TType::TYPE getType() const { return TType::TYPE_VECTOR3; }
};




class TCoord3SetField : public TPointerTypedField<vector<TCoord3>*>
{
public:
	TCoord3SetField(int x,int y,int z) 
		: TPointerTypedField<vector<TCoord3>*>(x,y,z) {};
	virtual shared_ptr<class TMeasure> getDefaultMeasure();
	virtual void accept(TFieldVisitor * v) { v->visitCoord3SetField(this); }
	virtual const string getTypeName() { return "coord3 set"; }
	virtual TType::TYPE getType() const { return TType::TYPE_COORD3SET; }
protected:
};






template<class T>
class TSparseTypedField : public TTypedFieldInterface<T>
{
public:
	TSparseTypedField(shared_ptr<TIndexMapper> p_IndexField) 
		: TTypedFieldInterface<T>()
		, m_IndexField(p_IndexField)
		, m_NonconstDummy(0)
		, m_ConstDummy(0)
		, m_Layer(0)
	{
		m_Values.resize(m_IndexField->getMaxIndex(), 0);
	};
	virtual ~TSparseTypedField() {};
	virtual class TLayer * getLayer() const { return m_Layer; }
	virtual void setLayer(class TLayer * p_Layer) { m_Layer = p_Layer; }

	virtual shared_ptr<class TMeasure> getDefaultMeasure() { throw string("pure virtual called"); return shared_ptr<class TMeasure>(); };
	virtual void accept(TFieldVisitor * v) { throw string("pure virtual called"); }

	virtual int getMaxX() const { return m_IndexField->getBaseIndexField()->getMaxX(); }	
	virtual int getMaxY() const { return m_IndexField->getBaseIndexField()->getMaxY(); }	
	virtual int getMaxZ() const { return m_IndexField->getBaseIndexField()->getMaxZ(); }	
	T & valuep(const TCoord3 & p) 
	{ 
		if(m_IndexField->vinside(p)) return m_Values[ m_IndexField->vcoord2idx(p) ];
		else return m_NonconstDummy;
	}
	const T & valuep(const TCoord3 & p) const 
	{ 
		if(m_IndexField->vinside(p)) return m_Values[ m_IndexField->vcoord2idx(p) ];
		else return m_ConstDummy;
	}
	virtual T & wvaluep(const TCoord3 & p) 
	{ 
		return valuep(p); 
	}
	virtual const T vvaluep(const TCoord3 & p) const 
	{ 
		return valuep(p); 
	}
	virtual unsigned int getMaxIndex() const { return (unsigned int)m_Values.size(); }
	virtual T & wvaluex(unsigned int idx) { return m_Values[idx]; } 
	virtual const T vvaluex(unsigned int idx) const { return m_Values[idx]; } 

	virtual void clear() { setEveryValue(0); }

	void setEveryValue(const T & value)
	{
		for(unsigned int i=0; i<m_Values.size(); i++) m_Values[i] = value;
	}
	virtual bool usingIndexField() const { return true; }
	virtual shared_ptr<TIndexMapper> getIndexField() const { return m_IndexField; }

	virtual void writeValueToStream(unsigned int x, std::ofstream * s) const 
	{ 
		s->write( (char*) &m_Values[x], sizeof(T) );
	}

	virtual void readValueFromStream(unsigned int x, std::ifstream * s) 
	{ 
		s->read( (char*) &m_Values[x], sizeof(T) );
	}

	virtual void writeToStream(std::ofstream * s) const
	{
		unsigned int count = this->getMaxIndex();
		s->write( (char*) &count, sizeof(unsigned int) );
		for(unsigned int x=0; x<count; x++)
		{
			writeValueToStream(x,s);
		}
	}

	static void readFromStream(TSparseTypedField<T> * p_Field, std::ifstream * s)
	{
		unsigned int count = 0;
		s->read( (char*) &count, sizeof(unsigned int) );
		for(unsigned int x=0; x<count; x++)
		{
			p_Field->readValueFromStream(x,s);
		}
	}

	T m_NonconstDummy;
	const T m_ConstDummy;
	vector<T> m_Values;
	shared_ptr<TIndexMapper> m_IndexField;
	mutable class TLayer * m_Layer;
};


template<class T>
class TPointerSparseTypedField : public TSparseTypedField<T>
{
public:
	TPointerSparseTypedField(shared_ptr<TIndexMapper> p_IndexField) 
		: TSparseTypedField<T>(p_IndexField)
	{
		setEveryValue(0);
	};

	virtual void clear()
	{
		for(unsigned int i=0; i<m_Values.size(); i++)
		{
			if(m_Values[i])
			{
				T v = m_Values[i];
				delete v;
				m_Values[i] = 0;
			}
		}
	}

	virtual ~TPointerSparseTypedField() 
	{
		clear();
	};
};



class TSparseFloatField3 : public TSparseTypedField<float>
{
public:
	TSparseFloatField3(shared_ptr<TIndexMapper> p_IndexField) : TSparseTypedField<float>(p_IndexField) {};
	virtual shared_ptr<class TMeasure> getDefaultMeasure();
	virtual void accept(TFieldVisitor * v) { v->visitFloatField(this); }
	virtual const string getTypeName() { return "float"; }
	virtual TType::TYPE getType() const { return TType::TYPE_FLOAT; }
protected:
};

class TSparseUintField3 : public TSparseTypedField<unsigned int>
{
public:
	TSparseUintField3(shared_ptr<TIndexMapper> p_IndexField) : TSparseTypedField<unsigned int>(p_IndexField) {};
	virtual shared_ptr<class TMeasure> getDefaultMeasure();
	virtual void accept(TFieldVisitor * v) { v->visitUintegerField(this); }
	virtual const string getTypeName() { return "uint"; }
	virtual TType::TYPE getType() const { return TType::TYPE_UINT; }
protected:
};

class TSparseUcharField3 : public TSparseTypedField<unsigned char>
{
public:
	TSparseUcharField3(shared_ptr<TIndexMapper> p_IndexField) : TSparseTypedField<unsigned char>(p_IndexField) {};
	virtual shared_ptr<class TMeasure> getDefaultMeasure();
	virtual void accept(TFieldVisitor * v) { v->visitUnsignedCharField(this); }
	virtual const string getTypeName() { return "uchar"; }
	virtual TType::TYPE getType() const { return TType::TYPE_UCHAR; }
protected:
};

class TSparseCoord3SetField : public TPointerSparseTypedField<vector<TCoord3>*>
{
public:
	TSparseCoord3SetField(shared_ptr<TIndexMapper> p_IndexField) 
		: TPointerSparseTypedField<vector<TCoord3>*>(p_IndexField) {};
	virtual shared_ptr<class TMeasure> getDefaultMeasure();
	virtual void accept(TFieldVisitor * v) { v->visitCoord3SetField(this); }
	virtual const string getTypeName() { return "coord3 set"; }
	virtual TType::TYPE getType() const { return TType::TYPE_COORD3SET; }
protected:
};

class TSparseCoord3WithFloatSetField : public TPointerSparseTypedField<vector<pair<TCoord3,float> >*>
{
public:
	TSparseCoord3WithFloatSetField(shared_ptr<TIndexMapper> p_IndexField) 
		: TPointerSparseTypedField<vector<pair<TCoord3,float> >*>(p_IndexField) {};
	virtual shared_ptr<class TMeasure> getDefaultMeasure();
	virtual void accept(TFieldVisitor * v) { v->visitCoord3WithFloatSetField(this); }
	virtual const string getTypeName() { return "coord3 w/ float set"; }
	virtual TType::TYPE getType() const { return TType::TYPE_COORD3WITHFLOATSET; }
protected:
};

class TSparseVector3Field : public TSparseTypedField<TVector3>
{
public:
	TSparseVector3Field(shared_ptr<TIndexMapper> p_IndexField) : TSparseTypedField<TVector3>(p_IndexField) {};
	virtual shared_ptr<class TMeasure> getDefaultMeasure();
	virtual void accept(TFieldVisitor * v) { v->visitVector3Field(this); }
	virtual const string getTypeName() { return "vector3 field"; }
	virtual TType::TYPE getType() const { return TType::TYPE_VECTOR3; }
protected:
};


class TFieldFactory
{
public:
	static TField * constructFromFile(const string & p_File); 
};






class TVirtualSparseFloatField_Maximize : public TTypedFieldInterface<float>
{
public:
	TVirtualSparseFloatField_Maximize(shared_ptr<TIndexMapper> p_IndexField, shared_ptr<TMeasure> * p_Measure1, shared_ptr<TMeasure> * p_Measure2) 
		: m_IndexField(p_IndexField)
		, m_Measure1(p_Measure1)
		, m_Measure2(p_Measure2)
		, m_Layer(0)
	{};
	virtual class TLayer * getLayer() const { return m_Layer; }
	virtual void setLayer(class TLayer * p_Layer) { m_Layer = p_Layer; }

	virtual shared_ptr<class TMeasure> getDefaultMeasure();
	virtual void accept(TFieldVisitor * v) { v->visitFloatField(this); }
	virtual const string getTypeName() { return "virtual, max"; }
	virtual TType::TYPE getType() const { return TType::TYPE_FLOAT; }

	virtual int getMaxX() const;
	virtual int getMaxY() const;
	virtual int getMaxZ() const;

	virtual float & wvaluep(const TCoord3 & p);
	virtual const float vvaluep(const TCoord3 & p) const;
	virtual float & wvaluex(unsigned int idx);
	virtual const float vvaluex(unsigned int idx) const;

	virtual bool usingIndexField() const { return true; }
	virtual shared_ptr<TIndexMapper> getIndexField() const { return m_IndexField; }

protected:
	shared_ptr<TMeasure> * m_Measure1; 
	shared_ptr<TMeasure> * m_Measure2; 
	shared_ptr<TIndexMapper> m_IndexField;
	mutable class TLayer * m_Layer;
};

TUnsignedCharField3* crop(const FIELD3<unsigned char>* field);

TUnsignedCharField3* scale(const FIELD3<unsigned char>* input, double factor);
