#ifndef FIELDADAPTORS_H
#define FIELDADAPTORS_H

#include "stdafx.h"
#include "field.h"
#include "indexedorigins.h"
#include "shortestpath.h"
#include "layer.h"

template<typename T>
class TTypedAdaptor : public TTypedFieldInterface<T>
{
public:
	TTypedAdaptor(const TField * p_AdaptedField)
		: m_AdaptedField(p_AdaptedField)
	{
	}
	virtual TLayer * getLayer() const { return m_Layer; }
	virtual void setLayer(class TLayer * p_Layer) { m_Layer = p_Layer; }

	virtual int getMaxX() const { return m_AdaptedField->getMaxX(); }
	virtual int getMaxY() const { return m_AdaptedField->getMaxY(); }
	virtual int getMaxZ() const { return m_AdaptedField->getMaxZ(); }
	virtual bool usingIndexField() const { return m_AdaptedField->usingIndexField(); }
 	virtual shared_ptr<TIndexMapper> getIndexField() const { return m_AdaptedField->getIndexField(); }

protected:
	class TLayer * m_Layer;
	const TField * m_AdaptedField;
};

static vector<shared_ptr<TField> > g_AllAdaptors;


class TFilteredVoxelsAdaptor : public TTypedAdaptor<unsigned char>
{
public:
	TFilteredVoxelsAdaptor(TField * p_Field)
		: TTypedAdaptor<unsigned char>(p_Field)
	{
	};
	virtual TType::TYPE adapts() const { return TType::TYPE_ANY; }
	virtual bool isVolatile() const { return true; }
	virtual const string getTypeName() { return "0 or 1"; }
	virtual const string getAdaptorName() const { return "filtered voxels"; }
	virtual const unsigned char vvaluep(const TCoord3 & p) const { return m_AdaptedField->getLayer()->m_Filter->test(p); }
	virtual TType::TYPE getType() const { return TType::TYPE_UCHAR; }
	virtual void accept(TFieldVisitor * v) { v->visitUnsignedCharField(this); }
	virtual TField * create(const TField * p_AdaptedField) const { return new TFilteredVoxelsAdaptor(static_cast<TTypedFieldInterface<TShortestPathSet*> *>( const_cast<TField *>(p_AdaptedField) ) ); }
	virtual shared_ptr<TMeasure> getDefaultMeasure() { return shared_ptr<TMeasure>(new TUnsignedCharFieldMeasure_Identity(this)); }
};


template<class T>
class TShortestPathSetFieldAdaptor : public TTypedAdaptor<T>
{
public:
	TShortestPathSetFieldAdaptor(TTypedFieldInterface<TShortestPathSet*> * p_SpSetField)
		: TTypedAdaptor<T>(p_SpSetField)
		, m_SpSetField(p_SpSetField)
	{
	};
	virtual TType::TYPE adapts() const { return TType::TYPE_SHORTESTPATHSET; }
	virtual unsigned int getMaxIndex() const { return m_SpSetField->getIndexField()->getMaxIndex(); }
protected:
	TTypedFieldInterface<TShortestPathSet*> * m_SpSetField;
};


class TShortestPathSetFieldAdaptor_FilteredPathSet : public TShortestPathSetFieldAdaptor<TShortestPathSet*>, public TLayer::TLayerObserver
{
public:
	TShortestPathSetFieldAdaptor_FilteredPathSet(TTypedFieldInterface<TShortestPathSet*> * p_SpSetField);
	virtual ~TShortestPathSetFieldAdaptor_FilteredPathSet()
	{}
	virtual bool isVolatile() const { return true; }
	virtual const string getTypeName() { return "sp set"; }
	virtual const string getAdaptorName() const { return "filtered shortest path"; }
	virtual TShortestPathSet* const vvaluep(const TCoord3 & p) const;
	virtual TShortestPathSet* const vvaluex(unsigned int idx) const;
	virtual TType::TYPE getType() const { return TType::TYPE_SHORTESTPATHSET; }
	virtual void accept(TFieldVisitor * v) { v->visitShortestPathSetField(this); }
	virtual TField * create(const TField * p_AdaptedField) const { return new TShortestPathSetFieldAdaptor_FilteredPathSet(static_cast<TTypedFieldInterface<TShortestPathSet*> *>( const_cast<TField *>(p_AdaptedField) ) ); }
	virtual shared_ptr<TMeasure> getDefaultMeasure() { return shared_ptr<TMeasure>(new TShortestPathSetFieldMeasure_PathCount(this)); }
	virtual void notifyLayerChanged()
	{
		getLayer()->onLayerChanged();
	}
protected:
	mutable TShortestPathSet volatilesps;
	mutable shared_ptr<TIndexedOrigins_Vector> volatileEft;
};








class TShortestPathSetFieldAdaptor_Length : public TShortestPathSetFieldAdaptor<float>
{
public:
	TShortestPathSetFieldAdaptor_Length(TTypedFieldInterface<TShortestPathSet*> * p_SpSetField) 
		: TShortestPathSetFieldAdaptor<float>(p_SpSetField)
	{};
	virtual const string getTypeName() { return "float"; }
	virtual const string getAdaptorName() const { return "maximum length (S-skel measure)"; }
	virtual const float vvaluep(const TCoord3 & p) const { return m_SpSetField->vvaluep(p) ? m_SpSetField->vvaluep(p)->getMaxLength() : 0; }
	virtual const float vvaluex(unsigned int x) const { return m_SpSetField->vvaluex(x) ? m_SpSetField->vvaluex(x)->getMaxLength() : 0; }
	virtual TType::TYPE getType() const { return TType::TYPE_FLOAT; }
	virtual void accept(TFieldVisitor * v) { v->visitFloatField(this); }
	virtual TField * create(const TField * p_AdaptedField) const { return new TShortestPathSetFieldAdaptor_Length(static_cast<TTypedFieldInterface<TShortestPathSet*> *>( const_cast<TField *>(p_AdaptedField) ) ); }
	virtual shared_ptr<class TMeasure> getDefaultMeasure(); 
};

class TShortestPathSetFieldAdaptor_PathSet : public TShortestPathSetFieldAdaptor<TIndexedOrigins*> 
{
public:
	TShortestPathSetFieldAdaptor_PathSet(TTypedFieldInterface<TShortestPathSet*> * p_SpSetField)
		: TShortestPathSetFieldAdaptor<TIndexedOrigins*>(p_SpSetField)
	{};
	virtual const string getTypeName() { return "indexed voxels"; }
	virtual const string getAdaptorName() const { return "path set"; }
	virtual TIndexedOrigins * const vvaluep(const TCoord3 & p) const { return m_SpSetField->vvaluep(p) ? m_SpSetField->vvaluep(p)->m_PathSet.get() : 0; }
	virtual TIndexedOrigins * const vvaluex(unsigned int x) const { return m_SpSetField->vvaluex(x) ? m_SpSetField->vvaluex(x)->m_PathSet.get() : 0; }
	virtual TType::TYPE getType() const { return TType::TYPE_INDEXEDORIGINS; }
	virtual void accept(TFieldVisitor * v);
	virtual TField * create(const TField * p_AdaptedField) const;
	virtual shared_ptr<class TMeasure> getDefaultMeasure();
	virtual shared_ptr<TIndexMapper> getIndexToField() const; 
};


class TShortestPathSetFieldAdaptor_Midpoints : public TShortestPathSetFieldAdaptor<TIndexedOrigins_Multimap*> 
{
public:
	TShortestPathSetFieldAdaptor_Midpoints(TTypedFieldInterface<TShortestPathSet*> * p_SpSetField);
	virtual const string getTypeName() { return "indexedorigins mm"; }
	virtual bool isVolatile() const { return true; }
	virtual const string getAdaptorName() const { return "midpoints"; }
	virtual TIndexedOrigins_Multimap * const vvaluep(const TCoord3 & p) const; 
	virtual TType::TYPE getType() const { return TType::TYPE_INDEXEDORIGINS_MULTIMAP; }
	virtual void accept(TFieldVisitor * v) { v->visitIndexedOriginsMultimapField(this); }
	virtual TField * create(const TField * p_AdaptedField) const;
	virtual shared_ptr<class TMeasure> getDefaultMeasure();
	virtual shared_ptr<TIndexMapper> getIndexToField() const; 
protected:
	mutable TIndexedOrigins_Multimap mm;
};


class TShortestPathSetFieldAdaptor_Eft : public TShortestPathSetFieldAdaptor<TIndexedOrigins*> 
{
public:
	TShortestPathSetFieldAdaptor_Eft(TTypedFieldInterface<TShortestPathSet*> * p_SpSetField)
		: TShortestPathSetFieldAdaptor<TIndexedOrigins*>(p_SpSetField)
	{};
	virtual const string getTypeName() { return "indexed voxels"; }
	virtual const string getAdaptorName() const { return "eft"; }
	virtual TIndexedOrigins * const vvaluep(const TCoord3 & p) const { return m_SpSetField->vvaluep(p) ? m_SpSetField->vvaluep(p)->m_Eft.get() : 0; }
	virtual TIndexedOrigins * const vvaluex(unsigned int x) const { return m_SpSetField->vvaluex(x) ? m_SpSetField->vvaluex(x)->m_Eft.get() : 0; }

	virtual TType::TYPE getType() const { return TType::TYPE_INDEXEDORIGINS; }
	virtual void accept(TFieldVisitor * v);
	virtual TField * create(const TField * p_AdaptedField) const;
	virtual shared_ptr<class TMeasure> getDefaultMeasure();
	virtual shared_ptr<TIndexMapper> getIndexToField() const; 

	virtual bool getDefaultRenderLines() { return true; }
	virtual bool getDefaultRenderAllVoxels() { return false; }
};


class TShortestPathSetFieldAdaptor_Seft : public TShortestPathSetFieldAdaptor<TIndexedOrigins*> 
{
public:
	TShortestPathSetFieldAdaptor_Seft (TTypedFieldInterface<TShortestPathSet*> * p_SpSetField)
		: TShortestPathSetFieldAdaptor<TIndexedOrigins*>(p_SpSetField)
	{};
	virtual const string getTypeName() { return "indexed voxels"; }
	virtual const string getAdaptorName() const { return "seft"; }
	virtual TIndexedOrigins * const vvaluep(const TCoord3 & p) const { return m_SpSetField->vvaluep(p) ? m_SpSetField->vvaluep(p)->m_Seft.get() : 0; }
	virtual TIndexedOrigins * const vvaluex(unsigned int x) const { return m_SpSetField->vvaluex(x) ? m_SpSetField->vvaluex(x)->m_Seft.get() : 0; }

	virtual TType::TYPE getType() const { return TType::TYPE_INDEXEDORIGINS; }
	virtual void accept(TFieldVisitor * v);
	virtual TField * create(const TField * p_AdaptedField) const;
	virtual shared_ptr<class TMeasure> getDefaultMeasure();
	virtual shared_ptr<TIndexMapper> getIndexToField() const; 

	virtual bool getDefaultRenderLines() { return true; }
	virtual bool getDefaultRenderAllVoxels() { return false; }
};



class TShortestPathSetFieldAdaptor_FpAngle : public TShortestPathSetFieldAdaptor<float> 
{
public:
	TShortestPathSetFieldAdaptor_FpAngle(TTypedFieldInterface<TShortestPathSet*> * p_SpSetField)
		: TShortestPathSetFieldAdaptor<float>(p_SpSetField)
	{};
	virtual const string getTypeName() { return "float"; }
	virtual const string getAdaptorName() const { return "fp angle"; }
	virtual const float vvaluep(const TCoord3 & p) const; 
	virtual const float vvaluex(unsigned int x) const; 

	virtual TType::TYPE getType() const { return TType::TYPE_FLOAT; }
	virtual void accept(TFieldVisitor * v);
	virtual TField * create(const TField * p_AdaptedField) const;
	virtual shared_ptr<class TMeasure> getDefaultMeasure();

};

class TShortestPathSetFieldAdaptor_Radius : public TShortestPathSetFieldAdaptor<float> 
{
public:
	TShortestPathSetFieldAdaptor_Radius(TTypedFieldInterface<TShortestPathSet*> * p_SpSetField)
		: TShortestPathSetFieldAdaptor<float>(p_SpSetField)
	{};
	virtual const string getTypeName() { return "float"; }
	virtual const string getAdaptorName() const { return "radius"; }
	virtual const float vvaluep(const TCoord3 & p) const; 
	virtual const float vvaluex(unsigned int x) const; 

	virtual TType::TYPE getType() const { return TType::TYPE_FLOAT; }
	virtual void accept(TFieldVisitor * v);
	virtual TField * create(const TField * p_AdaptedField) const;
	virtual shared_ptr<class TMeasure> getDefaultMeasure();
};


class TShortestPathSetFieldAdaptor_MaxPathVoxelDistance : public TShortestPathSetFieldAdaptor<float> 
{
public:
	TShortestPathSetFieldAdaptor_MaxPathVoxelDistance(TTypedFieldInterface<TShortestPathSet*> * p_SpSetField)
		: TShortestPathSetFieldAdaptor<float>(p_SpSetField)
	{};
	virtual const string getTypeName() { return "float"; }
	virtual const string getAdaptorName() const { return "eccentricity"; }
	virtual const float vvaluep(const TCoord3 & p) const; 
	virtual const float vvaluex(unsigned int x) const; 

	virtual TType::TYPE getType() const { return TType::TYPE_FLOAT; }
	virtual void accept(TFieldVisitor * v);
	virtual TField * create(const TField * p_AdaptedField) const;
	virtual shared_ptr<class TMeasure> getDefaultMeasure();
};



#endif
