#include "stdafx.h"

#include "filter.h"

#include "deltaomega.h"
#include "layer.h"
#include "skeletonizer.h"
#include "component.h"
#include "globals.h"

void TMeasure::init()	
{
	m_IsInitialized = true;
	if(m_AbstractField)
	{
		m_AbstractField->begin(); // exception can be thrown here
		m_MinValue = +(std::numeric_limits<float>::max)();
		m_MaxValue = -(std::numeric_limits<float>::max)();
		if(m_AbstractField->usingIndexField())
		{
			const TIndexMapper * IndexField = m_AbstractField->getIndexField().get();
			for(unsigned int x=0; x<IndexField->getMaxIndex(); x++)
			{
				const float v = this->vvaluex(x);
				if( fabs(v) < (std::numeric_limits<unsigned int>::max)()*0.5f  )
				{
					if(v < m_MinValue) m_MinValue = v;
					if(v > m_MaxValue) m_MaxValue = v;
				}
			}
		}
		else
		{
			for(unsigned int i=0; i<m_AbstractField->getMaxX(); i++)
				for(unsigned int j=0; j<m_AbstractField->getMaxY(); j++)
					for(unsigned int k=0; k<m_AbstractField->getMaxZ(); k++)
					{
						const float v = this->toFloat(TCoord3(i,j,k));
						if( fabs(v) < (std::numeric_limits<unsigned int>::max)()*0.5f  )
						{
							if(v < m_MinValue) m_MinValue = v;
							if(v > m_MaxValue) m_MaxValue = v;
						}
					}
		}
		if(m_MinValue == +(std::numeric_limits<float>::max)()) m_MinValue = -1;
		if(m_MaxValue == -(std::numeric_limits<float>::max)()) m_MaxValue = -1;
		m_AbstractField->end();
	}
	else
	{
		m_MinValue = -1;
		m_MaxValue = -1;
	}
	m_MaxValue = ceil(m_MaxValue);
}


float TMeasure::getMinValue() 
{
	if(! m_IsInitialized) this->init();
	return m_MinValue;
}

float TMeasure::getMaxValue() 
{
	if(! m_IsInitialized) this->init();
	return m_MaxValue;
}



void TFloatFilter::DrawGui(wxWindow * p_Window)
{
	new TFloatFilterPanel(this, p_Window, -1, wxPoint(5,5), wxSize(200,200) );
}

void TFilterAlwaysTrue::DrawGui(wxWindow * p_Window)
{
	new wxStaticText(p_Window, -1, "True filter", wxPoint(5,5));
}

float TFloatFieldMeasure_Normalized::toFloat(const TCoord3 & p) const
{ 
	if(!g_Mediator.getSkeletonizer().get()) return m_Field->vvaluep(p);
	return m_Field->vvaluep(p) / g_Mediator.getSkeletonizer().get()->m_NormalizeBy; 
}

string TFloatFieldMeasure_Normalized::getName() const
{ 
	if(!g_Mediator.getSkeletonizer().get()) return "float normalize by ???";
	return wxString::Format("float normalize by %0.4f", g_Mediator.getSkeletonizer().get()->m_NormalizeBy).ToStdString(); 
}



float TIndexedOriginsFieldMeasure_MinDistance::toFloat(const TCoord3 & p) const 
{ 
	if(!m_Field->vvaluep(p)) return 0;

	float mind = (std::numeric_limits<float>::max)();
	shared_ptr<TIndexedOrigins::TIterator> it( m_Field->vvaluep(p)->newIterator());
	for(it->init(); it->valid(); it->next())
	{
		const unsigned int idx = it->value();
		if( idx >= m_Field->getIndexToField()->getMaxIndex() ) continue;

		const TCoord3 c = m_Field->getIndexToField()->vidx2coord(idx);
		const float d = c.distance(p);
		if(d < mind) mind = d;
	}
	return mind;	
} 

float TIndexedOriginsFieldMeasureMultimap_MinDistance::toFloat(const TCoord3 & p) const 
{ 
	if(!m_Field->vvaluep(p)) return 0;

	float mind = (std::numeric_limits<float>::max)();

	TIndexedOrigins_Multimap * io = m_Field->vvaluep(p);
	TIndexedOrigins_Multimap::iterator it;
	for(it = io->begin(); it != io->end(); it++)
	{
		const unsigned int idx = it->second;
		if( idx >= m_Field->getIndexToField()->getMaxIndex() ) continue;

		const TCoord3 c = m_Field->getIndexToField()->vidx2coord(idx);
		const float d = c.distance(p);
		if(d < mind) mind = d;
	}
	return mind;	
} 

float TIndexedOriginsFieldMeasureMultimap_MaxDistance::toFloat(const TCoord3 & p) const 
{ 
	if(!m_Field->vvaluep(p)) return 0;

	float mind = (std::numeric_limits<float>::min)();

	TIndexedOrigins_Multimap * io = m_Field->vvaluep(p);
	TIndexedOrigins_Multimap::iterator it;
	for(it = io->begin(); it != io->end(); it++)
	{
		const TCoord3 c = m_Field->getIndexToField()->vidx2coord(it->second);
		const float d = c.distance(p);
		if(d > mind) mind = d;
	}
	return mind;	
} 

float TIndexedOriginsFieldMeasureMultimap_OriginCount::toFloat(const TCoord3 & p) const 
{
	if(!m_Field->vvaluep(p)) return 0;
	return m_Field->vvaluep(p)->size();	
}

TFloatFilterPanel::TFloatFilterPanel(TFloatFilter * p_Filter, wxWindow* parent, wxWindowID id, const wxPoint& pos, const wxSize& size, long style)
		: wxPanel(parent,id,pos,size,style), m_Filter(p_Filter)
{	
	int x = 5;
	int y = 5;

	new wxStaticText(this, -1, "Filter settings", wxPoint(x,y) );
	y += 15;

	float Rate = 0.01f;
	int Min = 0;
	int Max = 100000;
	if(p_Filter->m_Measure) 
	{	
		float mmin = 0.001f;
		float mmax = (*p_Filter->m_Measure)->getMaxValue();
		Rate = ( mmax - mmin ) / 1000.0f;
		Min = (mmin / Rate);
		Max = (mmax / Rate)+1;
	}
	

	{
		x += 20;
		y += 15;
		
		m_CheckBoxAlwaysTrue = new wxCheckBox(this, ID_ALWAYSTRUE, "Always True", wxPoint(x,y), wxDefaultSize);
		m_CheckBoxAlwaysTrue->SetValue(m_Filter->getAlwaysTrue());

		y += 20;
		
		float v = (float) m_Filter->m_LowerValue;
		new wxStaticText(this, -1, "Lower", wxPoint(x,y) );
		m_SpinCtrlLower = new wxSpinCtrlForFloats(this, ID_LOWERVALUE, Min, Max, Rate, v, wxPoint(x+40,y), wxSize(70,20) );
		y += 35;
		v = (float) m_Filter->m_UpperValue;
		new wxStaticText(this, -1, "Upper", wxPoint(x,y) );
		m_SpinCtrlUpper = new wxSpinCtrlForFloats(this, ID_UPPERVALUE, Min, Max, Rate, v, wxPoint(x+40,y), wxSize(70,20) );
		y += 35;
		x -= 20;
	}

	// Gebruik hier connect omdat ik niet weet wat wxEVT_COMMAND_SPINCTRL_UPDATED in een event table is.
	this->Connect( ID_LOWERVALUE, wxEVT_COMMAND_SPINCTRL_UPDATED, (wxObjectEventFunction) &TFloatFilterPanel::OnChangeLower );
	this->Connect( ID_UPPERVALUE, wxEVT_COMMAND_SPINCTRL_UPDATED, (wxObjectEventFunction) &TFloatFilterPanel::OnChangeUpper );
	this->Connect( ID_ALWAYSTRUE, wxEVT_COMMAND_CHECKBOX_CLICKED, (wxObjectEventFunction) &TFloatFilterPanel::OnChangeAlwaysTrue );
}


void TFloatFilterPanel::OnChangeLower(wxCommandEvent & event)
{
	if(m_SpinCtrlLower->GetFloat() >= 0.0f)
	{
		m_Filter->m_LowerValue = m_SpinCtrlLower->GetFloat();
	}
	(*m_Filter->m_Measure)->m_AbstractField->getLayer()->onLayerChanged();
}

void TFloatFilterPanel::OnChangeUpper(wxCommandEvent & event)
{
	m_Filter->m_UpperValue = m_SpinCtrlUpper->GetFloat();
	(*m_Filter->m_Measure)->m_AbstractField->getLayer()->onLayerChanged();
}

void TFloatFilterPanel::OnChangeAlwaysTrue(wxCommandEvent & event)
{
	m_Filter->m_AlwaysTrue = m_CheckBoxAlwaysTrue->IsChecked();
	if(m_Filter->m_Measure) (*m_Filter->m_Measure)->m_AbstractField->getLayer()->onLayerChanged();
}
 

TMeasuresBox::TMeasuresBox(wxWindow* parent, wxWindowID id, const wxString& value, const wxPoint& pos, const wxSize& size)
	: wxComboBox(parent,id,value,pos,size,0,0,wxCB_READONLY,wxDefaultValidator,"comboBox")
{
	m_MeasurePrototypes.push_back( shared_ptr<TMeasure>(new TMeasureNone()) );
	m_MeasurePrototypes.push_back( shared_ptr<TMeasure>(new TIntFieldMeasure_Identity(0)) );
	m_MeasurePrototypes.push_back( shared_ptr<TMeasure>(new TUintFieldMeasure_Identity(0)) );
	m_MeasurePrototypes.push_back( shared_ptr<TMeasure>(new TFloatFieldMeasure_Identity(0)) );
	m_MeasurePrototypes.push_back( shared_ptr<TMeasure>(new TFloatFieldMeasure_Sqrt(0)) );
	m_MeasurePrototypes.push_back( shared_ptr<TMeasure>(new TFloatFieldMeasure_Normalized(0)) );
	m_MeasurePrototypes.push_back( shared_ptr<TMeasure>(new TUnsignedCharFieldMeasure_Identity(0)) );
//	m_MeasurePrototypes.push_back( shared_ptr<TMeasure>(new TOrigSetFieldMeasure_MinimumDistance(0)) );
//	m_MeasurePrototypes.push_back( shared_ptr<TMeasure>(new TCoord2FieldMeasure_CoordCount(0)) );
	m_MeasurePrototypes.push_back( shared_ptr<TMeasure>(new TIndexedOriginsFieldMeasure_CoordCount(0)) );
	m_MeasurePrototypes.push_back( shared_ptr<TMeasure>(new TIndexedOriginsFieldMeasure_MinDistance(0)) );
	m_MeasurePrototypes.push_back( shared_ptr<TMeasure>(new TCoord3SetField_CoordCount(0)) );
	m_MeasurePrototypes.push_back( shared_ptr<TMeasure>(new TCoord3WithFloatSetField_CoordCount(0)) );
	m_MeasurePrototypes.push_back( shared_ptr<TMeasure>(new TCoord3WithFloatSetField_Minimum(0)) );
	m_MeasurePrototypes.push_back( shared_ptr<TMeasure>(new TCoord3WithFloatSetField_Maximum(0)) );
	m_MeasurePrototypes.push_back( shared_ptr<TMeasure>(new TVector3FieldMeasure_Length(0)) );
	m_MeasurePrototypes.push_back( shared_ptr<TMeasure>(new TIndexedOriginsFieldMeasureMultimap_Minimum(0)) );
	m_MeasurePrototypes.push_back( shared_ptr<TMeasure>(new TIndexedOriginsFieldMeasureMultimap_Maximum(0)) );
	m_MeasurePrototypes.push_back( shared_ptr<TMeasure>(new TIndexedOriginsFieldMeasureMultimap_MinDistance(0)) );
	m_MeasurePrototypes.push_back( shared_ptr<TMeasure>(new TIndexedOriginsFieldMeasureMultimap_MaxDistance(0)) );
	m_MeasurePrototypes.push_back( shared_ptr<TMeasure>(new TIndexedOriginsFieldMeasureMultimap_OriginCount(0)) );
	m_MeasurePrototypes.push_back( shared_ptr<TMeasure>(new TShortestPathSetFieldMeasure_PathCount(0)) );
	m_MeasurePrototypes.push_back( shared_ptr<TMeasure>(new TShortestPathSetFieldMeasure_MaximumCircularity(0)) );
	m_MeasurePrototypes.push_back( shared_ptr<TMeasure>(new TComponentSetFieldMeasure_Count(0)) );
	m_MeasurePrototypes.push_back( shared_ptr<TMeasure>(new TComponentSetFieldMeasure_MaximumArea(0)) );


}

void TMeasuresBox::fill()
{
	Clear();
	for(int i=0; i<m_MeasurePrototypes.size(); i++)
	{
		shared_ptr<TMeasure> PrototypeMeasure( m_MeasurePrototypes[i] );
		Append(m_MeasurePrototypes[i]->getName().c_str(), (void*)PrototypeMeasure.get() );
	}	
}

void TMeasuresBox::fillForSourceType(TType::TYPE p_FieldType, TMeasure::MEASURETYPE p_MeasureType)
{
	Clear();
	Append(m_MeasurePrototypes[0]->getName().c_str(), (void*)m_MeasurePrototypes[0].get() );
	for(int i=0; i<m_MeasurePrototypes.size(); i++)
	{
		// Measures that operate on the current fieldtype
		if(m_MeasurePrototypes[i]->getSourceType() == p_FieldType)
		{
			shared_ptr<TMeasure> PrototypeMeasure = m_MeasurePrototypes[i];
			Append(m_MeasurePrototypes[i]->getName().c_str(), (void*)PrototypeMeasure.get() );

			// Select measure p_MeasureType
			if( p_MeasureType && PrototypeMeasure->getMeasureType() == p_MeasureType )
			{
				SetSelection( GetCount()-1 );
			}
		}
	}
	
	// Select first measure if none selected
	if( GetSelection() == -1 && GetCount() > 0 ) SetSelection(0);
}



float TCoord3WithFloatSetField_Minimum::toFloat(const TCoord3 & p) const
{
	const vector<pair<TCoord3,float> > * v = m_Field->vvaluep(p);
	if(v == 0 || v->size() == 0) return 0;

	unsigned int minidx = 0;
	for(unsigned int x=0; x<v->size(); x++)
	{
		if((*v)[x].second < (*v)[minidx].second) minidx = x;
	}
	return (*v)[minidx].second < 0.00123f ? 0.00123f : (*v)[minidx].second;
}


float TCoord3WithFloatSetField_Maximum::toFloat(const TCoord3 & p) const
{
	const vector<pair<TCoord3,float> > * v = m_Field->vvaluep(p);
	if(v == 0 || v->size() == 0) return 0;

	unsigned int maxidx = 0;
	for(unsigned int x=0; x<v->size(); x++)
	{
		if((*v)[x].second > (*v)[maxidx].second) maxidx = x;
	}
	return (*v)[maxidx].second;
}
