#include "histogram.h"

#include "globals.h"
#include "main.h"
#include "renderer.h"
#include "abstractfilter.h"
#include "colormap.h"
#include "filter.h"


THistogram::THistogram()
	: m_Cumulative(false)
	, m_LogScale(true)
	, m_AxisXName("noname")
	, m_SelectedBin(0)
	, m_Bins(128)
	, m_LogBase(1000.0f)
{
	TGlobalRenderer * GR = TGlobalRenderer::instance();
	GR->registerMe( this );
}


void THistogram::render()
{
	TRenderer Renderer;
	Renderer.begin();
	glDisable(GL_LIGHTING);
    glDisable(GL_DEPTH_TEST);
	glDisable(GL_CULL_FACE); 
	const TGlobalRenderer * GR = TGlobalRenderer::instance();
	
	TMap::iterator it, jt, kt;

	const float AxisYText = 0.15f;

	const float AxisX = 6.0f;
	const float AxisY = 2.0f;

	const float maxx = m_Points.rbegin()->first;
	m_BinSize = maxx / m_Bins;
	
	glPushName(m_PickID);
	glPushName(0);

	// Render axes
	{
		glLoadName(1);
		glPushName(0);
		if(m_ColorMap)
		{
			TColor3 c;
			float cmh = 0.1f;
			for(unsigned int i=0; i<=m_Bins; i++)
			{
				float x = m_BinSize*i;
				m_ColorMap->getColor(x, &c);
				glColor3fv(c.rgb);
				float x0 = (float)i * (AxisX/m_Bins);
				float x1 = (float)(i+1)*(AxisX/m_Bins);
				glLoadName(i);

				glBegin(GL_QUADS);
					glVertex3f(x0, 0.0f, 0.0f);
					glVertex3f(x0, -AxisYText, 0.0f);
					glVertex3f(x1, -AxisYText, 0.0f);
					glVertex3f(x1, 0.0f, 0.0f);
				glEnd();
			}
		}
		glPopName();
		glLoadName(0);

		glBegin(GL_LINES);
		glColor3f(0.0, 0.0, 0.0);
		glVertex3f(0.0f, 0.0f, 0.0f);
		glVertex3f(AxisX, 0.0f, 0.0f);
		glVertex3f(0.0f, 0.0f, 0.0f);
		glVertex3f(0.0f, AxisY, 0.0f);
		glEnd();

	}

	// Determine max bin value
	unsigned int maxy = 0;
	{
		unsigned int  count = 0;
		jt = m_Points.lower_bound( 0.0f );
		for(unsigned int x=1; x<=m_Bins; x++)
		{
			it = jt;
			jt = m_Points.lower_bound( (float)(x+1)*m_BinSize );
			if( !m_Cumulative ) count = 0;
			for(kt = it; kt != jt; kt++)
			{
				count += kt->second;
			}
			maxy = (count > maxy ? count : maxy);
		}

	}

/*
	// Render options
	{
		glLoadName(2);
		glBegin(GL_QUADS);
			glColor3f(1.0f, 0.0f, 0.0f);
			glVertex3f(0.0f, 0.0f, 0.0f);
			glVertex3f(0.0f, -AxisYText, 0.0f);
			glVertex3f(-AxisYText, -AxisYText, 0.0f);
			glVertex3f(-AxisYText, 0.0f, 0.0f);
		glEnd();
*/

/*	
		glPushMatrix();
			const float fontscale = 0.15;
			glColor3f(0.5f, 0.5f, 0.5f);
			glScalef(fontscale, fontscale, fontscale);
			GR->m_Font->renderText( "O", TFont::FONT_RIGHTALIGN | TFont::FONT_TOPALIGN | TFont::FONT_NOBLEND );
		glPopMatrix();
		glLoadName(0);
	}
*/

	// Render axis texts
	{
		glEnable(GL_COLOR_MATERIAL);
		glColor3f(0.0, 0.0, 0.0);
	
		const float fontscale = 0.15;

		// Increase / decrease binsize controls
		glPushMatrix();
			glTranslatef(0.0f, -AxisYText, 0.0f);
			glScalef(fontscale, fontscale, fontscale);
			glLoadName(2);
			GR->m_Font->renderText( "<", TFont::FONT_LEFTALIGN | TFont::FONT_TOPALIGN | TFont::FONT_NOBLEND );
			glLoadName(3);
			glTranslatef(GR->m_Font->getWidth(), 0.0f, 0.0f );
			GR->m_Font->renderText( ">", TFont::FONT_LEFTALIGN | TFont::FONT_TOPALIGN | TFont::FONT_NOBLEND );
			glLoadName(0);
		glPopMatrix();

		// Increase / decrease log scale
		glPushMatrix();
			glTranslatef(0.0f, AxisYText, 0.0f);
			glRotatef(90.0f, 0.0f, 0.0f, 1.0f);
			glTranslatef(0.0f, AxisYText, 0.0f);
			glScalef(fontscale, fontscale, fontscale);
			glLoadName(4);
			GR->m_Font->renderText( "<", TFont::FONT_LEFTALIGN | TFont::FONT_NOBLEND );
			glLoadName(5);
			glTranslatef(GR->m_Font->getWidth(), 0.0f, 0.0f );
			GR->m_Font->renderText( ">", TFont::FONT_LEFTALIGN | TFont::FONT_NOBLEND );
			glLoadName(0);
		glPopMatrix();

		
		glPushMatrix();
			glTranslatef(0.5f, -AxisYText, 0.0f );
			glScalef(fontscale, fontscale, fontscale);
			GR->m_Font->renderText( wxString::Format("%s",m_AxisXName.c_str()).c_str(), TFont::FONT_LEFTALIGN | TFont::FONT_TOPALIGN );
		glPopMatrix();

		glPushMatrix();
			glTranslatef(AxisX, -AxisYText, 0.0f);
			glScalef(fontscale, fontscale, fontscale);
			glTranslatef(0.0f, -GR->m_Font->getHeight(), 0.0f);

			GR->m_Font->renderText( wxString::Format("%.2f",maxx).c_str(), TFont::FONT_RIGHTALIGN );
		glPopMatrix();

		glPushMatrix();
			glTranslatef(0.0f, AxisY, 0.0f);
			glScalef(fontscale, fontscale, fontscale);
			GR->m_Font->renderText( wxString::Format("%u",maxy).c_str() , TFont::FONT_RIGHTALIGN | TFont::FONT_TOPALIGN  );
		glPopMatrix();
	}

	// Render bins
	{
		glColor3f(1.0, 0.0, 0.0);
		jt = m_Points.lower_bound( 0.0f );
		float count = 0;
		for(unsigned int x=0; x<=m_Bins; x++)
		{
			it = jt;
			jt = m_Points.lower_bound( (float)(x+1)*m_BinSize );
			if( !m_Cumulative ) count = 0;
			for(kt = it; kt != jt; kt++)
			{
				count += kt->second;
			}

			float value = 0.0f;
			if(m_LogScale) 
			{
				value = (float)count / (float)maxy;
				//value = 1.0f + value * 9999.0f;
				//value = log( value ) / log( 10000.0f );
				//value = (log(value)/log(m_LogBase)) / (log( 10000.0f )/log(m_LogBase));
				
				value = 1.0f + value * (m_LogBase-1.0f);
				value = log( value ) / log( m_LogBase );
				
			}
			else
			{
				value = (float)count / (float)maxy;
			}

			// value between 0 and 1 

			float x0 = (float) x * (AxisX/m_Bins);
			float x1 = (float)(x+1)*(AxisX/m_Bins);
			float y = AxisY * value;
			glBegin(GL_POLYGON);
			glVertex3f( x0,		0.0f,	0.0f );
			glVertex3f( x1,		0.0f,	0.0f );
			glVertex3f( x1,		y,		0.0f );
			glVertex3f(	x0,		y,		0.0f );
			glEnd();
			glVertex3f(x,y,0);
		}
	}

	// Render selection
	{
		float x0 = (float) m_SelectedBin * (AxisX/m_Bins);
		float x1 = (float)(m_SelectedBin+1)*(AxisX/m_Bins);
		float y = AxisY;
		glBegin(GL_LINES);
		glColor3f(0,0,0);
		glVertex3f( x0,		y,		0.0f );
		glVertex3f( x0,		-AxisYText,	0.0f );
		glEnd();
	}

	glPopName();
	glPopName();
	Renderer.end();
}

void THistogram::OnPicked(vector<unsigned int> p_PickStack, class wxMouseEvent * event)
{
}


TLayerHistogram::TLayerHistogram(shared_ptr<TLayer> p_Layer)
	: THistogram()
	, m_Layer(p_Layer)
{
}


void TLayerHistogram::update(bool p_UseFilter)
{
	const TLayer * Layer = m_Layer.get();
	const TField * Field = Layer->m_Field.get();
	if(! Field->usingIndexField() ) throw string("! Field->usingIndexField()");
	const TIndexMapper * IndexField = Field->getIndexField().get();
	const TMeasure * Measure = Layer->m_FilterMeasure.get();
	const TFilter * Filter = Layer->m_Filter.get();

	m_PointCount = 0;
	m_Points.clear();
	for(unsigned int x=0; x<IndexField->getMaxIndex(); x++)
	{
		if(p_UseFilter && !Filter->test( IndexField->vidx2coord(x) )) continue;
		const float v = Measure->toFloat( IndexField->vidx2coord(x) );
		if(v == 0.0f)  continue;

		TMap::iterator it = m_Points.find( v );
		if( it == m_Points.end() )
		{
			it = m_Points.insert( TMap::value_type(v, 1) );
		}
		else
		{
			it->second++;
		}
		m_PointCount++;
	}

	m_AxisXName = Layer->m_Name.substr(0,16);
	m_ColorMap = Layer->m_ColorMap;
}

void TLayerHistogram::OnPicked(vector<unsigned int> p_PickStack, class wxMouseEvent * event)
{
	THistogram::OnPicked(p_PickStack, event);

	if(p_PickStack[0] == 1) // color map bar
	{
		if(event->LeftDown())
		{
			if(p_PickStack.size() <= 1) throw string("p_PickStack.size() <= 1");
			const unsigned int i = p_PickStack[1];
			m_SelectedBin = i;
			float v = m_BinSize * i;
			m_Layer->m_Filter->m_LowerValue = v;
			m_Layer->setNeedRedraw();
			g_Mediator.redrawCanvas();
		}
		else if(event->RightDown() || event->RightUp())
		{
			g_Mediator.getGLCanvas()->PopupMenu( new THistogramMenu(this), wxPoint(g_Mediator.getGLCanvas()->m_MouseX, g_Mediator.getGLCanvas()->m_MouseY) );
		}
	}
	else if(p_PickStack[0] == 2) // decrease bins 
	{
		m_Bins /= 2;
		m_Bins = max(m_Bins,4);
		m_SelectedBin /= 2;
		m_Layer->setNeedRedraw();
		g_Mediator.redrawCanvas();
	}
	else if(p_PickStack[0] == 3) // increase bins
	{
		m_Bins *= 2;
		m_SelectedBin *= 2;
		m_Layer->setNeedRedraw();
		g_Mediator.redrawCanvas();
	}
	else if(p_PickStack[0] == 4) // decrease base
	{
		m_LogBase /= 2.0f;
		if(m_LogBase < 2.0f) m_LogBase = 2.0f;
		m_Layer->setNeedRedraw();
		g_Mediator.redrawCanvas();
	}
	else if(p_PickStack[0] == 5) // increase base
	{
		m_LogBase *= 2.0f;
		m_Layer->setNeedRedraw();
		g_Mediator.redrawCanvas();
	}
}




// ---------------------------

THistogramMenu::THistogramMenu(TLayerHistogram * p_LayerHistogram)
	: wxMenu()
	, m_LayerHistogram(p_LayerHistogram)
{	
	AppendCheckItem( ID_CUMULATIVE, wxT("Cumulative") );
	FindItem(ID_CUMULATIVE)->Check( m_LayerHistogram->m_Cumulative );

	AppendCheckItem( ID_LOG, wxT("Log") );
	FindItem(ID_LOG)->Check( m_LayerHistogram->m_LogScale );

//	AppendCheckItem( ID_USEFILTER, wxT("Use Filter") );
//	FindItem(ID_USEFILTER)->Check( m_LayerHistogram->m_UseFilter );
}


void THistogramMenu::OnMenuChoice(wxCommandEvent& event)
{

	// Histogram options
	if(event.GetId() == ID_CUMULATIVE)
	{
		m_LayerHistogram->m_Cumulative = ! event.IsChecked();
		g_Mediator.redrawCanvas();
	}
	else if(event.GetId() == ID_LOG)
	{
		m_LayerHistogram->m_LogScale = ! event.IsChecked();
		g_Mediator.redrawCanvas();
	}
	else if(event.GetId() == ID_USEFILTER)
	{
//		m_LayerHistogram->m_UseFilter = ! event.IsChecked();
//		m_LayerHistogram->update();
//		g_Mediator.redrawCanvas();
	}

}


BEGIN_EVENT_TABLE(THistogramMenu, wxMenu)
    EVT_MENU_RANGE(ID_FIRST, ID_LAST, THistogramMenu::OnMenuChoice)
END_EVENT_TABLE()



























TScatterPlot::TScatterPlot()
	: m_AxisXName("noname")
{
	TGlobalRenderer * GR = TGlobalRenderer::instance();
	GR->registerMe( this );
}


void TScatterPlot::render()
{
	TRenderer Renderer;
	Renderer.begin();
	glDisable(GL_LIGHTING);
    glDisable(GL_DEPTH_TEST);
	glDisable(GL_CULL_FACE); 
	const TGlobalRenderer * GR = TGlobalRenderer::instance();
	
	TMap::iterator it, jt, kt;

	const float AxisYText = 0.15f;

	const float AxisX = 6.0f;
	const float AxisY = 2.0f;

	const float maxx = m_Points.rbegin()->first;
	const float maxy = m_Points.rbegin()->second;
	
	glPushName(m_PickID);
	glPushName(0);

	// Render axes
	{
		glLoadName(1);

		glBegin(GL_LINES);
		glColor3f(0.0, 0.0, 0.0);
		glVertex3f(0.0f, 0.0f, 0.0f);
		glVertex3f(AxisX, 0.0f, 0.0f);
		glVertex3f(0.0f, 0.0f, 0.0f);
		glVertex3f(0.0f, AxisY, 0.0f);
		glEnd();

	}


	// Render axis texts
	{
		glEnable(GL_COLOR_MATERIAL);
		glColor3f(0.0, 0.0, 0.0);
	
		const float fontscale = 0.15;

		// Increase / decrease binsize controls
		glPushMatrix();
			glTranslatef(0.0f, -AxisYText, 0.0f);
			glScalef(fontscale, fontscale, fontscale);
			glLoadName(2);
			GR->m_Font->renderText( "<", TFont::FONT_LEFTALIGN | TFont::FONT_TOPALIGN | TFont::FONT_NOBLEND );
			glLoadName(3);
			glTranslatef(GR->m_Font->getWidth(), 0.0f, 0.0f );
			GR->m_Font->renderText( ">", TFont::FONT_LEFTALIGN | TFont::FONT_TOPALIGN | TFont::FONT_NOBLEND );
			glLoadName(0);
		glPopMatrix();

		// Increase / decrease log scale
		glPushMatrix();
			glTranslatef(0.0f, AxisYText, 0.0f);
			glRotatef(90.0f, 0.0f, 0.0f, 1.0f);
			glTranslatef(0.0f, AxisYText, 0.0f);
			glScalef(fontscale, fontscale, fontscale);
			glLoadName(4);
			GR->m_Font->renderText( "<", TFont::FONT_LEFTALIGN | TFont::FONT_NOBLEND );
			glLoadName(5);
			glTranslatef(GR->m_Font->getWidth(), 0.0f, 0.0f );
			GR->m_Font->renderText( ">", TFont::FONT_LEFTALIGN | TFont::FONT_NOBLEND );
			glLoadName(0);
		glPopMatrix();

		
		glPushMatrix();
			glTranslatef(0.5f, -AxisYText, 0.0f );
			glScalef(fontscale, fontscale, fontscale);
			GR->m_Font->renderText( wxString::Format("%s",m_AxisXName.c_str()).c_str(), TFont::FONT_LEFTALIGN | TFont::FONT_TOPALIGN );
		glPopMatrix();

		glPushMatrix();
			glTranslatef(AxisX, -AxisYText, 0.0f);
			glScalef(fontscale, fontscale, fontscale);
			glTranslatef(0.0f, -GR->m_Font->getHeight(), 0.0f);

			GR->m_Font->renderText( wxString::Format("%.2f",maxx).c_str(), TFont::FONT_RIGHTALIGN );
		glPopMatrix();

		glPushMatrix();
			glTranslatef(0.0f, AxisY, 0.0f);
			glScalef(fontscale, fontscale, fontscale);
			GR->m_Font->renderText( wxString::Format("%u",maxy).c_str() , TFont::FONT_RIGHTALIGN | TFont::FONT_TOPALIGN  );
		glPopMatrix();
	}

	// Render graph
	{
		TMap::iterator it;
		glColor3f(1.0, 0.0, 0.0);
		glBegin(GL_LINE_STRIP);
		for(it = m_Points.begin(); it != m_Points.end(); it++)
		{
			glVertex3f(it->first / maxx, it->second / maxy, 0.0f);
		}
		glEnd();
	}



	glPopName();
	glPopName();
	Renderer.end();
}

void TScatterPlot::OnPicked(vector<unsigned int> p_PickStack, class wxMouseEvent * event)
{
}
