#include "stdafx.h"

#include "layer.h"

#include "globals.h"
#include "main.h" // still here because of ID_'s

#include "abstractfield.h"
#include "abstractfilter.h"
#include "colormap.h"
#include "deltaomega.h"
#include "indexedorigins_cells.h"
#include "skeletonizer.h"
#include "sskelsegmentation.h"
 
#include "filter.h"
#include "component.h"
#include "settings.h"

using namespace std;

int TLayer::m_LastId = 0;


using namespace std;

TAbstractLayer::TAbstractLayer(shared_ptr<TField> p_Field, string p_Name, string p_ShortName)
	: TLayer()
{
	m_Field = p_Field;
	m_Name = p_Name;
	if(m_Name == "") throw string("m_Name == \"\"");
	m_ShortName = p_ShortName;
	if(m_ShortName == "") m_ShortName = m_Name;
	m_Id = m_LastId++;
	m_Field->setLayer(this);

	// Create selection voxel set
	{
		m_Selection.reset( new TVoxelSetSelection(m_Field.get()) );
		m_Selection->setCheckedForRendering(true);
		m_VoxelSets.push_back( m_Selection );
	}
}

TLayer::TLayer(shared_ptr<TField> p_Field, string p_Name, string p_ShortName)
	: m_Field(p_Field)
	, m_CheckedForRendering(false)
	, m_RenderPrimitiveType(0)
	, m_RenderLines(false)
	, m_RenderSelectionIndicators(true)
	, m_RenderVoxelOutlines(false)
	, m_Depth(0)
{
	m_Id = m_LastId++;
	m_Field->setLayer(this);
	m_Name = p_Name;
	if(m_Name == "") throw string("m_Name == \"\"");
	m_ShortName = p_ShortName;
	if(m_ShortName == "") m_ShortName = m_Name;
	init();
}

void TLayer::init()
{
	m_FilterMeasure = m_Field->getDefaultMeasure();
	try
	{
		m_FilterMeasure->init();
	}
	catch(...) {}

	m_Filter.reset( new TFloatFilter(&m_FilterMeasure) );

	if( m_Filter->m_LowerValue == 0.0f ) m_Filter->m_LowerValue = 0.0001f;
	

	m_AllColorMaps.push_back( shared_ptr<TColorMapUsingMeasure>( new TNormalizedColorMapUsingMeasure(&m_FilterMeasure) ) );
	m_AllColorMaps.back()->init();
	m_AllColorMaps.push_back( shared_ptr<TColorMapUsingMeasure>( new TMonochromeColorMapUsingMeasure(&m_FilterMeasure) ) );
	m_AllColorMaps.back()->init();
	m_AllColorMaps.push_back( shared_ptr<TColorMapUsingMeasure>( new TInverseMonochromeColorMapUsingMeasure(&m_FilterMeasure) ) );
	m_AllColorMaps.back()->init();
	m_AllColorMaps.push_back( shared_ptr<TColorMapUsingMeasure>( new THeatMapUsingMeasure(&m_FilterMeasure) ) );
	m_AllColorMaps.back()->init();
	m_AllColorMaps.push_back( shared_ptr<TColorMapUsingMeasure>( new TBlueYellowMapUsingMeasure(&m_FilterMeasure) ) );
	m_AllColorMaps.back()->init();
	m_AllColorMaps.push_back( shared_ptr<TColorMapUsingMeasure>( new TDecompositionColorMapUsingMeasure(&m_FilterMeasure) ) );
	m_AllColorMaps.back()->init();
	m_AllColorMaps.push_back( shared_ptr<TColorMapUsingMeasure>( new TDecomposition2ColorMapUsingMeasure(&m_FilterMeasure) ) );
	m_AllColorMaps.back()->init();
	m_AllColorMaps.push_back( shared_ptr<TColorMapUsingMeasure>( new TColorizeColorMapUsingMeasure(&m_FilterMeasure) ) );
	m_AllColorMaps.back()->init();
	

	m_ColorMap = m_AllColorMaps[0];

	// Create selection voxel set
	{
		m_Selection.reset( new TVoxelSetSelection(m_Field.get()) );
		m_Selection->setCheckedForRendering(true);
		m_VoxelSets.push_back( m_Selection );
	}

	// Create voxel sets with all voxels
	if(m_Field->usingIndexField())
	{
		shared_ptr<TVoxelSetAllUsingIndexMapper> vs(new TVoxelSetAllUsingIndexMapper(m_Field.get()));
		vs->setName("All voxels");
		vs->setCheckedForRendering(true);
		m_VoxelSets.push_back(vs);
		m_AllVoxels = vs;
	}
	else
	{
		shared_ptr<TVoxelSetAll> vs(new TVoxelSetAll(m_Field.get()));
		vs->setName("All voxels");
		vs->setCheckedForRendering(true);
		m_VoxelSets.push_back(vs);
		m_AllVoxels = vs;
	}

	if( m_Field->getType() == TType::TYPE_INDEXEDORIGINS ) 
	{
		m_AllVoxels->setCheckedForRendering(false);
		m_RenderLines = true;
	}

}

void TLayer::selectVisibleVoxels(const class TLayer * Layer)
{
	vector<TCoord3> VisibleVoxels;
	Layer->getVisibleVoxels(&VisibleVoxels);
	for(unsigned int x=0; x<VisibleVoxels.size(); x++)
	{
		m_Selection->select( VisibleVoxels[x] );
	}
}

void TLayer::onLayerChanged()
{
	set<TLayerObserver*>::iterator it;
	for(it = m_Observers.begin(); it != m_Observers.end(); it++)
	{
		(*it)->notifyLayerChanged();
	}
	setNeedRedraw();
	g_Mediator.redrawCanvas();
}

void TLayer::onFieldChanged()
{
	m_FilterMeasure->init();
	m_Filter.reset( new TFloatFilter(&m_FilterMeasure) );
	if( m_Filter->m_LowerValue == 0.0f ) m_Filter->m_LowerValue = 0.0001f;
//	m_ColorMapMeasure->init();
	m_ColorMap->init();

	onLayerChanged();

	g_Mediator.updateUI();
	g_Mediator.redrawCanvas();
}


void TLayer::getVisibleVoxels(vector<TCoord3> * p_VisibleVoxels) const
{
	p_VisibleVoxels->clear();
	for(unsigned int x=0; x<m_VoxelSets.size(); x++)
	{
		if(m_VoxelSets[x]->getCheckedForRendering())
		{
			m_VoxelSets[x]->initTraversal();
			TCoord3 c;
			while(m_VoxelSets[x]->nextTraversal(c))
			{
				if( m_Filter->test(c) )
				{
					p_VisibleVoxels->push_back(c); 
				}
			}
		}
	}
}

void TLayer::writeToVtk(const std::string& fileName)
{
	std::ofstream f(fileName);
	f.precision(4);

	f << "# vtk DataFile Version 2.0" << std::endl;
	f << "Exported from Skeleton Sandbox" << std::endl;
	f << "ASCII" << std::endl;
	f << "DATASET STRUCTURED_POINTS" << std::endl;
	f << "DIMENSIONS " << m_Field->getMaxX() << " " << m_Field->getMaxY() << " " << m_Field->getMaxZ() << std::endl;
	f << "SPACING 1 1 1" << std::endl;
	f << "ORIGIN 0 0 0" << std::endl;
	f << "POINT_DATA " << m_Field->getMaxX() * m_Field->getMaxY() * m_Field->getMaxZ() << std::endl;
	f << "SCALARS scalars float" << std::endl;
	f << "LOOKUP_TABLE default" << std::endl;

	for (int k = 0; k < m_Field->getMaxZ(); k++)
	{
		for (int j = 0 ; j < m_Field->getMaxY(); j++)
		{
			for (int i = 0; i < m_Field->getMaxX(); i++)
			{
				TCoord3 p(i, j, k);
				float v = m_FilterMeasure->toFloat(p);
				f << (m_Filter->test(v) ? v : 0.0f) << std::endl;
			}
		}
	}
	
	f.close();
}


void TLayerSet::deselectAll()
{
	TLayerList::const_iterator it;
	for(it = m_VisibleList.begin(); it != m_VisibleList.end(); it++)
	{
		(*it)->setCheckedForRendering(false);
	}
}

TField * TLayerSet::getField(const string & p_ShortName)
{
	shared_ptr<TLayer> l = getLayer(p_ShortName);
	if(l) return l->m_Field.get();
	else return 0;
}

shared_ptr<TLayer> TLayerSet::getLayer(const string & p_ShortName)
{
	TLayerList::const_iterator it;
	for(it = m_VisibleList.begin(); it != m_VisibleList.end(); it++)
	{
		if((*it)->m_ShortName == p_ShortName)
			return *it;	
	}
	return shared_ptr<TLayer>( (TLayer*)0 );
}

int TLayerSet::getIndex(const TLayer * p_Layer) const
{
	TLayerList::const_iterator it;
	int count = 0;
	for(it = m_VisibleList.begin(); it != m_VisibleList.end(); it++)
	{
		if( (*it).get() == p_Layer)
		{
			return count;
		}
		count++;
	}
	return -1;
}

TField *  TLayerSet::newField(TType::TYPE p_Type, shared_ptr<TIndexMapper> p_IndexMapper, const string p_ShortName)
{
	if(!exists(p_ShortName))
	{
		if(p_Type == TType::TYPE_FLOAT)
		{
			shared_ptr<TSparseFloatField3> Field( new TSparseFloatField3(p_IndexMapper) );
			shared_ptr<TLayer> NewLayer( new TLayer(Field, p_ShortName, p_ShortName) );
			g_Mediator.addLayer( NewLayer ); 
		}
		else if(p_Type == TType::TYPE_UINT)
		{
			shared_ptr<TSparseUintField3> Field( new TSparseUintField3(p_IndexMapper) );
			shared_ptr<TLayer> NewLayer( new TLayer(Field, p_ShortName, p_ShortName) );
			g_Mediator.addLayer( NewLayer ); 
		}
		else if(p_Type == TType::TYPE_UCHAR)
		{
			shared_ptr<TSparseUcharField3> Field( new TSparseUcharField3(p_IndexMapper) );
			shared_ptr<TLayer> NewLayer( new TLayer(Field, p_ShortName, p_ShortName) );
			g_Mediator.addLayer( NewLayer ); 
		}
		else if(p_Type == TType::TYPE_COORD3SET)
		{
			shared_ptr<TSparseCoord3SetField> Field( new TSparseCoord3SetField(p_IndexMapper) );
			shared_ptr<TLayer> NewLayer( new TLayer(Field, p_ShortName, p_ShortName) );
			g_Mediator.addLayer( NewLayer ); 
		}
		else if(p_Type == TType::TYPE_COORD3WITHFLOATSET )
		{
			shared_ptr<TSparseCoord3WithFloatSetField> Field( new TSparseCoord3WithFloatSetField(p_IndexMapper) );
			shared_ptr<TLayer> NewLayer( new TLayer(Field, p_ShortName, p_ShortName) );
			g_Mediator.addLayer( NewLayer ); 
		}
		else if(p_Type == TType::TYPE_SHORTESTPATHSET)
		{
			shared_ptr<TShortestPathSetField> Field( new TShortestPathSetField(p_IndexMapper) );
			shared_ptr<TLayer> NewLayer( new TLayer(Field, p_ShortName, p_ShortName) );
			g_Mediator.addLayer( NewLayer ); 
		}
		else if(p_Type == TType::TYPE_NODE)
		{
			shared_ptr<TSskelNodeField> Field( new TSskelNodeField(p_IndexMapper) );
			shared_ptr<TLayer> NewLayer( new TLayer(Field, p_ShortName, p_ShortName) );
			g_Mediator.addLayer( NewLayer ); 
		}
		else if(p_Type == TType::TYPE_VECTOR3)
		{
			shared_ptr<TSparseVector3Field> Field( new TSparseVector3Field(p_IndexMapper) );
			shared_ptr<TLayer> NewLayer( new TLayer(Field, p_ShortName, p_ShortName) );
			g_Mediator.addLayer( NewLayer ); 
		}
		else if(p_Type == TType::TYPE_COMPONENTSET)
		{
			shared_ptr<TComponentSetField> Field( new TComponentSetField(p_IndexMapper) );
			shared_ptr<TLayer> NewLayer( new TLayer(Field, p_ShortName, p_ShortName) );
			g_Mediator.addLayer( NewLayer ); 
		}
		
		else throw string("TLayerSet: invalid type");
	}

	return getLayer(p_ShortName)->m_Field.get();
}

TField * TLayerSet::newIoField(shared_ptr<TIndexMapper> p_IndexMapper, shared_ptr<TIndexMapper> p_IndexTo, const string p_ShortName)
{
	if(!exists(p_ShortName))
	{
		if(!p_IndexTo) 
		{
			throw string("TLayerSet::getLayer(): p_IndexTo == 0");
		}
		shared_ptr<TSparseIOField> Field( new TSparseIOField(p_IndexMapper, p_IndexTo) );
		shared_ptr<TLayer> NewLayer( new TLayer(Field, p_ShortName, p_ShortName) );
		g_Mediator.addLayer( NewLayer ); 
	}
	return getLayer(p_ShortName)->m_Field.get();
}

bool TLayerSet::exists(const string & p_ShortName)
{
	TLayerList::const_iterator it;
	for(it = m_VisibleList.begin(); it != m_VisibleList.end(); it++)
	{
		if((*it)->m_ShortName == p_ShortName)
			return true;	
	}
	return false;
}

shared_ptr<TLayer> TLayerSet::getLayerByIndex(int p_Index)
{
	if(p_Index < m_VisibleList.size() && p_Index >= 0) return m_VisibleList[p_Index];
	else return shared_ptr<TLayer>(static_cast<TLayer*>(0));
}

shared_ptr<TLayer> TLayerSet::getLayerById(int p_Id)
{
	if(p_Id < m_Id2Layer.size() && p_Id >= 0) return m_Id2Layer[p_Id];
	else return shared_ptr<TLayer>(static_cast<TLayer*>(0));
}

void TLayerSet::insertLayer(shared_ptr<TLayer> p_Layer)
{
	if(p_Layer->m_ShortName == "" || exists(p_Layer->m_ShortName)) 
	{
		p_Layer->m_ShortName = string( wxString::Format("noname%i",p_Layer->m_Id) );
	}
	if(p_Layer->m_Id >= m_Id2Layer.size()) m_Id2Layer.resize(p_Layer->m_Id+1);
	m_Id2Layer[p_Layer->m_Id] = p_Layer;
	m_VisibleList.push_back( p_Layer );
}

void TLayerSet::insertLayerAfterLayer(shared_ptr<TLayer> p_Layer, TLayer * p_AfterLayer)
{
	vector<shared_ptr<TLayer> > NewLayers;
	for(int i=0; i<m_VisibleList.size(); i++)
	{
		NewLayers.push_back(m_VisibleList[i]);
		if(m_VisibleList[i].get() == p_AfterLayer)
		{
			NewLayers.push_back(p_Layer);
		}
	}
	if(p_AfterLayer == 0)
	{
		NewLayers.push_back(p_Layer);
	}
	m_VisibleList = NewLayers;

	if(p_Layer->m_Id >= m_Id2Layer.size()) m_Id2Layer.resize(p_Layer->m_Id+1);
	m_Id2Layer[p_Layer->m_Id] = p_Layer;
}

void TLayerSet::deleteLayer(const string & p_ShortName)
{
	if(p_ShortName == "") throw string("name == \"\"");
	TLayerList NewList;
	for(unsigned int x=0; x<m_VisibleList.size(); x++)
	{
		if(m_VisibleList[x]->m_ShortName != p_ShortName)
		{
			NewList.push_back( m_VisibleList[x] );
		}
		else
		{
			shared_ptr<TLayer> EmptyLayer( static_cast<TLayer*>(0) );
			m_Id2Layer[m_VisibleList[x]->m_Id] = EmptyLayer;		
		}

	}
	m_VisibleList = NewList;
	g_Mediator.updateUI();
}

void TLayerSet::deleteLayer(int i)
{
}

void TLayerSet::deleteLayer(const TLayer * p_Layer)
{
	TLayerList NewList;
	for(unsigned int x=0; x<m_VisibleList.size(); x++)
	{
		if(m_VisibleList[x].get() == p_Layer )
		{
			NewList.push_back( m_VisibleList[x] );
		}
		else
		{
			shared_ptr<TLayer> EmptyLayer( static_cast<TLayer*>(0) );
			m_Id2Layer[m_VisibleList[x]->m_Id] = EmptyLayer;		
		}

	}
	m_VisibleList = NewList;
	g_Mediator.updateUI();
}




void TFieldSelectionVisitor::visitFloatField(TTypedFieldInterface<float> * p_Field)
//void TFieldSelectionVisitor::visitFloatField(class TFloatField3 * p_Field)
{
	if(m_Coord.i == -1) return;

//	m_SelectionInfoTextCtrl->AppendText( 
//		wxString::Format("Coord: (%i,%i,%i)\n", m_Coord.x,m_Coord.y,m_Coord.z)
//	);

	m_SelectionInfoTextCtrl->AppendText( 
		wxString::Format("Float value: %f\n", p_Field->vvaluep(m_Coord))
	);

}

void TFieldSelectionVisitor::visitIntegerField(TTypedFieldInterface<int> * p_Field) 
{
	if(m_Coord.i == -1) return;

//	m_SelectionInfoTextCtrl->AppendText( 
//		wxString::Format("Coord: (%i,%i,%i)\n", m_Coord.x,m_Coord.y,m_Coord.z)
//	);

	m_SelectionInfoTextCtrl->AppendText( 
		wxString::Format("Int value: %i\n", p_Field->vvaluep(m_Coord))
	);
}

void TFieldSelectionVisitor::visitUintegerField(TTypedFieldInterface<unsigned int> * p_Field)
{
	if(m_Coord.i == -1) return;

//	m_SelectionInfoTextCtrl->AppendText( 
//		wxString::Format("Coord: (%u,%u,%u)\n", m_Coord.x,m_Coord.y,m_Coord.z)
//	);

	m_SelectionInfoTextCtrl->AppendText( 
		wxString::Format("Uint value: %u\n", p_Field->vvaluep(m_Coord))
	);
}


//void TFieldSelectionVisitor::visitUnsignedCharField(class TUnsignedCharField3 * p_Field) 
void TFieldSelectionVisitor::visitUnsignedCharField(TTypedFieldInterface<unsigned char> * p_Field)
{
	if(m_Coord.i == -1) return;

//	m_SelectionInfoTextCtrl->AppendText( 
//		wxString::Format("Coord: (%i,%i,%i)\n", m_Coord.x,m_Coord.y,m_Coord.z)
//	);

	m_SelectionInfoTextCtrl->AppendText( 
		wxString::Format("Uchar value: %i\n", p_Field->vvaluep(m_Coord))
	);


}




void TFieldSelectionVisitor::visitVector3Field(TTypedFieldInterface<TVector3> * p_Field)
//voiid TFieldSelectionVisitor::visitVector3Field(class TVector3Field3 * p_Field) 
{
	if(m_Coord.i == -1) return;

//	m_SelectionInfoTextCtrl->AppendText( 
//		wxString::Format("Coord: (%i,%i,%i)\n", m_Coord.x,m_Coord.y,m_Coord.z)
//	);

	m_SelectionInfoTextCtrl->AppendText( 
		wxString::Format("Vector norm: %f\n", p_Field->vvaluep(m_Coord).norm())
	);

	const TVector3 & v = p_Field->vvaluep(m_Coord);
	m_SelectionInfoTextCtrl->AppendText( wxString::Format("Vector value: (%f,%f,%f)", v.x, v.y, v.z) );

}


void TFieldSelectionVisitor::visitIndexedOriginsField(TTypedFieldInterface<class TIndexedOrigins*> *  p_Field)
// void TFieldSelectionVisitor::visitIndexedOriginsField(class TIndexedOriginsField * p_Field) 
{
	if(m_Coord.i == -1) return;
	
//	m_SelectionInfoTextCtrl->AppendText( 
//		wxString::Format("Coord: (%i,%i,%i)\n", m_Coord.x,m_Coord.y,m_Coord.z)
//	);

	TIndexedOrigins * io = p_Field->vvaluep(m_Coord);
	if(io == 0) return;

	const unsigned int UpperLimit = 30;
	int Count = 0;
	m_SelectionInfoTextCtrl->AppendText( wxString::Format("Voxel count: %i\n", io->vertexcount()) );
	const TCoord3 a = m_Coord;
//	m_SelectionInfoTextCtrl->AppendText( wxString::Format("IndexToField MaxIndex: %i\n", p_Field->getIndexToField()->getMaxIndex() ) );


	shared_ptr<TIndexedOrigins::TIterator> it(io->newIterator());
	for(it->init(); it->valid(); it->next())
	{
		const TCoord3 c = p_Field->getIndexToField()->vidx2coord(it->value());
		const float dist = c.distance(a);
		m_SelectionInfoTextCtrl->AppendText(wxString::Format("%i. idx:%i, (%i,%i,%i), dist: %.2f\n", Count, it->value(), c.x, c.y, c.z, dist));
		if(Count++ > UpperLimit) break;
	}


	if( 

		false // !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
		&& io->getType() == TIndexedOrigins::TYPE_CELLS
		)
	{
		TIndexedOrigins_Cells * osc = io->castCells();

		TIndexedOrigins_Cells::TContainerType::const_iterator jt;
		unsigned int maxl = 0;
		for(jt=osc->begin(); jt != osc->end(); jt++) maxl = (*jt)->m_Level > maxl ? (*jt)->m_Level : maxl;

		m_SelectionInfoTextCtrl->AppendText(wxString::Format("\nCells, count: %i, #levels: %i\n", osc->size(), maxl+1));

		//TIndexedOrigins_Vector::iterator kt;
		for(unsigned int l=0; l<=maxl; l++)
		{
			m_SelectionInfoTextCtrl->AppendText(wxString::Format("\nlevel: %i\n", l));
			for(jt=osc->begin(); jt != osc->end(); jt++)
			{
				if( l == maxl && (*jt)->m_Level == maxl )
				{
					m_SelectionInfoTextCtrl->AppendText(wxString::Format("%i ", (*jt)->m_Index));
				}
				else if( (*jt)->m_Level == l )
				{
						m_SelectionInfoTextCtrl->AppendText(wxString::Format("idx: %i, #vert: %i\n", (*jt)->m_Index, (*jt)->m_Vertices.size() ));
				}
			}
		}
	}
		
	m_SelectionInfoTextCtrl->ShowPosition(0);
}

void TFieldSelectionVisitor::visitIndexedOriginsMultimapField(TTypedFieldInterface<class TIndexedOrigins_Multimap*> *  p_Field)
{
	if(m_Coord.i == -1) return;

	TIndexedOrigins_Multimap * iomm = p_Field->vvaluep(m_Coord);
	if(iomm == 0) return;
	TIndexedOrigins_Multimap::iterator it;
	m_SelectionInfoTextCtrl->AppendText( wxString::Format("Voxel count: %i\n", iomm->vertexcount()) );
	int Count = 0;
	for(it = iomm->begin(); it != iomm->end(); it++)
	{
		float dist = (numeric_limits<float>::max)();
		TCoord3 c(-1,-1,-1);
		if(it->second != TIndexedOrigins::INVALIDINDEX) 
		{
			c = g_Mediator.getSkeletonizer()->m_BoundaryIndexField->vidx2coord(it->second);
			dist = c.distance(m_Coord);
		}

		if(dist == (numeric_limits<float>::max)()) dist = 0;
		m_SelectionInfoTextCtrl->AppendText(wxString::Format("%i. (%i,%i,%i) idx:%i, dist:%4.2f, value: %.3f\n", Count, c.x, c.y, c.z, it->second, dist, it->first));
		Count++;
		if(Count > 100) break;
	}
	
}








void TFieldSelectionVisitor::visitCoord3SetField(TTypedFieldInterface<vector<TCoord3>*> * p_Field)
{
	if(m_Coord.i == -1) return;

	m_SelectionInfoTextCtrl->AppendText( 
		wxString::Format("Coord: (%i,%i,%i)\n", m_Coord.x,m_Coord.y,m_Coord.z)
	);
	
	const vector<TCoord3> * vec = p_Field->vvaluep(m_Coord);
	if(!vec) 
	{
		m_SelectionInfoTextCtrl->AppendText( wxString::Format("Null-pointer") );
	}
	else
	{
		m_SelectionInfoTextCtrl->AppendText( wxString::Format("Size: %i\n", vec->size()) );
		for(unsigned int i=0; i<vec->size(); i++)
		{
			m_SelectionInfoTextCtrl->AppendText( wxString::Format("%i. (%i,%i,%i)\n", i, (*vec)[i].x, (*vec)[i].y, (*vec)[i].z) );
			if(i > 50) break;
		}

		m_SelectionInfoTextCtrl->ShowPosition(0);
	}
}

void TFieldSelectionVisitor::visitCoord3WithFloatSetField(TTypedFieldInterface<vector<pair<TCoord3,float> >*> * p_Field)
{
}


void TFieldSelectionVisitor::visitShortestPathSetField(TTypedFieldInterface<class TShortestPathSet*> *  p_Field)
{
	if(m_Coord.i == -1) return;

//	m_SelectionInfoTextCtrl->AppendText( 
//		wxString::Format("Coord: (%i,%i,%i)\n", m_Coord.x,m_Coord.y,m_Coord.z)
//	);

	const TShortestPathSet * sps = p_Field->vvaluep(m_Coord);
	if(sps == 0) return;

	const TSkeletonizer * Skel = g_Mediator.getSkeletonizer().get();
	m_SelectionInfoTextCtrl->AppendText( wxString::Format("Max length: %f\n", sps->getMaxLength()) );
	if(!g_Settings.m_ReleaseMode) m_SelectionInfoTextCtrl->AppendText( wxString::Format("Sourcevoxel: (%i,%i,%i)\n", sps->m_SourceVoxel.x, sps->m_SourceVoxel.y, sps->m_SourceVoxel.z) );
	unsigned int x;
	
	// Print userdata occuring in sps
	if(!g_Settings.m_ReleaseMode)
	{
		m_SelectionInfoTextCtrl->AppendText( "Userdata: " );
		set<unsigned int> Set;
		set<unsigned int>::iterator it;
		for(x=0; x<sps->m_Paths.size(); x++)
		{
			shared_ptr<TShortestPath> sp = sps->m_Paths[x];
//			Set.insert(sp->m_UserData);
		}
		for(it=Set.begin(); it != Set.end(); it++)
		{
			m_SelectionInfoTextCtrl->AppendText( wxString::Format("%i, ", *it) );
		}
	}
	m_SelectionInfoTextCtrl->AppendText( wxString::Format("\n") );
	m_SelectionInfoTextCtrl->AppendText( wxString::Format("Eccentricity: %.4f\n", sps->getEccentricity() ) );

	m_SelectionInfoTextCtrl->AppendText( wxString::Format("Number of paths: %i\n", sps->m_Paths.size()) );
	for(x=0; x<sps->m_Paths.size(); x++)
	{
		shared_ptr<TShortestPath> sp = sps->m_Paths[x];
		const TCoord3 begin = Skel->Omega->m_BackgroundBoundaryIndexField->vidx2coord( sp->m_Begin );
		const TCoord3 end = Skel->Omega->m_BackgroundBoundaryIndexField->vidx2coord( sp->m_End );
		//const TCoord3 middle = Skel->m_BoundaryIndexField->vidx2coord( sp->m_Middle );

		m_SelectionInfoTextCtrl->AppendText( wxString::Format("Path \t%i\n", x) );
		m_SelectionInfoTextCtrl->AppendText( wxString::Format("\tbegin: idx %i, \t(%i,%i,%i)\n", sps->m_Paths[x]->m_Begin, begin.x, begin.y, begin.z) );
		m_SelectionInfoTextCtrl->AppendText( wxString::Format("\tend: idx %i, \t(%i,%i,%i)\n", sps->m_Paths[x]->m_End, end.x, end.y, end.z) );
		m_SelectionInfoTextCtrl->AppendText( wxString::Format("\tlength: %.3f\n", sp->m_Length) );
		//m_SelectionInfoTextCtrl->AppendText( wxString::Format("\tmiddle: idx %i, \t(%i,%i,%i)\n", sps->m_Paths[x]->m_Middle, middle.x, middle.y, middle.z) );
		if(sp->m_Path)
			m_SelectionInfoTextCtrl->AppendText( wxString::Format("\t#voxels: %i\n", sp->m_Path->size() ) );
		else
			m_SelectionInfoTextCtrl->AppendText( wxString::Format("\t#voxels: no path stored\n") );

//		m_SelectionInfoTextCtrl->AppendText( wxString::Format("\tUserdata: %i\n", sp->m_UserData) );
//		m_SelectionInfoTextCtrl->AppendText( wxString::Format("\tUserdata2: %i\n", sp->m_UserData2) );
		if(!g_Settings.m_ReleaseMode) 
		{
			m_SelectionInfoTextCtrl->AppendText( wxString::Format("\teccentricity: %.4f\n", sp->getEccentricity(sps->m_SourceVoxel) ) );
			if(sp->m_LocalSheet)
			{
				TPoint3 pt = sp->m_LocalSheet->getPoint();
				TPoint3 n = sp->m_LocalSheet->getNormal();
				m_SelectionInfoTextCtrl->AppendText( wxString::Format("\tplane origin: \t(%.3f,%.3f,%.3f), Plane normal: (%.3f,%.3f,%.3f)\n", pt.x,pt.y,pt.z, n.x,n.y,n.z) );
			}
		}		
	}

	m_SelectionInfoTextCtrl->ShowPosition(0);
}

void TFieldSelectionVisitor::visitComponentSetField(TTypedFieldInterface<class TComponentSet*> *  p_Field)
{
	if(m_Coord.i == -1) return;

	m_SelectionInfoTextCtrl->AppendText( 
		wxString::Format("Coord: (%i,%i,%i)\n", m_Coord.x,m_Coord.y,m_Coord.z)
	);

	const TComponentSet * cs = p_Field->vvaluep(m_Coord);
	if(cs == 0) return;

	const TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	unsigned int x,y;

	m_SelectionInfoTextCtrl->AppendText( wxString::Format("is on loop: %i\n", cs->isOnLoop()) );

	m_SelectionInfoTextCtrl->AppendText( wxString::Format("Number of components: %i\n", cs->count()) );
	for(x=0; x<cs->m_Cells.size(); x++)
	{
		m_SelectionInfoTextCtrl->AppendText( wxString::Format("Cell %i, size: %i\n", x, cs->m_Cells[x]->vertexcount() ) );
	}

	m_SelectionInfoTextCtrl->AppendText( wxString::Format("Number of unique cells: %i\n", cs->count()) );
	set<shared_ptr<TIndexedOrigins_Cells> >::iterator it;
	x=0;
	for(it = cs->m_UniqueCells.begin(); it != cs->m_UniqueCells.end(); it++)
	{
		m_SelectionInfoTextCtrl->AppendText( wxString::Format("Unique cell #%i size: %i\n", x, (*it)->vertexcount() ) );
		x++;
	}

	m_SelectionInfoTextCtrl->AppendText( wxString::Format("Number of borders: %i\n", cs->m_Borders.size()) );
	for(x=0; x<cs->m_BorderEft.size(); x++)
	{
		if(cs->m_BorderEft[x].get() == 0) continue;

		const float diameter = cs->m_BorderDiameters.size() > 0 ? cs->m_BorderDiameters[x] : 0.0f;
		const float length = cs->m_BorderLengths.size() > 0 ? cs->m_BorderLengths[x] : 0.0f;
		const float ecc = length / (diameter*PI);
		m_SelectionInfoTextCtrl->AppendText( wxString::Format("Border: %i, size: %i, length: %f, diameter: %f, ecc: %f, cellsize: %i, eft: ", x, cs->m_Borders[x]->size(), length, diameter, ecc, cs->m_Border2Cell[x]->vertexcount() ) );
		for(y=0; y<cs->m_BorderEft[x]->size(); y++)
		{
			m_SelectionInfoTextCtrl->AppendText( wxString::Format("%i, ", cs->m_BorderEft[x]->at(y)) );
		}
		m_SelectionInfoTextCtrl->AppendText( "\n" );
	}

	m_SelectionInfoTextCtrl->ShowPosition(0);
}


void TFieldSelectionVisitor::visitNodeField(TTypedFieldInterface<vector<class TNode*>*> *  p_Field)
{
	if(m_Coord.i == -1) return;

	m_SelectionInfoTextCtrl->AppendText( 
		wxString::Format("Coord: (%i,%i,%i)\n", m_Coord.x,m_Coord.y,m_Coord.z)
	);
	
	if(!p_Field->vvaluep(m_Coord)) 
	{
		m_SelectionInfoTextCtrl->AppendText( wxString::Format("Null-pointer") );
	}
	else
	{
		const vector<TNode*> & Nodes = *p_Field->vvaluep(m_Coord);
		m_SelectionInfoTextCtrl->AppendText( wxString::Format("#nodes: %i\n", Nodes.size()) );
		unsigned int i;
		for(i=0; i<Nodes.size(); i++)
		{
			const TNode * N = Nodes[i];
			m_SelectionInfoTextCtrl->AppendText( wxString::Format("node %i, idx: %i, segid: %i, coord: (%i,%i,%i), ", i, N->m_Index, N->m_SegmentId, N->m_Coord.x, N->m_Coord.y, N->m_Coord.z) );

			if(N->m_Eft)
			{
				m_SelectionInfoTextCtrl->AppendText( wxString::Format("m_Eft, size: %i, ", N->m_Eft->size()) );
			}

			if(N->m_ShortestPath)
			{
				m_SelectionInfoTextCtrl->AppendText( wxString::Format("sp length: %.3f, ", N->m_ShortestPath->m_Length) );
			}
			
			if(N->m_ShortestPaths.size() > 0)
			{
				m_SelectionInfoTextCtrl->AppendText( wxString::Format("sps size: %i\n", N->m_ShortestPaths.size() ) );
				for(unsigned int j=0; j<N->m_ShortestPaths.size(); j++)
				{	
					m_SelectionInfoTextCtrl->AppendText( wxString::Format("\t%i, length: %.3f, ", j, N->m_ShortestPaths[j]->m_Length) );
				}
			}

			m_SelectionInfoTextCtrl->AppendText("\n");
			if(i > 50) break;
		}

		// Print edges
		m_SelectionInfoTextCtrl->AppendText( wxString::Format("\nEdges for each node\n") );
		for(i=0; i<Nodes.size(); i++)
		{
			const TNode * Node = Nodes[i];
			m_SelectionInfoTextCtrl->AppendText( wxString::Format("node %i, idx: %i, edgescount: %i \n", i, Node->m_Index, Node->m_Edges.size()) );

			for(unsigned int j=0; j<Node->m_Edges.size(); j++)
			{
				m_SelectionInfoTextCtrl->AppendText( wxString::Format("\tedge %i, to: %i, sim geo fp: %.1f, sim tp: %.3f\n", j, Node->m_Neighbors[j], Node->m_Edges[j]->m_SimilarityGeoFp, Node->m_Edges[j]->m_SimilarityTp) );
			}
			if(i > 50) break;
		}

		m_SelectionInfoTextCtrl->ShowPosition(0);
	}
}





TLayerPanel::TLayerPanel(wxWindow* parent, wxWindowID id, const wxPoint& pos, const wxSize& size, long style)
	: wxPanel(parent, id, pos, size, style)
	, m_Layer(0)
	, m_ColorMapMeasuresBox(0)
	, m_FilterMeasuresBox(0)
	, m_SelectionPanel(0)
	, m_LayerInfoPanel(0)
	, m_UseVoxelSet(0)
	, m_UseFilter(0)
{
	wxBoxSizer * Sizer = new wxBoxSizer(wxVERTICAL);

	m_Notebook = new wxNotebook(this, -1);
	Sizer->Add(m_Notebook, 1, wxALL | wxEXPAND, 0);


	// Voxelsets panel
	{
		//wxBoxSizer * Sizer = new wxBoxSizer( wxVERTICAL );
		wxPanel * Panel = new wxPanel(m_Notebook, -1, wxDefaultPosition, wxDefaultSize);

		int x = 5;
		int y = 5;

		// VoxelSet listbox
		{
			m_VoxelSetListBox = new TVoxelSetListBox(this, Panel, ID_VOXELSETLISTBOX, wxPoint(x,y), wxSize(220,50) );
			y += m_VoxelSetListBox->GetSize().y + 10;
		}


//		new wxButton(Panel, ID_CREATEVOXELSLICE, "New Slice", wxPoint(x,y));
//		x += 120;

//		new wxButton(Panel, ID_NEWVOXELSETFROMFILTER, "From filter", wxPoint(x,y));
//		x += 120;
//		y += 60;

		x = 5;
		// Voxel set options
		new wxStaticText(Panel, -1, "Voxel set options", wxPoint(x,y) );
		m_VoxelSetPanel = new wxPanel(Panel, -1, wxPoint(x,y), wxSize(220,250), wxSUNKEN_BORDER);
		wxBoxSizer * Sizer = new wxBoxSizer(wxVERTICAL);
		m_VoxelSetPanel->SetSizer(Sizer);


		y += 210;

		m_Notebook->AddPage(Panel, "Voxelset");
	}

	// Filter panel
	{
		wxPanel * Panel = new wxPanel(m_Notebook, -1, wxDefaultPosition, wxDefaultSize);
		m_Notebook->AddPage(Panel, "Filter");

		int x = 5;
		int y = 5;

		new wxStaticText(Panel, -1, "Measure type", wxPoint(x,y) );
		y += 25;
		m_FilterMeasuresBox = new TMeasuresBox(Panel, ID_FILTERMEASURESBOX, "", wxPoint(x,y), wxSize(200,20));
		y += 25;

		m_FilterPanel = new wxPanel(Panel, -1, wxPoint(x,y), wxSize(200,200), wxSUNKEN_BORDER);
	}


	// Colormap panel
	{
		wxPanel * Panel = new wxPanel(m_Notebook, -1, wxDefaultPosition, wxDefaultSize);
		m_Notebook->AddPage(Panel, "Colormap");

		int x = 5;
		int y = 5;

//		new wxStaticText(Panel, -1, "Colormap Measure", wxPoint(x,y));
//		y += 25;
//		m_ColorMapMeasuresBox = new TMeasuresBox(Panel, ID_COLORMAPMEASURESBOX, "", wxPoint(x,y)),
//		y += 25;
		new wxStaticText(Panel, -1, "Colormap type", wxPoint(x,y));
		y += 25;
		m_ColormapTypeBox = new wxComboBox(Panel, ID_COLORMAPTYPE, "", wxPoint(x,y), wxSize(200,20), 0, 0, wxCB_READONLY,wxDefaultValidator,"comboBox");
		y += 25;

		m_ColormapPanel = new wxPanel(Panel, -1, wxPoint(x,y), wxSize(200,200), wxSUNKEN_BORDER);
	}


	// Selection info
	{
		wxPanel * Panel = new wxPanel(m_Notebook, -1, wxDefaultPosition, wxDefaultSize);
		m_Notebook->AddPage(Panel, "Selection");

		int x = 5;
		int y = 5;
		
		m_SelectionInfoTextCtrl.reset( new wxTextCtrl(Panel, -1, "", wxPoint(x,y), wxSize(200,200), wxTE_MULTILINE) );

		//new wxStaticText(Panel, -1, "Selection info ", wxPoint(x,y));
		//y += 25;

		//m_SelectionPanel = new wxPanel(Panel, -1, wxPoint(x,y), wxSize(220,220));
		wxBoxSizer * Sizer = new wxBoxSizer(wxVERTICAL);
		Sizer->Add(m_SelectionInfoTextCtrl.get(), 1, wxALL | wxEXPAND);
		Panel->SetSizer( Sizer );
	}

	// Information about selected layer
	if(!g_Settings.m_ReleaseMode)
	{
		wxPanel * Panel = new wxPanel(m_Notebook, -1, wxDefaultPosition, wxDefaultSize);
		m_Notebook->AddPage(Panel, "Layer");

		int x = 5;
		int y = 5;
		
		//wxStaticText * Text = new wxStaticText(Panel, -1, "Layer info", wxPoint(x,y));
		//y += 15;

		m_LayerInfoTextCtrl.reset( new wxTextCtrl(Panel, -1, "", wxPoint(x,y), wxSize(200,200), wxTE_MULTILINE) );

		wxBoxSizer * Sizer = new wxBoxSizer(wxVERTICAL);
		//Sizer->Add(Text, 0, wxTOP);
		Sizer->Add(m_LayerInfoTextCtrl.get() , 1, wxALL | wxEXPAND);
		Panel->SetSizer( Sizer );
	}

	// Field specific
	{
		//wxPanel * Panel = new wxPanel(m_Notebook, -1, wxDefaultPosition, wxDefaultSize);
		//m_FieldSpecificPanel = new wxPanel(m_Notebook, -1, wxPoint(5,5), wxSize(200,200), wxSUNKEN_BORDER);
		//m_Notebook->AddPage(m_FieldSpecificPanel, "Field specific");
	}

//	this->Connect( ID_VOXELSETLISTBOX, wxEVT_COMMAND_LISTBOX_SELECTED, (wxObjectEventFunction) &TLayerPanel::OnVoxelSetClick );
//	this->Connect( ID_VOXELSETLISTBOX, wxEVT_COMMAND_CHECKLISTBOX_TOGGLED, (wxObjectEventFunction) &TLayerPanel::OnVoxelSetToggle );

//	this->Connect( ID_CREATEVOXELSLICE, wxEVT_COMMAND_BUTTON_CLICKED, (wxObjectEventFunction) &TLayerPanel::OnCreateVoxelSlice );
	this->Connect( ID_USEFILTER, wxEVT_COMMAND_CHECKBOX_CLICKED, (wxObjectEventFunction) &TLayerPanel::OnUseFilterClick );
	this->Connect( ID_NEWVOXELSETFROMFILTER, wxEVT_COMMAND_BUTTON_CLICKED, (wxObjectEventFunction) &TLayerPanel::OnNewVoxelSetFromFilter );
	this->Connect( ID_FILTERMEASURESBOX, wxEVT_COMMAND_COMBOBOX_SELECTED, (wxObjectEventFunction) &TLayerPanel::OnFilterMeasureSelect );
	this->Connect( ID_COLORMAPMEASURESBOX, wxEVT_COMMAND_COMBOBOX_SELECTED, (wxObjectEventFunction) &TLayerPanel::OnColorMapMeasureSelect );
	this->Connect( ID_COLORMAPTYPE, wxEVT_COMMAND_COMBOBOX_SELECTED, (wxObjectEventFunction) &TLayerPanel::OnColorMapTypeSelect );

	this->SetSizer(Sizer);
}

BEGIN_EVENT_TABLE(TLayerPanel, wxPanel)
	EVT_CHECKLISTBOX( ID_VOXELSETLISTBOX, TLayerPanel::OnVoxelSetToggle )
	EVT_LISTBOX( ID_VOXELSETLISTBOX, TLayerPanel::OnVoxelSetClick )
END_EVENT_TABLE()


BEGIN_EVENT_TABLE(TVoxelSetListBox, wxCheckListBox)
    EVT_MOUSE_EVENTS( TVoxelSetListBox::OnMouse )
END_EVENT_TABLE()

void TLayerPanel::updateForLayer(TLayer * p_Layer)
{
	m_Layer = p_Layer;
	assert(m_Layer);


	// Fill voxel set listbox
	{
		m_VoxelSetListBox->Clear();

		// Render voxelsets
		for(int i=0; i<m_Layer->m_VoxelSets.size(); i++)
		{
			shared_ptr<TVoxelSet> vs = m_Layer->m_VoxelSets.at(i);
			m_VoxelSetListBox->Append(vs->getName().c_str());
			m_VoxelSetListBox->Check(i,vs->getCheckedForRendering());
			if( vs.get() == p_Layer->m_CurrentVoxelSet.get() ) 
			{	
				m_VoxelSetListBox->Select(i);
			}
		}
	}

	// Update filter measure
	m_FilterMeasuresBox->Clear();
	m_FilterPanel->DestroyChildren();
	if(m_Layer->m_Filter && m_Layer->m_FilterMeasure)
	{
		TType::TYPE FieldType = m_Layer->m_Field->getType();
		shared_ptr<TMeasure> FilterMeasure = m_Layer->m_FilterMeasure;
		m_FilterMeasuresBox->fillForSourceType( FieldType, FilterMeasure ? FilterMeasure->getMeasureType() : TMeasure::MEASURETYPE_NONE );

		m_Layer->m_Filter->DrawGui(m_FilterPanel);
	}

	// Update colormap
	m_ColormapTypeBox->Clear();
	m_ColormapPanel->DestroyChildren();
	if(m_Layer->m_ColorMap)
	{

		/*
		// Color map measure
		shared_ptr<TField> Field = m_Layer->m_Field;
		TMeasure::MEASURETYPE MeasureType = (m_Layer->m_ColorMapMeasure ? m_Layer->m_ColorMapMeasure->getMeasureType() : TMeasure::MEASURETYPE_NONE);
		TMeasure::MEASURETYPE MeasureType = (m_Layer->m_ColorMapMeasure ? m_Layer->m_ColorMapMeasure->getMeasureType() : TMeasure::MEASURETYPE_NONE);
		m_ColorMapMeasuresBox->fillForSourceType(Field->getFieldType(), MeasureType);
		*/

		// Color map type
		for(unsigned int i=0; i<m_Layer->m_AllColorMaps.size(); i++)
		{
			m_ColormapTypeBox->Append( m_Layer->m_AllColorMaps[i]->getName().c_str() );
			if( m_Layer->m_AllColorMaps[i].get() == m_Layer->m_ColorMap.get() )
			{
				m_ColormapTypeBox->SetSelection(i);
			}
		}

		m_Layer->m_ColorMap->DrawGui(m_ColormapPanel);
	}



	if(m_SelectionInfoTextCtrl)
	{
		m_SelectionInfoTextCtrl->Clear();

		// Update selection panel
		if(!g_Settings.m_ReleaseMode)
		{
			m_SelectionInfoTextCtrl->AppendText( wxString("Display name: ") + m_Layer->getDisplayName().c_str() + "\n");
			m_SelectionInfoTextCtrl->AppendText( wxString("Internal name: ") + m_Layer->getInternalName().c_str() + "\n");
		}
		
		if(m_Layer->m_Selection)
		{
			TCoord3 p = m_Layer->m_Selection->getSingleSelection();
			TFieldSelectionVisitor Visitor( m_SelectionInfoTextCtrl.get(), p );
			//m_SelectionInfoTextCtrl->AppendText( string(string("Layer: ") + m_Layer->m_Name + "\n").c_str() );
			m_SelectionInfoTextCtrl->AppendText( wxString::Format("Number of selected voxels: %i\n", m_Layer->m_Selection->getSize()) );

			if(m_Layer->m_Selection->getSize() > 0)
			{
				m_SelectionInfoTextCtrl->AppendText( 
					wxString::Format("Selected voxel: (%i,%i,%i)\n", 
						m_Layer->m_Selection->getSingleSelection().x, 
						m_Layer->m_Selection->getSingleSelection().y, 
						m_Layer->m_Selection->getSingleSelection().z
					) );

				if(!g_Settings.m_ReleaseMode)
				{
					if( m_Layer->m_Field->usingIndexField() )
					{
						shared_ptr<TIndexMapper> IndexField = m_Layer->m_Field->getIndexField();
						if(IndexField->vinside(p)) 
						{
							m_SelectionInfoTextCtrl->AppendText( 
								wxString::Format("Coord2idx: %i\n", 
									IndexField->vcoord2idx(p) 
								) );

							TCoord3 coord = IndexField->vidx2coord( IndexField->vcoord2idx(p) 
							);
							wxString::Format("Idx2Coord( Coord2idx ) : %i %i %i\n", coord.i, coord.j, coord.k
								 );
						}
					}			
				}
			}

			m_Layer->m_Field->accept( &Visitor );
		}
	}

	// Update layer info panel
	if(m_LayerInfoTextCtrl)
	{
		m_LayerInfoTextCtrl->Clear();
		m_LayerInfoTextCtrl->AppendText( wxString::Format("Dimensions: (%i,%i,%i)\n", m_Layer->m_Field->getMaxX(), m_Layer->m_Field->getMaxY(), m_Layer->m_Field->getMaxZ() )  );
	
		if( m_Layer->m_FilterMeasure)
		{
			m_LayerInfoTextCtrl->AppendText( wxString::Format("Filter measure\n\tMin value: %f\n\tMax value: %f\n"
				, m_Layer->m_FilterMeasure->getMinValue() 
				, m_Layer->m_FilterMeasure->getMaxValue() 
				) );
		}
	
		if( m_Layer->m_Field->usingIndexField() )
		{
			m_LayerInfoTextCtrl->AppendText( 
				wxString::Format("Indexfield index count: %i\n", 
					m_Layer->m_Field->getIndexField()->getMaxIndex() 
				) );

		}
	}


	m_VoxelSetPanel->DestroyChildren();



}	


void TVoxelSetListBox::OnMouse(wxMouseEvent & event)
{
	event.Skip();
	if( event.RightDown() )
	{
		m_SelectionMenu.reset( new TSelectionMenu() );
		PopupMenu( m_SelectionMenu.get(), event.GetPosition() );
	}
}	



void TLayerPanel::OnVoxelSetClick(wxCommandEvent& event)
{
	if(!m_Layer) return;

	// Draw GUI for selected voxel set
	{
		m_VoxelSetPanel->DestroyChildren();
		const int sel = m_VoxelSetListBox->GetSelection();
		if(sel > 0)
		{
			shared_ptr<TVoxelSet> VoxelSet = m_Layer->m_VoxelSets[sel];
			g_Mediator.getCurrentLayer()->m_CurrentVoxelSet = VoxelSet;
			VoxelSet->DrawGui(m_VoxelSetPanel);
		}
	}
}

void TLayerPanel::OnVoxelSetToggle(wxCommandEvent& event)
{
	if(!m_Layer) return;

	int sel = event.GetInt();
	if(sel >= 0 )
	{
		m_Layer->m_VoxelSets.at(sel)->setCheckedForRendering( m_VoxelSetListBox->IsChecked(sel) );
	}

	g_Mediator.redrawCanvas();
}

void TLayerPanel::addVoxelSet(shared_ptr<TVoxelSet> p_VoxelSet)
{
	if(!m_Layer) return;

	assert(p_VoxelSet);

	// Add to list
	m_Layer->m_VoxelSets.push_back(p_VoxelSet);
}



void TLayerPanel::OnUseFilterClick(wxCommandEvent& event)
{
	if(!m_Layer) return;
	g_Mediator.redrawCanvas();
}


void TLayerPanel::OnNewVoxelSetFromFilter(wxCommandEvent& event)
{
	if(!m_Layer) return;

	TField * f = m_Layer->m_Field.get();
	if(!m_Layer->m_Filter) throw string("No filter selected");
	shared_ptr<TVoxelSet> vs( TVoxelSubset::constructByFilter(f, m_Layer->m_Filter ) );
	vs->setName( string("Filter of ")+m_Layer->m_Name );
	this->addVoxelSet(vs);
	this->updateForLayer(m_Layer);
}



void TLayerPanel::OnFilterMeasureSelect(wxCommandEvent& event)
{
	assert(m_Layer);

	int sel = m_FilterMeasuresBox->GetSelection();
	if(sel == -1) return;

	TField * Field = m_Layer->m_Field.get();

	shared_ptr<TMeasure> Measure( m_FilterMeasuresBox->getCloneOfSelectedMeasure(Field) );
	m_Layer->m_FilterMeasure = Measure;
	m_Layer->m_Filter.reset( new TFloatFilter(&m_Layer->m_FilterMeasure) );
	m_Layer->onFieldChanged();


	m_Layer->setNeedRedraw();
	g_Mediator.redrawCanvas();
	updateForLayer(m_Layer);
}


void TLayerPanel::OnColorMapMeasureSelect(wxCommandEvent& event)
{
	int sel = m_ColorMapMeasuresBox->GetSelection();
	if(sel == -1) return;

//	m_Layer->m_ColorMapMeasure.reset(  m_ColorMapMeasuresBox->getCloneOfSelectedMeasure(m_Layer->m_Field.get()) );
	m_Layer->m_ColorMap->init();

	
	m_Layer->setNeedRedraw();
	g_Mediator.redrawCanvas();
	updateForLayer(m_Layer);

}

	

void TLayerPanel::OnColorMapTypeSelect(wxCommandEvent& event)
{
	if(!m_Layer) return;

	m_Layer->m_ColorMap = m_Layer->m_AllColorMaps[event.GetInt()];
	m_Layer->m_ColorMap->init();
	m_Layer->setNeedRedraw();
	g_Mediator.redrawCanvas();
	updateForLayer(m_Layer);
}




void TLayer::setNeedRedraw()
{ 
	for(int i=0; i<m_VoxelSets.size(); i++)
	{
		m_VoxelSets[i]->m_NeedRedraw = true;
	}
}


void TLayer::setSelectionFromFilter(const TFilter * p_Filter)
{
	m_Selection->clear();

	unsigned int x,y,z;
	for(x=0; x<m_Field->getMaxX(); x++)
		for(y=0; y<m_Field->getMaxY(); y++)
			for(z=0; z<m_Field->getMaxZ(); z++)
			{
				TCoord3 c(x,y,z);
				if( p_Filter->test(c) )
				{
					m_Selection->select(c);
				}
			}


}


void TVisibleVoxelSet::initTraversal() const
{
	if(m_Dirty) m_ObservingLayer->getVisibleVoxels( &m_VoxelPositions );
	m_Dirty = false;
	m_TraversalIdx = 0; 

}

void TLayer::registerObserver(TLayerObserver* o) const
{ 
	m_Observers.insert(o); 
}

void TLayer::unregisterObserver(TLayerObserver* o) const
{ 
	if( m_Observers.find(o) != m_Observers.end() )
	{
		m_Observers.erase( m_Observers.find(o) );
	} 
}
