#ifndef LAYER_H
#define LAYER_H


#include "stdafx.h"
#include "wx/wxprec.h"
#include "wx/notebook.h"
#include "coord.h"
#include "field.h"
#include "voxelset.h"

#include <set>
using std::set;

class TLayer
{
public:
	friend class TLayerPanel;

	// Constructor
	TLayer(shared_ptr<TField> p_Field, string p_Name, string p_ShortName = "");
	void init();
	virtual ~TLayer()
	{
	}

	class TLayerObserver
	{
	public:
		TLayerObserver()
		{
		};
		TLayerObserver(const class TLayer * p_Layer) 
			: m_ObservingLayer(p_Layer) 
		{
			if(m_ObservingLayer) m_ObservingLayer->registerObserver(this);
		};
		virtual ~TLayerObserver()
		{
		}
		virtual void notifyLayerChanged() = 0;
	protected:
		const class TLayer * m_ObservingLayer;
	};



	// Data
	vector<shared_ptr<TVoxelSet> >	m_VoxelSets;
	shared_ptr<TVoxelSet> m_CurrentVoxelSet;

	shared_ptr<class TFloatFilter>			m_Filter;
	shared_ptr<class TMeasure>		m_FilterMeasure;
	shared_ptr<class TColorMapUsingMeasure>	m_ColorMap;
	shared_ptr<class TField>			m_Field;
	shared_ptr<class TLayerHistogram>		m_Histogram;

	vector<shared_ptr<class TColorMapUsingMeasure> > m_AllColorMaps;

	shared_ptr<TVoxelSetSelection>			m_Selection;
	shared_ptr<TVoxelSet>				m_AllVoxels;

	void setCheckedForRendering(bool b) { m_CheckedForRendering = b; }
	bool getCheckedForRendering() const { return m_CheckedForRendering; }

	void setNeedRedraw();
	void setSelectionFromFilter(const TFilter * p_Filter);
	void onFieldChanged();
	void onLayerChanged();

	void getVisibleVoxels(vector<TCoord3> * p_VisibleVoxels) const;
	void registerObserver(TLayerObserver* o) const;
	void unregisterObserver(TLayerObserver* o) const;
	string m_Filename;
	void selectVisibleVoxels(const class TLayer * Layer);
	string getDisplayName() const { return m_Name; }
	string getInternalName() const { return m_ShortName; }

	void writeToVtk( const string & p_Filename );

	int							m_Id;
	string						m_Name;
	string						m_ShortName;
	int							m_RenderPrimitiveType;
	bool						m_RenderLines;
	bool						m_RenderSelectionIndicators;
	bool						m_RenderVoxelOutlines;
	unsigned int				m_Depth;

protected:
	bool						m_CheckedForRendering;
	static int					m_LastId;
	mutable set<TLayerObserver*>		m_Observers;
	TLayer() {};
};

class TSubLayer : public TLayer
{
public:
	TSubLayer(shared_ptr<TField> p_Field, string p_Name, string p_ShortName = "")
		: TLayer(p_Field,p_Name,p_ShortName)
	{
	}
	
	virtual string getDisplayName() const { return m_Name; }
	virtual string getName() const { return m_Name; }

protected:
	shared_ptr<TLayer> m_SuperLayer;
};


class TAbstractLayer : public TLayer
{
public:
	TAbstractLayer(shared_ptr<TField> p_Field, string p_Name = "", string p_ShortName = "");
};

class TLayerSet
{
public:
	TLayerSet(const string p_Name)
		: m_Name(p_Name)
	{
		setCheckedForRendering(true);
	};

	shared_ptr<TLayer> getLayer(const string & p_Name);

	TField * getField(const string & p_ShortName);
	TField * newField(TType::TYPE p_Type, shared_ptr<TIndexMapper> p_Mapper, const string p_ShortName = "");
	TField * newIoField(shared_ptr<TIndexMapper> p_Mapper, shared_ptr<TIndexMapper> p_IndexTo, const string p_ShortName = "");
	int TLayerSet::getIndex(const TLayer * p_Layer) const;
	void insertLayer(shared_ptr<TLayer> p_Layer);
	void insertLayerAfterLayer(shared_ptr<TLayer> p_Layer, TLayer * p_AfterLayer);
	void deleteLayer(const string & p_Name);
	void deleteLayer(int i);
	void deleteLayer(const TLayer * p_Layer);
	bool exists(const string & p_Name);
	void deselectAll();
	typedef vector<shared_ptr<TLayer> > TLayerList;
	TLayerList::iterator begin() { return m_VisibleList.begin(); }
	TLayerList::iterator end() { return m_VisibleList.end(); }
	int count() const { return m_VisibleList.size(); }
	shared_ptr<TLayer> first() { return m_VisibleList[0]; }
	shared_ptr<TLayer> getLayerById(int p_Id);
	shared_ptr<TLayer> getLayerByIndex(int p_Index);
	const string getName() const { return m_Name; }
	bool getCheckedForRendering() const { return m_CheckedForRendering; }
	void setCheckedForRendering(bool b) { m_CheckedForRendering = b; }

	shared_ptr<class TSkeletonizer> m_Skeletonizer;
	shared_ptr<TLayer> m_CurrentLayer;
	
protected:
	TLayerList m_VisibleList;
	TLayerList m_Id2Layer;
	string m_Name;
	bool m_CheckedForRendering;
};



class TFieldSelectionVisitor : public TFieldVisitor
{
public:
	TFieldSelectionVisitor(class wxTextCtrl * p_TextCtrl, TCoord3 & p_Coord)
		: m_Coord(p_Coord)
		, m_SelectionInfoTextCtrl(p_TextCtrl)
	{
	}

	virtual void visitFloatField(TTypedFieldInterface<float> * p_Field);
	virtual void visitUintegerField(TTypedFieldInterface<unsigned int> * p_Field);
	virtual void visitUnsignedCharField(TTypedFieldInterface<unsigned char> * p_Field);
	virtual void visitCoord3SetField(TTypedFieldInterface<vector<TCoord3>*> * p_Field);
	virtual void visitCoord3WithFloatSetField(TTypedFieldInterface<vector<pair<TCoord3,float> >*> * p_Field);
	virtual void visitIntegerField(TTypedFieldInterface<int> *  p_Field);
	virtual void visitIndexedOriginsField(TTypedFieldInterface<class TIndexedOrigins*> *  p_Field);
	virtual void visitIndexedOriginsMultimapField(TTypedFieldInterface<class TIndexedOrigins_Multimap*> *  p_Field);
	virtual void visitShortestPathSetField(TTypedFieldInterface<class TShortestPathSet*> *  p_Field);
	virtual void visitNodeField(TTypedFieldInterface<vector<class TNode*>*> *  p_Field);
	virtual void visitComponentSetField(TTypedFieldInterface<class TComponentSet*> *  p_Field);

	virtual void visitVector3Field(TTypedFieldInterface<TVector3> * p_Field);
	
protected:
	TCoord3 & m_Coord;
	class wxTextCtrl * m_SelectionInfoTextCtrl;
};




class TVoxelSetListBox : public wxCheckListBox
{
public:
	TVoxelSetListBox(TLayerPanel *p_LayerPanel, wxWindow *parent, wxWindowID id,
                 const wxPoint& pos = wxDefaultPosition,
                 const wxSize& size = wxDefaultSize,
                 int nStrings = 0,
                 const wxString choices[] = NULL,
                 long style = 0,
                 const wxValidator& validator = wxDefaultValidator,
                 const wxString& name = wxListBoxNameStr)
		: wxCheckListBox(parent, id, pos, size, nStrings, choices, style, validator, name )
		, m_LayerPanel( p_LayerPanel )
	{
	};

	void OnMouse(wxMouseEvent & event);
	
	TLayerPanel * m_LayerPanel;
	shared_ptr<class TSelectionMenu> m_SelectionMenu;
    DECLARE_EVENT_TABLE()
};


class TLayerPanel : public wxPanel
{
public:
    TLayerPanel(wxWindow* parent, wxWindowID id, const wxPoint& pos = wxDefaultPosition, const wxSize& size = wxDefaultSize, long style = wxTAB_TRAVERSAL);
	void updateForLayer(TLayer * p_Layer);
	void addVoxelSet(shared_ptr<TVoxelSet> p_VoxelSet);
	void select(TCoord3 & p_Coord);

	class TMeasuresBox *		m_ColorMapMeasuresBox;
	class TMeasuresBox *		m_FilterMeasuresBox;

	TLayer * m_Layer;
	wxPanel *			m_SelectionPanel;
	wxPanel *			m_FilterPanel;
	wxPanel *			m_ColormapPanel;
	wxPanel *			m_LayerInfoPanel;
	wxPanel *			m_FieldSpecificPanel;
	shared_ptr<wxTextCtrl>		m_LayerInfoTextCtrl;
	shared_ptr<wxTextCtrl>		m_SelectionInfoTextCtrl;

	wxNotebook * m_Notebook;
	wxCheckBox * m_UseVoxelSet;
	TVoxelSetListBox * m_VoxelSetListBox;
	wxPanel * m_VoxelSetPanel;
	wxCheckBox *			m_UseFilter;
//	TFilterFactoryPanel	*	m_FilterFactoryPanel;

	wxComboBox *		m_ColormapTypeBox;
//	wxComboBox *		m_FilterIdBox;

	// Methods
	void OnCreateOrigSetSelection(wxCommandEvent& event);
	void OnUseFilterClick(wxCommandEvent& event);
	void OnNewVoxelSetFromFilter(wxCommandEvent& event);
	void OnFilterMeasureSelect(wxCommandEvent& event);
	void OnColorMapMeasureSelect(wxCommandEvent& event);
	//void OnFilterIdSelect(wxCommandEvent& event);
	void OnColorMapTypeSelect(wxCommandEvent& event);
	void OnVoxelSetClick(wxCommandEvent& event);
	void OnVoxelSetToggle(wxCommandEvent& event);

    DECLARE_EVENT_TABLE()

};




class TVisibleVoxelSet : public TVoxelSet, public TLayer::TLayerObserver
{
public:
	TVisibleVoxelSet(const TField * p_Field, const TLayer * p_ObservingLayer) : TVoxelSet(p_Field), TLayerObserver(p_ObservingLayer), m_Dirty(true) {};
	virtual VOXELSETTYPE getVoxelSetType() const { return VOXELSETTYPE_VISIBLEVOXEL; }

	// Accessors
	virtual void initTraversal() const;
	virtual bool nextTraversal(TCoord3 & c) const 
	{
		if(m_TraversalIdx >= m_VoxelPositions.size()) return false;
		c = m_VoxelPositions[m_TraversalIdx];
		m_TraversalIdx++;
		return true;
	}

	// Methods
	virtual void DrawGui(wxWindow * p_Window) {};
	const TCoord3 * getFirstElement() const { return &m_VoxelPositions[0]; } 
	void notifyLayerChanged() { m_Dirty = true; m_NeedRedraw = true; }
		
protected:
	mutable bool m_Dirty;
	mutable int m_TraversalIdx;
	mutable vector<TCoord3> m_VoxelPositions;
};





#endif
