#include "lotufo.h"

#include "field.h"

/*

Implementation of Lotufo's algorithm. There are 3 important datastructures:

1) V : heap of all to be updated pixels, sorted on distance to origin
2) m_Origin : function of pixel -> origin
3) m_Pointers : function of pixel -> entry V

*/

void TLotufo::stageInit()			                        
{								
	m_Distance.reset( new TFloatField3( m_Image->getMaxX(), m_Image->getMaxY(), m_Image->getMaxZ()) );
	m_FlagField.reset( TFlagField3::construct(m_Image, m_Distance.get(), 0) );
	m_Origins.reset( new TOrigSetField2(m_FlagField->dimX(),m_FlagField->dimY()) );
	m_Pointers = new FIELD<std::multimap<float,TCoord2>::iterator>(m_FlagField->getMaxX(), m_FlagField->getMaxY() );

	m_Tolerance2 = (int)(m_Tolerance*m_Tolerance+0.5f); 
}

void TLotufo::stageMain()
{
	vector<TCoord2> Np;
	Np.resize(8);

	// Initialize V, m_Origins, and m_Pointers
	{
		for(int i=0; i<m_FlagField->getMaxX(); i++)
		{
			for(int j=0; j<m_FlagField->getMaxY(); j++)
			{
				if(m_FlagField->narrowband(i,j))
				{
					// Update m_Origins->value(i,j)
					TOrigSet2 & os = m_Origins->value(i,j);
					const TCoord2 c(i,j); 

					os.erase(os.begin(),os.end());
					os.insert( TOrigSet2::value_type(0,c) ); // Insert point itself
					m_Pointers->value(i,j) = V.insert( TTemporaryPixels::value_type(0,c) );

					getNeighbors(c, m_NeighborCount, Np);
					for(int k=0; k<Np.size(); k++)
					{
						const TCoord2 & n = Np[k];
						if( ! m_FlagField->narrowband(n.x,n.y) ) continue;
						const ARITHTYPE d = c.ARITHDISTANCE(n);
						if( d > ARITHTOLERANCE ) continue;
						os.insert( TOrigSet2::value_type(d, n) );
					}
				}
			}
		}
	}

	{
		for(int i=0; i<m_FlagField->getMaxX(); i++)
		{
			for(int j=0; j<m_FlagField->getMaxY(); j++)
			{
				if(m_FlagField->known(i,j))
				{
					//m_Origins->value(i,j).insert( TOrigSet2::value_type(0, TCoord2(i,j)) );
				}
				else if(m_FlagField->unknown(i,j))
				{
					m_FlagField->value(i,j) = TFlagField3::NARROW_BAND;
					m_Pointers->value(i,j) = V.insert( TTemporaryPixels::value_type( 10000, TCoord2(i,j)) );
				}
			}
		}
	}


	TOrigSet2::const_iterator it, lt;
	TOrigSet2::iterator jt;
			
	
	// Do iteration
	while(V.size() != 0)
	{
		const TCoord2 p = V.begin()->second;
		m_FlagField->value(p.x, p.y) = TFlagField3::KNOWN;
		V.erase(V.begin());

		// For each a in N^T(p)
		getNeighbors(p, m_NeighborCount, Np);
		for(int i=0; i<Np.size(); i++)
		{
			const TCoord2 & a = Np[i];
			if(! m_FlagField->narrowband(a.x, a.y) ) continue; // for each a \in N^T(p)
			// Note that FlagField doesn't contain U pixels 

			const TOrigSet2 & Sp = m_Origins->value(p.x, p.y);
			TOrigSet2 & Sa = m_Origins->value(a.x, a.y);

			// *** This code extends Lotufo's code to multiple origins:

			// Insert Sp into Sa
			for(it = Sp.begin(); it != Sp.end(); it++)
			{
				// Only insert 'it' if it does not occur in Sa
				const ARITHTYPE d = a.ARITHDISTANCE(it->second);
				bool found = false;
				for(lt = Sa.lower_bound(d); lt != Sa.end(); lt++)
				{
					if(lt->first > d) break;
					if(lt->second == it->second) { found = true; break; }
				}
				if(!found) Sa.insert( TOrigSet2::value_type(d,it->second) );
			}

			// Prune: remove all origins at larger distance than d_min + threshold
			const ARITHTYPE d_min = Sa.begin()->first;
#ifdef FLOAT_ARITHMETIC
			jt = Sa.upper_bound(d_min + m_Tolerance);
#else
			const float d_min_root = sqrtf(d_min);
			jt = Sa.upper_bound(d_min + 2*m_Tolerance*d_min_root+ m_Tolerance2);
#endif
			Sa.erase(jt, Sa.end());

			// Insert (possible) new distance for a into V
			V.erase( m_Pointers->value(a.x, a.y) );
			m_Pointers->value(a.x, a.y) = V.insert( TTemporaryPixels::value_type(d_min, a) );
		}


	} // end while narrowband not empty

	m_AlreadyPerformed = true;
}

void TLotufo::stageEnd()
{
	// Fill m_Distance
	{
		for(int i=0;i<m_Distance->dimX();i++)
			for(int j=0;j<m_Distance->dimY();j++)
			{
				m_Distance->value(i,j) = (m_Origins->value(i,j).size())? m_Origins->value(i,j).begin()->second.distance(TCoord2(i,j)) : -1; 
			}
	}

	delete m_Pointers;

}



string TLotufo::getLatexName() const
{
	string sTolerance = "?";
	if(m_Tolerance <= 0.01f) { sTolerance = "0"; }
	else if(fabs(m_Tolerance-1.0f) <= 0.01f) { sTolerance = "1"; }
	else if(fabs(m_Tolerance-sqrt(2.0f)/2.0f) <= 0.01f) { sTolerance = "\\frac{1}{2}\\sqrt{2}"; }
	else if(fabs(m_Tolerance-sqrt(2.0f) <= 0.01f)) { sTolerance = "\\sqrt{2}"; }

	return wxString::Format("\\gcdt{}%i $\\epsilon%s$", m_NeighborCount, sTolerance.c_str());
}

string TLotufo::getShortName() const
{
	string sTolerance = "?";
	if(m_Tolerance <= 0.01f) { sTolerance = "0"; }
	else if(fabs(m_Tolerance-1.0f) <= 0.01f) { sTolerance = "1"; }
	else if(fabs(m_Tolerance-sqrt(2.0f)/2.0f) <= 0.01f) { sTolerance = "0.71"; }
	else if(fabs(m_Tolerance-sqrt(2.0f) <= 0.01f)) { sTolerance = "1.41"; }

	string r = wxString::Format("GTFT%i e%s", m_NeighborCount, sTolerance.c_str());
	return r;
}

void TLotufo::deleteFields() 
{
	//m_FlagField.reset();
	//m_Distance.reset();
	//m_Origins.reset();
}
