#include "stdafx.h"

#include "main.h"

//#include <crtdbg.h> // For debugging
#include "luainterface.h"

#include "globals.h"
#include "stdafx.h"

#include "camera2.h"
#include "controls.h"
#include "abstractfilter.h"
#include "field.h"
#include "layer.h"
#include "colormap.h"
#include "utils.h"
#include "renderer.h"

#include "skeletonizer.h"
#include "parameters.h"
#include "settings.h"

#include "model.h"

#include "histogram.h"
#include "deltaomega.h"
#include "font.h"
#include "indexedorigins_cells.h"

#include "exactdt.h"

#include "povray.h"
#include "action.h"
#include "component.h"
#include "shortestpath.h"
#include "fieldadaptors.h"
#include "scriptedfield.h"

#include "wx/filename.h"
#include "wx/splitter.h"
#include "wx/apptrait.h"
#include "wx/evtloop.h"

#include "logwriter.h"
#include "ivcon.h"


#include "boundarysegmentation.h"
#include "boundarysegmentation_old.h"
#include "sskelsegmentation.h"
#include "volumesegmentation.h"
#include "scalardata.h"

#include "parallelskeletonizer.h"
#include "rootgraph.h"
#include "rootwindow.h"


BEGIN_EVENT_TABLE(MyApp, wxApp)
	EVT_IDLE(MyApp::OnIdle)
END_EVENT_TABLE()

static void addFTLayer(const ParallelSkeletonizer& skeletonizer)
{
	TTypedFieldInterface<TIndexedOrigins_Vector*>* ftField = static_cast<TTypedFieldInterface<TIndexedOrigins_Vector*>*>(
		g_Mediator.getCurrentLayerSet()->newIoField(const_pointer_cast<TIndexField3>(skeletonizer.GetIndexField()),
			const_pointer_cast<TSubIndexMapper>(skeletonizer.GetBoundaryField()), "Feature transform"));

	ftField->clear();
	for(unsigned int x = 0; x < skeletonizer.GetFeatureTransform()->size(); x++)
	{	
		if (skeletonizer.GetFeatureTransform()->at(x).size() == 0) continue;
		ftField->wvaluex(x) = new TIndexedOrigins_Vector(skeletonizer.GetFeatureTransform()->at(x));
	}
	ftField->getLayer()->onFieldChanged();
}

static void addEftLayer(const ParallelSkeletonizer& skeletonizer)
{
	TTypedFieldInterface<TIndexedOrigins_Vector*>* eftField = static_cast<TTypedFieldInterface<TIndexedOrigins_Vector*>*>(
		g_Mediator.getCurrentLayerSet()->newIoField(const_pointer_cast<TIndexField3>(skeletonizer.GetIndexField()),
			const_pointer_cast<TSubIndexMapper>(skeletonizer.GetBoundaryField()), "Extended FT"));

	eftField->clear();
	for(unsigned int x = 0; x < skeletonizer.GetExtendedFT()->size(); x++)
	{	
		if (skeletonizer.GetExtendedFT()->at(x).size() == 0) continue;
		eftField->wvaluex(x) = new TIndexedOrigins_Vector(skeletonizer.GetExtendedFT()->at(x));
	}
	eftField->getLayer()->onFieldChanged();
}

static void addImportanceLayer(const ParallelSkeletonizer& skeletonizer)
{
	shared_ptr<TSparseFloatField3> field(const_pointer_cast<TSparseFloatField3>(skeletonizer.GetImportanceMeasureField()));
	shared_ptr<TLayer> importanceLayer(new TLayer(field, "Importance measure"));

	importanceLayer->m_FilterMeasure.reset(new TFloatFieldMeasure_Sqrt(field.get()));
	importanceLayer->onFieldChanged();

	g_Mediator.addLayer(importanceLayer);
}

static void addAngleLayer(const ParallelSkeletonizer& skeletonizer)
{
	shared_ptr<TSparseFloatField3> field(const_pointer_cast<TSparseFloatField3>(skeletonizer.GetAngleField()));
	shared_ptr<TLayer> angleLayer(new TLayer(field, "Geodesic angles [1 - (cos(angle) + 1) / 2]"));

	g_Mediator.addLayer(angleLayer);
}

static void addLayers(const ParallelSkeletonizer& skeletonizer)
{
	shared_ptr<TLayer> inputLayer(new TLayer(const_pointer_cast<TUnsignedCharField3>(skeletonizer.GetInputField()),
		"Input volume"));
	g_Mediator.addLayer(inputLayer);

	shared_ptr<TLayer> indexLayer(new TLayer(const_pointer_cast<TIndexField3>(skeletonizer.GetIndexField()), "Indices"));
	indexLayer->m_Filter->m_UpperValue = skeletonizer.GetIndexField()->getMaxIndex();
	g_Mediator.addLayer(indexLayer);

	shared_ptr<TLayer> boundaryLayer(new TLayer(const_pointer_cast<TSubIndexMapper>(skeletonizer.GetBoundaryField()),
		"Boundary indices"));
	boundaryLayer->m_Filter->m_UpperValue = skeletonizer.GetBoundaryField()->getMaxIndex();
	g_Mediator.addLayer(boundaryLayer);

	addFTLayer(skeletonizer);
	addEftLayer(skeletonizer);

	shared_ptr<TLayer> shortestPathSetLayer(new TLayer(const_pointer_cast<TShortestPathSetField>(skeletonizer.GetShortestPathSetField()),
		"Shortest path sets"));
	g_Mediator.addLayer(shortestPathSetLayer);

	shared_ptr<TLayer> cSkelLayer(new TLayer(const_pointer_cast<TSubIndexMapper>(skeletonizer.GetCurveSkeletonIndexField()),
		"Curve skeleton indices"));
	cSkelLayer->m_Filter->m_UpperValue = skeletonizer.GetCurveSkeletonIndexField()->getMaxIndex();
	g_Mediator.addLayer(cSkelLayer);

	shared_ptr<TSparseUintField3> lengthField(const_pointer_cast<TSparseUintField3>(skeletonizer.GetLengthField()));
	g_Mediator.addLayer(shared_ptr<TLayer>(new TLayer(lengthField, "Geodesic lengths")));

	if (skeletonizer.ShouldComputeAngle())
		addAngleLayer(skeletonizer);

	if (skeletonizer.ShouldComputeImportance())
		addImportanceLayer(skeletonizer);
}

void MyApp::OnVisualizeSelected(wxCommandEvent&)
{
	RootWindow dialog(MyApp::m_Instance->m_TopWindow);

	if (dialog.ShowModal() == wxID_OK)
	{
		shared_ptr<TFloatField3> field(dynamic_cast<TFloatField3*>(
			TFieldFactory::constructFromFile(dialog.GetFileName())));

		if(field != nullptr)
		{
			g_Mediator.addLayerSet(dialog.GetFileName());
			g_Mediator.addLayer(shared_ptr<TLayer>(new TLayer(field, "Skeleton field")));

			shared_ptr<RootGraph> graph = RootGraph::FromField(field);
			g_Mediator.addLayer(shared_ptr<TLayer>(new TLayer(graph->ToField(), "Initial graph")));

			graph->RemoveSmallComponents(dialog.GetMinComponentSize());
			g_Mediator.addLayer(shared_ptr<TLayer>(new TLayer(graph->ToField(), "Small components removed")));

			JunctionEndPointDetector detector;
			shared_ptr<TLayer> layer(new TLayer(graph->ToField(&detector), "Junctions and end points"));
			g_Mediator.addLayer(layer);

			wxLogMessage("(%d) Nodes: %d; junctions: %d; end points: %d",
				layer->m_Id, graph->GetNodes().size(),
				detector.GetJunctions().size(), detector.GetEndPoints().size());

			if (dialog.ShouldRemoveJunctionConnectedEndPoints())
			{
				graph->RemoveJunctionConnectedEndpoints(&detector);

				detector = JunctionEndPointDetector(); // Reset
				layer.reset(new TLayer(graph->ToField(&detector), "Junction-connected end points removed"));
				g_Mediator.addLayer(layer);

				wxLogMessage("(%d) Nodes: %d; junctions: %d; end points: %d",
					layer->m_Id, graph->GetNodes().size(),
					detector.GetJunctions().size(), detector.GetEndPoints().size());
			}
		}
		else wxLogError("Skeleton field must contain float values");
	}
}

void MyApp::OnSkeletonizeSelected(wxCommandEvent&)
{
	SkeletonizeWindow dialog(MyApp::m_Instance->m_TopWindow);

	if (dialog.ShowModal() == wxID_OK)
	{
		string fileName = dialog.GetFileName();

		ParallelSkeletonizer skeletonizer(fileName, dialog.GetProperties());
		skeletonizer.Perform();

		g_Mediator.addLayerSet(fileName);
		addLayers(skeletonizer);
	}
}

MyApp * MyApp::m_Instance = 0;

bool MyApp::OnInit()
{
	const bool RELEASE = g_Settings.m_ReleaseMode;

//	_CrtSetDbgFlag ( _CRTDBG_ALLOC_MEM_DF | _CRTDBG_LEAK_CHECK_DF );
	// Use this to detect memory leaks:
//	_CrtSetBreakAlloc( 94903 );

	m_LuaLines = 0;
	m_ToolPanel = 0;
	MyApp::m_Instance = this;
	g_Log = this;
	m_RedrawCanvas = true;

	if(false && !g_Settings.m_ReleaseMode)
	{
		TActionMenu * Menu = new TActionMenu("Miscellaneous");
		Menu->m_Actions.push_back( new TAction_Reconstruction() );
		Menu->m_Actions.push_back( new TAction_ComputeSimplifiedEft() );
		Menu->m_Actions.push_back( new TAction_ReverseEft() );
		Menu->m_Actions.push_back( new TAction_ReverseEftAngle() );

		
		Menu->m_Actions.push_back( new TAction_InverseEftComplete() );
		Menu->m_Actions.push_back( new TAction_ReverseEftAngle2() );
		
		Menu->m_Actions.push_back( new TAction_DistanceToFeatureCollection() );
		Menu->m_Actions.push_back( new TAction_ExtendCollapseToLoops() );
		

		
		g_Mediator.addActionMenu( Menu );
	}

	
	if(false && !g_Settings.m_ReleaseMode)
	{
		TActionMenu * Menu = new TActionMenu("S-skel segm.");
		Menu->m_Actions.push_back( new TAction_SkeletonizeForSskelSegmentation() );
		Menu->m_Actions.push_back( new TAction_PruneSpSetByGeoFpDistanceForSskelSegmentation() );
		Menu->m_Actions.push_back( new TAction_ComputePathAndDebug() );
		Menu->m_Actions.push_back( new TAction_SskelSegmenterUsingIntCurvesConnectedComponents() );
		Menu->m_Actions.push_back( new TAction_Debug_SameIntCurves() );
		Menu->m_Actions.push_back( new TAction_SegmentSskel() );
		g_Mediator.addActionMenu( Menu );
	}

	if(false && !g_Settings.m_ReleaseMode)
	{
		TActionMenu * Menu = new TActionMenu("Part-type segm.");
		Menu->m_Actions.push_back( new TAction_VolumeSegmentationSkeletonize() );
		Menu->m_Actions.push_back( new TAction_SetSegmSigmaThreshold() );
		Menu->m_Actions.push_back( new TAction_SetRobustComponentTau() );
		Menu->m_Actions.push_back( new TAction_SetSegmRhoSimilar() );
		

		Menu->m_Actions.push_back( new TAction_NewJunctionDetection() );
		Menu->m_Actions.push_back( new TAction_CurveSkeletonFeatureAngle() );
		Menu->m_Actions.push_back( new TAction_MergeJunctions() );
		
//		Menu->m_Actions.push_back( new TAction_MoveCriticalPoints2() );
		Menu->m_Actions.push_back( new TAction_VolumeSegmentationFlat() );
		Menu->m_Actions.push_back( new TAction_VolumeSegmentationFlat2() );
		

		Menu->m_Actions.push_back( new TAction_CleanupCriticalPoints() );
		Menu->m_Actions.push_back( new TAction_MoveCriticalPoints() );
		
		Menu->m_Actions.push_back( new TAction_VolumeSegmentationMapToMesh() );
//		Menu->m_Actions.push_back( new TAction_VolumeSegmentationLigatures() );

		g_Mediator.addActionMenu( Menu );
	}

	if(false && !g_Settings.m_ReleaseMode)
	{
		TActionMenu * Menu = new TActionMenu("Patch-type segm.");
		Menu->m_Actions.push_back( new TAction_SkeletonizeForBoundarySegmentation() );
		Menu->m_Actions.push_back( new TAction_MapSskelSegmUsingFt() );
		Menu->m_Actions.push_back( new TAction_MapSskelSegmUsingFtAndGeodesics() );
		Menu->m_Actions.push_back( new TAction_DistanceToField() );
		Menu->m_Actions.push_back( new TAction_BoundaryNormalFromInverseEft() );
		Menu->m_Actions.push_back( new TAction_ComputeNormalsClamped() );
		Menu->m_Actions.push_back( new TAction_BoundaryNormalUsingForwardEftAndShortestPaths() );
		Menu->m_Actions.push_back( new TAction_NotUsingSskelSegmentation() );
		Menu->m_Actions.push_back( new TAction_FillUsingClusters() );
		Menu->m_Actions.push_back( new TAction_MergeClusters() );
		Menu->m_Actions.push_back( new TAction_NotUsingSskelSegmentation3() );
		g_Mediator.addActionMenu( Menu );
	}


	g_AllAdaptors.push_back( shared_ptr<TField>( new TShortestPathSetFieldAdaptor_Eft(0)) );
	g_AllAdaptors.push_back( shared_ptr<TField>( new TShortestPathSetFieldAdaptor_Length(0)) );
	g_AllAdaptors.push_back( shared_ptr<TField>( new TShortestPathSetFieldAdaptor_PathSet(0)) );
	g_AllAdaptors.push_back( shared_ptr<TField>( new TShortestPathSetFieldAdaptor_Midpoints(0)) );
	g_AllAdaptors.push_back( shared_ptr<TField>( new TShortestPathSetFieldAdaptor_FilteredPathSet(0)) );
	g_AllAdaptors.push_back( shared_ptr<TField>( new TShortestPathSetFieldAdaptor_Radius(0)) );
	g_AllAdaptors.push_back( shared_ptr<TField>( new TShortestPathSetFieldAdaptor_FpAngle(0)) );
	g_AllAdaptors.push_back( shared_ptr<TField>( new TShortestPathSetFieldAdaptor_MaxPathVoxelDistance(0)) );
	
	if(!g_Settings.m_ReleaseMode)
	{
		g_AllAdaptors.push_back( shared_ptr<TField>( new TFilteredVoxelsAdaptor(0)) );
	}



	m_SelectionRendererOrder.push_back(TType::TYPE_COORD3WITHFLOATSET);
	m_SelectionRendererOrder.push_back(TType::TYPE_COORD3SET);
	m_SelectionRendererOrder.push_back(TType::TYPE_ORIGSET);
	m_SelectionRendererOrder.push_back(TType::TYPE_GEODESICSET);
	m_SelectionRendererOrder.push_back(TType::TYPE_INDEXEDORIGINS);
	m_SelectionRendererOrder.push_back(TType::TYPE_INDEXEDORIGINSSORTED);
	m_SelectionRendererOrder.push_back(TType::TYPE_COORD3SETSORTED);
	m_SelectionRendererOrder.push_back(TType::TYPE_DELTAOMEGAFLOATMAP);

	m_SelectionRendererOrder.push_back(TType::TYPE_PATHFIELD);
	m_SelectionRendererOrder.push_back(TType::TYPE_VECTOR3);
	m_SelectionRendererOrder.push_back(TType::TYPE_VECTOR2);
	m_SelectionRendererOrder.push_back(TType::TYPE_VECTORSET);
	m_SelectionRendererOrder.push_back(TType::TYPE_INDEXEDORIGINS_MULTIMAP);
	m_SelectionRendererOrder.push_back(TType::TYPE_SHORTESTPATHSET);
	m_SelectionRendererOrder.push_back(TType::TYPE_COMPONENTSET);
	


	m_ObservingCamera = shared_ptr<TObservingCamera>(
		new TObservingCamera( 0, 0, 10 )
		);

	m_RoamingCamera = shared_ptr<TRoamingCamera>(
		new TRoamingCamera( TPoint3(5.0f, 5.0f, -15.0f), (float)(2.0*PI/4.0), 0.0f ) 
		);

	m_CurrentCamera = m_ObservingCamera;
	m_Picking = false;

	MyFrame *frame;
	wxScrolledWindow * m_RightPanel;
	{
		int x,y;

		/* Create the main frame window */
		frame = m_TopWindow = new MyFrame(nullptr, "Skeleton Sandbox", wxPoint(50, 50), wxSize(1024, 768));
		frame->Maximize();
		
		/* Make a menubar */
		m_MenuBar = new wxMenuBar;

		// File menu
		{
			wxMenu *menu = new wxMenu;

			menu->Append(ID_PARALLEL_SKELETONIZE, "Skeletonize");
			frame->Bind(wxEVT_COMMAND_MENU_SELECTED, &MyApp::OnSkeletonizeSelected, this, ID_PARALLEL_SKELETONIZE);

			menu->Append(ID_ROOT_VISUALIZATION, "Visualize skeleton");
			frame->Bind(wxEVT_COMMAND_MENU_SELECTED, &MyApp::OnVisualizeSelected, this, ID_ROOT_VISUALIZATION);

			wxImage::AddHandler(new wxPNGHandler());

			//menu->Append(ID_OPENFIELD, wxT("Open previously computed skeleton"));
			//menu->Append(ID_SAVEFIELD, wxT("Save skeleton"));
			//menu->AppendSeparator();
			//menu->Append(ID_OPENMODEL, wxT("Open mesh"));
			//menu->Append(ID_CREATEMODEL, wxT("Create mesh from voxel shape"));
			//if(!RELEASE) menu->Append(ID_CONVERTMODEL, wxT("Convert mesh"));
			menu->AppendSeparator();
			menu->Append(ID_SAVESCREENSHOT, wxT("Save screenshot to disk"));
			menu->Append(ID_COPYSCREENSHOTTOCLIPBOARD, wxT("Copy screenshot to clipboard"));
			//menu->Append(ID_RESET, wxT("Reset"));

			if(false && !g_Settings.m_ReleaseMode)
			{
				menu->AppendSeparator();
				menu->Append(ID_OPENSCALARDATA, wxT("Open scalar data in default layerset"));
				menu->Append(ID_SKELETONIZESCALARDATA, wxT("Skeletonize scalar data using filter settings"));
			}

			menu->AppendSeparator();
	
			menu->Append(ID_OPENCAMERA, wxT("Open camera"));
			menu->Append(ID_SAVECAMERA, wxT("Save camera"));

			menu->AppendSeparator();
			menu->Append(ID_EXPORTVTK, wxT("Export selected field"));

			
			if(false && !g_Settings.m_ReleaseMode)
			{
				menu->AppendSeparator();
				menu->Append(ID_EXPORTPOVRAYLAYERSET, wxT("Export selected layerset as povray"));
				menu->Append(ID_EXPORTPOVRAYSKELETON, wxT("Export skeleton as povray"));
				menu->Append(ID_EXPORTPOVRAYFIELD, wxT("Export selected field as povray"));
				
			}

			m_MenuBar->Append(menu, wxT("&File"));
		}

		// Camera menu
		{
			m_CameraMenu = new wxMenu;
			m_CameraMenu->AppendCheckItem(ID_OBSERVINGCAMERA, wxT("Virtual trackball camera"));
			m_CameraMenu->FindItem(ID_OBSERVINGCAMERA)->Check(m_CurrentCamera->getCameraType() == TCamera::CAMERATYPE_OBSERVING );
			m_CameraMenu->AppendCheckItem(ID_ROAMINGCAMERA, wxT("Free camera"));
			m_CameraMenu->FindItem(ID_ROAMINGCAMERA)->Check(m_CurrentCamera->getCameraType() == TCamera::CAMERATYPE_ROAMING );

			if(!g_Settings.m_ReleaseMode)
			{
				m_CameraMenu->AppendSeparator();
				m_CameraMenu->AppendCheckItem(ID_PERSPECTIVEPROJECTION, wxT("Perspective"));
				m_CameraMenu->FindItem(ID_PERSPECTIVEPROJECTION)->Check(m_CurrentCamera->getProjectionMode() == TCamera::PROJECTIONMODE_PERSPECTIVE);
				m_CameraMenu->AppendCheckItem(ID_ORTHOGONALPROJECTION, wxT("Orthogonal"));
				m_CameraMenu->FindItem(ID_ORTHOGONALPROJECTION)->Check(m_CurrentCamera->getProjectionMode() == TCamera::PROJECTIONMODE_ORTHOGONAL);
				m_CameraMenu->AppendSeparator();
				m_CameraMenu->AppendCheckItem(ID_LIGHTING, wxT("Lighting"));
				m_CameraMenu->FindItem(ID_LIGHTING)->Check(g_Settings.m_Lighting);
				m_CameraMenu->AppendCheckItem(ID_ALLOWPICKINGOFSELECTION, wxT("Allow picking of selection"));
				m_CameraMenu->FindItem(ID_ALLOWPICKINGOFSELECTION)->Check(g_Settings.m_AllowPickingOfSelection);
//				m_CameraMenu->AppendCheckItem(ID_DISPLAYCSHISTOGRAM, wxT("Display CS Histogram"));
//				m_CameraMenu->FindItem(ID_DISPLAYCSHISTOGRAM)->Check(g_Settings.m_DisplayCsHistogram);
//				m_CameraMenu->AppendCheckItem(ID_EDITOBJECTMODE, wxT("Edit object mode"));
//				m_CameraMenu->FindItem(ID_EDITOBJECTMODE)->Check(g_Settings.m_EditObjectMode);
			}

			if(!g_Settings.m_ReleaseMode)
			{
				m_CameraMenu->AppendSeparator();
				m_CameraMenu->AppendCheckItem(ID_ROTATECAMERA0, wxT("Plane XZ"));
				m_CameraMenu->FindItem(ID_ROTATECAMERA0)->Check(g_Settings.m_RotateCamera == 0);
				m_CameraMenu->AppendCheckItem(ID_ROTATECAMERA1, wxT("Plane XY"));
				m_CameraMenu->FindItem(ID_ROTATECAMERA1)->Check(g_Settings.m_RotateCamera == 1);
				m_CameraMenu->AppendCheckItem(ID_ROTATECAMERA2, wxT("Plane YZ"));
				m_CameraMenu->FindItem(ID_ROTATECAMERA2)->Check(g_Settings.m_RotateCamera == 2);
			}

			m_CameraMenu->AppendCheckItem(ID_USEBLENDINGFORMODEL, wxT("Use blending for mesh"));
			m_CameraMenu->FindItem(ID_USEBLENDINGFORMODEL)->Check(g_Settings.m_UseBlendingForModel);
			m_CameraMenu->AppendCheckItem(ID_FLATSHADING, wxT("Use flat shading for mesh"));
			m_CameraMenu->FindItem(ID_FLATSHADING)->Check(g_Settings.m_FlatShading);
			m_CameraMenu->AppendCheckItem(ID_RENDERMODELWITHOUTSHADING, wxT("No shading on mesh"));
			m_CameraMenu->FindItem(ID_RENDERMODELWITHOUTSHADING)->Check(g_Settings.m_RenderModelWithoutShading);
			

			if(!g_Settings.m_ReleaseMode)
			{
				m_CameraMenu->AppendSeparator();
				
				
				m_CameraMenu->AppendCheckItem(ID_RENDERCSSELECTIONCOMPONENTS, wxT("Render Component Set field components for selection"));
				m_CameraMenu->FindItem(ID_RENDERCSSELECTIONCOMPONENTS)->Check(g_Settings.m_RenderCsSelectionComponents);

				m_CameraMenu->AppendCheckItem(ID_RENDERFOG, wxT("Render fog"));
				m_CameraMenu->FindItem(ID_RENDERFOG)->Check(g_Settings.m_RenderFog);
			}
									


			m_MenuBar->Append(m_CameraMenu, wxT("&Options"));
		}

		/*
		// DT menu
		if(!g_Settings.m_ReleaseMode)
		{
			wxMenu * menu = new wxMenu;
			menu->Append(ID_DT1, wxT("TFT comparison"));
			menu->Append(ID_DT2, wxT("Exact TFT four tolerances"));
			m_MenuBar->Append(menu, wxT("TFT"));
		}
		*/

		// Skeletonization menu
		if (false)
		{
			wxMenu * menu = new wxMenu;
			menu->Append(ID_SKELETONIZE, wxT("Skeletonize"), "Skeletonize a shape using the current settings.");

			menu->AppendSeparator();

			menu->AppendCheckItem(ID_PARAMETER_COMPUTECSKEL, wxT("Compute C-skel"), "Whether the curve skeleton is computed." );
			menu->FindItem(ID_PARAMETER_COMPUTECSKEL)->Check(g_Parameters.m_ComputeCskel);
			
			menu->AppendCheckItem(ID_PARAMETER_ONLYCSKEL, wxT("Only C-skel, no S-skel (faster)"), "Computes only the curve skeleton, not the surface skeleton, which is faster." );
			menu->FindItem(ID_PARAMETER_ONLYCSKEL)->Check(g_Parameters.m_OnlyCskelIterative);

			menu->AppendCheckItem(ID_PARAMETER_COMPUTEIMPORTANCEFORCSKEL, wxT("Compute importance for C-skel"), "Whether the collapse/importance measure for the curve skeleton is computed." );
			menu->FindItem(ID_PARAMETER_COMPUTEIMPORTANCEFORCSKEL)->Check(g_Parameters.m_ComputeImportanceForCskel);

			if(!RELEASE)
			{	
				menu->AppendCheckItem(ID_PARAMETER_EQUALIZESKELETON, wxT("Equalize importance measure"), "Equalize the C-skel measure by taking the square root." );
				menu->FindItem(ID_PARAMETER_EQUALIZESKELETON)->Check(g_Parameters.m_Equalize);
			}

			if(!RELEASE)
			{
				menu->AppendCheckItem(ID_PARAMETER_INVERTOBJECT, wxT("Compute background S-skel too"), "Also compute the surface skeleton of the background: the shape's outside.");
				menu->FindItem(ID_PARAMETER_INVERTOBJECT)->Check(g_Parameters.m_ComputeBackgroundSskel);
			}
				
			menu->AppendCheckItem(ID_PARAMETER_KEEPSHORTESTPATHS, wxT("Keep shortest paths"), "Do not discard the shortest paths so that they can be inspected." );
			menu->FindItem(ID_PARAMETER_KEEPSHORTESTPATHS)->Check(g_Parameters.m_KeepShortestPaths);

			menu->AppendCheckItem(ID_PARAMETER_KEEPFT, wxT("Keep FT (non-extended)"), "Keep the original non-extended feature transform.");
			menu->FindItem(ID_PARAMETER_KEEPFT)->Check(g_Parameters.m_KeepFt);

			menu->AppendCheckItem(ID_PARAMETER_EXTENDCOLLAPSETOLOOPS, wxT("Extend collapse to loops"), "Extend collapse to loops");
			menu->FindItem(ID_PARAMETER_EXTENDCOLLAPSETOLOOPS)->Check(g_Parameters.m_ExtendCollapseToLoops);
			
			

			if(!g_Settings.m_ReleaseMode) 
			{
				//menu->AppendSeparator();
				//menu->AppendCheckItem(ID_PARAMETER_PROCESSPARTLY, wxT("Process object partly") );
				//menu->FindItem(ID_PARAMETER_PROCESSPARTLY)->Check(g_Parameters.m_ProcessObjectPartly);

				menu->AppendCheckItem(ID_PARAMETER_CACHE_GEODESICS, wxT("Cache geodesics (speed-up)") );
				menu->FindItem(ID_PARAMETER_CACHE_GEODESICS)->Check(g_Parameters.m_CacheGeodesics);

				menu->AppendCheckItem(ID_PARAMETER_USESPATIALSUBDIVISION, wxT("Use spatial subdivision (speed-up)") );
				menu->FindItem(ID_PARAMETER_USESPATIALSUBDIVISION)->Check(g_Parameters.m_OptimizeSpatialSubdivision);
			}
			

			if(!g_Settings.m_ReleaseMode) 
			{
				menu->AppendSeparator();
				menu->AppendCheckItem(ID_PARAMETER_SPTYPE_DIJKSTRA, wxT("Dijkstra shortest paths") );
				menu->AppendCheckItem(ID_PARAMETER_SPTYPE_ASTAR, wxT("A* shortest paths (speed-up)") );
				menu->AppendCheckItem(ID_PARAMETER_SPTYPE_ALL, wxT("All shortest paths") );
				menu->FindItem(ID_PARAMETER_SPTYPE_DIJKSTRA + g_Parameters.m_GeodesicType)->Check(true);
			}

			menu->AppendSeparator();
			menu->Append(ID_PARAMETER_DILATION_DISTANCE, "Set C-skel dilation distance", "Dilation distance for detecting C-skel voxels.");

			if(!RELEASE)
			{
				menu->Append(ID_PARAMETER_SEFT_THRESHOLD, "Set simplified EFT threshold", "This setting determines when voxels are detected as surface-skeleton voxels.");
			}

			if(!g_Settings.m_ReleaseMode) 
			{
				menu->Append(ID_PARAMETER_SHAPEINFLATION_DISTANCE, wxString::Format("Set shape inflation distance"), "The distance by which the input shape is inflated. Use this for disconnected voxel shapes polygonal soup models.");
			}

			m_SkeletonizerMenu = menu;
			m_MenuBar->Append(menu, wxT("&Skeletonization"));
		}
		
		// Plug-in actions
		if(true)
		{
			unsigned int FreeId = ID_ACTION_BEGIN;
			for(unsigned int x=0; x<g_Mediator.m_ActionMenus.size(); x++)
			{	
				TActionMenu * Menu = g_Mediator.m_ActionMenus[x];
				wxMenu * menu = new wxMenu();
				for(unsigned int y=0; y<Menu->m_Actions.size(); y++)
				{
					menu->Append(FreeId, Menu->m_Actions[y]->getScreenName().c_str(), Menu->m_Actions[y]->getHelpString().c_str());
					Menu->m_Actions[y]->m_Id = FreeId;
					FreeId++;
					if(FreeId >= ID_ACTION_END) throw string("Too many actions");
				}
				m_MenuBar->Append(menu, Menu->m_Name.c_str() );
			}
		}



		// Debug menu
		if(false && !g_Settings.m_ReleaseMode)
		{
			wxMenu * menu = new wxMenu;
			menu->Append(ID_DEBUGISLOOP, wxT("IsLoop"));
			menu->Append(ID_DEBUGISLOOP2, wxT("IsLoop2"));
			menu->Append(ID_DEBUGDILATE2, wxT("dilate2"));
			menu->Append(ID_DEBUGSMOOTHSP, wxT("smooth sp"));

			m_MenuBar->Append(menu, wxT("&Debug"));
		}


		// Help menu
		{
			wxMenu * menu = new wxMenu;
			menu->Append(ID_HELP2, wxT("Help"));
			menu->Append(ID_ABOUT2, wxT("About"));
			m_MenuBar->Append(menu, wxT("&Help"));
		}


		frame->SetMenuBar(m_MenuBar);

		// Splitter vertical
		wxSplitterWindow * SplitterVertical = new wxSplitterWindow(frame, -1, wxPoint(0, 0), wxDefaultSize, wxSP_3D);
		SplitterVertical->SetMinimumPaneSize(20);
		wxPanel * LeftPanel = new wxPanel(SplitterVertical, -1, wxDefaultPosition, wxDefaultSize);
		LeftPanel->SetBackgroundColour( wxColour(0, 0, 0) );
		m_RightPanel = new wxScrolledWindow(SplitterVertical, -1, wxPoint(0,0), wxDefaultSize);
		frame->GetClientSize(&x,&y);
		SplitterVertical->SplitVertically( LeftPanel, m_RightPanel, -300 );





		// Splitter horizontal
		wxSplitterWindow * SplitterHorizontal = new wxSplitterWindow(LeftPanel, -1, wxPoint(0, 0), wxSize(100, 100), wxSP_3D);
		SplitterHorizontal->SetMinimumPaneSize(20);

		int gl_attrib[20] = { 
			WX_GL_RGBA, 
			WX_GL_MIN_RED, 1, 
			WX_GL_MIN_GREEN, 1,
			WX_GL_MIN_BLUE, 1, 
			WX_GL_DEPTH_SIZE, 1,
			WX_GL_MIN_ACCUM_RED, 1, 
			WX_GL_MIN_ACCUM_GREEN, 1, 
			WX_GL_MIN_ACCUM_BLUE, 1, 
			WX_GL_DOUBLEBUFFER,	
			GL_NONE };

		frame->SetCanvas( 
			new TGLCanvas(
				SplitterHorizontal, 
				-1, 
				wxPoint(0, 0), 
				//wxDefaultSize, 
				wxSize(640,480),
				0L,
				"GL canvas",
				gl_attrib
				)
			);


		wxWindow * TopPanel = frame->GetCanvas();

		wxPanel * BottomPanel = new wxPanel(SplitterHorizontal, -1, wxPoint(0, 0), wxDefaultSize);
		//BottomPanel->SetBackgroundColour( wxColour(0, 255, 255) );
		wxNotebook * nb = new wxNotebook(BottomPanel, -1, wxPoint(0,0), wxSize(600,400) );

		// Sizer
		wxBoxSizer * SplitterSizer = new wxBoxSizer( wxVERTICAL );
		SplitterSizer->Add( SplitterHorizontal, 1, wxEXPAND | wxALL, 0 );
		LeftPanel->SetSizer( SplitterSizer );

//		wxNotebook * nb = new wxNotebook(BottomPanel, -1, wxPoint(0,0), wxDefaultSize );

		// Split horizontal
		SplitterHorizontal->SplitHorizontally( TopPanel, BottomPanel, -100 );
//		SplitterHorizontal->SplitHorizontally( TopPanel, nb, -200 );

		wxBoxSizer * BottomSizer = new wxBoxSizer( wxVERTICAL );
		BottomSizer->Add(nb, 1, wxALL | wxEXPAND, 0);
		BottomPanel->SetSizer( BottomSizer );

		// Layers 
		{
			wxPanel * LogPanel = new wxPanel(nb, -1, wxDefaultPosition, wxDefaultSize );
			wxBoxSizer * LogLinesSizer = new wxBoxSizer( wxVERTICAL );
			LogLinesSizer->Add(m_LogLines = new wxTextCtrl( LogPanel, ID_LOGLINES, "", wxPoint(5,5), wxSize(600,300), wxTE_READONLY | wxTE_MULTILINE ) , 1, wxALL | wxEXPAND, 0);
			LogPanel->SetSizer( LogLinesSizer );
			m_LogLines->SetMaxLength( 0 );
			nb->AddPage(LogPanel, "Log");
		}

		if(false && !RELEASE)
		{
			wxPanel * LuaPanel = new wxPanel(nb, -1, wxDefaultPosition, wxDefaultSize );
			wxBoxSizer * LuaLinesSizer = new wxBoxSizer( wxVERTICAL );
			LuaLinesSizer->Add(m_LuaLines = new wxTextCtrl( LuaPanel, ID_LUALINES, "", wxPoint(5,5), wxSize(600,300), wxTE_MULTILINE ), 1, wxALL | wxEXPAND, 0);
			LuaPanel->SetSizer( LuaLinesSizer );
			m_LuaLines->SetMaxLength( 0 );
			nb->AddPage(LuaPanel, "Lua script");
		}
	}

	// Sizer
	wxBoxSizer * RightSizer = new wxBoxSizer( wxVERTICAL );

	// ToolPanel
	{
		wxPanel * Panel = m_ToolPanel = new TToolPanel(m_RightPanel
			, -1
			);
		RightSizer->Add( Panel, 1, wxEXPAND );
	}

	// LayerSettingsPanel
	{
		wxPanel * Panel = m_LayerPanel = new TLayerPanel(m_RightPanel
			, -1
			, wxDefaultPosition
			, wxDefaultSize 
			);
		RightSizer->Add( Panel, 1, wxEXPAND  );
	}

	m_RightPanel->SetScrollbars(0, 10, 0, frame->GetSize().y/10);
	m_RightPanel->EnableScrolling(true,true);
	m_RightPanel->SetSizer( RightSizer );
	
	frame->CreateStatusBar(2);
	m_StatusBar = frame->GetStatusBar();
	InitGL();

	TGlobalRenderer::instance()->registerMe( this );

	// Show this frame
	frame->Show(TRUE);
	wxSize Size = frame->GetSize();
	frame->SetSize(Size.x+1, Size.y+1);
	
	if(m_LuaLines) 
	{
		Lua_DoStuff();
		g_Mediator.getLuaProgram()->load();
		m_LuaLines->AppendText( g_Mediator.getLuaProgram()->getString().c_str() );
	}

	updateUI();

	//wxLog::DisableTimestamp();
	wxLog::SetActiveTarget(new wxLogTextCtrl(MyApp::m_LogLines));

	return TRUE;
}


/*
shared_ptr<TLayer> MyApp::getCurrentLayer()
{
	if( !m_ToolPanel || m_ToolPanel->m_LayerListBox->GetSelection() < 0 )
	{
		return shared_ptr<TLayer>( static_cast<TLayer*>(0) );
	}
	else 
	{
		TLayerSet::TLayerList::iterator it = g_Mediator.getCurrentLayerSet()->begin();
		int c = 0;
		int sel = m_ToolPanel->m_LayerListBox->GetSelection();
		if(sel >= 0) return g_Mediator.getCurrentLayerSet()->getLayerByIndex(sel);
		return shared_ptr<TLayer>( static_cast<TLayer*>(0) );
	}
}
*/

void MyApp::setCurrentLayer(shared_ptr<TLayer> p_Layer)
{
	int Index = g_Mediator.getCurrentLayerSet()->getIndex(p_Layer.get());
	m_ToolPanel->m_LayerListBox->SetSelection(Index);
}


int MyApp::OnExit()
{
	return 0;
}


int MyApp::MainLoop()
{
    if( argc >= 2 )
	{
		try
		{
			const string Command = string(argv[1]);
			const string Filename = string(argv[2]);
			std::ifstream File(Filename.c_str());
			if( File.is_open() )
			{
				File.close();
			}
			else
			{	
				throw string("Could not open file");
				return 1;
			}

			if(Command == "cskel")
			{
				g_Parameters.m_ComputeCskel = true;
				g_Parameters.m_OnlyCskelIterative = true;
				g_Parameters.m_OnlyKeepShortestPathsOnCskel = true;
				g_Parameters.m_KeepShortestPaths = true;
				g_Parameters.m_ComputeBackgroundSskel = false;
				g_Parameters.m_ComputeImportanceForCskel = true;

				TSkeletonizer * skel = new TSkeletonizer( Filename.c_str() );
				skel->init( Filename.c_str() );
				skel->perform();
				wxString file = Filename.c_str();
				file.Replace(".vtk.gz",".cskel.gz"); 
				file.Replace(".vtk",".cskel.gz"); 
				file.Replace(".im.gz",".cskel.gz"); 
				file.Replace(".im",".cskel.gz"); 
//				std::cout << string("Writing to '").c_str() << file.c_str() << string("'\n");
				if( string(file.c_str()) == Filename) throw string("New file = old file");
				skel->writeToFile( file.ToStdString() );			
			}
			else if(Command == "screenshot")
			{
				g_Settings.m_RenderAxis = false;

				TSkeletonizer * skel = TSkeletonizer::readFromFile( Filename.c_str() );

				// Select C-skel measure for rendering
				shared_ptr<TLayer> Layer = g_Mediator.getCurrentLayerSet()->getLayer("C-skel measure");
				if(Layer) Layer->setCheckedForRendering(true);
		
				// Do segmentation
				{
					TAction_MoveCriticalPoints SegmAction;
					SegmAction.perform();

				}

				// Open camera
				skel->tryOpenCamera(Filename.c_str());

				// Create mesh
				/*
				try
				{
					TSubIndexMapper * Boundary = static_cast<TSubIndexMapper *>( g_Mediator.getCurrentLayerSet()->getField("indexfield boundary") );
					if(!Boundary) throw string("!Boundary");
					TSubIndexMapper * Foreground = static_cast<TSubIndexMapper *>( g_Mediator.getCurrentLayerSet()->getField("indexfield fg") );
					if(!Foreground) throw string("!Foreground");
					g_Mediator.getSkeletonizer().get()->m_Model.reset( new TModel(Boundary, Foreground
					, (g_Parameters.m_ComputeBackgroundSskel ? g_Parameters.m_InvertBorder : 1) 
					) );
				}
				catch(...)
				{
				}
				*/

				wxString BaseFileName = Filename.c_str();
				BaseFileName.Replace(".cskel.gz",""); 
				BaseFileName.Replace(".cskel",""); 
				BaseFileName.Replace(".skel.gz",""); 
				BaseFileName.Replace(".skel",""); 

				// Write segmentation
				{
					// Select / deselect					
					g_Mediator.getCurrentLayerSet()->deselectAll();
					shared_ptr<TLayer> Layer = g_Mediator.getCurrentLayerSet()->getLayer("Final segmentation");
					if(Layer) Layer->setCheckedForRendering(true);


					// Update 
					updateUI();
					Layer->setNeedRedraw();
					redrawCanvas();
					this->getGLCanvas()->OnPaint( wxPaintEvent() );
					redrawCanvas();

					this->saveScreenshot( string(BaseFileName.c_str()) + "_segm.png" );
					
					//TPovray povray( g_Mediator.getSkeletonizer().get() );
					//povray.writeLayerSet( string(BaseFileName.c_str()) + "_segm_povray.pov", g_Mediator.getCurrentLayerSet().get() );
				}				


				// Write skeleton
				{
					// Select / deselect					
					{
						g_Mediator.getCurrentLayerSet()->deselectAll();
						
						g_Mediator.getCurrentLayerSet()->getLayer( TAction_CurveSkeletonFeatureAngle::getStaticName() )->setCheckedForRendering(true);
						g_Mediator.getCurrentLayerSet()->getLayer( "Junctions" )->setCheckedForRendering(true);
						TTypedFieldInterface<unsigned char>* Cp = static_cast<TTypedFieldInterface<unsigned char>*>( g_Mediator.getCurrentLayerSet()->getLayer( "Final critical points" )->m_Field.get() );
						Cp->getLayer()->setCheckedForRendering(true);
						shared_ptr<TLayer> Pathset = g_Mediator.getCurrentLayerSet()->getLayer( "spset fg pathset" );
						Pathset->setCheckedForRendering(true);
						//shared_ptr<TLayer> Eft = g_Mediator.getCurrentLayerSet()->getLayer( "spset eft pathset" );
						//Pathset->setCheckedForRendering(true);

						Pathset->selectVisibleVoxels(Cp->getLayer());
						//Eft->selectVisibleVoxels(Cp->getLayer());
						
						// Select the final critical points
						//for(unsigned int x=0; x<Cp->getMaxIndex(); x++)
						//{
						//	skel->m_ForegroundSpSetField->getLayer()->select( Cp->getIndexField()->vidx2coord(x) );
						//}
					}

					TSubIndexMapper * Boundary = static_cast<TSubIndexMapper *>( g_Mediator.getCurrentLayerSet()->getField("indexfield boundary") );
					if(!Boundary) throw string("!Boundary");
					TSubIndexMapper * Foreground = static_cast<TSubIndexMapper *>( g_Mediator.getCurrentLayerSet()->getField("indexfield fg") );
					if(!Foreground) throw string("!Foreground");
					g_Mediator.getSkeletonizer().get()->m_Model.reset( TModel::createUsingMarchingCubes(Boundary, Foreground, (g_Parameters.m_ComputeBackgroundSskel ? g_Parameters.m_InvertBorder : 1) ) );

					// Update 
					updateUI();
					Layer->setNeedRedraw();
					redrawCanvas();
					this->getGLCanvas()->OnPaint( wxPaintEvent() );
					redrawCanvas();

					this->saveScreenshot( string(BaseFileName.c_str()) + "_skel.png" );
				}
			}
			else throw string("Unknown command: ") + Command;
		}
		catch(const string & s)
		{
			*g_Log << "\n" << "Exception: " << s.c_str() << "\n";
		}
		catch(const std::bad_alloc &)
		{
			*g_Log << "std::bad_alloc : out of memory\n";
		}
		catch(...)
		{
			*g_Log << "Unknown exception\n";
		}
		return 0;
		m_RedrawCanvas = true;
	}

	return wxApp::MainLoop();
}



void MyApp::OnIdle(wxIdleEvent& event)
{
	if(m_RedrawCanvas)
		this->getFrame()->GetCanvas()->Refresh();

	event.Skip();
}

IMPLEMENT_APP(MyApp)

BEGIN_EVENT_TABLE(MyFrame, wxFrame)
	EVT_MENU_OPEN(MyFrame::OnMenuOpen)
    EVT_MENU_RANGE(ID_MENUFIRST, ID_MENULAST, MyFrame::OnMenuChoice)
	EVT_LISTBOX(ID_LAYERLISTBOX, MyFrame::OnLayerClick)
	EVT_CHECKLISTBOX(ID_LAYERLISTBOX, MyFrame::OnLayerCheckListBox)
	EVT_LISTBOX(ID_LAYERSETLISTBOX, MyFrame::OnLayerSetClick)
	EVT_CHECKLISTBOX(ID_LAYERSETLISTBOX, MyFrame::OnLayerSetCheckListBox)
	EVT_BUTTON(ID_LOADFIELDBUTTON, MyFrame::OnLoadField)
	EVT_BUTTON(ID_SAVEFIELDBUTTON, MyFrame::OnSaveField)
	EVT_RADIOBUTTON(ID_ROAMINGCAMERA, MyFrame::OnCameraChoice)
	EVT_RADIOBUTTON(ID_OBSERVINGCAMERA, MyFrame::OnCameraChoice)
	EVT_COMBOBOX(ID_LOOKDOWNAXIS, MyFrame::OnLookDownAxis)
	EVT_BUTTON(ID_NEEDREDRAW2, MyFrame::OnNeedRedraw)
	EVT_BUTTON(ID_PARSESELECTION, MyFrame::OnParseSelection)
	EVT_TEXT(ID_LUALINES, MyFrame::OnLuaEnterText)
END_EVENT_TABLE()




/* My frame constructor */
MyFrame::MyFrame(wxFrame *frame, const wxString& title, const wxPoint& pos,
    const wxSize& size, long style):
  wxFrame(frame, -1, title, pos, size, style)
{
    m_canvas = NULL;
}

/* Intercept menu commands */
void MyFrame::OnMenuChoice(wxCommandEvent& event)
{
	MyApp::instance()->OnMenuChoice(event);
}

void MyFrame::OnMenuOpen(wxMenuEvent& event)
{
	MyApp::instance()->OnMenuOpen(event);
}

void MyFrame::OnLayerClick(wxCommandEvent& event)
{
	MyApp::instance()->OnLayerClick(event);
}

void MyFrame::OnLayerSetClick(wxCommandEvent& event)
{
	MyApp::instance()->OnLayerSetClick(event);
}

void MyFrame::OnLayerCheckListBox(wxCommandEvent& event)
{
	MyApp::instance()->OnLayerCheckListBox(event);
}

void MyFrame::OnLayerSetCheckListBox(wxCommandEvent& event)
{
	MyApp::instance()->OnLayerSetCheckListBox(event);
}

void MyFrame::OnLoadField(wxCommandEvent& event)
{
	MyApp::instance()->OnLoadField(event);
}

void MyFrame::OnSaveField(wxCommandEvent& event)
{
	MyApp::instance()->OnSaveField(event);
}

void MyFrame::OnCameraChoice(wxCommandEvent& event)
{
	MyApp::instance()->OnCameraChoice(event);
}

void MyFrame::OnLookDownAxis(wxCommandEvent& event)
{
	MyApp::instance()->OnLookDownAxis(event);
}

void MyFrame::OnNeedRedraw(wxCommandEvent& event)
{
	g_Mediator.getCurrentLayer()->onLayerChanged();
	MyApp::instance()->redrawCanvas();
}

void MyFrame::OnParseSelection(wxCommandEvent& event)
{
	string sel = MyApp::instance()->m_LogLines->GetStringSelection();
	int x = -1;
	int y = -1;
	int z = -1;
	int result = sscanf(sel.c_str(), "(%d,%d,%d)", &x, &y, &z );
	if( result == 3 )
	{
		shared_ptr<TLayer> layer = g_Mediator.getCurrentLayer();
		if(layer)
		{
			TCoord3 c(x,y,z);
			if(layer->m_Selection) layer->m_Selection->select( c );
			MyApp::instance()->redrawCanvas();
		}
	}
	else 
	{
		*g_Log << wxString::Format("Could not parse selection, result = %i\n", result).c_str();
	}
}


void MyFrame::OnKeyDown(wxKeyEvent&event)
{
	if(! g_Mediator.getCurrentLayer()) return;
	const TLayer * Layer = g_Mediator.getCurrentLayer().get();

	const int & key = event.GetKeyCode();

	TCoord3 sel;
	if(Layer->m_Selection)
	{
		sel = g_Mediator.getCurrentLayer()->m_Selection->getSingleSelection();
		if(key == 81 && sel.x != -1)	// q 
			MyApp::instance()->moveSelection(-1, 0,0);
		else if( (key == 65)&& sel.x != -1)	// a, z
			MyApp::instance()->moveSelection( 1, 0,0);
		else if(key == 87 && sel.x != -1)	// w
			MyApp::instance()->moveSelection( 0,-1,0);
		else if(key == 83 && sel.x != -1)	// s
			MyApp::instance()->moveSelection( 0, 1,0);
		else if(key == 69 && sel.x != -1)	// e
			MyApp::instance()->moveSelection( 0,0, 1);
		else if(key == 68 && sel.x != -1)	// d
			MyApp::instance()->moveSelection( 0,0,-1);
		else if(key > 65 && key < 140 /* arbitrary */)
		{
			*g_Log << "Unbound key pressed: " << event.GetKeyCode() << "\n";
		}
	}
}

void MyFrame::OnLuaEnterText(wxCommandEvent & event)
{
	g_Mediator.getLuaProgram()->setString( event.GetString().ToStdString() );
}

BEGIN_EVENT_TABLE(TGLCanvas, wxGLCanvas)
    EVT_SIZE(TGLCanvas::OnSize)
    EVT_PAINT(TGLCanvas::OnPaint)
    EVT_ERASE_BACKGROUND(TGLCanvas::OnEraseBackground)
    EVT_MOUSE_EVENTS(TGLCanvas::OnMouse)
	EVT_MOUSEWHEEL(TGLCanvas::OnMouse)
	EVT_KEY_DOWN(MyFrame::OnKeyDown)
END_EVENT_TABLE()

TGLCanvas::TGLCanvas(wxWindow *parent, wxWindowID id,
    const wxPoint& pos, const wxSize& size, long style, const wxString& name, int* attribList) :
  wxGLCanvas(parent, id, attribList, pos, size, style, name)
{
	m_Context = new wxGLContext(this);
	m_Context->SetCurrent(*this);

	GLenum error = glewInit();
	if (error != GLEW_OK)
		wxLogError("GLEW error: %s", (const char*)glewGetErrorString(error));
}

TGLCanvas::~TGLCanvas(void)
{
}

void TGLCanvas::OwnPaint()
{
	MyApp::instance()->OnDrawGL();
}

void TGLCanvas::OnPaint( wxPaintEvent& event )
{
    /* must always be here */
    wxPaintDC dc(this);

	try
	{
		m_Context->SetCurrent(*this);
		MyApp::instance()->OnDrawGL();
		SwapBuffers();
	}
	catch(string & s)
	{
		*g_Log << "\n" << "Exception: " << s.c_str() << "\n";
	}
	catch(std::bad_alloc &)
	{
		*g_Log << "std::bad_alloc : out of memory\n";
	}
}

void TGLCanvas::OnSize(wxSizeEvent& event)
{
    // set GL viewport (not called by wxGLCanvas::OnSize on all platforms...)
	int x,y;
	this->GetClientSize(&x,&y);
	glViewport(0, 0, x, y);
	glMatrixMode(GL_PROJECTION); 
	glLoadIdentity();
	MyApp::instance()->getCamera()->setProjectionMatrix(x,y);
	Refresh(FALSE);
}

void TGLCanvas::OnEraseBackground(wxEraseEvent& event)
{
    /* Do nothing, to avoid flashing on MSW */
}


void TGLCanvas::OnMouse( wxMouseEvent& event )
{
	if(MyApp::instance()->m_Picking) return;

	event.Skip();

	MyApp * App = MyApp::instance();
	float x = m_MouseX = event.GetX();
	float y = m_MouseY = event.GetY();
	float dx = (x-m_PreviousX); 
	float dy = (y-m_PreviousY);

	static bool Dragging = false;
	if(
		!event.ControlDown() 
		&& !event.ShiftDown() 
		&& !Dragging 
		&& event.RightUp()
		)
	{
		// Do picking
		vector<unsigned int> PickStack;
		MyApp::instance()->attemptPick(&event, PickStack);

		if( PickStack.size() == 0 )
		{
			m_RenderMenu.reset( new TRenderMenu() );
			PopupMenu( m_RenderMenu.get(), event.GetPosition() );
		}
		else
		{
			TGlobalRenderer::instance()->processPick(PickStack, &event);
		}
		Dragging = false;
	}
	else if(event.ControlDown() || event.ShiftDown())
	{
		if(event.LeftDown() || event.RightDown())
		{
			// Do picking
			vector<unsigned int> PickStack;
			MyApp::instance()->attemptPick(&event, PickStack);

			if( PickStack.size() == 0 ) // If nothing picked
			{
				// If nothing picked, do deselection
				if( event.RightDown() && event.ShiftDown() )
				{
					// Deselect all in current layer
					shared_ptr<TLayer> Layer = g_Mediator.getCurrentLayer();
					if(Layer)
					{
						if(Layer->m_Selection) 
						{
							Layer->m_Selection->getSingleSelection() = TCoord3(-1,-1,-1);
							Layer->m_Selection->clear();
						}
						MyApp::instance()->redrawCanvas();
					}
				}
				else if( event.RightDown() && event.ControlDown() )
				{
					// Deselect all in all layers
					for(TLayerSet::TLayerList::iterator it = g_Mediator.getCurrentLayerSet()->begin(); it != g_Mediator.getCurrentLayerSet()->end(); it++)
					{
						if((*it)->m_Selection)	
						{
							(*it)->m_Selection->getSingleSelection() = TCoord3(-1,-1,-1);
							(*it)->m_Selection->clear();
						}
					}
					MyApp::instance()->redrawCanvas();
				}

				// Update layer controls
				{
					TLayer * Layer = g_Mediator.getCurrentLayer().get();
					if(Layer) MyApp::instance()->m_LayerPanel->updateForLayer(Layer);
				}
			}
			else 
			{
				TGlobalRenderer::instance()->processPick(PickStack, &event);
				MyApp::instance()->redrawCanvas();
			}

			Dragging = false;
		}
	}
	else if(event.RightDClick())
	{
		m_RenderMenu.reset( new TRenderMenu() );
		PopupMenu( m_RenderMenu.get(), event.GetPosition() );
	}
	else if(event.AltDown())
	{
		if(event.LeftIsDown())
		{
			// Alternative to zoom
			float w = dx/100.0f;
			App->getCamera()->move(w,w,false,true,false);
			App->redrawCanvas();
		}
	}
	else if(event.GetWheelRotation() != 0)
	{
		float w = (event.GetWheelRotation()) * 0.01f;
		App->getCamera()->move(w,w,false,true,false);
		App->redrawCanvas();
		Dragging = false;
	}
	else if( (!Dragging && event.RightUp()) )
	{
		PopupMenu( new TRenderMenu(), event.GetPosition() );
		Dragging = false;
	}
	else if( event.LeftUp() || event.RightUp() )
	{
		Dragging = false;
	}
	else if(event.Dragging())
	{
		App->getCamera()->move( dx/100.0f, dy/100.0f, event.LeftIsDown(), event.MiddleIsDown(), event.RightIsDown() );
		App->redrawCanvas();
		Dragging = true;
	}

	MyApp::instance()->updateStatusBar();

    m_PreviousX = x;
    m_PreviousY = y;

}

void MyApp::OnMenuChoice(wxCommandEvent& event)
{
	string s;

	if(event.GetId() == ID_MENU_EXIT)
	{
		m_TopWindow->Destroy();
	}
	else if(event.GetId() == ID_OPENFIELD)
	{
		OnLoadField(event);
	}
	else if(event.GetId() == ID_OPENSCALARDATA)
	{
		wxFileDialog FileDialog(m_TopWindow, "Choose volume data file", "", "", "*.raw; *.*", wxFD_OPEN);
		if( FileDialog.ShowModal() == wxID_OK)
		{
			TScalarData scalardata( FileDialog.GetPath().ToStdString()  );
		}
	}
	else if(event.GetId() == ID_SKELETONIZESCALARDATA)
	{
		TTypedFieldInterface<float> * Field = static_cast<TTypedFieldInterface<float> *>( g_Mediator.getDefaultLayerSet()->getField("Scalar data") );
		if(!Field) throw string("!Field");
		TScalarData::processFilteredVolume( Field );
	}
	else if(event.GetId() == ID_SAVEFIELD)
	{
		OnSaveField(event);
	}
	else if(event.GetId() == ID_CONVERTMODEL)
	{	
		wxFileDialog FileDialogOpen(m_TopWindow, "Choose mesh file to open", "", "", "*.*", wxFD_OPEN);
		if( FileDialogOpen.ShowModal() == wxID_OK)
		{
			wxFileDialog FileDialogSave(m_TopWindow, "Choose mesh file to save", "", "", "*.*", wxFD_SAVE);
			if( FileDialogSave.ShowModal() == wxID_OK)
			{
				string from = FileDialogOpen.GetPath().c_str();
				string to = FileDialogSave.GetPath().c_str();

				*g_Log << "Converting mesh '" << from << "' to '" << to << "' \n";
				char * in = const_cast<char*>( from.c_str() );
				char * out = const_cast<char*>( to.c_str() );
				char * names[3] = {"", in, out };
				shared_ptr<TIvCon> ivcon( new TIvCon() );
				int error = ivcon->ivcon_main(2, names);
				if(error) 
				{
					throw string("Error while converting");
				}
			}
		}

	}
	else if(event.GetId() == ID_OPENMODEL)
	{	
		wxFileDialog FileDialog(m_TopWindow, "Choose mesh file to open", "", "", "*.obj; *.off; *.obj.gz; *.ply; *.ply.gz", wxFD_OPEN);
		if( FileDialog.ShowModal() == wxID_OK)
		{
			unsigned int x = g_Mediator.getCurrentLayerSet()->count() > 0 ? g_Mediator.getCurrentLayerSet()->first()->m_Field->getMaxX() : 100;
			*g_Log << "Loading mesh... ";
			if(!g_Mediator.getSkeletonizer().get()) throw string("Skeletonizer not loaded");
			
			g_Mediator.getSkeletonizer().get()->m_Model.reset( new TModel(FileDialog.GetPath().ToStdString()
				, x
				, (g_Parameters.m_ComputeBackgroundSskel ? g_Parameters.m_InvertBorder : 1)
				) );
			*g_Log << " done\n";
		}
	}
	else if(event.GetId() == ID_CREATEMODEL)
	{	
		*g_Log << "Creating mesh... ";
		TSubIndexMapper * Boundary = static_cast<TSubIndexMapper *>( g_Mediator.getCurrentLayerSet()->getField("indexfield boundary") );
		if(!Boundary) throw string("!Boundary");
		TSubIndexMapper * Foreground = static_cast<TSubIndexMapper *>( g_Mediator.getCurrentLayerSet()->getField("indexfield fg") );
		if(!Foreground) throw string("!Foreground");
		//g_Mediator.getSkeletonizer().get()->m_Model.reset( new TModel(Boundary, Foreground
		//		, (g_Parameters.m_ComputeBackgroundSskel ? g_Parameters.m_InvertBorder : 1) 
		//		) );
		g_Mediator.getSkeletonizer().get()->m_Model.reset(
			TModel::createUsingMarchingCubes(Boundary, Foreground, (g_Parameters.m_ComputeBackgroundSskel ? g_Parameters.m_InvertBorder : 1))
		);
		*g_Log << " done\n";

	}
	else if(event.GetId() == ID_SAVESCREENSHOT)
	{
		wxFileDialog FileDialogSave(m_TopWindow, "Choose file to save", "", "", "Portable Network Graphics|*.png", wxFD_SAVE);
		if( FileDialogSave.ShowModal() == wxID_OK)
		{
			string to = FileDialogSave.GetPath().c_str();
			saveScreenshot(to);
		}
	}
	else if(event.GetId() == ID_COPYSCREENSHOTTOCLIPBOARD)
	{
		copyScreenshotToClipboard();
	}
	
/*
	else if(event.GetId() == ID_SAVEMODEL)
	{	
		if(m_Layers.size() > 0)
		{
			wxFileDialog FileDialog(m_TopWindow, "Choose filename to save to", "", "", "*.obj; *.off", wxSAVE);
			if( FileDialog.ShowModal() == wxID_OK)
			{
				*g_Log << "Saving model... ";
				g_Mediator.getSkeletonizer().get()->m_Model->writeToObjFile();
				*g_Log << " done" << endl;
			}
		}
	}
*/
	else if(	event.GetId() == ID_EXPORTPOVRAYLAYERSET
			||  event.GetId() == ID_EXPORTPOVRAYFIELD
			||	event.GetId() == ID_EXPORTPOVRAYSKELETON 
			||	event.GetId() == ID_EXPORTPOVRAYRECONSTRUCTION 
			||  event.GetId() == ID_EXPORTCURVESKELETONANDJUNCTIONS
			)
	{
		if(g_Mediator.getSkeletonizer().get())
		{
			wxFileDialog FileDialog(m_TopWindow, "Choose filename", "", "", "*.pov", wxFD_SAVE);
			if( FileDialog.ShowModal() == wxID_OK)
			{
				*g_Log << "Exporting, ";
				TPovray povray( g_Mediator.getSkeletonizer().get() );

				if( event.GetId() == ID_EXPORTPOVRAYFIELD )
					povray.writeLayerSet( FileDialog.GetPath().ToStdString(), 0, g_Mediator.getCurrentLayer().get() );
				else if( event.GetId() == ID_EXPORTPOVRAYLAYERSET )
					povray.writeLayerSet( FileDialog.GetPath().ToStdString(), g_Mediator.getCurrentLayerSet().get() );
				else if( event.GetId() == ID_EXPORTPOVRAYSKELETON  )
					povray.writePovraySkeleton( FileDialog.GetPath().ToStdString(), g_Mediator.getCurrentLayer().get() );
				else if( event.GetId() == ID_EXPORTPOVRAYRECONSTRUCTION )
					povray.writePovrayReconstruction( FileDialog.GetPath().ToStdString(), g_Mediator.getCurrentLayer().get() );
				else if( event.GetId() == ID_EXPORTCURVESKELETONANDJUNCTIONS )
					povray.writePovrayCurveSkeletonAndJunctions( FileDialog.GetPath().ToStdString() );
					

				*g_Log << " done\n";
			}
		}
	}
	else if(event.GetId() == ID_EXPORTVTK)
	{
		if(/*g_Mediator.getSkeletonizer().get() &&*/ g_Mediator.getCurrentLayer())
		{
			wxFileDialog fileDialog(m_TopWindow, "Export selected volume", wxEmptyString, wxEmptyString, "*.vtk", wxFD_SAVE);
			if (fileDialog.ShowModal() == wxID_OK)
			{
				g_Mediator.getCurrentLayer()->writeToVtk(fileDialog.GetPath().ToStdString());
			}
		}
	}
	else if(event.GetId() == ID_OPENCAMERA)
	{
	 	string directory = "cameras";
		if(g_Mediator.getSkeletonizer().get())
		{
			directory = wxFileName(g_Mediator.getSkeletonizer().get()->m_InputFilename.c_str()).GetPath(wxPATH_GET_VOLUME).c_str();
		}
		
		wxFileDialog FileDialog(m_TopWindow, "Choose camera filename", directory.c_str(), "", "*.cam", wxFD_OPEN);
		if( FileDialog.ShowModal() == wxID_OK)
		{
			std::ifstream s( FileDialog.GetPath().ToStdString() );
			m_RoamingCamera.reset( TRoamingCamera::readFromStream(&s) );
			m_CurrentCamera = m_RoamingCamera;
			s.close();
		}
	}
	else if(event.GetId() == ID_SAVECAMERA)
	{
		if( m_CurrentCamera->getCameraType() != TCamera::CAMERATYPE_ROAMING ) throw string("Can only store free camera");

	 	string directory = "cameras";
		if(g_Mediator.getSkeletonizer().get())
		{
			directory = wxFileName(g_Mediator.getSkeletonizer().get()->m_InputFilename.c_str()).GetPath(wxPATH_GET_VOLUME).c_str();
		}

		wxFileDialog FileDialog(m_TopWindow, "Choose camera filename", directory.c_str(), "", "*.cam", wxFD_SAVE);
		if( FileDialog.ShowModal() == wxID_OK)
		{
			std::ofstream s( FileDialog.GetPath().ToStdString() );
			static_cast<TRoamingCamera*>(m_CurrentCamera.get())->writeToStream(&s);
			s.close();
		}
	}
	else if(event.GetId() == ID_CROP)
	{
		TField * Field = g_Mediator.getCurrentLayer()->m_Field.get();
		if(Field->getType() == TType::TYPE_UCHAR)
		{
			TUnsignedCharField3 * UCF = static_cast<TUnsignedCharField3*>(Field);
			shared_ptr<TField> newField(crop(UCF));
			addLayer(shared_ptr<TLayer>(new TLayer(newField, "cropped") ));
		}
	}
	else if(event.GetId() == ID_OBSERVINGCAMERA)
	{
		m_CurrentCamera = m_ObservingCamera;
		m_CameraMenu->FindItem(ID_OBSERVINGCAMERA)->Check(true);
		m_CameraMenu->FindItem(ID_ROAMINGCAMERA)->Check(false);
		redrawCanvas();
	}
	else if(event.GetId() == ID_ROAMINGCAMERA)
	{
		m_CurrentCamera = m_RoamingCamera;
		m_CameraMenu->FindItem(ID_OBSERVINGCAMERA)->Check(false);
		m_CameraMenu->FindItem(ID_ROAMINGCAMERA)->Check(true);
		redrawCanvas();
	}
	else if(event.GetId() == ID_PERSPECTIVEPROJECTION)
	{
		m_CurrentCamera->setProjectionMode(TCamera::PROJECTIONMODE_PERSPECTIVE);
		m_CameraMenu->FindItem(ID_PERSPECTIVEPROJECTION)->Check(true);
		m_CameraMenu->FindItem(ID_ORTHOGONALPROJECTION)->Check(false);
		redrawCanvas();
	}
	else if(event.GetId() == ID_ORTHOGONALPROJECTION)
	{
		m_CurrentCamera->setProjectionMode(TCamera::PROJECTIONMODE_ORTHOGONAL);
		m_CameraMenu->FindItem(ID_PERSPECTIVEPROJECTION)->Check(false);
		m_CameraMenu->FindItem(ID_ORTHOGONALPROJECTION)->Check(true);
		redrawCanvas();
	}
	else if(event.GetId() >= ID_ROTATECAMERA0 && event.GetId() <= ID_ROTATECAMERA2)
	{
		g_Settings.m_RotateCamera = event.GetId() - ID_ROTATECAMERA0;
		m_CameraMenu->FindItem(ID_ROTATECAMERA0)->Check(g_Settings.m_RotateCamera == 0);
		m_CameraMenu->FindItem(ID_ROTATECAMERA1)->Check(g_Settings.m_RotateCamera == 1);
		m_CameraMenu->FindItem(ID_ROTATECAMERA2)->Check(g_Settings.m_RotateCamera == 2);
		redrawCanvas();
	}
	else if(event.GetId() == ID_LIGHTING)
	{
		g_Settings.m_Lighting = event.IsChecked();
		redrawCanvas();
	}
	else if(event.GetId() == ID_RENDER2D)
	{
		g_Settings.m_Render2D = event.IsChecked();
		redrawCanvas();
	}
	else if(event.GetId() == ID_DISPLAYCSHISTOGRAM)
	{
		g_Settings.m_DisplayCsHistogram = event.IsChecked();
		g_Mediator.getCurrentLayer()->setNeedRedraw();
		redrawCanvas();
	}
	else if(event.GetId() == ID_USEBLENDINGFORMODEL)
	{
		g_Settings.m_UseBlendingForModel = event.IsChecked();
		if(g_Mediator.getCurrentLayer())
		{
			g_Mediator.getCurrentLayer()->setNeedRedraw();
		}
		redrawCanvas();
	}
	else if(event.GetId() == ID_FLATSHADING)
	{
		g_Settings.m_FlatShading = event.IsChecked();
		redrawCanvas();
	}
	else if(event.GetId() == ID_RENDERMODELWITHOUTSHADING)
	{
		g_Settings.m_RenderModelWithoutShading = event.IsChecked();
		redrawCanvas();
	}
	
	
	else if(event.GetId() == ID_RENDERFOG)
	{
		g_Settings.m_RenderFog = event.IsChecked();
		redrawCanvas();
	}
	
	else if(event.GetId() == ID_RENDERCSSELECTIONCOMPONENTS)
	{
		g_Settings.m_RenderCsSelectionComponents = event.IsChecked();
		redrawCanvas();
	}
	
	else if(event.GetId() == ID_ALLOWPICKINGOFSELECTION)
	{
		g_Settings.m_AllowPickingOfSelection = event.IsChecked();
	}
	else if(event.GetId() == ID_EDITOBJECTMODE)
	{
		g_Settings.m_EditObjectMode = event.IsChecked();
	}

	else if(event.GetId() == ID_PARAMETER_EQUALIZESKELETON)
	{
		g_Parameters.m_Equalize = event.IsChecked();
	}
	else if(event.GetId() == ID_PARAMETER_KEEPFT)
	{
		g_Parameters.m_KeepFt = event.IsChecked();
	}
	else if(event.GetId() == ID_PARAMETER_EXTENDCOLLAPSETOLOOPS)
	{
		g_Parameters.m_ExtendCollapseToLoops = event.IsChecked();
	}
	
	else if(event.GetId() == ID_PARAMETER_KEEPSHORTESTPATHS)
	{
		g_Parameters.m_KeepShortestPaths = event.IsChecked();
	}
	else if(event.GetId() == ID_PARAMETER_ONLYCSKEL)
	{
		g_Parameters.m_OnlyCskelIterative = event.IsChecked();
	}
	else if(event.GetId() == ID_PARAMETER_COMPUTEIMPORTANCEFORCSKEL)
	{
		g_Parameters.m_ComputeImportanceForCskel = event.IsChecked();
	}
	else if(event.GetId() == ID_PARAMETER_INVERTOBJECT)
	{
		g_Parameters.m_ComputeBackgroundSskel = event.IsChecked();
	}
	else if(event.GetId() == ID_PARAMETER_COMPUTECSKEL)
	{
		g_Parameters.m_ComputeCskel = event.IsChecked();
	}
	else if(event.GetId() == ID_PARAMETER_PROCESSPARTLY)
	{
		g_Parameters.m_ProcessObjectPartly = event.IsChecked();
	}
	else if(event.GetId() == ID_PARAMETER_CACHE_GEODESICS)
	{
		g_Parameters.m_CacheGeodesics = event.IsChecked();
	}
	else if(event.GetId() == ID_PARAMETER_USESPATIALSUBDIVISION)
	{
		g_Parameters.m_OptimizeSpatialSubdivision = event.IsChecked();
	}
	
	
	else if(event.GetId() == ID_PARAMETER_SEFT_THRESHOLD)
	{
		double Input = g_Parameters.m_SimplifiedEftThreshold;
		wxString s = "";
		do
		{
			s = ::wxGetTextFromUser("Set simplified EFT threshold", "Determines which rho_S is required for a voxel to be detected as a surface skeleton voxel.", wxString::Format("%f",Input).c_str() );
			if(s == "") return;
		} 
		while( !s.ToDouble(&Input) );
		g_Parameters.m_SimplifiedEftThreshold = Input;
	}
	else if(event.GetId() == ID_PARAMETER_DILATION_DISTANCE)
	{
		double Input = g_Parameters.m_DilationDistance;
		wxString s = "";
		do
		{
			s = ::wxGetTextFromUser("Dilation distance", "Dilation distance", wxString::Format("%f",Input).c_str() );
			if(s == "") return;
		} 
		while( !s.ToDouble(&Input) );
		g_Parameters.m_DilationDistance = Input;
	}
	else if(event.GetId() == ID_PARAMETER_SHAPEINFLATION_DISTANCE)
	{
		double Input = g_Settings.m_InflateInput;
		wxString s = "";
		do
		{
			s = ::wxGetTextFromUser("Inflation distance", "Inflation distance", wxString::Format("%f",Input).c_str() );
			if(s == "") return;
		} 
		while( !s.ToDouble(&Input) );
		g_Settings.m_InflateInput = Input;
	}
	
	else if(event.GetId() >= ID_PARAMETER_SPTYPE_DIJKSTRA && event.GetId() <= ID_PARAMETER_SPTYPE_ALL)
	{
		for(int i=ID_PARAMETER_SPTYPE_DIJKSTRA; i<=ID_PARAMETER_SPTYPE_ALL; i++)
		{
			m_SkeletonizerMenu->FindItem(i)->Check( i == event.GetId() );
		}
		g_Parameters.m_GeodesicType = (TParameters::TGeodesicType) (TParameters::GEODESICTYPE_DIJKSTRA + event.GetId() - ID_PARAMETER_SPTYPE_DIJKSTRA);
	}	
	else if(event.GetId() == ID_SKELETONIZE)
	{
		wxFileDialog FileDialog(m_TopWindow, "Choose VTK file to skeletonize", "", "", "*.vtk;*.vtk.gz;*.im;*.im.gz;*.scn;*.scn.gz", wxFD_OPEN);
		if( FileDialog.ShowModal() == wxID_OK)
		{
//			if(g_Mediator.getSkeletonizer().get()) throw string("error: g_Mediator.getSkeletonizer().get()");
			TSkeletonizer * skel = new TSkeletonizer( FileDialog.GetPath().ToStdString() );
			skel->init( FileDialog.GetPath().ToStdString() );
			skel->perform();

			updateUI();
		}
	}
	else if(
			event.GetId() == ID_DEBUGISLOOP
			|| event.GetId() == ID_DEBUGISLOOP2
			|| event.GetId() == ID_DEBUGDILATE2
			|| event.GetId() == ID_DEBUGSMOOTHSP
			)

	{
		if(!g_Mediator.getSkeletonizer().get()) throw string("!g_Mediator.getSkeletonizer().get()");
		if( !g_Mediator.getCurrentLayer() ) throw string("!getCurrentLayer()");

		if( ! g_Mediator.getSkeletonizer().get()->m_SpSetField) throw string("m_SpSetField not available");

		// Create debug fields
		TOmega * Omega = g_Mediator.getSkeletonizer().get()->Omega.get();
		TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();


		TTypedFieldInterface<unsigned char> * debug1 = static_cast<TTypedFieldInterface<unsigned char>*>(  g_Mediator.getCurrentLayerSet()->newField(TType::TYPE_UCHAR, skel->m_BoundaryIndexField, "debug1" ) );
		TTypedFieldInterface<float> * debug2 = static_cast<TTypedFieldInterface<float>*>(  g_Mediator.getCurrentLayerSet()->newField(TType::TYPE_FLOAT, skel->m_BoundaryIndexField, "debug2" ) );
		TTypedFieldInterface<unsigned char> * debug3 = static_cast<TTypedFieldInterface<unsigned char>*>(  g_Mediator.getCurrentLayerSet()->newField(TType::TYPE_UCHAR, skel->m_BoundaryIndexField, "debug3" ) );
		debug1->clear();
		debug2->clear();
		debug3->clear();

		g_Settings.m_DebugOutput = true;

		if(g_Mediator.getCurrentLayer()->m_Selection->getSize() == 0) throw string("No selection");

		const TVoxelSet * vs = g_Mediator.getCurrentLayer()->m_Selection.get();
		TCoord3 c;
		vs->initTraversal();
		while(vs->nextTraversal(c))
		{
			//TCoord3 c = *it;
			if( c.x == -1 ) throw string("No selection");


			
			if(event.GetId() == ID_DEBUGISLOOP)
			{
				int Components;

				const TShortestPathSet * spset = g_Mediator.getSkeletonizer().get()->m_SpSetField->vvaluep(c);
				if(spset == 0) throw string("No shortest path set found");
				const TIndexedOrigins_Vector * sps = spset->m_PathSet->castConstVector();
				if(sps == 0) throw string("No path set");


				TIndexedOrigins_Vector DilatedShortestPaths;
				g_Mediator.getSkeletonizer()->Omega->m_DeltaOmega->dilate(g_Parameters.m_DilationDistance, sps, &DilatedShortestPaths, false);
		
				unsigned int x;
				for(x=0; x<DilatedShortestPaths.size(); x++)
				{
					debug3->wvaluex(DilatedShortestPaths[x]) = 1;
				}
				// Debug
				/*
				TIndexedOrigins_Vector::const_iterator it;
				{
					for(it = DilatedShortestPaths.begin(); it != DilatedShortestPaths.end(); it++)
					{
						const TCoord3 d = g_Mediator.getSkeletonizer()->m_BoundaryIndexField->vidx2coord(*it);
						debug2->wvaluep(d) = Omega->m_DeltaOmega->m_AuxiliaryList[ *it ].m_Distance;
					}
					debug2->getLayer()->onFieldChanged();
				}
				*/
		
				vector<shared_ptr<TIndexedOrigins_Vector> > Boundaries;
				bool isloop = Omega->m_DeltaOmega->isLoop( 
					&DilatedShortestPaths
					, Components
					, &Boundaries
				);
				
				TComponentSet * cs = new TComponentSet(c);
				Omega->m_DeltaOmega->computeComponentSetUsingCells2(
					&DilatedShortestPaths, 
					cs
					);

				*g_Log << "m_Cells.size(): " << (unsigned int)cs->m_Cells.size() << "\n";

				// Print cellsizes
				{
					TComponentSet::TCellSizes::iterator it;
					TComponentSet::TCellSizes sizes = cs->getCellSizes();
					*g_Log << "Cellsizes: ";
					for(it = sizes.begin(); it != sizes.end(); it++)
					{
						*g_Log << it->first << ", ";
					}
					*g_Log << "\n";
				}

				float TotalSize = 0.0f;
				if(cs->m_Cells.size() > 2) // Note: cs->m_Cells[0] is empty
					TotalSize = skel->m_BoundaryIndexField->getMaxIndex() - cs->getCellSizes().rbegin()->first;

				*g_Log << "TotalSize: " << TotalSize << "\n";


				for(x=0; x<cs->m_Cells.size(); x++)
				{
					if(!cs->m_Cells[x]) continue;

					*g_Log << "Cell " << x << ", size: " << (unsigned int)cs->m_Cells[x]->vertexcount() << "\n";
					shared_ptr<TIndexedOrigins::TIterator> it( cs->m_Cells[x]->newIterator() );
					it->init();
					while(it->valid())
					{
						debug2->wvaluex( it->value() ) = x;
						it->next();
					}
				}
				
				// For debugging purposes:
				TIndexedOrigins_Vector::const_iterator it;
				for(x=1; x<Boundaries.size(); x++)
				{
					for(it = Boundaries[x]->begin(); it != Boundaries[x]->end(); it++)
					{
						const TCoord3 d = g_Mediator.getSkeletonizer()->m_BoundaryIndexField->vidx2coord(*it);
						debug1->wvaluep(d) = x;
					}
				}
				
				delete cs;
				*g_Log << "IsLoop: " << isloop << "\n";
			}
			
			if(event.GetId() == ID_DEBUGISLOOP2)
			{
				const TShortestPathSet * spset = g_Mediator.getSkeletonizer().get()->m_SpSetField->vvaluep(c);
				if(spset == 0) throw string("No shortest path set found");
				const TIndexedOrigins_Vector * sps = spset->m_PathSet->castConstVector();
				if(sps == 0) throw string("No path set");


				TIndexedOrigins_Vector DilatedShortestPaths;
				g_Mediator.getSkeletonizer()->Omega->m_DeltaOmega->dilate(g_Parameters.m_DilationDistance, sps, &DilatedShortestPaths, true);
		
				unsigned int x;
				for(x=0; x<DilatedShortestPaths.size(); x++)
				{
					debug3->wvaluex(DilatedShortestPaths[x]) = 1;
				}

				TComponentSet * cs = new TComponentSet(c);
				Omega->m_DeltaOmega->computeComponentSetUsingCells2(
					&DilatedShortestPaths, 
					cs
					);
				cs->simplify(c);
				
				for(x=0; x<cs->m_Borders.size(); x++)
				{
					if(!cs->m_Borders[x]) continue;

					for(unsigned int y=0; y<cs->m_Borders[x]->size(); y++)
					{
						debug1->wvaluex( cs->m_Borders[x]->at(y) ) = x+1;
					}
				}
				delete cs;
			}
			if(event.GetId() == ID_DEBUGDILATE2)
			{
				TShortestPathSet * sps = g_Mediator.getSkeletonizer().get()->m_SpSetField->vvaluep(c);
				if(sps == 0) throw string("No shortest path set found");
				
				TIndexedOrigins_Vector Output;
//				g_Mediator.getSkeletonizer()->Omega->m_DeltaOmega->dilateUntilTwoComponents( 
//					4.0f
//					, sps->m_PathSet.get()
//					, &Output
//					, true
//				);
				for(unsigned int x=0; x<Output.size(); x++)
				{
					debug1->wvaluex(Output[x]) = 1;
				}
				//*g_Log << "Isloop: " << isloop << "\n";
			}

			if(event.GetId() == ID_DEBUGSMOOTHSP)
			{
				TShortestPathSet * sps = g_Mediator.getSkeletonizer().get()->m_SpSetField->vvaluep(c);
				if(sps == 0) throw string("No shortest path set found");

				TIndexedOrigins_Vector Output1;
				g_Mediator.getSkeletonizer()->Omega->m_DeltaOmega->dilate(
					g_Parameters.m_DilationDistance
					, sps->m_PathSet.get()
					, &Output1
					, false
				);

				TIndexedOrigins_Vector Output2;
				g_Mediator.getSkeletonizer()->Omega->m_DeltaOmega->erode(&Output1, &Output2);

				for(unsigned int x=0; x<Output2.size(); x++)
				{
					debug1->wvaluex(Output2[x]) = 1;
				}
				
			}

		} // end while

		if(debug1) debug1->getLayer()->onFieldChanged();
		if(debug2) debug2->getLayer()->onFieldChanged();
		if(debug3) debug3->getLayer()->onFieldChanged();
		g_Settings.m_DebugOutput = false;
	}
	
	else if(event.GetId() >= ID_ACTION_BEGIN && event.GetId() <= ID_ACTION_END)
	{
		for(unsigned int x=0; x<g_Mediator.m_ActionMenus.size(); x++)
		{	
			TActionMenu * Menu = g_Mediator.m_ActionMenus[x];
			for(unsigned int y=0; y<Menu->m_Actions.size(); y++)
			{
				if(Menu->m_Actions[y]->m_Id == event.GetId())
				{
					Menu->m_Actions[y]->perform();
					break;
				}
			}
		}
	}

	else if(event.GetId() == ID_HELP2)
	{
		string text = "";
		text += "Usage";
		text += "\n-----------------------";
		text += "\n1) Set options in Skeletonization menu.";
		text += "\n2) Choose Skeletonization -> Skeletonize to compute the skeleton of a voxel dataset with current options. This can take up to a few minutes.";
		text += "\n3) Inspect result by making layers visible (right) and navigating with mouse in 3D view (left).";

		text += "\n\nNavigation";
		text += "\n-----------------------";
		text += "\nUse Options menu to select either Virtual trackball or Free camera mode.";
		text += "\nLeft mousebutton: rotate. Middle mousebutton/scrollwheel: zoom. Right mousebutton: pan (free camera only).";
				
		text += "\n\nSelecting voxels";
		text += "\n-----------------------";
		text += "\nVoxel selection in all layers: hold Ctrl and left-click voxel(s).";
		text += "\nVoxel deselection in all layers: hold Ctrl and right-click voxel(s).";
		text += "\nDeselect all: hold Ctrl and right-click on white background.";
		text += "\nHold Shift instead of Ctrl to (de)select in current layer only.";

		wxMessageDialog dialog(0, text.c_str(), "Help", wxOK);
		dialog.ShowModal();
	}
	else if(event.GetId() == ID_ABOUT2)
	{
		wxMessageDialog dialog(0, "Skeleton Sandbox\nComputing multiscale curve and surface skeletons of binary voxel shapes\n 2008 Dennie Reniers, d.reniers@tue.nl\nhttp://www.win.tue.nl/~dreniers\n\nModified by Christiaan Arnoldus", "About", wxOK);
		dialog.ShowModal();
	}

}


void MyApp::OnMenuOpen(wxMenuEvent& event)
{
	for(unsigned int x=0; x<g_Mediator.m_ActionMenus.size(); x++)
	{	
		TActionMenu * Menu = g_Mediator.m_ActionMenus[x];
		for(unsigned int y=0; y<Menu->m_Actions.size(); y++)
		{
			TAction * Action = Menu->m_Actions[y];
			if(m_MenuBar->FindItem( Action->m_Id  ))
			{
				m_MenuBar->FindItem( Action->m_Id  )->Enable( Action->isAvailable() );
			}
		}
	}

}


void MyApp::InitGL()
{
    glEnable(GL_DEPTH_TEST);
	glEnable(GL_CULL_FACE); 
	glEnable(GL_NORMALIZE);

	m_CubeDisplayList = glGenLists(1);
	glNewList(m_CubeDisplayList,GL_COMPILE);
		glBegin(GL_QUADS);
		glNormal3f( 0.0F, 0.0F, 1.0F);
		glVertex3f( 0.5F, 0.5F, 0.5F); glVertex3f(-0.5F, 0.5F, 0.5F);
		glVertex3f(-0.5F,-0.5F, 0.5F); glVertex3f( 0.5F,-0.5F, 0.5F);

		glNormal3f( 0.0F, 0.0F,-1.0F);
		glVertex3f(-0.5F,-0.5F,-0.5F); glVertex3f(-0.5F, 0.5F,-0.5F);
		glVertex3f( 0.5F, 0.5F,-0.5F); glVertex3f( 0.5F,-0.5F,-0.5F);

		glNormal3f( 0.0F, 1.0F, 0.0F);
		glVertex3f( 0.5F, 0.5F, 0.5F); glVertex3f( 0.5F, 0.5F,-0.5F);
		glVertex3f(-0.5F, 0.5F,-0.5F); glVertex3f(-0.5F, 0.5F, 0.5F);

		glNormal3f( 0.0F,-1.0F, 0.0F);
		glVertex3f(-0.5F,-0.5F,-0.5F); glVertex3f( 0.5F,-0.5F,-0.5F);
		glVertex3f( 0.5F,-0.5F, 0.5F); glVertex3f(-0.5F,-0.5F, 0.5F);

		glNormal3f( 1.0F, 0.0F, 0.0F);
		glVertex3f( 0.5F, 0.5F, 0.5F); glVertex3f( 0.5F,-0.5F, 0.5F);
		glVertex3f( 0.5F,-0.5F,-0.5F); glVertex3f( 0.5F, 0.5F,-0.5F);

		glNormal3f(-1.0F, 0.0F, 0.0F);
		glVertex3f(-0.5F,-0.5F,-0.5F); glVertex3f(-0.5F,-0.5F, 0.5F);
		glVertex3f(-0.5F, 0.5F, 0.5F); glVertex3f(-0.5F, 0.5F,-0.5F);
		glEnd();		
	glEndList();

}



void MyApp::drawSelection(TLayer * p_Layer)
{
	if(!p_Layer->m_Selection) return;

	const TField * f = p_Layer->m_Field.get();
	
	TRenderer Renderer(f);
	Renderer.begin();

	glEnable(GL_COLOR_MATERIAL);

	if( p_Layer->m_RenderPrimitiveType == 1 ) 
	{
		glColor3f(1.0f, 1.0f, 1.0f);
	}
	else if( p_Layer->m_RenderPrimitiveType == 2 ) 
	{
		glColor3f(1.0f, 1.0f, 1.0f);
		glTranslatef(0.0f, 0.0f, -0.01f);
	}
	else 
	{
		glColor4f(0.5f, 0.5f, 0.5f, 0.0f);
		glPolygonOffset(0, -2);
		glEnable(GL_POLYGON_OFFSET_FILL);
		glLineWidth(2.0f);
		glPolygonMode(GL_FRONT_AND_BACK,GL_LINE);
		
		if(p_Layer->m_RenderVoxelOutlines)
		{	
			// What to do?
		}
	}

	if(p_Layer->m_Selection->getSingleSelection().x != -1)
	{
		Renderer.renderVoxel(p_Layer->m_Selection->getSingleSelection());
	}


	TCoord3 c;
	p_Layer->m_Selection->initTraversal();
	while(p_Layer->m_Selection->nextTraversal(c))
	{
		Renderer.renderVoxel(c);
	}


	glPolygonMode(GL_FRONT_AND_BACK,GL_FILL);
	glDisable(GL_POLYGON_OFFSET_FILL);
	Renderer.end();
}

void MyApp::drawBegin()
{
	// Do not set projection matrix here!
	// When picking, the projection matrix should be different.

	// Clear
	glClearColor( 1.0, 1.0, 1.0, 1 );
	glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

	// Camera
	glMatrixMode(GL_MODELVIEW);  
	glLoadIdentity();
//		m_CurrentCamera->setMatrices();

//		glPushMatrix();
//		glTranslatef(0, 0, -2.0f); 
//		glutSolidSphere(0.1f,6,6);
//		glPopMatrix();


	if(g_Settings.m_Lighting)
	{
		GLfloat light0_pos[4]   = { 0.5f, 0.2f, 0.7f, 1.0f };
		GLfloat light0_color[4] = { 0.7f, 0.7f, 0.7f, 1.0f }; 
		GLfloat light0_ambient[4] = { 0.2f, 0.2f, 0.2f, 1.0f };
		GLfloat black[4] = { 0.0f, 0.0f, 0.0f, 0.0f };
		
		// /*
		glLightfv(GL_LIGHT0, GL_POSITION, light0_pos);
		glLightfv(GL_LIGHT0, GL_DIFFUSE,  light0_color);  
		glLightfv(GL_LIGHT0, GL_AMBIENT, light0_ambient);
		glLightfv(GL_LIGHT0, GL_SPECULAR, black);
		glEnable(GL_LIGHT0);
		// */

		 /*
		GLfloat light1_pos[4]   = { -0.5f, -0.2f, -0.7f, 0.0f };
		glLightfv(GL_LIGHT1, GL_POSITION, light1_pos);
		glLightfv(GL_LIGHT1, GL_DIFFUSE,  light0_color);  
		glLightfv(GL_LIGHT1, GL_AMBIENT, light0_ambient);
		glLightfv(GL_LIGHT1, GL_SPECULAR, black);
		glEnable(GL_LIGHT1);
		// */

		glEnable(GL_LIGHTING);
		glShadeModel(GL_SMOOTH);
	}
	else
	{
		glDisable(GL_LIGHTING);
		glDisable(GL_LIGHT0);
	}

	// Setup lights
	m_CurrentCamera->setMatrices();

	if(g_Mediator.getCurrentLayerSet() && g_Mediator.getCurrentLayerSet()->count() > 0)
	{
		TLayer * Layer = g_Mediator.getCurrentLayerSet()->first().get();
		m_ObservingCamera->setMiddle( TPoint3(
			(float) Layer->m_Field->getMaxX() * g_Settings.m_VoxelScale / 2.0f 
			, (float) Layer->m_Field->getMaxY() * g_Settings.m_VoxelScale / 2.0f
			, (float) Layer->m_Field->getMaxZ() * g_Settings.m_VoxelScale / 2.0f
			));
	}

	
	if(g_Settings.m_RotateCamera == 0)
	{
	}
	else if(g_Settings.m_RotateCamera == 1)
	{
		glRotatef(90.0f, 1.0f, 0.0f, 0.0f);
	}
	else if(g_Settings.m_RotateCamera == 2)
	{
		glRotatef(-90.0f, 0.0f, 0.0f, 1.0f);
	}

}

void MyApp::drawEnd()
{
	glFinish();
}

void MyApp::drawVoxelSets(vector<TVoxelSet*> & p_VoxelSets, TVoxelSet::VOXELSETTYPE p_Type)
{

	for(int i=0; i<p_VoxelSets.size(); i++)
	{
		TVoxelSet * vs = p_VoxelSets[i];
		if(vs->getVoxelSetType() == p_Type)
		{
			const TField * Field = vs->getField();
			TLayer * Layer = Field->getLayer();
			
			TColorMapUsingMeasure * ColorMap = Field->getLayer()->m_ColorMap.get();

			try
			{
				drawVoxelSet(
					Layer,
					vs,
					Field,
					Layer->m_Filter.get(),
					ColorMap
					);
			}
			catch(string & s)
			{
				vs->setCheckedForRendering(false);
				Layer->setCheckedForRendering(false);
				throw string("Exception: layer rendering disabled: ") + s;
			}
		}
	}
}



void MyApp::drawVoxelSetsInCorrectOrder(vector<TVoxelSet*> & p_VoxelSets)
{
	if(!g_Settings.m_Render2D)
	{
		drawVoxelSets(p_VoxelSets, TVoxelSet::VOXELSETTYPE_VISIBLEVOXEL);
		drawVoxelSets(p_VoxelSets, TVoxelSet::VOXELSETTYPE_MULTISLICE);
		drawVoxelSets(p_VoxelSets, TVoxelSet::VOXELSETTYPE_SUBSET);
		drawVoxelSets(p_VoxelSets, TVoxelSet::VOXELSETTYPE_ALL);
	}
	else
	{
		drawVoxelSets(p_VoxelSets, TVoxelSet::VOXELSETTYPE_ALL);
		drawVoxelSets(p_VoxelSets, TVoxelSet::VOXELSETTYPE_SUBSET);
		drawVoxelSets(p_VoxelSets, TVoxelSet::VOXELSETTYPE_MULTISLICE);
		drawVoxelSets(p_VoxelSets, TVoxelSet::VOXELSETTYPE_VISIBLEVOXEL);
	}
}


void MyApp::drawVoxelSet(TLayer * p_Layer, TVoxelSet * p_VoxelSet, const TField * p_Field, const TFilter * p_Filter, TColorMapUsingMeasure * p_ColorMap)
{
	const TField * Field = p_Layer->m_Field.get();
	const int MaxX = p_Field->getMaxX();
	const int MaxY = p_Field->getMaxY();
	const int MaxZ = p_Field->getMaxZ();

	bool DisplayList = false;
	// Make a displaylist of this voxelset if possible
	if(p_VoxelSet->getVoxelSetType() == TVoxelSet::VOXELSETTYPE_ALL || p_VoxelSet->getVoxelSetType() == TVoxelSet::VOXELSETTYPE_VISIBLEVOXEL)
	{
		if(p_VoxelSet->m_DisplayList == 0)  p_VoxelSet->m_DisplayList = glGenLists(1);
		
		if(p_VoxelSet->m_NeedRedraw)
		{
			glDeleteLists(p_VoxelSet->m_DisplayList,1);
			glNewList(p_VoxelSet->m_DisplayList,GL_COMPILE_AND_EXECUTE);
		}
		else if(p_VoxelSet->m_DisplayList != 0)
		{
			glCallList(p_VoxelSet->m_DisplayList);
			return;
		}
			
		DisplayList = true;
	}

	p_Layer->m_Field->begin();

	TRenderer Renderer(Field);
	Renderer.begin();

	// Track color, needed for colormap
	glEnable(GL_COLOR_MATERIAL);
	glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);
	TColor3 Color = p_VoxelSet->getDefaultColor();

	if(Field->getIndexField() && p_VoxelSet->getVoxelSetType() == TVoxelSet::VOXELSETTYPE_ALL)
	{
		shared_ptr<TIndexMapper> IndexField = Field->getIndexField();
		const unsigned int MaxX = IndexField->getMaxIndex();
		unsigned int x;
		for(x=0; x<MaxX; x++)
		{
			const TCoord3 p = IndexField->vidx2coord(x);
			if( p_Filter->testx(x) )
			{
				p_ColorMap->getColor(x, &Color);
				glColor3fv(Color.rgb);

				Renderer.renderVoxel(p);
			}
		}
	}
	else if(Field->getIndexField())
	{
		TCoord3 p;
		p_VoxelSet->initTraversal();
		while(p_VoxelSet->nextTraversal(p))
		{
			if(!Field->getIndexField()->vinside(p)) continue;

			const unsigned int idx = Field->getIndexField()->vcoord2idx(p);
 			if( !p_Filter->testx(idx) ) continue;

			p_ColorMap->getColor(idx, &Color);
			glColor3fv(Color.rgb);
			Renderer.renderVoxel(p);
			m_VoxelsRendered++;					
		}
	}
	else 
	{
		TCoord3 p;
		p_VoxelSet->initTraversal();
		while(p_VoxelSet->nextTraversal(p))
		{
 			if( !p_Filter->test(p) ) continue;

			p_ColorMap->getColor(p, &Color);
			glColor3fv(Color.rgb);
			Renderer.renderVoxel(p);
			m_VoxelsRendered++;					
		}
	}
	p_VoxelSet->m_NeedRedraw = false;
	Renderer.end();
	p_Layer->m_Field->end();

	if(DisplayList)
	{
		glEndList(); // end display list
	}
}


void MyApp::drawLayerSet(TLayerSet * p_LayerSet)
{
	glPushName(m_PickID); // Put one name on the stack. Use glLoadName to replace that name
	glPushName(0);

	// Gather all voxelsets that should be rendered
	vector<TVoxelSet*> VoxelSets;
	for(TLayerSet::TLayerList::const_iterator it = p_LayerSet->begin(); it != p_LayerSet->end(); it++)
	{
		shared_ptr<TLayer> Layer  = *it;
		if(Layer->getCheckedForRendering()  )
		{
			for(int j=0; j<Layer->m_VoxelSets.size(); j++)
			{
				shared_ptr<TVoxelSet> vs = Layer->m_VoxelSets.at(j);
				if(vs->getCheckedForRendering())
				{
					VoxelSets.push_back( vs.get() );
				}
			}
		}
	}

	// Draw non-overlays
	drawVoxelSetsInCorrectOrder( VoxelSets );

	checkGlError();



//	glEnable(GL_DEPTH_TEST);

	// Render selections in order
	TSelectionRenderer SelectionRenderer;
	if(g_Mediator.getSkeletonizer() && !m_Picking || g_Settings.m_AllowPickingOfSelection)
	{
		for(int j=0; j<m_SelectionRendererOrder.size(); j++)
		{
			TType::TYPE FieldType = m_SelectionRendererOrder[j];

			for(TLayerSet::TLayerList::iterator it = g_Mediator.getCurrentLayerSet()->begin(); it != g_Mediator.getCurrentLayerSet()->end(); it++)
			{
				shared_ptr<TLayer> Layer  = *it;
				if(Layer->m_Selection && Layer->getCheckedForRendering() && Layer->m_Selection->getCheckedForRendering() )
					if(Layer->m_Selection->getSize() > 0)
						if(Layer->m_Field->getType() == FieldType)
							Layer->m_Field->accept(&SelectionRenderer);
			}
		}
	}

	glPopName();
	glPopName();

	checkGlError();
}

void MyApp::OnDrawGL()
{
	// - Render
	m_VoxelsRendered = 0;
	if( g_Mediator.getLuaProgram()->isDirty() )
	{
		g_Mediator.getLuaProgram()->save();
		g_Mediator.getLuaProgram()->perform();
	}

	if(m_Picking)
	{
//		m_StatusBar->SetStatusText("Picking...",0);
	}
	else 
	{
//		m_StatusBar->SetStatusText("Rendering...",0);
		int x,y;
		MyApp::instance()->getGLCanvas()->GetClientSize(&x,&y);
		glMatrixMode(GL_PROJECTION);  
		glLoadIdentity();
		getCamera()->setProjectionMatrix(x,y);
	}

	drawBegin();

	if( g_Mediator.getSkeletonizer() && g_Settings.m_RenderModel && !m_Picking )
	{
		TModel * Model = g_Mediator.getSkeletonizer()->m_Model.get();
		if(Model && g_Settings.m_RenderModelWithoutShading)
		{
			TRenderer Renderer;
			Renderer.begin();

			glPushAttrib(GL_CURRENT_BIT);

			glEnable(GL_COLOR_MATERIAL);
			glColorMaterial(GL_FRONT_AND_BACK, GL_DIFFUSE);
			glColor3f(0.0f, 0.0f, 0.0f);
			glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT);
			glColor3f(0.9f, 0.9f, 0.9f);
			glDisable(GL_LIGHTING);

			Model->render();
			glDisable(GL_COLOR_MATERIAL);

			Renderer.end();

			glPopAttrib();

			glClear(GL_DEPTH_BUFFER_BIT);
		}
	}

	// Fog
	if(g_Mediator.getSkeletonizer() && g_Settings.m_RenderFog && (*g_Mediator.getCurrentLayerSet()->begin()) )
	{
		
		shared_ptr<TLayer> Layer = *g_Mediator.getCurrentLayerSet()->begin();
		TPoint3 Middle(
			(float) Layer->m_Field->getMaxX()  / 2.0f * g_Settings.m_VoxelScale  
			, (float) Layer->m_Field->getMaxY() / 2.0f * g_Settings.m_VoxelScale
			, (float) Layer->m_Field->getMaxZ() / 2.0f * g_Settings.m_VoxelScale
			);
		TPoint3 pos = m_CurrentCamera->getPos();
		float fogstart = pos.distance(Middle) ;
		float fogsize = Middle.distance( TPoint3(0,0,0) ); 

		GLfloat fogColor[4]= {0.5f, 0.5f, 0.5f, 1.0f};
		glDisable(GL_FOG);
		glFogi(GL_FOG_MODE, GL_LINEAR);		// Fog Mode
		glFogfv(GL_FOG_COLOR, fogColor);			// Set Fog Color
		glFogf(GL_FOG_DENSITY, 0.35f);				// How Dense Will The Fog Be
		glHint(GL_FOG_HINT, GL_DONT_CARE);			// Fog Hint Value
		glFogf(GL_FOG_START, fogstart-fogsize);				// Fog Start Depth
		glFogf(GL_FOG_END, fogstart+fogsize);				// Fog End Depth
		glEnable(GL_FOG);					// Enables GL_FOG
	}
	else
	{
		glDisable(GL_FOG);
	}

	checkGlError();
	

	// Actual rendering here, including selections
	for(unsigned int l=0; l<g_Mediator.m_LayerSets.size(); l++)
	{
		if( g_Mediator.m_LayerSets[l]->getCheckedForRendering() )
		{
			drawLayerSet( g_Mediator.m_LayerSets[l].get() );
		}
	}

	// Render axis
	if(!m_Picking && g_Settings.m_RenderAxis) 
	{
		TRenderer Renderer;
		Renderer.begin();
		glDisable(GL_LIGHTING);
		utilDrawAxes();
		Renderer.end();
	}



	// Draw selections
	if( !m_Picking && g_Mediator.getSkeletonizer() )
	{
		TRenderer Renderer;
		Renderer.begin();

		glEnable(GL_DEPTH_TEST);
		int i=0;
		for(TLayerSet::TLayerList::iterator it = g_Mediator.getCurrentLayerSet()->begin(); it != g_Mediator.getCurrentLayerSet()->end(); it++)
		{
			if((*it)->m_Selection)
			{
				if(m_ToolPanel->m_LayerListBox->IsChecked(i))
				{
					if( (*it)->m_RenderSelectionIndicators )
					{
						if( (*it)->m_Selection->getSize() > 0)
						{
							drawSelection( (*it).get());
						}
					}
				}
			}
			i++;
		}
		Renderer.end();
	}
	
	// Draw model
	if( g_Mediator.getSkeletonizer() 
		&& g_Settings.m_RenderModel 
		&& !m_Picking 
		&& !g_Settings.m_RenderModelWithoutShading
	)
	{
		TModel * Model = g_Mediator.getSkeletonizer()->m_Model.get();
		if(Model != 0)
		{
			TRenderer Renderer;
			Renderer.begin();

			if( g_Settings.m_UseBlendingForModel ) glEnable(GL_BLEND);
			else glDisable(GL_BLEND);

//				glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);


			if (TGlobalRenderer::instance()->m_BlendEquationAvailable)
			{
				glBlendEquation(GL_MIN); 
//					glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

				glEnable(GL_COLOR_MATERIAL);
				glColor3f(1.0f, 1.0f, 1.0f);

				//glEnable(GL_COLOR_MATERIAL);
				//glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);
				//glColorMaterial(GL_FRONT_AND_BACK, GL_DIFFUSE);
				//glColor4f(0.9f, 0.9f, 0.9f, 0.7f);
				//glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT);
				//glColor4f(0.9f, 0.9f, 0.9f, 0.5f);

				//GLfloat light0_ambient[4] = { 0.4f, 0.4f, 0.4f, 0.5f };
				//glLightfv(GL_LIGHT0, GL_AMBIENT, light0_ambient);
			}
// */
// /*
			else
			{
//					glBlendEquation(GL_MIN);
				glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

				glEnable(GL_COLOR_MATERIAL);
				glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);
				glColor4f(0.2f, 0.2f, 0.2f, 0.2f);

				{
					GLfloat light0_ambient[4] = { 0.2f, 0.2f, 0.2f, 0.5f };
					glLightfv(GL_LIGHT0, GL_AMBIENT, light0_ambient);
				}
			}
// */



			Model->render();


			Renderer.end();
		}
	}




	drawEnd();

	checkGlError();
	m_RedrawCanvas = false;
}

void MyApp::OnLayerClick(wxCommandEvent& event)
{
	int sel = m_ToolPanel->m_LayerListBox->GetSelection();
	shared_ptr<TLayer> LayerByIndex = g_Mediator.getCurrentLayerSet()->getLayerByIndex(sel);
	g_Mediator.setCurrentLayer( LayerByIndex );

	MyApp::instance()->m_LayerPanel->updateForLayer( LayerByIndex.get() );
	updateStatusBar();
}

void MyApp::OnLayerSetClick(wxCommandEvent& event)
{
	if( event.GetInt() >= 0 && event.GetInt() < g_Mediator.m_LayerSets.size() )
	{
		g_Mediator.m_CurrentLayerSet = g_Mediator.m_LayerSets[ event.GetInt() ];
	}
	redrawCanvas();
	updateUI();
}


void MyApp::OnLayerCheckListBox(wxCommandEvent& event)
{
	int sel = event.GetInt();
	g_Mediator.getCurrentLayerSet()->getLayerByIndex(event.GetInt())->setCheckedForRendering( m_ToolPanel->m_LayerListBox->IsChecked(sel) );
	m_ToolPanel->m_LayerListBox->SetSelection(sel);
	OnLayerClick(event);
	redrawCanvas();
}

void MyApp::OnLayerSetCheckListBox(wxCommandEvent& event)
{
	int sel = event.GetInt();
	g_Mediator.m_LayerSets[ sel ]->setCheckedForRendering( m_ToolPanel->m_LayerSetListBox->IsChecked(sel) );
	m_ToolPanel->m_LayerSetListBox->SetSelection(sel);
	OnLayerSetClick(event);
	redrawCanvas();
}




void MyApp::updateStatusBar()
{
	shared_ptr<TLayer> Layer = g_Mediator.getCurrentLayer();
	if(Layer && Layer->m_Selection && Layer->m_FilterMeasure)
	{
		const TMeasure * Measure = Layer->m_FilterMeasure.get();
		TCoord3 c = Layer->m_Selection->getSingleSelection();
		if(c.x != -1)
		{
			if(Layer->m_Field->usingIndexField())
			{
				if(Layer->m_Field->getIndexField()->vinside(c))
				{
					unsigned int idx = Layer->m_Field->getIndexField()->vcoord2idx(c);
					m_StatusBar->SetStatusText( wxString::Format("Selected (%d,%d,%d) (idx: %i): %f", c.x, c.y, c.z, idx, Measure->vvaluex(idx)),0);
				}
			}
			else
			{
				m_StatusBar->SetStatusText( wxString::Format("Selected (%d,%d,%d): %f", c.x, c.y, c.z,Measure->toFloat(c)),0);
			}
		}
	}
}


void MyApp::OnLoadField(wxCommandEvent& event)
{
	wxFileDialog FileDialog(m_TopWindow, "Choose skeleton file to open", "", "", "*.skel;*.skel.gz;*.cskel;*.cskel.gz", wxFD_OPEN | wxFD_FILE_MUST_EXIST );
	if( FileDialog.ShowModal() == wxID_OK)
	{
		*g_Log << "Loading skeleton from file '" << FileDialog.GetPath().c_str() << "'...";
		TSkeletonizer * skel = TSkeletonizer::readFromFile( FileDialog.GetPath().ToStdString() );
		skel->tryOpenCamera( FileDialog.GetPath().ToStdString() );
		*g_Log << " done\n";
	}
}

void MyApp::OnSaveField(wxCommandEvent& event)
{
	if(!g_Mediator.getSkeletonizer()) throw string("!g_Mediator.getSkeletonizer()");

	int sel = m_ToolPanel->m_LayerListBox->GetSelection();
	if(sel == -1) return;

	wxString FilePath = wxFileName(g_Mediator.getSkeletonizer()->m_InputFilename.c_str()).GetPath(wxPATH_GET_VOLUME);
	wxFileDialog FileDialog(m_TopWindow, "Choose skeleton file to save to", FilePath, "", "*.skel;*.skel.gz;*.cskel;*.cskel.gz", wxFD_SAVE);
	if( FileDialog.ShowModal() == wxID_OK)
	{
		*g_Log << "Saving skeletonizer to '" << FileDialog.GetPath().c_str() << "'...";
		g_Mediator.getSkeletonizer()->writeToFile( FileDialog.GetPath().ToStdString() );
		*g_Log << " done.\n";
	}
}

void MyApp::OnCameraChoice(wxCommandEvent& event)
{
	int sel = event.GetId();
	if( sel == ID_ROAMINGCAMERA )
	{
		m_CurrentCamera = m_ObservingCamera;
		m_ObservingCameraButton->SetValue(false);
	}
	else if( sel == ID_OBSERVINGCAMERA )
	{
		m_CurrentCamera = m_RoamingCamera;
		m_RoamingCameraButton->SetValue(false);
	}
	redrawCanvas();
}

void MyApp::OnLookDownAxis(wxCommandEvent& event)
{
	int sel = m_LookDownAxisBox->GetSelection();
	if(sel == -1) return;
	m_CurrentCamera->snapToAxis(sel);
	redrawCanvas();
}




void MyApp::processHits(GLint hits, GLuint buffer[], vector<unsigned int> & p_Hits)
{
   unsigned int i, j;
   GLuint names, *ptr, minZ,*ptrNames, numberOfNames;

//   printf ("hits = %d\n", hits);
   ptr = (GLuint *) buffer;
   minZ = 0xffffffff;
   for (i = 0; i < hits; i++) {	
      names = *ptr;
	  ptr++;
	  if (*ptr < minZ) {
		  numberOfNames = names;
		  minZ = *ptr;
		  ptrNames = ptr+2;
	  }
	  
	  ptr += names+2;
	}
//  printf ("The closest hit names are ");
  ptr = ptrNames;
  for (j = 0; j < numberOfNames; j++,ptr++) {
//     printf ("%d ", *ptr);
	 p_Hits.push_back(*ptr);
  }
//  printf ("\n");
   
}


void MyApp::attemptPick(class wxMouseEvent * event, vector<unsigned int> & p_PickStack)
{
	shared_ptr<TLayer> Layer = g_Mediator.getCurrentLayer();
	if(!Layer) throw string("No layer selected");

	m_StatusBar->SetStatusText("Picking...",0);
	
	wxSize sz = MyApp::instance()->getGLCanvas()->GetClientSize();
	float x = event->GetX();
	float y = event->GetY();

	float screenwidth  = sz.x;
	float screenheight = sz.y; 
	y = screenheight - y; // reverse y coordinate
	
	// Set new projection matrix
	int Viewport[4];
	glGetIntegerv(GL_VIEWPORT, Viewport);
	int szx,szy;
	MyApp::instance()->getGLCanvas()->GetClientSize(&szx,&szy);
	glMatrixMode(GL_PROJECTION); 
	glPushMatrix();
	glLoadIdentity();
    gluPickMatrix(x, y, 2.0, 2.0, Viewport);
	getCamera()->setProjectionMatrix(szx,szy);
	checkGlError();

	// Initialize for selection
	int BufferSize = 1024; 
	//GLuint * Buffer = new GLuint[BufferSize];
	shared_ptr<GLuint> Buffer( new GLuint[BufferSize] );

	glSelectBuffer(BufferSize, Buffer.get());
	glRenderMode(GL_SELECT);
	glInitNames();
	m_Picking = true;
	MyApp::instance()->getGLCanvas()->OwnPaint();
	m_Picking = false;

	// Restore projection matrix
	glMatrixMode(GL_PROJECTION); 
	glPopMatrix();
	checkGlError();
	
	// Retrieve coord
	GLint Hits = glRenderMode(GL_RENDER);
	TField * Field = g_Mediator.getCurrentLayer()->m_Field.get();

//		*g_Log << "Hits: " << Hits << endl;
	if(Hits <= 0 || Hits >= BufferSize) 
	{
		if(Hits >= BufferSize)
		{
			*g_Log << "Picking error: Hits == 0 || Hits > BufferSize: " << Hits << ", " << BufferSize << "\n";
		}
		vector<unsigned int> Empty;
		p_PickStack = Empty;
	}
	else
	{
		processHits(Hits, Buffer.get(), p_PickStack);
	}

	m_StatusBar->SetStatusText("");
}



void MyApp::moveSelection(int p_x, int p_y, int p_z)
{
	for(TLayerSet::TLayerList::iterator it = g_Mediator.getCurrentLayerSet()->begin(); it != g_Mediator.getCurrentLayerSet()->end(); it++)
	{
		shared_ptr<TLayer> l = (*it);
		if(!l->m_Selection) continue;

		TCoord3 oldc = l->m_Selection->getSingleSelection();
		TCoord3 newc(oldc.x + p_x, oldc.y + p_y, oldc.z + p_z );

		l->m_Selection->erase( oldc );
		l->m_Selection->select( newc );
	}
	m_LayerPanel->updateForLayer( g_Mediator.getCurrentLayer().get() );
	redrawCanvas();
}


void MyApp::addLayer(shared_ptr<TLayer> p_Layer)
{
	shared_ptr<TLayer> After;
	if( g_Mediator.getCurrentLayerSet()->count() != 0 ) After = g_Mediator.getCurrentLayerSet()->getLayerByIndex(g_Mediator.getCurrentLayerSet()->count()-1);
	addLayer(p_Layer, After.get());
}

void MyApp::addLayer(shared_ptr<TLayer> p_Layer, TLayer * p_AfterLayer)
{
	if(! p_Layer.get()) throw string("addLayer(): p_Layer == 0");

//	int Selection = m_ToolPanel->m_LayerListBox->GetSelection();

	g_Mediator.getCurrentLayerSet()->insertLayerAfterLayer(p_Layer, p_AfterLayer);

	updateUI();

/*
	m_ToolPanel->m_LayerListBox->GetSize();
	if(Selection >= 0 && Selection < m_ToolPanel->m_LayerListBox->GetCount() ) 
	{
		m_ToolPanel->m_LayerListBox->SetSelection(Selection);
	}
	else if(Selection == -1)
	{
		m_ToolPanel->m_LayerListBox->SetSelection( m_ToolPanel->m_LayerListBox->GetCount()-1 );
	}
	m_LayerPanel->updateForLayer(p_Layer.get());
*/
}

void MyApp::updateUI()
{
	if(!g_Mediator.getCurrentLayer())
	{
		int c = m_ToolPanel->m_LayerListBox->GetCount()-1;
		if(c >= 0) m_ToolPanel->m_LayerListBox->SetSelection(c);
	}


	// Fill layerset listbox
	{
		int sel = m_ToolPanel->m_LayerSetListBox->GetSelection();
		m_ToolPanel->m_LayerSetListBox->Clear();
		unsigned int x;
		for(x=0; x<g_Mediator.m_LayerSets.size(); x++)
		{
			m_ToolPanel->m_LayerSetListBox->Append( wxString::Format("%d. %s ", x, g_Mediator.m_LayerSets[x]->getName().c_str() ) );
			m_ToolPanel->m_LayerSetListBox->Check( x, g_Mediator.m_LayerSets[x]->getCheckedForRendering() );
			if( g_Mediator.m_LayerSets[x].get() == g_Mediator.getCurrentLayerSet().get() )
			{
				m_ToolPanel->m_LayerSetListBox->SetSelection(x);
			}
		}
	}


	{
		int sel = m_ToolPanel->m_LayerListBox->GetSelection();
		m_ToolPanel->m_LayerListBox->Clear();
		int i=0;
		for(TLayerSet::TLayerList::iterator it = g_Mediator.getCurrentLayerSet()->begin(); it != g_Mediator.getCurrentLayerSet()->end(); it++)
		{
			string spaces = "";
			for(unsigned int x=0; x<(*it)->m_Depth; x++) spaces += "     ";
			m_ToolPanel->m_LayerListBox->Append( wxString::Format("%d. %s %s (%s)", (*it)->m_Id, spaces.c_str(), (*it)->m_Name.c_str(), (*it)->m_Field->getTypeName().c_str() ) );
			m_ToolPanel->m_LayerListBox->Check( i, (*it)->getCheckedForRendering() );
			if( (*it).get() == g_Mediator.getCurrentLayerSet()->m_CurrentLayer.get() )
				m_ToolPanel->m_LayerListBox->SetSelection(i);
			i++;
		}
	}


	if(g_Mediator.getCurrentLayer())
	{
		m_LayerPanel->updateForLayer(g_Mediator.getCurrentLayer().get());
	}
}




TToolPanel::TToolPanel(wxWindow* parent, wxWindowID id, const wxPoint& pos, const wxSize& size, long style)
	: wxPanel(parent, id, pos, size, style)
{
	wxBoxSizer * Sizer = new wxBoxSizer(wxVERTICAL);

//	m_Notebook = new wxNotebook(this, -1);
//	Sizer->Add(m_Notebook, 1, wxALL | wxEXPAND, 0);

	// Layer set
	{
//		wxPanel * Panel = new wxPanel(m_Notebook, -1, wxDefaultPosition, wxSize(300,150) );
//		m_Notebook->AddPage(Panel, "Layerset");
		wxPanel * Panel = this;
		
		int x = 5;
		int y = 5;
		new wxStaticText(Panel, -1, "Layer sets", wxPoint(x,y) );
		y += 15;
		m_LayerSetListBox = new TLayerSetListBox(Panel, ID_LAYERSETLISTBOX, wxPoint(x,y), wxSize(300,80) );
		y += m_LayerSetListBox->GetSize().y + 15;

		new wxStaticText(Panel, -1, "Layers in set", wxPoint(x,y) );
		y += 15;
		m_LayerListBox = new TLayerListBox(Panel, ID_LAYERLISTBOX, wxPoint(x,y), wxSize(300,300) );
		y += m_LayerListBox->GetSize().y + 15;
	}


	this->SetSizer(Sizer);
}


void TLayerListBox::OnMouse(wxMouseEvent & event)
{
	event.Skip();
	if(event.RightDown())
	{
		shared_ptr<TLayerMenu> LayerMenu( new TLayerMenu() );
		PopupMenu( LayerMenu.get(), event.GetPosition() );
	}
}

BEGIN_EVENT_TABLE(TLayerListBox, wxCheckListBox)
    EVT_MOUSE_EVENTS( TLayerListBox::OnMouse )
END_EVENT_TABLE()


void TLayerSetListBox::OnMouse(wxMouseEvent & event)
{
	event.Skip();
	if(event.RightDown())
	{
		shared_ptr<TLayerSetMenu> LayerSetMenu( new TLayerSetMenu() );
		PopupMenu( LayerSetMenu.get(), event.GetPosition() );
	}
}


BEGIN_EVENT_TABLE(TLayerSetListBox, wxCheckListBox)
    EVT_MOUSE_EVENTS( TLayerSetListBox::OnMouse )
END_EVENT_TABLE()


TLogWriter & MyApp::operator<<(int v)
{
	*m_LogLines << v;
	return *this;
}

TLogWriter & MyApp::operator<<(unsigned int v)
{
	m_LogLines->AppendText( wxString::Format("%i",v) );
	return *this;
}

TLogWriter & MyApp::operator<<(string v)
{
	m_LogLines->AppendText(v.c_str());
	return *this;
}

TLogWriter & MyApp::operator<<(const char v[])
{
	*m_LogLines << v;
	return *this;
}

TLogWriter & MyApp::operator<<(float v)
{
	*m_LogLines << v;
	return *this;
}



TSelectionMenu::TSelectionMenu()
	: wxMenu()
{	
	TLayer * layer = g_Mediator.getCurrentLayer().get();
	if(layer == 0) return;

	this->Append( ID_CREATEVOXELSLICE, wxT("New slice") );
	this->Append( ID_DELETEVOXELSET, wxT("Delete voxelset") );
	FindItem(ID_DELETEVOXELSET)->Enable(true);
	
	int sel = MyApp::instance()->m_LayerPanel->m_VoxelSetListBox->GetSelection();
	if(layer->m_VoxelSets[sel] == layer->m_AllVoxels) FindItem(ID_DELETEVOXELSET)->Enable(false);
	else if(layer->m_VoxelSets[sel] == layer->m_Selection) FindItem(ID_DELETEVOXELSET)->Enable(false);

	{
		wxMenu * menu = new wxMenu();
		for(int i=0; i<g_Mediator.getCurrentLayerSet()->count(); i++)
		{
			menu->Append( ID_FIRSTSELECTVISIBLE + i, wxString( g_Mediator.getCurrentLayerSet()->getLayerByIndex(i)->m_Name ) );
		}	
		
		this->Append( ID_SELECTVISIBLE, wxT("Select visible voxels from:"), menu );
	}

	{
		wxMenu * menu = new wxMenu();
		for(int i=0; i<g_Mediator.getCurrentLayerSet()->count(); i++)
		{
			menu->Append( ID_FIRSTMASKBYLAYER + i, wxString( g_Mediator.getCurrentLayerSet()->getLayerByIndex(i)->m_Name ) );
		}	
		
		this->Append( ID_MASKBY, wxT("Mask by visible voxels from:"), menu );
	}
}

void TSelectionMenu::OnMenuChoice(wxCommandEvent& event)
{
	TLayer * layer = g_Mediator.getCurrentLayer().get();
	if(event.GetId() == ID_CREATEVOXELSLICE)
	{
		if(layer) 
		{
			shared_ptr<TVoxelSet> vs( new TVoxelMultiSlice(layer->m_Field.get(), 0, 0) );
			layer->m_VoxelSets.push_back( vs );
			layer->setNeedRedraw();
			MyApp::instance()->updateUI();
		}
	}
	else if(event.GetId() == ID_DELETEVOXELSET)
	{
		if( layer )
		{	
			int sel = MyApp::instance()->m_LayerPanel->m_VoxelSetListBox->GetSelection();

			vector<shared_ptr<TVoxelSet> > NewVoxelSets;
			for(unsigned int x=0; x<layer->m_VoxelSets.size(); x++)
			{
				if(x != sel) NewVoxelSets.push_back( layer->m_VoxelSets[x] );
			}
			layer->m_VoxelSets = NewVoxelSets;
			MyApp::instance()->updateUI();
		}
	}
	else if(event.GetId() == ID_SELECTIONFROMVISIBLE)
	{
		if( layer )
		{	
			vector<TCoord3> VisibleVoxels;
			layer->getVisibleVoxels(&VisibleVoxels);
			layer->m_Selection->clear();
			for(unsigned int x=0; x<VisibleVoxels.size(); x++)
			{
				layer->m_Selection->select(VisibleVoxels[x]);
			}
			layer->setNeedRedraw();
			MyApp::instance()->redrawCanvas();
		}
	}
	else if(event.GetId() == ID_SELECTIONFROMVISIBLEALLLAYERS)
	{
		for(TLayerSet::TLayerList::iterator it = g_Mediator.getCurrentLayerSet()->begin(); it != g_Mediator.getCurrentLayerSet()->end(); it++)
		{
			TLayer * l = (*it).get();
			l->setSelectionFromFilter(layer->m_Filter.get());
			l->setNeedRedraw();
		}	
		MyApp::instance()->redrawCanvas();
	}


	else if(event.GetId() == ID_SELECTIONFROMFILTER)
	{
		if( layer )
		{
			layer->setSelectionFromFilter(layer->m_Filter.get());
			layer->setNeedRedraw();
			MyApp::instance()->redrawCanvas();
		}
	}
	else if(event.GetId() == ID_SELECTIONFROMFILTERALLLAYERS)
	{
		for(TLayerSet::TLayerList::iterator it = g_Mediator.getCurrentLayerSet()->begin(); it != g_Mediator.getCurrentLayerSet()->end(); it++)
		{
			TLayer * l = (*it).get();
			l->setSelectionFromFilter(layer->m_Filter.get());
			l->setNeedRedraw();
		}	
		MyApp::instance()->redrawCanvas();
	}



	else if(event.GetId() >= ID_FIRSTSELECTVISIBLE && event.GetId() <= ID_LASTSELECTVISIBLE)
	{
		const TLayer * SelectLayer = g_Mediator.getCurrentLayerSet()->getLayerByIndex( event.GetId() - ID_FIRSTSELECTVISIBLE  ).get();
		g_Mediator.getCurrentLayer()->selectVisibleVoxels(SelectLayer);
		
		MyApp::instance()->updateUI();
		MyApp::instance()->redrawCanvas();

	}

	else if(event.GetId() >= ID_FIRSTMASKBYLAYER && event.GetId() <= ID_LASTMASKBYLAYER)
	{
		const TLayer * MaskLayer = g_Mediator.getCurrentLayerSet()->getLayerByIndex( event.GetId() - ID_FIRSTMASKBYLAYER ).get();
		shared_ptr<TVoxelSet> vs(new TVisibleVoxelSet(g_Mediator.getCurrentLayer()->m_Field.get(), MaskLayer));
		vs->m_Name = "Masked by " + MaskLayer->m_Name;
		g_Mediator.getCurrentLayer()->m_VoxelSets.push_back( vs );
		MyApp::instance()->updateUI();
		MyApp::instance()->redrawCanvas();

	}
		
}

BEGIN_EVENT_TABLE(TSelectionMenu, wxMenu)
    EVT_MENU_RANGE(ID_FIRST, ID_LAST, TSelectionMenu::OnMenuChoice)
END_EVENT_TABLE()


TRenderMenu::TRenderMenu()
	: wxMenu()
{
	Append(ID_FIELDCHANGED, wxT("Set field changed"));
	Append(ID_NEEDREDRAW, wxT("Set need redraw"));
	Append(ID_CAMERARESET, wxT("Camera reset"));
	AppendCheckItem(ID_RENDERAXIS, wxT("Render axis"));
	FindItem(ID_RENDERAXIS)->Check( g_Settings.m_RenderAxis);
	
	AppendCheckItem(ID_RENDERMODEL, wxT("Render model"));
	FindItem(ID_RENDERMODEL)->Check(g_Settings.m_RenderModel);
	AppendCheckItem(ID_RENDERMODELWITHOUTSHADING, wxT("No shading on model"));
	FindItem(ID_RENDERMODELWITHOUTSHADING)->Check(g_Settings.m_RenderModelWithoutShading);

	if(!g_Settings.m_ReleaseMode)
	{
		AppendCheckItem(ID_PREVENTRENDERINGTOOMUCH, wxT("Prevent rendering too much"));
		FindItem(ID_PREVENTRENDERINGTOOMUCH)->Check(g_Settings.m_PreventRenderingTooMuch);
	}

}

void TRenderMenu::OnMenuChoice(wxCommandEvent& event)
{
	if(!MyApp::instance() || !g_Mediator.getCurrentLayer())
	{
		return;
	}
	shared_ptr<TLayer> currentlayer = g_Mediator.getCurrentLayer();

	bool checked = ! event.IsChecked();
	if( FindItem(event.GetId())->IsCheckable() ) FindItem(event.GetId())->Check( checked );

	if(event.GetId() == ID_RENDERBLACKANDWHITE)
	{
		g_Settings.m_RenderBlackAndWhite = checked;
		MyApp::instance()->redrawCanvas();
	}
	else if(event.GetId() == ID_RENDERAXIS)
	{
		g_Settings.m_RenderAxis = checked;
		MyApp::instance()->redrawCanvas();
	}

	else if(event.GetId() == ID_RENDERMODEL)
	{
		g_Settings.m_RenderModel = checked;
		MyApp::instance()->redrawCanvas();
	}
	else if(event.GetId() == ID_NEEDREDRAW)
	{
		currentlayer->setNeedRedraw();
		MyApp::instance()->redrawCanvas();
	}
	else if(event.GetId() == ID_FIELDCHANGED)
	{
		currentlayer->onFieldChanged();
		currentlayer->setNeedRedraw();
		MyApp::instance()->redrawCanvas();
	}
	else if(event.GetId() == ID_CAMERARESET)
	{
		MyApp::instance()->m_CurrentCamera->reset();
		MyApp::instance()->redrawCanvas();
	}
	else if(event.GetId() == ID_PREVENTRENDERINGTOOMUCH)
	{
		g_Settings.m_PreventRenderingTooMuch = checked;
		for(TLayerSet::TLayerList::iterator it = g_Mediator.getCurrentLayerSet()->begin(); it != g_Mediator.getCurrentLayerSet()->end(); it++)
		{
			(*it)->setNeedRedraw();
		}
		MyApp::instance()->redrawCanvas();
	}
	else if(event.GetId() == ID_RENDERMODELWITHOUTSHADING)
	{
		g_Settings.m_RenderModelWithoutShading = checked;
		MyApp::instance()->redrawCanvas();	
	}


	
}


BEGIN_EVENT_TABLE(TRenderMenu, wxMenu)
    EVT_MENU_RANGE(ID_FIRST, ID_LAST, TRenderMenu::OnMenuChoice)
END_EVENT_TABLE()




TLayerMenu::TLayerMenu()
	: wxMenu()
{	
	if(! g_Mediator.getCurrentLayer() ) return;
	shared_ptr<TLayer> layer = g_Mediator.getCurrentLayer();
	
	// Submenu
	wxMenu * menu = new wxMenu;
	int items = 0;
	for(unsigned int x=0; x<g_AllAdaptors.size(); x++)
	{
		if(layer->m_Field->getType() == g_AllAdaptors[x]->adapts() || g_AllAdaptors[x]->adapts() == TType::TYPE_ANY)
		{
			menu->Append( ID_FIRSTADDADAPTOR+x, g_AllAdaptors[x]->getAdaptorName().c_str() ); 
			items++;
		}
	}
	if(items > 0)
	{
		Append(0, wxT("Add adaptor"), menu);
	}
	else delete menu;

	if(!g_Settings.m_ReleaseMode)
	{
		Append(ID_ADDSCRIPTEDLAYER, wxT("Add scripted layer (using indexfield of cur. layer)"));
		Append(ID_RENAMELAYER, wxT("Rename layer"));

		Append( ID_DELETELAYER, wxT("Delete layer") );
		FindItem(ID_DELETELAYER)->Enable(false);
	}

	AppendSeparator();

	AppendCheckItem(ID_RENDERCUBES, wxT("Render cubes"));
	FindItem(ID_RENDERCUBES)->Check(layer->m_RenderPrimitiveType == 0);
	AppendCheckItem(ID_RENDERSPHERES, wxT("Render spheres"));
	FindItem(ID_RENDERSPHERES)->Check(layer->m_RenderPrimitiveType == 1);
	
	
	if(!g_Settings.m_ReleaseMode)
	{
		AppendCheckItem(ID_RENDERQUADS, wxT("Render quads"));
		FindItem(ID_RENDERQUADS)->Check(layer->m_RenderPrimitiveType == 2);
	}

	AppendSeparator();

	AppendCheckItem(ID_RENDERLINES, wxT("Render lines"));
	FindItem(ID_RENDERLINES)->Check(layer->m_RenderLines);

	AppendCheckItem(ID_RENDERVOXELOUTLINES, wxT("Render voxel outlines"));
	FindItem(ID_RENDERVOXELOUTLINES)->Check(layer->m_RenderVoxelOutlines);

	AppendCheckItem(ID_RENDERSELECTIONINDICATORS, wxT("Render selection indicators"));
	FindItem(ID_RENDERSELECTIONINDICATORS)->Check(layer->m_RenderSelectionIndicators);

	/*
	if(!g_Settings.m_ReleaseMode)
	{
		AppendSeparator();
		Append( ID_UPDATEHISTOGRAMUSINGFILTER, wxT("Update histogram using filter") );
		FindItem(ID_UPDATEHISTOGRAMUSINGFILTER)->Enable(false);
		Append( ID_UPDATEHISTOGRAM, wxT("Update histogram") );
		FindItem(ID_UPDATEHISTOGRAM)->Enable(false);
		Append( ID_DISABLEHISTOGRAM, wxT("Disable histogram") );
		FindItem(ID_DISABLEHISTOGRAM)->Enable(false);
	}
	*/

	if( g_Mediator.getCurrentLayer() )
	{
		if(FindItem(ID_UPDATEHISTOGRAMUSINGFILTER)) FindItem(ID_UPDATEHISTOGRAMUSINGFILTER)->Enable(true);
		if(FindItem(ID_UPDATEHISTOGRAM)) FindItem(ID_UPDATEHISTOGRAM)->Enable(true);
		if( g_Mediator.getCurrentLayer()->m_Histogram )
		{
			if(FindItem(ID_DISABLEHISTOGRAM)) FindItem(ID_DISABLEHISTOGRAM)->Enable(true);
		}
		if(FindItem(ID_DELETELAYER)) FindItem(ID_DELETELAYER)->Enable(true);
	}
}


void TLayerMenu::OnMenuChoice(wxCommandEvent& event)
{
	bool checked = ! event.IsChecked();
	if(! g_Mediator.getCurrentLayer() ) return;
	shared_ptr<TLayer> layer = g_Mediator.getCurrentLayer();

	if(
		event.GetId() == ID_RENDERCUBES 
		|| event.GetId() == ID_RENDERSPHERES
		|| event.GetId() == ID_RENDERQUADS
		)
	{
		if(event.GetId() == ID_RENDERCUBES)
			layer->m_RenderPrimitiveType = 0;
		else if(event.GetId() == ID_RENDERSPHERES)
			layer->m_RenderPrimitiveType = 1;
		else if(event.GetId() == ID_RENDERQUADS)
			layer->m_RenderPrimitiveType = 2;

		layer->setNeedRedraw();
		MyApp::instance()->redrawCanvas();
	}
	else if(event.GetId() == ID_RENDERLINES)
	{
		layer->m_RenderLines = checked;		
		layer->setNeedRedraw();
		MyApp::instance()->redrawCanvas();
	}
	else if(event.GetId() == ID_RENDERVOXELOUTLINES)
	{
		layer->m_RenderVoxelOutlines = checked;		
		layer->setNeedRedraw();
		MyApp::instance()->redrawCanvas();
	}
	
	else if(event.GetId() == ID_RENDERSELECTIONINDICATORS)
	{
		layer->m_RenderSelectionIndicators = checked;
		layer->setNeedRedraw();
		MyApp::instance()->redrawCanvas();
	}
	else if(event.GetId() == ID_RENAMELAYER)
	{
		//layer->m_RenderSelectionIndicators = checked;
		wxTextEntryDialog * Dialog = new wxTextEntryDialog(0, "Enter display name", "Enter display name", layer->m_Name.c_str());
		if( Dialog->ShowModal() == wxID_OK )
		{	
			layer->m_Name = Dialog->GetValue().c_str();
		}
		
		MyApp::instance()->updateUI();
		
	}


	else if(event.GetId() == ID_DISABLEHISTOGRAM)
	{
		layer->m_Histogram.reset();
		MyApp::instance()->redrawCanvas();
	}
	else if(event.GetId() == ID_DELETELAYER)
	{
		g_Mediator.getCurrentLayerSet()->deleteLayer( layer->m_ShortName );
		MyApp::instance()->updateUI();
		MyApp::instance()->redrawCanvas();
	}

	else if(event.GetId() >= ID_FIRSTADDADAPTOR && event.GetId() <= ID_LASTADDADAPTOR)
	{
		shared_ptr<TLayer> CurrentLayer = g_Mediator.getCurrentLayer();
		if(!CurrentLayer) throw string("No currentlayer");
			
		int adaptor = event.GetId()-ID_FIRSTADDADAPTOR;
		TField * FieldAdaptor = g_AllAdaptors[adaptor]->create(CurrentLayer->m_Field.get());
		shared_ptr<TField> Field( FieldAdaptor );
		shared_ptr<TLayer> NewLayer( new TLayer(Field, FieldAdaptor->getAdaptorName() ) );
		NewLayer->m_Depth = 1;
		MyApp::instance()->addLayer(NewLayer, CurrentLayer.get() );
		if(Field->getLayer() == 0) throw string("Field->m_Layer == 0");
	}

	else if(event.GetId() == ID_ADDSCRIPTEDLAYER)
	{
		shared_ptr<TField> Field;
		shared_ptr<TLayer> Layer;
		try
		{
			TScriptedField * sf = new TScriptedField( layer->m_Field->getIndexField() );
			Field.reset(sf);
			sf->m_SubjectLayerId = layer->m_Id;
			Layer.reset(new TLayer(Field, "Scripted") );
			g_Mediator.addLayer( Layer );
		}
		catch(string & error)
		{
			*g_Log << "Warning: " << error << "\n";
		}
	}
}


BEGIN_EVENT_TABLE(TLayerMenu, wxMenu)
    EVT_MENU_RANGE(ID_FIRST, ID_LAST, TLayerMenu::OnMenuChoice)
END_EVENT_TABLE()



// ---------------------------------

TLayerSetMenu::TLayerSetMenu()
	: wxMenu()
{	
	shared_ptr<TLayerSet> LayerSet = g_Mediator.getCurrentLayerSet();
	if(LayerSet->m_Skeletonizer)
	{
		Append(ID_SAVESKELETON, wxString::Format("Save skeleton to disk"));
	}

	if(LayerSet->exists("Scalar data"))
	{
		shared_ptr<TLayer> Layer = LayerSet->getLayer("Scalar data");
		Append(ID_SKELETONIZESCALARDATA, wxString::Format("Skeletonize volume (%.2f,%.2f)", Layer->m_Filter->m_LowerValue, Layer->m_Filter->m_UpperValue));
	}
}

void TLayerSetMenu::OnMenuChoice(wxCommandEvent& event)
{
	bool checked = ! event.IsChecked();

	shared_ptr<TLayerSet> LayerSet = g_Mediator.getCurrentLayerSet();
	if(event.GetId() == ID_SKELETONIZESCALARDATA)
	{
		TTypedFieldInterface<float> * Field = static_cast<TTypedFieldInterface<float> *>( LayerSet->getField("Scalar data") );
		if(!Field) throw string("!Field");
		TScalarData::processFilteredVolume( Field );
	}
	else if(event.GetId() == ID_SAVESKELETON)
	{
		if(!LayerSet->m_Skeletonizer) throw string("!LayerSet->m_Skeletonizer");
		MyApp::instance()->OnSaveField(event);
	}
}

BEGIN_EVENT_TABLE(TLayerSetMenu, wxMenu)
    EVT_MENU_RANGE(ID_FIRST, ID_LAST, TLayerSetMenu::OnMenuChoice)
END_EVENT_TABLE()


// ---------------------------------

void MyApp::OnPicked(vector<unsigned int> p_PickStack, wxMouseEvent * event)
{
	if(!g_Mediator.getCurrentLayer()) throw string("No layer selected");

	const int c = p_PickStack.at(0);
	if(c == 0) return;
	
	TCoord3 Selected(-1,-1,-1);
	{
		shared_ptr<TLayer> Layer( g_Mediator.getCurrentLayer() ); 
		const int maxx = Layer->m_Field->getMaxX();
		const int maxy = Layer->m_Field->getMaxY();
		const int maxz = Layer->m_Field->getMaxZ();
		const int z = c / (maxx * maxy);
		const int remainder = c % (maxx * maxy);
		const int x = remainder % maxx;
		const int y = remainder / maxx;
		Selected = TCoord3(x,y,z); 
	}

	if(g_Settings.m_EditObjectMode)
	{
		shared_ptr<TLayer> Layer( g_Mediator.getCurrentLayer() ); 
		if(Layer->m_Field->getType() == TType::TYPE_UCHAR)
		{
			TUnsignedCharField3 * uf = static_cast<TUnsignedCharField3*>(Layer->m_Field.get());
			if(event->LeftDown()) 
				uf->valuep(Selected) = 1;	
			else if(event->RightDown())
				uf->valuep(Selected) = 0;	
		}
		else throw string("Can only edit uchar field");
	}
	else
	{
		if(event->ShiftDown() && event->LeftIsDown())
		{
			// Select in current layer
			shared_ptr<TLayer> Layer( g_Mediator.getCurrentLayer() ); 
			if(Layer->m_Selection) Layer->m_Selection->select(Selected);
		}
		else if(event->ControlDown() && event->LeftIsDown())
		{
			// Select in all layers
			for(TLayerSet::TLayerList::iterator it = g_Mediator.getCurrentLayerSet()->begin(); it != g_Mediator.getCurrentLayerSet()->end(); it++)
			//for(int i=0; i<m_Layers.size(); i++)
			{
				shared_ptr<TLayer> Layer = (*it);
				if(Layer->m_Selection) Layer->m_Selection->select(Selected);
			}
		}
		else if(event->ShiftDown() && event->RightDown())
		{
			// Remove selected from m_MultipleSelection
			shared_ptr<TLayer> Layer( g_Mediator.getCurrentLayer() ); 
			if(Layer->m_Selection) Layer->m_Selection->erase(Selected);
		}
		else if(event->ControlDown() && event->RightIsDown() )
		{
			// Remove selected from m_MultipleSelection
			for(TLayerSet::TLayerList::iterator it = g_Mediator.getCurrentLayerSet()->begin(); it != g_Mediator.getCurrentLayerSet()->end(); it++)
			{
				shared_ptr<TLayer> l = (*it);
				if(l->m_Selection) l->m_Selection->erase(Selected);
			}
		}
	}

	// Update 
	m_LayerPanel->updateForLayer( g_Mediator.getCurrentLayer().get() );
	redrawCanvas();
}


void MyApp::saveScreenshot(string p_Filename)
{
	SaveScreenshot(this->getGLCanvas(), p_Filename);
}

void MyApp::copyScreenshotToClipboard()
{
	CopyScreenshotToClipboard(this->getGLCanvas());
}
