#include "stdafx.h"

#include "model.h"

#include "globals.h"

#include "settings.h"
#include "logwriter.h"
#include "ivcon.h"

#include "utils.h"
#include "layer.h"
#include "CIsoSurface.h"
#include "colormap.h"

TModel::TModel()
	: m_DoAxisRotate(false)
{}

TModel::TModel(TIndexMapper * p_Boundary, TIndexMapper * p_Inside, int p_Border)
	: m_Border(p_Border)
	, m_DoAxisRotate(false)
{
	m_VertexBuffer.reserve( p_Boundary->getMaxIndex() * 4 );
	m_IndexBuffer.reserve( m_VertexBuffer.capacity() );
	m_Normals.reserve( m_IndexBuffer.capacity() );
	unsigned int x;
	for(x=0; x<p_Boundary->getMaxIndex(); x++)
	{
		const TCoord3 p = p_Boundary->vidx2coord(x);

		TCoord3 np;
		for(np.x=p.x-1; np.x<=p.x+1; np.x++) // neighbors
		for(np.y=p.y-1; np.y<=p.y+1; np.y++)
		for(np.z=p.z-1; np.z<=p.z+1; np.z++)
		{
			if( p_Inside->vinside(np) ) continue;
			if( (p.x == np.x) + (p.y == np.y) + (p.z == np.z) != 2 ) continue; // Only 6-neighbors

			const float W = 1.0f;
			TPoint3 pa,pb,pc,pd; 
			TVector3 n;
			if( np.z < p.z ) 
			{
				pa = TPoint3(p.x   , p.y   , p.z  );
				pb = TPoint3(p.x    , p.y + W, p.z  );
				pc = TPoint3(p.x + W, p.y + W, p.z   );
				pd = TPoint3(p.x + W, p.y  , p.z  );
				n = TVector3(0.0f, 0.0f, -1.0f);
			}
			else if( np.z > p.z ) 
			{
				pd = TPoint3(p.x   , p.y   , p.z + W);
				pc = TPoint3(p.x   , p.y + W, p.z + W);
				pb = TPoint3(p.x + W, p.y + W, p.z + W);
				pa = TPoint3(p.x + W, p.y   , p.z + W);
				n = TVector3(0.0f, 0.0f, +1.0f);
			}
			else if( np.x < p.x ) 
			{
				pa = TPoint3(p.x, p.y   , p.z    );
				pb = TPoint3(p.x, p.y    , p.z + W );
				pc = TPoint3(p.x, p.y + W, p.z + W  );
				pd = TPoint3(p.x, p.y + W, p.z   );
				n = TVector3(-1.0f, 0.0f, 0.0f);
			}
			else if( np.x > p.x ) 
			{
				pd = TPoint3(p.x+W, p.y   , p.z    );
				pc = TPoint3(p.x+W, p.y    , p.z + W );
				pb = TPoint3(p.x+W, p.y + W, p.z + W  );
				pa = TPoint3(p.x+W, p.y + W, p.z   );
				n = TVector3(+1.0f, 0.0f, 0.0f);
			}
			else if( np.y < p.y ) 
			{
				pd = TPoint3(p.x, p.y   , p.z    );
				pc = TPoint3(p.x, p.y    , p.z + W );
				pb = TPoint3(p.x+W, p.y, p.z + W  );
				pa = TPoint3(p.x+W, p.y, p.z   );
				n = TVector3(0.0f, -1.0f, 0.0f);
			}
			else if( np.y > p.y ) 
			{
				pa = TPoint3(p.x  , p.y+W, p.z    );
				pb = TPoint3(p.x  , p.y+W, p.z + W );
				pc = TPoint3(p.x+W, p.y+W, p.z + W );
				pd = TPoint3(p.x+W, p.y+W, p.z   );
				n = TVector3(0.0f, +1.0f, 0.0f);
			}	
			else continue;

			unsigned int idx = m_VertexBuffer.size();
			m_VertexBuffer.push_back( pa );
			m_VertexBuffer.push_back( pb );
			m_VertexBuffer.push_back( pc );
			m_VertexBuffer.push_back( pd );

			m_IndexBuffer.push_back( idx + 0 );
			m_IndexBuffer.push_back( idx + 1 );
			m_IndexBuffer.push_back( idx + 2 );
			m_IndexBuffer.push_back( idx + 2 );
			m_IndexBuffer.push_back( idx + 3 );
			m_IndexBuffer.push_back( idx + 0 );
			m_Normals.push_back( n );
			m_Normals.push_back( n );
			m_Normals.push_back( n );
			m_Normals.push_back( n );
			m_Normals.push_back( n );
			m_Normals.push_back( n );

		}
	}
	computeMinMax();


	m_Resize = 1.0f;
	m_UseCulling = true;
}


void TModel::computeMinMax()
{
	m_MaxX = -(std::numeric_limits<float>::max)();
	m_MaxY = -(std::numeric_limits<float>::max)();
	m_MaxZ = -(std::numeric_limits<float>::max)();
	m_MinX = +(std::numeric_limits<float>::max)();
	m_MinY = +(std::numeric_limits<float>::max)();
	m_MinZ = +(std::numeric_limits<float>::max)();
	// Min/max
	for(unsigned int i=0; i<m_VertexBuffer.size(); i++)
	{
		if( m_VertexBuffer[i].x > m_MaxX ) m_MaxX = m_VertexBuffer[i].x;
		if( m_VertexBuffer[i].y > m_MaxY ) m_MaxY = m_VertexBuffer[i].y;
		if( m_VertexBuffer[i].z > m_MaxZ ) m_MaxZ = m_VertexBuffer[i].z;
		if( m_VertexBuffer[i].x < m_MinX ) m_MinX = m_VertexBuffer[i].x;
		if( m_VertexBuffer[i].y < m_MinY ) m_MinY = m_VertexBuffer[i].y;
		if( m_VertexBuffer[i].z < m_MinZ ) m_MinZ = m_VertexBuffer[i].z;
	}
}


TModel::TModel(const string & p_Filename, int p_BaseX, int p_Border)
	: m_Border(p_Border)
	, m_DoAxisRotate(true)
{
	int i;

	string Filename = p_Filename;
	bool wasCompressed = false;

	// Decompress using gzip.exe if necessary
	{
		//wxString wxFilename = p_File.c_str();
		if( Filename.substr(Filename.length() - 3,3) == ".gz"  )
		{
			decompressAndWait(Filename);
			Filename = Filename.substr(0, Filename.length() - 3);
			wasCompressed = true;
		}
	}
	
	m_MaxX = -(std::numeric_limits<float>::max)();
	m_MaxY = -(std::numeric_limits<float>::max)();
	m_MaxZ = -(std::numeric_limits<float>::max)();
	m_MinX = +(std::numeric_limits<float>::max)();
	m_MinY = +(std::numeric_limits<float>::max)();
	m_MinZ = +(std::numeric_limits<float>::max)();
	
	if( Filename.substr(Filename.length() - 4,4) == ".ply" )
	{
		readPly(Filename);

		unsigned int i;

		computeMinMax();

		m_UseCulling = true;

		// Create normals
		m_Normals.resize(m_IndexBuffer.size());
		for(i=0; i<m_IndexBuffer.size()/3; i++)
		{
			unsigned int idx1 = m_IndexBuffer[i*3+0];
			unsigned int idx2 = m_IndexBuffer[i*3+1];
			unsigned int idx3 = m_IndexBuffer[i*3+2];
			TVector3 p = m_VertexBuffer[idx1];
			TVector3 a = m_VertexBuffer[idx2];
			TVector3 b = m_VertexBuffer[idx3];
			a.subtract(p);
			b.subtract(p);
			a.cross(a,b);
			a.normalize();
			m_Normals[i*3+0] = a;
			m_Normals[i*3+1] = a;
			m_Normals[i*3+2] = a;

			/*
			// Debug
			if(i<10)
			{
				*g_Log << wxString::Format("0: %.4f, %.4f, %.4f\n", m_Normals[i*3+0].x, m_Normals[i*3+0].y, m_Normals[i*3+0].z).c_str();
				*g_Log << wxString::Format("1: %.4f, %.4f, %.4f\n", m_Normals[i*3+1].x, m_Normals[i*3+1].y, m_Normals[i*3+1].z).c_str();
				*g_Log << wxString::Format("2: %.4f, %.4f, %.4f\n", m_Normals[i*3+2].x, m_Normals[i*3+2].y, m_Normals[i*3+2].z).c_str();
			}
			*/
		}


	}
	else
	{
		string Output = "test.obj";
		char output[] = "\0";
	//	char * names[2] = {const_cast<char*>(Output.c_str()), const_cast<char*>(p_Filename.c_str()) };
		char * names[3] = {"", const_cast<char*>(Filename.c_str()), output };
		shared_ptr<TIvCon> ivcon( new TIvCon() );
		int error = ivcon->ivcon_main(2, names);
		if(error) throw string("Error reading '"+Filename+"'");
	//	*g_Log << "cor3_num: " << cor3_num << "\n";
		

		m_VertexBuffer.resize(ivcon->cor3_num);

		for(i=0; i<ivcon->cor3_num; i++)
		{
			m_VertexBuffer[i] = TPoint3(ivcon->cor3[0][i], ivcon->cor3[1][i], ivcon->cor3[2][i]);
			if( m_VertexBuffer[i].x > m_MaxX ) m_MaxX = m_VertexBuffer[i].x;
			if( m_VertexBuffer[i].y > m_MaxY ) m_MaxY = m_VertexBuffer[i].y;
			if( m_VertexBuffer[i].z > m_MaxZ ) m_MaxZ = m_VertexBuffer[i].z;
			if( m_VertexBuffer[i].x < m_MinX ) m_MinX = m_VertexBuffer[i].x;
			if( m_VertexBuffer[i].y < m_MinY ) m_MinY = m_VertexBuffer[i].y;
			if( m_VertexBuffer[i].z < m_MinZ ) m_MinZ = m_VertexBuffer[i].z;
		}

		m_IndexBuffer.resize(ivcon->face_num*3);
		m_Normals.resize(ivcon->face_num*3);
		for(i=0; i<ivcon->face_num; i++)
		{
			if( ivcon->face_order[i] != 3 ) *g_Log << "warning: face_order[i] != 3: " << ivcon->face_order[i] << "\n";
			m_IndexBuffer[i*3+0] = ivcon->face[0][i];
			m_IndexBuffer[i*3+1] = ivcon->face[1][i];
			m_IndexBuffer[i*3+2] = ivcon->face[2][i];

			m_Normals[i*3+0] = TVector3( ivcon->vertex_normal[0][0][i], ivcon->vertex_normal[1][0][i], ivcon->vertex_normal[2][0][i] );
			m_Normals[i*3+0].normalize();
			m_Normals[i*3+1] = TVector3( ivcon->vertex_normal[0][1][i], ivcon->vertex_normal[1][1][i], ivcon->vertex_normal[2][1][i] );
			m_Normals[i*3+1].normalize();
			m_Normals[i*3+2] = TVector3( ivcon->vertex_normal[0][2][i], ivcon->vertex_normal[1][2][i], ivcon->vertex_normal[2][2][i] );
			m_Normals[i*3+2].normalize();
		}

		m_UseCulling = true;
	}

	const float dx = m_MaxX-m_MinX;
	const float dy = m_MaxY-m_MinY;
	const float dz = m_MaxZ-m_MinZ;
	
	if(p_BaseX)
	{
		// y-axis in polygonal model corresponds to x-axis in voxel model
		// -2 because of 1-pixel border around voxel model
//		m_Resize = ( ((float) (p_BaseX-m_Border*2) )* MyApp::m_VoxelScale ) / dy;
		m_Resize = ( ((float) (p_BaseX-m_Border*2) ) ) / dy;
	}
	else
	{
		m_Resize = 10.0f/dx;
	}

	if(wasCompressed)
	{
		compressAndWait(Filename);
	}
};


static int vertex_i = 0;
static int vertex_j = 0;
static int face_j = 0;
TModel * static_model = 0;

static int vertex_cb(p_ply_argument argument) 
{
    long eol;
    ply_get_argument_user_data(argument, NULL, &eol);
	double d = ply_get_argument_value(argument);
	static_model->m_VertexBuffer[vertex_j].pos[vertex_i] = d;
    //*g_Log << wxString::Format("%g", d).c_str();
    if (eol) 
	{
		vertex_j++;
		vertex_i = 0;
		//*g_Log << "\n";
	}
    else 
	{
		vertex_i++;
		//*g_Log <<" ";
	}
    return 1;
}

static int face_cb(p_ply_argument argument) {
    long length, value_index;
    ply_get_argument_property(argument, NULL, &length, &value_index);
	if(value_index >= 0 && value_index <= 2) // value_index can be -1 !
	{ 
		double d = ply_get_argument_value(argument);
		static_model->m_IndexBuffer[face_j*3 + value_index] = d;
		if(value_index == 2) face_j++;
	}
    return 1;
}

void TModel::readPly(const string p_Name)
{
	vertex_i = 0;
	vertex_j = 0;
	face_j = 0;
	static_model = this;

    long nvertices, ntriangles;
    p_ply ply = ply_open(p_Name.c_str(), NULL, 0L, NULL);
    if (!ply) throw string("Couldn't read ply");
    if (!ply_read_header(ply)) throw string("Couldn't read ply");
    nvertices = ply_set_read_cb(ply, "vertex", "x", vertex_cb, NULL, 0);
    ply_set_read_cb(ply, "vertex", "y", vertex_cb, NULL, 0);
    ply_set_read_cb(ply, "vertex", "z", vertex_cb, NULL, 1);
    ntriangles = ply_set_read_cb(ply, "face", "vertex_indices", face_cb, NULL, 0);
//    *g_Log << wxString::Format("Vertices: %ld\nTriangles: %ld\n", nvertices, ntriangles).c_str();
	m_VertexBuffer.resize( nvertices );
	m_IndexBuffer.resize( ntriangles * 3 );
    if (!ply_read(ply)) throw string("Couldn't read ply");
    ply_close(ply);
	
	/*
	// Debug
	unsigned int x;
	*g_Log << "First ... vertices\n";
	for(x=0; x<20; x++)
	{
		if(x >= m_VertexBuffer.size()) break;
		*g_Log << m_VertexBuffer[x].x << " " << m_VertexBuffer[x].y << " " << m_VertexBuffer[x].z << "\n";
	}

	*g_Log << "First ... faces\n";
	for(x=0; x<20; x++)
	{
		if(x*3 >= m_IndexBuffer.size()) break;
		*g_Log << m_IndexBuffer[x*3+0] << " " << m_IndexBuffer[x*3+1] << " " << m_IndexBuffer[x*3+2] << "\n";
	}
	*/
}



void TModel::render()
{
	unsigned int i;

	
	const float dx = m_MaxX-m_MinX;
	const float dy = m_MaxY-m_MinY;
	const float dz = m_MaxZ-m_MinZ;

	if(g_Settings.m_FlatShading)
		glShadeModel(GL_FLAT);
	else
		glShadeModel(GL_SMOOTH);

	if(	m_UseCulling ) 
	{
		glEnable(GL_CULL_FACE);
		glCullFace(GL_BACK);
	}
	else glDisable(GL_CULL_FACE);

	glPushMatrix();

	glScalef(g_Settings.m_VoxelScale, g_Settings.m_VoxelScale, g_Settings.m_VoxelScale);
	glTranslatef(m_Border, m_Border, m_Border);
	if( m_DoAxisRotate )
	{
		glRotatef(-90.f, 1.0f, 0.0f, 0.0f);
		glRotatef(-90.f, 0.0f, 0.0f, 1.0f);
	}
//	glTranslatef(-m_MinX+1.0f, -m_MinY+1.0f, -m_MinZ+1.0f);

	glScalef(m_Resize, m_Resize, m_Resize);
	glTranslatef(-m_MinX, -m_MinY, -m_MinZ);


	// Render model base
	if(false)
	{
		glBegin(GL_POLYGON);
			glVertex3f(m_MinX+dx, m_MinY, m_MinZ);
			glVertex3f(m_MinX+dx, m_MinY+dy, m_MinZ);
			glVertex3f(m_MinX, m_MinY+dy, m_MinZ);
			glVertex3f(m_MinX, m_MinY, m_MinZ);
		glEnd();

		glBegin(GL_POLYGON);
			glVertex3f(m_MinX+dx, m_MinY, m_MinZ);
			glVertex3f(m_MinX+dx, m_MinY, m_MinZ+dz);
			glVertex3f(m_MinX, m_MinY, m_MinZ+dz);
			glVertex3f(m_MinX, m_MinY, m_MinZ);
		glEnd();
	}


	glBegin(GL_TRIANGLES);
	for(i=0; i<m_IndexBuffer.size(); i++)
	{
		if( m_Normals.size() > 0 ) glNormal3fv( m_Normals[ i ].pos );
		if( m_Colors.size() == m_IndexBuffer.size() ) glColor3fv( m_Colors[i].rgb );
		glVertex3fv( m_VertexBuffer[ m_IndexBuffer[i] ].pos );
	}
	glEnd();
// */
	glCullFace(GL_BACK);

	glPopMatrix();
}

















TModel * TModel::createUsingMarchingCubes(TIndexMapper * p_Boundary, TIndexMapper * p_Foreground, int p_Border)
{
	shared_ptr<TFloatField3> field( new TFloatField3(p_Foreground->getBaseIndexField()->getMaxX(), p_Foreground->getBaseIndexField()->getMaxY(), p_Foreground->getBaseIndexField()->getMaxZ()) );
	field->clear();

//	const float OBJECTVALUE = 100.0f;
	unsigned int x;
	for(x=0; x<p_Foreground->getMaxIndex(); x++)
	{
		field->valuep( p_Foreground->vidx2coord(x) ) = 1.0f;
	}
	// Slightly inflate the shape to make smoother isosurface
	/*
	TFloatField3 * Image = field.get();
	unsigned int k;
	for(x=0; x<p_Foreground->getBaseIndexField()->getMaxX(); x++)
	for(y=0; y<p_Foreground->getBaseIndexField()->getMaxY(); y++)
	for(z=0; z<p_Foreground->getBaseIndexField()->getMaxZ(); z++)
	{
		Image->value(x,y,z) = 1000.0f;	
	}
	
	for(k=0; k<p_Foreground->getMaxIndex(); k++)
	{
		const int W = 3;

		const TCoord3 p = p_Foreground->vidx2coord(k);
		int a,b,c;
		for(a=p.x-W; a<=p.x+W; a++)
		for(b=p.y-W; b<=p.y+W; b++)
		for(c=p.z-W; c<=p.z+W; c++)
		{
			if( a < 0 || b < 0 || c < 0 || a >= field->getMaxX() || b >= field->getMaxY() || c >= field->getMaxZ() ) 
				continue;

			const TCoord3 q(a,b,c);
			const float newvalue = p.distance(q);
			const float oldvalue = Image->valuep(q);

			if(newvalue < oldvalue)
			{
				Image->valuep(q) = newvalue; 
			}
		}
	}
	shared_ptr<TLayer> NewLayer( new TLayer(field, "create mesh field") );
	g_Mediator.addLayer( NewLayer );
	*/


	CIsoSurface<float> surface;
	surface.GenerateSurface( field->data(), 0.5f, p_Foreground->getBaseIndexField()->getMaxX()-1, p_Foreground->getBaseIndexField()->getMaxY()-1, p_Foreground->getBaseIndexField()->getMaxZ()-1, 1.0f, 1.0f, 1.0f );
	if(!g_Settings.m_ReleaseMode) *g_Log << "Constructed surface is valid: " << surface.IsSurfaceValid() << ", vertices: " << surface.m_nVertices << ", triangles: " << surface.m_nTriangles << "\n";

	TModel * model = new TModel();
	model->m_Border = p_Border;
	model->m_VertexBuffer.resize( surface.m_nVertices );
	for(x=0; x<surface.m_nVertices; x++)
	{
		model->m_VertexBuffer[x] = surface.m_ppt3dVertices[x];
	}
	model->computeMinMax();
	model->m_IndexBuffer.resize( surface.m_nTriangles*3 );
	for(x=0; x<surface.m_nTriangles*3; x+=3 )
	{
		model->m_IndexBuffer[x] = surface.m_piTriangleIndices[x+2];
		model->m_IndexBuffer[x+1] = surface.m_piTriangleIndices[x+1];
		model->m_IndexBuffer[x+2] = surface.m_piTriangleIndices[x];
	}
	
	model->m_Normals.resize( model->m_IndexBuffer.size() );
	for(x=0; x<model->m_IndexBuffer.size(); x++)
	{
		model->m_Normals[x] = TVector3(surface.m_pvec3dNormals[model->m_IndexBuffer[x] ]);
		model->m_Normals[x].scale(+1.0f);
	}
	model->m_Resize = 1.0f;
	model->m_UseCulling = true;

	return model;
}


void TModel::computeColorsUsingSegmentation(TTypedFieldInterface<unsigned int>* p_Input)
{
	if(p_Input == 0) return;

	TModel * model = this;
	model->m_Colors.resize( model->m_IndexBuffer.size() );
	



	glPushMatrix();
	glMatrixMode(GL_MODELVIEW);  
	glLoadIdentity();

	glTranslatef(m_Border, m_Border, m_Border);
	if( m_DoAxisRotate )
	{
		glRotatef(-90.f, 1.0f, 0.0f, 0.0f);
		glRotatef(-90.f, 0.0f, 0.0f, 1.0f);
	}
	glScalef(m_Resize, m_Resize, m_Resize);
	glTranslatef(-m_MinX, -m_MinY, -m_MinZ);


	GLfloat matrix[16];

	// Render model base
	vector<unsigned int> Labels;
	Labels.resize( m_IndexBuffer.size(), 0 );


	for(unsigned int i=0; i<m_IndexBuffer.size(); i+=3)
	{
		TVector3 pv(0.0f, 0.0f, 0.0f);

		// Point i + 0
		{
			const unsigned int j = i + 0;
			glPushMatrix();
			const TPoint3 q = m_VertexBuffer[ m_IndexBuffer[j] ].pos;
			glTranslatef(q.x,q.y,q.z);
			glGetFloatv(GL_MODELVIEW_MATRIX, matrix);
			pv.add( TPoint3( matrix[12], matrix[13], matrix[14] ) );
			glPopMatrix();
		}
		// Point i + 1
		{
			const unsigned int j = i + 1;
			glPushMatrix();
			const TPoint3 q = m_VertexBuffer[ m_IndexBuffer[j] ].pos;
			glTranslatef(q.x,q.y,q.z);
			glGetFloatv(GL_MODELVIEW_MATRIX, matrix);
			pv.add( TPoint3( matrix[12], matrix[13], matrix[14] ) );
			glPopMatrix();
		}		
		// Point i + 2
		{
			const unsigned int j = i + 2;
			glPushMatrix();
			const TPoint3 q = m_VertexBuffer[ m_IndexBuffer[j] ].pos;
			glTranslatef(q.x,q.y,q.z);
			glGetFloatv(GL_MODELVIEW_MATRIX, matrix);
			pv.add( TPoint3( matrix[12], matrix[13], matrix[14] ) );
			glPopMatrix();
		}
		pv.scale(1.0f/3.0f);

		const TCoord3 p( pv.x, pv.y, pv.z );
		
		// Find nearest voxel in p_Input
		TCoord3 np;
		const char W = 1;
		float closest = 10000.0f;
		TColor3 closestcolor(1.0f, 1.0f, 1.0f);
		for(np.x=p.x-W; np.x<=p.x+W; np.x++) // neighbors
		for(np.y=p.y-W; np.y<=p.y+W; np.y++)
		for(np.z=p.z-W; np.z<=p.z+W; np.z++)
		{
			if(!p_Input->getIndexField()->vinside(np)) continue;

			if(np.distance(p) < closest)
			{	
				p_Input->getLayer()->m_ColorMap->getColor(np, &closestcolor);
			}
		}
		//Labels[i] = closestlabel;
		model->m_Colors[i+0] = closestcolor;
		model->m_Colors[i+1] = closestcolor;
		model->m_Colors[i+2] = closestcolor;
	}
	glPopMatrix();



}


