#include "mullikin.h"

/*
  Note:
  There is an error in Cuisenaire's thesis, in section 5.3 in the SSED algorithm, the lines
  test(p,(-1,0)) 
  and
  test(p,(0,-1)) 
  need to be swapped.

 */ 

#include "Geometry.h"
#include "field.h"

void TMullikinDt::test2(const int px, const int py, const int nx, const int ny)
{
	if ( m_FlagField->inside(px,py)  
		&& px+nx>=0 && py+ny>=0 && px+nx<M && py+ny<N )
	{
		TOrigSet2 & Sp = m_Origins->value(px,py);  // this origin set
		const TOrigSet2 & Sa = m_Origins->value(px+nx, py+ny); // neighbor origin set

		TOrigSet2::const_iterator it, lt;
		TOrigSet2::iterator  jt;
		// Insert S(a) into S(p)
		for(it = Sa.begin(); it != Sa.end(); it++)
		{
			// Only insert 'it' if it does not occur in S(p)
			const ARITHTYPE d = TCoord2(px,py).ARITHDISTANCE(it->second);
			bool found = false;
			for(lt = Sp.lower_bound(d); lt != Sp.end(); lt++)
			{
				if(lt->first > d) break;
				if(lt->second == it->second) { found = true; break; }
			}
			if(!found) Sp.insert( TOrigSet2::value_type(d,it->second) );
		}

		// Prune C: remove all origins at larger distance than d_min + threshold
		const ARITHTYPE d_min = Sp.begin()->first;
#ifdef FLOAT_ARITHMETIC
		jt = Sp.upper_bound(d_min + m_Tolerance);
#else
		const float d_min_root = sqrtf(d_min);
		jt = Sp.upper_bound(d_min + 2*m_Tolerance*d_min_root + m_Tolerance2);
#endif
		Sp.erase(jt, Sp.end());
	}
}

void TMullikinDt::stageInit()
{
	M = m_Image->getMaxX();
	N = m_Image->getMaxY();
	m_Distance.reset( new TFloatField3( m_Image->getMaxX(), m_Image->getMaxY(), m_Image->getMaxZ()) );
	m_FlagField.reset( TFlagField3::construct(m_Image, m_Distance.get(), 0) );
	m_Origins.reset( new TOrigSetField2(m_FlagField->dimX(),m_FlagField->dimY()) );

	m_Tolerance2 = (int)(m_Tolerance*m_Tolerance+0.5f); 
}

void TMullikinDt::stageMain() 
{
	if(m_AlreadyPerformed) throw string("Edt already performed"); 

	int i,j;
	for(i=0; i<M; i++)
	{
		for(j=0; j<N; j++)
		{
			if(m_FlagField->narrowband(i,j)) // points on boundary
			{
				m_Origins->value(i,j).insert( TOrigSet2::value_type(0, TCoord2(i,j)) );
			}
		}
	}


	if(m_NeighborCount == 4)
	{
		// forward scan
		for(j=0; j<N; j++)
		{
			for(i=0; i<M; i++)
			{
				test2(i,j,0,-1);
				test2(i,j,-1,0);
			}
			for(i=M-1; i>=0; i--)
			{
				test2(i,j,1,0);
			}
		}

		// backward scan
		for(j=N-1; j>=0; j--)
		{
			for(i=0; i<M; i++)
			{
				test2(i,j,0,1);
				test2(i,j,-1,0);
			}

			for(i=M-1; i>=0; i--)
			{
				test2(i,j,1,0);
			}
		}
	}
	else if(m_NeighborCount == 8)
	{
		// forward scan
		for(j=0; j<N; j++)
		{
			for(i=0; i<M; i++)
			{
				test2(i,j,-1,-1);
				test2(i,j, 0,-1);
				test2(i,j, 1,-1);
				test2(i,j,-1,0);
			}
			for(i=M-1; i>=0; i--)
			{
				test2(i,j, 1,0);
			}
		}

		// backward scan
		for(j=N-1; j>=0; j--)
		{
			for(i=0; i<M; i++)
			{
				test2(i,j,-1, 1);
				test2(i,j, 0, 1);
				test2(i,j, 1, 1);
				test2(i,j,-1, 0);
			}

			for(i=M-1; i>=0; i--)
			{
				test2(i,j, 1, 0);
			}
		}
	}
	else throw string("Invalid neighborcount");

	
	m_AlreadyPerformed = true;
}

void TMullikinDt::stageEnd() 
{
	for(int i=0; i<M; i++)
	{
		for(int j=0; j<N; j++)
		{
			m_Distance->value(i,j) = (m_Origins->value(i,j).size())? m_Origins->value(i,j).begin()->second.distance(TCoord2(i,j)) : -1; 
		}
	}
}


string TMullikinDt::getLatexName() const
{
	string sTolerance = "?";
	if(m_Tolerance <= 0.01f) { sTolerance = "0"; }
	else if(fabs(m_Tolerance-1.0f) <= 0.01f) { sTolerance = "1"; }
	else if(fabs(m_Tolerance-sqrt(2.0f)/2.0f) <= 0.01f) { sTolerance = "\\frac{1}{2}\\sqrt{2}"; }
	else if(fabs(m_Tolerance-sqrt(2.0f) <= 0.01f)) { sTolerance = "\\sqrt{2}"; }

	string r = wxString::Format("\\mcdt{}%i $\\epsilon%s$", m_NeighborCount, sTolerance.c_str());
	return r;
}



string TMullikinDt::getShortName() const
{
	string sTolerance = "?";
	if(m_Tolerance <= 0.01f) { sTolerance = "0"; }
	else if(fabs(m_Tolerance-1.0f) <= 0.01f) { sTolerance = "1"; }
	else if(fabs(m_Tolerance-sqrt(2.0f)/2.0f) <= 0.01f) { sTolerance = "0.71"; }
	else if(fabs(m_Tolerance-sqrt(2.0f) <= 0.01f)) { sTolerance = "1.41"; }

	string r = wxString::Format("MTFT%i e%s", m_NeighborCount, sTolerance.c_str());
	return r;
}

void TMullikinDt::deleteFields() 
{
	//m_FlagField.reset();
	//m_Distance.reset();
	//m_Origins.reset();
}