#include "stdafx.h"

#include "parallelskeletonizer.h"

#include "danielsson3d.h"
#include "skeletonizerthread.h"
#include "utils.h"

SkeletonizerThreadContext::SkeletonizerThreadContext(std::shared_ptr<TIndexField3> indexField)
	: indexField(indexField)
{
	extendedFT.reset(new std::vector<const TIndexedOrigins_Set>());
	extendedFT->resize(indexField->getMaxIndex());

	shortestPathSetField.reset(new TShortestPathSetField(indexField));

	double minCachedPathLength = 10.0;
	int maxGeodesicCacheSize = 40000000;
	geodesicCache.reset(new GeodesicCache(minCachedPathLength, maxGeodesicCacheSize));

	curveSkeletonIndexField.reset(new TSubIndexMapper(indexField.get()));
	importanceMeasureField.reset(new TSparseFloatField3(indexField));

	angleField.reset(new TSparseFloatField3(indexField));
	lengthField.reset(new TSparseUintField3(indexField));
}

std::shared_ptr<std::vector<const TIndexedOrigins_Set>> SkeletonizerThreadContext::GetExtendedFT()
{
	return extendedFT;
}

std::shared_ptr<const std::vector<const TIndexedOrigins_Set>> SkeletonizerThreadContext::GetExtendedFT() const
{
	return extendedFT;
}

std::shared_ptr<TShortestPathSetField> SkeletonizerThreadContext::GetShortestPathSetField()
{
	return shortestPathSetField;
}

std::shared_ptr<const TShortestPathSetField> SkeletonizerThreadContext::GetShortestPathSetField() const
{
	return shortestPathSetField;
}

std::shared_ptr<GeodesicCache> SkeletonizerThreadContext::GetGeodesicCache()
{
	return geodesicCache;
}

void SkeletonizerThreadContext::AddCurveSkeletonPoint(unsigned int baseIndex)
{
	wxMutexLocker lock(curveSkeletonMutex);

	TCoord3 point = indexField->vidx2coord(baseIndex);
	unsigned int curveSkeletonIndex = curveSkeletonIndexField->m_SubIndex.size();

	curveSkeletonIndexField->m_Coord2IdxMap.insert(pair<TCoord3, unsigned int>(point, curveSkeletonIndex));
	curveSkeletonIndexField->m_SubIndex.push_back(baseIndex);
}

std::shared_ptr<const TSubIndexMapper> SkeletonizerThreadContext::GetCurveSkeletonIndexField() const
{
	return curveSkeletonIndexField;
}

std::shared_ptr<TSparseFloatField3> SkeletonizerThreadContext::GetImportanceMeasureField()
{
	return importanceMeasureField;
}

std::shared_ptr<const TSparseFloatField3> SkeletonizerThreadContext::GetImportanceMeasureField() const
{
	return importanceMeasureField;
}

std::shared_ptr<TSparseFloatField3> SkeletonizerThreadContext::GetAngleField()
{
	return angleField;
}

std::shared_ptr<const TSparseFloatField3> SkeletonizerThreadContext::GetAngleField() const
{
	return angleField;
}

std::shared_ptr<TSparseUintField3> SkeletonizerThreadContext::GetLengthField()
{
	return lengthField;
}

std::shared_ptr<const TSparseUintField3> SkeletonizerThreadContext::GetLengthField() const
{
	return lengthField;
}

ParallelSkeletonizer::ParallelSkeletonizer(std::string fileName, const wxPropertyGridInterface* properties)
	: fileName(fileName), removedVoxels(0), acceptedVoxels(0)
{
	scale = properties->GetPropertyValueAsDouble("scale");
	dilationDistance = properties->GetPropertyValueAsDouble("dilationDistance");
	computeAngle = properties->GetPropertyValueAsBool("computeAngle");
	computeImportance = properties->GetPropertyValueAsBool("computeImportance");
	tau = properties->GetPropertyValueAsDouble("tau");
	maxCosine = std::cos((std::max)(properties->GetPropertyValueAsInt("minAngle"), 0) / 180.0 * M_PI); 
	acceptCosine = std::cos((std::min)(properties->GetPropertyValueAsInt("acceptAngle"), 180) / 180.0 * M_PI);  
	maxThreads = properties->GetPropertyValueAsInt("maxThreads");
}

double ParallelSkeletonizer::GetDilationDistance() const
{
	return dilationDistance;
}

bool ParallelSkeletonizer::ShouldComputeAngle() const
{
	return computeAngle;
}

double ParallelSkeletonizer::GetTau() const
{
	return tau;
}

double ParallelSkeletonizer::GetMaxCosine() const
{
	return maxCosine;
}

double ParallelSkeletonizer::GetAcceptCosine() const
{
	return acceptCosine;
}

bool ParallelSkeletonizer::ShouldComputeImportance() const
{
	return computeImportance;
}

void ParallelSkeletonizer::Perform()
{
	wxLogMessage("Initializing fields");
	InitializeFields();

	wxLogMessage("Resolution %dx%dx%d", inputField->getMaxX(), inputField->getMaxY(), inputField->getMaxZ());
	wxLogMessage("Object voxels %u", indexField->getMaxIndex());
	wxLogMessage("Boundary voxels %u", boundaryField->getMaxIndex());

	wxLogMessage("Initializing adjacency lists");
	InitializeAdjacencyLists();

	wxLogMessage("Computing feature transform");

	wxStopWatch watch;
	ComputeFeatureTransform();
	wxLogMessage("Computing feature transform took %ld ms", watch.Time());

	int threads = (std::max)(1, (std::min)(maxThreads, wxThread::GetCPUCount()));

	wxLogMessage("Parallel processing of all voxels on %d thread(s)", threads);

	watch.Start();
	ProcessInParallel(threads);
	wxLogMessage("Parallel processing took %ld ms", watch.Time());

	wxLogMessage("Curve skeleton voxels %d", context->GetCurveSkeletonIndexField()->getMaxIndex());
	wxLogMessage("Removed by angle test %d", removedVoxels);
	wxLogMessage("Accepted by angle test %d", acceptedVoxels);

	wxLogMessage("Peak memory usage %u MB", getProcessInfo().PeakWorkingSetSize / 1024u / 1024u);
}

std::shared_ptr<const TUnsignedCharField3> ParallelSkeletonizer::GetInputField() const
{
	return inputField;
}

std::shared_ptr<const TIndexField3> ParallelSkeletonizer::GetIndexField() const
{
	return indexField;
}

std::shared_ptr<const TSubIndexMapper> ParallelSkeletonizer::GetBoundaryField() const
{
	return boundaryField;
}

std::shared_ptr<const std::vector<const TDeltaOmega::TAdjacencyStruct>> ParallelSkeletonizer::GetAdjacencyLists() const
{
	return adjacencyLists;
}

std::shared_ptr<const std::vector<const TIndexedOrigins_Vector>> ParallelSkeletonizer::GetFeatureTransform() const
{
	return featureTransform;
}

std::shared_ptr<const std::vector<const TIndexedOrigins_Set>> ParallelSkeletonizer::GetExtendedFT() const
{
	return context->GetExtendedFT();
}

std::shared_ptr<const TShortestPathSetField> ParallelSkeletonizer::GetShortestPathSetField() const
{
	return context->GetShortestPathSetField();
}

std::shared_ptr<const TSubIndexMapper> ParallelSkeletonizer::GetCurveSkeletonIndexField() const
{
	return context->GetCurveSkeletonIndexField();
}

std::shared_ptr<const TSparseFloatField3> ParallelSkeletonizer::GetImportanceMeasureField() const
{
	return context->GetImportanceMeasureField();
}

std::shared_ptr<const TSparseFloatField3> ParallelSkeletonizer::GetAngleField() const
{
	return context->GetAngleField();
}

std::shared_ptr<const TSparseUintField3> ParallelSkeletonizer::GetLengthField() const
{
	return context->GetLengthField();
}

void ParallelSkeletonizer::InitializeFields()
{
	std::shared_ptr<TUnsignedCharField3> rawVolume(static_cast<TUnsignedCharField3*>(TFieldFactory::constructFromFile(fileName)));

	if (std::abs(scale - 1.0f) >= .1)
	{
		wxLogMessage("Scaling input field by a factor %.1lf", scale);
		std::unique_ptr<TUnsignedCharField3> scaled(::scale(rawVolume.get(), scale));
		inputField.reset(::crop(scaled.get()));
	}
	else inputField = rawVolume;

	indexField.reset(new TIndexField3(inputField->getMaxX(), inputField->getMaxY(), inputField->getMaxZ()));
	boundaryField.reset(new TSubIndexMapper(indexField.get()));

	SetObjectVoxels();
	SetBoundaryVoxels();
	RemoveSurroundedVoxels();
	IndexVoxels();

	context.reset(new SkeletonizerThreadContext(indexField));
}

void ParallelSkeletonizer::SetObjectVoxels()
{
	for (unsigned int z = 0; z < inputField->getMaxZ(); z++)
	{
		for (unsigned int y = 0; y < inputField->getMaxY(); y++)
		{
			for (unsigned int x = 0; x < inputField->getMaxX(); x++)
			{
				unsigned char inputValue = inputField->value(x, y, z);

				if (inputValue != 0)
				{
					indexField->value(x, y, z) = 0;
				}
			}
		}
	}
}

void ParallelSkeletonizer::SetBoundaryVoxels()
{
	for (unsigned int z = 0; z < indexField->getMaxZ(); z++)
	{
		for (unsigned int y = 0; y < indexField->getMaxY(); y++)
		{
			for (unsigned int x = 0; x < indexField->getMaxX(); x++)
			{
				unsigned int indexValue = indexField->value(x, y, z);

				if (indexValue == 0)
				{
					if (indexField->value(x - 1, y, z) == TIndexMapper::OUTSIDE ||
						indexField->value(x + 1, y, z) == TIndexMapper::OUTSIDE ||
						indexField->value(x, y - 1, z) == TIndexMapper::OUTSIDE ||
						indexField->value(x, y + 1, z) == TIndexMapper::OUTSIDE ||
						indexField->value(x, y, z - 1) == TIndexMapper::OUTSIDE ||
						indexField->value(x, y, z + 1) == TIndexMapper::OUTSIDE)
					{
						indexField->value(x, y, z) = TIndexMapper::BOUNDARY;
					}
				}
			}
		}
	}
}

void ParallelSkeletonizer::RemoveSurroundedVoxels()
{
	std::vector<TCoord3> remove;

	for (unsigned int z = 0; z < indexField->getMaxZ(); z++)
	{
		for (unsigned int y = 0; y < indexField->getMaxY(); y++)
		{
			for (unsigned int x = 0; x < indexField->getMaxX(); x++)
			{
				unsigned int indexValue = indexField->value(x, y, z);

				if (indexValue == TIndexMapper::BOUNDARY)
				{
					if (indexField->value(x - 1, y, z) >= TIndexMapper::BOUNDARY &&
						indexField->value(x + 1, y, z) >= TIndexMapper::BOUNDARY &&
						indexField->value(x, y - 1, z) >= TIndexMapper::BOUNDARY &&
						indexField->value(x, y + 1, z) >= TIndexMapper::BOUNDARY &&
						indexField->value(x, y, z - 1) >= TIndexMapper::BOUNDARY &&
						indexField->value(x, y, z + 1) >= TIndexMapper::BOUNDARY)
					{
						remove.push_back(TCoord3(x, y, z));
					}
				}
			}
		}
	}

	for (std::vector<TCoord3>::const_iterator i = remove.begin(); i != remove.end(); i++)
	{
		TCoord3 point = *i;
		indexField->valuep(point) = TIndexMapper::OUTSIDE;
	}
}

void ParallelSkeletonizer::IndexVoxels()
{
	for (unsigned int z = 0; z < indexField->getMaxZ(); z++)
	{
		for (unsigned int y = 0; y < indexField->getMaxY(); y++)
		{
			for (unsigned int x = 0; x < indexField->getMaxX(); x++)
			{
				TCoord3 point(x, y, z);
				unsigned int indexValue = indexField->valuep(point);

				if (indexValue != TIndexMapper::OUTSIDE)
				{
					indexField->m_Voxels.push_back(point);

					unsigned int baseIndex = indexField->m_Voxels.size() - 1;
					indexField->valuep(point) = baseIndex;

					if (indexValue == TIndexMapper::BOUNDARY)
					{
						boundaryField->m_SubIndex.push_back(baseIndex);

						unsigned int boundaryIndex = boundaryField->m_SubIndex.size() - 1;
						boundaryField->m_Coord2IdxMap.insert(std::pair<TCoord3, unsigned int>(point, boundaryIndex));
					}
				}
			}
		}
	}
}

void ParallelSkeletonizer::InitializeAdjacencyLists()
{
	adjacencyLists.reset(new std::vector<const TDeltaOmega::TAdjacencyStruct>());
	adjacencyLists->resize(boundaryField->getMaxIndex());

	int emptyWarnings = 0;

	for (unsigned int i = 0; i < adjacencyLists->size(); i++)
	{
		unsigned int edges = 0;
		const TCoord3& point = boundaryField->vidx2coord(i);

		for (int z = -1; z <= 1; z++)
		{
			for (int y = -1; y <= 1; y++)
			{
				for (int x = -1; x <= 1; x++)
				{
					TCoord3 neighbourPoint(point.x + x, point.y + y, point.z + z);
					unsigned int neighbourIndex = boundaryField->vcoord2idx(neighbourPoint);

					if (neighbourIndex == TIndexMapper::OUTSIDE || neighbourIndex == i ||
						!HaveCommonInsideVoxel(point, neighbourPoint))
					{
						continue;
					}

					assert(edges < TDeltaOmega::MAXNSIZE);

					adjacencyLists->at(i).m_Edges[edges].m_ToVertex = neighbourIndex;
					adjacencyLists->at(i).m_Edges[edges].m_Weight =
						2 - (neighbourPoint.x == point.x) - (neighbourPoint.y == point.y) - (neighbourPoint.z == point.z);

					edges++;
				}
			}
		}

		if (edges == 0) emptyWarnings++;
		adjacencyLists->at(i).m_EdgeCount = edges;
	}

	if (emptyWarnings > 0)
		wxLogWarning("Empty adjacency list for %d boundary voxels", emptyWarnings);
}

bool ParallelSkeletonizer::HaveCommonInsideVoxel(const TCoord3& point, const TCoord3& neighbourPoint)
{
	for (int z = (std::min)(point.z, neighbourPoint.z) - 1; z <= (std::max)(point.z, neighbourPoint.z) + 1; z++)
	{
		for (int y = (std::min)(point.y, neighbourPoint.y) - 1; y <= (std::max)(point.y, neighbourPoint.y) + 1; y++)
		{
			for (int x = (std::min)(point.x, neighbourPoint.x) - 1; x <= (std::max)(point.x, neighbourPoint.x) + 1; x++)
			{
				TCoord3 candidate(x, y, z);

				if (std::abs(candidate.x - point.x) <= 1 &&
					std::abs(candidate.y - point.y) <= 1 &&
					std::abs(candidate.z - point.z) <= 1 && 
					std::abs(candidate.x - neighbourPoint.x) <= 1 &&
					std::abs(candidate.y - neighbourPoint.y) <= 1 &&
					std::abs(candidate.z - neighbourPoint.z) <= 1 && 
					!boundaryField->vinside(candidate) && indexField->vinside(candidate))
				{
					return true;
				}
			}
		}
	}

	return false;
}

void ParallelSkeletonizer::ComputeFeatureTransform()
{
	unsigned int maxFeaturePoints = 1;
	TDanielsson3d algorithm(indexField.get(), boundaryField.get(), maxFeaturePoints);
	algorithm.perform();

	featureTransform.reset(new std::vector<const TIndexedOrigins_Vector>());
	for (std::vector<TIndexedOrigins_Vector>::const_iterator i = algorithm.m_Origins.begin();
		i != algorithm.m_Origins.end(); i++)
	{
		const TIndexedOrigins_Vector& points = *i;
		featureTransform->push_back(points);
	}
}

void ParallelSkeletonizer::ProcessInParallel(int threadCount)
{
	std::vector<std::shared_ptr<SkeletonizerThread>> threads;

	for (int i = 0; i < threadCount; i++)
	{
		std::shared_ptr<SkeletonizerThread> thread(new SkeletonizerThread(this, context, i, threadCount));
		threads.push_back(thread);

		if (thread->Create() != wxTHREAD_NO_ERROR)
			wxLogError("Failed to create thread %d", i);
		else if (thread->Run() != wxTHREAD_NO_ERROR)
			wxLogError("Failed to run thread %d", i);
	}

	removedVoxels = 0, acceptedVoxels = 0;

	for (std::vector<std::shared_ptr<SkeletonizerThread>>::const_iterator i = threads.begin(); i != threads.end(); i++)
	{
		std::shared_ptr<SkeletonizerThread> thread = *i;

		if (thread->Wait() != 0)
			wxLogError("Thread %d faulted", thread->GetThreadID());

		removedVoxels += thread->GetRemovedVoxels();
		acceptedVoxels += thread->GetAcceptedVoxels();
	}
}
