#include "stdafx.h"

#include "povray.h"

#include "settings.h"
#include "field.h"
#include "abstractfilter.h"
#include "layer.h"
#include "colormap.h"
#include "main.h"
#include "camera2.h"
#include "skeletonizer.h"
#include "model.h"
#include "globals.h"

#include <map>
using namespace std;

TPovray::TPovray(class TSkeletonizer * p_Skeletonizer)
	: m_Skeletonizer(p_Skeletonizer)
{

}

void TPovray::writeLayerSet(const string & p_OutputFilename, TLayerSet * p_LayerSet, TLayer * p_Layer)
{
	std::ofstream f( p_OutputFilename.c_str() );

	f << 
	"// Skeletonizer output of model '"+m_Skeletonizer->m_InputFilename+"'\n"
	"// Copyright 2006 Dennie Reniers, D.Reniers@tue.nl, http://www.win.tue.nl/~dreniers\n"
	"#version unofficial megapov 1.21;\n"

	"#declare THRESHOLD = 0.9;\n"
	"#declare RENDER_MESH = 1;\n";

				
	
	f<<"#macro BLOBPOINT ( P_POS, P_COLOR )\n"
	"sphere { P_POS, 0.15, 1 pigment { color rgb P_COLOR } finish { diffuse 1 specular 0.3 } }\n"
	"#end\n"
	"\n"
	"#macro BOXPOINT( P_POS, P_COLOR )\n"
	"box { P_POS-<0.05,0.05,0.05>, P_POS+<0.05,0.05,0.05> pigment { color rgb P_COLOR } finish { diffuse 1 specular 0.3 } }\n"
	"#end\n"
	"\n"
	"#macro SPHEREPOINT(P_POS, P_COLOR, P_RADIUS)\n"
	"sphere { P_POS, 0.3 pigment { color rgb P_COLOR } finish { diffuse 1 specular 0.3 } }\n"
	"#end\n";

	std::map<shared_ptr<TLayer>, bool> IsBlobMap;

	unsigned int x=1;
	TLayerSet::TLayerList::iterator it;
	if(p_LayerSet)
	{
		for(it = p_LayerSet->begin(); it != p_LayerSet->end(); it++)
		{
			if( (*it)->getCheckedForRendering())
			{
				unsigned int Type = 1;
				//float BlobRadius = 0.3f;
				if( 
					(*it)->m_ShortName == "Junctions" 
					|| (*it)->m_ShortName == "Candidate critical points" 
					|| (*it)->m_ShortName == "Final critical points" 
					)
				{
					Type = 2;
				}
				else if(
					(*it)->m_Field.get() == g_Mediator.getSkeletonizer()->m_SurfaceSkeletonField
					|| (*it)->m_Field.get() == g_Mediator.getSkeletonizer()->m_CurveSkeletonField.get()
					|| (*it)->m_Field.get() == g_Mediator.getSkeletonizer()->m_SkeletonField.get()
					 )
				{
					Type = 1;
				}
				else if((*it)->m_Field->getIndexField().get() == g_Mediator.getSkeletonizer()->m_BoundaryIndexField.get() )
				{
					Type = 2;
				}

				f << "#declare RENDER_" << x << " = " << Type << "; // " << (*it)->m_Name << "\n";


				f << "#macro B"<<x<<"( P_POS, P_COLOR )\n";
				f << "BLOBPOINT(P_POS,P_COLOR)\n";
				f << "#end\n";

				f << "#macro R"<<x<<"( P_POS, P_COLOR )\n";
				f << "BOXPOINT(P_POS,P_COLOR)\n";
				f << "//SPHERE(P_POS,P_COLOR,0.3)\n";
				f << "#end\n";
				f << "\n";
				x++;
			}
		}
	}


	f<<
	"#declare ROTATION = (clock + 0.1) * 2*pi;\n"
//	"#declare R = 0.3;\n"
//	"#declare R_C = 0.15;\n"
//	"#declare R_S = 0.15;\n"
//	"#declare R_B = 0.3;\n"
	
	"#declare COLOR = color rgb <0.7,0.7,0.7>;\n"
	"#declare SPEC = 0.3;\n"
	"background{ rgb <1,1,1> }\n"
	"global_settings { max_trace_level 200 }\n"
	"camera {\n"
	"angle 40\n"
	;

	TVector3 CamLoc = MyApp::instance()->m_CurrentCamera->getPos();
	TVector3 CamLookAt = MyApp::instance()->m_CurrentCamera->getLookAt();

	f << wxString::Format( "location <%f,%f,%f>\n", CamLoc.x, CamLoc.y, CamLoc.z).c_str();
	f << wxString::Format( "look_at <%f,%f,%f>\n", CamLookAt.x, CamLookAt.y, CamLookAt.z).c_str();

	// Two things are done with the following line.
	// 1. make right-handedness like opengl
	// 2. use -1 instead of -4/3 for square pixels, is needed because perspective angle for opengl is along y-axis, 
	// while for pov-ray it is along x-axis. With square pixels this doesn't matter.
	// 
	f << "right <-1,0,0>\n"; 
	f << "}\n";		

	f << wxString::Format( "light_source { <%f,%f,%f> color rgb <1,1,1> } \n", CamLoc.x, CamLoc.y, CamLoc.z).c_str();


	x = 1;
	if(p_LayerSet)
	{
		for(it = p_LayerSet->begin(); it != p_LayerSet->end(); it++)
		{
			shared_ptr<TLayer> Layer = *it;
			if( (*it)->getCheckedForRendering())
			{
				f << "#if (RENDER_" << x << "= 1)\n";
				f << "blob { threshold THRESHOLD\n";
				writePovrayField(&f, Layer.get(), x, true );
				f<< "pigment { color rgb <1.0,1.0,1.0> }\n"
				"no_shadow\n"
				"}\n";
				f << "#end\n";
				f << "#if (RENDER_" << x << "= 2)\n";
				writePovrayField(&f, Layer.get(), x, false );
				f << "\n#end\n";
			
				x++;
			}

		}	
	}
	else if(p_Layer)
	{
		throw string("Doesn't support");
//		f << "#if (RENDER_" << x << ")\n";
//		writePovrayField(&f, p_Layer,  );
//		x++;
//		f << "#end\n";
	}


	// Output mesh
	if(m_Skeletonizer->m_Model)
	{	
		const TModel * Model = m_Skeletonizer->m_Model.get();
		f << 
		"#if (RENDER_MESH)\n"
		"mesh {\n"
		;
		for(unsigned int i=0; i<Model->m_IndexBuffer.size(); i=i+3)
		{
			const TPoint3 & a = Model->m_VertexBuffer[ Model->m_IndexBuffer[i+0] ];
			const TVector3 & na = Model->m_Normals[ i+0 ];
			const TPoint3 & b = Model->m_VertexBuffer[ Model->m_IndexBuffer[i+1] ];
			const TVector3 & nb = Model->m_Normals[ i+1 ];
			const TPoint3 & c = Model->m_VertexBuffer[ Model->m_IndexBuffer[i+2] ];
			const TVector3 & nc = Model->m_Normals[ i+2 ];
			f << wxString::Format("smooth_triangle { <%.3f,%.3f,%.3f> <%.3f,%.3f,%.3f> <%.3f,%.3f,%.3f> <%.3f,%.3f,%.3f> <%.3f,%.3f,%.3f> <%.3f,%.3f,%.3f> }\n"
				,a.x,a.y,a.z
				,na.x, na.y, na.z 
				,b.x,b.y,b.z
				,nb.x, nb.y, nb.z 
				,c.x,c.y,c.z
				,nc.x, nc.y, nc.z 
				).c_str();
		}
		f << "translate <" << -Model->m_MinX << ", " << -Model->m_MinY << ", " << -Model->m_MinZ << ">\n";
		f << "scale " << (Model->m_Resize * g_Settings.m_VoxelScale) << "\n";
		f << "rotate <-90.0, 0.0, 0.0> \n";
		f << "rotate <0.0, -90.0, 0.0> \n";
//			f << "pigment { color rgb <1,1,1> transmit .9 }\n";
		
		f << "pigment {\n"
			"aoi\n"
			"color_map {\n"
			"[0.3 color rgb <1.0, 1.0, 1.0> transmit 1.0]\n"
			"[0.8 color rgb <1.0, 1.0, 1.0> transmit 0.0]\n"
			"}\n } \n";

		f << 
		"no_shadow\n"
		"}\n"
		"#end\n"
		;
	}
		
	f.close();
}



void TPovray::writePovrayField(std::ofstream * p_Stream, const TLayer * p_Layer, unsigned int p_Id, bool p_IsBlob)
{
	if(p_Layer == 0) throw string("p_Layer == 0");
	const TLayer * Layer = p_Layer;
	const TField * Field = Layer->m_Field.get();
	const TFilter * Filter = Layer->m_Filter.get();
	const TColorMap * ColorMap = Layer->m_ColorMap.get();
//	const TMeasure * ColorMapMeasure = Layer->m_ColorMapMeasure.get();
	const TMeasure * ColorMapMeasure = Layer->m_FilterMeasure.get();


	const float impstep = 0.00001f;

	std::ofstream & f = *p_Stream;
	f << "// Layer " << p_Layer->getDisplayName() << "\n";

	const TVoxelSet * VoxelSet = 0;
	for(unsigned int x=0; x<Layer->m_VoxelSets.size(); x++)
	{
		if( Layer->m_VoxelSets[x]->getVoxelSetType() != TVoxelSet::VOXELSETTYPE_ALL   
			&& Layer->m_VoxelSets[x]->getVoxelSetType() != TVoxelSet::VOXELSETTYPE_SELECTION
			)
		{
			VoxelSet = Layer->m_VoxelSets[x].get();
			f << "// Using voxelset " << VoxelSet->getName() << "\n";
		}
	}
	if(VoxelSet)
	{
		VoxelSet->initTraversal();
		TCoord3 p;
		TColor3 c;

		while( VoxelSet->nextTraversal(p) )
		{
			if(!Filter->test(p)) continue;

			ColorMap->getColor(ColorMapMeasure->toFloat(p), &c);
			float px = p.x * g_Settings.m_VoxelScale;
			float py = p.y * g_Settings.m_VoxelScale;
			float pz = p.z * g_Settings.m_VoxelScale;

			f << wxString::Format("%s%i(<%.1f,%.1f,%.1f>, <%.3f,%.3f,%.3f>) "
			, p_IsBlob ? "B" : "R"
			,p_Id
			,px,py,pz
			,c.r, c.g, c.b
			).c_str();	
		}
	}
	else
	{
		TColor3 c;
		unsigned int x;
		if( Field->usingIndexField() ) 
		for(x=0; x<Field->getIndexField()->getMaxIndex(); x++)
		{
			const TCoord3 p = Field->getIndexField()->vidx2coord(x);
			if( Filter->test(p) )
			{
				ColorMap->getColor(ColorMapMeasure->toFloat(p), &c);
				float px = p.x * g_Settings.m_VoxelScale;
				float py = p.y * g_Settings.m_VoxelScale;
				float pz = p.z * g_Settings.m_VoxelScale;

				f << wxString::Format("%s%i(<%.1f,%.1f,%.1f>, <%.3f,%.3f,%.3f>) "
				, p_IsBlob ? "B" : "R"
				,p_Id
				,px,py,pz
				,c.r, c.g, c.b
				).c_str();
			}
		}
	}

	/*
	if(IsBlob) { 
		f <<	
		"pigment { color rgb <1.0,1.0,1.0> }\n"
		"no_shadow\n"
		"}\n"
		;
	}
	*/

}



void TPovray::writePovraySkeleton(const string & p_OutputFilename, const TLayer * p_Layer)
{
	if(p_Layer == 0) throw string("p_Layer == 0");
	const TLayer * Layer = p_Layer;
	if(p_Layer->m_Field->getType() != TType::TYPE_FLOAT) throw string("Not a float field, please select skeleton layer");
	const TTypedFieldInterface<float> * Field = static_cast<TTypedFieldInterface<float>*>(p_Layer->m_Field.get());
	const TTypedFieldInterface<float>* RadiusField = static_cast<TTypedFieldInterface<float>*>( g_Mediator.getCurrentLayerSet()->getField( "Radius" ) );

	const float impstep = 0.00001f;


	try
	{

	const float midx = (float)(Field->getMaxX() ) / 2.0f;
	const float midy = (float)(Field->getMaxY() ) / 2.0f;
	const float midz = (float)(Field->getMaxZ() ) / 2.0f;
	
	//TPoint3 CamLoc( -2*m_Root.x, Field->getMaxY()*2.0f, 0 );
	//TPoint3 CamLookAt( midx, midy, midz );

	TVector3 CamLoc = MyApp::instance()->m_CurrentCamera->getPos();
	TVector3 CamLookAt = MyApp::instance()->m_CurrentCamera->getLookAt();


	// ofstream f( "_test.pov" );
	std::ofstream f( p_OutputFilename.c_str() );
	f << 
	"// Skeletonizer output of model '"+m_Skeletonizer->m_InputFilename+"'\n"
	"// Copyright 2006 Dennie Reniers, D.Reniers@tue.nl, http://www.win.tue.nl/~dreniers\n"
	"#version unofficial megapov 1.21;\n"
	//"#include \"colors.inc\"\n"
	"#declare RENDER_CS = 1;\n"
	"#declare RENDER_SEL = 1;\n"
	"#declare RENDER_SS = 0;\n"
	"#declare RENDER_MESH = 1;\n"
//	"#declare RENDER_REC = 1;\n"

	"#declare CS_RAD = 0.2;\n"
	"#declare CS_THRESHOLD = 1.0;\n"

	"#declare SS_RAD = 0.22;\n"
	"#declare SS_THRESHOLD = 0.8;\n"

	"#declare SEL_RAD = 0.3;\n"
	"#declare SEL_THRESHOLD = 1.0;\n"


	"#declare REC_THRESHOLD = 1.0;\n"
	"#declare REC_T = 0.0004;\n"

//	"#declare IMP_T = 0.005;\n"
//	"#declare MAXGEODESICLENGTH_T = " << m_Skeletonizer->m_MaximumGeodesicLength/m_Skeletonizer->m_NormalizeBy << ";\n"
	"#declare CS_T = " << Layer->m_Filter->m_LowerValue << ";\n"
	"#declare SS_T = 0.0004;\n"

	"#declare SPEC = 0.3;\n"
	"#declare ROTATION = (clock + 0.1) * 2*pi;\n"
	"background{ rgb <1,1,1> }\n"
	"global_settings { max_trace_level 200 }\n"
	"camera {\n"
	"angle 40\n";
	f << wxString::Format( "location <%f,%f,%f>\n", CamLoc.x, CamLoc.y, CamLoc.z).c_str();
 /*
	f << wxString::Format( "location <%f+sin(ROTATION)*%f,%f,%f+cos(ROTATION)*%f>\n", 
		CamLookAt.x, 
		max(midx,midz) * 10,
		CamLookAt.y, 
		CamLookAt.z,
		max(midx,midz) * 10
		).c_str();
// */
	f << wxString::Format( "look_at <%f,%f,%f>\n", CamLookAt.x, CamLookAt.y, CamLookAt.z).c_str();
//	f << "scale " << g_Settings.m_VoxelScale << "\n";
	
	// Two things are done with the following line.
	// 1. make right-handedness like opengl
	// 2. use -1 instead of -4/3 for square pixels, is needed because perspective angle for opengl is along y-axis, 
	// while for pov-ray it is along x-axis. With square pixels this doesn't matter.
	// 
	f << "right <-1,0,0>\n"; 
	f << "}\n";		

	f << wxString::Format( "light_source { <%f,%f,%f> color rgb <1,1,1> } \n", CamLoc.x, CamLoc.y, CamLoc.z).c_str();

/*	f << wxString::Format( "light_source { <%f+sin(ROTATION)*%f,%f,%f+cos(ROTATION)*%f> color rgb <1,1,1> } \n",
		CamLookAt.x, 
		max(midx,midz) * 12,
		CamLookAt.y, 
		CamLookAt.z,
		max(midx,midz) * 12
		).c_str();
*/


	const unsigned int X = Field->getMaxX();
	const unsigned int Y = Field->getMaxY();
	const unsigned int Z = Field->getMaxZ();

	// Output Curve Skeleton
	if(m_Skeletonizer->m_CurveSkeletonField)
	{
		multimap<float,TCoord3> S;
	
		for(unsigned int k=0; k<Z; k++)
		{
			for(unsigned int j=0; j<Y; j++)
			{
				for(unsigned int i=0; i<X; i++)
				{
					const TCoord3 p(i,j,k);
					if( m_Skeletonizer->m_CurveSkeletonField->vvaluep(p) != 0.0f )
					{
						S.insert( pair<float,TCoord3>(Field->vvaluep(p),p) );
					}
				}
			}
		}

		f << 
		"#if (RENDER_CS)\n"
		"blob\n { threshold CS_THRESHOLD\n"
		;
		TColor3 c;
		
		multimap<float,TCoord3>::reverse_iterator it;

		float previmp = 0;
		int impthresholdcount = 0;
		bool first = true;
		for(it = S.rbegin(); it != S.rend(); it++)
		{
			const float imp = it->first;
			const TCoord3 & p = it->second;
			float x = p.x * g_Settings.m_VoxelScale;
			float y = p.y * g_Settings.m_VoxelScale;
			float z = p.z * g_Settings.m_VoxelScale;
			Layer->m_ColorMap->getColor(p, &c);

			if( 
				// imp < 0.3f &&
				 (int)(previmp / impstep) != (int)(imp / impstep) ) // Detect importance going over impstep
			{
				if(!first) f << "#end\n";
				first = false;

				f << "#if (CS_T <= " << (float)((int)(imp / impstep)-1)*impstep << ")\n";
				impthresholdcount++;
			}

			f << wxString::Format("sphere { <%.3f,%.3f,%.3f>, CS_RAD, 1 pigment { color rgb <%.3f, %.3f, %.3f> } finish { diffuse 1 specular SPEC } }\n"
				,x,y,z
				,c.r,c.g,c.b
				).c_str();
			previmp = imp;
		}
		if(!first) f << "#end\n";


		f <<	
		"pigment { color rgb <1,1,1> }\n"
//		"finish { phong 1 }\n"
		"no_shadow\n"
		"}\n"
		"#end\n"
		;
	}


	// Output Surface Skeleton
	if(m_Skeletonizer->m_ForegroundSurfaceSkeletonField)
	{
		multimap<float,TCoord3> S;
		for(unsigned int k=0; k<Z; k++)
		{
			for(unsigned int j=0; j<Y; j++)
			{
				for(unsigned int i=0; i<X; i++)
				{
					const TCoord3 p(i,j,k);
					if( m_Skeletonizer->m_ForegroundSurfaceSkeletonField->vvaluep(p) != 0.0f )
					{
						S.insert( pair<float,TCoord3>(Field->vvaluep(p),p) );
					}
				}
			}
		}

		f << 
		"#if (RENDER_SS)\n"
		"blob\n { threshold SS_THRESHOLD\n"
		;
		TColor3 c;
		
		multimap<float,TCoord3>::reverse_iterator it;

		float previmp = 0;
		int impthresholdcount = 0;
		bool first = true;
		for(it = S.rbegin(); it != S.rend(); it++)
		{
			const float imp = it->first;
			const TCoord3 & p = it->second;
			float x = p.x * g_Settings.m_VoxelScale;
			float y = p.y * g_Settings.m_VoxelScale;
			float z = p.z * g_Settings.m_VoxelScale;
			Layer->m_ColorMap->getColor(p, &c);

			if( 
				// imp < 0.05f
				// && 
				(int)(previmp / impstep) != (int)(imp / impstep) ) // Detect importance going over impstep
			{
				if(!first) f << "#end\n";
				first = false;

				f << "#if (SS_T <= " << (float)((int)(imp / impstep)-1)*impstep << ")\n";
				impthresholdcount++;
			}

			f << wxString::Format("sphere { <%.3f,%.3f,%.3f>, SS_RAD, 1 pigment { color rgb <%.3f, %.3f, %.3f> } finish { diffuse 1 specular SPEC } }\n"
				,x,y,z
				,c.r,c.g,c.b
				).c_str();
			previmp = imp;
		}
		if(!first) f << "#end\n";


		f <<	
		"pigment { color rgb <1,1,1> }\n"
//		"finish { phong 1 }\n"
		"no_shadow\n"
		"}\n"
		"#end\n"
		;
	}
			
	
	
	// Output mesh
	if(m_Skeletonizer->m_Model)
	{	
		const TModel * Model = m_Skeletonizer->m_Model.get();

		f << 
		"#if (RENDER_MESH)\n"
		"mesh {\n"
		;
		for(unsigned int i=0; i<Model->m_IndexBuffer.size(); i=i+3)
		{
			const TPoint3 & a = Model->m_VertexBuffer[ Model->m_IndexBuffer[i+0] ];
			const TVector3 & na = Model->m_Normals[ i+0 ];
			const TPoint3 & b = Model->m_VertexBuffer[ Model->m_IndexBuffer[i+1] ];
			const TVector3 & nb = Model->m_Normals[ i+1 ];
			const TPoint3 & c = Model->m_VertexBuffer[ Model->m_IndexBuffer[i+2] ];
			const TVector3 & nc = Model->m_Normals[ i+2 ];
			f << wxString::Format("smooth_triangle { <%.3f,%.3f,%.3f> <%.3f,%.3f,%.3f> <%.3f,%.3f,%.3f> <%.3f,%.3f,%.3f> <%.3f,%.3f,%.3f> <%.3f,%.3f,%.3f> }\n"
				,a.x,a.y,a.z
				,na.x, na.y, na.z 
				,b.x,b.y,b.z
				,nb.x, nb.y, nb.z 
				,c.x,c.y,c.z
				,nc.x, nc.y, nc.z 
				).c_str();
		}
//		f << "translate <" << -Model->m_MinX << ", " << -Model->m_MinY << ", " << -Model->m_MinZ << ">\n";
		f << "scale " << (Model->m_Resize * g_Settings.m_VoxelScale) << "\n";
//		f << "rotate <-90.0, 0.0, 0.0> \n";
//		f << "rotate <0.0, -90.0, 0.0> \n";
//			f << "pigment { color rgb <1,1,1> transmit .9 }\n";
		
		f << "pigment {\n"
			"aoi\n"
			"color_map {\n"
			"[0.3 color rgb <1.0, 1.0, 1.0> transmit 1.0]\n"
			"[0.8 color rgb <1.0, 1.0, 1.0> transmit 0.0]\n"
			"}\n } \n";

		f << 
		"no_shadow\n"
		"}\n"
		"#end\n"
		;
	}

	// Output selection
	if( m_Skeletonizer->m_TotalCollapseField && Layer->m_Selection->getSingleSelection().x != -1  )
	{
		f << 
		"#if (RENDER_SEL)\n"
		"blob\n { threshold SEL_THRESHOLD\n"
		;
		TColor3 c;

		const TVoxelSet * vs = Layer->m_Selection.get();
		TCoord3 q;
		vs->initTraversal();
		while(vs->nextTraversal(q))
		{
			TVector3 s(q.x, q.y, q.z);
			s.scale( g_Settings.m_VoxelScale );

			const TIndexedOrigins_Vector * io = m_Skeletonizer->m_TotalCollapseField->valuep(q)->castVector();
			TIndexedOrigins_Vector::const_iterator jt;
			for(jt = io->begin(); jt != io->end(); jt++)
			{
				const TCoord3 & p = g_Mediator.getSkeletonizer()->m_BoundaryIndexField->vidx2coord(*jt);
				TVector3 v(p.x,p.y,p.z);
				v.scale( g_Settings.m_VoxelScale );
				f << wxString::Format("sphere { <%.3f,%.3f,%.3f>, SEL_RAD, 1 pigment { color rgb <1.0,0.0,1.0>} finish { diffuse 1 specular SPEC }  }\n"
					,v.x,v.y,v.z
					).c_str();
			}
		}
		
		f <<	
//		"pigment { color rgb <1,0,1> }\n"
//		"finish { diffuse 0.7  }\n"
		"no_shadow\n"
		"}\n"
		"#end\n";
		
		vs->initTraversal();
		while(vs->nextTraversal(q))
		{
			TVector3 s(q.x,q.y,q.z);
			s.scale( g_Settings.m_VoxelScale );
			f << wxString::Format("sphere { <%.3f,%.3f,%.3f>, 0.3 pigment { color rgb <1.0,1.0,1.0>} finish { diffuse 1 specular SPEC } }\n"
				, s.x, s.y, s.z
				).c_str();

			if(m_Skeletonizer->m_EftField)
			{
				const TIndexedOrigins_Vector * io = m_Skeletonizer->m_EftField->vvaluep(q)->castVector();
				TIndexedOrigins_Vector::const_iterator jt;
				for(jt = io->begin(); jt != io->end(); jt++)
				{
					const TCoord3 & p = g_Mediator.getSkeletonizer()->m_BoundaryIndexField->vidx2coord(*jt);
					TVector3 v(p.x,p.y,p.z);
					v.scale( g_Settings.m_VoxelScale );
					f << wxString::Format("cylinder { <%.3f,%.3f,%.3f>, <%.3f,%.3f,%.3f>, 0.1 pigment { color rgb <0.8,0.8,0.8>} finish { diffuse 1 specular SPEC } }\n"
						, s.x, s.y, s.z
						, v.x, v.y, v.z
						).c_str();
				}
			}
		}
	}





	f.close();

	}
	catch(string & s)
	{
		*g_Log << "exception caught: " + s + ", ";
	}

	// end output povray
}



void TPovray::writePovrayReconstruction(const string & p_OutputFilename, const TLayer * p_Layer)
{
	if(p_Layer == 0) throw string("p_Layer == 0");
	const TLayer * Layer = p_Layer;
	if(p_Layer->m_Field->getType() != TType::TYPE_FLOAT) throw string("Not a floatfield");
	const TTypedFieldInterface<float> * Field = static_cast<TTypedFieldInterface<float>*>(p_Layer->m_Field.get());

	const TFilter * Filter = Layer->m_Filter.get();

	const unsigned int X = Field->getMaxX();
	const unsigned int Y = Field->getMaxY();
	const unsigned int Z = Field->getMaxZ();
	

	shared_ptr<TFlagField3> RecFlagField( new TFlagField3(X,Y,Z) );
	for(unsigned int k=0; k<Z; k++)
	{
		for(unsigned int j=0; j<Y; j++)
		{
			for(unsigned int i=0; i<X; i++)
			{
				const TCoord3 p(i,j,k);
				if( Filter->test(p) )
				{
					RecFlagField->valuep(p) = TFlagField3::INSIDE;
				}
			}
		}
	}
	RecFlagField->constructNarrowBand();
	

	const float impstep = 0.00001f;

	try
	{
/*
	wxString ObjectName = wxFileName(m_InputFilename.c_str()).GetName().c_str();
	ObjectName.Replace("128","");
	ObjectName.Replace("256","");
	ObjectName.Replace("384","");
	ObjectName.Replace("512","");
*/	

	const float midx = (float)(Field->getMaxX() ) / 2.0f;
	const float midy = (float)(Field->getMaxY() ) / 2.0f;
	const float midz = (float)(Field->getMaxZ() ) / 2.0f;

	TVector3 CamLoc = MyApp::instance()->m_CurrentCamera->getPos();
	TVector3 CamLookAt = MyApp::instance()->m_CurrentCamera->getLookAt();


	std::ofstream f( p_OutputFilename.c_str() );
	f << 
	"// Skeletonizer output of model '"+m_Skeletonizer->m_InputFilename+"'\n"
	"// Object boundary reconstruction\n"
	"// Copyright 2006 Dennie Reniers, D.Reniers@tue.nl, http://www.win.tue.nl/~dreniers\n"
	"#version unofficial megapov 1.21;\n"

	"#declare ROTATION = (clock + 0.1) * 2*pi;\n"
	"#declare B_RAD = 0.4;\n"
	"#declare B_THRESHOLD = 0.8;\n"
	"#declare B_COLOR = color rgb <0.7,0.7,0.7>;\n"
	"#declare SPEC = 0.3;\n"
	"background{ rgb <1,1,1> }\n"
	"global_settings { max_trace_level 200 }\n"
	"camera {\n"
	"angle 40\n"
	;
	f << wxString::Format( "location <%f,%f,%f>\n", CamLoc.x, CamLoc.y, CamLoc.z).c_str();
	f << wxString::Format( "look_at <%f,%f,%f>\n", CamLookAt.x, CamLookAt.y, CamLookAt.z).c_str();

	// Two things are done with the following line.
	// 1. make right-handedness like opengl
	// 2. use -1 instead of -4/3 for square pixels, is needed because perspective angle for opengl is along y-axis, 
	// while for pov-ray it is along x-axis. With square pixels this doesn't matter.
	// 
	f << "right <-1,0,0>\n"; 
	f << "}\n";		

	f << wxString::Format( "light_source { <%f,%f,%f> color rgb <1,1,1> } \n", CamLoc.x, CamLoc.y, CamLoc.z).c_str();

	f << 
	"blob\n { threshold B_THRESHOLD\n";

	const unsigned int X = Field->getMaxX();
	const unsigned int Y = Field->getMaxY();
	const unsigned int Z = Field->getMaxZ();

	
	for(unsigned int k=0; k<Z; k++)
	{
		for(unsigned int j=0; j<Y; j++)
		{
			for(unsigned int i=0; i<X; i++)
			{
				const TCoord3 p(i,j,k);
				if( RecFlagField->valuep(p) == TFlagField3::NARROW_BAND )
				{
					float x = p.x * g_Settings.m_VoxelScale;
					float y = p.y * g_Settings.m_VoxelScale;
					float z = p.z * g_Settings.m_VoxelScale;

					f << wxString::Format("sphere { <%.3f,%.3f,%.3f>, B_RAD, 1 pigment { B_COLOR } finish { diffuse 1 specular SPEC } }\n"
						,x,y,z
						).c_str();
				}
			}
		}
	}

	f <<	
	"pigment { color rgb <1.0,1.0,1.0> }\n"
	"no_shadow\n"
	"}\n"
	;


	f.close();

	}
	catch(string & s)
	{
		*g_Log << "\nException caught: " + s + ", ";
	}

	// end output povray
}


void TPovray::writePovrayCurveSkeletonAndJunctions(const string & p_OutputFilename)
{
//	if(p_Layer == 0) throw string("p_Layer == 0");

	TTypedFieldInterface<unsigned char>* LoopField = static_cast<TTypedFieldInterface<unsigned char>*>( g_Mediator.getCurrentLayerSet()->getField("C-skel detector") );
	if(!LoopField) throw string("!LoopField");

	TTypedFieldInterface<float>* CurveSkeletonField = static_cast<TTypedFieldInterface<float>*>( g_Mediator.getCurrentLayerSet()->getField("C-skel measure") );
	if(!CurveSkeletonField) throw string("!CurveSkeletonField");
	const TField * Field = CurveSkeletonField;
	
	TTypedFieldInterface<unsigned char>* FilteredJunctions = static_cast<TTypedFieldInterface<unsigned char>*>( g_Mediator.getCurrentLayerSet()->getField( "Junctions" ) );
	if(!FilteredJunctions) throw string("!FilteredJunctions");


	const TFilter * CSFilter = m_Skeletonizer->m_CurveSkeletonField->m_Layer->m_Filter.get();
	const TColorMap * CSColorMap = m_Skeletonizer->m_CurveSkeletonField->m_Layer->m_ColorMap.get();
	const TMeasure * CSColorMapMeasure = m_Skeletonizer->m_CurveSkeletonField->m_Layer->m_FilterMeasure.get();

	const float impstep = 0.00001f;

	// Render rho and junctions of loopfield

	try
	{
	const float midx = (float)(Field->getMaxX() ) / 2.0f;
	const float midy = (float)(Field->getMaxY() ) / 2.0f;
	const float midz = (float)(Field->getMaxZ() ) / 2.0f;

	TVector3 CamLoc = MyApp::instance()->m_CurrentCamera->getPos();
	TVector3 CamLookAt = MyApp::instance()->m_CurrentCamera->getLookAt();

	std::ofstream f( p_OutputFilename.c_str() );
	f << 
	"// Output for model '"+m_Skeletonizer->m_InputFilename+"'\n"
//	"// Field " << Layer->m_Name << "\n"
	"// Copyright 2006 Dennie Reniers, D.Reniers@tue.nl, http://www.win.tue.nl/~dreniers\n"
	"#version unofficial megapov 1.21;\n"

	"#declare RENDER_NONFILTEREDJUNCTIONS = 0;\n"
	"#declare RENDER_FILTEREDJUNCTIONS = 0;\n"
	"#declare RENDER_RHO = 1;\n"
	"#declare RENDER_LOOPFIELD = 0;\n"
	"#declare RENDER_MESH = 1;\n"
	"#declare RENDER_CRITICALPOINTS = 1;\n"
	
	"#declare ROTATION = (clock + 0.1) * 2*pi;\n"
	"#declare JUNCTIONRAD = 0.4;\n"
	"#declare CPRAD = 0.4;\n"
	"#declare RHORAD = 0.2;\n"
	"#declare LOOPFIELDRAD = 0.2;\n"
	
	"#declare THRESHOLD = 0.8;\n"
	"#declare LOOPFIELDCOLOR = color rgb <0.2, 0.8, 1.0>;\n"
	"#declare JUNCTIONCOLOR = color rgb <0.0, 0.0, 0.0>;\n"
	"#declare CPCOLOR = color rgb <0.0, 0.0, 0.0>;\n"
	
	"#declare SPEC = 0.2;\n"
	"#declare CPSPEC = 5.0;\n"
	
	"background{ rgb <1,1,1> }\n"
	"global_settings { max_trace_level 200 }\n"
	"camera {\n"
	"angle 40\n"
	;
	f << wxString::Format( "location <%f,%f,%f>\n", CamLoc.x, CamLoc.y, CamLoc.z).c_str();
	f << wxString::Format( "look_at <%f,%f,%f>\n", CamLookAt.x, CamLookAt.y, CamLookAt.z).c_str();

	// Two things are done with the following line.
	// 1. make right-handedness like opengl
	// 2. use -1 instead of -4/3 for square pixels, is needed because perspective angle for opengl is along y-axis, 
	// while for pov-ray it is along x-axis. With square pixels this doesn't matter.
	// 
	f << "right <-1,0,0>\n"; 
	f << "}\n";		

	f << wxString::Format( "light_source { <%f,%f,%f> color rgb <1,1,1> } \n", CamLoc.x, CamLoc.y, CamLoc.z).c_str();
	
	// Output rho
	f << 
	"#if (RENDER_RHO)\n"
	"blob\n { threshold THRESHOLD\n";
		
	TColor3 c;

	const unsigned int X = Field->getMaxX();
	const unsigned int Y = Field->getMaxY();
	const unsigned int Z = Field->getMaxZ();

	for(unsigned int k=0; k<Z; k++)
	{
		for(unsigned int j=0; j<Y; j++)
		{
			for(unsigned int i=0; i<X; i++)
			{
				const TCoord3 p(i,j,k);
				if( CSFilter->test(p) )
				{
					CSColorMap->getColor(CSColorMapMeasure->toFloat(p), &c);
					float x = p.x * g_Settings.m_VoxelScale;
					float y = p.y * g_Settings.m_VoxelScale;
					float z = p.z * g_Settings.m_VoxelScale;

					f << wxString::Format("sphere { <%.3f,%.3f,%.3f>, RHORAD, 1 pigment { color rgb <%.3f,%.3f,%.3f> } finish { diffuse 1 specular SPEC } }\n"
						,x,y,z
						,c.r, c.g, c.b
						).c_str();
				}
			}
		}
	}

	f <<	
	"pigment { color rgb <1.0,1.0,1.0> }\n"
	"no_shadow\n"
	"}\n"
	"#end\n"
	;

	// Output nonfiltered junctions
	f << 
	"#if (RENDER_NONFILTEREDJUNCTIONS)\n";
	for(unsigned int k=0; k<Z; k++)
	{
		for(unsigned int j=0; j<Y; j++)
		{
			for(unsigned int i=0; i<X; i++)
			{
				const TCoord3 p(i,j,k);
				if( LoopField->vvaluep(p) >= 3.0f )
				{
					float x = p.x * g_Settings.m_VoxelScale;
					float y = p.y * g_Settings.m_VoxelScale;
					float z = p.z * g_Settings.m_VoxelScale;

					f << wxString::Format("sphere { <%.3f,%.3f,%.3f>, JUNCTIONRAD pigment { JUNCTIONCOLOR } finish { diffuse 1 specular CPSPEC } }\n"
						,x,y,z
						).c_str();
				}
			}
		}
	}
	f << "#end\n";


	// Output filtered junctions
	f << 
	"#if (RENDER_FILTEREDJUNCTIONS)\n";
	for(unsigned int k=0; k<Z; k++)
	{
		for(unsigned int j=0; j<Y; j++)
		{
			for(unsigned int i=0; i<X; i++)
			{
				const TCoord3 p(i,j,k);
				if( FilteredJunctions->vvaluep(p) >= 1 )
				{
					float x = p.x * g_Settings.m_VoxelScale;
					float y = p.y * g_Settings.m_VoxelScale;
					float z = p.z * g_Settings.m_VoxelScale;

					f << wxString::Format("sphere { <%.3f,%.3f,%.3f>, JUNCTIONRAD pigment { JUNCTIONCOLOR } finish { diffuse 1 specular CPSPEC } }\n"
						,x,y,z
						).c_str();
				}
			}
		}
	}
	f << "#end\n";



	// Output loopfield
	f << 
	"#if (RENDER_LOOPFIELD)\n"
	"blob\n { threshold THRESHOLD\n";
	for(unsigned int k=0; k<Z; k++)
	{
		for(unsigned int j=0; j<Y; j++)
		{
			for(unsigned int i=0; i<X; i++)
			{
				const TCoord3 p(i,j,k);
				if( LoopField->vvaluep(p) >= 2.0f )
				{
					float x = p.x * g_Settings.m_VoxelScale;
					float y = p.y * g_Settings.m_VoxelScale;
					float z = p.z * g_Settings.m_VoxelScale;

					f << wxString::Format("sphere { <%.3f,%.3f,%.3f>, LOOPFIELDRAD, 1 pigment { LOOPFIELDCOLOR } finish { diffuse 1 specular SPEC } }\n"
						,x,y,z
						).c_str();
				}
			}
		}
	}
	f <<	
	"pigment { color rgb <1.0,1.0,1.0> }\n"
	"no_shadow\n"
	"}\n"
	"#end\n"
	;



	// Output mesh
	if(m_Skeletonizer->m_Model)
	{	
		const TModel * Model = m_Skeletonizer->m_Model.get();
		f << 
		"#if (RENDER_MESH)\n"
		"mesh {\n"
		;
		for(unsigned int i=0; i<Model->m_IndexBuffer.size(); i=i+3)
		{
			const TPoint3 & a = Model->m_VertexBuffer[ Model->m_IndexBuffer[i+0] ];
			const TVector3 & na = Model->m_Normals[ i+0 ];
			const TPoint3 & b = Model->m_VertexBuffer[ Model->m_IndexBuffer[i+1] ];
			const TVector3 & nb = Model->m_Normals[ i+1 ];
			const TPoint3 & c = Model->m_VertexBuffer[ Model->m_IndexBuffer[i+2] ];
			const TVector3 & nc = Model->m_Normals[ i+2 ];
			f << wxString::Format("smooth_triangle { <%.3f,%.3f,%.3f> <%.3f,%.3f,%.3f> <%.3f,%.3f,%.3f> <%.3f,%.3f,%.3f> <%.3f,%.3f,%.3f> <%.3f,%.3f,%.3f> }\n"
				,a.x,a.y,a.z
				,na.x, na.y, na.z 
				,b.x,b.y,b.z
				,nb.x, nb.y, nb.z 
				,c.x,c.y,c.z
				,nc.x, nc.y, nc.z 
				).c_str();
		}
//		f << "translate <" << -Model->m_MinX << ", " << -Model->m_MinY << ", " << -Model->m_MinZ << ">\n";
		f << "scale " << (Model->m_Resize * g_Settings.m_VoxelScale) << "\n";
//		f << "rotate <-90.0, 0.0, 0.0> \n";
//		f << "rotate <0.0, -90.0, 0.0> \n";
//			f << "pigment { color rgb <1,1,1> transmit .9 }\n";
		
		f << "pigment {\n"
			"aoi\n"
			"color_map {\n"
			"[0.3 color rgb <1.0, 1.0, 1.0> transmit 1.0]\n"
			"[0.8 color rgb <1.0, 1.0, 1.0> transmit 0.0]\n"
			"}\n } \n";

		f << 
		"no_shadow\n"
		"}\n"
		"#end\n"
		;
	}

	f.close();

	}
	catch(string & s)
	{
		*g_Log << "\nException caught: " + s + ", ";
	}
}






