#include "stdafx.h"

#include "renderer.h"

#include "globals.h"
#include "settings.h"
#include "field.h"
#include "layer.h"
#include "utils.h"
#include "indexedoriginsfield.h"
#include "colormap.h"
#include "deltaomega.h"

#include "font.h"
#include "skeletonizer.h"
#include "histogram.h"
#include "component.h"
#include "indexedorigins_cells.h"

 
shared_ptr<TGlobalRenderer> TGlobalRenderer::m_GlobalRenderer = shared_ptr<TGlobalRenderer>(static_cast<TGlobalRenderer*>(0));

TGlobalRenderer::TGlobalRenderer()
{
	m_PickProcessors.push_back(0);
	m_Font.reset( new TFont() );
	m_BlendEquationAvailable = (void *) wglGetProcAddress("glBlendEquation") != 0;
};

void TGlobalRenderer::registerMe(TPickProcessor * p_PickProcessor)
{
	p_PickProcessor->m_PickID = m_PickProcessors.size();
	m_PickProcessors.push_back(p_PickProcessor);
}

void TGlobalRenderer::processPick(vector<unsigned int> p_PickStack, class wxMouseEvent * event)
{
	if(p_PickStack.size() == 0) throw string("Empty pick stack");
	if(p_PickStack.size() == 1) throw string("p_PickStack.size() == 1");

	vector<unsigned int> NewStack;
	for(int i=1; i<p_PickStack.size(); i++)
	{
		NewStack.push_back(p_PickStack[i]);
	}
	m_PickProcessors.at(p_PickStack[0])->OnPicked(NewStack, event);
}


TRenderer::TRenderer(const TField * p_Field)
	: m_Field(p_Field)
	, m_EndCalled(false)
	, m_2D( m_Field ? p_Field->getMaxZ() == 1 : 0)
{
	if(m_Field)
	{
		m_MaxX = m_Field->getMaxX();
		m_MaxY = m_Field->getMaxY();
		m_MaxZ = m_Field->getMaxZ();
	}

	sphereQuadric = gluNewQuadric();
}


TRenderer::~TRenderer()
{
	gluDeleteQuadric(sphereQuadric);

	if(!m_EndCalled) 
	{
		throw string("Renderer.end() not called");
	}
}


void TRenderer::begin()
{
	glGetBooleanv(GL_DEPTH_TEST, &m_DepthTest);
	glGetBooleanv(GL_DEPTH_WRITEMASK, &m_DepthMask);
	glGetBooleanv(GL_CULL_FACE, &m_CullFace);
	glGetBooleanv(GL_LIGHTING, &m_Light);
	glGetBooleanv(GL_BLEND, &m_Blend);
	glGetBooleanv(GL_POLYGON_OFFSET_FILL, &m_PolygonOffsetFill);
	glGetBooleanv(GL_COLOR_MATERIAL, &m_ColorMaterial);
	glGetFloatv(GL_LINE_WIDTH, &m_LineWidth);
	m_GR = TGlobalRenderer::instance();
	if(m_GR->m_BlendEquationAvailable) glGetIntegerv(GL_BLEND_EQUATION, &m_BlendEquation);

	glPushMatrix();
	if(m_Field)
	{
		glScalef(g_Settings.m_VoxelScale, g_Settings.m_VoxelScale, g_Settings.m_VoxelScale);
	}
}

void TRenderer::end()
{
	m_EndCalled = true;
	// Restore
	glPopMatrix();
	if(m_DepthTest) glEnable(GL_DEPTH_TEST); else glDisable(GL_DEPTH_TEST);
	if(m_DepthMask) glDepthMask(true); else glDepthMask(false);
	if(m_Light) glEnable(GL_LIGHTING); else glDisable(GL_LIGHTING);
	if(m_CullFace) glEnable(GL_CULL_FACE); else glDisable(GL_CULL_FACE);
	if(m_Blend) glEnable(GL_BLEND); else glDisable(GL_BLEND);
	if(m_PolygonOffsetFill) glEnable(GL_POLYGON_OFFSET_FILL); else glDisable(GL_POLYGON_OFFSET_FILL);
	if(m_ColorMaterial) glEnable(GL_COLOR_MATERIAL); else glDisable(GL_COLOR_MATERIAL);
	if(m_GR->m_BlendEquationAvailable) glBlendEquation(m_BlendEquation);
	glLineWidth(m_LineWidth);
}

void TRenderer::renderVoxel(const TCoord3 & p)
{
	glPushMatrix();
	//glTranslatef(m_MaxX-p.x+0.5f, m_MaxY-p.y+0.5f, p.z+0.5f); // upside down
	glTranslatef(p.x+0.5f, p.y+0.5f, p.z+0.5f); 

	glLoadName(p.z * m_MaxX * m_MaxY + p.y * m_MaxX + p.x); 
	
	if(m_Field->getLayer() && m_Field->getLayer()->m_RenderPrimitiveType == 1) 
	{
		gluSphere(sphereQuadric, g_Settings.m_RenderSphereSize, 8, 8);
	}
	else if(m_Field->getLayer() && m_Field->getLayer()->m_RenderPrimitiveType == 2) 
	{
		utilDrawQuad();
	}
	else
	{
		utilDrawCube();
	}

	glPopMatrix();
} 

void TRenderer::renderLine(const TCoord3 & from, const TCoord3 & to)
{
//	const TVector3 v( to.x - from.x, to.y - from.y, to.z - from.z );
//	renderVector(from, v);

//	TVector3 f(m_MaxX - from.x + 0.5f, m_MaxY - from.y + 0.5f, from.z + 0.5f); // upside down
//	TVector3 t(m_MaxX - to.x + 0.5f, m_MaxY - to.y + 0.5f, to.z + 0.5f); // upside down
	
	if(! g_Mediator.isPicking() )
	{
		const TVector3 f(from.x + 0.5f, from.y + 0.5f, from.z + 0.5f); // upside down
		TVector3 t(to.x + 0.5f, to.y + 0.5f, to.z + 0.5f); // upside down
		glBegin(GL_LINES);
			glVertex3f(f.x,f.y,f.z);
			glVertex3f(t.x,t.y,t.z);
		glEnd();
	}
}

void TRenderer::renderVector(const TCoord3 & from, const TVector3 & dir)
{
//	TVector3 f(m_MaxX - from.x + 0.5f, m_MaxY - from.y + 0.5f, from.z + 0.5f);
	TVector3 f(from.x + 0.5f, from.y + 0.5f, from.z + 0.5f);

//	TVector3 d( -dir.x, -dir.y, dir.z );
	TVector3 d( dir.x, dir.y, dir.z );
	d.scale(3.0f);
	TVector3 t = f;
	t.add( d );

	//     t
	//   / | \
	// m2  m  m1
	//     |
	//     |
	//     f----- dp

	TVector3 m1 = d; 
	m1.scale( (m1.length()-0.5f) / m1.length() );
	m1.add(f);

	TVector3 m2 = m1;	
	TVector3 dp( d.y, -d.x, d.z );
	if(dp.length() == 0) return;
	dp.normalize();
	dp.scale(0.05f); // arrow size
	m1.add(dp);
	m2.subtract(dp);

	glBegin(GL_LINES);
		glColor3f(1.0f, 0.3f, 0.3f);
		glVertex3f(f.x,f.y,f.z);
		glColor3f(0.3f, 1.0f, 0.3f);
		glVertex3f(t.x,t.y,t.z);

		//glVertex3f(t.x,t.y,t.z);
		//glVertex3f(m1.x,m1.y,m1.z);

		//glVertex3f(t.x,t.y,t.z);
		//glVertex3f(m2.x,m2.y,m2.z);
	glEnd();
}

void TRenderer::setColorByStatus(const TPath * p_Path, bool p_ForLine)
{
/*
	if(p_Path->m_Status == TPath::STATUS_ACTIVE)
		setColor(TColor3(0.5f,0.9f,0.5f));
	else if(p_Path->m_Status == TPath::STATUS_STALLED)
		setColor(TColor3(0.9f,0.5f,0.5f));
	else if(p_Path->m_Status == TPath::STATUS_MERGED_NECESSARY)
		setColor(TColor3(0.0f,0.7f,0.7f));
	else if(p_Path->m_Status == TPath::STATUS_MERGED_WALKOVER)
		setColor(TColor3(0.7f,0.7f,0.0f));
	else if(p_Path->m_Status == TPath::STATUS_MERGED_CLEANUP)
		setColor(TColor3(0.0f,0.0f,0.0f));

	if(p_ForLine)
	{
		TColor3 c = m_Color;
		c.add( TColor3(0.1f, 0.1f, 0.1f) );
		setColor(c);
	}
*/
 }




// =====


TSelectionRenderer::TSelectionRenderer()
{
}

void TSelectionRenderer::visitFloatField(TTypedFieldInterface<float> * p_Field)
//void TSelectionRenderer::visitFloatField(class TFloatField3 * p_Field)
{
}

void TSelectionRenderer::visitIntegerField(TTypedFieldInterface<int> * p_Field)
{
}

void TSelectionRenderer::visitUintegerField(TTypedFieldInterface<unsigned int> * p_Field)
{
}


void TSelectionRenderer::visitUnsignedCharField(TTypedFieldInterface<unsigned char> * p_Field)
//void TSelectionRenderer::visitUnsignedCharField(class TUnsignedCharField3 * p_Field)
{
}

void TSelectionRenderer::visitVector2Field(class TVector2Field3 * p_Field)
{
}


void TSelectionRenderer::visitIndexedOriginsField(TTypedFieldInterface<class TIndexedOrigins*> *  p_Field)
{
	TRenderer Renderer(p_Field);
	Renderer.begin();
	glPolygonOffset(0, -1);
	glEnable(GL_POLYGON_OFFSET_FILL);
	glEnable(GL_POLYGON_OFFSET_LINE);

	glEnable(GL_COLOR_MATERIAL);
	glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);
	glColor4f(0.2f, 0.2f, 0.2f, 0.2f);


	const TIndexMapper * IndexToField = p_Field->getIndexToField().get();
	if(!IndexToField) throw string("!IndexToField");
	const TVoxelSet * vs = p_Field->getLayer()->m_Selection.get();
	TCoord3 q;
	vs->initTraversal();
	while(vs->nextTraversal(q))
	{
		const TCoord3 s = q;

		TIndexedOrigins * io = p_Field->vvaluep(s);
		if(!io) continue;
				
		shared_ptr<TIndexedOrigins::TIterator> jt(io->newIterator());
		for(jt->init(); jt->valid(); jt->next())
		{
			const TCoord3 & c = IndexToField->vidx2coord(jt->value());
			
			glColor3f(1.0f, 0.0f, 1.0f);
			Renderer.renderVoxel(c);			

			// Render outline
			if(p_Field->getLayer()->m_RenderVoxelOutlines)
			{
				glEnable(GL_LINE_SMOOTH);
				glEnable(GL_BLEND);
				glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
				glHint (GL_LINE_SMOOTH_HINT, GL_NICEST);
				glPolygonOffset(-1, -1);
				glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
				glColor3f(0.0f, 0.0f, 0.0f);
				Renderer.renderVoxel(c);
				glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
				glPolygonOffset(0, -1);
			}

			if(p_Field->getLayer()->m_RenderLines)
			{
				glColor3f(0.0f, 0.0f, 0.0f);
				Renderer.renderLine(s,c); 
			}
			glDisable(GL_BLEND);
			glDisable(GL_LINE_SMOOTH);
		}
	}


	glDisable(GL_POLYGON_OFFSET_FILL);
	glDisable(GL_POLYGON_OFFSET_LINE);
	Renderer.end();
}

void TSelectionRenderer::visitIndexedOriginsMultimapField(TTypedFieldInterface<class TIndexedOrigins_Multimap*> *  p_Field)
{
	TRenderer Renderer(p_Field);
	Renderer.begin();
	glPolygonOffset(0, -1);
	glEnable(GL_POLYGON_OFFSET_FILL);
	glEnable(GL_COLOR_MATERIAL);


	shared_ptr<TColorMap> ColorMap = p_Field->getLayer()->m_ColorMap;

	const TFilter * Filter = static_cast<TFloatFilter*>( p_Field->getLayer()->m_Filter.get() );
	bool UseFilter = 
		p_Field->getLayer()->m_FilterMeasure->getMeasureType() == TMeasure::MEASURETYPE_INDEXEDORIGINS_MULTIMAP_MIN
		|| p_Field->getLayer()->m_FilterMeasure->getMeasureType() == TMeasure::MEASURETYPE_INDEXEDORIGINS_MULTIMAP_MAX;

	const TVoxelSet * vs = p_Field->getLayer()->m_Selection.get();
	TCoord3 q;
	vs->initTraversal();
	while(vs->nextTraversal(q))
	{
		const TCoord3 s = q;
  
		TIndexedOrigins_Multimap * io = p_Field->vvaluep(s);
		if(!io) continue;
		const float minvalue = io->begin()->first;
		const float maxvalue = io->rbegin()->first;

		TIndexedOrigins_Multimap::iterator kt;
		for(kt = io->begin(); kt != io->end(); kt++)
		{
			if(kt->second == TIndexedOrigins::INVALIDINDEX) continue;

			const TCoord3 & c = p_Field->getIndexToField()->vidx2coord(kt->second);
			if(UseFilter && !Filter->test(kt->first)) continue;

			TColor3 Col;
			ColorMap->getColor(kt->first, &Col);
			glColor3f( Col.r, Col.g, Col.b );
			Renderer.renderVoxel(c);			

			if(p_Field->getLayer()->m_RenderLines)
			{
				glColor3f(0.0f, 0.0f, 0.0f);
				Renderer.renderLine(s,c); 
			}
			
		}
	}


	glDisable(GL_POLYGON_OFFSET_FILL);
	Renderer.end();
}




void TSelectionRenderer::visitVector3Field(TTypedFieldInterface<TVector3> * p_Field)
{
	TRenderer Renderer(p_Field);
	Renderer.begin();
	Renderer.setColor(TColor3(0.3f,0.3f,0.3f));
//	glEnab(GL_DEPTH_TEST);
	glDisable(GL_LIGHTING);
	glLineWidth(2.0f);

	const TVoxelSet * vs = p_Field->getLayer()->m_Selection.get();
	TCoord3 q;
	vs->initTraversal();
	while(vs->nextTraversal(q))
	{
		const TCoord3 s = q;

		Renderer.renderVector(s, p_Field->vvaluep(s) );
	}

	//glEnable(GL_DEPTH_TEST);
	Renderer.end();
}







void TSelectionRenderer::visitCoord3SetField(TTypedFieldInterface<vector<TCoord3>*> * p_Field)
{
	TRenderer Renderer(p_Field);
	Renderer.begin();
	glEnable(GL_DEPTH_TEST);
	glDisable(GL_LIGHTING);

	const TVoxelSet * vs = p_Field->getLayer()->m_Selection.get();
	TCoord3 q;
	vs->initTraversal();
	while(vs->nextTraversal(q))
	{
		const TCoord3 s = q;

		vector<TCoord3> * io = p_Field->vvaluep(s);
		if(io == 0) continue; 
		Renderer.setColor(TColor3(1.0f,0.0f,1.0f));
		
		for(unsigned int x=0; x<io->size(); x++)
		{
//			Renderer.renderVoxel( (*io)[x] );
		}

		if(p_Field->getLayer()->m_RenderLines)
		{
			Renderer.setColor(TColor3(0.0f,0.0f,0.0f));
			for(unsigned int x=0; x<io->size(); x++)
			{
				// Renderer.renderLine( s, (*io)[x] ); 
				const TCoord3 & from = s;
				const TCoord3 & to = (*io)[x];

				if(! g_Mediator.isPicking())
				{
					const TVector3 f(from.x + 0.5f, from.y + 0.5f, from.z + 0.5f); // upside down
					const TVector3 t(to.x + 0.5f, to.y + 0.5f, to.z + 0.5f); // upside down
					glBegin(GL_LINES);
						glColor3f(1.0f, 0.0f, 0.0f);
						glVertex3f(f.x,f.y,f.z);
						glColor3f(0.0f, 1.0f, 0.0f);
						glVertex3f(t.x,t.y,t.z);
					glEnd();
				}
			}
		}
	}
	Renderer.end();
}

void TSelectionRenderer::visitCoord3WithFloatSetField(TTypedFieldInterface<vector<pair<TCoord3,float> >*> * p_Field)
{
	TRenderer Renderer(p_Field);
	Renderer.begin();
	glEnable(GL_DEPTH_TEST);
	glDisable(GL_LIGHTING);
//	glPointSize(2.0f);

	TColorMap * ColorMap = p_Field->getLayer()->m_ColorMap.get();
	TColor3 Color;
	const TVoxelSet * vs = p_Field->getLayer()->m_Selection.get();
	TCoord3 q;
	vs->initTraversal();
	while(vs->nextTraversal(q))
	{
		const TCoord3 s = q;

		vector< pair<TCoord3,float> > * io = p_Field->vvaluep(s);
		if(io == 0) continue; 
		Renderer.setColor(TColor3(1.0f,0.0f,1.0f));
		
		for(unsigned int x=0; x<io->size(); x++)
		{
//			Renderer.renderVoxel( (*io)[x] );
		}

		if(p_Field->getLayer()->m_RenderLines)
		{
			Renderer.setColor(TColor3(0.0f,0.0f,0.0f));
			for(unsigned int x=0; x<io->size(); x++)
			{
				// Renderer.renderLine( s, (*io)[x] ); 
				const TCoord3 & from = s;
				const TCoord3 & to = (*io)[x].first;

				if(! g_Mediator.isPicking())
				{
					const TVector3 f(from.x + 0.5f, from.y + 0.5f, from.z + 0.5f); // upside down
					const TVector3 t(to.x + 0.5f, to.y + 0.5f, to.z + 0.5f); // upside down
					ColorMap->getColor((*io)[x].second, &Color);
					glColor3f(Color.r, Color.g, Color.b);

					if(from.x==to.x&&from.y==to.y&&from.z==to.z)
					{
						glPushMatrix();
						glTranslatef(f.x,f.y,f.z);
						glScalef(0.3f, 0.3f, 0.3f);
						utilDrawCube();
						glPopMatrix();
					}
					else
					{
						glBegin(GL_LINES);
							glVertex3f(f.x,f.y,f.z);
							glVertex3f(t.x,t.y,t.z);
						glEnd();
					}
				}
			}
		}
	}
	Renderer.end();
}

void TSelectionRenderer::visitShortestPathSetField(TTypedFieldInterface<class TShortestPathSet*> *  p_Field)
{
	TRenderer Renderer(p_Field);
	Renderer.begin();
	glDisable(GL_LIGHTING);

	const TVoxelSet * vs = p_Field->getLayer()->m_Selection.get();
	TCoord3 q;
	vs->initTraversal();
	while(vs->nextTraversal(q))
	{
		const TShortestPathSet * sps = p_Field->vvaluep(q);
		if(!sps) continue;

		for(unsigned int x=0; x<sps->m_Paths.size(); x++)
		{
			shared_ptr<TShortestPath> sp = sps->m_Paths[x];
			
			// Render local sheet
			if(sp->m_LocalSheet)
			{
				const TPoint3 origin = sp->m_LocalSheet->getPoint();
				TVector3 normal = sp->m_LocalSheet->getNormal();
				normal.scale(3.0f);
				normal.add(origin);
 
				glBegin(GL_LINES);
					glColor3f(1.0f, 0.0f, 0.0f);
					glVertex3f(origin.x, origin.y, origin.z);
					glColor3f(0.0f, 1.0f, 0.0f);
					glVertex3f(normal.x, normal.y, normal.z);
				glEnd();


				TVector3 vp;
				TVector3 vq;
				sp->m_LocalSheet->vectorInPlane(vp);
				glColor3f(0.2f, 0.2f, 0.2f);
				vq = vp.cross(sp->m_LocalSheet->getNormal());
				vp.scale(2.0f);
				vq.scale(2.0f);
				TVector3 c1,c2,c3,c4;
				c1.add(origin);
				c1.subtract(vp);
				c1.add(vq);
				c2.add(origin);
				c2.add(vp);
				c2.add(vq);
				c3.add(origin);
				c3.add(vp);
				c3.subtract(vq);
				c4.add(origin);
				c4.subtract(vp);
				c4.subtract(vq);

				glBegin(GL_LINE_STRIP);
					glVertex3f(c1.x, c1.y, c1.z);
					glVertex3f(c2.x, c2.y, c2.z);
					glVertex3f(c3.x, c3.y, c3.z);
					glVertex3f(c4.x, c4.y, c4.z);
					glVertex3f(c1.x, c1.y, c1.z);
				glEnd();
			}
		}
	}
	Renderer.end();
}

void TSelectionRenderer::visitComponentSetField(TTypedFieldInterface<class TComponentSet*> *  p_Field)
{
	const TCoord3 & p = p_Field->getLayer()->m_Selection->getSingleSelection();
	if(!p_Field->getIndexField()->vinside(p)) return;

	const TComponentSet * cs = p_Field->vvaluep(p);
	if(!cs) return;

	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	TRenderer Renderer(p_Field);
	Renderer.begin();
	glEnable(GL_COLOR_MATERIAL);

	unsigned int x;

	// Render bordereft
	glColor3f( 1.0f, 0.0f, 1.0f );
	for(x=0; x<cs->m_BorderEft.size(); x++)
	{	
		const TIndexedOrigins_Vector * vec = cs->m_BorderEft[x].get();
		for(unsigned int y=0; y<vec->size(); y++)
		{
			Renderer.renderVoxel( skel->m_BoundaryIndexField->vidx2coord( (*vec)[y] ) );
		}
	}

	// Render border
	glColor3f( 0.5f, 0.0f, 0.5f );
	for(x=0; x<cs->m_Borders.size(); x++)
	{	
		const TIndexedOrigins_Vector * vec = cs->m_Borders[x].get();
		for(unsigned int y=0; y<vec->size(); y++)
		{
			Renderer.renderVoxel( skel->m_BoundaryIndexField->vidx2coord( (*vec)[y] ) );
		}
	}

	if(g_Settings.m_RenderCsSelectionComponents)
	{
		// Render cells
		TNormalizedColorMap ColorMap(0,cs->m_Cells.size()-1.0f);
		for(x=1; x<cs->m_Cells.size(); x++)
		// for(x=0; x<cs->m_Cells.size(); x++) // render dilated path also
		{
			TColor3 color;
			ColorMap.getColor(x, &color);
			glColor3fv( color.rgb );

			TIndexedOrigins_Cells * io = cs->m_Cells[x].get();
			shared_ptr<TIndexedOrigins::TIterator> it( io->newIterator() );
			it->init();
			while(it->valid())
			{
				unsigned int idx = it->value();
				Renderer.renderVoxel( skel->m_BoundaryIndexField->vidx2coord(idx) );
				it->next();
			}
		}
	}

	Renderer.end();
}


void TSelectionRenderer::visitNodeField(TTypedFieldInterface<vector<class TNode*>*> *  p_Field)
{
/*
	TRenderer Renderer(p_Field);
	Renderer.begin();
	glEnable(GL_DEPTH_TEST);
	glDisable(GL_LIGHTING);

	TColorMap * ColorMap = p_Field->getLayer()->m_ColorMap.get();
	TColor3 Color;
	const TVoxelSet * vs = p_Field->getLayer()->m_Selection.get();
	TCoord3 q;
	vs->initTraversal();
	while(vs->nextTraversal(q))
	{
		const TCoord3 s = q;

		if(p_Field->vvaluep(s) == 0) continue; 
		vector<TNode*> * io = p_Field->vvaluep(s);
		Renderer.setColor(TColor3(1.0f,0.0f,1.0f));
		
		for(unsigned int x=0; x<io->size(); x++)
		{
//			Renderer.renderVoxel( (*io)[x] );
		}

		if(p_Field->getLayer()->m_RenderLines)
		{
			Renderer.setColor(TColor3(0.0f,0.0f,0.0f));
			for(unsigned int x=0; x<io->size(); x++)
			{
				// Renderer.renderLine( s, (*io)[x] ); 
				const TCoord3 & from = s;
				const TCoord3 & to = (*io)[x].first;

				if(! MyApp::instance()->m_Picking)
				{
					const TVector3 f(from.x + 0.5f, from.y + 0.5f, from.z + 0.5f); // upside down
					const TVector3 t(to.x + 0.5f, to.y + 0.5f, to.z + 0.5f); // upside down
					ColorMap->getColor((*io)[x].second, &Color);
					glColor3f(Color.r, Color.g, Color.b);

					if(from.x==to.x&&from.y==to.y&&from.z==to.z)
					{
						glPushMatrix();
						glTranslatef(f.x,f.y,f.z);
						glScalef(0.3f, 0.3f, 0.3f);
						utilDrawCube();
						glPopMatrix();
					}
					else
					{
						glBegin(GL_LINES);
							glVertex3f(f.x,f.y,f.z);
							glVertex3f(t.x,t.y,t.z);
						glEnd();
					}
				}
			}
		}
	}
	Renderer.end();
*/
 }




void TSelectionRenderField_IndexedOrigins::onSelectionChanged()
{
	if(m_Selection->m_Field->getType() != TType::TYPE_INDEXEDORIGINS) throw string("onSelectionChanged(): invalid type");
	const TSparseTypedField<TIndexedOrigins*> * Field = static_cast<const TSparseTypedField<TIndexedOrigins*> *>( m_Selection->m_Field );

	TCoord3 p;
	clear();
	unsigned int x=0;

	m_Selection->initTraversal();
	while(m_Selection->nextTraversal(p))
	{
		if(Field->valuep(p) == 0) continue;
		TIndexedOrigins_Vector * vec = Field->valuep(p)->castVector();
		for(x=0; x<vec->size(); x++)
		{
			m_Values[(*vec)[x]] = 1;
		}
	}
	m_Layer->onFieldChanged();
}


