#include "stdafx.h"

#include "shortestpath.h"
#include "globals.h"
#include "skeletonizer.h"

using namespace std;

float TShortestPathSet::getMaxLength() const
{ 
	if(m_Paths.size() == 0) return 0;
	
	unsigned int idx = 0;
	for(unsigned int y=0; y<m_Paths.size(); y++)
	{
		if( m_Paths[y]->m_Length > m_Paths[idx]->m_Length ) idx = y;

	}
	return m_Paths[idx]->m_Length;
}

float TShortestPathSet::getEccentricity() const
{
	if(m_Paths.size() == 0) return 0;
	
	float maxd = 0.0f;
	for(unsigned int y=0; y<m_Paths.size(); y++)
	{
		const float d = m_Paths[y]->getEccentricity(m_SourceVoxel);
		if( d > maxd ) 
		{
			maxd = d;
		}

	}
	return maxd;
}

float TShortestPath::getRhoDivDt(const TCoord3 & p) const
{
	float radius = 0.0f;
	{
		const float dist = p.distance( g_Mediator.getSkeletonizer().get()->m_BackgroundBoundaryIndexField->vidx2coord(m_Begin) );
		if(dist > radius ) radius  = dist;
	}
	{
		const float dist = p.distance( g_Mediator.getSkeletonizer().get()->m_BackgroundBoundaryIndexField->vidx2coord(m_End) );
		if(dist > radius ) radius = dist;
	}
	return m_Length / radius;
}


float TShortestPath::getMaxRadius(const TCoord3 & p) const
{
	const float d1 = p.distance( g_Mediator.getSkeletonizer().get()->m_BackgroundBoundaryIndexField->vidx2coord(m_Begin) );
	const float d2 = p.distance( g_Mediator.getSkeletonizer().get()->m_BackgroundBoundaryIndexField->vidx2coord(m_End) );;
	return d1 > d2 ? d1 : d2;
}

float TShortestPath::getEccentricity(const TCoord3 & p) const
{
	if(getMaxDistance() == (std::numeric_limits<float>::max)())
		computeMaxDistance(p);
	const float dist = g_Mediator.getSkeletonizer().get()->m_BackgroundBoundaryIndexField->vidx2coord(m_Begin).distance( g_Mediator.getSkeletonizer().get()->m_BackgroundBoundaryIndexField->vidx2coord(m_End) );
	return getMaxDistance() / getMaxRadius(p);
}



TShortestPathSet::TPaths TShortestPathSet::getThreeLongestPaths() const
{
	TPaths paths;
	typedef multimap<float, shared_ptr<TShortestPath>, greater<float> > TMap;
	TMap mapp;
	TMap::iterator it;
	for(unsigned int y=0; y<m_Paths.size(); y++)
	{
		shared_ptr<TShortestPath> sp = m_Paths[y];
		mapp.insert( TMap::value_type(sp->m_Length, sp) );
	}
	it=mapp.begin(); 
	if(it == mapp.end()) return paths;
	paths.push_back( it->second ); it++;
	if(it == mapp.end()) return paths;
	paths.push_back( it->second ); it++;
	if(it == mapp.end()) return paths;
	paths.push_back( it->second );
	return paths;
}


float TShortestPathSet::getRadius() const
{
	if(m_Paths.size() == 0) return 0.0f;

	return m_Paths[0]->getMaxRadius(m_SourceVoxel);
}

float TShortestPathSet::getMaxRadius() const
{
	if(m_Eft->size() == 0) return 0.0f;

	float radius = 0.0f;
	for(unsigned int x=0; x<m_Eft->size(); x++)
	{
		const float dist = m_SourceVoxel.distance( g_Mediator.getSkeletonizer().get()->m_BackgroundBoundaryIndexField->vidx2coord((*m_Eft)[x] ) );
		if(dist > radius ) radius  = dist;
	}
	return radius;
}


float TShortestPathSetFieldMeasure_MaximumCircularity::toFloat(const TCoord3 & p) const
{
	TShortestPathSet * sps = m_Field->vvaluep(p);
	if(!sps || sps->m_Paths.size() == 0) return 0;

	unsigned int idx = 0;
	for(unsigned int x=0; x<sps->m_Paths.size(); x++)
	{
		if(sps->m_Paths[x]->getCircularity() > sps->m_Paths[idx]->getCircularity() ) idx = x;
	}
	return sps->m_Paths[idx]->getCircularity();
}

void TShortestPathSet::updatePathSet()
{
	TIndexedOrigins_Set MergedGeo;

	// Compute pathset 
	if(m_PathSet)
	{	
		m_PathSet->clear();
		for(unsigned int x=0; x<m_Paths.size(); x++)
		{
			if(m_Paths[x]->m_Path)
			{
				MergedGeo.merge( m_Paths[x]->m_Path.get() );
			}
		}
		m_PathSet->merge( &MergedGeo );
	}
}



void TShortestPath::computeMaxDistance(const TCoord3 & p) const
{
	if(!m_Path) return;

	float maxdist = 0.0f;
	for(unsigned int x=0; x<m_Path->size(); x++)
	{
		const unsigned int pidx = (*m_Path)[x];
		const TCoord3 pcoord = g_Mediator.getSkeletonizer().get()->m_BoundaryIndexField->vidx2coord( pidx );
		const float dist = p.distance( pcoord );
		if(dist > maxdist) maxdist = dist;
	}
	m_MaxDistance = maxdist;
}

void TShortestPath::writeToStream(std::ofstream * s) const
{
	s->write( (char*) &m_Begin, sizeof(unsigned int) );
	s->write( (char*) &m_End, sizeof(unsigned int) );
	s->write( (char*) &m_Length, sizeof(float) );
	s->write( (char*) &m_MaxDistance, sizeof(float) );

	char containspath = m_Path.get() == 0 ? 0 : 1;
	s->write( &containspath, 1 );
	if(containspath) 
	{
		m_Path->writeToStream(s);
	}

	char containsplane = m_LocalSheet.get() == 0 ? 0 : 1;
	s->write( &containsplane, 1 );
	if(containsplane)
	{
		s->write( (char*) &m_LocalSheet->m_Normal.x, sizeof(float)*3 );
		s->write( (char*) &m_LocalSheet->m_Point.x, sizeof(float)*3 );
		s->write( (char*) &m_LocalSheet->m_D, sizeof(float) );
	}
}

TShortestPath * TShortestPath::readFromStream(std::ifstream * s)
{
	TShortestPath * p = new TShortestPath();
	s->read( (char*) &p->m_Begin, sizeof(unsigned int) );
	s->read( (char*) &p->m_End, sizeof(unsigned int) );
	s->read( (char*) &p->m_Length, sizeof(float) );
	s->read( (char*) &p->m_MaxDistance, sizeof(float) );
	p->m_MaxDistance = (std::numeric_limits<float>::max)(); // Todo: necessary because not all files contain correct maxdistance
	
	char containspath = 0;
	s->read( &containspath, 1 );
	assert(containspath == 0 || containspath == 1);
	if(containspath) 
	{
		p->m_Path.reset( TIndexedOrigins_Vector::readFromStream(s) );
	}

	char containsplane = 0;
	s->read( &containsplane, 1 );
	assert(containsplane == 0 || containsplane == 1);
	if(containsplane)
	{
		p->m_LocalSheet.reset( new TPlane() );
		s->read( (char*) &p->m_LocalSheet->m_Normal.x, sizeof(float)*3 );
		s->read( (char*) &p->m_LocalSheet->m_Point.x, sizeof(float)*3 );
		s->read( (char*) &p->m_LocalSheet->m_D, sizeof(float) );
	}


	return p;
}

void TShortestPathSet::writeToStream(std::ofstream * s) const
{
	// Write eft
	s->write( (char*) &m_SourceVoxel.x, sizeof(float)*3 );
	m_Eft->writeToStream(s);

	// Write simplified eft
	{
		char containsseft = m_Seft.get() == 0 ? 0 : 1;
		s->write( &containsseft, 1 );
		if(containsseft) m_Seft->writeToStream(s);
	}

	// Write path set
	{
		char containspathset = m_PathSet.get() == 0 ? 0 : 1;
		s->write( &containspathset, 1 );
		if(containspathset) m_PathSet->writeToStream(s);
	}

	// Write individual paths
	{
		unsigned int pathcount = m_Paths.size();
		s->write( (char*) &pathcount, sizeof(unsigned int) );
		for(unsigned int y=0; y<m_Paths.size(); y++)
			m_Paths[y]->writeToStream(s);
	}
}

TShortestPathSet * TShortestPathSet::readFromStream(std::ifstream * s)
{
	TShortestPathSet * sps = new TShortestPathSet();
	s->read( (char*) &sps->m_SourceVoxel.x, sizeof(float)*3 );
	sps->m_Eft.reset( TIndexedOrigins_Vector::readFromStream(s) );
	
	// Read simplified eft
	char containsseft = 0;
	s->read( &containsseft, 1 );
	if(containsseft) sps->m_Seft.reset( TIndexedOrigins_Vector::readFromStream(s) );


	// Read path set
	char containspathset = 0;
	s->read( &containspathset, 1 );
	if(containspathset) sps->m_PathSet.reset( TIndexedOrigins_Vector::readFromStream(s) );
	unsigned int pathcount = 0;
	s->read( (char*) &pathcount, sizeof(unsigned int) );
	assert(pathcount < 256);
	sps->m_Paths.resize(pathcount);
	for(unsigned int y=0; y<pathcount; y++)
		sps->m_Paths[y].reset( TShortestPath::readFromStream(s) );
	return sps;
}

TShortestPathSetField * TShortestPathSetField::readFromStream(std::ifstream * s, shared_ptr<TIndexMapper> p_IndexField)
{
	TShortestPathSetField * field = new TShortestPathSetField(p_IndexField);
	s->read( (char*)&field->m_SeftTau, sizeof(field->m_SeftTau) );
	unsigned int count = 0;
	s->read( (char*)&count, sizeof(count) );
	for(unsigned int x=0; x<count; x++)
	{
		field->wvaluex(x) = TShortestPathSet::readFromStream(s);
	}
	return field;	
}

void TShortestPathSetField::writeToStream(std::ofstream * s) const
{
	s->write( (char*)&this->m_SeftTau, sizeof(this->m_SeftTau) );
	unsigned int count = this->getMaxIndex();
	s->write( (char*) &count, sizeof(unsigned int) );
	for(unsigned int x=0; x<count; x++)
	{
		writeValueToStream(x,s);
	}
}


