#ifndef SHORTESTPATH_H
#define SHORTESTPATH_H

#include "stdafx.h"
#include "field.h"
#include "indexedorigins.h"
#include <vector>
using std::vector;
#include <set>
using std::set;
#include <limits>
#include "filter.h"

class TShortestPath
{
public:
	TShortestPath(unsigned int p_Begin, unsigned int p_End, float p_Length, shared_ptr<TIndexedOrigins_Vector> p_Path)
		: m_Begin(p_Begin)
		, m_End(p_End)
		, m_Length(p_Length)
		, m_MaxDistance((std::numeric_limits<float>::max)())
		, m_Middle(TIndexedOrigins::INVALIDINDEX)
		, m_Path(p_Path)
	{
		;
	}
	TShortestPath()
		: m_Begin(TIndexedOrigins::INVALIDINDEX)
		, m_End(TIndexedOrigins::INVALIDINDEX)
		, m_Length((std::numeric_limits<float>::max)())
		, m_MaxDistance((std::numeric_limits<float>::max)())
		, m_Middle(TIndexedOrigins::INVALIDINDEX)
	{
		;
	}
	virtual ~TShortestPath()
	{
	}

	// Data 
	unsigned int m_Begin;
	unsigned int m_End;
	unsigned int m_Middle;
	float m_Length;	
	mutable float m_MaxDistance;
	shared_ptr<TIndexedOrigins_Vector> m_Path;
	shared_ptr<TPlane> m_LocalSheet;

	// Methods
	unsigned int getMiddle() const { return m_Middle; }
	float getCircularity() const { return 0; }
	float getMaxDistance() const { return m_MaxDistance; }
	float getEccentricity(const TCoord3 & p) const;
	float getRhoDivDt(const TCoord3 & p) const;
	float getMaxRadius(const TCoord3 & p) const;

	void computeMaxDistance(const TCoord3 & p) const;

	void writeToStream(std::ofstream * s) const;
	static TShortestPath * readFromStream(std::ifstream * s);
};

typedef pair<TCoord3,TShortestPath*> TSpp;


class TShortestPathSet
{
public:
	typedef vector<shared_ptr<TShortestPath> > TPaths;

	TShortestPathSet(const TCoord3 p_SourceVoxel, shared_ptr<TIndexedOrigins_Vector> p_Eft)
		: m_Eft(p_Eft)
		, m_SourceVoxel(p_SourceVoxel)
	{
	}
	
	// Data
	TCoord3 m_SourceVoxel;
	shared_ptr<TIndexedOrigins_Vector> m_Eft;
	shared_ptr<TIndexedOrigins_Vector> m_Seft;
	shared_ptr<TIndexedOrigins_Vector> m_PathSet;
	TPaths m_Paths;

	// Methods
	float getMaxLength() const;
	TPaths getThreeLongestPaths() const;
	float getEccentricity() const;
	float getRadius() const;
	float getMaxRadius() const;
	void updatePathSet();
	void writeToStream(std::ofstream * s) const;
	static TShortestPathSet * readFromStream(std::ifstream * s);
	TShortestPathSet() {};
protected:
};


class TShortestPathSetFieldMeasure_PathCount : public TTypedMeasure<TShortestPathSet*>
{
public:
	TShortestPathSetFieldMeasure_PathCount(TTypedFieldInterface<TShortestPathSet*> * p_Field) 
		: TTypedMeasure<TShortestPathSet*>(p_Field) {};

	virtual float toFloat(const TCoord3 & p) const { return m_Field->vvaluep(p) ? m_Field->vvaluep(p)->m_Paths.size() : 0; }
	virtual float vvaluex(const unsigned int x) const { return m_Field->vvaluex(x) ? m_Field->vvaluex(x)->m_Paths.size() : 0; }
	virtual string getName() const { return "path count"; };
	virtual MEASURETYPE getMeasureType() const { return MEASURETYPE_SHORTESTPATHSET_PATHCOUNT; }
	virtual TType::TYPE getSourceType() const { return TType::TYPE_SHORTESTPATHSET; }
	virtual TMeasure * create(TField * p_Field) const { return new TShortestPathSetFieldMeasure_PathCount(static_cast<TTypedFieldInterface<TShortestPathSet*>*>(p_Field)); }
};


class TShortestPathSetFieldMeasure_MaximumCircularity : public TTypedMeasure<TShortestPathSet*>
{
public:
	TShortestPathSetFieldMeasure_MaximumCircularity(TTypedFieldInterface<TShortestPathSet*> * p_Field) 
		: TTypedMeasure<TShortestPathSet*>(p_Field) {};

	virtual float toFloat(const TCoord3 & p) const;
	virtual float vvaluex(const unsigned int x) const { return toFloat( m_Field->getIndexField()->vidx2coord(x) ); }
	virtual string getName() const { return "max circularity"; };
	virtual MEASURETYPE getMeasureType() const { return MEASURETYPE_SHORTESTPATHSET_MAXCIRCULARITY; }
	virtual TType::TYPE getSourceType() const { return TType::TYPE_SHORTESTPATHSET; }
	virtual TMeasure * create(TField * p_Field) const { return new TShortestPathSetFieldMeasure_MaximumCircularity(static_cast<TTypedFieldInterface<TShortestPathSet*>*>(p_Field)); }
};

class TShortestPathSetField : public TPointerSparseTypedField<TShortestPathSet*>
{
public:
	TShortestPathSetField(shared_ptr<TIndexMapper> p_IndexField)
		: TPointerSparseTypedField<TShortestPathSet*>(p_IndexField)
		, m_SeftTau(4.0f)
	{
	}
	virtual ~TShortestPathSetField() 
	{ 
		clear(); 
	}
	virtual const string getTypeName() { return "shortest path set"; }
	virtual TType::TYPE getType() const { return TType::TYPE_SHORTESTPATHSET; }
	virtual void accept(TFieldVisitor * v) { v->visitShortestPathSetField(this); }
	virtual shared_ptr<TMeasure> getDefaultMeasure() { return shared_ptr<TMeasure>(new TShortestPathSetFieldMeasure_PathCount(this)); }
	virtual void writeValueToStream(unsigned int x, std::ofstream * s) const 
	{ 
		vvaluex(x)->writeToStream(s);
	}
	virtual void readValueFromStream(unsigned int x, std::ifstream * s) 
	{ 
		wvaluex(x) = TShortestPathSet::readFromStream(s);
	}
	static TShortestPathSetField * readFromStream(std::ifstream * s, shared_ptr<TIndexMapper> p_IndexField);
	virtual void writeToStream(std::ofstream * s) const;

	float m_SeftTau;
};


#endif

