#include "stdafx.h"

#include "skeletonizer.h"

#include "settings.h"
#include "globals.h"
#include "camera2.h"
#include "wx/filename.h"
#include "wx/file.h"
#include "field.h"
#include "abstractfilter.h"
#include "utils.h"
#include "indexedoriginsfield.h"
#include "indexedorigins_cells.h"
#include "deltaomega.h"
#include "result.h"
#include "colormap.h"
#include "model.h"
#include "danielsson3d.h"
#include "histogram.h"
#include "component.h"
#include "geometry.h"
#include "fieldadaptors.h"
#include "scriptedfield.h"
#include "logwriter.h"
#include "action.h"

using namespace std;



TSkeletonizer::TSkeletonizer(const string & p_Filename)
	: m_Initialized(false)
	, m_NormalizeBy(1.0f)
	, m_SurfaceSkeletonField(0)
	, m_ForegroundSurfaceSkeletonField(0)
	, m_BackgroundSurfaceSkeletonField(0)
	, m_InputFilename(p_Filename)
{
	g_Mediator.addLayerSet(p_Filename);
	shared_ptr<TSkeletonizer> skel( this );
	g_Mediator.getCurrentLayerSet()->m_Skeletonizer = skel;
}

void TSkeletonizer::init(const string & p_Filename)
{
	m_InputFilename = p_Filename;
	const bool RELEASE = g_Settings.m_ReleaseMode;
	m_Result.add("peak page file usage before", niceprintuint(getProcessInfo().PeakPagefileUsage) );
	m_Result.add("filename", wxFileName(p_Filename).GetName().ToStdString() );


	*g_Log << "Loading field \"" << p_Filename.c_str() << "\"...\n"; g_Mediator.yield(); 
	wxStopWatch StopWatch;
	StopWatch.Start(0);
	m_Result.add("memory before loading field", niceprintuint(getProcessInfo().WorkingSetSize) );
	TUnsignedCharField3* Image = static_cast<TUnsignedCharField3*>( TFieldFactory::constructFromFile(p_Filename) );

	if (abs(g_Parameters.m_ScaleFactor - 1) >= .1)
	{
		wxLogMessage("Scale volume by a factor %.1lf", g_Parameters.m_ScaleFactor);
		std::unique_ptr<TUnsignedCharField3> scaled(scale(Image, g_Parameters.m_ScaleFactor));
		delete Image;
		Image = crop(scaled.get());
	}
	
	// Todo: possible inflate of model, for polygonal soups
	// Also: remove disconnected components
	if(g_Settings.m_InflateInput)
	{
		*g_Log << "Inflating field by " << g_Settings.m_InflateInput << " voxels.\n";
		inflate(Image);
	}
		
//	removeDisconnectedComponents(Image);

	m_Result.add("time for loading field", (float)StopWatch.Time()/1000.0f);
	m_Result.add("memory after loading field", niceprintuint(getProcessInfo().WorkingSetSize) );
	if(!RELEASE) *g_Log << "done\n"; g_Mediator.yield();

	m_TotalTimeWithInit.start();
	init(Image);
}

void TSkeletonizer::inflate(TUnsignedCharField3* Image)
{
	unsigned int x,y,z;
	const int W = g_Settings.m_InflateInput;
	for(z=0; z<Image->getMaxZ(); z++)
	for(y=0; y<Image->getMaxY(); y++)
	for(x=0; x<Image->getMaxX(); x++)
	{
		const TCoord3 p(x,y,z);
		if( Image->valuep(p) & 1 )
		{
			int a,b,c;
			for(a=x-W; a<=x+W; a++)
			for(b=y-W; b<=y+W; b++)
			for(c=z-W; c<=z+W; c++)
			{
				const TCoord3 q(a,b,c);
				if(p.distance2(q) <= W*W)
				{
					Image->valuep(q) |= 16;
				}
			}
		}
	}

	for(z=0; z<Image->getMaxZ(); z++)
	for(y=0; y<Image->getMaxY(); y++)
	for(x=0; x<Image->getMaxX(); x++)
	{
		const TCoord3 p(x,y,z);
		if( Image->valuep(p) & 16 )
		{
			Image->valuep(p) = 1;
		}
	}
}

void TSkeletonizer::removeDisconnectedComponents(TUnsignedCharField3* Image)
{
	unsigned int x,y,z;
	
	vector<unsigned int> ComponentSizes;
	ComponentSizes.push_back(0);
	ComponentSizes.push_back(0);
	unsigned int MaxSizeId = 0;

	for(z=0; z<Image->getMaxZ(); z++)
	for(y=0; y<Image->getMaxY(); y++)
	for(x=0; x<Image->getMaxX(); x++)
	{
		const TCoord3 p(x,y,z);
		if( Image->valuep(p) != 1 ) continue;
		
		ComponentSizes.push_back(0);
		if(ComponentSizes.size() > 255) throw string("Too many disconnected components in shape");

		queue<TCoord3> Q;
		Q.push(p);

		while(!Q.empty())
		{
			const TCoord3 p = Q.front();
			Q.pop();
			Image->valuep(p) = ComponentSizes.size();
			ComponentSizes.back()++;

			TCoord3 np;
			for(np.x=p.x-1; np.x<=p.x+1; np.x++) // neighbors
			for(np.y=p.y-1; np.y<=p.y+1; np.y++)
			for(np.z=p.z-1; np.z<=p.z+1; np.z++)
			{
				if(Image->valuep(np) == 1)
				{
					Image->valuep(np) = ComponentSizes.size();
					Q.push(np);
				}						
			}
		}

		if(ComponentSizes.back() > MaxSizeId) MaxSizeId = ComponentSizes.size();
	}

	if( ComponentSizes.size() >= 3 )
	{
		*g_Log << "Removing " << (unsigned int)(ComponentSizes.size() - 3) << " disconnected components.\n";
	}

	for(z=0; z<Image->getMaxZ(); z++)
	for(y=0; y<Image->getMaxY(); y++)
	for(x=0; x<Image->getMaxX(); x++)
	{
		const TCoord3 p(x,y,z);
		if( Image->valuep(p) != MaxSizeId ) Image->valuep(p) = 0;
		else Image->valuep(p) = 1;
	}
}

void TSkeletonizer::init(TUnsignedCharField3* Image )
{
	const bool RELEASE = g_Settings.m_ReleaseMode;

	m_TotalTimeWithInit.start();

	wxStopWatch StopWatch;
	StopWatch.Start(0);

	if(!RELEASE) *g_Log << "Constructing omega... "; g_Mediator.yield();
	Omega.reset( new TOmega() );
	if(!RELEASE) *g_Log << "done\n"; g_Mediator.yield();
	m_Result.add("memory after constructing omega", niceprintuint(getProcessInfo().WorkingSetSize) );

	if(!RELEASE) *g_Log << "Init index fields.... "; g_Mediator.yield();
	StopWatch.Start(0);
	Omega->initIndexFields(Image); 
	m_Result.add("time for init flagfield", (float)StopWatch.Time()/1000.0f );
	m_Result.add("memory after init flagfield", niceprintuint(getProcessInfo().WorkingSetSize) );
	delete Image; 
	if(!RELEASE) *g_Log << "done\n"; g_Mediator.yield();

	StopWatch.Start(0);
	if(!RELEASE) *g_Log << "Constructing AdjacencyList... ";
	Omega->initDeltaOmega();
	if(!RELEASE) *g_Log << "done\n";
	m_Result.add("time for init deltaomega", (float)StopWatch.Time()/1000.0f );
	m_Result.add("memory after init deltaomega", niceprintuint(getProcessInfo().WorkingSetSize) );

	StopWatch.Start(0);
	Omega->initFT();
	m_Result.add("time for init FT", (float)StopWatch.Time()/1000.0f );
	m_Result.add("memory after init FT", (unsigned int) getProcessInfo().WorkingSetSize );
	

	m_Result.add("boundary voxels", Omega->m_BoundaryIndexField->getMaxIndex() );
	m_Result.add("dimensions", string(wxString::Format("%i x %i x %i", Omega->m_ForegroundIndexField->getMaxX(), Omega->m_ForegroundIndexField->getMaxY(), Omega->m_ForegroundIndexField->getMaxZ()).c_str()) );
	m_Result.add("object voxel count", Omega->m_ForegroundIndexField->getMaxIndex() );
	m_Result.add("epoch", wxString::Format("%i", (int)wxDateTime::Now().GetTicks()).ToStdString()  );

	*g_Log << "Dimensions: " << Omega->m_ForegroundIndexField->getMaxX() << "x" << Omega->m_ForegroundIndexField->getMaxY() << "x" << Omega->m_ForegroundIndexField->getMaxZ() << "\n";
	*g_Log << "#Boundary voxels: " << Omega->m_BoundaryIndexField->getMaxIndex() << "\n";
	*g_Log << "#Object voxels: " << Omega->m_ForegroundIndexField->getMaxIndex() << "\n";


	if(!RELEASE) *g_Log << "Creating spatial subdivision... "; g_Mediator.yield();
	Omega->m_DeltaOmega->createSpatialSubdivision();
	if(!RELEASE) *g_Log << " done\n";
	m_Result.add("time for spatial subdivisions", (float)StopWatch.Time()/1000.0f );
	m_Result.add("memory after spatial subdivision", niceprintuint(getProcessInfo().WorkingSetSize) );

	addOmegaLayers();
}

void TSkeletonizer::addOmegaLayers()
{
	if(Omega->m_ForegroundIndexField) 
	{
		shared_ptr<TLayer> NewLayer( new TLayer(Omega->m_ForegroundIndexField, "indexfield fg") );
		NewLayer->m_AllVoxels->setCheckedForRendering( false );
		NewLayer->m_Filter->m_UpperValue = (std::numeric_limits<unsigned int>::max)()/2;
		g_Mediator.addLayer(NewLayer);
	}

	if(Omega->m_BackgroundIndexField) 
	{
		shared_ptr<TLayer> NewLayer( new TLayer(Omega->m_BackgroundIndexField, "indexfield bg") );
		NewLayer->m_AllVoxels->setCheckedForRendering( false );
		NewLayer->m_Filter->m_UpperValue = (std::numeric_limits<unsigned int>::max)()/2;
		g_Mediator.addLayer(NewLayer);
	}

	if(Omega->m_BoundaryIndexField) 
	{
		shared_ptr<TLayer> NewLayer( new TLayer(Omega->m_BoundaryIndexField, "indexfield boundary") );
		NewLayer->m_AllVoxels->setCheckedForRendering( false );
		NewLayer->m_Filter->m_UpperValue = (std::numeric_limits<unsigned int>::max)()/2;
		g_Mediator.addLayer(NewLayer);
	}

	if(Omega->m_BackgroundBoundaryIndexField && Omega->m_BackgroundBoundaryIndexField.get() != Omega->m_BoundaryIndexField.get()) 
	{
		shared_ptr<TLayer> NewLayer( new TLayer(Omega->m_BackgroundBoundaryIndexField, "indexfield boundary bg") );
		NewLayer->m_AllVoxels->setCheckedForRendering( false );
		NewLayer->m_Filter->m_UpperValue = (std::numeric_limits<unsigned int>::max)()/2;
		g_Mediator.addLayer(NewLayer);
	}
}



void TSkeletonizer::perform()
{
	const bool RELEASE = g_Settings.m_ReleaseMode;

	TDeltaOmega & DO = *Omega->m_DeltaOmega;


	if( Omega->m_ForegroundIndexField->getMaxX() <= 128 
		&& Omega->m_ForegroundIndexField->getMaxY() <= 128
		&& Omega->m_ForegroundIndexField->getMaxZ() <= 128
		)
	{
		*g_Log << "Adjusting settings for small resolution shape.\n";
		g_Parameters.m_MinimumMergedGeodesicsVertexCount = 0;
		g_Parameters.m_DiscardSmallComponentBoundaries = 0;
		g_Parameters.m_SimplifiedEftThreshold = 1.0f;
	}

	// Create fields 
	if(!RELEASE) *g_Log << "Creating fields... "; g_Mediator.yield();
	wxStopWatch StopWatch;
	StopWatch.Start(0);


	m_BoundaryIndexField = Omega->m_BoundaryIndexField;
	if(Omega->m_BackgroundBoundaryIndexField) m_BackgroundBoundaryIndexField = Omega->m_BackgroundBoundaryIndexField;
	else m_BackgroundBoundaryIndexField = Omega->m_BoundaryIndexField;


	m_ForegroundSskelIndexField.reset( new TSubIndexMapper(Omega->m_ForegroundIndexField.get()) );
	m_ForegroundSskelIndexField->m_SubIndex.reserve( Omega->m_ForegroundIndexField->getMaxIndex() / 10 );
	m_SskelIndexField = m_ForegroundSskelIndexField;


 
	if(g_Parameters.m_ComputeBackgroundSskel)
	{
		m_BackgroundSskelIndexField.reset( new TSubIndexMapper(Omega->m_BackgroundIndexField.get()) );
		m_BackgroundSskelIndexField->m_SubIndex.reserve( Omega->m_BackgroundIndexField->getMaxIndex() / 10 );
	}

	if(g_Parameters.m_ComputeCskel)
	{
		m_CskelIndexField.reset( new TSubIndexMapper(Omega->m_ForegroundIndexField.get()) );
		m_CskelIndexField->m_SubIndex.reserve( Omega->m_ForegroundIndexField->getMaxIndex() / 1000 );
		m_LoopField.reset( new TSparseUcharField3(m_CskelIndexField) );
	}

	
	if(g_Parameters.m_ComputeCskel && g_Parameters.m_ComputeImportanceForCskel)
	{
		m_CurveSkeletonField.reset( new TSparseFloatField3(m_CskelIndexField) );
		m_CurveSkeletonField->m_Values.reserve( m_CskelIndexField->m_SubIndex.capacity() );
	}


	m_ForegroundSpSetField.reset( new TShortestPathSetField(m_SskelIndexField));
	m_ForegroundSpSetField->m_Values.reserve( m_SskelIndexField->m_SubIndex.capacity() );
	m_ForegroundSpSetField->m_SeftTau = g_Parameters.m_SimplifiedEftThreshold;


	if(g_Parameters.m_ComputeBackgroundSskel)
	{
		m_BackgroundSpSetField.reset( new TShortestPathSetField(m_BackgroundSskelIndexField));
		m_BackgroundSpSetField->m_Values.reserve( m_BackgroundSskelIndexField->m_SubIndex.capacity() );
	}

	if( g_Parameters.m_KeepTotalCollapse ) 
	{
		m_TotalCollapseField.reset( new TSparseIOField(Omega->m_ForegroundIndexField, Omega->m_BoundaryIndexField));
	}
	

	m_Result.add("time for creating fields", (float)StopWatch.Time()/1000.0f );
	m_Result.add("memory after creating fields", (unsigned int) getProcessInfo().WorkingSetSize );
	*g_Log << "done\n"; g_Mediator.yield();


	const bool USELOGFILE = g_Parameters.m_UseLogFile;
	TLogFileWriter * logfile = USELOGFILE ? new TLogFileWriter("log.txt") : 0;

	m_NormalizeBy =	g_Parameters.m_Equalize 
		? sqrtf(Omega->m_BoundaryIndexField->getMaxIndex()): Omega->m_BoundaryIndexField->getMaxIndex();

	wxStopWatch TotalTime; 
	TotalTime.Start(0);




	{
		unsigned int MaximumCollapseSize = 0;
		

		if(g_Parameters.m_ComputeBackgroundSskel)
		{
			*g_Log << "Processing background voxels, ";
			m_ProcessingBackground = true;
			m_BaseIndexField = Omega->m_BackgroundIndexField;
			m_SskelIndexField = m_BackgroundSskelIndexField;
			m_SurfaceSkeletonField = m_BackgroundSurfaceSkeletonField;
			m_SpSetField = m_BackgroundSpSetField;
			Omega->m_FT = Omega->m_BackgroundFt;

			processAllVoxels();
		}

		{
			*g_Log << "Processing foreground voxels, ";
			m_ProcessingBackground = false;
			m_BaseIndexField = Omega->m_ForegroundIndexField;
			m_SskelIndexField = m_ForegroundSskelIndexField;
			m_SurfaceSkeletonField = m_ForegroundSurfaceSkeletonField;
			m_SpSetField = m_ForegroundSpSetField;
			Omega->m_FT = Omega->m_ForegroundFt;

			if(g_Parameters.m_OnlyCskelIterative)
				processSmart();
			else
				processAllVoxels();
		}

 
//		m_Result.add("time for geodesics", m_GeodesicTime);
//		m_Result.add("time for collapse", m_CollapseTime);
//		m_Result.add("time for loop detection", m_IsLoopTime);
//		m_Result.add("time for cskel points", m_TimeSpentOnCskelPoints);
//		m_Result.add("time for non-cskel points", m_TimeSpentOnOtherPoints);


	}
	
	m_Result.add("memory after collapse", niceprintuint(getProcessInfo().WorkingSetSize) );

	if(g_Parameters.m_KeepFt)
	{	
		{
			TTypedFieldInterface<TIndexedOrigins_Vector*>* Output = static_cast<TTypedFieldInterface<TIndexedOrigins_Vector*>*>( g_Mediator.getCurrentLayerSet()->newIoField( Omega->m_ForegroundIndexField, Omega->m_BoundaryIndexField, "FG FT" ) );
			Output->clear();
			for(unsigned int x=0; x<Omega->m_ForegroundFt->m_Origins.size(); x++)
			{	
				if((Omega->m_ForegroundFt->m_Origins)[x].size() == 0) continue;
				Output->wvaluex(x) = new TIndexedOrigins_Vector( (Omega->m_ForegroundFt->m_Origins)[x] );
			}
			Output->getLayer()->onFieldChanged();
		}

		if( Omega->m_BackgroundFt )
		{
			TTypedFieldInterface<TIndexedOrigins_Vector*>* Output = static_cast<TTypedFieldInterface<TIndexedOrigins_Vector*>*>( g_Mediator.getCurrentLayerSet()->newIoField( Omega->m_BackgroundIndexField, Omega->m_BackgroundBoundaryIndexField, "BG FT" ) );
			Output->clear();
			for(unsigned int x=0; x<Omega->m_BackgroundFt->m_Origins.size(); x++)
			{	
				if((Omega->m_BackgroundFt->m_Origins)[x].size() == 0) continue;

				Output->wvaluex(x) = new TIndexedOrigins_Vector( (Omega->m_BackgroundFt->m_Origins)[x] );
			}
			Output->getLayer()->onFieldChanged();
		}
	}

	Omega->m_FT.reset();
	if(Omega->m_ForegroundFt)
	{
		if(! Omega->m_ForegroundFt.unique()) throw string("!Omega->m_ForegroundFt.unique()");
		if(!RELEASE) *g_Log << "Resetting ForegroundFt\n";
		Omega->m_ForegroundFt.reset();
	}

	m_Result.add("memory after freeing FT", niceprintuint(getProcessInfo().WorkingSetSize) );

	if(logfile) { *logfile << "\ndone with phases\n"; logfile->flush(); }


	// ==========================================================================
	float TotalTimeWithInit;
	// Stop timing
	{
		const float WithoutInit = (float)TotalTime.Time()/1000.0f;
		m_Result.add("time for total without initialization", WithoutInit );
		
		TotalTimeWithInit = (float)m_TotalTimeWithInit.stop();
		m_Result.add("time for total with initialization", TotalTimeWithInit );

		m_Result.add("time for initialization", TotalTimeWithInit-WithoutInit );
	}



	// Process information
	{
		TDeltaOmega::TGeodesicCache::iterator it;
		unsigned int c = 0;
		for(it = DO.m_Geodesics.begin(); it != DO.m_Geodesics.end(); it++)
		{
			if(it->second->m_Path)
				c += it->second->m_Path->vertexcount();
		}
		
		m_Result.add("geodesic count", DO.m_ComputedGeodesics );
		m_Result.add("shortest paths in cache", (unsigned int)DO.m_Geodesics.size() );
		m_Result.add("voxel count in cached shortest paths", c);
//		m_Result.add("surface collapse count", m_SurfaceCollapseCount  );
//		m_Result.add("peak page file usage after", niceprintuint(getProcessInfo().PeakPagefileUsage) );

		unsigned int mismatch = m_ForegroundSskelIndexField->m_SubIndex.size() - (Omega->m_ForegroundIndexField->getMaxIndex() / 10);
		m_Result.add("foreground ss index mismatch", niceprintuint(mismatch) );

	}


	// =================================================================================================
	// Add layers
	if(logfile) { *logfile << "\nbefore adding layers\n"; logfile->flush(); }
	if(!RELEASE) *g_Log << "Adding layers...\n"; g_Mediator.yield();

	addLayers();

	m_Result.add("memory after adding layers", niceprintuint(getProcessInfo().WorkingSetSize) );
	
	if(g_Settings.m_TryOpenMesh)
	{
		if( tryOpenMesh() )
		{
		}
		else
		{
			m_Model.reset( TModel::createUsingMarchingCubes(
					Omega->m_BoundaryIndexField.get()
					, Omega->m_ForegroundIndexField.get()
					, (g_Parameters.m_ComputeBackgroundSskel ? g_Parameters.m_InvertBorder+1 : 1)
					) );
		}
	}

	m_Result.add("memory after opening model", niceprintuint(getProcessInfo().WorkingSetSize) );


	// =================================================================================================
	// Open camera
	tryOpenCamera(m_InputFilename);


	// =================================================================================================
	if(logfile) { *logfile << "\nbefore output resultsfield\n"; logfile->flush(); }


	if(logfile) { delete logfile; logfile = 0; }

	if(!RELEASE) *g_Log << "Clearing geodesic cache...\n"; g_Mediator.yield();
	Omega->m_DeltaOmega->clearGeodesicCache();
	if(!g_Parameters.m_KeepShortestPaths)
	{
		for(unsigned int x=0; x<m_ForegroundSpSetField->getMaxIndex(); x++)
		{
			TShortestPathSet * sps = m_ForegroundSpSetField->vvaluex(x);
			for(unsigned int y=0; y<sps->m_Paths.size(); y++)
			{
				sps->m_Paths[y]->m_Path.reset();
			}
		}
	}

	m_Result.add("memory clear geodesics", niceprintuint(getProcessInfo().WorkingSetSize) );
	


	if(!RELEASE)
	{
		TStringPairProcessor Parameters;
		g_Parameters.acceptStringPairProcessor(&Parameters);
		*g_Log << "\nParameters: \n";
		Parameters.printToLog();
		*g_Log << "\nResults: \n";
		m_Result.printToLog();

		// Print timings etc. to all results file
		string hashResults = m_Result.getPerlHash();
		string hashParameters = Parameters.getPerlHash();

		string line = "push @results, {'results'=>"+hashResults+", 'parameters'=>"+hashParameters+"};\n";
		std::ofstream file("results/allresults.txt", ios_base::app);
		file << line;
		file.close();
	}

	*g_Log << "\nFinished in " << TotalTimeWithInit << "s \n";

	if(g_Parameters.m_ExtendCollapseToLoops)
	{
		TAction_ExtendCollapseToLoops Extend;
		Extend.perform();
	}

	g_Mediator.redrawCanvas();
	m_Initialized = true;
}


void TSkeletonizer::addLayers()
{
	if(m_ForegroundSskelIndexField) 
	{
		shared_ptr<TLayer> NewLayer( new TLayer(m_ForegroundSskelIndexField, "S-skel index field") );
		NewLayer->m_AllVoxels->setCheckedForRendering( false );
		g_Mediator.addLayer(NewLayer);

	}
	if(m_BackgroundSskelIndexField) 
	{
		shared_ptr<TLayer> NewLayer( new TLayer(m_BackgroundSskelIndexField, "S-skel index field (bg)") );
		NewLayer->m_AllVoxels->setCheckedForRendering( false );
		g_Mediator.addLayer(NewLayer);

	}
	if(m_CskelIndexField) 
	{
		shared_ptr<TLayer> NewLayer( new TLayer(m_CskelIndexField, "C-skel index field") );
		NewLayer->m_AllVoxels->setCheckedForRendering( false );
		g_Mediator.addLayer(NewLayer);
	}

	if(m_FtField)
	{
		shared_ptr<TLayer> NewLayer( new TLayer(m_FtField, "FT") );
		NewLayer->m_AllVoxels->setCheckedForRendering( false );
		NewLayer->m_RenderPrimitiveType = 1;
		NewLayer->m_RenderLines = true;
		NewLayer->m_RenderSelectionIndicators = false;
		g_Mediator.addLayer( NewLayer );
	}

	if(m_TotalCollapseField) 
	{
		shared_ptr<TLayer> NewLayer( new TLayer(m_TotalCollapseField, "Collapse") );
		NewLayer->m_Selection->setCheckedForRendering( true );
		NewLayer->m_AllVoxels->setCheckedForRendering( false );
		g_Mediator.addLayer( NewLayer );
	}

	if(m_LoopField) 
	{	
		shared_ptr<TLayer> LoopFieldLayer( new TLayer(m_LoopField, "C-skel detector", "C-skel detector") );
		LoopFieldLayer->m_Filter->m_LowerValue = 0.9f;
		g_Mediator.addLayer( LoopFieldLayer );
	}

	if(m_CurveSkeletonField) 
	{
		shared_ptr<TLayer> NewLayer( new TLayer(m_CurveSkeletonField, "C-skel measure", "C-skel measure") );
		NewLayer->m_FilterMeasure.reset( new TFloatFieldMeasure_Sqrt(m_CurveSkeletonField.get()) );
		NewLayer->onFieldChanged();
		g_Mediator.addLayer(NewLayer);
	}

	if(m_ForegroundSpSetField)
	{
		shared_ptr<TLayer> NewLayer( new TLayer(m_ForegroundSpSetField, "sp set") );
		NewLayer->m_AllVoxels->setCheckedForRendering(false);
		g_Mediator.addLayer(NewLayer);

		addShortestPathSetAdaptorField(m_ForegroundSpSetField.get(), "spset fg");
		m_ForegroundSurfaceSkeletonField = static_cast<TTypedFieldInterface<float>*>( g_Mediator.getCurrentLayerSet()->getField("spset fg maximum length") );
		if(!m_ForegroundSurfaceSkeletonField) throw string("!m_ForegroundSurfaceSkeletonField");

		if(!g_Mediator.getCurrentLayerSet()->getLayer("spset fg maximum length")) throw string("!spset fg maximum length");
		g_Mediator.getCurrentLayerSet()->getLayer("spset fg maximum length")->m_Filter->m_LowerValue = 5.0f;
	}


	if(m_BackgroundSpSetField)
	{
		shared_ptr<TLayer> NewLayer( new TLayer(m_BackgroundSpSetField, "sp set (bg)") );
		g_Mediator.addLayer(NewLayer);

		addShortestPathSetAdaptorField(m_BackgroundSpSetField.get(), "spset bg");
		m_BackgroundSurfaceSkeletonField = static_cast<TTypedFieldInterface<float>*>( g_Mediator.getCurrentLayerSet()->getField("spset bg maximum length") );
		if(!m_BackgroundSurfaceSkeletonField) throw string("!m_BackgroundSurfaceSkeletonField");

		if(!g_Mediator.getCurrentLayerSet()->getLayer("spset bg maximum length")) throw string("!spset bg maximum length");
		g_Mediator.getCurrentLayerSet()->getLayer("spset bg maximum length")->m_Filter->m_LowerValue = 12.0f;
	}
}


TSkeletonizerThread::TSkeletonizerThread(unsigned int p_From, unsigned int p_To, TSkeletonizer * p_Skel)
	: wxThread(wxTHREAD_JOINABLE)
	, m_From(p_From)
	, m_To(p_To)
	, m_Skel(p_Skel)
{
	m_GeodesicTime = 0;
	m_CollapseTime = 0;
	m_IsLoopTime = 0;
	m_TimeSpentOnCskelPoints = 0;
	m_TimeSpentOnOtherPoints = 0;
	m_SurfaceCollapseCount = 0;

	EFT = new TIndexedOrigins_Vector();
	EFT->reserve(32);
	MergedGeodesics = new TIndexedOrigins_Vector();
	MergedGeodesics->reserve( g_Mediator.getSkeletonizer()->Omega->m_DeltaOmega->m_AdjacencyList.size() / 2 );
	SurfaceCollapse = new TIndexedOrigins_Cells();
	SurfaceCollapse->reserve( g_Mediator.getSkeletonizer()->Omega->m_DeltaOmega->m_AdjacencyList.size() / 2 );

	if( !g_Parameters.m_OptimizeSpatialSubdivision ) 
	{
		m_FindConnectedComponentsField = static_cast<TTypedFieldInterface<unsigned int>*>( g_Mediator.getCurrentLayerSet()->newField(TType::TYPE_UINT, g_Mediator.getSkeletonizer()->m_BoundaryIndexField, "Find connected components field"));
		m_FindConnectedComponentsField->clear();
	}

}

TSkeletonizerThread::~TSkeletonizerThread()
{

	delete EFT;
	delete MergedGeodesics;
	delete SurfaceCollapse;
}

void TSkeletonizerThread::OnExit()
{
}

void * TSkeletonizerThread::Entry()
{
	processAllVoxels();
	return 0;
}

void TSkeletonizerThread::processAllVoxels()
{
	*g_Log << "From " << m_From << " to " << m_To << "\n";
	for(unsigned int Index=m_From; Index < m_To; Index++)
	{
		const TCoord3 p = g_Mediator.getSkeletonizer()->m_BaseIndexField->vidx2coord(Index);
		processVoxel(p, Index);
	}
}

void TSkeletonizer::processAllVoxels()
{
	// /*
	*g_Log << "parallel processing of all voxels on "<< wxThread::GetCPUCount() << " CPUs \n";
	const unsigned int EndIndex = !g_Parameters.m_ProcessObjectPartly ? m_BaseIndexField->getMaxIndex() : m_BaseIndexField->getMaxIndex() / 4;

	TSkeletonizerThread Thread(0, EndIndex, this);
	Thread.processAllVoxels();

/*
	vector<wxThread*> threads;
	threads.push_back( new TSkeletonizerThread(0,EndIndex, this) );
	//threads.push_back( new TSkeletonizerThread(0,EndIndex/2, this) );
//	threads.push_back( new TSkeletonizerThread(EndIndex/2,EndIndex, this) );

	unsigned int x;
	for(x=0; x<threads.size(); x++)
	{
		if ( threads[x]->Create() != wxTHREAD_NO_ERROR )
		{
			*g_Log << "Can't create thread " << x << "!";
	    }
	}

	for(x=0; x<threads.size(); x++)
	{
		if ( threads[x]->Run() != wxTHREAD_NO_ERROR )
		{
			*g_Log << "Can't create thread " << x << "!";
	    }
	}

	for(x=0; x<threads.size(); x++)
	{
		threads[x]->Wait();
	}

	for(x=0; x<threads.size(); x++)
	{
		delete threads[x];
	}

	//delete thread2;
	// */
// */


 /*
	*g_Log << "\n["; for(unsigned int k=0; k<m_BaseIndexField->getMaxIndex()/5000+1; k++) *g_Log << "."; *g_Log << "]\n"; 
	*g_Log << "[";
	TPrecisionTimer PrecisionTimer; 
	PrecisionTimer.start();

	const unsigned int EndIndex = !g_Parameters.m_ProcessObjectPartly ? m_BaseIndexField->getMaxIndex() : m_BaseIndexField->getMaxIndex() / 4;
	for(unsigned int Index=0; Index < EndIndex; Index++)
	{
		if(Index % 5000 == 0) 
		{
			*g_Log << ".";
			g_Mediator.yield();
		}

		const TCoord3 p = m_BaseIndexField->vidx2coord(Index);
		processVoxel(p, Index);

	}
	const float Time = PrecisionTimer.stop();
	*g_Log << "] " << Time << "s\n";
	// */
}

void TSkeletonizer::processSmart()
{
	// Will not be an actual thread:
	TSkeletonizerThread Thread(0, 0, this);

	*g_Log << "iterative processing, following C-skel:\n";
	shared_ptr<TSparseUcharField3> Visited(new TSparseUcharField3(m_BaseIndexField) );

	unsigned int FoundCskelVoxels = 0;
	
	queue<TCoord3> RecentlyProcessed;
	*g_Log << "[";
	unsigned int Index=0;
	const unsigned int Threshold = (float)Omega->m_ForegroundIndexField->getMaxIndex() * 0.001f;
	while(FoundCskelVoxels < Threshold && Index < m_BaseIndexField->getMaxIndex() )
	{
		// Find start
		stack<TCoord3> S;
		while( Index<m_BaseIndexField->getMaxIndex() )
		{
			const TCoord3 p = m_BaseIndexField->vidx2coord(Index);
			if(Visited->valuep(p)) { Index++; continue; }
			Thread.processVoxel(p, Index);
			Visited->valuep(p) = 1;
			if( m_CskelIndexField->vcoord2idx(p) < TIndexField3::OUTSIDE )
			{
				S.push(p);
				FoundCskelVoxels++;
//				*g_Log << "Found starting voxel " << p.x << ", " << p.y << ", " << p.z << "\n";
				*g_Log << "s";
				Index++;
				break;
			}
			Index++;
		}

		TCoord3 np;
		while(!S.empty())
		{
			TCoord3 p = S.top();
			S.pop();
			RecentlyProcessed.push(p);

			for(np.x=p.x-1; np.x<=p.x+1; np.x++) // neighbors
			for(np.y=p.y-1; np.y<=p.y+1; np.y++)
			for(np.z=p.z-1; np.z<=p.z+1; np.z++)
			{
				if(m_BaseIndexField->vcoord2idx(np) == TIndexField3::OUTSIDE ) continue;
				if( Visited->valuep(np) ) continue;

				Thread.processVoxel(np, m_BaseIndexField->vcoord2idx(np) );
				Visited->valuep(np) = 1;
				if( m_CskelIndexField->vcoord2idx(np) != TIndexField3::OUTSIDE )
				{
					S.push(np);
					FoundCskelVoxels++;
					if(FoundCskelVoxels % 100 == 0)
					{
						*g_Log << ".";
						g_Mediator.yield();
					}
				}
			}
		}
	}
	*g_Log << "]\n";
}


void TSkeletonizerThread::processVoxel(const TCoord3 & p, unsigned int Index)
{
	// Create Extended FT
	computeEFT(p, EFT);
	if(EFT->size() == 0) return;
	PrecisionTimer.start();

	// Compute shortest-path set
	shared_ptr<TIndexedOrigins_Vector> Eft( new TIndexedOrigins_Vector(*EFT) );
	
	TShortestPathSet * sps = new TShortestPathSet(p, Eft);
	if(g_Parameters.m_KeepShortestPaths || g_Parameters.m_ComputeCskel) sps->m_PathSet.reset( new TIndexedOrigins_Vector() );
	computePathSet(sps);

	const float LastGeodesicTime = PrecisionTimer.stop();
	m_GeodesicTime += LastGeodesicTime;

	const TIndexedOrigins_Vector * const mergedgeodesics = sps->m_PathSet.get();

	bool IsCurveSkeleton = false;
	if(g_Parameters.m_ComputeCskel)
	{
		// Only compute collapse for possible candidates: this improves the time needed for determining C-skel
		// However, a low value is required to capture fine tubular structures
		if(mergedgeodesics->vertexcount() > g_Parameters.m_MinimumMergedGeodesicsVertexCount) // parameter
		{
			int Components = 2;

			PrecisionTimer.start();
			
			// /*
			m_Skel->m_MutexIsLoop.Lock();
			DilatedShortestPaths.clear();
			//dilate(p,mergedgeodesics,&DilatedShortestPaths,false);
			g_Mediator.getSkeletonizer()->Omega->m_DeltaOmega->dilate(g_Parameters.m_DilationDistance, mergedgeodesics,&DilatedShortestPaths,false);
			IsCurveSkeleton = g_Mediator.getSkeletonizer()->Omega->m_DeltaOmega->isLoop(
				&DilatedShortestPaths, 
				Components);
			m_Skel->m_MutexIsLoop.Unlock();
			// */

			// Alternative
			// /*
			// IsCurveSkeleton = Omega->m_DeltaOmega->isLoop2( sps->m_Eft.get(), sps->m_PathSet.get(), Components);
			// */

			const float LastLoopTime = PrecisionTimer.stop();
			m_IsLoopTime += LastLoopTime;

			
			if(IsCurveSkeleton)
			{	
				m_Skel->m_Mutex.Lock();
				m_Skel->m_CskelIndexField->m_Coord2IdxMap.insert( pair<TCoord3, unsigned int>(p, (unsigned int)m_Skel->m_CskelIndexField->m_SubIndex.size() ) );
				m_Skel->m_CskelIndexField->m_SubIndex.push_back(Index);
				m_Skel->m_LoopField->m_Values.push_back( Components );
				m_Skel->m_Mutex.Unlock();
			
				if(
					m_Skel->m_CurveSkeletonField
					|| m_Skel->m_TotalCollapseField
					)
				{
					SurfaceCollapse->resize(0);
					PrecisionTimer.start();
					unsigned int ComponentCount;
					float TotalSize = 0.0f;
					if(g_Parameters.m_OptimizeSpatialSubdivision)
					{
						//ComponentCount = m_Skel->Omega->m_DeltaOmega->computeCollapseUsingCells(&DilatedShortestPaths, SurfaceCollapse);
						//TotalSize = mergedgeodesics->vertexcount() + SurfaceCollapse->vertexcount();
						TComponentSet * cs = new TComponentSet(p);
						m_Skel->Omega->m_DeltaOmega->computeComponentSetUsingCells2(
							&DilatedShortestPaths, 
							cs
							);
						
						// Count cells larger than RobustComponentCount
						ComponentCount = 0;
						for(unsigned int x=0; x<cs->m_Cells.size(); x++)
						{
							const float size = (float)cs->m_Cells[x]->vertexcount() / (float)m_Skel->m_BoundaryIndexField->getMaxIndex();
							if(size > g_Parameters.m_RobustComponentTau)
							{
								ComponentCount++;
							}
						}

						//if(cs->m_Cells.size() > 2) // Note: cs->m_Cells[0] is empty
						if( ComponentCount >= 2)
						{
							TotalSize = m_Skel->m_BoundaryIndexField->getMaxIndex() - cs->getCellSizes().rbegin()->first;
						}

						delete cs;

					}
					else
					{
						unsigned int x;

						for(x=0; x<DilatedShortestPaths.size(); x++)
							m_FindConnectedComponentsField->wvaluex( DilatedShortestPaths[x] ) = 1;

						m_FindConnectedComponentsField->getLayer()->m_Filter->m_LowerValue = 0.5f;
						m_FindConnectedComponentsField->getLayer()->m_Filter->m_UpperValue = 1.5f;
			
						vector<set<unsigned int> > Sizes;
						m_Skel->Omega->m_DeltaOmega->findConnectedComponents(m_FindConnectedComponentsField->getLayer()->m_Filter.get(), m_FindConnectedComponentsField, true, &Sizes);
						ComponentCount = Sizes.size() - 1;
						TotalSize = 0.0f;
						for(x=0; x<Sizes.size(); x++)
							if(Sizes[x].size() > TotalSize)  TotalSize = Sizes[x].size();
						TotalSize = g_Mediator.getSkeletonizer()->m_BoundaryIndexField->getMaxIndex() - TotalSize;

						m_FindConnectedComponentsField->clear();
					}

					m_SurfaceCollapseCount++;
					const float LastCollapseTime = PrecisionTimer.stop();
					m_TimeSpentOnCskelPoints += LastCollapseTime;
					m_CollapseTime += LastCollapseTime;

				
					// Uncomment to assign rho = 0.0f to C-skel loops. Some problems in non-loop cases (hand).
					if(ComponentCount <= 2) 
					{
						TotalSize = 0.0f;
						m_Skel->m_LoopField->m_Values.back() = 1;
					}

					if(g_Parameters.m_Equalize) TotalSize = sqrt(TotalSize);

					if(m_Skel->m_CurveSkeletonField)
					{
						m_Skel->m_CurveSkeletonField->m_Values.push_back( TotalSize  );
					}

					if(m_Skel->m_TotalCollapseField) 
					{
						TIndexedOrigins_Vector * tc = new TIndexedOrigins_Vector();
						tc->merge( SurfaceCollapse );
						tc->merge( mergedgeodesics );
						m_Skel->m_TotalCollapseField->m_Values[Index] = tc;
					}
				}

				m_TimeSpentOnCskelPoints += LastGeodesicTime;
				m_TimeSpentOnCskelPoints += LastLoopTime;
			}
			else
			{
				m_TimeSpentOnOtherPoints += LastGeodesicTime;
				m_TimeSpentOnOtherPoints += LastLoopTime;			
			}
								
		} 
	}
	else // Do not compute curve skeleton
	{
		// skip
	}


	bool AssignedSps = false;
	if( !m_Skel->m_ProcessingBackground || !g_Parameters.m_UseBoundingBoxForBgFt )
	{
		TAction_ComputeSimplifiedEft::process(sps, g_Parameters.m_SimplifiedEftThreshold);
		if(sps->m_Seft->size() > 1 || IsCurveSkeleton)
		{
			m_Skel->m_Mutex.Lock();
			m_Skel->m_SskelIndexField->m_Coord2IdxMap.insert( pair<TCoord3, unsigned int>(p, (unsigned int)m_Skel->m_SskelIndexField->m_SubIndex.size() ) );
			m_Skel->m_SskelIndexField->m_SubIndex.push_back( Index );
			m_Skel->m_SpSetField->m_Values.push_back( sps  );
			m_Skel->m_Mutex.Unlock();
			
			AssignedSps = true;
		}
	}
	else // Background
	{
		// TODO TODO
		// If all voxels on foreground boundary then add path set
		{
			TAction_ComputeSimplifiedEft::process(sps, g_Parameters.m_SimplifiedEftThreshold);
			if(sps->m_Seft->size() > 1 || IsCurveSkeleton)
			{
				m_Skel->m_Mutex.Lock();
				m_Skel->m_SskelIndexField->m_Coord2IdxMap.insert( pair<TCoord3, unsigned int>(p, (unsigned int)m_Skel->m_SskelIndexField->m_SubIndex.size() ) );
				m_Skel->m_SskelIndexField->m_SubIndex.push_back( Index );
				m_Skel->m_SpSetField->m_Values.push_back( sps  );
				m_Skel->m_Mutex.Unlock();
				
				AssignedSps = true;
			}
		}
	}

	if(!g_Parameters.m_KeepShortestPaths)
	{
		if(sps->m_PathSet) sps->m_PathSet->clear();
	}

	// Delete paths: not possible: these are cached
//	{
//		for(unsigned int x=0; x<sps->m_Paths.size(); x++)
//			sps->m_Paths[x]->m_Path.reset();
//	}

	if(!AssignedSps) delete sps;
}




void TSkeletonizerThread::computeEFT(const TCoord3 & p, TIndexedOrigins_Vector * p_Output, bool p_Symmetrical)
{
	static TIndexedOrigins_Set ios;
	const vector<TIndexedOrigins_Vector> & FT = m_Skel->Omega->m_FT->m_Origins;

	ios.clear();
	ios.merge( &FT[ m_Skel->m_BaseIndexField->valuep(p) ] );
	TCoord3 np;
	for(np.x=p.x-1; np.x<=p.x+1; np.x++) // neighbors
	for(np.y=p.y-1; np.y<=p.y+1; np.y++)
	for(np.z=p.z-1; np.z<=p.z+1; np.z++)
	{
		if( m_Skel->m_BaseIndexField->vinside(np) )
		{
			const unsigned int npIndex = m_Skel->m_BaseIndexField->valuep(np);
			if(np.x < p.i || np.y < p.j || np.z < p.k) continue;  // parameter
			if( &FT[npIndex] )
			{
				ios.merge( &FT[npIndex] );
			}
		}
	}
	p_Output->resize(0);
	p_Output->merge( &ios );
}

void TSkeletonizerThread::computePathSet(TShortestPathSet * p_Par, const bool p_Debug)
{
	const bool & Debug = p_Debug;

	assert(p_Par->m_Eft);

	// Create shortest path (shortest geodesic) between each two points in the EFT
	float length = 0;
	TIndexedOrigins_Vector::const_iterator it,jt;
	p_Par->m_Paths.reserve(p_Par->m_Eft->size() * (p_Par->m_Eft->size()-1));
	for(it = p_Par->m_Eft->begin(); it != p_Par->m_Eft->end(); it++)
	{
		for(jt = it; jt != p_Par->m_Eft->end(); jt++)
		{
			if(*it == *jt) continue; 

			if( m_Skel->m_ProcessingBackground ) 
			{
				const unsigned int begin = *it;
				const unsigned int end = *jt;
				char fg = 0; 
				if( begin < m_Skel->Omega->m_BoundaryIndexField->getMaxIndex() ) fg++;
				if( end < m_Skel->Omega->m_BoundaryIndexField->getMaxIndex() ) fg++;
				if(fg == 2)
				{
					shared_ptr<TShortestPath> spinfo = m_Skel->Omega->m_DeltaOmega->getShortestPath(*it, *jt);
					p_Par->m_Paths.push_back( spinfo);
				}
				else if(fg == 1)
				{
					shared_ptr<TIndexedOrigins_Vector> io( static_cast<TIndexedOrigins_Vector*>(0) );
					shared_ptr<TShortestPath> spinfo( new TShortestPath(begin,end,999.9f, io ) );
					p_Par->m_Paths.push_back( spinfo);
				}
				else 
				{
					shared_ptr<TIndexedOrigins_Vector> io( static_cast<TIndexedOrigins_Vector*>(0) );
					shared_ptr<TShortestPath> spinfo( new TShortestPath(begin,end,0.0001f,io ) );
					p_Par->m_Paths.push_back( spinfo);
				}
			}
			else
			{
				shared_ptr<TShortestPath> spinfo = m_Skel->Omega->m_DeltaOmega->getShortestPath(*it, *jt);
				p_Par->m_Paths.push_back( spinfo);
			}
		}
	}
	if(p_Par->m_Paths.size() == 0) return;

	// Compute local tangent plane
	{
		const TCoord3 & p = p_Par->m_SourceVoxel;
		for(unsigned int x=0; x<p_Par->m_Paths.size(); x++)
		{
			shared_ptr<TShortestPath> sp_p = p_Par->m_Paths[x];
			const TCoord3 pa = m_Skel->Omega->m_BackgroundBoundaryIndexField->vidx2coord( sp_p->m_Begin );
			const TCoord3 pb = m_Skel->Omega->m_BackgroundBoundaryIndexField->vidx2coord( sp_p->m_End );

			// Plane equation: <N,(p-O)>=0 
			TVector3 N; 
			TVector3 O;
			TVector3 a(pa.x, pa.y, pa.z);
			a.x -= p.x; 
			a.y -= p.y; 
			a.z -= p.z;
			
			TVector3 b(pb.x, pb.y, pb.z);
			b.x -= p.x; 
			b.y -= p.y; 
			b.z -= p.z;
			N.add(a);
			N.subtract(b);
			N.normalize();
			// N = (a-b) / ||a-b||
			O.add(a); 
			O.add(b); 
			O.scale(0.5f);
			// O = (a+b)/2

			O.x += p.x;
			O.y += p.y;
			O.z += p.z;

			p_Par->m_Paths[x]->m_LocalSheet.reset( new TPlane(O, N) );
		}
	}

	p_Par->updatePathSet();
}


/*
void TSkeletonizerThread::dilate(const TCoord3 & p_Coord, const TIndexedOrigins_Vector * p_Input, TIndexedOrigins_Vector * p_Output, bool p_Reset)
{
	if(p_Output->size() < p_Input->size()) p_Output->reserve( p_Input->size() * 2 );

	g_Mediator.getSkeletonizer()->Omega->m_DeltaOmega->dilate(
		g_Parameters.m_DilationDistance
		, p_Input
		, p_Output
		, p_Reset
		);
}
*/



void TSkeletonizer::addShortestPathSetAdaptorField(TShortestPathSetField * p_Field, const string p_ShortName)
{
	if(p_Field->getType() != TType::TYPE_SHORTESTPATHSET) throw string("!spset");

	TShortestPathSetField* AdaptThisField = static_cast<TShortestPathSetField*>( p_Field );

	if(true) // !!!
	{		
		TTypedFieldInterface<float> * TypedAdaptorField = new TShortestPathSetFieldAdaptor_Length(AdaptThisField);
		shared_ptr<TField> AdaptorField( TypedAdaptorField );
		shared_ptr<TLayer> AdaptorLayer( new TLayer(AdaptorField, AdaptorField->getAdaptorName(), p_ShortName + " maximum length") );
		AdaptorLayer->m_AllVoxels->setCheckedForRendering(true);
		AdaptorLayer->m_RenderLines = true;
		AdaptorLayer->m_Depth = 1;
		g_Mediator.addLayer(AdaptorLayer, p_Field->getLayer()  );
	}

	if(true) // !!!
	{		
		TTypedFieldInterface<float> * TypedAdaptorField = new TShortestPathSetFieldAdaptor_Radius(AdaptThisField);
		shared_ptr<TField> AdaptorField( TypedAdaptorField );
		shared_ptr<TLayer> AdaptorLayer( new TLayer(AdaptorField, AdaptorField->getAdaptorName(), p_ShortName + " radius" ) );
		AdaptorLayer->m_Depth = 1;
		g_Mediator.addLayer(AdaptorLayer, p_Field->getLayer()  );
	}

	if(true) // !!!
	{		
		TTypedFieldInterface<float> * TypedAdaptorField = new TShortestPathSetFieldAdaptor_FpAngle(AdaptThisField);
		shared_ptr<TField> AdaptorField( TypedAdaptorField );
		shared_ptr<TLayer> AdaptorLayer( new TLayer(AdaptorField, AdaptorField->getAdaptorName(), p_ShortName + " angle" ) );
		AdaptorLayer->m_Depth = 1;
		g_Mediator.addLayer(AdaptorLayer, p_Field->getLayer()  );
	}

	{		
		TTypedFieldInterface<TIndexedOrigins*> * TypedAdaptorField = new TShortestPathSetFieldAdaptor_Eft(AdaptThisField);
		shared_ptr<TField> AdaptorField( TypedAdaptorField );
		shared_ptr<TLayer> AdaptorLayer( new TLayer(AdaptorField, AdaptorField->getAdaptorName(), p_ShortName + " eft" ) );
		AdaptorLayer->m_FilterMeasure.reset( new TIndexedOriginsFieldMeasure_MinDistance(TypedAdaptorField) );
		AdaptorLayer->m_Filter.reset( new TFloatFilter(&AdaptorLayer->m_FilterMeasure) );
		AdaptorLayer->m_Filter->m_LowerValue = 6.0f;
		AdaptorLayer->m_AllVoxels->setCheckedForRendering(false);
		AdaptorLayer->m_RenderLines = true;
		AdaptorLayer->m_Depth = 1;
		g_Mediator.addLayer(AdaptorLayer, p_Field->getLayer() );
	}


	{		
		TTypedFieldInterface<TIndexedOrigins*> * TypedAdaptorField = new TShortestPathSetFieldAdaptor_Seft(AdaptThisField);
		shared_ptr<TField> AdaptorField( TypedAdaptorField );
		shared_ptr<TLayer> AdaptorLayer( new TLayer(AdaptorField, AdaptorField->getAdaptorName() + wxString::Format(" tau=%.2f", p_Field->m_SeftTau).ToStdString(), p_ShortName + " seft") );
		AdaptorLayer->m_FilterMeasure.reset( new TIndexedOriginsFieldMeasure_CoordCount(TypedAdaptorField) );
		AdaptorLayer->m_Filter.reset( new TFloatFilter(&AdaptorLayer->m_FilterMeasure) );
		AdaptorLayer->m_Filter->m_LowerValue = 2.0f;
		AdaptorLayer->m_AllVoxels->setCheckedForRendering(false);
		AdaptorLayer->m_RenderLines = true;
		AdaptorLayer->m_Depth = 1;
		g_Mediator.addLayer(AdaptorLayer, p_Field->getLayer() );
	}

	if(false)
	{
		TTypedFieldInterface<float> * TypedAdaptorField = new TShortestPathSetFieldAdaptor_MaxPathVoxelDistance(AdaptThisField);
		shared_ptr<TField> AdaptorField( TypedAdaptorField );
		shared_ptr<TLayer> AdaptorLayer( new TLayer(AdaptorField, AdaptorField->getAdaptorName(), p_ShortName + " eccentricity") );
		AdaptorLayer->m_Depth = 1;
		AdaptorLayer->m_AllVoxels->setCheckedForRendering(true);
//		AdaptorLayer->m_Filter->m_LowerValue = 1.2f;
		AdaptorLayer->m_RenderLines = true;
		g_Mediator.addLayer(AdaptorLayer, p_Field->getLayer()  );
	}

	if(true)
	{	
		// Detect whether there is at least one path set available
		bool IsAvailable = false;
		for(unsigned int x=0; x<AdaptThisField->getMaxIndex(); x++)
		{
			if(AdaptThisField->vvaluex(x)->m_PathSet && AdaptThisField->vvaluex(x)->m_PathSet->size() != 0)
			{
				IsAvailable = true;
			}
		}
		if(IsAvailable)
		{
			TTypedFieldInterface<TIndexedOrigins*> * TypedAdaptorField = new TShortestPathSetFieldAdaptor_PathSet(AdaptThisField);
			shared_ptr<TField> AdaptorField( TypedAdaptorField );
			shared_ptr<TLayer> AdaptorLayer( new TLayer(AdaptorField, AdaptorField->getAdaptorName(), p_ShortName + " pathset") );
			AdaptorLayer->m_AllVoxels->setCheckedForRendering(false);
			AdaptorLayer->m_RenderLines = false;
			AdaptorLayer->m_Depth = 1;
			g_Mediator.addLayer(AdaptorLayer, p_Field->getLayer()  );
		}
	}

	// Add combination of curve and surface skeleton measures
	if( g_Mediator.getCurrentLayerSet()->getField("C-skel measure") 
		&& g_Mediator.getCurrentLayerSet()->getField("spset fg maximum length") 
		)
	{	
		m_SkeletonField.reset( new TVirtualSparseFloatField_Maximize(
			m_SskelIndexField,	
			&g_Mediator.getCurrentLayerSet()->getLayer("C-skel measure")->m_FilterMeasure, 
			&g_Mediator.getCurrentLayerSet()->getLayer("spset fg maximum length")->m_FilterMeasure) );
		shared_ptr<TLayer> NewLayer( new TLayer(m_SkeletonField, "Combined C-skel and S-skel measure", "combined measure fg") );
		NewLayer->m_Filter->m_LowerValue = g_Mediator.getCurrentLayerSet()->getLayer("spset fg maximum length")->m_FilterMeasure->getMinValue();

		NewLayer->setCheckedForRendering(false);
		g_Mediator.addLayer(NewLayer);
	}

}



bool TSkeletonizer::tryOpenCamera(const string & Filename)
{
	wxString camerafile = Filename.c_str();
	camerafile.Replace(".gz","");
	camerafile.Replace(".vtk",".cam");
	camerafile.Replace(".skel",".cam");
	camerafile.Replace(".cskel",".cam");
	std::ifstream s( camerafile.ToStdString() );
	bool succesful = false;
	*g_Log << "Trying to open " << camerafile.c_str() << "... ";
	if(s.is_open())
	{
		*g_Log << "success.\n";
		g_Mediator.setRoamingCamera( shared_ptr<TRoamingCamera>( TRoamingCamera::readFromStream(&s) ) );
		s.close();
		succesful = true;
	}
	else
	{
		*g_Log << "success.\n";
		// If not found with same name try camera.cam
		string camerafile = wxFileName(Filename.c_str()).GetPath(wxPATH_GET_VOLUME | wxPATH_GET_SEPARATOR	).c_str();
		camerafile += "camera.cam";
		std::ifstream s( camerafile.c_str() );
		*g_Log << "failed.\n";
		*g_Log << "Trying to open " << camerafile.c_str() << "... ";
		if(s.is_open())
		{
			g_Mediator.setRoamingCamera( shared_ptr<TRoamingCamera>(TRoamingCamera::readFromStream(&s)) );
			s.close();
			*g_Log << "Opened " << camerafile.c_str() << "\n";
			succesful = true;
		}
		else
		{
			*g_Log << "success.\n";
			// If not found try camera.cam in upper dir
			string camerafile = wxFileName(Filename.c_str()).GetPath(wxPATH_GET_VOLUME | wxPATH_GET_SEPARATOR	).c_str();
			camerafile += "../camera.cam";
			std::ifstream s( camerafile.c_str() );
			*g_Log << "failed.\n";
			*g_Log << "Trying to open " << camerafile.c_str() << "... ";
			if(s.is_open())
			{
				g_Mediator.setRoamingCamera( shared_ptr<TRoamingCamera>( TRoamingCamera::readFromStream(&s) ) );
				s.close();
				*g_Log << "Opened " << camerafile.c_str() << "\n";
				succesful = true;
			}
		}
	}
	return succesful;
}

bool TSkeletonizer::tryOpenMesh()
{
	// Open model
	*g_Log << "Trying to open original mesh for display... "; g_Mediator.yield();
	bool modelsucces = false;
	try
	{
		vector<wxString> PossibleFilenames;

		wxString ObjectFilePath = wxFileName(m_InputFilename.c_str()).GetPath(wxPATH_GET_VOLUME);
		wxString ObjectFileName = wxFileName(m_InputFilename.c_str()).GetFullName(); 
		ObjectFileName.Replace("128","");
		ObjectFileName.Replace("256","");
		ObjectFileName.Replace("384","");
		ObjectFileName.Replace("512","");
		ObjectFileName.Replace(".vtk.gz", "");
		ObjectFileName.Replace(".vtk", "");
		ObjectFileName.Replace(".skel.gz", "");
		ObjectFileName.Replace(".skel", "");
		ObjectFileName.Replace(".cskel.gz", "");
		ObjectFileName.Replace(".cskel", "");
		PossibleFilenames.push_back( ObjectFilePath + "/normals/" + ObjectFileName + ".obj" );
		PossibleFilenames.push_back( ObjectFilePath + "/" + ObjectFileName + ".obj" );
		PossibleFilenames.push_back( ObjectFilePath + "/" + ObjectFileName + ".obj.gz" );
		PossibleFilenames.push_back( ObjectFilePath + "/normals/" + ObjectFileName + ".ply" );
		PossibleFilenames.push_back( ObjectFilePath + "/" + ObjectFileName + ".ply" );
		PossibleFilenames.push_back( ObjectFilePath + "/" + ObjectFileName + ".ply.gz" );

		for(unsigned int x=0; x<PossibleFilenames.size(); x++)
		{
			wxString s = PossibleFilenames[x];
			if( wxFile::Exists(s) )
			{
				if(!g_Settings.m_ReleaseMode) *g_Log << "Opening model " << s.c_str() << "\n";
				m_Model.reset( new TModel( 
					s.ToStdString()
					, Omega->m_ForegroundIndexField->getMaxX() 
					, (g_Parameters.m_ComputeBackgroundSskel ? g_Parameters.m_InvertBorder+1 : 1)
					) );
				modelsucces = true;
				break;
			}
			else
			{
				if(!g_Settings.m_ReleaseMode) *g_Log << "Cannot open " << s << "\n";
			}
		}
	}
	catch(string & s)
	{
		*g_Log << "\nException caught: " + s + ", ";
	}
	if(g_Settings.m_ReleaseMode)
	{
		if(!modelsucces) *g_Log << " failed.\n";
		else *g_Log << " success.\n";
	}
	return modelsucces;
}


void TSkeletonizer::writeToFile(const string & p_File)
{
//	if(!m_BackgroundSskelIndexField) throw string("Should have background");
	bool DoCompress = false;
	string Filename = p_File;
	if( Filename.substr(Filename.length() - 3,3) == ".gz"  )
	{
		Filename = Filename.substr(0, Filename.length() - 3);
	}
	ofstream s(Filename.c_str(), ios::binary);

	// Write omega
	unsigned int stringsize = m_InputFilename.size();
	s.write( (char*) &stringsize, sizeof(unsigned int) );
	s.write( m_InputFilename.c_str(), m_InputFilename.size()*sizeof(string::value_type) );
	this->Omega->writeToStream(&s);

	// Foreground sskel-index field 
	if(this->m_ForegroundSskelIndexField)
	{
		unsigned int id = ID_FG_SSKEL_INDEX;
		s.write( (char*) &id, sizeof(unsigned int) );
		this->m_ForegroundSskelIndexField->writeToStream(&s);
	}
	
	// Foreground sskel spset field 
	if(this->m_ForegroundSpSetField)
	{
		unsigned int  id = ID_FG_SSKEL_SPSET;
		s.write( (char*) &id, sizeof(unsigned int) );
		this->m_ForegroundSpSetField->writeToStream(&s);
	}
	
	// Background sskel-index field 
	if(this->m_BackgroundSskelIndexField)
	{
		unsigned int  id = ID_BG_SSKEL_INDEX;
		s.write( (char*) &id, sizeof(unsigned int) );
		this->m_BackgroundSskelIndexField->writeToStream(&s);
	}

	// Foreground spset field 
	if(this->m_BackgroundSpSetField)
	{
		unsigned int  id = ID_BG_SSKEL_SPSET;
		s.write( (char*) &id, sizeof(unsigned int) );
		this->m_BackgroundSpSetField->writeToStream(&s);
	}

	// C-skel index field 
	if(this->m_CskelIndexField)
	{
		unsigned int  id = ID_CSKEL_INDEX;
		s.write( (char*) &id, sizeof(unsigned int) );
		this->m_CskelIndexField->writeToStream(&s);
	}

	if(this->m_LoopField)
	{
		unsigned int  id = ID_CSKEL_LOOPFIELD;
		s.write( (char*) &id, sizeof(unsigned int) );
		this->m_LoopField->writeToStream(&s);
	}

	if(this->m_CurveSkeletonField)
	{
		unsigned int  id = ID_CSKEL_COLLAPSE;
		s.write( (char*) &id, sizeof(unsigned int) );
		this->m_CurveSkeletonField->writeToStream(&s);
	}
	

	s.close();

	// Compress
	{
		compressAndWait(Filename);
	}
			
}

TSkeletonizer * TSkeletonizer::readFromFile(const string & p_File)
{
	bool wasCompressed = false;
	string Filename = p_File;
	// Decompress using gzip.exe if necessary
	{
		//wxString wxFilename = p_File.c_str();
		if( Filename.substr(Filename.length() - 3,3) == ".gz"  )
		{
			decompressAndWait(Filename);

			Filename = Filename.substr(0, Filename.length() - 3);
			if( !wxFile::Exists(Filename.c_str()) )
			{
				throw string("Couldn't decompress \"") + Filename + "\"\n";
			}
			wasCompressed = true;
		}
	}

	if(! wxFile::Exists(Filename.c_str()) )
	{
		throw string("Cannot find ") + Filename.c_str();
	}
	ifstream s(Filename.c_str(), ios::binary);
	if(!s.is_open() || s.bad()) throw string("Bad file");

	// Read filename
	unsigned int stringsize = 0;
	s.read( (char*) &stringsize, sizeof(unsigned int) );
	assert(stringsize < 256);
	string InputFilename;
	InputFilename.resize(stringsize);
	s.read( (char*) InputFilename.data(), stringsize );

	// WARNING: overwriting inputfilename
	InputFilename = Filename;

	// Read omega
	TSkeletonizer * skel = new TSkeletonizer(Filename.c_str() );

	skel->m_InputFilename = InputFilename.c_str();
	skel->Omega.reset( TOmega::readFromStream(&s) );
	skel->addOmegaLayers();
	skel->m_BoundaryIndexField = skel->Omega->m_BoundaryIndexField;
	skel->m_BackgroundBoundaryIndexField = skel->Omega->m_BackgroundBoundaryIndexField;
	skel->m_SskelIndexField = skel->m_ForegroundSskelIndexField;

	// *g_Log << "Creating spatial subdivision... "; g_Mediator.yield();
	skel->Omega->m_DeltaOmega->createSpatialSubdivision();
	// *g_Log << " done\n";


	if(!s.is_open() || s.bad()) throw string("Bad file");
	while(!s.eof())
	{
		unsigned int fieldid = 0;
		s.read( (char*) &fieldid, sizeof(unsigned int) );
		// *g_Log << "id: " << fieldid << ", ";
		if(fieldid == ID_FG_SSKEL_INDEX)
		{
			shared_ptr<TSubIndexMapper> IndexField( TSubIndexMapper::readFromStream(&s, skel->Omega->m_ForegroundIndexField.get() ) );
			skel->m_ForegroundSskelIndexField = IndexField;
			skel->m_SskelIndexField = skel->m_ForegroundSskelIndexField;
//			shared_ptr<TLayer> IndexLayer( new TLayer(IndexField, "S-skel indexfield fg") );
//			g_Mediator.addLayer( IndexLayer );		
		}
		else if(fieldid == ID_FG_SSKEL_SPSET)
		{
			if(!skel->m_ForegroundSskelIndexField) throw string("!skel->m_ForegroundSskelIndexField");

			shared_ptr<TShortestPathSetField> spsf( TShortestPathSetField::readFromStream(&s, skel->m_ForegroundSskelIndexField) );
			skel->m_ForegroundSpSetField = spsf;
			skel->m_SpSetField = skel->m_ForegroundSpSetField;
			shared_ptr<TLayer> Layer( new TLayer(spsf, "spset fg") );
//			g_Mediator.addLayer( Layer );		
//			skel->addShortestPathSetAdaptorField(spsf.get(), "spset fg");
//			skel->m_ForegroundSurfaceSkeletonField = static_cast<TTypedFieldInterface<float>*>( g_Mediator.getCurrentLayerSet()->getField("spset fg maximum length") );
//			if(skel->m_ForegroundSurfaceSkeletonField) skel->m_ForegroundSurfaceSkeletonField->getLayer()->m_Filter->m_LowerValue = 12.0f;
		}
		else if(fieldid == ID_BG_SSKEL_INDEX)
		{
			if(!skel->Omega->m_BackgroundIndexField) throw string("!skel->Omega->m_BackgroundIndexField");

			shared_ptr<TSubIndexMapper> IndexField( TSubIndexMapper::readFromStream(&s, skel->Omega->m_BackgroundIndexField.get() ) );
			skel->m_BackgroundSskelIndexField = IndexField;
//			shared_ptr<TLayer> IndexLayer( new TLayer(IndexField, "Background S-skel indexfield") );
//			g_Mediator.addLayer( IndexLayer );	
		}
		else if(fieldid == ID_BG_SSKEL_SPSET)
		{
			if(!skel->m_BackgroundSskelIndexField) throw string("!skel->m_BackgroundSskelIndexField");

			shared_ptr<TShortestPathSetField> spsf( TShortestPathSetField::readFromStream(&s, skel->m_BackgroundSskelIndexField) );
			skel->m_BackgroundSpSetField = spsf;
//			shared_ptr<TLayer> Layer( new TLayer(spsf, "spset bg") );
//			g_Mediator.addLayer( Layer );		
//			skel->addShortestPathSetAdaptorField(spsf.get(), "spset bg");
//			skel->m_BackgroundSurfaceSkeletonField = static_cast<TTypedFieldInterface<float>*>( g_Mediator.getCurrentLayerSet()->getField("spset bg length") );
//			skel->m_BackgroundSurfaceSkeletonField->getLayer()->m_Filter->m_LowerValue = 12.0f;
		}
		else if(fieldid == ID_CSKEL_INDEX)
		{
			shared_ptr<TSubIndexMapper> IndexField( TSubIndexMapper::readFromStream(&s, skel->Omega->m_ForegroundIndexField.get() ) );
			skel->m_CskelIndexField = IndexField;
//			shared_ptr<TLayer> IndexLayer( new TLayer(IndexField, "C-skel indexfield") );
//			g_Mediator.addLayer( IndexLayer );		
		}
		else if(fieldid == ID_CSKEL_LOOPFIELD)
		{
			TSparseUcharField3 * f = new TSparseUcharField3(skel->m_CskelIndexField);
			TSparseTypedField<unsigned char>::readFromStream(f, &s);
			skel->m_LoopField.reset( f );
//			shared_ptr<TLayer> Layer( new TLayer(skel->m_LoopField, "C-skel detector", "C-skel detector") );
//			g_Mediator.addLayer( Layer );		
		}
		else if(fieldid == ID_CSKEL_COLLAPSE)
		{
			TSparseFloatField3 * f = new TSparseFloatField3(skel->m_CskelIndexField);
			TSparseTypedField<float>::readFromStream(f, &s);
			skel->m_CurveSkeletonField.reset( f );
//			shared_ptr<TLayer> Layer( new TLayer(skel->m_CurveSkeletonField, "C-skel measure", "C-skel measure") );
//			g_Mediator.addLayer( Layer );		
		}
		else if(!g_Settings.m_ReleaseMode) *g_Log << "Warning: unknown field id.";
	}

	skel->addLayers();

	if(s.bad()) *g_Log << "Warning: s.bad()\n";
	s.close();

	if(g_Settings.m_TryOpenMesh)
	{
		skel->tryOpenMesh();
	}

	if(wasCompressed)
	{
		compressAndWait(Filename.c_str());
	}


	return skel;
}




