#include "stdafx.h"

#include "sskelsegmentation.h"

#include "globals.h"
#include "skeletonizer.h"
#include "deltaomega.h"
#include "layer.h"
#include "fieldadaptors.h"
#include "Geometry.h"
#include "logwriter.h"

using namespace std;





void TAction_SkeletonizeForSskelSegmentation::perform_main()
{
	if( m_Filename != "" )
	{
		bool UseCSkel = false;
		g_Parameters.m_ComputeCskel = UseCSkel;
		g_Parameters.m_ComputeImportanceForCskel = UseCSkel;
		g_Parameters.m_ComputeBackgroundSskel = true;
		g_Parameters.m_InvertBorder = 5;
		g_Parameters.m_UseBoundingBoxForBgFt = false;

		g_Parameters.m_OnlyCskelIterative = false;
		g_Parameters.m_OnlyKeepShortestPathsOnCskel = false;
		g_Parameters.m_Equalize = true;

		g_Parameters.m_KeepShortestPaths = true;
		g_Parameters.m_CacheGeodesics = true;

		TSkeletonizer * skel = new TSkeletonizer( m_Filename );
		skel->perform();

		if(skel->m_SurfaceSkeletonField) skel->m_SurfaceSkeletonField->getLayer()->m_Filter->m_LowerValue = 10.0f;

		g_Mediator.updateUI();
	}
}

bool TAction_SkeletonizeForSskelSegmentation::isAvailable()
{
	return true;
}


void TAction_SkeletonizeForSskelSegmentation::perform_init()
{
	m_Filename = "";
	wxFileDialog FileDialog(0, "Choose field file to open", "models", "", "*.vtk;*.vtk.gz", wxFD_OPEN);
	if( FileDialog.ShowModal() == wxID_OK)
	{
		m_Filename = FileDialog.GetPath().c_str();
	}	
}


// ----------------------------------------------------------------------




TSskelSegmenter::TSskelSegmenter()
	: m_RemoveDilatedYCurve(true)
{
}



void TNodeGraph::computeSegmentStats()
{
	// Determine edges within this segment with large dissimilarities 
	unsigned int x;
	
	// Determine amount of segments
	m_Segments.clear();
	{
		unsigned int maxsegid = 0;
		for(x=0; x<m_Nodes.size(); x++)
		{
			unsigned int SegId = m_Nodes[x].m_SegmentId;
			if(SegId > maxsegid) maxsegid = SegId;
		}
		m_Segments.resize(maxsegid+1);
		for(x=0; x<m_Segments.size(); x++) m_Segments[x].m_Index = x;
	}

	for(x=0; x<m_Nodes.size(); x++)
	{
		unsigned int SegId = m_Nodes[x].m_SegmentId;
		m_Segments[SegId].m_Nodes.push_back( &m_Nodes[x] );
		if(m_Nodes[x].m_ShortestPath)
		{
			float length = m_Nodes[x].m_ShortestPath->m_Length;
			if(length < m_Segments[SegId].m_MinimumImportance)  m_Segments[SegId].m_MinimumImportance = length;
			if(length > m_Segments[SegId].m_MaximumImportance)  m_Segments[SegId].m_MaximumImportance = length;
		}
	}


	// And neighbors
	for(x=0; x<m_Edges.size(); x++)
	{
		unsigned int FromId = m_Nodes[m_Edges[x]->m_From].m_SegmentId;
		unsigned int ToId = m_Nodes[m_Edges[x]->m_To].m_SegmentId;
		m_Segments[FromId].m_Neighbors.insert(ToId);
		m_Segments[ToId].m_Neighbors.insert(FromId);
	}

	computeSpecificSegmentStats();
}

void TYNetwork::computeSpecificSegmentStats()
{
	for(unsigned int x=1; x<m_Segments.size(); x++)
	{
		TNodeComponent & comp = m_Segments[x];
		float maximp = 0;
		for(unsigned int y=0; y<comp.m_Nodes.size(); y++)
		{
			TNode * N = comp.m_Nodes[y];
			if(N->m_ShortestPaths.size() == 0) continue;

			unsigned int minidx = 0;
			for(unsigned int z=0; z<N->m_ShortestPaths.size(); z++)
			{
				if(N->m_ShortestPaths[z]->m_Length < N->m_ShortestPaths[minidx]->m_Length) minidx = z;
			}
			if(N->m_ShortestPaths[minidx]->m_Length > maximp) maximp = N->m_ShortestPaths[minidx]->m_Length;
		}
		comp.m_Importance = maximp;
	}
}


void TNodeGraph::printSegmentStats()
{
	for(unsigned int x=0; x<m_Segments.size(); x++)
	{
		TSskelSegment & ss = m_Segments[x]; 
		if(ss.m_Nodes.size() > 0)
		{
			*g_Log << "Segment id: " << ss.m_Index << ", count: " << (unsigned int)ss.m_Nodes.size() << ", importance: " << ss.m_Importance << "\n";
			g_Mediator.yield();
		}
	}
}



void TNodeGraph::initializeGraph(TTypedFieldInterface<TShortestPathSet*> * SPSF, const string p_Name)
{
	unsigned int x,y;

	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();

	m_Nodes.clear();
	m_Edges.clear();
	m_Segments.clear();
	
	// Make nodes
	{
		unsigned int CurrentNode = 0;
		for(x=0; x<SPSF->getIndexField()->getMaxIndex(); x++)
		{
			const TCoord3 p = SPSF->getIndexField()->vidx2coord(x);
			const TShortestPathSet * sps = SPSF->vvaluex(x);
			if(sps == 0) continue;
			for(y=0; y<sps->m_Paths.size(); y++)
			{
				shared_ptr<TShortestPath> sp = sps->m_Paths[y];
				
				m_Nodes.push_back( TNode(CurrentNode,p) );
				m_Nodes.back().m_ShortestPath = sp;
//				sp->m_UserData = CurrentNode;

				CurrentNode++;
			}
		}
	}
	m_Visited.resize( m_Nodes.size() * 8 );
	m_vc = 0;

	// Make nodefield
	m_NodeField = static_cast<TTypedFieldInterface<vector<TNode*>*>*>( g_Mediator.getCurrentLayerSet()->newField(TType::TYPE_NODE, SPSF->getIndexField(), p_Name ) );
	if(!m_NodeField) throw string("!NodeField");
	m_NodeField->clear();
	for(x=0; x<m_Nodes.size(); x++)
	{
		TNode * N = &m_Nodes[x];
		const TCoord3 p = N->m_Coord;
		if(!m_NodeField->vvaluep(p)) m_NodeField->wvaluep(p) = new vector<TNode*>();
		m_NodeField->vvaluep(p)->push_back(N);
	}

	// Make edges
	if(true)
	{
		float WEIGHTS[4];
		WEIGHTS[0] = 0.0001f; WEIGHTS[1] = 0.9016f; WEIGHTS[2] = 1.289f; WEIGHTS[3] = 1.615f;
		TCoord3 np;
		for(x=0; x<m_NodeField->getMaxIndex(); x++)
		{
			const TCoord3 p = m_NodeField->getIndexField()->vidx2coord(x);
			if(m_NodeField->vvaluep(p) == 0) continue;

			vector<TNode*> & nodesp = *m_NodeField->vvaluep(p);
//			TShortestPath* sp = m_Nodes[x].m_ShortestPath.get();

			for(np.x=p.x-1; np.x<=p.x+1; np.x++) // neighbors
			for(np.y=p.y-1; np.y<=p.y+1; np.y++)
			for(np.z=p.z-1; np.z<=p.z+1; np.z++)
			{
				if(m_NodeField->vvaluep(np) == 0) continue;

				vector<TNode*> & nodesnp = *m_NodeField->vvaluep(np);
				for(y=0; y<nodesp.size(); y++)
				for(unsigned int z=0; z<nodesnp.size(); z++)
				{
					TNode * n1 = nodesp[y];
					TNode * n2 = nodesnp[z];
					if(n1 == n2) continue;

					bool Neighbor6 = (p.x == np.x) + (p.y == np.y) + (p.z == np.z) >= 2;
					if(!Neighbor6) continue;

					shared_ptr<TEdge> Edge( new TEdge(n1->m_Index, n2->m_Index, Neighbor6) );
					char weight = 3;
					if( np.x == p.x ) weight--;
					if( np.y == p.y ) weight--;
					if( np.z == p.z ) weight--;

					Edge->m_Distance = WEIGHTS[weight];
					n1->m_Edges.push_back( Edge.get() );
					m_Edges.push_back(Edge);
					n1->m_Neighbors.push_back( n2->m_Index );
				}
			}
		}
	}


	// Compute similarities
	{
		for(x=0; x<m_Nodes.size(); x++)
		{
			TNode * Node = &m_Nodes[x];
			for(y=0; y<Node->m_Edges.size(); y++)
			{
				TEdge * Edge = Node->m_Edges[y];
				const TNode * FromNode = &m_Nodes[ Edge->m_From ];
				const TNode * ToNode = &m_Nodes[ Edge->m_To ];

				const TShortestPath * sp = FromNode->m_ShortestPath.get();
				const TShortestPath * nsp = ToNode->m_ShortestPath.get();

				// Compute geo fp
				if(false)
				{
					const TCoord3 pa = skel->m_BoundaryIndexField->vidx2coord( sp->m_Begin );
					const TCoord3 pb = skel->m_BoundaryIndexField->vidx2coord( sp->m_End );
					const TCoord3 qa = skel->m_BoundaryIndexField->vidx2coord( nsp->m_Begin );
					const TCoord3 qb = skel->m_BoundaryIndexField->vidx2coord( nsp->m_End );
					const float paqa = pa.distance(qa);
					const float pbqb = pb.distance(qb);
					const float paqb = pa.distance(qb);
					const float pbqa = pb.distance(qa);

					shared_ptr<TShortestPath> p1 = paqa+pbqb < paqb+pbqa ? 
					skel->Omega->m_DeltaOmega->getShortestPath(sp->m_Begin, nsp->m_Begin)
					: skel->Omega->m_DeltaOmega->getShortestPath(sp->m_Begin, nsp->m_End)
					;

					shared_ptr<TShortestPath> p2 = paqa+pbqb < paqb+pbqa ? 
					skel->Omega->m_DeltaOmega->getShortestPath(sp->m_End, nsp->m_End)
					: skel->Omega->m_DeltaOmega->getShortestPath(sp->m_End, nsp->m_Begin)
					;

					Edge->m_SimilarityGeoFp = max(p1->m_Length, p2->m_Length);
				}
				
				// Compute tp discont.
				{
					const float signeddot = sp->m_LocalSheet->getNormal().dot(nsp->m_LocalSheet->getNormal());
					Edge->m_SimilarityTp = fabs(signeddot);
				}
			}
		}
	}

	m_NodeField->getLayer()->m_FilterMeasure.reset( new TSskelNodeFieldMeasure_Count(m_NodeField) );
	m_NodeField->getLayer()->onFieldChanged();
}


void TNodeGraph::initializeIntCurveGraph(TTypedFieldInterface<TShortestPathSet*> * SPSF, string p_Name)
{
	// Make node for each voxel for each three feature voxels
	unsigned int x,y;

	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	if(SPSF == 0)
	{
		SPSF = static_cast<TTypedFieldInterface<TShortestPathSet*> *>( g_Mediator.getCurrentLayerSet()->getField( TAction_PruneSpSetByGeoFpDistanceForSskelSegmentation::getStaticName() ) );
	}

	m_Nodes.clear();
	m_Edges.clear();
	m_Segments.clear();

	// Make nodes
	{
		m_Nodes.reserve(256);
		for(x=0; x<SPSF->getIndexField()->getMaxIndex(); x++)
		{
			const TCoord3 p = SPSF->getIndexField()->vidx2coord(x);
			const TShortestPathSet * sps = SPSF->vvaluex(x);
			if(sps == 0) continue;

			// Do this for each three paths in sps and nsps
			for(unsigned int e1=0; e1<sps->m_Eft->size(); e1++)
			for(unsigned int e2=e1+1; e2<sps->m_Eft->size(); e2++)
			for(unsigned int e3=e2+1; e3<sps->m_Eft->size(); e3++)
			{
				set<unsigned int> eftset;
				shared_ptr<TIndexedOrigins_Vector> eft( new TIndexedOrigins_Vector() );
				eft->push_back( (*sps->m_Eft)[e1] ); eftset.insert( (*sps->m_Eft)[e1] );
				eft->push_back( (*sps->m_Eft)[e2] ); eftset.insert( (*sps->m_Eft)[e2] );
				eft->push_back( (*sps->m_Eft)[e3] ); eftset.insert( (*sps->m_Eft)[e3] );

				const unsigned int nodeidx = m_Nodes.size();
				m_Nodes.push_back( TNode(nodeidx, p) );
				m_Nodes.back().m_Eft = eft;

				// Assign corresponding shortest paths				
				for(unsigned int y=0; y<sps->m_Paths.size(); y++)
				{
					shared_ptr<TShortestPath> sp = sps->m_Paths[y];
					if( eftset.find(sp->m_Begin) != eftset.end() 
						&& eftset.find(sp->m_End) != eftset.end() )
					{
						m_Nodes.back().m_ShortestPaths.push_back( sp );
					}
				}
			}
		}
	}
	m_Visited.resize( m_Nodes.size() * 8 );
	m_vc = 0;

	
	// Make nodefield
	m_NodeField = static_cast<TTypedFieldInterface<vector<TNode*>*>*>( g_Mediator.getCurrentLayerSet()->newField(TType::TYPE_NODE, SPSF->getIndexField(), p_Name) );
	if(!m_NodeField) throw string("!NodeField");
	m_NodeField->clear();
	
	for(x=0; x<m_Nodes.size(); x++)
	{
		TNode * N = &m_Nodes[x];
		const TCoord3 p = N->m_Coord;
		if(!m_NodeField->vvaluep(p)) m_NodeField->wvaluep(p) = new vector<TNode*>();
		m_NodeField->vvaluep(p)->push_back(N);
	}



	// Make edges
	if(true)
	{
		float WEIGHTS[4];
		WEIGHTS[0] = 0.0001f; WEIGHTS[1] = 0.9016f; WEIGHTS[2] = 1.289f; WEIGHTS[3] = 1.615f;
		TCoord3 np;
		for(x=0; x<m_NodeField->getMaxIndex(); x++)
		{
			const TCoord3 p = m_NodeField->getIndexField()->vidx2coord(x);
			if(m_NodeField->vvaluep(p) == 0) continue;

			vector<TNode*> & nodesp = *m_NodeField->vvaluep(p);
//			TShortestPath* sp = m_Nodes[x].m_ShortestPath.get();

			for(np.x=p.x-1; np.x<=p.x+1; np.x++) // neighbors
			for(np.y=p.y-1; np.y<=p.y+1; np.y++)
			for(np.z=p.z-1; np.z<=p.z+1; np.z++)
			{
				if(m_NodeField->vvaluep(np) == 0) continue;

				vector<TNode*> & nodesnp = *m_NodeField->vvaluep(np);
				for(y=0; y<nodesp.size(); y++)
				for(unsigned int z=0; z<nodesnp.size(); z++)
				{
					TNode * n1 = nodesp[y];
					TNode * n2 = nodesnp[z];
					if(n1 == n2) continue;

					bool Neighbor6 = (p.x == np.x) + (p.y == np.y) + (p.z == np.z) >= 2;
					if(!Neighbor6) continue;

					shared_ptr<TEdge> Edge( new TEdge(n1->m_Index, n2->m_Index, Neighbor6) );
					char weight = 3;
					if( np.x == p.x ) weight--;
					if( np.y == p.y ) weight--;
					if( np.z == p.z ) weight--;

					Edge->m_Distance = WEIGHTS[weight];
					n1->m_Edges.push_back( Edge.get() );
					m_Edges.push_back(Edge);
					n1->m_Neighbors.push_back( n2->m_Index );
				}
			}
		}
	}

	m_NodeField->getLayer()->m_FilterMeasure.reset( new TSskelNodeFieldMeasure_Count(m_NodeField) );
	m_NodeField->getLayer()->onFieldChanged();
}


void TNodeGraph::getSeeds(vector<TNode*> *p_Output, float p_Threshold)
{
	unsigned int x;
	for(x=0; x<m_Nodes.size(); x++)
	{
		TNode * N = &m_Nodes[x];
		bool found = false;
		for(unsigned int y=0; y<N->m_ShortestPaths.size(); y++)
		{
			if(N->m_ShortestPaths[y]->m_Length < p_Threshold) 
			{
				found = true;
			}
		}
		if(!found && N->m_Eft->size() >= 3)
		{
			p_Output->push_back(N);
		}
	}
}

void TNodeGraph::getSeedsUsing2Criterion(vector<TNode*> *p_Output, float p_Threshold)
{
	unsigned int x;
	for(x=0; x<m_Nodes.size(); x++)
	{
		TNode * N = &m_Nodes[x];
		unsigned int count = 0;
		for(unsigned int y=0; y<N->m_ShortestPaths.size(); y++)
		{
			if(N->m_ShortestPaths[y]->m_Length > p_Threshold) 
			{
				count++;
			}
		}
		if(count >= 2 && N->m_Eft->size() >= 3)
		{
			p_Output->push_back(N);
		}
	}
}

void TNodeGraph::convertToThisGraph(const vector<TNode*> *p_Input, vector<TNode*> *p_Output)
{
	vector<TNode*>::const_iterator it;
	p_Output->reserve(p_Input->size());
	for(it = p_Input->begin(); it != p_Input->end(); it++)
	{
		vector<TNode*> * nodes = m_NodeField->vvaluep( (*it)->m_Coord);
		if(!nodes) continue;
		for(unsigned int x=0; x<nodes->size(); x++)
		{
			p_Output->push_back((*nodes)[x]);
		}
	}
}

TTypedFieldInterface<unsigned int>* TNodeGraph::outputNodes(const vector<TNode*>* p_Input, string p_Name)
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	vector<TNode*>::const_iterator it;
	TTypedFieldInterface<unsigned int>* Output = static_cast<TTypedFieldInterface<unsigned int>*>( g_Mediator.getCurrentLayerSet()->newField( TType::TYPE_UINT, m_NodeField->getIndexField(), p_Name ) );
	Output->clear();
	for(it = p_Input->begin(); it != p_Input->end(); it++)
	{
		Output->wvaluep( (*it)->m_Coord) = 1;
	}
	Output->getLayer()->onFieldChanged();
	return Output;
}

void TNodeGraph::extend(const vector<TNode*>* p_Input, vector<TNode*> *p_Output, float p_Threshold)
{	
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();

	// Extend pruned to non-pruned 
	typedef queue<TNode*> TMap;
	TMap S;

	m_vc=0;
	vector<TNode*>::const_iterator it;
	for(it = p_Input->begin(); it != p_Input->end(); it++)
	{
		S.push( *it );
	}

	unsigned int x;
	while(!S.empty())
	{
		int Ssize = S.size();
		int vcsize = m_Visited.size();

		TNode * N = S.front();
		S.pop();

		m_Visited[m_vc] = N->m_Index; m_vc++;
		if(N->m_IsKnown) continue;
		N->m_IsKnown = true;

		p_Output->push_back( N );
			
		for(x=0; x<N->m_Neighbors.size(); x++)
		{
			TNode * NN = &m_Nodes[ N->m_Neighbors[x] ];
			TEdge * NE = N->m_Edges[x];

			unsigned int count = 0;
			for(unsigned int y=0; y<NN->m_ShortestPaths.size(); y++) 
				if(NN->m_ShortestPaths[y]->m_Length > p_Threshold) 
					count++;
			
			if(	
				count >= 2
				&& !NN->m_IsKnown 
				&& TAction_SskelSegmenterUsingIntCurvesConnectedComponents::sameIntCurvesNodes(N,NN,p_Threshold)
				)
			{
				S.push( NN );
			}
		}
	}

	reset();

}

void TNodeGraph::dilate(const vector<TNode*> * p_Input, vector<TNode*> * p_Output, float p_Distance)
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	
	unsigned int x;
	typedef multimap<float, TNode*> TMap;
	TMap S;

	m_vc = 0;
	vector<TNode*>::const_iterator it;
	for(it=p_Input->begin(); it != p_Input->end(); it++) 
	{
		TNode * N = *it;
		N->m_Distance = 0.0f;
		//N->m_SegmentId = 1;
		S.insert( TMap::value_type(N->m_Distance, N) );
	}

	p_Output->reserve( p_Input->size() );
	TNode * N  = 0;
	while(!S.empty())
	{
		N = S.begin()->second;
		S.erase(S.begin() );

		//if(N->m_IsKnown) continue;
		N->m_IsKnown = true;
		m_Visited[m_vc] = N->m_Index; m_vc++;
		if(N->m_Distance > p_Distance) continue;
		p_Output->push_back(N);

		for(x=0; x<N->m_Neighbors.size(); x++)
		{
			TNode * NN = &m_Nodes[ N->m_Neighbors[x] ];
			TEdge * NE = N->m_Edges[x];

			const float newd = N->m_Distance + NE->m_Distance;
			if(	
				newd < NN->m_Distance
				)
			{
				NN->m_Distance = newd;
				//NN->m_SegmentId = 1;
				//m_Visited[m_vc] = NN->m_Index; m_vc++;

				// Do distance-based dilation
				S.insert( TMap::value_type( newd, NN ) );
			}
		}
	}

	reset();
}

void TNodeGraph::dilate2x2(const vector<TNode*> * p_Input, vector<TNode*> * p_Output)
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	
	typedef multimap<float, TNode*> TMap;
	TMap S;

	TCoord3 np;
	m_vc = 0;
	vector<TNode*>::const_iterator it;
	for(it=p_Input->begin(); it != p_Input->end(); it++) 
	{
		TNode * N = *it;
		const TCoord3 p = N->m_Coord;
		for(np.x=p.x-1; np.x<=p.x+1; np.x++) // neighbors
		for(np.y=p.y-1; np.y<=p.y+1; np.y++)
		for(np.z=p.z-1; np.z<=p.z+1; np.z++)
		{
//			if(np.x > p.i || np.y > p.j || np.z > p.k) continue;  // parameter
			if( (p.x == np.x) + (p.y == np.y) + (p.z == np.z) != 2 ) continue;

			vector<TNode*> * nodes = m_NodeField->vvaluep(np);
			if(nodes == 0) continue;
			for(unsigned int y=0; y<nodes->size(); y++)
			{
				p_Output->push_back( (*nodes)[y] );
			}
		}
	}
}


void TNodeGraph::floodfill(const TTypedFieldInterface<unsigned int> * p_Input)
{
	unsigned int x,y;

	// Give segmentid 1 to all nodes in input
	for(x=0; x<p_Input->getMaxIndex(); x++)
	{
		if(p_Input->vvaluex(x) == 0) continue;
		const TCoord3 p = p_Input->getIndexField()->vidx2coord(x);

		vector<TNode*> * nodes = m_NodeField->vvaluep(p);
		if(nodes == 0) continue;
		for(y=0; y<nodes->size(); y++)
		{
			TNode * N =(*nodes)[y];
			N->m_SegmentId = 1;
		}
	}

	// Now do a floodfill of the unfilled nodes
	unsigned int m_MaxSegmentId = 2;
	
	for(x=0; x<m_Nodes.size(); x++)
	{	
		TNode * N = &m_Nodes[x];
		if(
			N->m_SegmentId == 0 // not yet filled
			)
		{
			queue<unsigned int> S;
			S.push(N->m_Index);

			while(!S.empty())
			{
				TNode * N = &m_Nodes[S.front()];
				S.pop();
				N->m_SegmentId = m_MaxSegmentId;

				for(y=0; y<N->m_Edges.size(); y++)
				{
					TNode * NN = &m_Nodes[N->m_Neighbors[y]];
					TEdge * NE = N->m_Edges[y];
					if(
						(NN->m_SegmentId == 0 || NN->m_SegmentId == 1)
						&& NE->m_6Neighbor // Only 6 neighbors

						)
					{
						if(NN->m_SegmentId == 0) S.push(NN->m_Index);
						NN->m_SegmentId = m_MaxSegmentId;
					}
				}
			}
			m_MaxSegmentId++;
		}
	}

}


bool TSskelGraph::areNeighbors(TEdge * p_Edge, TYNetwork * p_Network)
{
	// See if there is a neighboring Y voxel in p_Network
	// If so, use minimum length as maximum geo fp distance
	float minlength = (numeric_limits<float>::max)();
	const TNode * n1 = &m_Nodes[p_Edge->m_From];
	const TNode * n2 = &m_Nodes[p_Edge->m_To];
	
	TCoord3 np;
	{
		const TCoord3 p = n1->m_Coord;
		
		for(np.x=p.x-1; np.x<=p.x+1; np.x++) // neighbors
		for(np.y=p.y-1; np.y<=p.y+1; np.y++)
		for(np.z=p.z-1; np.z<=p.z+1; np.z++)
		{
			vector<TNode*> * pnodes = p_Network->m_NodeField->vvaluep(np);
			if(pnodes != 0) 
			{
				for(unsigned int x=0; x<pnodes->size(); x++)
				{
					TNode * N = (*pnodes)[x];
					if(N->m_SegmentId == 0) continue;
					for(unsigned int y=0; y<N->m_ShortestPaths.size(); y++)
					{
						if( N->m_ShortestPaths[y]->m_Length < minlength )
							minlength = N->m_ShortestPaths[y]->m_Length;
					}
				}
			}
		}
	}


	{
		const TCoord3 p = n2->m_Coord;
		
		for(np.x=p.x-1; np.x<=p.x+1; np.x++) // neighbors
		for(np.y=p.y-1; np.y<=p.y+1; np.y++)
		for(np.z=p.z-1; np.z<=p.z+1; np.z++)
		{
			vector<TNode*> * pnodes = p_Network->m_NodeField->vvaluep(np);
			if(pnodes != 0) 
			{
				for(unsigned int x=0; x<pnodes->size(); x++)
				{
					TNode * N = (*pnodes)[x];
					if(N->m_SegmentId == 0) continue;
					for(unsigned int y=0; y<N->m_ShortestPaths.size(); y++)
					{
						if( N->m_ShortestPaths[y]->m_Length < minlength )
							minlength = N->m_ShortestPaths[y]->m_Length;
					}

				}
			}
		}
	}

//	return min( p_Edge->m_SimilarityGeoFp < minlength, m_ThresholdHigh );
	return p_Edge->m_SimilarityGeoFp < minlength * 0.8f; 
}	


void TSskelGraph::floodfillUsingAreNeighbors(TYNetwork * p_Network)
{
	unsigned int x,y;
	unsigned int SegId = 1;
	
	// Determine segments, first try
	*g_Log << "Propagation-based segment determination\n";
	queue<unsigned int> S;
	{
		for(x=0; x<m_Nodes.size(); x++)
		{	
			TNode * N = &m_Nodes[x];
			if(
				N->m_SegmentId == 0 // not yet filled
				//&& canGoThere(N)
				)
			{
				S.push(N->m_Index);

				while(!S.empty())
				{
					TNode * N = &m_Nodes[S.front()];
					S.pop();
					N->m_SegmentId = SegId;

					for(y=0; y<N->m_Edges.size(); y++)
					{
						TNode * NN = &m_Nodes[N->m_Neighbors[y]];
						TEdge * NE = N->m_Edges[y];
						if(
							NN->m_SegmentId == 0 
							&& areNeighbors(NE, p_Network) 
						//	&& canGoThere(NN)
							)
						{
							NN->m_SegmentId = SegId;
							S.push(NN->m_Index);
						}
					}
				}
				SegId++;
			}
		}
	}
}

void TSskelGraph::unifyLocalSheetNormals()
{
	*g_Log << "Unify normals\n";

	computeSegmentStats();

	m_vc = 0;
	unsigned int x,z;

	// Set isknown to false (to be sure)
	for(z=0; z<m_Nodes.size(); z++)
	{
		m_Nodes[z].m_IsKnown = false;
	}

	// Now unify per segment
	for(z=1; z<m_Segments.size(); z++)
	{
		TSskelSegment & ss = m_Segments[z];
		if(ss.m_Nodes.size() == 0) continue;
	
		queue<TNode*> Q;
		Q.push(ss.m_Nodes[0]);
		while(!Q.empty())
		{
			TNode * N = Q.front();
			Q.pop();

			if(N->m_IsKnown) continue;
			N->m_IsKnown = true;

			m_Visited[m_vc] = N->m_Index; 
			m_vc++;

			for(x=0; x<N->m_Neighbors.size(); x++)
			{
				TNode * NN = &m_Nodes[ N->m_Neighbors[x] ];
				TEdge * NE = N->m_Edges[x];

				if(NN->m_SegmentId != z) continue;

				if(!NN->m_IsKnown)
				{
					const float dot = N->m_ShortestPath->m_LocalSheet->getNormal().dot( NN->m_ShortestPath->m_LocalSheet->getNormal() );
					if( dot < 0.0f )
					{
						NN->m_ShortestPath->m_LocalSheet->reverse();
					}
					Q.push(NN);
				}
			}
		}
	}	

	reset();
}


void TNodeGraph::removeSegment(unsigned int p_SegmentId)
{
	typedef multimap<float, TNode*> TMap;
	unsigned int x,y;
	TMap S;
	for(y=0; y<m_Nodes.size(); y++)
	{
		TNode * N = &m_Nodes[y];
		if(N->m_SegmentId == p_SegmentId ) 
		{
			for(x=0; x<N->m_Neighbors.size(); x++)
			{
				TNode * NN = &m_Nodes[ N->m_Neighbors[x] ];
				TEdge * NE = N->m_Edges[x];
				if(!NE->m_6Neighbor) continue;
				
				if( NN->m_SegmentId != p_SegmentId ) 
				{	
					S.insert( TMap::value_type(0.0f, N) );
					N->m_Distance = 0.0f;
					break;
				}
			}
		}
	}

	while(!S.empty())
	{
		TNode * N = S.begin()->second;
		S.erase(S.begin() );

		if(N->m_IsKnown) continue;
		N->m_IsKnown = true;
		m_Visited[m_vc] = N->m_Index; 
		m_vc++;

		// Determine which segment N gets	
		{
			TNode * maxdotN = 0;
			float maxdot = 0;
			for(x=0; x<N->m_Neighbors.size(); x++)
			{
				TNode * NN = &m_Nodes[ N->m_Neighbors[x] ];
				TEdge * NE = N->m_Edges[x];
				if(!NE->m_6Neighbor) continue;
				
				if(
					NE->m_SimilarityTp > maxdot
					&& NN->m_SegmentId != p_SegmentId
					) 
				{
					maxdot = NE->m_SimilarityTp;
					maxdotN = NN;
				}
			}
			//if(!maxdotN) throw string("!maxdotN");
			if(maxdotN)
			{
				N->m_SegmentId = maxdotN->m_SegmentId;
			}
		}

		for(x=0; x<N->m_Neighbors.size(); x++)
		{
			TNode * NN = &m_Nodes[ N->m_Neighbors[x] ];
			TEdge * NE = N->m_Edges[x];
			if(!NE->m_6Neighbor) continue;
			
			const float newd = N->m_Distance + NE->m_Distance;
			if(	
				!NN->m_IsKnown 
				&& newd < NN->m_Distance
				&& NN->m_SegmentId == p_SegmentId
				)
			{
				NN->m_Distance = newd;

				S.insert( TMap::value_type( newd, NN ) );
			}
		}
	}

	// If there are any nodes with segment id left, remove
	for(y=0; y<m_Nodes.size(); y++)
	{
		TNode * N = &m_Nodes[y];
		if(N->m_SegmentId == p_SegmentId ) 
		{
			N->m_SegmentId = 0;
		}
	}
}

void TNodeGraph::reset()
{
	// Reset
	for(unsigned int x=0; x<m_vc; x++)
	{
		m_Nodes[m_Visited[x]].m_IsKnown = false;
		m_Nodes[m_Visited[x]].m_Previous = 0;
		m_Nodes[m_Visited[x]].m_Distance = (numeric_limits<float>::max)();
	}
	m_vc = 0;
}

void TNodeGraph::resetSegmentIds()
{
	for(unsigned int x=0; x<m_Nodes.size(); x++)
	{
		m_Nodes[x].m_SegmentId = 0;
	}
}


void TNodeGraph::checkForResetError()
{
	// Reset
	unsigned int x;
	for(x=0; x<m_Nodes.size(); x++)
	{
		if( m_Nodes[x].m_IsKnown != false ) 
		{
			*g_Log << "Error in m_IsKnown " << m_Nodes[x].m_IsKnown << " \n";
			break;
		}
		if( m_Nodes[x].m_Previous != 0 )
		{
			*g_Log << "Error in m_Previous " << ((int)m_Nodes[x].m_Previous) << " \n";
			break;
		}

		if(m_Nodes[x].m_Distance != (numeric_limits<float>::max)())
		{
			*g_Log << "Error in " << m_Nodes[x].m_Distance << " \n";
			break;
		}

	}
	if(x != m_Nodes.size())
	{
		*g_Log << "Error: not reset node idx " << x << "\n";
	}
	m_vc = 0;
}

void TNodeGraph::outputSegmentation(const string & p_Name)
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();

	TTypedFieldInterface<unsigned int>* Output = static_cast<TTypedFieldInterface<unsigned int>*>( g_Mediator.getCurrentLayerSet()->newField( TType::TYPE_UINT, m_NodeField->getIndexField(), p_Name ) );
	Output->clear();

	for(unsigned int x=0; x<m_NodeField->getMaxIndex(); x++)
	{
		const TCoord3 p = m_NodeField->getIndexField()->vidx2coord(x);
		vector<TNode*> * NodeSet = m_NodeField->vvaluex(x);
		if(NodeSet == 0) continue;
		
		TNode * N = 0;
		for(unsigned int y=0; y<NodeSet->size(); y++)
		{
			if( (*NodeSet)[y]->m_SegmentId != 0 )
			{
				if( N == 0 )  N = (*NodeSet)[y];
				else
				{
					// How to map multiple nodes to a single voxel?
					// 1. choose node with maximum rho

					TNode * Ny = (*NodeSet)[y];
					if( Ny->m_ShortestPath && Ny->m_ShortestPath->m_Length > N->m_ShortestPath->m_Length )
					{
						N = (*NodeSet)[y];
					}
				}
			}
		}
		
		if(N != 0)
			Output->wvaluep(p) = N->m_SegmentId;

	}
	
	Output->getLayer()->onFieldChanged();
}


/*
void TNodeGraph::renumber()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();

	map<unsigned int, unsigned int> Renumber;
	for(unsigned int x=0; x<m_NodeField->getMaxIndex(); x++)
	{
		const TCoord3 p = m_NodeField->getIndexField()->vidx2coord(x);
		vector<TNode*> * NodeSet = m_NodeField->vvaluex(x);
		if(NodeSet == 0) continue;
		
		TNode * N = 0;
		for(unsigned int y=0; y<NodeSet->size(); y++)
		{
			unsigned int & id = (*NodeSet)[y]->m_SegmentId;
			if( id != 0 )
			{
				if(Renumber.find(id) == Renumber.end()) Renumber.insert( pair<unsigned int,unsigned int>(id,Renumber.size()+1) );
				id = Renumber[id];
			}
		}
	}
}
*/


void TNodeGraph::outputSegmentationMinColoring(const string & p_Name)
{
	if(m_Segments.size()==0) throw string("m_Segments.size()==0");
	unsigned int x;

	set<unsigned int>::iterator it;
	set<unsigned int> NonFreeColors;
	for(x=0; x<m_Segments.size(); x++)
	{
		NonFreeColors.clear();
		for(it = m_Segments[x].m_Neighbors.begin(); it != m_Segments[x].m_Neighbors.end(); it++)
		{
			NonFreeColors.insert( m_Segments[*it].m_MinColor );
		}

		unsigned int lowestfree = 1;
		while( NonFreeColors.find( lowestfree ) != NonFreeColors.end() )
		{
			lowestfree++;
		}

		m_Segments[x].m_MinColor = lowestfree;
	}

	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	TTypedFieldInterface<unsigned int>* Output = static_cast<TTypedFieldInterface<unsigned int>*>( g_Mediator.getCurrentLayerSet()->newField( TType::TYPE_UINT, m_NodeField->getIndexField(), p_Name ) );
	Output->clear();
	for(x=0; x<m_NodeField->getMaxIndex(); x++)
	{
		const TCoord3 p = m_NodeField->getIndexField()->vidx2coord(x);
		vector<TNode*> * NodeSet = m_NodeField->vvaluex(x);
		if(NodeSet == 0 || NodeSet->size() == 0) continue;
	
		TNode * N = 0;
		for(unsigned int y=0; y<NodeSet->size(); y++)
		{
			if( (*NodeSet)[y]->m_SegmentId != 0 )
			{
				if( N == 0 )  N = (*NodeSet)[y];
				else
				{
					// How to map multiple nodes to a single voxel?
					// 1. choose node with maximum rho

					if( (*NodeSet)[y]->m_ShortestPath && (*NodeSet)[y]->m_ShortestPath->m_Length > N->m_ShortestPath->m_Length )
					{
						N = (*NodeSet)[y];
					}
				}
			}
		}
		
		if(N != 0)
			Output->wvaluep(p) = m_Segments[ N->m_SegmentId ].m_MinColor;
	}
	
	Output->getLayer()->onFieldChanged();
	Output->getLayer()->m_ColorMap = Output->getLayer()->m_AllColorMaps[5];
}


void TNodeGraph::outputEverySegment()
{
	*g_Log << "output every segment\n";
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();

	if(m_Segments.size() == 0) throw string("m_Segments.size() == 0");

	for(unsigned int y=1; y<m_Segments.size(); y++)
	{
		TNodeComponent & comp = m_Segments[y];
		 
		TTypedFieldInterface<unsigned int>* Output = static_cast<TTypedFieldInterface<unsigned int>*>( g_Mediator.getCurrentLayerSet()->newField( TType::TYPE_UINT, m_NodeField->getIndexField(), wxString::Format("Segment %i", comp.m_Index).ToStdString()) );
		Output->clear();

		for(unsigned int x=0; x<comp.m_Nodes.size(); x++)
		{
			Output->wvaluep(comp.m_Nodes[x]->m_Coord) = 1;
		}
	
		Output->getLayer()->onFieldChanged();
	}
}


void TNodeGraph::pruneSegments(unsigned int p_Threshold, unsigned int p_ToId)
{
	*g_Log << "prune segments\n";
	unsigned int x,y;
	if(m_Segments.size()==0) throw string("m_Segments.size()==0");
	for(x=0; x<m_Segments.size(); x++)
	{
		TNodeComponent & ss = m_Segments[x];
		if(ss.m_Nodes.size() >= p_Threshold)
		{
	//		*g_Log << "Segment id: " << ss.m_Index << ", count: " << ss.m_Nodes.size();
	//		*g_Log << "\n";
		}
		else
		{
			for(y=0; y<ss.m_Nodes.size(); y++) 
			{
				ss.m_Nodes[y]->m_SegmentId = p_ToId;
			}
		}
	}

	computeSegmentStats();
}


void TYNetwork::computeYCurveComponents(float p_ThresholdLow, float p_ThresholdHigh, vector<TNode*> * p_Output, string p_Name)
{
	*g_Log << "Compute Y components, using threshold low " << p_ThresholdLow << ", threshold high: " << p_ThresholdHigh << "\n";
	*g_Log << "#nodes: " << (unsigned int)m_Nodes.size() << "\n";
	*g_Log << "[";
	unsigned int x,y;

	typedef queue<TNode*> TMap;
	TMap S;

	unsigned int SegId = 1;
	m_vc = 0;
	for(y=0; y<m_Nodes.size(); y++)
	{
		TNode * N = &m_Nodes[y];

		if(N->m_SegmentId == 0)
		{	
			//*g_Log << "."; g_Mediator.Yield();
			S.push(N);

			while(!S.empty())
			{
				TNode * N = S.front();
				S.pop();
				if(N->m_IsKnown) continue;
				N->m_IsKnown = true;
				N->m_SegmentId = SegId;
				m_Visited[m_vc++] = N->m_Index;


				bool nlow = false;
				for(unsigned int z=0; z<N->m_ShortestPaths.size(); z++)
				{
					if(N->m_ShortestPaths[z]->m_Length < p_ThresholdHigh ) nlow = true;
				}


				for(x=0; x<N->m_Neighbors.size(); x++)
				{
					TNode * NN = &m_Nodes[ N->m_Neighbors[x] ];
					TEdge * NE = N->m_Edges[x];


					bool nnlow = false;
					for(unsigned int z=0; z<NN->m_ShortestPaths.size(); z++)
					{
						if(NN->m_ShortestPaths[z]->m_Length < p_ThresholdHigh ) nnlow = true;
					}

					const float Threshold = nlow || nnlow ? p_ThresholdLow : p_ThresholdHigh;
	
					if(	(!NN->m_IsKnown)
						&& 
						TAction_SskelSegmenterUsingIntCurvesConnectedComponents::sameIntCurvesNodes(N,NN, Threshold) 
						)
					{
						S.push(	NN );
					}
				}
			}
			SegId++;
		}
	}
	reset();
	*g_Log << "]"; 
	*g_Log << "#components: " << SegId << "\n";
	*g_Log << "\n";

	computeSegmentStats();
	outputSegmentation(p_Name);

	// Remove Y-curves that have a lower importance than threshold
	for(y=0; y<m_Segments.size(); y++)
	{	
		TNodeComponent & comp = m_Segments[y];
		if(comp.m_Importance < p_ThresholdHigh)
		{
			for(x=0; x<comp.m_Nodes.size(); x++)
			{
				comp.m_Nodes[x]->m_SegmentId = 0;
			}
		}
	}

	// Output
	for(y=0; y<m_Nodes.size(); y++)
	{
		TNode * N = &m_Nodes[y];
		if(N->m_SegmentId != 0) 
			p_Output->push_back(N);
	}

	computeSegmentStats();
}








// ----------------------------------------------------------------------

bool TAction_ComputePathAndDebug::isAvailable()
{
	return g_Mediator.getSskelSegmenter().get() != 0;
}

void TAction_ComputePathAndDebug::perform_main()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	TSskelSegmenter * segm = g_Mediator.getSskelSegmenter().get();
	TVoxelSetSelection * selection = g_Mediator.getCurrentLayer()->m_Selection.get();

	vector<TCoord3> SelVoxels;
	{
		selection->initTraversal();
		TCoord3 c;
		while(selection->nextTraversal(c))
		{
			SelVoxels.push_back(c);
		}
	}
	if(SelVoxels.size() != 2) throw string("Please select two voxels");

	static long UseAreNeighbors;
	{
		wxString Choices[] = { "0. Yes", "1. No" };
		wxString Result = ::wxGetSingleChoice("UseAreNeighbors", "UseAreNeighbors", 2, Choices );
		if(Result == "") return;
		Result.Left(1).ToLong(&UseAreNeighbors);
	}

	TTypedFieldInterface<vector<TNode*>*>* NodeField= segm->m_SskelGraph.m_NodeField;
	if(!NodeField) throw string("!NodeField");
	if(!NodeField->vvaluep(SelVoxels[0])) throw string("!NodeField->vvaluep(SelVoxels[0])");
	if(!NodeField->vvaluep(SelVoxels[1])) throw string("!NodeField->vvaluep(SelVoxels[1])");
	if(NodeField->vvaluep(SelVoxels[0])->size() > 1) *g_Log << "Warning: voxels contains multiple nodes, picking first\n";
	if(NodeField->vvaluep(SelVoxels[1])->size() > 1) *g_Log << "Warning: voxels contains multiple nodes, picking first\n";

	TNode * Node1 = NodeField->vvaluep(SelVoxels[0])->at(0);
	TNode * Node2 = NodeField->vvaluep(SelVoxels[1])->at(0);
	vector<TNode*> Path;
	float length = segm->m_SskelGraph.computePath(Node1, Node2, UseAreNeighbors == 0, &Path, &segm->m_IntCurveGraph);
	*g_Log << "Path length: " << length << ", nodes: " << (unsigned int)Path.size() << "\n"; 
	unsigned int x,y;
	{
		TTypedFieldInterface<unsigned int>* Debug= static_cast<TTypedFieldInterface<unsigned int>*>(g_Mediator.getCurrentLayerSet()->newField(TType::TYPE_UINT, segm->m_SskelGraph.m_NodeField->getIndexField(), "Debug: path") );
		Debug->clear();
		
		for(x=0; x<Path.size(); x++)
		{
			Debug->wvaluep( Path[x]->m_Coord ) = x+1; 
		}

		if(Path.size() > 0)
		{
			for(x=0; x<Path.size()-1; x++)
			{
				TNode * cur = Path[x];
				TNode * nxt = Path[x+1];
				for(y=0; y<cur->m_Neighbors.size(); y++)
				{
					if( &segm->m_SskelGraph.m_Nodes[cur->m_Neighbors[y]] == nxt)
					{
						*g_Log << "Edge from " << x << " to " << (x+1) << "\tsim geo fp: " << cur->m_Edges[y]->m_SimilarityGeoFp << ", sim tp: " << cur->m_Edges[y]->m_SimilarityTp << "\n";
					}
				}
			}
		}

		Debug->getLayer()->onFieldChanged();
	}
}





// -------------------------------------------------------------------

bool TAction_PruneSpSetByGeoFpDistanceForSskelSegmentation::isAvailable()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	if(!skel) return false;
	return skel->m_SpSetField && skel->m_CurveSkeletonField; 
}

TShortestPathSet * TAction_PruneSpSetByGeoFpDistanceForSskelSegmentation::process(TShortestPathSet * sps, float p_LowerValue)
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();

	unsigned int y,a;
	// Detect equivalence sets
	typedef set< set<unsigned int> > TEqSet;
	TEqSet eqset;
	{
		for(y=0; y<sps->m_Eft->size(); y++) 
		{
			set<unsigned int> s;
			unsigned int idx = (*sps->m_Eft)[y];
			if(skel && idx >= skel->m_BoundaryIndexField->getMaxIndex()) continue;

			s.insert(idx);
			eqset.insert(s);
		}

		TEqSet::iterator it;
		TEqSet::iterator jt;
		for(y=0; y<sps->m_Paths.size(); y++) 
		{
			TShortestPath * sp = sps->m_Paths[y].get();
			if( sp->m_Length < p_LowerValue )
			{
				unsigned int idx1 = sp->m_Begin;
				unsigned int idx2 = sp->m_End;		 	
				for(it = eqset.begin(); it != eqset.end(); it++) if(it->find(idx1) != it->end()) break;
				for(jt = eqset.begin(); jt != eqset.end(); jt++) if(jt->find(idx2) != jt->end()) break;

				// Now merge jt to it
				if(it != jt)
				{
					set<unsigned int>::iterator kt;
					for(kt=jt->begin(); kt != jt->end(); kt++) 
					{
						unsigned int idx = *kt;
						set<unsigned int> & s = const_cast<set<unsigned int> & >(*it);
						s.insert(idx);
					}
		
					eqset.erase(jt);
				}
			}
		}
	}

	// Now for each pair of items in eqset a path should remain
	TShortestPathSet::TPaths NewPaths;
	{
		TEqSet::iterator it;
		TEqSet::iterator jt;

		// For each two equivalence sets 
		for(it = eqset.begin(); it != eqset.end(); it++)
		for(jt = it; jt != eqset.end(); jt++)
		{
			if(it == jt) continue;
		
			unsigned int idx1 = *it->begin(); 
			unsigned int idx2 = *jt->begin();
			for(a=0; a<sps->m_Paths.size(); a++)
			{
				shared_ptr<TShortestPath> sp = sps->m_Paths[a];
				if( 
					(sp->m_Begin == idx1 && sp->m_End == idx2)
					||
					(sp->m_Begin == idx2 && sp->m_End == idx1)
					)
				{
					NewPaths.push_back( sp );
				}
			}
			
		}
	}

	/*
	bool KeepCskel = false;
	// If no shortest paths left but C-skel voxel then insert longest path
	if( KeepCskel && NewPaths.size() == 0 )
	{
		if( skel->m_CurveSkeletonField && skel->m_CurveSkeletonField->vvaluep( sps->m_SourceVoxel ) > p_LowerValue )
		{
			shared_ptr<TShortestPath> Longest;
			{
				for(a=0; a<sps->m_Paths.size(); a++)
				{
					if( !Longest || sps->m_Paths[a]->m_Length > Longest->m_Length  ) Longest = sps->m_Paths[a];
				}
			}

			NewPaths.clear();
			NewPaths.push_back( Longest );
		}
	}
	*/
	
	{
		shared_ptr<TIndexedOrigins_Vector> NewEft( new TIndexedOrigins_Vector() );
		set<unsigned int> EftSet;
		for(a=0; a<NewPaths.size(); a++)
		{
			EftSet.insert( NewPaths[a]->m_Begin  );
			EftSet.insert( NewPaths[a]->m_End );
		}
		set<unsigned int>::iterator it;
		for(it = EftSet.begin(); it != EftSet.end(); it++)
		{
			NewEft->push_back( *it );
		}

		TShortestPathSet * newsps = new TShortestPathSet( sps->m_SourceVoxel, NewEft);
		newsps->m_PathSet.reset( new TIndexedOrigins_Vector() );
		newsps->m_Paths = NewPaths;
		newsps->updatePathSet();
		return newsps;
	}
}

void TAction_PruneSpSetByGeoFpDistanceForSskelSegmentation::perform_main()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();

	if(!m_SPSF) throw string("!m_SPSF");
	if(!m_Filter) throw string("!m_Filter");

	/*
	bool KeepCskel = true;
	if(false)
	{
		long Option;
		{
			wxString Choices[] = { "0. No", "1. Yes" };
			wxString Result = ::wxGetSingleChoice("Keep C-skel", "Keep C-skel", 2, Choices );
			if(Result == "") return;
			Result.Left(1).ToLong(&Option);
		}
		KeepCskel = Option == 1 ? true : false;
	}
	*/
	
	if(m_ShortName == "") throw string("m_ShortName=0");
	TTypedFieldInterface<TShortestPathSet*>* Output = static_cast<TTypedFieldInterface<TShortestPathSet*>*>( g_Mediator.getCurrentLayerSet()->newField( TType::TYPE_SHORTESTPATHSET, m_SPSF->getIndexField(), m_ShortName ) );
	Output->clear();
	for(unsigned int x=0; x<m_SPSF->getMaxIndex(); x++)
	{
		const TCoord3 p = m_SPSF->getIndexField()->vidx2coord(x);
		TShortestPathSet * sps = m_SPSF->vvaluex(x);
		if(sps == 0) continue;

		TShortestPathSet * newsps = process(sps, m_Filter->m_LowerValue);
		Output->wvaluep(p) = newsps;
	}		

	Output->getLayer()->m_Name = m_ShortName + wxString::Format(" %.2f",m_Filter->m_LowerValue);
	Output->getLayer()->onFieldChanged();
	Output->getLayer()->m_Filter->m_LowerValue = 2.99f;
	g_Mediator.updateUI();

}


bool TAction_Debug_SameIntCurves::isAvailable()
{
//	return g_Mediator.getCurrentLayer() && g_Mediator.getCurrentLayer()->m_Field->getType() == TType::TYPE_SHORTESTPATHSET; 
	return g_Mediator.getCurrentLayer() && g_Mediator.getCurrentLayer()->m_Field->getType() == TType::TYPE_NODE; 
}

void TAction_Debug_SameIntCurves::perform_main()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	TSskelSegmenter * segm = g_Mediator.getSskelSegmenter().get();
	TVoxelSetSelection * selection = g_Mediator.getCurrentLayer()->m_Selection.get();

	if(g_Mediator.getCurrentLayer()->m_Field->getType() != TType::TYPE_NODE) throw string("TAction_FilterShortestPathSets: not a node field");
	TTypedFieldInterface<vector<TNode*>*> * Node = static_cast<TTypedFieldInterface<vector<TNode*>*>*>( g_Mediator.getCurrentLayer()->m_Field.get() );

	vector<TCoord3> SelVoxels;
	{
		selection->initTraversal();
		TCoord3 c;
		while(selection->nextTraversal(c))
		{
			SelVoxels.push_back(c);
		}
	}
	if(SelVoxels.size() != 2) throw string("Please select two voxels");
			
	//TShortestPathSet * sps = SPSF->vvaluep(SelVoxels[0]);
	//TShortestPathSet * nsps = SPSF->vvaluep(SelVoxels[1]);
	//if(!sps || !nsps) throw string("!sps || !nsps");
	TNode * n = (*Node->vvaluep(SelVoxels[0]))[0];
	TNode * nn = (*Node->vvaluep(SelVoxels[1]))[0];


	const float Threshold = 10.0f;
	*g_Log << "Using maximum threshold: " << Threshold << "\n";
	bool result = TAction_SskelSegmenterUsingIntCurvesConnectedComponents::sameIntCurvesNodes(n, nn, Threshold, true);
	*g_Log << "Result: " << result << "\n";
}

// ----------------------------------------------------------------------

bool TAction_SskelSegmenterUsingIntCurvesConnectedComponents::sameIntCurves(const TShortestPathSet * sps, const TShortestPathSet * nsps, float p_MaximumTolerance, bool p_Debug)
{
/*
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	unsigned int x,y,z;

	if(sps->m_Eft->size() != 3) throw string("sps->m_Eft.size() != 3");
	if(nsps->m_Eft->size() != 3) throw string("nsps->m_Eft.size() != 3");

	float mat[3][3];
	for(z=0; z<sps->m_Eft->size(); z++)
	for(y=0; y<nsps->m_Eft->size(); y++)
	{
		const TCoord3 a = skel->m_BoundaryIndexField->vidx2coord((*sps->m_Eft)[z]) ;
		const TCoord3 b = skel->m_BoundaryIndexField->vidx2coord((*nsps->m_Eft)[y]) ;

		//mat[y][z] = a.distance(b);
		mat[y][z] = skel->Omega->m_DeltaOmega->getShortestPath((*sps->m_Eft)[z], (*nsps->m_Eft)[y])->m_Length;
	}

	// Find minimum path length in sps 
	float mind = (numeric_limits<float>::max)();
	{
		for(x=0; x<sps->m_Paths.size(); x++)
		{
			if( sps->m_Paths[x]->m_Length < mind ) mind = sps->m_Paths[x]->m_Length;
		}
		for(x=0; x<nsps->m_Paths.size(); x++)
		{
			if( nsps->m_Paths[x]->m_Length < mind ) mind = nsps->m_Paths[x]->m_Length;
		}
	}
	const float Threshold = min( mind, p_MaximumTolerance );  

	if(p_Debug)
	{
		*g_Log << mat[0][0] << ", " << mat[1][1] << ", " << mat[2][2] << "\n";
		*g_Log << mat[0][0] << ", " << mat[1][2] << ", " << mat[2][1] << "\n";
		*g_Log << mat[0][1] << ", " << mat[1][2] << ", " << mat[2][0] << "\n";
		*g_Log << mat[0][1] << ", " << mat[1][0] << ", " << mat[2][2] << "\n";
		*g_Log << mat[0][2] << ", " << mat[1][0] << ", " << mat[2][1] << "\n";
		*g_Log << mat[0][2] << ", " << mat[1][1] << ", " << mat[2][0] << "\n";
		*g_Log << "Threshold: " << Threshold << "\n";
	}

	
	return  
		   (mat[0][0] < Threshold && mat[1][1] < Threshold && mat[2][2] < Threshold) 
		|| (mat[0][0] < Threshold && mat[1][2] < Threshold && mat[2][1] < Threshold) 

		|| (mat[0][1] < Threshold && mat[1][2] < Threshold && mat[2][0] < Threshold) 
		|| (mat[0][1] < Threshold && mat[1][0] < Threshold && mat[2][2] < Threshold) 

		|| (mat[0][2] < Threshold && mat[1][0] < Threshold && mat[2][1] < Threshold) 
		|| (mat[0][2] < Threshold && mat[1][1] < Threshold && mat[2][0] < Threshold) 
	;
*/
	return false;
}

bool TAction_SskelSegmenterUsingIntCurvesConnectedComponents::sameIntCurvesNodes(const TNode * n1, const TNode * n2, float p_MaximumTolerance, bool p_Debug)
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	unsigned int y,z;

	if(n1->m_Eft->size() != 3) throw string("n1->m_Eft.size() != 3");
	if(n2->m_Eft->size() != 3) throw string("n2->m_Eft.size() != 3");

	float mat[3][3];
	for(z=0; z<n1->m_Eft->size(); z++)
	for(y=0; y<n2->m_Eft->size(); y++)
	{
		if((*n1->m_Eft)[z] >= skel->m_BoundaryIndexField->getMaxIndex()) continue;
		if((*n2->m_Eft)[y] >= skel->m_BoundaryIndexField->getMaxIndex()) continue;
		const TCoord3 a = skel->m_BoundaryIndexField->vidx2coord((*n1->m_Eft)[z]) ;
		const TCoord3 b = skel->m_BoundaryIndexField->vidx2coord((*n2->m_Eft)[y]) ;

		//mat[y][z] = a.distance(b);
		mat[y][z] = skel->Omega->m_DeltaOmega->getShortestPath((*n1->m_Eft)[z], (*n2->m_Eft)[y])->m_Length;
	}

	/*
	// Find minimum path length in sps 
	float mind = (numeric_limits<float>::max)();
	{
		for(x=0; x<n1->m_ShortestPaths.size(); x++)
		{
			if( n1->m_ShortestPaths[x]->m_Length < mind ) mind = n1->m_ShortestPaths[x]->m_Length;
		}
		for(x=0; x<n2->m_ShortestPaths.size(); x++)
		{
			if( n2->m_ShortestPaths[x]->m_Length < mind ) mind = n2->m_ShortestPaths[x]->m_Length;
		}
	}
	const float Threshold = min( mind, p_MaximumTolerance );  // todo, 20 = threshold
	*/
	const float Threshold = p_MaximumTolerance;

	if(p_Debug)
	{
		*g_Log << mat[0][0] << ", " << mat[1][1] << ", " << mat[2][2] << "\n";
		*g_Log << mat[0][0] << ", " << mat[1][2] << ", " << mat[2][1] << "\n";
		*g_Log << mat[0][1] << ", " << mat[1][2] << ", " << mat[2][0] << "\n";
		*g_Log << mat[0][1] << ", " << mat[1][0] << ", " << mat[2][2] << "\n";
		*g_Log << mat[0][2] << ", " << mat[1][0] << ", " << mat[2][1] << "\n";
		*g_Log << mat[0][2] << ", " << mat[1][1] << ", " << mat[2][0] << "\n";
		*g_Log << "Threshold: " << Threshold << "\n";
	}

	
	return  
		   (mat[0][0] < Threshold && mat[1][1] < Threshold && mat[2][2] < Threshold) 
		|| (mat[0][0] < Threshold && mat[1][2] < Threshold && mat[2][1] < Threshold) 

		|| (mat[0][1] < Threshold && mat[1][2] < Threshold && mat[2][0] < Threshold) 
		|| (mat[0][1] < Threshold && mat[1][0] < Threshold && mat[2][2] < Threshold) 

		|| (mat[0][2] < Threshold && mat[1][0] < Threshold && mat[2][1] < Threshold) 
		|| (mat[0][2] < Threshold && mat[1][1] < Threshold && mat[2][0] < Threshold) 
	;
}

float TSskelGraph::computePath(TNode * p_From, TNode * p_To, bool p_UseAreNeighbors, vector<TNode*> * p_Path, TYNetwork * p_Network)
{
	typedef multimap<float, TNode*> TMap;
	TMap S;
	S.insert( TMap::value_type(0.0f, p_From) );
	p_From->m_Distance = 0.0f;
	m_Visited.resize( m_Nodes.size() * 4 );
	m_vc=0;
	m_Visited[m_vc] = p_From->m_Index; m_vc++;

	TNode * N  = 0;
	float FoundDist = (numeric_limits<float>::max)();
	unsigned int x;
	while(!S.empty())
	{
		//const float dist = S.begin()->first;
		N = S.begin()->second;
		S.erase(S.begin() );

		if(N->m_IsKnown) continue;
		N->m_IsKnown = true;
		m_Visited[m_vc] = N->m_Index; m_vc++;
		
		if(N == p_To) 
		{ 
			FoundDist = N->m_Distance;
			break; 
		}
		
		for(x=0; x<N->m_Neighbors.size(); x++)
		{
			TNode * NN = &m_Nodes[ N->m_Neighbors[x] ];
			TEdge * NE = N->m_Edges[x];
			
			const float newd = N->m_Distance + NE->m_Distance;
			if(	!NN->m_IsKnown 
				&& (!p_UseAreNeighbors || areNeighbors( NE, p_Network))
//				&& canGoThere(NN)
				&& newd < NN->m_Distance
				)
			{
				NN->m_Previous = N;
				NN->m_Distance = newd;
				m_Visited[m_vc] = NN->m_Index; m_vc++;
				S.insert( TMap::value_type( newd, NN ) );
			}
		}
	}

	if(N == p_To && p_Path != 0)
	{
		vector<TNode*> Reversed;
		Reversed.push_back( N );
		while(N != p_From)
		{
			N = N->m_Previous;
			Reversed.push_back( N );
		}
		p_Path->reserve(Reversed.size());
		for(x=Reversed.size()-1; x!=0; x--)
		{
			p_Path->push_back(Reversed[x]);
		}
	}

	reset();

	return FoundDist;
}


bool TAction_SskelSegmenterUsingIntCurvesConnectedComponents::isAvailable()
{
	return g_Mediator.getSkeletonizer().get() != 0;
}


void TAction_SskelSegmenterUsingIntCurvesConnectedComponents::renumber(TTypedFieldInterface<unsigned int> * p_Cluster)
{
	if(!p_Cluster) throw string("!p_Cluster");
	unsigned int x;
	vector<unsigned int> Renumber;

	unsigned int maxid = 0;
	for(x=0; x<p_Cluster->getMaxIndex(); x++)
	{
		if(p_Cluster->vvaluex(x) > maxid) maxid = p_Cluster->vvaluex(x);
	}
	Renumber.resize(maxid+1,0);

	unsigned int freeid = 1;
	for(x=0; x<p_Cluster->getMaxIndex(); x++)
	{
		const unsigned int id = p_Cluster->vvaluex(x);
		if( id != 0 )
		{
			if(Renumber[id] == 0)
			{
				Renumber[id] = freeid++;
			}
		}
	}

	for(x=0; x<p_Cluster->getMaxIndex(); x++)
	{
		const unsigned int id = p_Cluster->vvaluex(x);
		if( id != 0 )
		{
			p_Cluster->wvaluex(x) = Renumber[id];
		}
	}

	p_Cluster->getLayer()->onFieldChanged();
}

void TAction_SskelSegmenterUsingIntCurvesConnectedComponents::perform_main()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	if(!g_Mediator.getSskelSegmenter()) g_Mediator.getSskelSegmenter().reset( new TSskelSegmenter() );
	if(!m_FgBg) if(!g_Mediator.getBgSskelSegmenter()) g_Mediator.getBgSskelSegmenter().reset( new TSskelSegmenter() );


	const string FgBg = m_FgBg ? " (fg)" : " (bg)";
	TField * Skel = m_FgBg ? skel->m_ForegroundSurfaceSkeletonField : skel->m_BackgroundSurfaceSkeletonField;
	TTypedFieldInterface<TShortestPathSet*> * SPSF = m_FgBg ? skel->m_ForegroundSpSetField.get() : skel->m_BackgroundSpSetField.get();
	TSskelSegmenter * segm = m_FgBg ? g_Mediator.getSskelSegmenter().get() : g_Mediator.getBgSskelSegmenter().get();
	
	static double ThresholdHigh = 10.0f;
	{
		{
			wxString s = "";
			do
			{
				s = ::wxGetTextFromUser("High threshold", "High threshold", wxString::Format("%f",ThresholdHigh).c_str() );
				if(s == "") return;
			} 
			while( !s.ToDouble(&ThresholdHigh) );
		}
	}
	static double ThresholdLow = 5.0f;
	{
		{
			wxString s = "";
			do
			{
				s = ::wxGetTextFromUser("Low threshold", "Low threshold", wxString::Format("%f",ThresholdLow).c_str() );
				if(s == "") return;
			} 
			while( !s.ToDouble(&ThresholdLow) );
		}
	}

	TTypedFieldInterface<TShortestPathSet*> * SPSFPruned = 0;
	{
		Skel->getLayer()->m_Filter->m_LowerValue = ThresholdHigh;
		Skel->getLayer()->m_Filter->m_UpperValue = 500.0f; // todo
		//TAction_PruneSpSetByGeoFpDistance Prune;
		TAction_PruneSpSetByGeoFpDistanceForSskelSegmentation Prune;
		Prune.m_ShortName = "spset pruned" + FgBg;
		Prune.m_SPSF = SPSF;
		Prune.m_Filter = Skel->getLayer()->m_Filter.get();
		Prune.perform();
		SPSFPruned = static_cast<TTypedFieldInterface<TShortestPathSet*> *>( g_Mediator.getCurrentLayerSet()->getField( Prune.m_ShortName)  );
	}

	TTypedFieldInterface<TShortestPathSet*> * SPSFNonpruned = 0;
	{
		Skel->getLayer()->m_Filter->m_LowerValue = ThresholdLow;
		Skel->getLayer()->m_Filter->m_UpperValue = 500.0f; // todo
		//TAction_PruneSpSetByGeoFpDistance Prune;
		TAction_PruneSpSetByGeoFpDistanceForSskelSegmentation Prune;
		Prune.m_ShortName = "spset not pruned" + FgBg;
		Prune.m_SPSF = SPSF;
		Prune.m_Filter = Skel->getLayer()->m_Filter.get();
		Prune.perform();
		SPSFNonpruned =  static_cast<TTypedFieldInterface<TShortestPathSet*> *>(g_Mediator.getCurrentLayerSet()->getField( Prune.m_ShortName) );

		if( !g_Mediator.getCurrentLayerSet()->exists(Prune.m_ShortName + " eft") )
		{		
			TTypedFieldInterface<TIndexedOrigins*> * TypedAdaptorField = new TShortestPathSetFieldAdaptor_Eft(SPSFNonpruned);
			shared_ptr<TField> AdaptorField( TypedAdaptorField );
			shared_ptr<TLayer> AdaptorLayer( new TLayer(AdaptorField, "   " + AdaptorField->getAdaptorName() ) );
			AdaptorLayer->m_ShortName = Prune.m_ShortName + " eft";
			AdaptorLayer->m_FilterMeasure.reset( new TIndexedOriginsFieldMeasure_MinDistance(TypedAdaptorField) );
			AdaptorLayer->m_Filter.reset( new TFloatFilter(&AdaptorLayer->m_FilterMeasure) );
			AdaptorLayer->m_AllVoxels->setCheckedForRendering(false);
			AdaptorLayer->m_RenderLines = true;
			g_Mediator.addLayer(AdaptorLayer, g_Mediator.getCurrentLayerSet()->getLayer("spset not pruned") );
		}
	}


	// Test, segment Y-curve at tau=high
	if(false)
	{
		// Opmerking: node field blijft niet bewaard, hierdoor probleem bij selecteren nodefield in GUI.
		TYNetwork & Y = *new TYNetwork();
		Y.initializeIntCurveGraph(SPSFPruned, "Y-network pruned nodes" + FgBg);
		vector<TNode*> Bla;
		Y.computeYCurveComponents(ThresholdLow, ThresholdHigh, &Bla, "Y-network pruned");
		Y.pruneSegments(10, 0);
		Y.outputSegmentationMinColoring("Y-network pruned min-color" + FgBg);
		Y.resetSegmentIds();
	}

	
	*g_Log << "Initialize int. curve. graph\n";
	segm->m_IntCurveGraph.initializeIntCurveGraph(SPSFNonpruned, "Y-network non-pruned nodes" + FgBg);
	segm->m_IntCurveGraph.checkForResetError();
		


	*g_Log << "Extend\n";
	vector<TNode*> ExtendedNodes;

	segm->m_IntCurveGraph.computeYCurveComponents(ThresholdLow, ThresholdHigh, &ExtendedNodes, "Y-network non-pruned before remove" + FgBg);
	segm->m_IntCurveGraph.outputSegmentationMinColoring("Y curves min-color" + FgBg);


	*g_Log << "Initialize S-skel graph\n";
	segm->m_SskelGraph.initializeGraph(SPSFPruned, "S-skel nodes " + FgBg);

/*
	// Gebruik flood fill using areneighbors
	segm->m_SskelGraph.floodfillUsingAreNeighbors(&segm->m_IntCurveGraph);
	segm->m_SskelGraph.outputSegmentation("Flood fill using areneighbors");
	segm->m_SskelGraph.computeSegmentStats();
	segm->m_SskelGraph.outputSegmentationMinColoring("Segmentation min-color");
	return;
*/

	vector<TNode*> ExtendedNodes2;
	*g_Log << "Convert extended nodes\n";
	segm->m_SskelGraph.convertToThisGraph(&ExtendedNodes, &ExtendedNodes2);
	ExtendedNodes.swap( vector<TNode*>() );
//	segm->m_SskelGraph.checkForResetError();
//	segm->m_SskelGraph.outputNodes(&ExtendedNodes2, "Extended2");
	

	vector<TNode*> DilatedNodes;
	*g_Log << "Dilate " << (unsigned int)ExtendedNodes2.size() << " nodes \n";
	segm->m_SskelGraph.dilate(&ExtendedNodes2, &DilatedNodes, 2.0f);
//	segm->m_SskelGraph.dilate2x2(&ExtendedNodes2, &DilatedNodes);
	ExtendedNodes2.swap( vector<TNode*>() );
	TTypedFieldInterface<unsigned int>* DilatedField = segm->m_SskelGraph.outputNodes(&DilatedNodes, "Dilated " + FgBg);
//	segm->m_SskelGraph.checkForResetError();

	*g_Log << "Flood fill\n";
	segm->m_SskelGraph.floodfill(DilatedField);
//	segm->m_SskelGraph.computeSegmentStats();
//	segm->m_SskelGraph.outputSegmentation("Flood filled");
	
	segm->m_SskelGraph.computeSegmentStats();
	segm->m_SskelGraph.pruneSegments(10, 1); 
	
	if(segm->m_RemoveDilatedYCurve) segm->m_SskelGraph.removeSegment(1);
	else 
	{
		*g_Log << "Warning: removal of dilated Y-network disabled\n";
		for(unsigned int x=0; x<segm->m_SskelGraph.m_Nodes.size(); x++)
		{
			unsigned int & sid = segm->m_SskelGraph.m_Nodes[x].m_SegmentId;
			if(sid == 1) sid = 0;
		}
	}
	
	segm->m_SskelGraph.computeSegmentStats();

	{
		const string name = "S-skel segmentation" + FgBg;
		segm->m_SskelGraph.outputSegmentation(name);
		renumber( static_cast<TTypedFieldInterface<unsigned int>*>(g_Mediator.getCurrentLayerSet()->getField(name)) );
	}

	if(true) // Enable/disable Min-color
	{
		const string name = "S-skel segmentation min-color" + FgBg;
		segm->m_SskelGraph.outputSegmentationMinColoring(name);
	}

	segm->m_SskelGraph.reset();

	if(true) // Enable/disable cleanup
	{
		vector<string> Layers;
		Layers.push_back("spset pruned");
		Layers.push_back("Y-network non-pruned nodes");
		Layers.push_back("Y-network non-pruned before remove");
		Layers.push_back("Dilated");
		
		for(unsigned int x=0; x<Layers.size(); x++)
		{
			g_Mediator.getCurrentLayerSet()->deleteLayer( Layers[x] + " (fg)" );
			g_Mediator.getCurrentLayerSet()->deleteLayer( Layers[x] + " (bg)" );
		}
	}

	return;
}


void TAction_SegmentSskel::perform_main()
{
	// Fg
	{
		TAction_SskelSegmenterUsingIntCurvesConnectedComponents Action;
		Action.m_FgBg = true;
		Action.perform();
	}

	// Bg
	if( g_Mediator.getSkeletonizer()->m_BackgroundSpSetField )
	{
		TAction_SskelSegmenterUsingIntCurvesConnectedComponents Action;
		Action.m_FgBg = false;
		Action.perform();
	}
}