#include "stdafx.h"

#include "utils.h"

#include "globals.h"
#include "logwriter.h"
#include "main.h"

#include "skeletonizer.h"
#include "deltaomega.h"
#include "layer.h"
#include "colormap.h"
#include "wx/bitmap.h"
#include "wx/clipbrd.h"
#include "wx/glcanvas.h"
#include "wx/image.h"
#include "wx/string.h"
#include "wx/wfstream.h"
#include "wx/zstream.h"
#include "logwriter.h"

using namespace std;


void utilDrawAxes()
{
	glEnable(GL_COLOR_MATERIAL);
	glBegin(GL_LINES);
		glColor3f(1.0f, 0.0f, 0.0f);
		glVertex3f(10.0f, 0.0f, 0.0f);
		glVertex3f(0.0f, 0.0f, 0.0f);
	glEnd();

	glBegin(GL_LINES);
		glColor3f(0.0f, 1.0f, 0.0f);
		glVertex3f(0.0f, 10.0f, 0.0f);
		glVertex3f(0.0f, 0.0f, 0.0f);
	glEnd();

	glBegin(GL_LINES);
		glColor3f(0.0f, 0.0f, 1.0f);
		glVertex3f(0.0f, 0.0f, 10.0f);
		glVertex3f(0.0f, 0.0f, 0.0f);
	glEnd();
}


void utilDrawCube()
{
	glBegin(GL_QUADS);
    glNormal3f( 0.0F, 0.0F, 1.0F);
    glVertex3f( 0.5F, 0.5F, 0.5F); glVertex3f(-0.5F, 0.5F, 0.5F);
    glVertex3f(-0.5F,-0.5F, 0.5F); glVertex3f( 0.5F,-0.5F, 0.5F);

    glNormal3f( 0.0F, 0.0F,-1.0F);
    glVertex3f(-0.5F,-0.5F,-0.5F); glVertex3f(-0.5F, 0.5F,-0.5F);
    glVertex3f( 0.5F, 0.5F,-0.5F); glVertex3f( 0.5F,-0.5F,-0.5F);

    glNormal3f( 0.0F, 1.0F, 0.0F);
    glVertex3f( 0.5F, 0.5F, 0.5F); glVertex3f( 0.5F, 0.5F,-0.5F);
    glVertex3f(-0.5F, 0.5F,-0.5F); glVertex3f(-0.5F, 0.5F, 0.5F);

    glNormal3f( 0.0F,-1.0F, 0.0F);
    glVertex3f(-0.5F,-0.5F,-0.5F); glVertex3f( 0.5F,-0.5F,-0.5F);
    glVertex3f( 0.5F,-0.5F, 0.5F); glVertex3f(-0.5F,-0.5F, 0.5F);

    glNormal3f( 1.0F, 0.0F, 0.0F);
    glVertex3f( 0.5F, 0.5F, 0.5F); glVertex3f( 0.5F,-0.5F, 0.5F);
    glVertex3f( 0.5F,-0.5F,-0.5F); glVertex3f( 0.5F, 0.5F,-0.5F);

    glNormal3f(-1.0F, 0.0F, 0.0F);
    glVertex3f(-0.5F,-0.5F,-0.5F); glVertex3f(-0.5F,-0.5F, 0.5F);
    glVertex3f(-0.5F, 0.5F, 0.5F); glVertex3f(-0.5F, 0.5F,-0.5F);
    glEnd();
}

void utilDrawCubeSide(int p_Side)
{
	glBegin(GL_QUADS);

	if(p_Side == 5)
	{
	    glNormal3f( 0.0F, 0.0F, 1.0F);
	    glVertex3f( 0.5F, 0.5F, 0.5F); glVertex3f(-0.5F, 0.5F, 0.5F);
		glVertex3f(-0.5F,-0.5F, 0.5F); glVertex3f( 0.5F,-0.5F, 0.5F);
	}

	else if(p_Side == 4)
	{
	    glNormal3f( 0.0F, 0.0F,-1.0F);
		glVertex3f(-0.5F,-0.5F,-0.5F); glVertex3f(-0.5F, 0.5F,-0.5F);
	    glVertex3f( 0.5F, 0.5F,-0.5F); glVertex3f( 0.5F,-0.5F,-0.5F);
	}

	else if(p_Side == 3)
	{
	    glNormal3f( 0.0F, 1.0F, 0.0F);
		glVertex3f( 0.5F, 0.5F, 0.5F); glVertex3f( 0.5F, 0.5F,-0.5F);
	    glVertex3f(-0.5F, 0.5F,-0.5F); glVertex3f(-0.5F, 0.5F, 0.5F);
	}

	else if(p_Side == 2)
	{
	    glNormal3f( 0.0F,-1.0F, 0.0F);
		glVertex3f(-0.5F,-0.5F,-0.5F); glVertex3f( 0.5F,-0.5F,-0.5F);
	    glVertex3f( 0.5F,-0.5F, 0.5F); glVertex3f(-0.5F,-0.5F, 0.5F);
	}

	else if(p_Side == 1)
	{
	    glNormal3f( 1.0F, 0.0F, 0.0F);
		glVertex3f( 0.5F, 0.5F, 0.5F); glVertex3f( 0.5F,-0.5F, 0.5F);
	    glVertex3f( 0.5F,-0.5F,-0.5F); glVertex3f( 0.5F, 0.5F,-0.5F);
	}

	else if(p_Side == 0)
	{
		glNormal3f(-1.0F, 0.0F, 0.0F);
		glVertex3f(-0.5F,-0.5F,-0.5F); glVertex3f(-0.5F,-0.5F, 0.5F);
		glVertex3f(-0.5F, 0.5F, 0.5F); glVertex3f(-0.5F, 0.5F,-0.5F);
	}
	else throw string("invalid side");

	glEnd();
}

void utilDrawQuad()
{
	glBegin(GL_QUADS);
    glNormal3f( 0.0F, 0.0F,-1.0F);
    glVertex3f(-0.5F,-0.5F,0.0F); glVertex3f(-0.5F, 0.5F,0.0F);
    glVertex3f( 0.5F, 0.5F,0.0F); glVertex3f( 0.5F,-0.5F,0.0F);
	glEnd();
}

void checkGlError()
{
	const int Error = glGetError();
	if( Error != GL_NO_ERROR )
	{
		string s( (char*) gluErrorString(Error) );
		*g_Log << "OpenGL error " << Error << ": " << s.c_str() << "\n";
		// assert(false);
	}
}


string floattostring(float v)
{
	char b[16];
	sprintf(b, "%.2f", v);
	return string(b);
}
 
string uinttostring(unsigned long v)
{
	char b[16];
	sprintf(b, "%u", v);
	return string(b);
}

string niceprintuint(unsigned long v)
{
	unsigned int number = v;

	// Copied from http://bdn2.borland.com/article/17183
	const int size = 15;    // max number of digits
	  static char str[size];  // string to return
	  char *ptr1, *ptr2;      // place holders
	  char tempStr[size];     // workplace
	  int counter = 0;        // three's counter
	  ptr1 = tempStr;
	  do {
		// grab rightmost digit and add value of character zero
		*ptr1++ = (char)(number % 10) + '0';
		// strip off rightmost digit
		number /= 10;
		// if moved over three digits insert comma into string
		if (number &&  !(++counter % 3))
		  *ptr1++ = ',';
		// continue until number equal zero
	  } while(number);
	  // this loop reverses characters in a string
	  for( --ptr1, ptr2 = str; ptr1 >= tempStr; --ptr1)
		*ptr2++ = *ptr1;
	  // add the zero string terminator
	  *ptr2 = 0;
	  return string(str);
}

string niceprintfloat(float v)
{
	char b[16];
	if(v < 1000)
	{
		sprintf(b, "%.2f", v);
		return string(b);
	}
	else
	{
		size_t number = v;
		return niceprintuint( number );
	}
}


const PROCESS_MEMORY_COUNTERS & getProcessInfo()
{
    static PROCESS_MEMORY_COUNTERS pmc;

    // Print the process identifier.

	DWORD processID;
	GetWindowThreadProcessId( (HWND)MyApp::instance()->GetTopWindow()->GetHandle(), &processID);  
//	DWORD processID = GetProcessId( (HANDLE) MyApp::instance()->GetTopWindow()->GetHandle(); );
    //printf( "\nProcess ID: %u\n", processID );

    // Print information about the memory usage of the process.

    HANDLE hProcess = OpenProcess(  PROCESS_QUERY_INFORMATION |
                                    PROCESS_VM_READ,
                                    FALSE, processID );
    if (NULL == hProcess) return pmc;

    if ( GetProcessMemoryInfo( hProcess, &pmc, sizeof(pmc)) )
    {
  //      printf( "\tPageFaultCount: 0x%08X\n", pmc.PageFaultCount );
 //       printf( "\tPeakWorkingSetSize: 0x%08X\n", 
 //                 pmc.PeakWorkingSetSize );
 //       printf( "\tWorkingSetSize: 0x%08X\n", pmc.WorkingSetSize );
 //       printf( "\tQuotaPeakPagedPoolUsage: 0x%08X\n", 
 //                 pmc.QuotaPeakPagedPoolUsage );
//        printf( "\tQuotaPagedPoolUsage: 0x%08X\n", 
//                  pmc.QuotaPagedPoolUsage );
//        printf( "\tQuotaPeakNonPagedPoolUsage: 0x%08X\n", 
//                  pmc.QuotaPeakNonPagedPoolUsage );
//        printf( "\tQuotaNonPagedPoolUsage: 0x%08X\n", 
//                  pmc.QuotaNonPagedPoolUsage );
        //printf( "\tPagefileUsage: 0x%08X\n", pmc.PagefileUsage ); 
//        printf( "\tPeakPagefileUsage: 0x%08X\n", 
//                  pmc.PeakPagefileUsage );

    }

    CloseHandle( hProcess );
	return pmc;
}






#define INLINE_ROTATE(a,i,j,k,l) g=a[i][j];h=a[k][l];a[i][j]=g-s*(h+g*tau);a[k][l]=h+s*(g-h*tau);
#define INLINE_MAX_ROTATIONS 50

void GeomUtils::mult(float* v,float k)
{  v[0] *= k; v[1] *= k; v[2] *= k;  }

float GeomUtils::norm(float* v)
{  return sqrt(v[0]*v[0]+v[1]*v[1]+v[2]*v[2]);  }	
	

float GeomUtils::normalize(float* v)
{
   float l = v[0]*v[0]+v[1]*v[1]+v[2]*v[2];
   if (l<1.0e-7) { 
   		   v[0] *=1.0e4; v[1] *= 1.0e4; v[2] *= 1.0e4;
   		   l = v[0]*v[0]+v[1]*v[1]+v[2]*v[2]; 
   		   l = sqrt(l); v[0] /= l; v[1] /= l; v[2] /= l;
   		   return l / 1.0e4;
   		 }
   if (l>1.0e-8) { l = sqrt(l); v[0] /= l; v[1] /= l; v[2] /= l; }
//   else cout<<"GeomUtils::normalize: length too small: "<<l<<endl;
   return l;
}
float GeomUtils::grad_prod(int i,float* v0,float* v1,float* v2,float a)
{							//Given triangle v0v1v2 and vertex i (0,1,or 2) in it, returns grad_bf(i)*grad_bf(i+1)*area
   float l1[3],l2[3], *pi,*pj,*pk;
   
   if (a<=0)   a = area(v0,v1,v2);			//compute area of v0v1v2, if not given
   
   if (i==0) 	  { pi = v0; pj = v1; pk = v2; }	//reorder vertices
   else if (i==1) { pi = v1; pj = v2; pk = v0; }
   else /*i==2*/  { pi = v2; pj = v0; pk = v1; } 
  
   l1[0] = pi[0]-pk[0]; l1[1] = pi[1]-pk[1]; l1[2] = pi[2]-pk[2];
   l2[0] = pj[0]-pk[0]; l2[1] = pj[1]-pk[1]; l2[2] = pj[2]-pk[2];
   
   return -dot(l1,l2)/(4*a);
}


float GeomUtils::area(float* p1,float* p2,float* p3)
{
  float a = (p1[0]-p2[0])*(p1[0]-p2[0])+(p1[1]-p2[1])*(p1[1]-p2[1])+(p1[2]-p2[2])*(p1[2]-p2[2]);
  float b = (p2[0]-p3[0])*(p2[0]-p3[0])+(p2[1]-p3[1])*(p2[1]-p3[1])+(p2[2]-p3[2])*(p2[2]-p3[2]);
  float c = (p1[0]-p3[0])*(p1[0]-p3[0])+(p1[1]-p3[1])*(p1[1]-p3[1])+(p1[2]-p3[2])*(p1[2]-p3[2]);
  return (0.25* sqrt(fabs((double)4.0*a*c - (a-b+c)*(a-b+c))));
}
	                  
void GeomUtils::grad_bf(int i,float* v0,float* v1,float* v2,float a,float* n,float* g)
{
   float N[3],pl[3],ln[3], *pi,*pj,*pk;
   
   if (!n)   { normal(v0,v1,v2,ln); n=ln; }		//compute normal of v0v1v2, if not given
   if (a<=0)   a = area(v0,v1,v2);			//compute area of v0v1v2, if not given
   
   if (i==0) 	  { pi = v0; pj = v1; pk = v2; }	//reorder vertices
   else if (i==1) { pi = v1; pj = v2; pk = v0; }
   else /*i==2*/  { pi = v2; pj = v0; pk = v1; } 
   
   pl[0] = pi[0]+n[0]; pl[1] = pi[1]+n[1]; pl[2] = pi[2]+n[2]; 
 
   normalDirection(pj,pk,pl,N);				//compute N = normal to basis-func plane pjpkpl times 2*area(pjpkpl)
   float ht = 0.5/a;					//ht = correction-factor for N's projection on plane of v0v1v2
   
   float pr = N[0]*n[0] + N[1]*n[1] + N[2]*n[2];	//pr = projection of N along n

   g[0] = ht*(pr*n[0]-N[0]); g[1] = ht*(pr*n[1]-N[1]); g[2] = ht*(pr*n[2]-N[2]); 
   
   //REMARK: The whole idea of the above stuff is to produce a grad-vector whose length is
   // 1/h, where h is the height of triangle p0p1p2, corresponding to vertex 'i'. 
   //Looking at the formulas, not normalizing N, and multiplying it with 'ht', achieves exactly this.
}



float GeomUtils::dot(const float* a,const float* b)
{
   return a[0]*b[0] + a[1]*b[1] + a[2]*b[2];
}

void GeomUtils::barycenter(float* p1, float* p2, float* p3, float* center)
{
  center[0] = (p1[0]+p2[0]+p3[0]) / 3.0;
  center[1] = (p1[1]+p2[1]+p3[1]) / 3.0;
  center[2] = (p1[2]+p2[2]+p3[2]) / 3.0;
}

// Jacobi iteration for the solution of eigenvectors/eigenvalues of a nxn
// real symmetric matrix. Square nxn matrix a; size of matrix in n;
// output eigenvalues in w; and output eigenvectors in v. Resulting
// eigenvalues/vectors are sorted in decreasing order; eigenvectors are
// normalized.
template<class T> static int jacobi_n(T **a, int n, T *w, T **v)
{
  int i, j, k, iq, ip, numPos;
  T tresh, theta, tau, t, sm, s, h, g, c, tmp;
  T bspace[4], zspace[4];
  T *b = bspace, *z = zspace;

  if (n > 4) {  b = new T[n];  z = new T[n];  }	// only allocate memory if the matrix is large

  for (ip=0; ip<n; ip++) 			// initialize
  { 
    for (iq=0; iq<n; iq++) v[ip][iq] = 0.0; 
    v[ip][ip] = 1.0; b[ip] = w[ip] = a[ip][ip]; z[ip] = 0.0;
  }

  for (i=0; i<INLINE_MAX_ROTATIONS; i++) 	// begin rotation sequence
  {
    sm = 0.0;
    for (ip=0; ip<n-1; ip++) 
      for (iq=ip+1; iq<n; iq++) sm += fabs(a[ip][iq]);
    if (sm == 0.0) break;

    if (i < 3)  tresh = 0.2*sm/(n*n);           // first 3 sweeps
    else tresh = 0.0;

    for (ip=0; ip<n-1; ip++) 
    {
      for (iq=ip+1; iq<n; iq++) 
      {
        g = 100.0*fabs(a[ip][iq]);

        // after 4 sweeps
        if (i > 3 && (fabs(w[ip])+g) == fabs(w[ip]) && (fabs(w[iq])+g) == fabs(w[iq]))
            a[ip][iq] = 0.0;
        else if (fabs(a[ip][iq]) > tresh) 
        {
          h = w[iq] - w[ip];
          if ( (fabs(h)+g) == fabs(h))
	    t = (a[ip][iq]) / h;
          else 
          {
            theta = 0.5*h / (a[ip][iq]);
            t = 1.0 / (fabs(theta)+sqrt(1.0+theta*theta));
            if (theta < 0.0) t = -t;
          }
          c = 1.0 / sqrt(1+t*t);
          s = t*c;
          tau = s/(1.0+c);
          h = t*a[ip][iq];
          z[ip] -= h;
          z[iq] += h;
          w[ip] -= h;
          w[iq] += h;
          a[ip][iq]=0.0;

          // ip already shifted left by 1 unit
          for (j = 0;j <= ip-1;j++) 
          { INLINE_ROTATE(a,j,ip,j,iq) }
          // ip and iq already shifted left by 1 unit
          for (j = ip+1;j <= iq-1;j++) 
          { INLINE_ROTATE(a,ip,j,j,iq) }
          // iq already shifted left by 1 unit
          for (j=iq+1; j<n; j++) 
          { INLINE_ROTATE(a,ip,j,iq,j) }
          for (j=0; j<n; j++) 
          { INLINE_ROTATE(v,j,ip,j,iq) }
        }
      }
    }

    for (ip=0; ip<n; ip++) 
    {
      b[ip] += z[ip];
      w[ip] = b[ip];
      z[ip] = 0.0;
    }
  }

  //// this should be NEVER called
  if ( i >= INLINE_MAX_ROTATIONS )
  {
    cout<<"Jacobi: Error extracting eigenfunctions"<<endl;
    return 0;
  }

  // sort eigenfunctions                 these changes do not affect accuracy 
  for (j=0; j<n-1; j++)                  // boundary incorrect
  {
    k = j; tmp = w[k];
    for (i=j+1; i<n; i++)                // boundary incorrect, shifted already
    {
      if (w[i] >= tmp)                   // why exchage if same?
      { k = i; tmp = w[k]; }
    }
    if (k != j) 
    {
      w[k] = w[j]; w[j] = tmp;
      for (i=0; i<n; i++) 
      { tmp = v[i][j]; v[i][j] = v[i][k]; v[i][k] = tmp; }
    }
  }

  // ensure eigenvector consistency (i.e., Jacobi can compute vectors that
  // are negative of one another (.707,.707,0) and (-.707,-.707,0). This can
  // reek havoc in hyperstreamline/other stuff. We will select the most
  // positive eigenvector.
  for (j=0; j<n; j++)
  {
    for (numPos=0, i=0; i<n; i++)
      if ( v[i][j] >= 0.0 ) numPos++;
    if ( numPos < ceil(double(n)/double(2.0)) )
      for(i=0; i<n; i++) v[i][j] *= -1.0;
  }

  if (n > 4)					//release if we allocated anything
  { delete [] b; delete [] z; }
  return 1;
}




#undef INLINE_ROTATE
#undef INLINE_MAX_ROTATIONS


int JacobiN(float **a, int n, float *w, float **v)
{
  return jacobi_n(a,n,w,v);
}


//
//int GeomUtils::JacobiN(double **a, int n, double *w, double **v)
//{
//  return jacobi_n(a,n,w,v);
//}




void PCA2(const TIndexedOrigins_Vector * p_Io, float* norm, float* u1, float* u2, float* barycenter)
{
//	const TBoundary & Boundary = g_Mediator.getSkeletonizer()->m_BoundaryIndexField;
	TIndexedOrigins_Vector::const_iterator it;

//	float * pi;
	float p[3]; p[0] = 0; p[1] = 0; p[2] = 0;
	float m[6]; 							
	float *M[3],*V[3],W[3],MM0[3],MM1[3],MM2[3],VV0[3],VV1[3],VV2[3],d[3]; 

	M[0] = MM0; M[1] = MM1; M[2] = MM2; V[0] = VV0; V[1] = VV1; V[2] = VV2; 
	m[0] = m[1] = m[2] = m[3] = m[4] = m[5] = 0;

	TIndexMapper * Boundary = g_Mediator.getSkeletonizer()->m_BoundaryIndexField.get();
	
	// Calculate weighted barycenter
	if(barycenter[0] == 0.0f && barycenter[1] == 0.0f && barycenter[2] == 0.0f)
	{
		for(it = p_Io->begin(); it != p_Io->end(); it++)
		{
			p[0] += Boundary->vidx2coord( *it ).x; 
			p[1] += Boundary->vidx2coord( *it ).y;
			p[2] += Boundary->vidx2coord( *it ).z;
		}
		
		p[0] /= (float)p_Io->vertexcount();
		p[1] /= (float)p_Io->vertexcount();
		p[2] /= (float)p_Io->vertexcount();
		barycenter[0] = p[0]; 
		barycenter[1] = p[1]; 
		barycenter[2] = p[2];
	}
	else
	{
		p[0] = barycenter[0];
		p[1] = barycenter[1];
		p[2] = barycenter[2];
	}
	
	// Calculate covariance matrix
	for(it = p_Io->begin(); it != p_Io->end(); it++)
	{
		//pi  = (float*) &Boundary[*it]; 
		d[0] = (Boundary->vidx2coord(*it).x - p[0]); 
		d[1] = (Boundary->vidx2coord(*it).y -p[1]); 
		d[2] = (Boundary->vidx2coord(*it).z -p[2]);
		
		m[0] += d[0]*d[0]; 
		m[1] += d[0]*d[1]; 
		m[2] += d[0]*d[2];		//Compute m[] += pi x pi
		m[3] += d[1]*d[1]; 
		m[4] += d[1]*d[2]; 
		m[5] += d[2]*d[2];
	}

	// Eigen code:
	//const float k = 1.0e+4f;						//Bias M[][] since values are really very small (O(1.0e-6))
	const float k = 1.0e4;
	M[0][0] = k*m[0]; M[0][1] = k*m[1]; M[0][2] = k*m[2];		//Fill-in symmetric M[][] for eigen-computation
	M[1][0] = k*m[1]; M[1][1] = k*m[3]; M[1][2] = k*m[4];
	M[2][0] = k*m[2]; M[2][1] = k*m[4]; M[2][2] = k*m[5];

	JacobiN(M,3,W,V);					//Compute the eigenvalues/vectors of M[][] in W[],V[]
									//Results are returned in decreasing eigenval-order W[0]>W[1]>W[2]

	W[0] = fabs(W[0]); W[1] = fabs(W[1]); W[2] = fabs(W[2]);

	norm[0] = VV0[2]; norm[1] = VV1[2]; norm[2] = VV2[2];		//Normal = eigenvec of smallest eigenval
	u1[0]   = VV0[0]; u1[1]   = VV1[0]; u1[2]   = VV2[0];		//u1     = eigenvec of largest eigenval
	u2[0]   = VV0[1]; u2[1]   = VV1[1]; u2[2]   = VV2[1];

	GeomUtils::normalize(norm);						//Normalize the three vectors (norm,u1,n2)
//	if (onorm && GeomUtils::dot(onorm,norm)<0)  			//If the caller gave an orientation-hint for the normal,
//	{  norm[0] *= -1; norm[1] *= -1; norm[2] *= -1;  }			//use it (swap computed normal along hint's orientation)

	float vn = GeomUtils::dot(u1,norm);					//Correct u1 so it's surely perp on norm
	u1[0] -= vn*norm[0]; u1[1] -= vn*norm[1]; u1[2] -= vn*norm[2];	//and normalized
	GeomUtils::normalize(u1);
	GeomUtils::cross(u1,norm,u2);					//Define u2 as norm x u1, so we have now a basis
	
}

void SaveScreenshot(wxGLCanvas * p_Canvas, string p_Filename)
{
	const unsigned int WIDTH = p_Canvas->GetClientSize().GetX();
	const unsigned int HEIGHT = p_Canvas->GetClientSize().GetY();
	wxImage img(WIDTH, HEIGHT);

	unsigned char * imgdata = new unsigned char[WIDTH*HEIGHT*4];
	glReadBuffer(GL_BACK);
	glReadPixels(0, 0, WIDTH, HEIGHT, GL_RGBA, GL_UNSIGNED_BYTE, imgdata ); 
	unsigned int TotalRead = WIDTH * HEIGHT;

	int minx = (std::numeric_limits<int>::max)(); 
	int miny = (std::numeric_limits<int>::max)(); 
	int maxx = 0;
	int maxy = 0;

	for(unsigned int a=0; a<TotalRead; a++ )
	{
		unsigned char r = imgdata[(a*4)+0]; 
		unsigned char g = imgdata[(a*4)+1]; 
		unsigned char b = imgdata[(a*4)+2]; 

		int x = a % WIDTH;
		int y = HEIGHT-1- (a / WIDTH);

		if( !(r == 255 && g == 255 && b == 255) )
		{
			if(x < minx) minx = x;
			if(y < miny) miny = y;
			if(x > maxx) maxx = x;
			if(y > maxy) maxy = y;
		}

		img.SetRGB(x, y, r,g,b);
	}

	delete imgdata;

	const int BORDER = 5;
	minx -= BORDER;
	miny -= BORDER; 
	maxx += BORDER;
	maxy += BORDER; 
	if(minx < 0) minx = 0;
	if(miny < 0) miny = 0;
	if(maxx > WIDTH-1) maxx = WIDTH-1;
	if(maxy > HEIGHT-1) maxy = HEIGHT-1;

	img.GetSubImage( wxRect(minx,miny,maxx-minx,maxy-miny) ).SaveFile(p_Filename.c_str(), wxBITMAP_TYPE_PNG);
}

void CopyScreenshotToClipboard(wxGLCanvas * p_Canvas)
{
	const unsigned int WIDTH = p_Canvas->GetClientSize().GetX();
	const unsigned int HEIGHT = p_Canvas->GetClientSize().GetY();
	wxImage img(WIDTH, HEIGHT);

	unsigned char * imgdata = new unsigned char[WIDTH*HEIGHT*4];
	glReadBuffer(GL_BACK);
	glReadPixels(0, 0, WIDTH, HEIGHT, GL_RGBA, GL_UNSIGNED_BYTE, imgdata ); 
	unsigned int TotalRead = WIDTH * HEIGHT;

	int minx = (std::numeric_limits<int>::max)(); 
	int miny = (std::numeric_limits<int>::max)(); 
	int maxx = 0;
	int maxy = 0;

	for(unsigned int a=0; a<TotalRead; a++ )
	{
		unsigned char r = imgdata[(a*4)+0]; 
		unsigned char g = imgdata[(a*4)+1]; 
		unsigned char b = imgdata[(a*4)+2]; 

		int x = a % WIDTH;
		int y = HEIGHT-1- (a / WIDTH);

		if( !(r == 255 && g == 255 && b == 255) )
		{
			if(x < minx) minx = x;
			if(y < miny) miny = y;
			if(x > maxx) maxx = x;
			if(y > maxy) maxy = y;
		}

		img.SetRGB(x, y, r,g,b);
	}

	delete imgdata;

	const int BORDER = 5;
	minx -= BORDER;
	miny -= BORDER; 
	maxx += BORDER;
	maxy += BORDER; 
	if(minx < 0) minx = 0;
	if(miny < 0) miny = 0;
	if(maxx > WIDTH-1) maxx = WIDTH-1;
	if(maxy > HEIGHT-1) maxy = HEIGHT-1;

	wxBitmap bitmap( img.GetSubImage( wxRect(minx,miny,maxx-minx,maxy-miny) ) );
	
	if (wxTheClipboard->Open())
	{
		wxTheClipboard->SetData( new wxBitmapDataObject(bitmap) );
		wxTheClipboard->Close();
	}

	
}




void createColoring(const TTypedFieldInterface<unsigned int> * p_Input, int p_ComponentCount, string p_Name)
{
	const bool Output = false;
	TSkeletonizer * sc = g_Mediator.getSkeletonizer().get();

	// Determine neighbors of components
	vector< set< unsigned int > > Neighbors;
	Neighbors.resize( p_ComponentCount );
	sc->Omega->m_DeltaOmega->determineNeighborhoods( p_Input, Neighbors );
	set<unsigned int>::iterator it;
	unsigned int x;
	for(x=0; x<p_ComponentCount; x++)
	{
		if(Output) *g_Log << "Neighbors of comp " << x << ": ";
		for(it = Neighbors[x].begin(); it != Neighbors[x].end(); it++)
		{
			if(Output) *g_Log << *it << ", ";
		}
		if(Output) *g_Log << "\n";
	}


	vector<unsigned int> Colors;
	for(x=0; x<Neighbors.size(); x++)
	{
		set<unsigned int> NonFreeColors;
		for(it = Neighbors[x].begin(); it != Neighbors[x].end(); it++)
		{
			if( *it < Colors.size() )
			{
				NonFreeColors.insert( Colors[*it] );
			}
		}

		unsigned int lowestfree = 0;
		while( NonFreeColors.find( lowestfree ) != NonFreeColors.end() )
		{
			lowestfree++;
		}

		Colors.push_back( lowestfree );
//			if(Output) *g_Log << "Color: " << lowestfree << "\n";
	}


	// Create new decomposition coloring
	{
		shared_ptr<TSparseUintField3> NewColoring( new TSparseUintField3(sc->Omega->m_BoundaryIndexField) );
		TIndexedOrigins_Set::iterator jt;

		int color = 0;
		for(x=0; x<p_Input->getIndexField()->getMaxIndex(); x++)
		{
			if(p_Input->vvaluex(x) != 0)
			{
				NewColoring->wvaluex(x) = Colors[ p_Input->vvaluex(x) - 1 ] + 1;
			}
		}

		shared_ptr<TLayer> NewLayer( new TLayer(NewColoring, p_Name) );
		NewLayer->m_VoxelSets[0]->setCheckedForRendering(true);
		NewLayer->m_ColorMap = NewLayer->m_AllColorMaps[5];
		NewLayer->m_ColorMap->init();
		NewLayer->m_ColorMap->m_Lower = 0.0f;
		MyApp::instance()->addLayer( NewLayer );

//		m_SegmentationLayers.push_back( NewLayer );
	}
}

void decompressZlib(const string& inputFile, const string& outputFile)
{
	wxFileInputStream input(inputFile);
	wxZlibInputStream zlib(input);

	wxFileOutputStream output(outputFile);
	zlib.Read(output);
}

void decompressAndWait(const string & p_File)
{
	SHELLEXECUTEINFO see = {0};
	see.cbSize = sizeof(SHELLEXECUTEINFO);
	see.hwnd = 0;
	see.lpVerb = L"";
	see.fMask = SEE_MASK_NOCLOSEPROCESS; 
	see.lpFile = L"gzip";
	see.hProcess = 0;
	string parameters = string("-d -f -q ") + p_File;
	see.lpParameters = wxString( parameters );
	see.nShow = SW_HIDE;
	int result = (int)ShellExecuteEx(&see);
	WaitForSingleObject(see.hProcess, 20000); // Wait for max 20s
	CloseHandle(see.hProcess);
}

void compressAndWait(const string & p_File)
{
	SHELLEXECUTEINFO see = {0};
	see.cbSize = sizeof(SHELLEXECUTEINFO);
	see.hwnd = 0;
	see.lpVerb = L"";
	see.fMask = SEE_MASK_NOCLOSEPROCESS; 
	see.lpFile = L"gzip";
	see.hProcess = 0;
	string parameters = string("-f -q ") + p_File;
	see.lpParameters = wxString( parameters );
	see.nShow = SW_HIDE;
	int result = (int)ShellExecuteEx(&see);
	WaitForSingleObject(see.hProcess, 20000); // Wait for max 20s
	CloseHandle(see.hProcess);
}
