#include "stdafx.h"

#include "volumesegmentation.h"

#include "parameters.h"
#include "globals.h"
#include "skeletonizer.h"
#include "layer.h"
#include "component.h"
#include "logwriter.h"
#include "colormap.h"
#include "wx/filename.h"
#include "model.h" 

bool TAction_VolumeSegmentationSkeletonize::isAvailable()
{
	return true;
}


void TAction_VolumeSegmentationSkeletonize::perform_init()
{
	m_Filename = "";
	wxFileDialog FileDialog(0, "Choose field file to open", "models", "", "*.vtk;*.vtk.gz", wxFD_OPEN);
	if( FileDialog.ShowModal() == wxID_OK)
	{
		m_Filename = FileDialog.GetPath().c_str();
	}	
}

void TAction_VolumeSegmentationSkeletonize::perform_main()
{
	if( m_Filename != "" )
	{
		g_Parameters.m_ComputeCskel = true;
		g_Parameters.m_OnlyCskelIterative = true;
		g_Parameters.m_OnlyKeepShortestPathsOnCskel = true;
		g_Parameters.m_KeepShortestPaths = true;
		g_Parameters.m_ComputeBackgroundSskel = false;
		g_Parameters.m_ComputeImportanceForCskel = true;


		TSkeletonizer * skel = new TSkeletonizer( m_Filename );
		skel->init(m_Filename);
		skel->perform();
		g_Mediator.updateUI();
	}
}






// ------------------------------------------------------------------------------------------


void TAction_VolumeSegmentation::selectCriticalPointsFromSelection()
{
	g_Mediator.getCurrentLayer()->m_Selection->initTraversal();
	TCoord3 c;
	while(g_Mediator.getCurrentLayer()->m_Selection->nextTraversal(c))
	{
		m_CriticalPoints.push_back(c);
	}
}

void TAction_VolumeSegmentation::selectCriticalPointFromJunctionField()
{
	unsigned int x;
	m_CriticalPoints.clear();
	TTypedFieldInterface<unsigned char>* Junctions = static_cast<TTypedFieldInterface<unsigned char>*>( g_Mediator.getCurrentLayerSet()->getField( "Junctions" ) );
	for(x=0; x<Junctions->getMaxIndex(); x++)
	{
		if(Junctions->vvaluex(x) > 0) m_CriticalPoints.push_back( Junctions->getIndexField()->vidx2coord(x) );
	}
}

void TAction_VolumeSegmentation::segment()
{
	const bool Output = false;
	TSkeletonizer * sc = g_Mediator.getSkeletonizer().get();
	
	const float BS = g_Mediator.getSkeletonizer()->m_BoundaryIndexField->getMaxIndex();

	TPrecisionTimer PrecisionTimer;
	PrecisionTimer.start();

	// =============================================================================================
	*g_Log << "Determine potential components of critical points\n";
	typedef multimap<float, shared_ptr<TIndexedOrigins_Set> > TPotentialComponents;
	TPotentialComponents PotComp;
	for(unsigned int x=0; x<m_CriticalPoints.size(); x++)
	{
		*g_Log << ".";
		TComponentSet Collapses( m_CriticalPoints.at(x) );
		TIndexedOrigins_Vector DilatedPathSet;
		g_Mediator.getSkeletonizer()->Omega->m_DeltaOmega->dilate(
			g_Parameters.m_DilationDistance
			, sc->m_SpSetField->vvaluep( m_CriticalPoints.at(x) )->m_PathSet->castConstVector()
			, &DilatedPathSet
			, true
		);

		sc->Omega->m_DeltaOmega->computeComponentSetUsingCells(&DilatedPathSet, &Collapses, true);

		unsigned int maxid = 0;
		{
			for(int j=1; j<Collapses.m_Cells.size(); j++) // note: collapse 0 is geodesic itself
			{
				if(Collapses.m_Cells[j]->vertexcount() > Collapses.m_Cells[maxid]->vertexcount() ) 
				{
					maxid = j;
				}
			}
		}

		// Add \partial\Omega \setminus maxid as component
		{
			shared_ptr<TIndexedOrigins_Set> s( new TIndexedOrigins_Set() );
			for(int j=1; j<Collapses.m_Cells.size(); j++)
			{
				if(j != maxid)
				{
					s->merge(Collapses.m_Cells[j].get());
				}
			}
			PotComp.insert( TPotentialComponents::value_type( ((float)s->vertexcount()/BS),s) );
		}
		
		// Add all components except largest
		{
			for(int j=1; j<Collapses.m_Cells.size(); j++)
			{
				if(j != maxid)
				{
					shared_ptr<TIndexedOrigins_Set> s( new TIndexedOrigins_Set() );
					s->merge(Collapses.m_Cells[j].get());
					PotComp.insert( TPotentialComponents::value_type( ((float)s->vertexcount())/BS,s) );
				}
			}
		}
	}

	// Insert boundary as component
	{
		shared_ptr<TIndexedOrigins_Set> s( new TIndexedOrigins_Set() );
		for(unsigned int x=0; x<g_Mediator.getSkeletonizer()->m_BoundaryIndexField->getMaxIndex(); x++)
		{
			s->insert( x );
		}
		PotComp.insert( TPotentialComponents::value_type( ((float)s->vertexcount())/BS,s) );
	}

	*g_Log << "\n";

	// =============================================================================================
	*g_Log << "Determine final components\n";
	
	// Debug info
	TPotentialComponents::iterator jt;
	if(true)
	{
		*g_Log << "Scales available: ";
		int componentcount = 0;
		for(jt = PotComp.begin(); jt != PotComp.end(); jt++)
		{
			*g_Log << wxString::Format("%.4f, ", jt->first);

			// Add potential components as layers
			if(false)
			{
				shared_ptr<TSparseUcharField3> NewField( new TSparseUcharField3(sc->Omega->m_BoundaryIndexField) );
				TIndexedOrigins_Set::iterator kt;
				for(kt = jt->second->begin(); kt != jt->second->end(); kt++)
				{
					NewField->wvaluep( g_Mediator.getSkeletonizer()->m_BoundaryIndexField->vidx2coord(*kt) ) = 1;
				}
				shared_ptr<TLayer> NewLayer( new TLayer(NewField, wxString::Format("Potential Component %i: %.4f", componentcount, jt->first).ToStdString()) );
				NewLayer->m_VoxelSets[0]->setCheckedForRendering(true);
				g_Mediator.addLayer( NewLayer );
			}
			componentcount++;
		}
		*g_Log << "\n";
	}

	float PreviousScale = -1.0f;
	unsigned int PreviousComponents = 0;
	float Scale = -1.0f;
	for(jt = PotComp.begin(); jt != PotComp.end(); jt++)
	{
//	if(jt->first < 0.001) continue;

	*g_Log << "\nProcessing scale " << wxString::Format("%.4f", jt->first) << "\n";

	Scale = jt->first;
	if(Scale - PreviousScale < 0.001)
	{
		*g_Log << "Skip scale\n";
		continue;
	}
	PreviousScale = Scale;

	vector<shared_ptr<TIndexedOrigins_Set> > FinalComp;
	vector<unsigned int> Parent;
	TPotentialComponents::iterator it;
	int componentcount = 0;
	for(it = PotComp.begin(); it != PotComp.end(); it++)
	{
		if(Output) *g_Log << "Processing potential component " << componentcount  << ", size: " << wxString::Format("%.4f", it->first).c_str() << "\n";
		else { *g_Log << "."; g_Mediator.yield(); }

		componentcount++;

		if( it->first < Scale  ) 
		{
//			if(Output) *g_Log << "\tToo small scale.\n";
			continue;
		}


		// See if difference with all final components is either empty or the full set
		unsigned int minidx = 0;
		TIndexedOrigins_Set OverlapSet;
		OverlapSet.clear();

		shared_ptr<TIndexedOrigins_Set> larger = it->second;
		vector<unsigned int> Overlap;
		Overlap.resize( FinalComp.size() );
		// vector<unsigned int> OverlappingComponents;
		unsigned int TotalOverlap = 0;
		for(unsigned int x=0; x<FinalComp.size(); x++)
		{
			//Intersection.clear();
			const TIndexedOrigins_Set * smaller = FinalComp[x].get();
			TIndexedOrigins_Set::iterator jt = larger->begin();
			TIndexedOrigins_Set::const_iterator kt = smaller->begin();

			bool isOverlapping = false;
			while(jt != larger->end() && kt != smaller->end())
			{
				if(*jt < *kt) jt++;
				else if(*jt > *kt) kt++;
				else if(*jt == *kt) 
				{ 
					OverlapSet.insert(*jt);
					Overlap[x]++;
					TotalOverlap++;
					jt++; 
					kt++; 
					//isOverlapping = true;
				}
			}
//			if(Output) *g_Log << "\tOverlap with " << x << " = " << Overlap[x] << "\n";
		}

		// Determine amount of components that x overlaps
		unsigned int OverlappingComponents = 0;
		for(unsigned int x=0; x<Overlap.size(); x++)
		{
			// if( Overlap[x] > 0 ) OverlappingComponents++;
			
			if( (float)Overlap[x] > 0.5f * (float)FinalComp[x]->vertexcount() ) OverlappingComponents++;
		}
	
		bool valid = false;
		shared_ptr<TIndexedOrigins_Set> ValidLayer;
		
		int contribution = larger->vertexcount() - TotalOverlap;
//		if(Output) *g_Log << "\tCurrent size: " << larger->vertexcount() << ", contribution: " << contribution << "\n";


		if(OverlappingComponents == 0)
		{
			// There is no overlap
//			if(Output) *g_Log << "\tNo overlap\n";
			
			if( ((float)contribution/BS) >= 0.001f )
			{
				ValidLayer = larger;
				valid = true;
			}
			else 
			{
				valid = false;
//				if(Output) *g_Log << "\tContribution too small\n";
			}
		}
		else // Overlappingcomponents > 0
		{
			// Invert overlapset
			shared_ptr<TIndexedOrigins_Set> InvertedOverlapSet( new TIndexedOrigins_Set() );
			TIndexedOrigins_Set::iterator jt;
			for(jt = larger->begin(); jt != larger->end(); jt++)
			{
				if( OverlapSet.find(*jt) == OverlapSet.end() )
				{
					InvertedOverlapSet->insert( *jt );
				}
			}

//			if(	((float)contribution/BS) >= 0.05f )

			bool merge = false;
			if( (float)contribution/BS < 0.005f )
			{
				merge = true;
				valid = false;
			}
			else if(	
				( (float)contribution/BS >= 0.02f )
				&&
				( (float)contribution/BS >= ((float)TotalOverlap)/BS/16.0f )
			
			)
			{
				ValidLayer = InvertedOverlapSet;
				valid = true;
			}
			else // Contribution too small
			{
				if( it->first > 0.99f ) // Do not merge whole boundary component
				{
					merge = true;
				}
				else
				{
					// Todo
					merge = false;
					//merge = true;
					valid = false;
				}
			}

			if(merge)
			{
				// Contribution is too small, merge with others 
				vector<TIndexedOrigins_Set*> Others;
				for(int i=0; i<Overlap.size(); i++)
				{
					if(Overlap[i] > 0 )
					{
						Others.push_back( FinalComp[ i ].get() );
					}
				}
				sc->Omega->m_DeltaOmega->mergeCollapseWithOthers( InvertedOverlapSet.get(), Others );
				if(Output) *g_Log << "\tContribution too small: doing merge\n";
			}
		} 


		// Add new layer
		if(valid)
		{
			if(Output) *g_Log << "\tAdded final component " << (unsigned int)FinalComp.size() << "\n";

			FinalComp.push_back( ValidLayer );
		}
		else
		{
			if(Output) *g_Log <<"\tDiscarded component\n";
		}

	}


	// Add segmentation field
	*g_Log << "Found " << (unsigned int)FinalComp.size() << " components\n";
	if(FinalComp.size() > 1)
	{
		shared_ptr<TSparseUintField3> OldColoring( new TSparseUintField3(sc->Omega->m_BoundaryIndexField) );
		{
			if(FinalComp.size() >= 255) throw string("Warning: #components > 255");

			TIndexedOrigins_Set::iterator jt;

			if(Output) *g_Log << "Finalcomp size: " << (unsigned int)FinalComp.size() << "\n";
			int color = 0;
			for(int i=0; i<FinalComp.size(); i++)
			{
				for(jt = FinalComp[i]->begin(); jt != FinalComp[i]->end(); jt++)
				{
					const TIndexedOrigins::TIndex & idx = *jt;
					OldColoring->wvaluep( g_Mediator.getSkeletonizer()->m_BoundaryIndexField->vidx2coord(idx) ) = i+1;
				}
			}

			if(false) // Debug: add old coloring
			{
				shared_ptr<TLayer> NewLayer( new TLayer(OldColoring, "Decomposition 1") );
				NewLayer->m_VoxelSets[0]->setCheckedForRendering(true);
				g_Mediator.addLayer( NewLayer );
			}
		}

		if(FinalComp.size() != PreviousComponents)
		{
			createColoring(OldColoring.get(), FinalComp.size(), wxString::Format("Hier segm lvl %i, scale %.4f", FinalComp.size(), Scale).ToStdString() );
			PreviousComponents = FinalComp.size();
		}
		else *g_Log << "Do no add field: FinalComp.size() == PreviousComponents\n";
	}
	else *g_Log << "Do no add field: FinalComp.size() == 0\n";
	
	if(!m_SegmentAll)
	{
		break;
	}

	} // end for scale

	PrecisionTimer.stop();
	float time = PrecisionTimer.stop();
	*g_Log << wxString::Format("\nFinished decomposition in %.2f seconds\n",time ).c_str();
}

// ------------------------------------------------------------------------------------------

bool TAction_VolumeSegmentationFinestScale::isAvailable()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	if(!skel) return false;
	return g_Mediator.getCurrentLayerSet()->exists( "Junctions" );
}


void TAction_VolumeSegmentationFinestScale::perform_main()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();

	selectCriticalPointFromJunctionField();
	m_SegmentAll = false;
	segment();
}

// ------------------------------------------------------------------------------------------

bool TAction_VolumeSegmentationFinestScaleUsingSelectedCp::isAvailable()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	if(!skel) return false;
	return true;
}


void TAction_VolumeSegmentationFinestScaleUsingSelectedCp::perform_main()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();

	selectCriticalPointsFromSelection();
	m_SegmentAll = false;
	segment();
}


// ------------------------------------------------------------------------------------------

bool TAction_VolumeSegmentationAllScales::isAvailable()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	if(!skel) return false;
	return g_Mediator.getCurrentLayerSet()->exists( "Junctions" );
}


void TAction_VolumeSegmentationAllScales::perform_main()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();

	selectCriticalPointFromJunctionField();
	m_SegmentAll = true;
	segment();
}

// ------------------------------------------------------------------------------------------

bool TAction_CurveSkeletonFeatureAngle::isAvailable()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	if(!skel) return false;
	return skel->m_LoopField.get() != 0;
}


float TAction_CurveSkeletonFeatureAngle::computeGeodesicness(const TCoord3 p, const class TIndexedOrigins_Vector * p_Path, const class TIndexedOrigins_Vector * p_Eft, TTypedFieldInterface<unsigned int> * p_Debug)
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	vector<float> Distances( p_Path->size(), 0.0f );
	float minangle = 1000.0f;

	// For each feature point
	//const TIndexedOrigins_Vector * eft = sps->m_Seft.get();
//	const TIndexedOrigins_Vector * eft = sps->m_Eft.get();
	for(unsigned int y=0; y<p_Eft->size(); y++)
	{

		const float maximumdist = skel->Omega->m_DeltaOmega->computeDistancesOnIndexes( p_Eft->at(y), p_Path, &Distances );
		const float halfdist = maximumdist / 2.0f;

		vector<unsigned int> Indexes;
		unsigned int a,b;
		for(unsigned int z=0; z<p_Path->size(); z++)
		{
			if( Distances[z] > halfdist - 1.0f && Distances[z] < halfdist + 1.0f ) // range of 2.0f
			{
				const unsigned int idx = (*p_Path)[z];
				Indexes.push_back( z );
			}
		}

		// Find the two indexes that are furthest appart

		float maxdist = 0.0f;
		unsigned int mina = (std::numeric_limits<unsigned int>::max)();
		unsigned int minb = (std::numeric_limits<unsigned int>::max)();
		for(a=0; a<Indexes.size(); a++)
		for(b=a+1; b<Indexes.size(); b++)
		{
			const float dist = skel->m_BoundaryIndexField->vidx2coord( (*p_Path)[Indexes[a]] ).distance( skel->m_BoundaryIndexField->vidx2coord( (*p_Path)[Indexes[b]] ) );
			if( dist > maxdist )
			{
				maxdist = dist;
				mina = Indexes[a];
				minb = Indexes[b];
			}
		}
		if(mina == (std::numeric_limits<unsigned int>::max)() || minb == (std::numeric_limits<unsigned int>::max)()) continue;
		const float dista = Distances[ mina ]; 
		const float distb = Distances[ minb ];
		shared_ptr<TShortestPath> sp = skel->Omega->m_DeltaOmega->getShortestPath( (*p_Path)[mina], (*p_Path)[minb] );
		const float distc = sp->m_Length;
		if(p_Debug)
		{
			for(unsigned int z=0; z<sp->m_Path->size(); z++)
			{
				p_Debug->wvaluex( (*sp->m_Path)[z] ) = 1;
			}	
		}

		float angle =  distc / (dista + distb);

		if(angle > 1.0f) angle = 1.0f;
		if(angle < 0.0f) angle = 0.0f;

		if(angle < minangle)
		{
			minangle = angle;
		}
	}

	return minangle;
}

void TAction_CurveSkeletonFeatureAngle::perform_main()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	TTypedFieldInterface<TShortestPathSet*> * SPSF = skel->m_SpSetField.get();

	const TTypedFieldInterface<TComponentSet*>* CsField = static_cast<TTypedFieldInterface<TComponentSet*>*>( g_Mediator.getCurrentLayerSet()->getField( "Component Sets" ) );
	if(!CsField) throw string("!CsField");
	

	TTypedFieldInterface<float>* Output = static_cast<TTypedFieldInterface<float>*>( g_Mediator.getCurrentLayerSet()->newField( TType::TYPE_FLOAT, skel->m_CskelIndexField, getName() ) );
	Output->clear();

	{
		unsigned int x;
		for(x=0; x<skel->m_CskelIndexField->getMaxIndex(); x++)
		{
			const TCoord3 p = skel->m_CskelIndexField->vidx2coord(x);
			const TShortestPathSet * sps = SPSF->vvaluep(p);
			const TComponentSet * cs = CsField->vvaluep(p);
			if(sps == 0) continue;
			if(cs == 0) continue;
			if(cs->m_Borders.size() < 2) continue;


			// Method using difference in border lengthts
			float minangle = 1.0f;
			if(false)
			{
				if(cs->m_Borders.size() == 2)
				{
					const float max = cs->m_BorderLengths[0] > cs->m_BorderLengths[1] ? cs->m_BorderLengths[0] : cs->m_BorderLengths[1];
					const float min = cs->m_BorderLengths[0] > cs->m_BorderLengths[1] ? cs->m_BorderLengths[1] : cs->m_BorderLengths[0];
					minangle = 1.0f - (max-min)/max;
				}
				else
				{
					minangle = 0.01f;
				}
				
			}
			else
			// Old method using shortest path between midpoints
			{ 
				
				unsigned int y;
				for(y=0; y<cs->m_Borders.size(); y++)
				{
					const float angle = computeGeodesicness(p, cs->m_Borders[y].get(), cs->m_BorderEft[y].get(), 0 );
					if(angle < minangle) minangle = angle;
				}
			}

			Output->wvaluep(p) = minangle;

		}
	}


	Output->getLayer()->onFieldChanged();
	Output->getLayer()->m_ColorMap->m_Lower = 0.01f;
	Output->getLayer()->setCheckedForRendering(false);
	//Output2->getLayer()->onFieldChanged();
	//Output2->getLayer()->m_ColorMap->m_Lower = 0.01f;
	
};


// ------------------------------------------------------------------------------------------

bool TAction_ComputeComponentSets::isAvailable()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	if(!skel) return false;
	return g_Mediator.getSkeletonizer()->m_SpSetField.get() != 0;
}


void TAction_ComputeComponentSets::perform_main()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();

	TTypedFieldInterface<TComponentSet*>* Output = static_cast<TTypedFieldInterface<TComponentSet*>*>( g_Mediator.getCurrentLayerSet()->newField( TType::TYPE_COMPONENTSET, skel->m_CskelIndexField, "Component Sets") );
	Output->clear();

	TTypedFieldInterface<TShortestPathSet*> * SPSF = static_cast<TTypedFieldInterface<TShortestPathSet*>*>( skel->m_SpSetField.get() );
	unsigned int x;
	for(x=0; x<skel->m_CskelIndexField->getMaxIndex(); x++)
	{
		const TCoord3 p = skel->m_CskelIndexField->vidx2coord(x);
		TShortestPathSet * sps = SPSF->vvaluep(p);
		if(!sps) continue;
		
		Output->wvaluex(x) = new TComponentSet(sps);
	}
	Output->getLayer()->onFieldChanged();
}


// ------------------------------------------------------------------------------------------

bool TAction_SetRobustComponentTau::isAvailable()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	if(!skel) return false;
	return true;
}


void TAction_SetRobustComponentTau::perform_main()
{
	wxString s = "";
	double Threshold = g_Parameters.m_RobustComponentTau;
	do
	{
		s = ::wxGetTextFromUser("Robust Component Tau", "Robust Component Tau", wxString::Format("%f",Threshold).c_str() );
		if(s == "") return;
	} 
	while( !s.ToDouble(&Threshold) );
	g_Parameters.m_RobustComponentTau = Threshold;
}

// ------------------------------------------------------------------------------------------

bool TAction_SetSegmSigmaThreshold::isAvailable()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	if(!skel) return false;
	return true;
}


void TAction_SetSegmSigmaThreshold::perform_main()
{
	wxString s = "";
	double Threshold = g_Parameters.m_Segmentation_MinimumSigma;
	do
	{
		s = ::wxGetTextFromUser("Minimum geodesicness (sigma)", "Minimum geodesicness (sigma)", wxString::Format("%f",Threshold).c_str() );
		if(s == "") return;
	} 
	while( !s.ToDouble(&Threshold) );
	g_Parameters.m_Segmentation_MinimumSigma = Threshold;
}

// ------------------------------------------------------------------------------------------

bool TAction_SetSegmRhoSimilar::isAvailable()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	if(!skel) return false;
	return true;
}


void TAction_SetSegmRhoSimilar::perform_main()
{
	wxString s = "";
	double Threshold = g_Parameters.m_Segmentation_RhoSimilar;
	do
	{
		s = ::wxGetTextFromUser("Body/limb rho similarity threshold", "Body/limb rho similarity threshold", wxString::Format("%f",Threshold).c_str() );
		if(s == "") return;
	} 
	while( !s.ToDouble(&Threshold) );
	g_Parameters.m_Segmentation_RhoSimilar = Threshold;
}


// ------------------------------------------------------------------------------------------

bool TAction_MoveCriticalPoints::isAvailable()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	if(!skel) return false;
	return true;
	
}

double TAction_MoveCriticalPoints::gauss_d2(double x, double s)
{
	const double teller = - exp(-(x*x) / (2.0*s*s) ) * (s-x) * (s+x);
	const double nummer = sqrt(PI * 2.0f) * pow(s,5.0);
	return teller / nummer;
}


void TAction_MoveCriticalPoints::findShortestPathOnCskel(unsigned int p_Start, unsigned int p_End, TIndexedOrigins_Vector * p_Result)
{
	TTypedFieldInterface<unsigned char>* Cskel = static_cast<TTypedFieldInterface<unsigned char>*>( g_Mediator.getCurrentLayerSet()->getField( "C-skel detector" ) );
	if(!Cskel) throw string("!Cskel");
	
	vector<unsigned int> Previous( Cskel->getMaxIndex(), (std::numeric_limits<unsigned int>::max)());
	vector<float> Distance( Cskel->getMaxIndex(), (std::numeric_limits<float>::max)());

	std::multimap<float, unsigned int> Q;
	Q.insert(pair<float,unsigned int>(0.0f,p_Start) );
	Previous[p_Start] = p_Start;
	Distance[p_Start] = 0.0f;
	unsigned int pidx = (std::numeric_limits<unsigned int>::max)();
	while(!Q.empty())
	{
		pidx = Q.begin()->second;
		Q.erase(Q.begin());
		if(pidx == p_End)
		{
			break;
		}
		const TCoord3 p = Cskel->getIndexField()->vidx2coord(pidx);

		const int W = 2;
		TCoord3 np;
		for(np.x=p.x-W; np.x<=p.x+W; np.x++) // neighbors
		for(np.y=p.y-W; np.y<=p.y+W; np.y++)
		for(np.z=p.z-W; np.z<=p.z+W; np.z++)
		{
			if(!Cskel->getIndexField()->vinside(np)) continue;

			const unsigned int npidx = Cskel->getIndexField()->vcoord2idx(np);
			if(Distance[pidx] + 1.0f < Distance[npidx])
			{
				Distance[npidx] = Distance[pidx] + 1.0f;
				Previous[npidx] = pidx;
				Q.insert(pair<float,unsigned int>(Distance[npidx],npidx) );
			}
		}
	}
	if(p_Result == 0) throw string("p_Result == 0");

	if(pidx != (std::numeric_limits<unsigned int>::max)())
	{
		// Trace back
		while(pidx != p_Start)
		{
			p_Result->push_back(pidx);
			pidx = Previous[pidx];
		}
	}

}

void TAction_MoveCriticalPoints::perform_main()
{
	unsigned int x;
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	//TTypedFieldInterface<TShortestPathSet*> * SPSF = skel->m_ForegroundSpSetField.get();


	if(!g_Mediator.getCurrentLayerSet()->exists("Junctions"))
	{
		TAction_NewJunctionDetection Action;
		Action.perform();
		TTypedFieldInterface<unsigned char>* Junctions = static_cast<TTypedFieldInterface<unsigned char>*>( g_Mediator.getCurrentLayerSet()->getField("Junctions" ) );
		if(!Junctions) throw string("!Junctions");
	}

	const TTypedFieldInterface<TShortestPathSet*> * SPSF = skel->m_ForegroundSpSetField.get();
	if(!SPSF) throw string("!SPSF");

	if(!g_Mediator.getCurrentLayerSet()->exists(TAction_CurveSkeletonFeatureAngle::getStaticName()))
	{
		TAction_CurveSkeletonFeatureAngle Action;
		Action.perform();
	}
	TTypedFieldInterface<float>* Measure = static_cast<TTypedFieldInterface<float>*>( g_Mediator.getCurrentLayerSet()->getField(TAction_CurveSkeletonFeatureAngle::getStaticName() ) );
	if(!Measure) throw string("!Measure");

	TTypedFieldInterface<float>* CskelRho = static_cast<TTypedFieldInterface<float>*>( g_Mediator.getCurrentLayerSet()->getField("C-skel measure") );

	// Small rho's have high sigma's
	// Problem, in boxwith2boxes the tips are found as critical points
	if(false)
	{
			TTypedFieldInterface<float>* CskelMeasure = static_cast<TTypedFieldInterface<float>*>( g_Mediator.getCurrentLayerSet()->getField("C-skel measure") );
		
		// float maxrho = CskelMeasure->getLayer()->m_FilterMeasure->getMaxValue() / 5.0f;
		const float maxrho = 25.0f;
		
		*g_Log << "Make C-skel point with small rho's have high geodesicness, maxrho=" << maxrho << "\n";
		for(x=0; x<Measure->getMaxIndex(); x++)
		{
			const TShortestPathSet * sps = SPSF->vvaluep(Measure->getIndexField()->vidx2coord(x));
			if(sps->getMaxLength() < maxrho && Measure->wvaluex(x) < 0.9f)
			{
				Measure->wvaluex(x) = 0.9f;
			}
		}
		Measure->getLayer()->onFieldChanged();
	}

	TTypedFieldInterface<unsigned char>* Junctions = static_cast<TTypedFieldInterface<unsigned char>*>( g_Mediator.getCurrentLayerSet()->getField("Junctions" ) );
	if(!Junctions) throw string("!Junctions");



	// Make a flat segmentation using all the junctions
	TTypedFieldInterface<unsigned char>* FlatSegmentation = 0;
	const bool FlatSegmentationPerJunction = true; // *** PARAMETER
	
	// Do segmentation for whole shape at once
	if(!FlatSegmentationPerJunction)
	{
		TAction_VolumeSegmentationFlat Action;
		Action.m_Name = wxString::Format("Flat segmentation for moving critical points").c_str();
		Action.m_FillZeros = false;
		Junctions->getLayer()->m_Selection->clear();
		for(x=0; x<Junctions->getMaxIndex(); x++)
		{
			if(Junctions->vvaluex(x) == 0) continue;
			const TCoord3 p = Junctions->getIndexField()->vidx2coord(x);
			Junctions->getLayer()->m_Selection->select( p );
		}
		g_Mediator.setCurrentLayer( Junctions->getLayer() );
		Action.perform();
		FlatSegmentation = static_cast<TTypedFieldInterface<unsigned char>*>( g_Mediator.getCurrentLayerSet()->getField( Action.m_Name ) );
		if(!FlatSegmentation) throw string("!FlatSegmentation");
	}


	TTypedFieldInterface<unsigned char>* Cskel = static_cast<TTypedFieldInterface<unsigned char>*>( g_Mediator.getCurrentLayerSet()->getField( "C-skel detector" ) );
	if(!Cskel) throw string("!Cskel");


	// Make copy of measure
	if(false)
	{
		TTypedFieldInterface<float>* CopyMeasure = static_cast<TTypedFieldInterface<float>*>( g_Mediator.getCurrentLayerSet()->newField(TType::TYPE_FLOAT, skel->m_CskelIndexField, TAction_CurveSkeletonFeatureAngle::getStaticName() + " copy" ) );
		if(!CopyMeasure) throw string("!CopyMeasure");
		for(x=0; x<Measure->getMaxIndex(); x++) CopyMeasure->wvaluex(x) = Measure->vvaluex(x);
		CopyMeasure->getLayer()->onFieldChanged();
	}

	TTypedFieldInterface<TComponentSet*>* CsField = static_cast<TTypedFieldInterface<TComponentSet*>*>( g_Mediator.getCurrentLayerSet()->getField( "Component Sets" ) );
	if(!CsField) throw string("!CsField");
	
//	TTypedFieldInterface<unsigned char>* TempComponentSetField = static_cast<TTypedFieldInterface<unsigned char>*>( g_Mediator.getCurrentLayerSet()->newField(TType::TYPE_UCHAR, skel->m_BoundaryIndexField, "Temp cs field" ) );

	TTypedFieldInterface<unsigned char>* JunctionBlocker = static_cast<TTypedFieldInterface<unsigned char>*>( g_Mediator.getCurrentLayerSet()->newField(TType::TYPE_UCHAR, Junctions->getIndexField(), "Junction blocker" ) );
	{
		JunctionBlocker->clear();
		for(x=0; x<Junctions->getMaxIndex(); x++)
		{
			if(Junctions->vvaluex(x) == 0.0f) continue;
			const TCoord3 p = Junctions->getIndexField()->vidx2coord(x);

			const int W = 2;
			TCoord3 np;
			for(np.x=p.x-W; np.x<=p.x+W; np.x++) // neighbors
			for(np.y=p.y-W; np.y<=p.y+W; np.y++)
			for(np.z=p.z-W; np.z<=p.z+W; np.z++)
			{
				if(!Junctions->getIndexField()->vinside(np)) continue;
				JunctionBlocker->wvaluep(np) = Junctions->vvaluex(x);
			}
		}

		// Test: voeg aan junctionblocker stukken C-skel toe die tussen dezelfde junction ids zitten
		unsigned int y;
		for(x=0; x<Junctions->getMaxIndex(); x++)
		for(y=x+1; y<Junctions->getMaxIndex(); y++)
		{
			if(Junctions->vvaluex(x) == 0) continue;
			if(Junctions->vvaluex(y) == 0) continue;
			if(Junctions->vvaluex(x) != Junctions->vvaluex(y)) continue;

			const TCoord3 px = Junctions->getIndexField()->vidx2coord(x);
			const TCoord3 py = Junctions->getIndexField()->vidx2coord(y);
			const float juncdist = px.distance(py);
			// Vind korste pad tussen twee junctions over C-skel

			TIndexedOrigins_Vector path;
			path.clear();
			findShortestPathOnCskel(x,y,&path);
			for(unsigned int a=0; a<path.size(); a++)
			{
				const TCoord3 p = Junctions->getIndexField()->vidx2coord( path[a] );

				const int W = 2;
				TCoord3 np;
				for(np.x=p.x-W; np.x<=p.x+W; np.x++) // neighbors
				for(np.y=p.y-W; np.y<=p.y+W; np.y++)
				for(np.z=p.z-W; np.z<=p.z+W; np.z++)
				{
					if(!Junctions->getIndexField()->vinside(np)) continue;
					JunctionBlocker->wvaluep(np) = Junctions->vvaluex(x);
				}
			}
		}

		JunctionBlocker->getLayer()->onFieldChanged();
	}

//	TTypedFieldInterface<float>* RhoDerivative = static_cast<TTypedFieldInterface<float>*>( g_Mediator.getCurrentLayerSet()->newField(TType::TYPE_FLOAT, skel->m_CskelIndexField, "RhoDerivative" ) );
//	RhoDerivative->clear();

	unsigned int maxjunctionid = 0;
	for(x=0; x<Junctions->getMaxIndex(); x++)
	{
		if(Junctions->vvaluex(x) > maxjunctionid) maxjunctionid = Junctions->vvaluex(x);
	}


	// Junction Id -> { (Cp, real Cp) }
	typedef map<unsigned int, map<unsigned int, TCriticalPointInfo> > TJunctionId2Cp;
	TJunctionId2Cp JunctionId2Cp;
	typedef map<unsigned int, float > TJunctionId2Rho;
	TJunctionId2Rho JunctionId2RhoMin;
	TJunctionId2Rho JunctionId2RhoMax;
	typedef map<unsigned int, vector<unsigned int> > TJunctionId2Junctions;
	TJunctionId2Junctions JunctionId2Junctions;

	// For each junction id, search for new critical points starting from junctions
	for(unsigned int z=1; z<=maxjunctionid; z++)
	{
		// For each junction, do distance propagation
		vector<float> Distance( Junctions->getMaxIndex(), (std::numeric_limits<float>::max)() );
		vector<unsigned int> Previous( Junctions->getMaxIndex(), (std::numeric_limits<unsigned int>::max)() );
		
		const bool KeepAllSegmentations = false; 

		// Do segmentation per junction
		vector<float> ComponentCollapses;
		{
			TAction_VolumeSegmentationFlat Action;
			if(KeepAllSegmentations)
				Action.m_Name = wxString::Format("Flat segmentation for moving critical points %i", z).c_str();
			else
				Action.m_Name = "Flat segmentation for moving critical points";

			Action.m_FillZeros = false;
			Action.m_DilationDistance = 4.0f;
			Junctions->getLayer()->m_Selection->clear();
			for(x=0; x<Junctions->getMaxIndex(); x++)
			{
				if(Junctions->vvaluex(x) == 0) continue;
				//if(Junctions->vvaluex(x) != z) continue;
				TComponentSet * cs = CsField->vvaluex(x); 

				const TCoord3 p = Junctions->getIndexField()->vidx2coord(x);
				if(Junctions->vvaluex(x) == z || cs->isOnLoop() )
					Junctions->getLayer()->m_Selection->select( p );
			}
			g_Mediator.setCurrentLayer( Junctions->getLayer() );
			Action.perform();
			if(FlatSegmentationPerJunction)
			{
				FlatSegmentation = static_cast<TTypedFieldInterface<unsigned char>*>( g_Mediator.getCurrentLayerSet()->getField( Action.m_Name ) );
				if(!FlatSegmentation) throw string("!FlatSegmentation");
			}

			// Determine collapse per component
			{
				ComponentCollapses.resize( Action.m_Sizes.size() );
				for(x=1; x<Action.m_Sizes.size(); x++)
				{
					const float compsize = (float)Action.m_Sizes[x].size();
					ComponentCollapses[x] = compsize / (float)Action.m_Output->getMaxIndex();
				}
			}
		}



		typedef multimap<float, unsigned int> TMap;
		TMap Q;
		Q.clear();
		
		// Push all junctions with id "z" onto queue
		float MinJuncLength = 0.0f;
		float MaxJuncLength = (std::numeric_limits<float>::max)();
		float MaxJuncEft = 0.0f;
		for(x=0; x<Junctions->getMaxIndex(); x++)
		{
			if(Junctions->vvaluex(x) == z)
			{
				Q.insert( TMap::value_type(0.0f, x) );
				Distance[x] = 0.0f;
				const TShortestPathSet * sps = SPSF->vvaluep(Junctions->getIndexField()->vidx2coord(x));
				if( sps->getMaxLength() > MinJuncLength ) MinJuncLength = sps->getMaxLength();
				if( sps->getMaxLength() < MaxJuncLength ) MaxJuncLength = sps->getMaxLength();
				JunctionId2Junctions[z].push_back(x);
			}
		}
		JunctionId2RhoMin[z] = MinJuncLength;
		JunctionId2RhoMax[z] = MaxJuncLength;
		if(Q.size() == 0) continue;

		set<unsigned int> OtherJunctionsReached;
		while(!Q.empty())
		{
			const unsigned int pidx = Q.begin()->second;
			const TCoord3 p = Junctions->getIndexField()->vidx2coord( pidx );
			Q.erase( Q.begin() );


			// Stop when other junction reached
			if( JunctionBlocker->vvaluex(pidx) != z 
				&& JunctionBlocker->vvaluex(pidx) != 0
				) 
			{
				OtherJunctionsReached.insert( JunctionBlocker->vvaluex(pidx) );
				continue;
			}

			TCoord3 np;
			for(np.x=p.x-1; np.x<=p.x+1; np.x++) // neighbors
			for(np.y=p.y-1; np.y<=p.y+1; np.y++)
			for(np.z=p.z-1; np.z<=p.z+1; np.z++)
			{
//				if( (p.x == np.x) + (p.y == np.y) + (p.z == np.z) != 2 ) continue; // Only 6-neighbors
				const unsigned int npidx = Junctions->getIndexField()->vcoord2idx(np);
				if(npidx == TIndexMapper::OUTSIDE) continue;

				float WEIGHT = p.toVector().distance( np.toVector() );

				// Option: bias weight with measure to prefer path going over red voxels
				WEIGHT += (1.0f - Measure->vvaluep(np));


				const float newd = Distance[pidx] + WEIGHT;
				if( newd < Distance[npidx] )
				{
					Distance[npidx] = newd;
					Previous[npidx] = pidx;
					Q.insert( TMap::value_type(newd,npidx) );
				}
			}		
		} // end while


		// For each component, find the junction reached//
		// or if no junction reached, the farthest voxel reached
		map<unsigned int, unsigned int> ComponentToPathEnd;
		map<unsigned int, set<unsigned int> > ComponentToVoxels;
		unsigned int FoundJunctionIdx = 0;
		for(x=0; x<Distance.size(); x++)
		{
			const TCoord3 p = Junctions->getIndexField()->vidx2coord(x);
			const float dist = Distance[x];
			if(dist == (std::numeric_limits<float>::max)()) continue;

			// See if ONE of the eft is in any component
			 /*
			const TShortestPathSet * sps = SPSF->vvaluep( p );
			set<unsigned int> Components;
			for(unsigned int y=0; y<sps->m_Eft->size(); y++)
			{
				const unsigned int idx = (*sps->m_Eft)[y];
				const unsigned int id = FlatSegmentation->vvaluex(idx);
				if(id == 0) continue;
				Components.insert(id);
			}
			std::set<unsigned int>::iterator it;
			for(it = Components.begin(); it != Components.end(); it++)
			{
				unsigned int componentid = *it;
				ComponentToVoxels[componentid].insert( x );
				
				if( ComponentToFarthest.find(componentid) == ComponentToFarthest.end()
					|| dist > Distance[ ComponentToFarthest[componentid] ] )
				{
					ComponentToFarthest[componentid] = x;
				}
			}
			// */
			
			 // /*
			// See if all of the EFT is in one component
			unsigned int componentid = (std::numeric_limits<unsigned int>::max)();
			{
				// See if ALL of the eft is in ONE component
				const TShortestPathSet * sps = SPSF->vvaluep( p );
				if(sps == 0) continue;
				set<unsigned int> Components;
				for(unsigned int y=0; y<sps->m_Eft->size(); y++)
				{
					const unsigned int idx = (*sps->m_Eft)[y];
					// compid = TempComponentSetField->vvaluex(idx);
					const unsigned int id = FlatSegmentation->vvaluex(idx);
					if(id == 0) continue;
					Components.insert(id);
				}
				if(Components.size() == 1) componentid = *Components.begin();
			}
			if(componentid == (std::numeric_limits<unsigned int>::max)()) continue;
			ComponentToVoxels[componentid].insert( x );
			
			if( ComponentToPathEnd.find(componentid) == ComponentToPathEnd.end()
				|| dist > Distance[ ComponentToPathEnd[componentid] ] )
			{
				ComponentToPathEnd[componentid] = x;
			}
			// */
		}

		// Overwrite farthest voxels with junction voxels
		//if(false) // ***************
		for(x=0; x<Distance.size(); x++)
		{
			const TCoord3 p = Junctions->getIndexField()->vidx2coord(x);
			if( !( JunctionBlocker->vvaluep(p) != 0 && JunctionBlocker->vvaluep(p) != z) ) continue;
			

			const float dist = Distance[x];
			if(dist == (std::numeric_limits<float>::max)()) continue;

			unsigned int componentid = (std::numeric_limits<unsigned int>::max)();
			{
				// See if ALL of the eft is in ONE component
				const TShortestPathSet * sps = SPSF->vvaluep( p );
				if(sps == 0) continue;
				set<unsigned int> Components;
				for(unsigned int y=0; y<sps->m_Eft->size(); y++)
				{
					const unsigned int idx = (*sps->m_Eft)[y];
					// compid = TempComponentSetField->vvaluex(idx);
					const unsigned int id = FlatSegmentation->vvaluex(idx);
					if(id == 0) continue;
					Components.insert(id);
				}
				if(Components.size() == 1) componentid = *Components.begin();
			}
			if(componentid == (std::numeric_limits<unsigned int>::max)()) continue;
		
			ComponentToPathEnd[componentid] = x;
		}


		// Now for each component, trace back to junction, making a path for each component
		typedef std::map<float, unsigned int> TPath;
		map<unsigned int, TPath > ComponentToPath;
		map<unsigned int, unsigned int> ComponentToJunction;
		{
			std::map<unsigned int, unsigned int>::iterator it;
			for(it = ComponentToPathEnd.begin(); it != ComponentToPathEnd.end(); it++)
			{	
				const unsigned int compid = it->first;
				const unsigned int farthest = it->second;
				const float distance = Distance[farthest];
				x=farthest;
				while( Junctions->vvaluex(x) == 0 )
				{
					const float d = Distance[x];
					ComponentToPath[it->first].insert( TPath::value_type(d, x) );
					x = Previous[x];
				}
				ComponentToJunction[it->first] = x;
			}
		}

		// Find critical points
		const unsigned int CriticalPointMethod = 3;
		if(CriticalPointMethod == 1)
		{
			std::map<unsigned int, TPath >::iterator it;
			for(it = ComponentToPath.begin(); it != ComponentToPath.end(); it++)
			{	
				const unsigned int compid = it->first;
				TPath & path = it->second;

				const unsigned int pathsize = path.size();
				// const float junceft = JunctionId2Eft[z];
				//const float junceft = SPSF->vvaluep( Junctions->getIndexField()->vidx2coord( path.begin()->second) )->getRadius();


				unsigned int skipped = 0;
//				bool reachedotherjunction = false;
//				bool hasbeenabovethreshold = false;
				unsigned int firstincomp = (std::numeric_limits<unsigned int>::max)();
				
				float CumulativeDistance = 0.0f;
				unsigned int ZeroMeasures = 0;
				const float pathstartradius = SPSF->vvaluep( skel->m_CskelIndexField->vidx2coord( path.begin()->second ) )->getMaxRadius();
				
				// Determine farthest point from p to \gamma(p) where p = begin of path
				float coverdistance = 0.0f;
				{
					const TCoord3 p = skel->m_CskelIndexField->vidx2coord( path.begin()->second );
					TIndexedOrigins_Vector & sp = *SPSF->vvaluep(p)->getThreeLongestPaths()[0]->m_Path.get();
					float max = 0.0f;
					for(unsigned int x=0; x<sp.size(); x++)
					{
						const float d = p.distance2( skel->m_BoundaryIndexField->vidx2coord( sp[x] ) );
						if(d > max) max = d;
					}
					coverdistance = sqrtf(max) * 0.9f;
				}

				
				TPath::iterator jt,kt;
				*g_Log << "== Junction: " << z << ", Path for compid: " << compid << ", number of balls: " << (unsigned int)JunctionId2Junctions[z].size() << "\n";
				*g_Log << "Path size: " << (unsigned int)path.size() << " voxels\n";
				*g_Log << "Cover: " << coverdistance << "\n";


				*g_Log << "Skipped measures: ";
				for(jt = path.begin(); jt != path.end(); jt++)
				{
					const float measure = Measure->vvaluex(jt->second);

					if(measure < 0.001f ) 
					{
						// Dit zou niet nodig hoeven zijn, maar anders worden soms onterechte CP's gevonden
						// op takken die niet in Geodesicness field voorkomen maar wel in C-skel
						ZeroMeasures++;
						if(ZeroMeasures > 5) 
						{
							*g_Log << "Abort because measure < 0.001f (probably invalid C-skel branch)\n";
							break;
						}
					}

					/*
					// Print debugging info
					{
						if(jt != path.begin())
						{
							kt = jt;
							kt--;
							const TCoord3 prevc = skel->m_CskelIndexField->vidx2coord( kt->second );
							const TCoord3 thisc = skel->m_CskelIndexField->vidx2coord( jt->second );
							CumulativeDistance += thisc.distance( prevc );
						}
						*g_Log << "(" << CumulativeDistance << "," << measure << "), ";
					}
					*/

					float avgmeasure = Measure->vvaluex(jt->second) ;
					{
						TPath::iterator kt = jt;
						kt++;
						avgmeasure += Measure->vvaluex(kt->second);
						kt++;
						avgmeasure += Measure->vvaluex(kt->second);
						avgmeasure /= 3.0f;
						*g_Log << "(" << avgmeasure << "), ";
					}

					if(ComponentToVoxels[compid].find(jt->second) != ComponentToVoxels[compid].end()
						&& 
						firstincomp == (std::numeric_limits<unsigned int>::max)()
						)
					{
						firstincomp = jt->second;
					}

					// If within inscribed ball of one of the junctions
					const TCoord3 p = Cskel->getIndexField()->vidx2coord(jt->second);
					bool within = false;
					float DistanceToNearestJunction = (std::numeric_limits<float>::max)();
					float RadiusOfNearestJunction= 0.0f;
					unsigned int NearestJunctionIdx = 0;
					if(false)
					{
						for(unsigned int x=0; x<JunctionId2Junctions[z].size(); x++)
						{
							const TCoord3 junccoord = Junctions->getIndexField()->vidx2coord( JunctionId2Junctions[z][x] );
							const float juncradius = SPSF->vvaluep(junccoord)->getMaxRadius();
							const float dist =  p.distance(junccoord);
							if(dist < DistanceToNearestJunction) 
							{
								DistanceToNearestJunction = dist;
								RadiusOfNearestJunction = juncradius;
								NearestJunctionIdx = JunctionId2Junctions[z][x];
							}
							if( dist <= juncradius * 1.01f )
							{
								within = true;
							}
						}
					}
					else
					{
						const TCoord3 q = Cskel->getIndexField()->vidx2coord(path.begin()->second);
						if(p.distance(q) < coverdistance) 
						{
							within = true;
						}
						
					}


					if(within)
					{
						// skip
					}
				
					else if(firstincomp != (std::numeric_limits<unsigned int>::max)()
						&& 
						//Measure->vvaluex(jt->second) >= g_Parameters.m_Segmentation_MinimumSigma ) // - 0.1f)
						avgmeasure >= g_Parameters.m_Segmentation_MinimumSigma )
					{
						*g_Log << "\nFound voxel above threshold: [" << jt->second << "], Measure: " << Measure->vvaluex((jt)->second) << ", DistanceToNearestJunction: " << DistanceToNearestJunction << ", RadiusOfNearestJunction: " << RadiusOfNearestJunction << ", NearestJunctionIdx: " << NearestJunctionIdx <<"\n";
						JunctionId2Cp[z][jt->second]  = TCriticalPointInfo(jt->second, true, false, ComponentCollapses[compid], pathstartradius, ComponentToJunction[it->first], path.begin()->second );
						break;
					}
					else if(firstincomp == (std::numeric_limits<unsigned int>::max)())
					{
						// skip if first voxel in component still not found
					}
					else if(JunctionBlocker->vvaluex(jt->second) == z)
					{
						// skip if still at junction
					}
					else if(JunctionBlocker->vvaluex(jt->second) != 0 && JunctionBlocker->vvaluex(jt->second) != z)
					{
						// if other junction reached: break
						*g_Log << "\n\t\tReached other junction, no CP\n";
						//JunctionId2Cp[z][firstincomp] = TCriticalPointInfo(firstincomp, false, true, ComponentCollapses[compid], pathstartradius );
						break;
					}

					skipped++;
				}
				*g_Log << "Skipped " << skipped << " voxels.\n";
				
//				else // Reached end of branch
				{
//					*g_Log << "Reached end, so take first: " << firstincomp << ", hasbeenabovethreshold: " << hasbeenabovethreshold << "\n";
//					JunctionId2Cp[z][firstincomp] = TCriticalPointInfo(firstincomp, false, true, ComponentCollapses[compid], pathstartradius);
				}

			}
		}
		else if(CriticalPointMethod == 2)
		{
			// Idea: kijk naar verloop van hoge sigma 's (islands)
		

			//TTypedFieldInterface<unsigned int>* DebugField = static_cast<TTypedFieldInterface<unsigned int>*>( g_Mediator.getCurrentLayerSet()->getField(TType"DebugField" ) );
			//if(!DebugField) throw string("!DebugField");

			//DebugField->clear();


			std::map<unsigned int, TPath >::iterator it;
			for(it = ComponentToPath.begin(); it != ComponentToPath.end(); it++)
			{	
				const unsigned int compid = it->first;
				TPath & path = it->second;

				const unsigned int pathsize = path.size();
				const unsigned int pathbegin = path.begin()->second;


				TPath::iterator jt,kt;
				*g_Log << "== Junction: " << z << ", Path for compid: " << compid << ", number of balls: " << (unsigned int)JunctionId2Junctions[z].size() << "\n";
				*g_Log << "Path size: " << (unsigned int)path.size() << " voxels\n";
				vector< vector<unsigned int> > Islands;
				Islands.push_back(vector<unsigned int>() );
				const float pathstartradius = SPSF->vvaluep( skel->m_CskelIndexField->vidx2coord( path.begin()->second ) )->getMaxRadius();

				for(jt = path.begin(); jt != path.end(); jt++)
				{
					const unsigned int idx = jt->second;
					const float measure = Measure->vvaluex(idx);

					if(JunctionBlocker->vvaluex(idx) != 0) continue;
					
					if(false) // Move outside ball
					{
						bool within = false;
						const TCoord3 p = Cskel->getIndexField()->vidx2coord(jt->second);
						float DistanceToNearestJunction = (std::numeric_limits<float>::max)();
						float RadiusOfNearestJunction= 0.0f;
						unsigned int NearestJunctionIdx = 0;

						for(unsigned int x=0; x<JunctionId2Junctions[z].size(); x++)
						{
							const TCoord3 junccoord = Junctions->getIndexField()->vidx2coord( JunctionId2Junctions[z][x] );
							const float juncradius = SPSF->vvaluep(junccoord)->getMaxRadius();
							const float dist =  p.distance(junccoord);
							if(dist < DistanceToNearestJunction) 
							{
								DistanceToNearestJunction = dist;
								RadiusOfNearestJunction = juncradius;
								NearestJunctionIdx = JunctionId2Junctions[z][x];
							}
							if( dist <= juncradius * 1.01f )
							{
								within = true;
								break;
							}
						}
						if(within) continue;
					}			

					float avgmeasure = 0.0f;
					bool computedavg = false;
					for(;;)
					{
						TPath::iterator kt = jt;
						avgmeasure += Measure->vvaluex(kt->second) ;
						kt++;
						if(kt == path.end()) break;
						avgmeasure += Measure->vvaluex(kt->second);
						kt++;
						if(kt == path.end()) break;
						avgmeasure += Measure->vvaluex(kt->second);
						kt++;
						if(kt == path.end()) break;
						avgmeasure += Measure->vvaluex(kt->second);
						kt++;
						if(kt == path.end()) break;
						avgmeasure += Measure->vvaluex(kt->second);
						avgmeasure /= 5.0f;
						*g_Log << "(" << idx << ", " << avgmeasure << "), ";
						computedavg = true;
						break;
					}
					if(!computedavg) continue;
									
					if(
						measure > g_Parameters.m_Segmentation_MinimumSigma
						&& avgmeasure > g_Parameters.m_Segmentation_MinimumSigma)
					{
						Islands.back().push_back(idx);
					}
					else if(Islands.back().size() != 0 && avgmeasure < g_Parameters.m_Segmentation_MinimumSigma - 0.1f)
					{
						Islands.push_back(vector<unsigned int>() );
					}
				}
				*g_Log << "\n";

				unsigned int firstbegin = (std::numeric_limits<unsigned int>::max)();
				unsigned int MinIsland = (std::numeric_limits<unsigned int>::max)();

				float minradius = (std::numeric_limits<float>::max)();
				for(unsigned int x=0; x<Islands.size(); x++)
				{
					if(Islands[x].size() == 0) continue;
					unsigned int islandbegin = Islands[x][0];
					//const float radius = SPSF->vvaluep( skel->m_CskelIndexField->vidx2coord(islandbegin) )->getMaxRadius();
					const float measure = Measure->vvaluex(islandbegin);

					// Take min Jordan length of cp
					const float RelativeRho = 1.0f - (CsField->vvaluex(pathbegin)->getMaxBorderLength() - CsField->vvaluex(islandbegin)->getMinBorderLength() ) / CsField->vvaluex(pathbegin)->getMaxBorderLength();

					// Take minimum length of island
					float islandmin = (std::numeric_limits<float>::max)();
					for(unsigned int y=0; y<Islands[x].size(); y++)
					{
						const float relrho = 1.0f - (CsField->vvaluex(pathbegin)->getMaxBorderLength() - CsField->vvaluex(Islands[x][y])->getMinBorderLength() ) / CsField->vvaluex(pathbegin)->getMaxBorderLength();
						if( relrho < islandmin ) islandmin = relrho;

					}


					*g_Log << "Island " << x << ", islandbegin: " << islandbegin << ", size: " << (unsigned int)Islands[x].size() << ", relrho: " << RelativeRho << ", islandmin: " << islandmin << ", measure: " << measure << "\n";
					
					if(firstbegin == (std::numeric_limits<unsigned int>::max)())
						firstbegin = islandbegin;

					if(RelativeRho < g_Parameters.m_Segmentation_RhoSimilar 
						&& MinIsland == (std::numeric_limits<unsigned int>::max)()
						) 
					{
						MinIsland = x;
						// Continue because we want to print info of other islands
						//break;
					}
				}


				if(MinIsland == (std::numeric_limits<unsigned int>::max)())
				{	
					if(firstbegin != (std::numeric_limits<unsigned int>::max)())
					{
						unsigned int mincp = firstbegin;
						*g_Log << "Did not find cp, take first of island [" << mincp << "]\n";
						JunctionId2Cp[z][mincp]  = TCriticalPointInfo(mincp, true, false, ComponentCollapses[compid], 0.0f, ComponentToJunction[compid], path.begin()->second  );
					}
					else
					{
						/*
						*g_Log << "Did not find cp, take first [" << mincp << "]\n";
						mincp = pathbegin;
						JunctionId2Cp[z][mincp]  = TCriticalPointInfo(mincp, true, false, ComponentCollapses[compid], 0.0f, ComponentToJunction[compid] );
						*/
					}
				}
				else
				{
					// Search on island for point with higher geodesicness
					unsigned int mincp = (std::numeric_limits<unsigned int>::max)();
					if(true)
					{
						*g_Log << "Searching for more suitable point: ";
						for(unsigned int x=0; x<Islands[MinIsland].size(); x++)
						{
							if(Measure->vvaluex(Islands[MinIsland][x]) > 0.8f)
							{
								mincp = Islands[MinIsland][x];
								*g_Log << "found.";
								break;
							}
						}
						*g_Log << "\n";
					}
					else
					{
						mincp = Islands[MinIsland][0];
					}

					*g_Log << "Found cp [" << mincp << "]\n";
					JunctionId2Cp[z][mincp]  = TCriticalPointInfo(mincp, true, false, ComponentCollapses[compid], 0.0f, ComponentToJunction[compid],path.begin()->second  );
				}

			}
		}
		else if(CriticalPointMethod == 3)
		{
			// Idea: pak eerste punt op branch zodat \theta > \theta_threshold  en trace back zolang sigma > sigma threshold
		

			//TTypedFieldInterface<unsigned int>* DebugField = static_cast<TTypedFieldInterface<unsigned int>*>( g_Mediator.getCurrentLayerSet()->getField(TType"DebugField" ) );
			//if(!DebugField) throw string("!DebugField");

			//DebugField->clear();


			std::map<unsigned int, TPath >::iterator it;
			for(it = ComponentToPath.begin(); it != ComponentToPath.end(); it++)
			{	
				const unsigned int compid = it->first;
				TPath & path = it->second;

				const unsigned int pathsize = path.size();
				const unsigned int pathbegin = path.begin()->second;


				TPath::iterator jt,kt;
				*g_Log << "== Junction: " << z << ", Path for compid: " << compid << ", number of balls: " << (unsigned int)JunctionId2Junctions[z].size() << "\n";
				*g_Log << "Path size: " << (unsigned int)path.size() << " voxels\n";
//				const float pathstartradius = SPSF->vvaluep( skel->m_CskelIndexField->vidx2coord( path.begin()->second ) )->getMaxRadius();

				*g_Log << "Searching forward for low rho: ";
				unsigned int FoundFirstMinSigma = (std::numeric_limits<unsigned int>::max)();
				for(jt = path.begin(); jt != path.end(); jt++)
				{
					const unsigned int idx = jt->second;
					if(JunctionBlocker->vvaluex(idx) != 0) continue;
					
					const float measure = Measure->vvaluex(idx);
					
					// Compute avgmeasure
					/*
					float avgmeasure = 0.0f;
					{
						TPath::iterator lt = jt;
						const unsigned int WindowSize = 5;
						for(unsigned int i=0; i<WindowSize; i++)
						{
							if(lt == path.end()) break;
							avgmeasure += Measure->vvaluex(lt->second);
							lt++;
						}
						avgmeasure /= (float)i;
					}
					// */


					// Take min Jordan length of cp
					const float RelativeRho = 1.0f - (CsField->vvaluex(pathbegin)->getMaxBorderLength() - CsField->vvaluex(idx)->getMaxBorderLength() ) / CsField->vvaluex(pathbegin)->getMaxBorderLength();

					*g_Log << "(" << idx << ", rho:" << RelativeRho << ", sigma: " << measure << ")"; // ", avgsig: " << avgmeasure << "), ";
					
					if(	measure > 0.75f 
						//&& avgmeasure > 0.7f
						&& FoundFirstMinSigma == (std::numeric_limits<unsigned int>::max)()
						)
					{
						FoundFirstMinSigma = idx;
					}

					if(RelativeRho < g_Parameters.m_Segmentation_RhoSimilar
						//&& measure > g_Parameters.m_Segmentation_MinimumSigma
						&& measure > 0.75f
					)
					{
						*g_Log << " found. ";
						kt = jt;
						break;
					}
				}
				*g_Log << "\n";
				
				
				if(jt == path.end())
				{
					if(FoundFirstMinSigma != (std::numeric_limits<unsigned int>::max)())
					{
						*g_Log << "Reached end of path\n";
						unsigned int mincp = FoundFirstMinSigma;
						*g_Log << "Take cp [" << mincp << "]\n";
						JunctionId2Cp[z][mincp]  = TCriticalPointInfo(mincp, true, false, CskelRho->vvaluex(mincp), 0.0f, ComponentToJunction[compid], path.begin()->second );
					}
				}
				else
				{
					*g_Log << "Searching backward for low sigma: ";
					for(jt = kt; jt != path.begin(); jt--)
					{
						const unsigned int idx = jt->second;
						// if(JunctionBlocker->vvaluex(idx) != 0) continue;

						const float measure = Measure->vvaluex(idx);

						// Compute avgmeasure
						float avgmeasure = 0.0f;
						{
							TPath::iterator lt = jt;
							const unsigned int WindowSize = 5;
							unsigned int i;
							for(i=0; i<WindowSize; i++)
							{
								if(lt == path.begin()) break;
								avgmeasure += Measure->vvaluex(lt->second);
								lt--;
							}
							avgmeasure /= (float)i;
						}

						*g_Log << "(" << idx << ", sigma:" << measure << ", avgsigma: " << avgmeasure << "), ";
						if(measure < g_Parameters.m_Segmentation_MinimumSigma)
						//if(	measure < 0.8f && avgmeasure < 0.8f)
						{
							jt++;
							*g_Log << "\n";
							const unsigned int mincp = jt->second;
							*g_Log << "Found cp [" << mincp << "]\n";
							JunctionId2Cp[z][mincp]  = TCriticalPointInfo(mincp, true, false, CskelRho->vvaluex(mincp), 0.0f, ComponentToJunction[compid], path.begin()->second );

							break;
						}
					}
					if(jt == path.begin())
					{
						*g_Log << "Reached begin of path\n";
						const unsigned int mincp = path.begin()->second;
						*g_Log << "Found cp [" << mincp << "]\n";
						JunctionId2Cp[z][mincp]  = TCriticalPointInfo(mincp, true, false, CskelRho->vvaluex(mincp), 0.0f, ComponentToJunction[compid], path.begin()->second  );

					}
					*g_Log << "\n";
				}

			}


		}

	} // for z

	// Delete junctions that did not lead to CP 
	for(x=0; x<Junctions->getMaxIndex(); x++)
	{
		if(Junctions->vvaluex(x) == 0) continue;
		if(JunctionId2Cp[ Junctions->vvaluex(x) ].size() == 0) 
			Junctions->wvaluex(x) = 0;
	}	

	TTypedFieldInterface<unsigned char>* FinalCp = 0;
	// Make field of critical points
	{
		TTypedFieldInterface<unsigned char>* Cp = static_cast<TTypedFieldInterface<unsigned char>*>( g_Mediator.getCurrentLayerSet()->newField(TType::TYPE_UCHAR, skel->m_CskelIndexField, "Candidate critical points" ) );
		FinalCp = Cp;
		Cp->clear();
		TJunctionId2Cp::iterator jt;
		for(jt = JunctionId2Cp.begin(); jt != JunctionId2Cp.end(); jt++)
		{
			std::map<unsigned int, TCriticalPointInfo>::iterator it;
			for(it = jt->second.begin(); it != jt->second.end(); it++)
			{
				if(it->second.m_Valid)
				{
					Cp->wvaluex( it->first ) = jt->first;
				}
			}
		}
		Cp->getLayer()->onFieldChanged();
		Cp->getLayer()->m_RenderPrimitiveType = 1;
	}


	{
		// Print new junctionid2cp information
		*g_Log << "---------------------\n";
		TJunctionId2Cp::iterator it;
		for(it = JunctionId2Cp.begin(); it != JunctionId2Cp.end(); it++)
		{
			*g_Log << "Junction " << it->first << ", CP: ";
			std::map<unsigned int, TCriticalPointInfo>::iterator jt;
			for(jt = it->second.begin(); jt != it->second.end(); jt++)
			{
				*g_Log << jt->second.m_Index << " (" << jt->second.m_Valid << "), ";
			}
			*g_Log << "\n";
			*g_Log << "rhomin: " << JunctionId2RhoMin[jt->second.m_Index] << "\n";
			*g_Log << "rhomax: " << JunctionId2RhoMax[jt->second.m_Index] << "\n";


		//	const float ratio = (--(--CpSortedOnRho.end()))->first / (--CpSortedOnRho.end())->first;
		//	*g_Log << "Ratio highest and one-before highest: " << ratio << "\n";
		}
	}

	*g_Log << "Filtering candidate critical-points.\n";
	// Filter critical points
	{
		TTypedFieldInterface<unsigned char>* Cp = static_cast<TTypedFieldInterface<unsigned char>*>( g_Mediator.getCurrentLayerSet()->newField(TType::TYPE_UCHAR, skel->m_CskelIndexField, "Final critical points" ) );
		Cp->clear();

		if(true)
		{
			*g_Log << "New method\n";
			FinalCp = Cp;
			TJunctionId2Cp::iterator jt;
			for(jt = JunctionId2Cp.begin(); jt != JunctionId2Cp.end(); jt++)
			{
				set<unsigned int> RemoveThese;

				const unsigned int juncid = jt->first;
				map<unsigned int, TCriticalPointInfo> & CpInfo = jt->second;
				std::multimap<float, unsigned int, std::greater<float> > ValidsSorted;

				// Compute avg Jordan-curve length of junctions with this ID
				float junclength = 0.0f;
				{
					*g_Log << "Computing avglength. ";
					float minlength = (std::numeric_limits<float>::max)();
					float maxlength = 0.0f;
					for(unsigned int x=0; x<JunctionId2Junctions[juncid].size(); x++)
					{
						const float length = CsField->vvaluex(JunctionId2Junctions[juncid][x] )->getMaxBorderLength();
						if(length < minlength) minlength = length;
						if(length > maxlength) maxlength = length;
					}
					junclength  = (minlength + maxlength)/2.0f;
					//*g_Log << ": sum i " << avglength << "\n";
				}

				const float juncrho = JunctionId2RhoMax[jt->first];
				*g_Log << "\nJunction " << jt->first << ", juncrhomin: " << JunctionId2RhoMin[jt->first] << ", juncrhomax: " << JunctionId2RhoMax[jt->first] << ", juncrho: " << juncrho << ", junclength: " << junclength << "\n"; //, junceft: " << junceft << "\n";
				
				if(false)
				{
					// Determine size of junction component
					// Do a flat segmentation using the critical points
					float junccompsize = 0.0f;
 
					TAction_VolumeSegmentationFlat Action;
					//Action.m_Name = "Flat segmentation for determining junction component ";
					Action.m_Name = wxString::Format("Flat segmentation for determining junction component %i", juncid).c_str();
					Action.m_FillZeros = false;
					Action.m_DilationDistance = 4.0f;

					std::map<unsigned int, TCriticalPointInfo>::iterator it;
					Junctions->getLayer()->m_Selection->clear();
					for(it = CpInfo.begin(); it != CpInfo.end(); it++)
					{
						const TCoord3 p = Junctions->getIndexField()->vidx2coord( it->second.m_Index );
						Junctions->getLayer()->m_Selection->select( p );
					}
					g_Mediator.setCurrentLayer( Junctions->getLayer() );
					Action.perform();

					// Find segment with junction
					const TShortestPathSet * sps = SPSF->vvaluep( Junctions->getIndexField()->vidx2coord(jt->first) );
					const unsigned int idx = sps->m_Eft->at(0);
					*g_Log << "Idx: " << idx << "\n";
					float mincompsize = (std::numeric_limits<float>::max)();
					for(unsigned int x=0; x<Action.m_Sizes.size(); x++)
					{
						const float size = Action.m_Sizes[x].size();
						if(size < 10) continue;
						*g_Log << "Component " << x << ", size: " << size << ",\t";
						if(size < mincompsize) mincompsize = size;
						
						if(Action.m_Sizes[x].find(idx) != Action.m_Sizes[x].end())
						{
							junccompsize = size;
						} 
					}

					*g_Log << "Size of junction component: " << junccompsize << ", mincompsize: " << mincompsize << "\n";
				}
				//", sumofrhorel: " << sumofrhorel << "\n";

				// First do a pass collection information
				set<unsigned int> ValidCps;

				std::map<unsigned int, TCriticalPointInfo>::iterator it;
				for(it = CpInfo.begin(); it != CpInfo.end(); it++)
				{
					TCriticalPointInfo & cpinfo = it->second;
					const unsigned int idx = it->first;
					const bool valid = it->second.m_Valid;
	//				const float collapse = it->second.m_Collapse;

					const TCoord3 p = skel->m_CskelIndexField->vidx2coord( it->first ) ;

					
					
					if(false)
					{
						// Using \rho_\skel:
						cpinfo.m_RelativeRho = 1.0f - (juncrho - cpinfo.m_Rho) / juncrho; 
					}
					else
					{
						// Using |\Gamma| (length counted in voxels):

						//const float jordanlength = (CsField->vvaluex(cpinfo.m_Index)->getMinBorderLength() + CsField->vvaluex(cpinfo.m_Index)->getMaxBorderLength() ) / 2.0f;
						// Take min borderlength of cp
						//cpinfo.m_RelativeRho = 1.0f - (avglength - jordanlength ) / avglength;

						{
							const float min = CsField->vvaluex(cpinfo.m_Index)->getMinBorderLength();
							const float max = CsField->vvaluex(cpinfo.m_Index)->getMaxBorderLength();
							//const float length = fabs(min - avglength) < fabs(max - avglength) ? min : max;
							const float length = (min+max)/2.0f;
							cpinfo.m_Rho = length;
						}

						float pathbeginrho = 0.0f;
						{
							const float min = CsField->vvaluex(cpinfo.m_PathStartIndex)->getMinBorderLength();
							const float max = CsField->vvaluex(cpinfo.m_PathStartIndex)->getMaxBorderLength();
							//const float length = fabs(min - avglength) < fabs(max - avglength) ? min : max;
							const float length = (min+max)/2.0f;
							pathbeginrho = length;
						}
						
						// Take max borderlength of cp
						cpinfo.m_RelativeRho = 1.0f - (junclength  - cpinfo.m_Rho ) / junclength ;

					}


					//cpinfo.m_RelativeRho = 1.0f - fabs(cpinfo.m_Rho - cpinfo.m_PathStartRho) / cpinfo.m_PathStartRho;
					cpinfo.m_Radius = SPSF->vvaluep(p)->getRadius();
					cpinfo.m_RelativeRadius = 1.0f - (fabs(cpinfo.m_Radius - cpinfo.m_PathStartRadius) / cpinfo.m_PathStartRadius);



					// Always remove cp's with > 1.0 relative rho
					if(cpinfo.m_RelativeRho >= 1.0f)
					{
						cpinfo.m_RelativeRho = 1.0f; // Dit clampen is nodig om max1-max2 te limiteren.
						//RemoveThese.insert(idx);
					}

					if(cpinfo.m_Valid)
					{
						ValidCps.insert( idx );
						ValidsSorted.insert( pair<float,unsigned int>(cpinfo.m_RelativeRho,idx) );
						if(!cpinfo.m_Default) 
						{
							*g_Log << wxString::Format("CP: %i, valid: %i, default: %i, collapse: %.4f, rho: %.2f, rhorel: %.2f, radius: %.2f, radiusrel: %.2f, pathstartradius: %.2f\n", idx, cpinfo.m_Valid, cpinfo.m_Default, cpinfo.m_Collapse, cpinfo.m_Rho, cpinfo.m_RelativeRho, cpinfo.m_Radius, cpinfo.m_RelativeRadius, cpinfo.m_PathStartRadius).c_str();
						}
					}
		
					if(!valid)
					{
						RemoveThese.insert(idx);
					}
				}

				float max1 = 0.0f;
				float max2 = 0.0f;
				float max3 = 0.0f;
				unsigned int max1idx = (std::numeric_limits<unsigned int>::max)();
				unsigned int max2idx = (std::numeric_limits<unsigned int>::max)();
				unsigned int max3idx = (std::numeric_limits<unsigned int>::max)();
				if(ValidsSorted.size() == 1) 
				{
					*g_Log << "*** Please note: ValidsSorted.size() < 3.\n";
					std::multimap<float, unsigned int, std::greater<float> >::iterator kt = ValidsSorted.begin();
					max1 = kt->first;
					max1idx = kt->second;
					max2 = -1.0f;
					max3 = -1.0f;
				}
				else if(ValidsSorted.size() == 2)
				{
					std::multimap<float, unsigned int, std::greater<float> >::iterator kt = ValidsSorted.begin();
					max1 = kt->first;
					max1idx = kt->second;
					kt++;
					max2 = kt->first;
					max2idx = kt->second;
					max3 = -1.0f;
				}
				else 
				{
					std::multimap<float, unsigned int, std::greater<float> >::iterator kt = ValidsSorted.begin();
					max1 = kt->first;
					max1idx = kt->second;
					kt++;
					max2 = kt->first;
					max2idx = kt->second;
					kt++;
					max3 = kt->first;
					max3idx = kt->second;
				}
				float diff23 = max2 - max3;
				float diff12 = max1 - max2;

				*g_Log << "All cps: " << (unsigned int)CpInfo.size() << ", valid cps: " << (unsigned int)ValidCps.size() << "\n";
				*g_Log << "max1: " << max1 << ", max2: " << max2 << ", max3: " << max3 << ", ";
				*g_Log << "diff12: " << diff12 << ", diff23: " << diff23 << "\n";

				// Examples:
				// (0.99, 0.72, 0.4), should be removed: 0.99 (both above threshold but diff12 is high)
				// (0.8, 0.75, 0.4), should be removed: 0.8, 0.75
				// (0.77, 0.75, 0.72), should be removed: none

				bool close12 = diff12 < 0.2f;
				bool close23 = diff23 < 0.2f;
				//*g_Log << "close12: " << close12 << ", close23: " << close23 << "\n";

				unsigned int removed = 0;
				const float Threshold = 0.6f; // g_Parameters.m_Segmentation_RhoSimilar;
				//if(kt3->first == 0.0f)
				//{
				//	*g_Log << "max3 = 0.0f\n";
				//	if(kt1->first < Threshold) { RemoveThese.insert(kt1->second); removed++; }
				//	if(kt2->first < Threshold) { RemoveThese.insert(kt2->second); removed++; }
				//}
				//else 
				if(max1 > Threshold)
				{
					*g_Log << "max1 > Threshold\n";
					
					if(close12 && !close23)
					{
						*g_Log << "close12 && !close23\n";
						*g_Log << "Remove both\n";
						RemoveThese.insert(max1idx);
						RemoveThese.insert(max2idx);
						removed = 2;
					}
					else if(close12 && close23)
					{
						*g_Log << "close12 && close23\n";
						*g_Log << "Remove none\n";

						// /*
						// Werkt niet bij Ant: zijn body is groot vergeleken met de rest
						std::multimap<float, unsigned int, std::greater<float> > SortedOnRho;
						for(it = CpInfo.begin(); it != CpInfo.end(); it++)
						{
							//if(it->first > g_Parameters.m_Segmentation_RhoSimilar)
							//{
							SortedOnRho.insert( pair<float,unsigned int>(it->second.m_Collapse, it->second.m_Index) );
							//}
						}
						if( SortedOnRho.size() >= 3 )
						{
							*g_Log << "Removing point with maximum rho\n";
							std::multimap<float, unsigned int, std::greater<float> >::iterator kt1 = SortedOnRho.begin();
							std::multimap<float, unsigned int, std::greater<float> >::iterator kt2 = kt1; kt2++;
							const float ratio = kt2->first / kt1->first;
							*g_Log << "Looking for largest rho_cskel: ratio: " << ratio << "\n";
							if(ratio < 0.6f)
							{
								*g_Log << "Removing candidate critical point with largest rho: " << kt1->second << " \n";
								RemoveThese.insert(kt1->second);
							}
						}
						// */

					}
					else if(!close12 && close23)
					{
						*g_Log << "!close12 && close23\n";
						*g_Log << "Remove one\n";
						RemoveThese.insert(max1idx);
						removed = 1;
					}
					else if(!close12 && !close23)
					{
						*g_Log << "!close12 && !close23\n";
						*g_Log << "Remove one\n";
						RemoveThese.insert(max1idx);
						removed = 1;
					}

				}
				else
				{
					*g_Log << "kt1->first < Threshold \n";
				}
				/*
				else if(kt1->first > Threshold)
				{
					*g_Log << "Remove one, diff12 > 0.2f\n";
					*g_Log << "diff12: " << diff12 << "\n";
					RemoveThese.insert(kt1->second);
				}
				else if(
					kt2->first > g_Parameters.m_Segmentation_RhoSimilar
					&& kt1->first > g_Parameters.m_Segmentation_RhoSimilar
					&& ValidsSorted.size() >= 3
					)
				{
					*g_Log << "Nothing removed \n";

					std::multimap<float, unsigned int, std::greater<float> > SortedOnRho;
					for(it = CpInfo.begin(); it != CpInfo.end(); it++)
					{
						//if(it->first > g_Parameters.m_Segmentation_RhoSimilar)
						//{
						SortedOnRho.insert( pair<float,unsigned int>(it->second.m_Collapse, it->second.m_Index) );
						//}
					}
					if( SortedOnRho.size() >= 3 )
					{
						std::multimap<float, unsigned int, std::greater<float> >::iterator kt1 = SortedOnRho.begin();
						std::multimap<float, unsigned int, std::greater<float> >::iterator kt2 = kt1; kt2++;
						const float ratio = kt2->first / kt1->first;
						*g_Log << "Looking for largest rho_cskel: ratio: " << ratio << "\n";
						if(ratio < 0.6f)
						{
							*g_Log << "Removing candidate critical point with largest rho: " << kt1->second << " \n";
							RemoveThese.insert(kt1->second);
						}
					}
				}
				// */

				std::set<unsigned int>::iterator kt;
				*g_Log << "RemoveThese: ";
				for(kt = RemoveThese.begin(); kt != RemoveThese.end(); kt++)
				{
					*g_Log << *kt << ", ";
				}
				*g_Log << "\n";
				for(it = jt->second.begin(); it != jt->second.end(); it++)
				{
					const unsigned int idx = it->first;
					if(RemoveThese.find( idx ) == RemoveThese.end())
					{
						Cp->wvaluex( idx ) = juncid;
					}
				}


			}
		}
		else
		{
			*g_Log << " Old method\n";
			FinalCp = Cp;
			TJunctionId2Cp::iterator jt;
			for(jt = JunctionId2Cp.begin(); jt != JunctionId2Cp.end(); jt++)
			{
				set<unsigned int> RemoveThese;

				const unsigned int juncid = jt->first;
				map<unsigned int, TCriticalPointInfo> & CpInfo = jt->second;
				multimap<float, unsigned int> ValidsSorted;

				const float juncrho = JunctionId2RhoMax[jt->first];
				//const float junceft = JunctionId2Eft[jt->first];
				*g_Log << "\nJunction " << jt->first << ", juncrhomin: " << JunctionId2RhoMin[jt->first] << ", juncrhomax: " << JunctionId2RhoMax[jt->first] << ", juncrho: " << juncrho << "\n"; //, junceft: " << junceft << "\n";
				//", sumofrhorel: " << sumofrhorel << "\n";

				// First do a pass collection information
				set<unsigned int> ValidCps;

				std::map<unsigned int, TCriticalPointInfo>::iterator it;
				for(it = CpInfo.begin(); it != CpInfo.end(); it++)
				{
					TCriticalPointInfo & cpinfo = it->second;
					const unsigned int idx = it->first;
					const bool valid = it->second.m_Valid;
	//				const float collapse = it->second.m_Collapse;

					const TCoord3 p = skel->m_CskelIndexField->vidx2coord( it->first ) ;

					cpinfo.m_Rho = SPSF->vvaluep(p)->getMaxLength();
					cpinfo.m_RelativeRho = 1.0f - (juncrho - cpinfo.m_Rho) / juncrho; 
					//cpinfo.m_RelativeRho = 1.0f - fabs(cpinfo.m_Rho - cpinfo.m_PathStartRho) / cpinfo.m_PathStartRho;
					cpinfo.m_Radius = SPSF->vvaluep(p)->getRadius();
					//cpinfo.m_RelativeRadius = 1.0f - (fabs(cpinfo.m_Radius - junceft) / junceft);

					if(cpinfo.m_Valid)
					{
						ValidCps.insert( idx );
						ValidsSorted.insert( pair<float,unsigned int>(cpinfo.m_Rho,idx) );
					}
		
					if(!valid)
					{
						RemoveThese.insert(idx);
					}
					if(cpinfo.m_Default)
					{
						RemoveThese.insert(idx);
					}
				}


				multimap<float, unsigned int> & sorted = ValidsSorted;
				if(sorted.size() == 0) { continue; }

				// Print info
				bool AllAboveThreshold = true;
				bool AllBelowThreshold = true;
				set<unsigned int> SimilarToJunction;
				for(it = CpInfo.begin(); it != CpInfo.end(); it++)
				{
					TCriticalPointInfo & cpinfo = it->second;
					const unsigned int idx = it->first;

					const float combined = cpinfo.m_RelativeRho + cpinfo.m_RelativeRadius;
				
					if(!cpinfo.m_Default) 
					{
						*g_Log << wxString::Format("CP: %i, valid: %i, default: %i, collapse: %.4f, rho: %.2f, rhorel: %.2f, rhorel^2: %.2f, radius: %.2f, radiusrel: %.2f, pathstartradius: %.2f, combined: %.2f", idx, cpinfo.m_Valid, cpinfo.m_Default, cpinfo.m_Collapse, cpinfo.m_Rho, cpinfo.m_RelativeRho, cpinfo.m_RelativeRho*cpinfo.m_RelativeRho, cpinfo.m_Radius, cpinfo.m_RelativeRadius, cpinfo.m_PathStartRadius, combined).c_str();
					}
					
					const float value = cpinfo.m_RelativeRho;
					//const float value = cpinfo.m_RelativeRho*cpinfo.m_RelativeRho; // Use squared rho

					if(cpinfo.m_Valid && value > g_Parameters.m_Segmentation_RhoSimilar) 
					{
						SimilarToJunction.insert(idx);
						*g_Log << " *** remove because similar (" << value << " > " << g_Parameters.m_Segmentation_RhoSimilar << ")";
					}		
					else if(!cpinfo.m_Valid) 
					{
						if(!cpinfo.m_Default)
							*g_Log << " *** remove because invalid  ";
					}

					*g_Log << "\n";		
				}

				if(  SimilarToJunction.size() >= 3 // && ValidCps.size() == SimilarToJunction.size() 
					)
				{
					/*
					*g_Log << "All valid Cp's removed. Put junction as CP\n";
					for(it = CpInfo.begin(); it != CpInfo.end(); it++)
					{
						RemoveThese.insert(it->second.m_Index);
					}
					Cp->wvaluex() = z;
					*/
					*g_Log << "All valid Cp's removed. Remove max collapse\n";
					TCriticalPointInfo * max = &CpInfo.begin()->second;
					for(it = CpInfo.begin(); it != CpInfo.end(); it++)
					{
						if( it->second.m_Collapse > max->m_Collapse )
						{
							max = &it->second;
						}
					}
					RemoveThese.insert(max->m_Index);
				}
				else
				{
					for(std::set<unsigned int>::iterator it=SimilarToJunction.begin(); it != SimilarToJunction.end(); it++)
						RemoveThese.insert(*it);
				}

		

				std::set<unsigned int>::iterator kt;
				*g_Log << "RemoveThese: ";
				for(kt = RemoveThese.begin(); kt != RemoveThese.end(); kt++)
				{
					*g_Log << *kt << ", ";
				}
				*g_Log << "\n";
				for(it = jt->second.begin(); it != jt->second.end(); it++)
				{
					const unsigned int idx = it->first;
					if(RemoveThese.find( idx ) == RemoveThese.end())
					{
						Cp->wvaluex( idx ) = juncid;
					}
				}

			}
		}

		Cp->getLayer()->onFieldChanged();
		Cp->getLayer()->m_RenderPrimitiveType = 1;
	}


	// Remove critical points that produce similar part cuts
	if(g_Parameters.m_RemoveSimilarCriticalPoints)	
	{
		*g_Log << "=== Removing critical points that have basically the same part cut\n";
		TDeltaOmega * DO = skel->Omega->m_DeltaOmega.get();
		TTypedFieldInterface<unsigned int>* TempField = static_cast<TTypedFieldInterface<unsigned int>*>( g_Mediator.getCurrentLayerSet()->newField(TType::TYPE_UINT, skel->m_BoundaryIndexField, "TempField2" ) );
		TempField->clear();

		TTypedFieldInterface<unsigned char>* Cp = static_cast<TTypedFieldInterface<unsigned char>*>( g_Mediator.getCurrentLayerSet()->getField("Final critical points" ) );
		for(x=0; x<Cp->getMaxIndex(); x++)
		{
			if(Cp->vvaluex(x) == 0) continue;
			
			const TCoord3 p = Cp->getIndexField()->vidx2coord(x);
			const TShortestPathSet * sps = SPSF->vvaluep(p);
			if(sps == 0) continue;

			TempField->getLayer()->onFieldChanged();

			TDeltaOmega::TDistancePropagationParameters par;
			par.m_TargetFilter = TempField->getLayer()->m_Filter.get();
			par.m_StartVector = sps->m_PathSet.get();
			par.m_StopDistance = 4.0f;
			set<unsigned int> Targets;
			par.m_OutputTargets = &Targets;

			DO->distancePropagation(par);
			set<unsigned int> TargetValues;
			std::set<unsigned int>::iterator it;
			for(it = Targets.begin(); it != Targets.end(); it++)
			{
				if( TempField->vvaluex(*it) != 0 
					&& TempField->vvaluex(*it) != Cp->vvaluex(x)
					)
				{
					TargetValues.insert( TempField->vvaluex(*it) );	
				}
			}
			if(TargetValues.size() != 0)
			{
				*g_Log << "Removed CP value: " << Cp->vvaluex(x) << " (idx:" << x << ")\n";
				Cp->wvaluex(x) = 0;
			}
			else
			{
				// Write this path to tempfield
				for(unsigned int a=0; a<sps->m_PathSet->size(); a++)
				{
					unsigned int idx = (*sps->m_PathSet)[a];
					TempField->wvaluex(idx) = Cp->vvaluex(x);
				}
			}
		}
	}
//	*/



	// Final segmentation
	{
		TAction_VolumeSegmentationFlat2 Action;
		Action.m_Name = "Final segmentation";
		Action.m_FillZeros = true;
		Action.m_DilationDistance = 2.0f;
		Action.m_ForceTwoComponentsPerCp = g_Parameters.m_ForceTwoComponentsPerCriticalPoint;
		Action.m_MakeMinColor = true;
//		Action.m_RemoveSmallComponents = 0.0001f;
		
		// Insert all junctions into except these
		for(x=0; x<Junctions->getMaxIndex(); x++)
		{
			if(Junctions->vvaluex(x) == 0) continue;
			Action.m_ForceTwoComponentsPerCpExceptThese.insert( Junctions->getIndexField()->vidx2coord(x) );
		}

		FinalCp->getLayer()->m_Selection->clear();
		for(x=0; x<FinalCp->getMaxIndex(); x++)
		{
			if(FinalCp->vvaluex(x) == 0) continue;
			const TCoord3 p = FinalCp->getIndexField()->vidx2coord(x);
			FinalCp->getLayer()->m_Selection->select( p );
		}
		g_Mediator.setCurrentLayer( FinalCp->getLayer() );
		Action.perform();

		// Write information file
		{
			const string filename = g_Mediator.getSkeletonizer()->m_InputFilename;
			if(filename != "") 
			{
				wxString BaseFileName = filename.c_str();
				wxString Fullname = wxFileName( BaseFileName ).GetFullName();
				BaseFileName.Replace(".cskel.gz",""); 
				BaseFileName.Replace(".cskel",""); 
				BaseFileName.Append("_segmresult.pl");
				if( string(BaseFileName.c_str()) != filename)
				{
					std::ofstream f(BaseFileName.ToStdString());
					f << "push @results, { filename=>qq|" << Fullname.c_str() << "|, segmentcount=> " << Action.m_OutputSegmentCount << " };";
					f << "\n";
					f.close();
				}
			}
			else
			{
				*g_Log << "Warning: m_InputFilename == ''\n";
			}
		}
	

	}
}


// ------------------------------------------------------------------------------------------

bool TAction_NewJunctionDetection::isAvailable()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	if(!skel) return false;
	return true;
}


void TAction_NewJunctionDetection::perform_main()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	TDeltaOmega * DO = skel->Omega->m_DeltaOmega.get();
	const TTypedFieldInterface<TShortestPathSet*> * SPSF = skel->m_ForegroundSpSetField.get();

//	TTypedFieldInterface<float>* Output = static_cast<TTypedFieldInterface<float>*>( g_Mediator.getCurrentLayerSet()->newField( TType::TYPE_FLOAT, skel->Omega->m_BoundaryIndexField, "Output" ) );
//	Output->clear();

	TTypedFieldInterface<unsigned char>* Cskel = static_cast<TTypedFieldInterface<unsigned char>*>( g_Mediator.getCurrentLayerSet()->getField( "C-skel detector" ) );
	if(!Cskel) throw string("!Cskel");

	TTypedFieldInterface<TComponentSet*>* OutputCs = static_cast<TTypedFieldInterface<TComponentSet*>*>( g_Mediator.getCurrentLayerSet()->newField( TType::TYPE_COMPONENTSET, skel->m_CskelIndexField, "Component Sets") );
	OutputCs->clear();

//	TTypedFieldInterface<unsigned int>* NewEftField = static_cast<TTypedFieldInterface<unsigned int>*>( g_Mediator.getCurrentLayerSet()->newField( TType::TYPE_UINT, skel->Omega->m_BoundaryIndexField, "NewEftField" ) );
//	NewEftField->clear();

	TTypedFieldInterface<unsigned char>* Junctions = static_cast<TTypedFieldInterface<unsigned char>*>( g_Mediator.getCurrentLayerSet()->newField(TType::TYPE_UCHAR, skel->m_CskelIndexField, "Junctions" ) );
	Junctions->clear();

	TTypedFieldInterface<unsigned int>* DebugField = static_cast<TTypedFieldInterface<unsigned int>*>( g_Mediator.getCurrentLayerSet()->newField(TType::TYPE_UINT, skel->m_BoundaryIndexField, "DebugField" ) );
	DebugField->clear();


	unsigned int x,y;
	TIndexedOrigins_Vector DilatedShortestPaths;
	TIndexedOrigins_Vector Boundary;
	TIndexedOrigins_Vector DilatedPathSet;

	unsigned int FreeId = 1;
	for(x=0; x<Cskel->getMaxIndex(); x++)
	{
		const TCoord3 p = Cskel->getIndexField()->vidx2coord(x);

		const TShortestPathSet * sps = SPSF->vvaluep(p);
		if(sps == 0) continue;

		DilatedShortestPaths.clear();
		DO->dilate(
			g_Parameters.m_DilationDistance
			, sps->m_PathSet.get()
			, &DilatedShortestPaths
			, true // reset
			);

		if(OutputCs->vvaluep(p) != 0) 
		{
			throw string("OutputCs->vvaluex(x) != 0");
		}
		TComponentSet * cs = new TComponentSet(p);
		skel->Omega->m_DeltaOmega->computeComponentSetUsingCells2( // Dit was eerst
		//skel->Omega->m_DeltaOmega->computeComponentSetUsingCells( // Waarom deze niet gebruiken?
			&DilatedShortestPaths
			, cs
			//, false // do not fill shortest path set
			);
		if(OutputCs->vvaluep(p)) { throw string("OutputCs already assigned"); }
		OutputCs->wvaluep(p) = cs;

		cs->simplify(p);
		unsigned int BorderCount = 0;
		for(y=0; y<cs->m_Borders.size(); y++) 
			if(cs->m_Borders[y]->size() > 5) BorderCount++;

		if(Cskel->vvaluex(x) >= 3 && BorderCount >= 3)
		{
			Junctions->wvaluep(p) = FreeId++; // cs->m_Borders.size();
		}
		cs->projectEft(sps, DebugField);

		// Enable for computing border lengths, important for alternative geodesicness measure (which doesn't work)
		 cs->computeBorderLengths(DebugField);
	}

	/*
	// Compute eccentricity
	TTypedFieldInterface<float>* Ecc = static_cast<TTypedFieldInterface<float>*>( g_Mediator.getCurrentLayerSet()->newField(TType::TYPE_FLOAT, skel->m_CskelIndexField, "Eccentricity" ) );
	Ecc->clear();
	for(x=0; x<Cskel->getMaxIndex(); x++)
	{
		const TCoord3 p = Cskel->getIndexField()->vidx2coord(x);
		// Get minimum eccentricity 
		const TComponentSet * cs = OutputCs->vvaluex(x);
		//float minecc = (std::numeric_limits<float>::max)();
		float maxecc = 0.0f;
		for(y=0; y<cs->m_BorderLengths.size(); y++)
		{
			const float diameter = cs->m_BorderDiameters[y];
			const float length = cs->m_BorderLengths[y];
			const float ecc = length / (diameter*PI);
			if(ecc > maxecc) maxecc = ecc;
		}
		Ecc->wvaluex(x) = maxecc;
	}
	Ecc->getLayer()->onFieldChanged();
	Ecc->getLayer()->m_ColorMap->m_Lower = 1.0f;
	Ecc->getLayer()->m_ColorMap->m_Upper = 2.0f;
	*/

	/*
		// Make copy of old junctions
		TTypedFieldInterface<unsigned char>* OldJunctions = static_cast<TTypedFieldInterface<unsigned char>*>( g_Mediator.getCurrentLayerSet()->newField(TType::TYPE_UCHAR, skel->m_CskelIndexField, "Old Junctions" ) );
		if(!OldJunctions) throw string("!OldJunctions");
		for(x=0; x<Junctions->getMaxIndex(); x++)
		{
			OldJunctions->wvaluex(x) = Junctions->vvaluex(x);
		}
		OldJunctions->getLayer()->onFieldChanged();
		OldJunctions->getLayer()->m_RenderPrimitiveType = 1;
	*/

	if(true)
	{
		TAction_MergeJunctions Action;
		Action.perform();
	}

//	Output->getLayer()->onFieldChanged();
	Junctions->getLayer()->onFieldChanged();
	Junctions->getLayer()->m_RenderPrimitiveType = 1;
	Junctions->getLayer()->setCheckedForRendering(false);

//	NewEftField->getLayer()->onFieldChanged();
	OutputCs->getLayer()->onFieldChanged();
	
}


// ------------------------------------------------------------------------------------------

bool TAction_MergeJunctions::isAvailable()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	if(!skel) return false;
	return g_Mediator.getCurrentLayerSet()->exists("Junctions");
}

void TAction_MergeJunctions::updateComponentSets(TTypedFieldInterface<unsigned char> * Junctions)
{
	TTypedFieldInterface<TComponentSet*>* CsField = static_cast<TTypedFieldInterface<TComponentSet*>*>( g_Mediator.getCurrentLayerSet()->getField( "Component Sets" ) );
	if(!CsField) throw string("!CsField");

	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	TDeltaOmega * DO = skel->Omega->m_DeltaOmega.get();

	TTypedFieldInterface<unsigned int>* DebugField = static_cast<TTypedFieldInterface<unsigned int>*>( g_Mediator.getCurrentLayerSet()->newField(TType::TYPE_UINT, skel->m_BoundaryIndexField, "Helpfield for updateComponentSets" ) );
	DebugField->clear();

	const TTypedFieldInterface<TShortestPathSet*> * SPSF = skel->m_ForegroundSpSetField.get();
	if(!SPSF) throw string("!SPSF");

	unsigned int x;
	unsigned int maxid = 0;
	for(x=0; x<Junctions->getMaxIndex(); x++)
	{
		if( Junctions->vvaluex(x) == 0 ) continue;
		if(Junctions->wvaluex(x) > maxid) maxid = Junctions->wvaluex(x);
	}

	// Update component sets
	for(unsigned int z=1; z<=maxid; z++)
	{
		TIndexedOrigins_Vector MergedPathSet;
		MergedPathSet.clear();
		unsigned int x;
		for(x=0; x<Junctions->getMaxIndex(); x++)
		{
			if(Junctions->vvaluex(x) != z) continue;
			const TCoord3 p = Junctions->getIndexField()->vidx2coord(x);
			if(!CsField->vvaluep(p)) continue;
			const TShortestPathSet * sps = SPSF->vvaluep(p);
			if(sps == 0) continue;
			if(sps->m_PathSet.get() == 0) throw string("!sps");
	
			MergedPathSet.merge( sps->m_PathSet.get() );
		}

	
		for(x=0; x<Junctions->getMaxIndex(); x++)
		{
			if(Junctions->vvaluex(x) != z) continue;
			const TCoord3 p = Junctions->getIndexField()->vidx2coord(x);
			if(!CsField->vvaluep(p)) continue;

			TIndexedOrigins_Vector DilatedPathSet;
			DilatedPathSet.clear();
			DO->dilate(
				g_Parameters.m_DilationDistance
				, &MergedPathSet
				, &DilatedPathSet
				, true // reset
				);

			TComponentSet * cs = new TComponentSet(p);
			skel->Omega->m_DeltaOmega->computeComponentSetUsingCells2(&DilatedPathSet, cs
	//			, false // do not fill shortest path set
				);
			
			if(CsField->vvaluep(p)) { delete CsField->vvaluep(p); CsField->wvaluep(p) = 0; }
			CsField->wvaluep(p) = cs;
			
			cs->simplify(p);
			cs->projectEft(SPSF->vvaluep(p), DebugField );
		}
	}

	CsField->getLayer()->onFieldChanged();
	CsField->getLayer()->m_AllVoxels->setCheckedForRendering(false);
}


void TAction_MergeJunctions::perform_main()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	TDeltaOmega * DO = skel->Omega->m_DeltaOmega.get();

	TTypedFieldInterface<unsigned char>* Junctions = static_cast<TTypedFieldInterface<unsigned char>*>( g_Mediator.getCurrentLayerSet()->getField("Junctions" ) );
	if(!Junctions) throw string("!Junctions");

	const TTypedFieldInterface<TShortestPathSet*> * SPSF = skel->m_ForegroundSpSetField.get();
	if(!SPSF) throw string("!SPSF");

//	TTypedFieldInterface<unsigned int>* DebugField = static_cast<TTypedFieldInterface<unsigned int>*>( g_Mediator.getCurrentLayerSet()->newField(TType::TYPE_UINT, skel->m_BoundaryIndexField, "DebugField" ) );
//	DebugField->clear();



	// Merge junctions that are within each other's inscribed ball radius
	unsigned int x,y;
	if(true)
	{
		for(x=0; x<Junctions->getMaxIndex(); x++)
		for(y=x+1; y<Junctions->getMaxIndex(); y++)
		{
			if( Junctions->vvaluex(x) == 0 ) continue;
			if( Junctions->vvaluex(y) == 0 ) continue;

			const TCoord3 px = Junctions->getIndexField()->vidx2coord(x);
			const TCoord3 py = Junctions->getIndexField()->vidx2coord(y);
			const TShortestPathSet * spsx = SPSF->vvaluep(px);
			const TShortestPathSet * spsy = SPSF->vvaluep(py);
			
			if( px.distance(py) < spsx->getMaxRadius() + spsy->getMaxRadius() * 1.1f ) // *** PARAMETER
			{
				// Do merge
				Junctions->wvaluex(y) = Junctions->vvaluex(x);
			}
		}
	}
	else // Merge junctions that are within distance of 5.0f
	{
		for(x=0; x<Junctions->getMaxIndex(); x++)
		for(y=x+1; y<Junctions->getMaxIndex(); y++)
		{
			if( Junctions->vvaluex(x) == 0 ) continue;
			if( Junctions->vvaluex(y) == 0 ) continue;

			const TCoord3 px = Junctions->getIndexField()->vidx2coord(x);
			const TCoord3 py = Junctions->getIndexField()->vidx2coord(y);
			if( px.distance(py) < 5.1f )
			{
				// Do merge
				Junctions->wvaluex(y) = Junctions->vvaluex(x);
			}
		}
	}
		
	// Rename junctions
	map<unsigned int, unsigned int> Map;
	unsigned int FreeId = 1;
	for(x=0; x<Junctions->getMaxIndex(); x++)
	{
		if(Junctions->vvaluex(x) == 0) continue;
		if(!Map[Junctions->vvaluex(x)]) Map[ Junctions->vvaluex(x) ] = FreeId++;
		Junctions->wvaluex(x) = Map[Junctions->vvaluex(x)];
	}


	// Todo: test
	// updateComponentSets(Junctions);

	Junctions->getLayer()->onFieldChanged();
}

// ------------------------------------------------------------------------------------------

TAction_VolumeSegmentationFlat::TAction_VolumeSegmentationFlat()
	: m_FillZeros(true)
	, m_Name( getStaticName() )
	, m_DilationDistance( g_Parameters.m_DilationDistance )
	, m_ForceTwoComponentsPerCp(false)
	, m_MakeMinColor(false)
	, m_RemoveSmallComponents(0.0f)
{
};


bool TAction_VolumeSegmentationFlat::isAvailable()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	if(!skel) return false;
	return true;
}



void TAction_VolumeSegmentationFlat::perform_main()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	TDeltaOmega * DO = skel->Omega->m_DeltaOmega.get();

	vector<TCoord3> CriticalPoints;
	g_Mediator.getCurrentLayer()->m_Selection->initTraversal();
	TCoord3 c;
	while(g_Mediator.getCurrentLayer()->m_Selection->nextTraversal(c))
	{
		CriticalPoints.push_back(c);
	}
	if(CriticalPoints.size() == 0) 
	{
		throw string("No critical points selected");
	}
	
	const TTypedFieldInterface<TShortestPathSet*> * SPSF = skel->m_ForegroundSpSetField.get();
	if(!SPSF) throw string("!SPSF");


	const TTypedFieldInterface<TComponentSet*>* CsField = static_cast<TTypedFieldInterface<TComponentSet*>*>( g_Mediator.getCurrentLayerSet()->getField( "Component Sets" ) );
	if(!CsField) throw string("!CsField");
	
	//const TTypedFieldInterface<unsigned char>* Junctions = static_cast<TTypedFieldInterface<unsigned char>*>( g_Mediator.getCurrentLayerSet()->getField("Junctions" ) );
	//if(!Junctions) throw string("!Junctions");

	TTypedFieldInterface<unsigned char>* SpSet = static_cast<TTypedFieldInterface<unsigned char>*>( g_Mediator.getCurrentLayerSet()->newField(TType::TYPE_UINT, skel->m_BoundaryIndexField, "SpSet" ) );
	SpSet->clear();

	TIndexedOrigins_Vector DilatedPathSet;
	TIndexedOrigins_Vector ErodedPathSet;
	unsigned int x,y;
	//for(x=0; x<Junctions->getMaxIndex(); x++)
	for(x=0; x<CriticalPoints.size(); x++)
	{
		const TCoord3 p = CriticalPoints[x];
		//const TCoord3 p = Junctions->getIndexField()->vidx2coord(x);
		//if(Junctions->vvaluep(p) == 0) continue;
	
		const TShortestPathSet * sps = SPSF->vvaluep(p);
		if(sps == 0) continue;
		DilatedPathSet.clear();
		if(m_ForceTwoComponentsPerCp && m_ForceTwoComponentsPerCpExceptThese.find(p) == m_ForceTwoComponentsPerCpExceptThese.end() )
		{
			//DO->dilateUntilTwoComponents(m_DilationDistance, sps->m_PathSet.get(), &DilatedPathSet);
			TComponentSet ComponentSet(sps, 2.0f);

			
			TComponentSet * cs = &ComponentSet;
			
			if(cs->m_Cells.size() <= 1) continue;

			TComponentSet::TCellSizes sizes = cs->getCellSizes();
			TComponentSet::TCellSizes::iterator jt,end;
			
			{
				shared_ptr<TIndexedOrigins_Cells> cell = cs->m_Cells[0];
				
				shared_ptr<TIndexedOrigins::TIterator> it( cell->newIterator() );
				it->init();
				while(it->valid())
				{
					SpSet->wvaluex( it->value() ) = 1;
					it->next();
				}
			}

			end = sizes.end();
			end--;
			end--;
			for(jt = sizes.begin(); jt != end; jt++)
			{
				shared_ptr<TIndexedOrigins_Cells> cell = jt->second;
				
				shared_ptr<TIndexedOrigins::TIterator> it( cell->newIterator() );
				it->init();
				while(it->valid())
				{
					SpSet->wvaluex( it->value() ) = 1;
					it->next();
				}
			}
		}
		else
		{
			DO->dilate(m_DilationDistance, sps->m_PathSet.get(), &DilatedPathSet);

			assert( sps->m_PathSet.get() != 0 );
			for(y=0; y<DilatedPathSet.size(); y++)
			{
				SpSet->wvaluex( DilatedPathSet[y] ) = 1;
			}		
		}
	}

	SpSet->getLayer()->onFieldChanged();		

	m_Output = static_cast<TTypedFieldInterface<unsigned int>*>( g_Mediator.getCurrentLayerSet()->newField(TType::TYPE_UINT, skel->m_BoundaryIndexField, m_Name ) );
	m_Output->clear();
	DO->findConnectedComponents(SpSet->getLayer()->m_Filter.get(), m_Output, true, &m_Sizes);

	// Remove small components
	if(m_RemoveSmallComponents > 0.000001f)
	{
		for(x=0; x<skel->m_BoundaryIndexField->getMaxIndex(); x++)
		{
			if(m_Output->vvaluex(x) == 0) continue;
			
			if( (float)m_Sizes[m_Output->vvaluex(x)].size() / (float)m_Output->getMaxIndex() < m_RemoveSmallComponents )
			{
				m_Output->wvaluex(x) = 0;
			}
		}
	}

	if(m_FillZeros) DO->fillZerosByPropagation(m_Output);
	
	if(m_MakeMinColor) 
	{
		skel->Omega->m_DeltaOmega->makeMinColoring( m_Output );
	}

	m_Output->getLayer()->onFieldChanged();		
}

// ------------------------------------------------------------------------------------------


TAction_VolumeSegmentationFlat2::TAction_VolumeSegmentationFlat2()
	: TAction_VolumeSegmentationFlat()
{
	m_MakeMinColor = true;
};


void TAction_VolumeSegmentationFlat2::perform_main()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	TDeltaOmega * DO = skel->Omega->m_DeltaOmega.get();

	vector<TCoord3> CriticalPoints;
	g_Mediator.getCurrentLayer()->m_Selection->initTraversal();
	TCoord3 c;
	while(g_Mediator.getCurrentLayer()->m_Selection->nextTraversal(c))
	{
		CriticalPoints.push_back(c);
	}
	if(CriticalPoints.size() == 0) throw string("No critical points selected");

	const TTypedFieldInterface<TShortestPathSet*> * SPSF = skel->m_ForegroundSpSetField.get();
	if(!SPSF) throw string("!SPSF");

	TTypedFieldInterface<unsigned int>* TempField = static_cast<TTypedFieldInterface<unsigned int>*>( g_Mediator.getCurrentLayerSet()->newField(TType::TYPE_UINT, skel->m_BoundaryIndexField, "TempField" ) );
	TempField->clear();

//	const TTypedFieldInterface<TComponentSet*>* CsField = static_cast<TTypedFieldInterface<TComponentSet*>*>( g_Mediator.getCurrentLayerSet()->getField( "Component Sets" ) );
//	if(!CsField) throw string("!CsField");

	vector< shared_ptr<TComponentSet> > componentsets;
	TComponentSet::TCellSizes cells;
	multimap<float, set<unsigned int> > Sorted;
	unsigned int x,y;
	for(x=0; x<CriticalPoints.size(); x++)
	{
		const TCoord3 p = CriticalPoints[x];

		//const TCoord3 p = skel->m_CskelIndexField->vidx2coord(x);
		TShortestPathSet * sps = SPSF->vvaluep(p);
		if(!sps) continue;
//	Output->wvaluex(x) = new TComponentSet(sps);
		shared_ptr<TComponentSet> cs( new TComponentSet(sps) );
		componentsets.push_back(cs);

		//TComponentSet * cs = CsField->vvaluep(p);
		//if(cs == 0) continue;
		TComponentSet::TCellSizes sizes = cs->getCellSizes();
		TComponentSet::TCellSizes::reverse_iterator jt = sizes.rbegin();

		TempField->clear();
		{
			shared_ptr<TIndexedOrigins::TIterator> it( jt->second->newIterator() );
			it->init();
			while(it->valid())
			{
				TempField->wvaluex( it->value() ) = 1;
				it->next();
			}
		}
		jt++;
		{
			shared_ptr<TIndexedOrigins::TIterator> it( jt->second->newIterator() );
			it->init();
			while(it->valid())
			{
				TempField->wvaluex( it->value() ) = 2;
				it->next();
			}
		}
		DO->fillZerosByPropagation(TempField);

		set<unsigned int> Component;
		for(y=0; y<TempField->getMaxIndex(); y++)
		{
			if(TempField->vvaluex(y) == 2)
			{
				Component.insert(y);
			}
		}
		Sorted.insert( pair<float,set<unsigned int> >((float)Component.size(), Component) );
	}

	m_Output = static_cast<TTypedFieldInterface<unsigned int>*>( g_Mediator.getCurrentLayerSet()->newField(TType::TYPE_UINT, skel->m_BoundaryIndexField, m_Name ) );
	m_Output->clear();
	
	unsigned int CompId=1;
	//TComponentSet::TCellSizes::reverse_iterator jt;
	std::multimap<float, set<unsigned int> >::reverse_iterator jt;
	for(jt=Sorted.rbegin(); jt != Sorted.rend(); jt++)
	{
		std::set<unsigned int>::iterator it;
		for(it = jt->second.begin(); it != jt->second.end(); it++)
		{
			m_Output->wvaluex( *it ) = CompId;
		}
		/*
		shared_ptr<TIndexedOrigins::TIterator> it( jt->second.newIterator() );
		it->init();
		while(it->valid())
		{
			m_Output->wvaluex( it->value() ) = CompId;
			it->next();
		}
		*/
		CompId++;	
	}	

	m_OutputSegmentCount = CompId;

	// Label zeros
	TIndexedOrigins_Vector Zeroes;
	Zeroes.reserve( m_Output->getMaxIndex() );
	for(x=0; x<m_Output->getMaxIndex(); x++)
	{
		if( m_Output->vvaluex(x) == 0 ) 
		{
			Zeroes.push_back(x);
		}
	}

	vector<set<unsigned int> > Sizes;
	DO->findConnectedComponents(&Zeroes, &Sizes);

	for(x=0; x<Sizes.size(); x++)
	{
		std::set<unsigned int>::iterator it;
		for(it = Sizes[x].begin(); it != Sizes[x].end(); it++)
		{
			m_Output->wvaluex(*it) = CompId;
		}
		CompId++;
	}	


	if(m_MakeMinColor) 
	{
//		TTypedFieldInterface<unsigned int>* MinColor = static_cast<TTypedFieldInterface<unsigned int>*>( g_Mediator.getCurrentLayerSet()->newField(TType::TYPE_UINT, skel->m_BoundaryIndexField, m_Name ) );
	
		skel->Omega->m_DeltaOmega->makeMinColoring( m_Output );
	}
	m_Output->getLayer()->onFieldChanged();		
}

// ------------------------------------------------------------------------------------------

bool TAction_CleanupCriticalPoints::isAvailable()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	if(!skel) return false;
	return true;
}


void TAction_CleanupCriticalPoints::perform_main()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	TDeltaOmega * DO = skel->Omega->m_DeltaOmega.get();

	
	const TTypedFieldInterface<unsigned char>* Junctions = static_cast<TTypedFieldInterface<unsigned char>*>( g_Mediator.getCurrentLayerSet()->getField("Junctions" ) );
	if(!Junctions) throw string("!Junctions");

	TTypedFieldInterface<unsigned char>* Cp = static_cast<TTypedFieldInterface<unsigned char>*>( g_Mediator.getCurrentLayerSet()->getField("New critical points" ) );



	/*
	TTypedFieldInterface<float>* Measure = static_cast<TTypedFieldInterface<float>*>( g_Mediator.getCurrentLayerSet()->getField(TAction_CurveSkeletonFeatureAngle::getStaticName() ) );
	if(!Measure) throw string("!Measure");

	unsigned int maxjunctionid = 0;
	unsigned int x;
	for(x=0; x<Junctions->getMaxIndex(); x++)
	{
		if(Junctions->vvaluex(x) > maxjunctionid) maxjunctionid = Junctions->vvaluex(x);
	}

	// For each critical point, search for other critical point, but do not go through junctions
	vector<unsigned int> DiscardThese;
	for(x=0; x<Cp->getMaxIndex(); x++)
	{
		if( Cp->vvaluex(x) == 0 ) continue;

		vector<float> Distance( Cp->getMaxIndex(), (std::numeric_limits<float>::max)() );
		vector<unsigned int> Previous( Cp->getMaxIndex(), (std::numeric_limits<unsigned int>::max)() );

		typedef multimap<float, unsigned int> TMap;
		TMap Q;
		Q.clear();
		
		*g_Log << "Processing CP " << x << " from junction " << Cp->vvaluex(x) << "\n";
		Q.insert( TMap::value_type(0.0f,x) );
		Distance[x] = 0.0f;
		Previous[x] = x;
		bool JunctionIsClose = false;
		float FoundOtherJunctionAtDistance = -1.0f;
		bool FoundOtherCp = false;
		float FoundOtherCpAtDistance = -1.0f;

		while(!Q.empty())
		{
			const float pidxdist = Q.begin()->first;
			const unsigned int pidx = Q.begin()->second;
			Q.erase( Q.begin() );

			const TCoord3 p = Cp->getIndexField()->vidx2coord(pidx);
			
			if(pidx != x && Junctions->vvaluex(pidx) != 0
				&& FoundOtherJunctionAtDistance == -1.0f
				) 
			{
				// WARNING: other junction besides the corresponding junction maybe found because of C-skel thickness
				if( Junctions->vvaluex(pidx) != Cp->vvaluex(x) ) *g_Log << "Warning: found junction id different from CP id.";

				// Trace back to critical point
				JunctionIsClose = true;
				FoundOtherJunctionAtDistance = pidxdist;
				unsigned int a = Previous[pidx];
				while(a != a);
				{
					if( Measure->vvaluex(a) < 0.8f )
					{
						JunctionIsClose = false;
					} 
					a = Previous[a];
				}
				continue; // Stop path when junction reached
			}

			if(pidx != x && Cp->vvaluex(pidx) != 0)
			{
				FoundOtherCp = true;
				FoundOtherCpAtDistance = pidxdist;
				continue;
			}
			

			TCoord3 np;
			for(np.x=p.x-1; np.x<=p.x+1; np.x++) // neighbors
			for(np.y=p.y-1; np.y<=p.y+1; np.y++)
			for(np.z=p.z-1; np.z<=p.z+1; np.z++)
			{
				if( (p.x == np.x) + (p.y == np.y) + (p.z == np.z) != 2 ) continue; // Only 6-neighbors
				
				const float WEIGHT = p.toVector().distance( np.toVector() );

				const unsigned int npidx = Junctions->getIndexField()->vcoord2idx(np);
				const float newd = Distance[pidx] + WEIGHT;
				if( newd < Distance[npidx] )
				{
					Distance[npidx] = newd;
					Previous[npidx] = pidx;
					Q.insert( TMap::value_type(newd,npidx) );
				}
			}	
		}
	
		if( JunctionIsClose && FoundOtherCp ) 
		{
			*g_Log << "Discarding critical point. ";
			DiscardThese.push_back(x);
		}
		else
		{
			*g_Log << "Keeping critical point. ";
		}
		*g_Log << "JunctionIsClose: " << JunctionIsClose << ", dist: " << FoundOtherJunctionAtDistance << ", FoundOtherCp: " << FoundOtherCp << ", dist: " << FoundOtherCpAtDistance << "\n";
	}

	for(x=0; x<DiscardThese.size(); x++)
	{
		Cp->wvaluex( DiscardThese[x] ) = 0;
	}


	Cp->getLayer()->onFieldChanged();
	*/
}




// ------------------------------------------------------------------------------------------

bool TAction_MoveCriticalPoints2::isAvailable()
{
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	if(!skel) return false;
	return true;
	
}

void TAction_MoveCriticalPoints2::perform_main()
{

	unsigned int x;
	TSkeletonizer * skel = g_Mediator.getSkeletonizer().get();
	//TTypedFieldInterface<TShortestPathSet*> * SPSF = skel->m_ForegroundSpSetField.get();


	if(!g_Mediator.getCurrentLayerSet()->exists("Junctions"))
	{
		TAction_NewJunctionDetection Action;
		Action.perform();
		TTypedFieldInterface<unsigned char>* Junctions = static_cast<TTypedFieldInterface<unsigned char>*>( g_Mediator.getCurrentLayerSet()->getField("Junctions" ) );
		if(!Junctions) throw string("!Junctions");

		/*
		// Make copy of old junctions
		TTypedFieldInterface<unsigned char>* OldJunctions = static_cast<TTypedFieldInterface<unsigned char>*>( g_Mediator.getCurrentLayerSet()->newField(TType::TYPE_UCHAR, skel->m_CskelIndexField, "Old Junctions" ) );
		if(!OldJunctions) throw string("!OldJunctions");
		for(x=0; x<Junctions->getMaxIndex(); x++)
		{
			OldJunctions->wvaluex(x) = Junctions->vvaluex(x);
		}
		OldJunctions->getLayer()->onFieldChanged();
		OldJunctions->getLayer()->m_RenderPrimitiveType = 1;
		*/
	}

	if(!g_Mediator.getCurrentLayerSet()->exists(TAction_CurveSkeletonFeatureAngle::getStaticName()))
	{
		TAction_CurveSkeletonFeatureAngle Action;
		Action.perform();
	}



	TTypedFieldInterface<unsigned char>* Junctions = static_cast<TTypedFieldInterface<unsigned char>*>( g_Mediator.getCurrentLayerSet()->getField("Junctions" ) );
	if(!Junctions) throw string("!Junctions");





	TTypedFieldInterface<unsigned char>* NewCp = static_cast<TTypedFieldInterface<unsigned char>*>( g_Mediator.getCurrentLayerSet()->newField(TType::TYPE_UCHAR, skel->m_CskelIndexField, "New critical points" ) );
	NewCp->clear();

	TTypedFieldInterface<unsigned char>* Cskel = static_cast<TTypedFieldInterface<unsigned char>*>( g_Mediator.getCurrentLayerSet()->getField( "C-skel detector" ) );
	if(!Cskel) throw string("!Cskel");
	TTypedFieldInterface<float>* Measure = static_cast<TTypedFieldInterface<float>*>( g_Mediator.getCurrentLayerSet()->getField(TAction_CurveSkeletonFeatureAngle::getStaticName() ) );
	if(!Measure) throw string("!Measure");

	const TTypedFieldInterface<TShortestPathSet*> * SPSF = skel->m_ForegroundSpSetField.get();
	if(!SPSF) throw string("!SPSF");
	const TTypedFieldInterface<TComponentSet*>* CsField = static_cast<TTypedFieldInterface<TComponentSet*>*>( g_Mediator.getCurrentLayerSet()->getField( "Component Sets" ) );
	if(!CsField) throw string("!CsField");
	
	TTypedFieldInterface<unsigned char>* SegmentationJunction = static_cast<TTypedFieldInterface<unsigned char>*>( g_Mediator.getCurrentLayerSet()->newField(TType::TYPE_UCHAR, skel->m_BoundaryIndexField, "SegmentationJunction" ) );
	SegmentationJunction->clear();

	
	// Create junction blocker field
	TTypedFieldInterface<unsigned char>* JunctionBlocker = static_cast<TTypedFieldInterface<unsigned char>*>( g_Mediator.getCurrentLayerSet()->newField(TType::TYPE_UCHAR, Junctions->getIndexField(), "Junction blocker" ) );
	{
		JunctionBlocker->clear();
		for(x=0; x<Junctions->getMaxIndex(); x++)
		{
			if(Junctions->vvaluex(x) == 0.0f) continue;
			const TCoord3 p = Junctions->getIndexField()->vidx2coord(x);

			TCoord3 np;
			for(np.x=p.x-1; np.x<=p.x+1; np.x++) // neighbors
			for(np.y=p.y-1; np.y<=p.y+1; np.y++)
			for(np.z=p.z-1; np.z<=p.z+1; np.z++)
			{
				if(!Junctions->getIndexField()->vinside(np)) continue;
				JunctionBlocker->wvaluep(np) = Junctions->vvaluex(x);
			}
		}
		JunctionBlocker->getLayer()->onFieldChanged();
	}



	const float ANGLE = 0.8f;
	unsigned int maxjunctionid = 0;
	for(x=0; x<Junctions->getMaxIndex(); x++)
	{
		if(Junctions->vvaluex(x) > maxjunctionid) maxjunctionid = Junctions->vvaluex(x);
	}
	vector<bool> DiscardedJunctions( maxjunctionid+1, false );

	vector< set<unsigned int> > JunctionId2Cp( maxjunctionid+1, set<unsigned int>() );
	vector< set<unsigned int> > JunctionId2JunctionPos( maxjunctionid+1, set<unsigned int>() );

	// For each junction id, search for new critical points starting from junctions
	for(unsigned int z=1; z<=maxjunctionid; z++)
	{
		// For each junction, do distance propagation
		vector<float> Distance( Junctions->getMaxIndex(), (std::numeric_limits<float>::max)() );
		vector<unsigned int> Previous( Junctions->getMaxIndex(), (std::numeric_limits<unsigned int>::max)() );
		
		typedef multimap<float, unsigned int> TMap;
		TMap Q;
		Q.clear();
		
		// Push all junctions with id "z" onto queue
		float MaxJuncLength = 0.0f;
		for(x=0; x<Junctions->getMaxIndex(); x++)
		{
			if(Junctions->vvaluex(x) == z)
			{
				JunctionId2JunctionPos[z].insert(x);
				Q.insert( TMap::value_type(0.0f, x) );
				Distance[x] = 0.0f;
				const TShortestPathSet * sps = SPSF->vvaluep(Junctions->getIndexField()->vidx2coord(x));
				if( sps->getMaxLength() > MaxJuncLength ) MaxJuncLength = sps->getMaxLength();
			}
		}
		if(Q.size() == 0) continue;

		// Fill SegmentationJunction with components associated with junction, where a 
		// CP is allowed to go.
		unsigned int maxcompid = 1;
		const float BODYTHRESHOLD = 0.5f;
		SegmentationJunction->clear();
		for(x=0; x<Junctions->getMaxIndex(); x++)
		{
			if(Junctions->vvaluex(x) != z) continue;
			TComponentSet * cs = CsField->vvaluex(x);
			TComponentSet::TCellSizes sizes = cs->getCellSizes();
			if(sizes.size() < 3) break;

			TComponentSet::TCellSizes::iterator k = sizes.end(); k--;
			TComponentSet::TCellSizes::iterator kmin1 = k; kmin1--;
			TComponentSet::TCellSizes::iterator kmin2 = kmin1; kmin2--;

			// Three options: 0,1 or 2 body parts
			unsigned int bodyparts = 0;
			
			TComponentSet::TCellSizes::iterator parts;

			// First check for all body parts, this is when all three sizes are about the same
			if( kmin2->first / kmin1->first < BODYTHRESHOLD )
			{
				bodyparts = 2;
				parts = kmin2;
			}
			//else if( kmin1->first / k->first < BODYTHRESHOLD )
			else
			{
				bodyparts = 1;
				parts = kmin1;
			}
			//else 
			//{
			//	bodyparts = 0;
			//	parts = k;
			//}
			parts++;
			*g_Log << "Junction " << z << " has " << bodyparts << " body\n";

			for(TComponentSet::TCellSizes::iterator jt = sizes.begin(); jt != parts; jt++)
			{
				shared_ptr<TIndexedOrigins_Cells> cell = jt->second;
				
				shared_ptr<TIndexedOrigins::TIterator> it( cell->newIterator() );
				it->init();
				while(it->valid())
				{
					SegmentationJunction->wvaluex( it->value() ) = maxcompid;
					it->next();
				}
				maxcompid++;
			}

			break;
		}
	

		// Project component set
		vector<unsigned int> MapComponentToCp( maxcompid+1, (std::numeric_limits<unsigned int>::max)() );


		vector<unsigned int> NewCriticalPoints;
		set<unsigned int> OtherJunctionsReached;
		while(!Q.empty())
		{
			const unsigned int pidx = Q.begin()->second;
			const TCoord3 p = Junctions->getIndexField()->vidx2coord( pidx );
			Q.erase( Q.begin() );


			// Stop when other junction reached
			if( JunctionBlocker->vvaluex(pidx) != z 
				&& JunctionBlocker->vvaluex(pidx) != 0
				// && Distance[pidx] > 5.0f 
				) 
			{
				OtherJunctionsReached.insert( JunctionBlocker->vvaluex(pidx) );
				continue;
			}

			const TShortestPathSet * sps = SPSF->vvaluep(p);
			assert(sps != 0);

			// if( sps->getMaxLength() < MaxJuncLength * 0.75f ) // parameter
			{
				// See if ALL of the eft is in ONE component
				const TShortestPathSet * sps = SPSF->vvaluep( p );
				set<unsigned int> Components;
				for(unsigned int x=0; x<sps->m_Eft->size(); x++)
				{
					const unsigned int idx = (*sps->m_Eft)[x];
					// compid = TempComponentSetField->vvaluex(idx);
					const unsigned int compid = SegmentationJunction->vvaluex(idx);
					if(compid == 0) continue;
					Components.insert(compid);
				}
				
				if(Components.size() != 1) { ; } // If not only one component among eft voxels then skip
				else if( 
					MapComponentToCp[*Components.begin()] == (std::numeric_limits<unsigned int>::max)() // component not yet assigned
					)
				{
					if( Measure->vvaluex(pidx) > ANGLE )
					{
						// Found new CP
						MapComponentToCp[*Components.begin()] = pidx;
						NewCriticalPoints.push_back(pidx);
						JunctionId2Cp[z].insert( pidx );
						continue;
					}
				}
				else continue;
			}

			TCoord3 np;
			for(np.x=p.x-1; np.x<=p.x+1; np.x++) // neighbors
			for(np.y=p.y-1; np.y<=p.y+1; np.y++)
			for(np.z=p.z-1; np.z<=p.z+1; np.z++)
			{
				if( (p.x == np.x) + (p.y == np.y) + (p.z == np.z) != 2 ) continue; // Only 6-neighbors
				
				const float WEIGHT = p.toVector().distance( np.toVector() );

				const unsigned int npidx = Junctions->getIndexField()->vcoord2idx(np);
				const float newd = Distance[pidx] + WEIGHT;
				if( newd < Distance[npidx] )
				{
					Distance[npidx] = newd;
					Q.insert( TMap::value_type(newd,npidx) );
				}
			}		
		} // end while


		// Keep all junctions
		if(true) // *** PARAMETER
		{
			for(unsigned int y=0; y<NewCriticalPoints.size(); y++)
			{
				NewCp->wvaluex(NewCriticalPoints[y]) = z;
			}
		}
		else
		// Print information and keep junctions 
		{	
			*g_Log << "Junctions " << z << ": found " << (unsigned int)NewCriticalPoints.size() << " critical points: ";
			for(unsigned int y=0; y<NewCriticalPoints.size(); y++)
			{
				*g_Log << NewCriticalPoints[y] << ", ";
			}
			*g_Log << "  reached " << (unsigned int)OtherJunctionsReached.size() << " junctions: ";
			std::set<unsigned int>::iterator it;
			for(it = OtherJunctionsReached.begin(); it != OtherJunctionsReached.end(); it++)
			{
				*g_Log << *it << ", ";	
			}
			if(NewCriticalPoints.size() + OtherJunctionsReached.size() >= 3) 
			{
				*g_Log << " keep.";
				for(unsigned int y=0; y<NewCriticalPoints.size(); y++)
				{
					NewCp->wvaluex(NewCriticalPoints[y]) = z;
				}
			}
			else
			{
				*g_Log << " discard.";
				DiscardedJunctions[z] = true;
			}
			*g_Log << "\n";
		}
	}


	// Delete discarded junctions
	{
		for(x=0; x<Junctions->getMaxIndex(); x++)
		{
			if( DiscardedJunctions[Junctions->vvaluex(x)] ) Junctions->wvaluex(x) = 0;
		}
	}

	Junctions->getLayer()->onFieldChanged();
	Junctions->getLayer()->m_RenderPrimitiveType = 1;
	NewCp->getLayer()->onFieldChanged();
	NewCp->getLayer()->m_RenderPrimitiveType = 1;
	NewCp->getLayer()->setCheckedForRendering(true);

	SegmentationJunction->getLayer()->onFieldChanged();



	// Final segmentation
	{
		TAction_VolumeSegmentationFlat Action;
		Action.m_Name = "Final segmentation";
		Action.m_FillZeros = true;
		NewCp->getLayer()->m_Selection->clear();
		for(x=0; x<NewCp->getMaxIndex(); x++)
		{
			if(NewCp->vvaluex(x) == 0) continue;
			const TCoord3 p = NewCp->getIndexField()->vidx2coord(x);
			NewCp->getLayer()->m_Selection->select( p );
		}
		g_Mediator.setCurrentLayer( NewCp->getLayer() );
		Action.perform();
	}
}


// ------------------------------------------------------------------------------------------


TAction_VolumeSegmentationLigatures::TAction_VolumeSegmentationLigatures()
{
}


void TAction_VolumeSegmentationLigatures::perform_main()
{
	const TTypedFieldInterface<TShortestPathSet*> * SPSF = g_Mediator.getSkeletonizer()->m_ForegroundSpSetField.get();
	if(!SPSF) throw string("!SPSF");

	TTypedFieldInterface<float>* Cskel = static_cast<TTypedFieldInterface<float>*>( g_Mediator.getCurrentLayerSet()->getField( "C-skel measure" ) );
	if(!Cskel) throw string("!Cskel");

	TTypedFieldInterface<unsigned int>* LigatureCount = static_cast<TTypedFieldInterface<unsigned int>*>( g_Mediator.getCurrentLayerSet()->newField(TType::TYPE_UINT, g_Mediator.getSkeletonizer()->m_BoundaryIndexField, "Ligature count" ) );
	LigatureCount->clear();
	for(unsigned int x=0; x<Cskel->getMaxIndex(); x++)
	{
		const TCoord3 p = Cskel->getIndexField()->vidx2coord( x );
		if( ! Cskel->getLayer()->m_Filter->testx(x) ) continue;

		const TShortestPathSet * sps = SPSF->vvaluep(p);
		if(sps == 0) continue;

		TIndexedOrigins_Vector DilatedShortestPaths;
		g_Mediator.getSkeletonizer()->Omega->m_DeltaOmega->dilate(g_Parameters.m_DilationDistance, sps->m_PathSet.get(), &DilatedShortestPaths, true);
		for(unsigned int y=0; y<DilatedShortestPaths.size(); y++)
		{
			LigatureCount->wvaluex( (DilatedShortestPaths)[y] )++;
		}
		
	}
	LigatureCount->getLayer()->onFieldChanged();
}


// ------------------------------------------------------------------------------------------


TAction_VolumeSegmentationMapToMesh::TAction_VolumeSegmentationMapToMesh()
{
}


void TAction_VolumeSegmentationMapToMesh::perform_main()
{
	TTypedFieldInterface<unsigned int>* Segmentation = static_cast<TTypedFieldInterface<unsigned int>*>( g_Mediator.getCurrentLayerSet()->getField( "Final segmentation" ) );
	if(!g_Mediator.getSkeletonizer()->m_Model) throw string("No model loaded");
	g_Mediator.getSkeletonizer()->m_Model->computeColorsUsingSegmentation(Segmentation);
}

