#include "stdafx.h"

#include "voxelset.h"

#include "globals.h"
#include "abstractfilter.h"
#include "layer.h"
#include "controls.h"

using namespace std;

void TVoxelSetAll::DrawGui(wxWindow * p_Window)
{
	new wxStaticText(p_Window, -1, "Contains all voxels", wxPoint(5,5), wxDefaultSize);
}

bool TVoxelSetAll::nextTraversal(TCoord3 & Coord) const
{
	const int maxx = m_Field->getMaxX();
	const int maxy = m_Field->getMaxY();
	const int maxz = m_Field->getMaxZ();
	const int Size = maxx * maxy * maxz;
	if(m_TraversalIdx >= Size) 
	{
		return false;
	}
	
	const int c = m_TraversalIdx;
	const int z = c / (maxx * maxy);
	const int remainder = c % (maxx * maxy);
	const int x = remainder % maxx;
	const int y = remainder / maxx;
	Coord = TCoord3(x,y,z); 
	m_TraversalIdx++;
	return true;
}


// --------------------

TVoxelSetAllUsingIndexMapper::TVoxelSetAllUsingIndexMapper(const TField * p_Field)
	: TVoxelSet(p_Field) 
{	
	if( !p_Field->usingIndexField() ) throw string("TVoxelSetAllUsingIndexMapper(): field does not use indexmapper");
} 

void TVoxelSetAllUsingIndexMapper::DrawGui(wxWindow * p_Window)
{
	new wxStaticText(p_Window, -1, "Contains all voxels (of the associated indexfield)", wxPoint(5,5), wxDefaultSize);
}

bool TVoxelSetAllUsingIndexMapper::nextTraversal(TCoord3 & p_Coord) const
{
	if(m_TraversalIdx >= m_Field->getIndexField()->getMaxIndex() ) 
	{
		return false;
	}
	p_Coord = m_Field->getIndexField()->vidx2coord(m_TraversalIdx);; 
	m_TraversalIdx++;
	return true;
}



// --------------------

void TVoxelSetSelection::initTraversal() const 
{
	it = m_MultipleSelection.begin();
}

bool TVoxelSetSelection::nextTraversal(TCoord3 & c) const 
{
	if(it == m_MultipleSelection.end()) return false;
	c = *it;
	it++;
	return true;
}

void TVoxelSetSelection::DrawGui(wxWindow * p_Window) 
{
}

void TVoxelSetSelection::notifyObservers()
{
	for(set<class TSelectionObserver*>::iterator it=m_Observers.begin(); it!=m_Observers.end(); it++) (*it)->onSelectionChanged();
}




// --------------------

shared_ptr<TVoxelSubset> TVoxelSubset::constructByFilter(const TField * p_Field, shared_ptr<TFilter> p_Filter)
{
	const TField * f = p_Field;
	shared_ptr<TVoxelSubset> vv( new TVoxelSubset(p_Field) );

	for(int k=0; k<f->getMaxZ(); k++)
	{
		for(int j=0; j<f->getMaxY(); j++)
		{
			for(int i=0; i<f->getMaxX(); i++)
			{
				if(p_Filter->test(TCoord3(i,j,k)))
				{
					vv->m_VoxelPositions.push_back( TCoord3(i,j,k) );
				}
			}
		}
	}

	return vv;
}


shared_ptr<TVoxelSubset> TVoxelSubset::constructByMask(const TField * p_Field, const TLayer * p_MaskLayer)
{
	if(!p_MaskLayer) throw string("TVoxelSubset::constructByMask(): !p_Layer");
	shared_ptr<TVoxelSubset> vv( new TVoxelSubset(p_Field) );
	vv->m_Name = "Masked by " + p_MaskLayer->m_Name;
	p_MaskLayer->getVisibleVoxels(&vv->m_VoxelPositions);
	return vv;
}



void TVoxelSubset::DrawGui(wxWindow * p_Window)
{
	int y = 5;
	new wxStaticText(p_Window, -1, "Voxel subset options", wxPoint(5,y), wxDefaultSize);
	y += 15;
	new wxStaticText(p_Window, -1, wxString::Format("Voxelcount: %i", m_VoxelPositions.size()), wxPoint(5,y), wxDefaultSize);
}


// --------------------





void TVoxelMultiSlice::initTraversal() const
{
	m_TraversalIdx = 0;
}

bool TVoxelMultiSlice::nextTraversal(TCoord3 & c) const
{
	int i = m_TraversalIdx;

	if(m_Axis == 0)
	{
		if(i >= m_SliceCount * m_Field->getMaxY() * m_Field->getMaxZ()) return false;
		int slice = i / ( m_Field->getMaxY() * m_Field->getMaxZ() );
		if(slice >= m_Field->getMaxX()) return false;
		i = i % ( m_Field->getMaxY() * m_Field->getMaxZ() );
		c = TCoord3(m_Value + slice, i / m_Field->getMaxZ(), i % m_Field->getMaxZ());
	}
	else if(m_Axis == 1)
	{
		if(i >= m_SliceCount * m_Field->getMaxX() * m_Field->getMaxZ()) return false;
		int slice = i / ( m_Field->getMaxX() * m_Field->getMaxZ() );
		if(slice >= m_Field->getMaxY()) return false;
		i = i % ( m_Field->getMaxX() * m_Field->getMaxZ() );
		c = TCoord3(i % m_Field->getMaxX(), m_Value + slice, i / m_Field->getMaxX());
	}
	else if(m_Axis == 2)
	{
		if(i >= m_SliceCount * m_Field->getMaxX() * m_Field->getMaxY()) return false;
		int slice = i / ( m_Field->getMaxX() * m_Field->getMaxY() );
		if(slice >= m_Field->getMaxZ()) return false;
		i = i % ( m_Field->getMaxX() * m_Field->getMaxY() );
		c = TCoord3(i % m_Field->getMaxX(), i / m_Field->getMaxX(), m_Value + slice);
	}
	else throw string("Invalid m_Axis");
	m_TraversalIdx++;
	return true;
}

void TVoxelMultiSlice::DrawGui(wxWindow * p_Window)
{
//	p_Window->GetSizer()->Add(
		new TVoxelMultiSlicePanel(p_Window, -1, this, wxPoint(0,0), p_Window->GetSize() );
//		, 1
//		, wxALL | wxEXPAND
//		, 0
//		);
//	p_Window->GetSizer()->Layout();
}






TVoxelMultiSlicePanel::TVoxelMultiSlicePanel(wxWindow* parent, wxWindowID id, TVoxelMultiSlice * p_VoxelSlice, const wxPoint& pos, const wxSize& size, long style)
	: wxPanel(parent,id,pos,size, wxSUNKEN_BORDER), m_VoxelSlice(p_VoxelSlice)
{
	wxPanel * Panel = this;

	int x = 5;
	int y = 5;

	new wxStaticText(Panel, -1, "Multislice options", wxPoint(x,y));
	y += 25;

	m_UseAaSliceX = new wxRadioButton(Panel, ID_SLICEAXISCHOICE, "X", wxPoint(x+10,y), wxDefaultSize);
	m_AaSliceX = new wxSliderWithTextCtrlForInts(Panel, ID_SLICEAXISX, 0, m_VoxelSlice->getField()->getMaxX()-1, m_VoxelSlice->m_ValueX, wxPoint(x+40, y), wxSize(100,30));
	y += 50;

	m_UseAaSliceY = new wxRadioButton(Panel, ID_SLICEAXISCHOICE, "Y", wxPoint(x+10,y), wxDefaultSize);
	m_AaSliceY = new wxSliderWithTextCtrlForInts(Panel, ID_SLICEAXISY, 0, m_VoxelSlice->getField()->getMaxY()-1, m_VoxelSlice->m_ValueY, wxPoint(x+40, y), wxSize(100,30));
	y += 50;

	m_UseAaSliceZ = new wxRadioButton(Panel, ID_SLICEAXISCHOICE, "Z", wxPoint(x+10,y), wxDefaultSize);
	m_AaSliceZ = new wxSliderWithTextCtrlForInts(Panel, ID_SLICEAXISZ, 0, m_VoxelSlice->getField()->getMaxZ()-1, m_VoxelSlice->m_ValueZ, wxPoint(x+40, y), wxSize(50,30));
	y += 50;

	new wxStaticText(Panel, -1, "#", wxPoint(x+10,y), wxDefaultSize);
	m_SliceCountControl = new wxSliderWithTextCtrlForInts(Panel, ID_SLICECOUNT, 1, 100, m_VoxelSlice->m_SliceCount, wxPoint(x+40, y), wxSize(50,30));
	y += 50;

	m_UseAaSliceX->SetValue( m_VoxelSlice->m_Axis == 0 );
	m_UseAaSliceY->SetValue( m_VoxelSlice->m_Axis == 1 );
	m_UseAaSliceZ->SetValue( m_VoxelSlice->m_Axis == 2 );

	this->Connect( ID_SLICEAXISX, wxEVT_COMMAND_SLIDER_UPDATED, (wxObjectEventFunction) &TVoxelMultiSlicePanel::OnChangeX );
	this->Connect( ID_SLICEAXISY, wxEVT_COMMAND_SLIDER_UPDATED, (wxObjectEventFunction) &TVoxelMultiSlicePanel::OnChangeY );
	this->Connect( ID_SLICEAXISZ, wxEVT_COMMAND_SLIDER_UPDATED, (wxObjectEventFunction) &TVoxelMultiSlicePanel::OnChangeZ );
	this->Connect( ID_SLICECOUNT, wxEVT_COMMAND_SLIDER_UPDATED, (wxObjectEventFunction) &TVoxelMultiSlicePanel::OnChangeSliceCount );
}

void TVoxelMultiSlicePanel::OnSliceAxisChoice(wxCommandEvent & event)
{
	if( event.GetEventObject() == m_UseAaSliceX)
	{
		m_VoxelSlice->m_Axis = 0;
	}
	else if( event.GetEventObject() == m_UseAaSliceY)
	{
		m_VoxelSlice->m_Axis = 1;

	}
	else if( event.GetEventObject() == m_UseAaSliceZ)
	{
		m_VoxelSlice->m_Axis = 2;
	}
	g_Mediator.redrawCanvas();
}

void TVoxelMultiSlicePanel::OnChangeX(wxCommandEvent & event)
{
	m_UseAaSliceX->SetValue(TRUE);
	m_UseAaSliceY->SetValue(FALSE);
	m_UseAaSliceZ->SetValue(FALSE);
	m_VoxelSlice->m_Axis = 0;
	m_VoxelSlice->m_Value = event.GetInt();
	m_VoxelSlice->m_ValueX = event.GetInt();
	g_Mediator.redrawCanvas();
}

void TVoxelMultiSlicePanel::OnChangeY(wxCommandEvent & event)
{
	m_UseAaSliceX->SetValue(FALSE);
	m_UseAaSliceY->SetValue(TRUE);
	m_UseAaSliceZ->SetValue(FALSE);

	m_VoxelSlice->m_Axis = 1;
	m_VoxelSlice->m_Value = event.GetInt();
	m_VoxelSlice->m_ValueY = event.GetInt();
	g_Mediator.redrawCanvas();
}

void TVoxelMultiSlicePanel::OnChangeZ(wxCommandEvent & event)
{
	m_UseAaSliceX->SetValue(FALSE);
	m_UseAaSliceY->SetValue(FALSE);
	m_UseAaSliceZ->SetValue(TRUE);
	m_VoxelSlice->m_Axis = 2;
	m_VoxelSlice->m_Value = event.GetInt();
	m_VoxelSlice->m_ValueZ = event.GetInt();
	g_Mediator.redrawCanvas();
}

void TVoxelMultiSlicePanel::OnChangeSliceCount(wxCommandEvent & event)
{
	m_VoxelSlice->m_SliceCount = event.GetInt();
	g_Mediator.redrawCanvas();
}



