#ifndef VOXELSET_H
#define VOXELSET_H

#include "stdafx.h"
#include <vector>
using std::vector;
#include <set>
using std::set;
//#include "stdafx.h"
#include "coord.h"
#include "geometry.h"
#include "abstractfilter.h"
#include "wx/panel.h"

class TVoxelSet 
{
public:
	// Types
	enum VOXELSETTYPE { VOXELSETTYPE_NONE, VOXELSETTYPE_ALL, VOXELSETTYPE_SELECTION, VOXELSETTYPE_SUBSET, VOXELSETTYPE_SLICE, VOXELSETTYPE_ORIGSET, VOXELSETTYPE_COORD2SET, VOXELSETTYPE_MULTISLICE, VOXELSETTYPE_VISIBLEVOXEL };

	// Accessors
	TVoxelSet(const class TField * p_Field) 
		: m_Field(p_Field)
		, m_DefaultColor(TColor3(0.0f,1.0f,0.0f))
		, m_CanBePicked(true)
		, m_ApplyFilter(true)
		, m_RenderOverlay(false)
		, m_DisplayList(0)
		, m_ApplyColormap(true)
		, m_Name("")
		, m_NeedRedraw(true)
	{
	};

	void setName(string p_Name) { m_Name = p_Name; }
	virtual string getName() const { return m_Name;  }
	const class TField * getField() const { return m_Field; }
	const TColor3 & getDefaultColor() const { return m_DefaultColor; }

	void setCanBePicked(bool b) { m_CanBePicked = b; } 
	bool getCanBePicked() const { return m_CanBePicked; }

	void setApplyFilter(bool b) { m_ApplyFilter = b; }
	bool getApplyFilter() const { return m_ApplyFilter; }

	void setApplyColorMap(bool b) { m_ApplyColormap = b; }
	bool getApplyColorMap() const { return m_ApplyColormap; }

	void setRenderAsOverlay(bool b) { m_RenderOverlay = b; }
	bool getRenderAsOverlay() const { return m_RenderOverlay; }

	void setCheckedForRendering(bool b) { m_CheckedForRendering = b; }
	bool getCheckedForRendering() const { return m_CheckedForRendering; }

	virtual VOXELSETTYPE getVoxelSetType() const = 0;

	virtual void initTraversal() const = 0;
	virtual bool nextTraversal(TCoord3 & c) const = 0;

	// Methods
	virtual void DrawGui(wxWindow * p_Window) = 0;
	

	unsigned int m_DisplayList;
	bool m_NeedRedraw;

	const TField * m_Field;
	string m_Name;
	TColor3 m_DefaultColor;
	bool m_CanBePicked;
	bool m_ApplyFilter;
	bool m_ApplyColormap;
	bool m_RenderOverlay;
	bool m_CheckedForRendering;
};


class TVoxelSetSelection : public TVoxelSet
{
public:
	TVoxelSetSelection(const TField * p_Field) 
		: TVoxelSet(p_Field) 
		, m_SingleSelection(TCoord3())
		{};

	virtual VOXELSETTYPE getVoxelSetType() const { return VOXELSETTYPE_SELECTION; }

	string getName() const 
	{ 
		return wxString::Format("Selection (%u)", (unsigned int)m_MultipleSelection.size() ).ToStdString();  
	}

	void select(const TCoord3 & p)
	{
		m_SingleSelection = p;
		m_MultipleSelection.insert(p);
		notifyObservers();
	}

	void clear()
	{
		m_SingleSelection = TCoord3(-1,-1,-1);
		m_MultipleSelection.clear();
		notifyObservers();
	}

	bool erase(TCoord3 & p)
	{
		if(m_SingleSelection == p) m_SingleSelection = TCoord3(-1,-1,-1);
		TMultipleSelection::iterator jt = m_MultipleSelection.find(p);
		if(jt != m_MultipleSelection.end())
		{
			m_MultipleSelection.erase(jt);
			notifyObservers();
			return true;	
		}
		return false;
	}	

	void registerObserver(class TSelectionObserver * p_Observer)
	{
		m_Observers.insert( p_Observer );
	}
	
	TCoord3 getSingleSelection()
	{
		return m_SingleSelection;
	}
	int getSize() const { return m_MultipleSelection.size(); }


	virtual void initTraversal() const;
	virtual bool nextTraversal(TCoord3 & c) const;
	virtual void DrawGui(wxWindow * p_Window);

protected:
	TCoord3						m_SingleSelection;
	typedef	set<TCoord3>		TMultipleSelection;
	mutable TMultipleSelection			m_MultipleSelection;
	mutable TMultipleSelection::iterator it;
	set<class TSelectionObserver*> m_Observers;
	
	void notifyObservers();
};



class TSelectionObserver
{
public:
	TSelectionObserver(class TVoxelSetSelection * p_Selection) 
		: m_Selection(p_Selection)
	{
		m_Selection->registerObserver(this);
	}

	virtual void onSelectionChanged() = 0;
protected:
	class TVoxelSetSelection * m_Selection;
};




class TVoxelSetAll : public TVoxelSet
{
public:
	TVoxelSetAll(const TField * p_Field) : TVoxelSet(p_Field) {};

	// Accessors
	virtual void initTraversal() const { m_TraversalIdx = 0; }
	virtual bool nextTraversal(TCoord3 & Coord) const;

	virtual VOXELSETTYPE getVoxelSetType() const { return VOXELSETTYPE_ALL; }

	// Methods
	virtual void DrawGui(wxWindow * p_Window);

protected:
	mutable unsigned int m_TraversalIdx;
};

class TVoxelSetAllUsingIndexMapper : public TVoxelSet
{
public:
	TVoxelSetAllUsingIndexMapper(const TField * p_Field);

	// Accessors
	virtual void initTraversal() const { m_TraversalIdx = 0; }
	virtual bool nextTraversal(TCoord3 & Coord) const;

	virtual VOXELSETTYPE getVoxelSetType() const { return VOXELSETTYPE_ALL; }

	// Methods
	virtual void DrawGui(wxWindow * p_Window);

protected:
	mutable unsigned int m_TraversalIdx;
};


class TVoxelSubset : public TVoxelSet
{
public:
	// Constructors
	TVoxelSubset(const TField * p_Field) : TVoxelSet(p_Field) {};
	static shared_ptr<TVoxelSubset> constructByFilter(const class TField * p_Field, shared_ptr<TFilter> p_Filter);
	static shared_ptr<TVoxelSubset> constructByMask(const class TField * p_Field, const class TLayer * p_MaskLayer);

	// Accessors
	virtual void initTraversal() const { m_TraversalIdx = 0; }
	virtual bool nextTraversal(TCoord3 & c) const 
	{
		if(m_TraversalIdx >= m_VoxelPositions.size()) return false;
		c = m_VoxelPositions[m_TraversalIdx];
		m_TraversalIdx++;
		return true;
	}

	virtual VOXELSETTYPE getVoxelSetType() const { return VOXELSETTYPE_SUBSET; }

	// Methods
	virtual void DrawGui(wxWindow * p_Window);
	const TCoord3 * getFirstElement() const { return &m_VoxelPositions[0]; } 

protected:
	mutable unsigned int m_TraversalIdx;
	vector<TCoord3> m_VoxelPositions;
};



class TVoxelMultiSlice : public TVoxelSet
{
public:
	friend class TVoxelMultiSlicePanel;

	// Constructors
	TVoxelMultiSlice(const TField * p_Field, int p_Axis = 1, int p_Value = 1)
		: TVoxelSet(p_Field), m_Axis(p_Axis), m_Value(p_Value), m_ValueX(m_Value), m_ValueY(m_Value), m_ValueZ(m_Value) 
		, m_SliceCount(1)
	{
		m_Name = "Multislice";
	};

	// Accessors
	inline int getAxis() const { return m_Axis; }
	virtual VOXELSETTYPE getVoxelSetType() const { return VOXELSETTYPE_MULTISLICE; }

	// Methods
	virtual void initTraversal() const;
	virtual bool nextTraversal(TCoord3 & c) const;
	virtual void DrawGui(wxWindow * p_Window);

protected:
	mutable int m_TraversalIdx;
	int m_Axis;
	int m_Value;
	int m_ValueX;
	int m_ValueY;
	int m_ValueZ;
	int m_SliceCount;
};







class TVoxelMultiSlicePanel : public wxPanel
{
public:
	// Types
	enum { ID_SLICEAXISCHOICE, ID_SLICEAXISX, ID_SLICEAXISY, ID_SLICEAXISZ, ID_SLICECOUNT };

	TVoxelMultiSlicePanel(wxWindow* parent, wxWindowID id, TVoxelMultiSlice * p_VoxelSlice, const wxPoint& pos = wxDefaultPosition, const wxSize& size = wxDefaultSize, long style = wxTAB_TRAVERSAL);

private:
	class wxRadioButton *			m_UseAaSliceX;
	class wxRadioButton *			m_UseAaSliceY;
	class wxRadioButton *			m_UseAaSliceZ;
	wxPanel *				m_AaSliceX;
	wxPanel *				m_AaSliceY;
	wxPanel *				m_AaSliceZ;
	wxPanel *				m_SliceCountControl;
	TVoxelMultiSlice *			m_VoxelSlice;

	void OnSliceAxisChoice(wxCommandEvent & event);
	void OnChangeX(wxCommandEvent & event);
	void OnChangeY(wxCommandEvent & event);
	void OnChangeZ(wxCommandEvent & event);
	void OnChangeSliceCount(wxCommandEvent & event);
};



#endif
