/***************************************************************************
 *   Copyright (C) 2009 by Hessel Hoogendorp                               *
 *   bugs.ccc@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


// ----------------------------------------------------------------------------
// Includes
// ----------------------------------------------------------------------------
#include "ccc.h"
#include "ccc_settings.h"
#include "ccc_global_function_repository.h"
#include "ccc_call_graph.h"
#include "ccc_call_graph_printer_dot.h"
#include "ccc_call_graph_printer_tlp.h"
#include "ccc_call_graph_printer_sql.h"
#include "ccc_verbose_print.h"
#include "ccc_debug.h"
#include <libcci/cci.h>


// ----------------------------------------------------------------------------
// Static member declarations
// ----------------------------------------------------------------------------
CCC_GlobalFunctionRepository CCC_Constructor::m_globalFunctionRepository;
FunctionCallVector CCC_Constructor::m_initializingCalls;
FunctionCallVector CCC_Constructor::m_finalizingCalls;


// ----------------------------------------------------------------------------
// Call graph construction
// ----------------------------------------------------------------------------
void CCC_Constructor::ConstructCallGraph()
{
	StringVector* pInputFileNames = CCC_ConstructorSettings::GetInputFileNames();
	StringVector::iterator itInputFileName = pInputFileNames->begin();

	VERBOSE_PRINT("Reading " << pInputFileNames->size() << " call information files.");
	while(itInputFileName != pInputFileNames->end())
	{
		ProcessCallInfoFile(*itInputFileName);
		itInputFileName++;
	}

	VERBOSE_PRINT("Resolving functions overriding virtuals.");
	ResolveFunctionsOverridingVirtuals();

	VERBOSE_PRINT("Validating.");
	Validate();

	PrintGraph();

	// Clean up.
	m_globalFunctionRepository.Finalize();
}

void CCC_Constructor::ProcessCallInfoFile(std::string szFileName)
{
	// Read the local functions stored in the specified file.
	LocalFunctionMap localFunctions;
	switch(CCC_ConstructorSettings::GetInputType())
	{
		case CCC_IT_Binary:
		{
			CCI_DeserializerBin deserializer(szFileName);
			deserializer.Deserialize(localFunctions, m_initializingCalls, m_finalizingCalls);
			break;
		}
		case CCC_IT_Xml:
		{
			CCI_DeserializerXml deserializer(szFileName);
			deserializer.Deserialize(localFunctions, m_initializingCalls, m_finalizingCalls);
			break;
		}
	}

	// Add the read local functions to the global function repository.
	m_globalFunctionRepository.AddLocalFunctionMap(&localFunctions);
}

void CCC_Constructor::ResolveFunctionsOverridingVirtuals()
{
	// Retrieve all global functions.
 	GlobalFunctionMap* pGlobalFunctions = m_globalFunctionRepository.GetGlobalFunctions();

	// Iterate over all global functions.
	GlobalFunctionMap::iterator itGlobalFunction = pGlobalFunctions->begin();
	while(itGlobalFunction != pGlobalFunctions->end())
	{
		CCC_GlobalFunction* pOverridingGlobalFunction = itGlobalFunction->second;

		// Iterate over all CCI_LocalFunctions in this global function.
		LocalFunctionVector* pLocalFunctions = pOverridingGlobalFunction->GetLocalFunctions();
		LocalFunctionVector::iterator itLocalFunction = pLocalFunctions->begin();
		while(itLocalFunction != pLocalFunctions->end())
		{
			CCI_LocalFunction* pLocalFunction = *itLocalFunction;

			// Check whether this is a virtual method.
			if(pLocalFunction->IsFlagVirtual())
			{
				// Retrieve the signatures of the methods it overrides.
				StringVector* pOverriddenMethodSignatures = pLocalFunction->GetOverriddenMethodSignatures();

				// For each method this method overrides, add a link from the
				// overriden method to this method.
				StringVector::iterator itOverriddenMethodSignature = pOverriddenMethodSignatures->begin();
				while(itOverriddenMethodSignature != pOverriddenMethodSignatures->end())
				{
					// Retrieve the global function belonging to the signature
					// of the overridden method.
					CCC_GlobalFunction* pOverriddenGlobalFunction = m_globalFunctionRepository.GetGlobalFunction(*itOverriddenMethodSignature);

					// NOTE: The assert below triggers when an overridden virtual is
					// filtered out, but one or more of its overriding methods are not.
					// In such a case, the overriding methods still have a signature
					// reference to the filtered method, which is what we encounter here.
					// However, the callinfo extractor should remove these references, so
					// this is actually an error in the callinfo extractor.
					cci_assert_msg_allow(pOverriddenGlobalFunction != NULL, "Signature '" << (*itOverriddenMethodSignature) << "' not found in Global Function Repository.");

					if(pOverriddenGlobalFunction != NULL)
						pOverriddenGlobalFunction->AddOverridingMethod(pOverridingGlobalFunction->GetSignature());

					itOverriddenMethodSignature++;
				}
			}

			itLocalFunction++;
		}

		itGlobalFunction++;
	}
}

void CCC_Constructor::Validate()
{
	// Check whether validation can be skipped.
	if(!CCC_ConstructorSettings::GetValidateInput())
		return;

	//
	// Check whether all called functions are known.
	//
	
	// Retrieve all global functions.
	GlobalFunctionMap* pGlobalFunctions = m_globalFunctionRepository.GetGlobalFunctions();

	// Iterate over all global functions.
	GlobalFunctionMap::iterator itGlobalFunction = pGlobalFunctions->begin();
	while(itGlobalFunction != pGlobalFunctions->end())
	{
		CCC_GlobalFunction* pOverridingGlobalFunction = itGlobalFunction->second;

		// Iterate over all CCI_LocalFunctions in this global function.
		LocalFunctionVector* pLocalFunctions = pOverridingGlobalFunction->GetLocalFunctions();
		LocalFunctionVector::iterator itLocalFunction = pLocalFunctions->begin();
		while(itLocalFunction != pLocalFunctions->end())
		{
			CCI_LocalFunction* pLocalFunction = *itLocalFunction;

			// Check whether this CCI_LocalFunction has a function definition.
			if(pLocalFunction->HasFunctionDefinition())
			{
				// Retrieve the function definition belonging to this CCI_LocalFunction.
				CCI_FunctionDefinition* pFunctionDefinition = pLocalFunction->GetFunctionDefinition();

				// Retrieve the vector of function calls that are made in this function definition.
				FunctionCallVector* pFunctionCalls = pFunctionDefinition->GetFunctionCalls();

				// Iterate over all function calls to validate them individually.
				FunctionCallVector::iterator itFunctionCall = pFunctionCalls->begin();
				while(itFunctionCall != pFunctionCalls->end())
				{
					CCI_FunctionCall* pFunctionCall = *itFunctionCall;

					// Retrieve the signature of the callee.
					std::string calleeSignature = pFunctionCall->GetSignature();

					if(pFunctionCall->IsTypePointer())
					{
						// This is a call through a pointer to function or
						// a call through a pointer to member.

						// Validate that this signature has an entry in the pointer signature
						// mapping.
						if(!m_globalFunctionRepository.ContainsGlobalFunctionWithPointerSignature(calleeSignature))
						{
							cci_assert_msg_allow
							(
								false,
								std::endl <<
								"\t{"                                                                                      << std::endl <<
								"\t\tProblem  : Function not found in the 'pointer signature-to-global function mapping'." << std::endl <<
								"\t\tCause    : Call candidates may have been filtered out during extraction."             << std::endl <<
							    "\t\tSolution : Add '-fc-filter-defn-conservative' to the command line of ccie."           << std::endl <<
								"\t\tSignature: " << calleeSignature                                                       << std::endl <<
								"\t}"
							);
						}
					}
					else
					{
						// This is a "plain" call. That is, not a call through a pointer.

						// Validate that a global function with this signature exists.
						if(!m_globalFunctionRepository.ContainsGlobalFunctionWithSignature(calleeSignature))
						{
							cci_assert_msg_allow
							(
								false,
								std::endl <<
								"\t{"                                                                              << std::endl <<
								"\t\tProblem  : Function not found in the 'signature-to-global function mapping'." << std::endl <<
								"\t\tCause    : Possible call to compiler-specific built-in function."             << std::endl <<
								"\t\tSignature: " << calleeSignature                                               << std::endl <<
								"\t}"
							);
						}
					}

					itFunctionCall++;
				}
			}

			itLocalFunction++;
		}

		itGlobalFunction++;
	}
}

void CCC_Constructor::PrintGraph()
{
	// Build the call graph.
	VERBOSE_PRINT("Creating graph.");
	CCC_CallGraph* pCallGraph = CCC_CallGraph::Create(&m_globalFunctionRepository, &m_initializingCalls, &m_finalizingCalls);

	// Validate the call graph.
	VERBOSE_PRINT("Validating graph.");
	pCallGraph->Validate();

	// Print the call graph.
	VERBOSE_PRINT("Serializing graph.");
	switch(CCC_ConstructorSettings::GetOutputFormat())
	{
		case CCC_OF_Dot:
		{
			CCC_CallGraphPrinterDot dotPrinter(CCC_ConstructorSettings::GetOutputFileName());
			dotPrinter.Print(pCallGraph);
			break;
		}
		case CCC_OF_Tulip:
		{
			CCC_CallGraphPrinterTlp tlpPrinter(CCC_ConstructorSettings::GetOutputFileName());
			tlpPrinter.Print(pCallGraph);
			break;
		}
		case CCC_OF_Sqlite:
		{
			CCC_CallGraphPrinterSql sqlPrinter(CCC_ConstructorSettings::GetOutputFileName());
			sqlPrinter.Print(pCallGraph);
			break;
		}
	}
	
	delete pCallGraph;
}


//-----------------------------------------------------------------------------
// Debugging
//-----------------------------------------------------------------------------
void CCC_Constructor::Print()
{
	std::cout << "Call graph";
	std::cout << std::endl << "{";
		m_globalFunctionRepository.Print("\t");

		std::cout << std::endl << "\tInitializing calls";
		std::cout << std::endl << "\t{";
		FunctionCallVector::iterator itInitializingCall = m_initializingCalls.begin();
		while(itInitializingCall != m_initializingCalls.end())
		{
			(*itInitializingCall)->PrintOut("\t\t");
			itInitializingCall++;
		}
		std::cout << std::endl << "\t}" << std::endl;
		
		std::cout << std::endl << "\tFinalizing calls";
		std::cout << std::endl << "\t{";
		FunctionCallVector::iterator itFinalizingCall = m_finalizingCalls.begin();
		while(itFinalizingCall != m_finalizingCalls.end())
		{
			(*itFinalizingCall)->PrintOut("\t\t");
			itFinalizingCall++;
		}
		std::cout << std::endl << "\t}" << std::endl;
	std::cout << std::endl << "}" << std::endl;
}
