/***************************************************************************
 *   Copyright (C) 2009 by Hessel Hoogendorp                               *
 *   bugs.ccc@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


// ----------------------------------------------------------------------------
// Includes
// ----------------------------------------------------------------------------
#include "ccc_call_graph.h"
#include "ccc_full_call_graph.h"
#include "ccc_partial_call_graph.h"
#include "ccc_global_function_repository.h"
#include "ccc_call_graph_node.h"
#include "ccc_call_graph_edge.h"
#include "ccc_settings.h"
#include "ccc_verbose_print.h"
#include <sstream>


// ----------------------------------------------------------------------------
// Static initializers
// ----------------------------------------------------------------------------
int GraphNode::m_iNextFreeId = 1;
int GraphEdge::m_iNextFreeId = 1;


// ----------------------------------------------------------------------------
// Construction & destruction
// ----------------------------------------------------------------------------
CCC_CallGraph::CCC_CallGraph()
{
	m_iNextFreeCallCandidateSetId = 0;
	m_plfRootFunctionNode         = NULL;
	m_pfcCallToMainEntryPoint     = NULL;

	DirectoryGraphNode* pgnRootDirectoryNode = new DirectoryGraphNode("/");
	m_nodes.push_back(pgnRootDirectoryNode);
	m_nodeFromDirectory["/"] = pgnRootDirectoryNode;
}

CCC_CallGraph::~CCC_CallGraph()
{
	VERBOSE_PRINT("Cleaning up call graph.");
	// Delete the CCI_LocalFunction of the fictional Root node, if it exists.
	if(m_plfRootFunctionNode != NULL)
	{
		delete m_plfRootFunctionNode;
		m_plfRootFunctionNode = NULL;
	}

	// Delete the CCI_FunctionCall of the fictional Root node to the main entry
	// point, if it exists.
	if(m_pfcCallToMainEntryPoint != NULL)
	{
		delete m_pfcCallToMainEntryPoint;
		m_pfcCallToMainEntryPoint = NULL;
	}
	
	// Delete all the nodes of the graph.
	std::vector<GraphNode*>::iterator itGraphNode = m_nodes.begin();
	while(itGraphNode != m_nodes.end())
	{
		delete *itGraphNode;
		itGraphNode++;
	}
	m_nodes.clear();
	
	// Delete all the edges of the graph.
	std::vector<GraphEdge*>::iterator itGraphEdge = m_edges.begin();
	while(itGraphEdge != m_edges.end())
	{
		delete *itGraphEdge;
		itGraphEdge++;
	}
	m_edges.clear();	
	VERBOSE_PRINT("Done cleaning up.");
}

CCC_CallGraph* CCC_CallGraph::Create(CCC_GlobalFunctionRepository* pGlobalFunctionRepository, FunctionCallVector* pInitializingCalls, FunctionCallVector* pFinalizingCalls)
{
	cci_assert(pGlobalFunctionRepository != NULL);
	cci_assert(pInitializingCalls        != NULL);
	cci_assert(pFinalizingCalls          != NULL);

	std::string szEntryPoint = CCC_ConstructorSettings::GetEntryPointName();
	CCC_CallGraph* pCallGraph;
	if(szEntryPoint == "")
	{
		pCallGraph = new CCC_FullCallGraph();
	}
	else
	{
		pCallGraph = new CCC_PartialCallGraph(szEntryPoint);
	}

	pCallGraph->Build(pGlobalFunctionRepository, pInitializingCalls, pFinalizingCalls);
	return pCallGraph;
}

void CCC_CallGraph::Build(CCC_GlobalFunctionRepository* pGlobalFunctionRepository,
						  FunctionCallVector* pInitializingCalls,
		   				  FunctionCallVector* pFinalizingCalls)
{
	m_pGlobalFunctionRepository = pGlobalFunctionRepository;
	m_pInitializingCalls        = pInitializingCalls;
	m_pFinalizingCalls          = pFinalizingCalls;

	m_pMainEntryPoint = GetMainEntryPoint();
	InsertNodes();
	InsertCallEdges();

	VERBOSE_PRINT("Call graph has " << m_nodes.size() << " nodes and " << m_edges.size() << " edges.");
}


// ----------------------------------------------------------------------------
// Node insertion
// ----------------------------------------------------------------------------
void CCC_CallGraph::InsertFunctionGraphNode(CCI_LocalFunction* pLocalFunction)
{
	// If pLocalFunction has static linkage, and if there is already a function
	// node in the graph that satisfies the following properties, then merge the
	// supplied function with that node.
	//
	// 1. Has an equal definition file name.
	// 2. Has an equal position within the definition file.
	// 3. Has an equal unmangled FQN.

	if(pLocalFunction->IsFlagStaticLinkage())
	{
		std::string szMangledSignature   = pLocalFunction->GetSignature();
		std::string szUnmangledSignature = UnmangleSignature(szMangledSignature);

		CCI_SourceLoc* pSourceLoc;
		if(pLocalFunction->HasFunctionDefinition())
		{
			pSourceLoc = pLocalFunction->GetFunctionDefinition()->GetSourceLoc();
		}
		else
		{
			pSourceLoc = pLocalFunction->GetFunctionDeclaration()->GetSourceLoc();
		}

		std::stringstream ss;
		ss << pSourceLoc->GetFileName() << "\n" << pSourceLoc->GetLine() << "\n" << pSourceLoc->GetColumn() << "\n" << szUnmangledSignature;
		std::string szMergingFQN = ss.str();

		if (m_nodeFromUnmangledFQN.find(szMergingFQN) == m_nodeFromUnmangledFQN.end())
		{
			// A node with these properties does not yet exist, so create it now.
			FunctionGraphNode* pFunctionGraphNode = new FunctionGraphNode(pLocalFunction);
			m_nodes.push_back(pFunctionGraphNode);
			m_nodeFromFunction[pLocalFunction] = pFunctionGraphNode;

			// Make sure the new Function node is properly contained by either a class or a file.
			GraphNode* pContainerNode = GetContainerNode(pLocalFunction);

			// Insert an edge from the Container node to the Function node.
			m_edges.push_back(new ContainsGraphEdge(pContainerNode, pFunctionGraphNode));

			// Finally, register this shared node in the respective mapping.
			m_nodeFromUnmangledFQN[szMergingFQN] = pFunctionGraphNode;
		}
		else
		{
			// A node with these properties already exists, 

			// Retrieve the existing function node.
			FunctionGraphNode* pFunctionGraphNode = m_nodeFromUnmangledFQN[szMergingFQN];

			// Insert a new mapping from this local function to the existing graph node.
			m_nodeFromFunction[pLocalFunction] = pFunctionGraphNode;

			// Finally, register this shared node in the respective mapping.
			m_nodeFromUnmangledFQN[szMergingFQN] = pFunctionGraphNode;
		}
	}
	else
	{
		// THE ORIGINAL IMPLEMENTATION

		// Create and insert a Function node for this local function.
		FunctionGraphNode* pFunctionGraphNode = new FunctionGraphNode(pLocalFunction);
		m_nodes.push_back(pFunctionGraphNode);
		m_nodeFromFunction[pLocalFunction] = pFunctionGraphNode;

		// Make sure the new Function node is properly contained by either a class or a file.
		GraphNode* pContainerNode = GetContainerNode(pLocalFunction);

		// Insert an edge from the Container node to the Function node.
		m_edges.push_back(new ContainsGraphEdge(pContainerNode, pFunctionGraphNode));

		// END ORIGINAL IMPLEMENTATION
	}
}

GraphNode* CCC_CallGraph::GetContainerNode(CCI_LocalFunction* pLocalFunction)
{
	if(pLocalFunction->HasClassName())
	{
		// Retrieve the name of this function's class.
		std::string szClassName = pLocalFunction->GetClassName();

		// If a node for this class does not yet exists, we need to create it.
		// Otherwise, we can simply use the existing Class node.
		std::map<std::string, ClassGraphNode*>::iterator itClassGraphNode = m_nodeFromClass.find(szClassName);
		if(itClassGraphNode != m_nodeFromClass.end())
			return itClassGraphNode->second;

		// If a node for this class does not exist yet, create it, insert it and
		// make sure it is properly contained by the file in which it is defined.
		ClassGraphNode* pClassGraphNode = new ClassGraphNode(szClassName);
		m_nodes.push_back(pClassGraphNode);
		m_nodeFromClass[szClassName] = pClassGraphNode;

		// Retrieve the File node that contains this Class node.
		FileGraphNode* pFileGraphNode = GetFileNode(pLocalFunction->GetFunctionDeclaration()->GetSourceLoc());

		// Insert an edge from the File node to the Class node.
		ContainsGraphEdge* pFileGraphEdge = new ContainsGraphEdge(pFileGraphNode, pClassGraphNode);
		m_edges.push_back(pFileGraphEdge);
		m_edgeFromClass[szClassName] = pFileGraphEdge;

		// Return the newly created Class node as the container node.
		return pClassGraphNode;
	}
	else
	{
		// If this function has a definition, use the definition's file name. Otherwise,
		// use the declaration's file name.
		CCI_SourceLoc* pSourceLoc = (pLocalFunction->HasFunctionDefinition())
									? pLocalFunction->GetFunctionDefinition()->GetSourceLoc()
									: pLocalFunction->GetFunctionDeclaration()->GetSourceLoc();

		// Retrieve the File node that is to contain this function and use it
		// as the container node.
		return GetFileNode(pSourceLoc);
	}
}

ClassGraphNode* CCC_CallGraph::GetClassNode(std::string szClassName)
{
	// If the class node already exists, simply return a pointer to it.
	std::map<std::string, ClassGraphNode*>::iterator itClassNode = m_nodeFromClass.find(szClassName);
	if(itClassNode != m_nodeFromClass.end())
		return itClassNode->second;
	
	// If the class node does not yet exist, create and insert it, and then
	// return a pointer to the newly created node.
	ClassGraphNode* pClassGraphNode = new ClassGraphNode(szClassName);
	m_nodes.push_back(pClassGraphNode);
	m_nodeFromClass[szClassName] = pClassGraphNode;
	return pClassGraphNode;
}

FileGraphNode* CCC_CallGraph::GetFileNode(CCI_SourceLoc* pSourceLoc)
{
	// Retrieve the filename belonging to this SourceLoc.
	std::string szFileName = pSourceLoc->GetFileName();

	// If the File node already exists, simply return a pointer to it.
	std::map<std::string, FileGraphNode*>::iterator itFileNode = m_nodeFromFile.find(szFileName);
	if(itFileNode != m_nodeFromFile.end())
		return itFileNode->second;

	// If the File node does not yet exist, create it now.
	FileGraphNode* pFileGraphNode = new FileGraphNode(pSourceLoc, pSourceLoc->GetFileDisplayName());
	m_nodes.push_back(pFileGraphNode);
	m_nodeFromFile[szFileName] = pFileGraphNode;

	// Make sure all parent directories of this File node have a corresponding Folder node.
	std::vector<std::string> fullDirNames;
	std::vector<std::string> dispDirNames;
	pSourceLoc->SplitFilePath(fullDirNames, dispDirNames);

	GraphNode* pChildGraphNode = pFileGraphNode;
	for(int i = fullDirNames.size() - 1; i >= 0; i--)
	{
		// If the Directory node for this directory already exists, then the Nodes
		// for its parent directories exist as well and we can terminate.
		std::map<std::string, DirectoryGraphNode*>::iterator itDirectory = m_nodeFromDirectory.find(fullDirNames[i]);
		if(itDirectory != m_nodeFromDirectory.end())
		{
			// Create an edge from the Directory node to the current child node.
			m_edges.push_back(new ContainsGraphEdge(itDirectory->second, pChildGraphNode));
			break;
		}

		// Create a new Directory node for this directory.
		DirectoryGraphNode* pDirectoryGraphNode = new DirectoryGraphNode(dispDirNames[i]);
		m_nodes.push_back(pDirectoryGraphNode);
		m_nodeFromDirectory[fullDirNames[i]] = pDirectoryGraphNode;

		// Create an edge from the Directory node to the current child node.
		m_edges.push_back(new ContainsGraphEdge(pDirectoryGraphNode, pChildGraphNode));
		
		// Make the new Directory node the current child node.
		pChildGraphNode = pDirectoryGraphNode;
	}

	return pFileGraphNode;
}

std::string CCC_CallGraph::UnmangleSignature(std::string szMangledSignature)
{
	// Retrieve the position in the signature string where the mangling starts.
	std::string::size_type pos = szMangledSignature.find("{sl:", 0);
	if (pos == std::string::npos)
	{
		// This signature does not appear to be mangled at all.
		return szMangledSignature;
	}

	// Remove the mangling from the signature string and return the result.
	return szMangledSignature.substr(0, pos);
}


// ----------------------------------------------------------------------------
// Edge insertion
// ----------------------------------------------------------------------------
void CCC_CallGraph::InsertCallEdges()
{
	// Process all the local functions that were added to the graph.
	std::map<CCI_LocalFunction*, FunctionGraphNode*>::iterator itLocalFunction = m_nodeFromFunction.begin();
	while(itLocalFunction != m_nodeFromFunction.end())
	{
		InsertCallEdges(itLocalFunction->first);
		itLocalFunction++;
	}

	// Process the initializing and finalizing function calls.
	if(CCC_ConstructorSettings::GetIncludeInitFinit())
	{
		if(InsertRootFunctionNodeAndCallEdgeToEntryPoint())
		{
			ProcessInitializingFunctionCalls();
			ProcessFinalizingFunctionCalls();
		}
	}
}

void CCC_CallGraph::InsertCallEdges(CCI_LocalFunction* pLocalFunction)
{
	// If this function does not have a definition, we do not have to look for
	// function calls, since it is obviously not going to make any.
	if(!pLocalFunction->HasFunctionDefinition())
		return;

	// Retrieve the function calls that are made in this function definition.
	FunctionCallVector* pFunctionCalls = pLocalFunction->GetFunctionDefinition()->GetFunctionCalls();

	// Iterate over each function call and retrieve its set of call candidates.
	FunctionCallVector::iterator itFunctionCall = pFunctionCalls->begin();
	while(itFunctionCall != pFunctionCalls->end())
	{
		InsertCallEdges(pLocalFunction, *itFunctionCall);
		itFunctionCall++;
	}
}

void CCC_CallGraph::InsertCallEdges(CCI_LocalFunction* pSourceNode, CCI_FunctionCall* pFunctionCall)
{
	GlobalFunctionVector callCandidates;
	GetCallCandidates(pFunctionCall, callCandidates);
	
	// Iterate over each call candidate of this function call, and
	// decide what to do with it.
	GlobalFunctionVector::iterator itCallCandidate = callCandidates.begin();
	while(itCallCandidate != callCandidates.end())
	{
		CCC_GlobalFunction* pGlobalFunction = *itCallCandidate;

		LocalFunctionVector localFunctions;
		GetPreferredLocalFunctions(pGlobalFunction, localFunctions);
		//GetPreferredLocalFunctionsForFunctionCall(pGlobalFunction, pFunctionCall, localFunctions);

		LocalFunctionVector::iterator itLocalFunction = localFunctions.begin();
		while(itLocalFunction != localFunctions.end())
		{
			// Insert an edge for this preferred local function.
			FunctionGraphNode* pSourceGraphNode      = m_nodeFromFunction[pSourceNode];
			FunctionGraphNode* pDestinationGraphNode = m_nodeFromFunction[*itLocalFunction];
			CallsGraphEdge* pGraphEdge = new CallsGraphEdge(pSourceGraphNode, pDestinationGraphNode, pFunctionCall, callCandidates.size(), m_iNextFreeCallCandidateSetId);
			m_edges.push_back(pGraphEdge);
			m_edgeFromCall[pFunctionCall] = pGraphEdge;

			itLocalFunction++;
		}

		itCallCandidate++;
	}

	// All of the call candidates for this function call have been associated with
	// the same id. It is time to increase the next free id, so it can be used to
	// form the next set.
	m_iNextFreeCallCandidateSetId++;
}

bool CCC_CallGraph::InsertRootFunctionNodeAndCallEdgeToEntryPoint()
{
	// First, find the signature and the global function corresponding to the
	// 'main' function.
	if(m_pMainEntryPoint == NULL)
		return false;

	// Create a CCI_LocalFunction object for the fictional Root node and create
	// a CCI_FunctionCall object for the call from the fictional Root to the main
	// entry point. The StringVector is a required parameter of the ctor of the
	// CCI_LocalFunction.
	StringVector dummy;
	if((m_plfRootFunctionNode = new CCI_LocalFunction("Root", "Root", "Root", "", new CCI_FunctionDeclaration(new CCI_SourceLoc("Root", 0, 0, "")), NULL, CCI_LFF_PUBLIC, &dummy)) == NULL) return false;
	if((m_pfcCallToMainEntryPoint = new CCI_FunctionCall(CCI_FCT_DirectFunction, new CCI_SourceLoc("Root", 0, 0, ""), "Root")) == NULL) return false;

	// Next, insert a fictive Root node that will:
	// 1. Call all initializers.
	// 2. Call 'main'.
	// 3. Call all finalizers.
	FunctionGraphNode* pGraphRootNode = new FunctionGraphNode(m_plfRootFunctionNode);
	m_nodes.push_back(pGraphRootNode);
	m_nodeFromFunction[m_plfRootFunctionNode] = pGraphRootNode;

	// Then, retrieve all the preferred local functions of the 'main' global
	// function we just found.
	LocalFunctionVector localFunctions;
	GetPreferredLocalFunctions(m_pMainEntryPoint, localFunctions);

	// Use the first local function to retrieve the name of the file containing
	// the entry-point function.
	CCI_LocalFunction* pMainFunction = localFunctions[0];
	CCI_SourceLoc* pMainSourceLoc = (pMainFunction->HasFunctionDefinition())
	                                ? pMainFunction->GetFunctionDefinition()->GetSourceLoc()
	                                : pMainFunction->GetFunctionDeclaration()->GetSourceLoc();
	FileGraphNode* pMainFileNode = GetFileNode(pMainSourceLoc);

	// Insert a containment edge from the entry-point file to the fictive Root node.
	ContainsGraphEdge* pContainsGraphEdge = new ContainsGraphEdge(pMainFileNode, pGraphRootNode);
	m_edges.push_back(pContainsGraphEdge);

	// Finally, create edges from the fictional 'Root' node to each entry-point
	// local function.
	LocalFunctionVector::iterator itLocalFunction = localFunctions.begin();
	while(itLocalFunction != localFunctions.end())
	{
		FunctionGraphNode* pSourceNode      = m_nodeFromFunction[m_plfRootFunctionNode];
		FunctionGraphNode* pDestinationNode = m_nodeFromFunction[*itLocalFunction];

		CallsGraphEdge* pGraphEdge = new CallsGraphEdge(pSourceNode, pDestinationNode, m_pfcCallToMainEntryPoint, 1, m_iNextFreeCallCandidateSetId);
		m_edges.push_back(pGraphEdge);
		m_edgeFromCall[m_pfcCallToMainEntryPoint] = pGraphEdge;

		itLocalFunction++;
	}

	// Increase the next free call candidate set id, since we just used this one.
	m_iNextFreeCallCandidateSetId++;

	return true;
}

void CCC_CallGraph::ProcessInitializingFunctionCalls()
{
	FunctionCallVector::iterator itFunctionCall = m_pInitializingCalls->begin();
	while(itFunctionCall != m_pInitializingCalls->end())
	{
		InsertCallEdges(m_plfRootFunctionNode, *itFunctionCall);
		itFunctionCall++;
	}
}

void CCC_CallGraph::ProcessFinalizingFunctionCalls()
{
	FunctionCallVector::iterator itFunctionCall = m_pFinalizingCalls->begin();
	while(itFunctionCall != m_pFinalizingCalls->end())
	{
		InsertCallEdges(m_plfRootFunctionNode, *itFunctionCall);
		itFunctionCall++;
	}
}


// ----------------------------------------------------------------------------
// Helpers
// ----------------------------------------------------------------------------
void CCC_CallGraph::GetCallCandidates(CCI_FunctionCall* pFunctionCall, GlobalFunctionVector & callCandidates)
{
	// Make sure the vector of call candidate signatures is empty.
	callCandidates.clear();
	std::string szSignature = pFunctionCall->GetSignature();

	// Compose the set of call candidates, depending on what type of function
	// call this is.
	if(pFunctionCall->IsTypeDirect())
	{
		// There is only one call candidate, since this is a 'direct' call.
		CCC_GlobalFunction* pGlobalFunction = m_pGlobalFunctionRepository->GetGlobalFunction(szSignature);
		if(pGlobalFunction != NULL)
		{
			callCandidates.push_back(pGlobalFunction);
		}
	}
	else if(pFunctionCall->IsTypeObject())
	{
		// The function with the signature of this call is a call candidate.
		CCC_GlobalFunction* pGlobalFunction = m_pGlobalFunctionRepository->GetGlobalFunction(szSignature);
		if(pGlobalFunction != NULL)
		{
			callCandidates.push_back(pGlobalFunction);

			// And so are all of its overriders.
			StringVector* pOverriders = pGlobalFunction->GetOverridingMethods();
			StringVector::iterator itOverrider = pOverriders->begin();
			while(itOverrider != pOverriders->end())
			{
				callCandidates.push_back(m_pGlobalFunctionRepository->GetGlobalFunction(*itOverrider));
				itOverrider++;
			}
		}
	}
	else if(pFunctionCall->IsTypePointer())
	{
		// All functions that have a matching pointer signature are a call
		// candidate. If a matching function is a method and virtual, then all
		// methods that override it are also a call candidate.
		GlobalFunctionVector* pGlobalFunctions = m_pGlobalFunctionRepository->GetGlobalFunctionVector(szSignature);

		if (pFunctionCall->IsTypePointerToMember())
		{
			// All functions that have a matching pointer signature are a call
			// candidate, but also all functions that override these functions
			// are call candidates.
			GlobalFunctionVector::iterator itPointerMatch = pGlobalFunctions->begin();
			while (itPointerMatch != pGlobalFunctions->end())
			{
				// The function with this signature is a call candidate.
				// THERE IS NO NEED FOR STATIC LINKAGE CHECK HERE
				callCandidates.push_back(*itPointerMatch);

				// Also, all functions that override the function with this signature
				// are a call candidate.
				StringVector* pOverriders = (*itPointerMatch)->GetOverridingMethods();
				StringVector::iterator itOverrider = pOverriders->begin();
				while(itOverrider != pOverriders->end())
				{
					// DO NOT CHECK FOR STATIC LINKAGE HERE, SINCE OVERRIDERS ARE
					// VALID CALL CANDIDATES EVEN IF THEY HAVE STATIC LINKAGE.
					callCandidates.push_back(m_pGlobalFunctionRepository->GetGlobalFunction(*itOverrider));
					itOverrider++;
				}

				// Advance to the next global function.
				itPointerMatch++;
			}
		}
		else
		{
			// All functions that have a matching pointer signature are a call
			// candidate.
			// DO NOT CHECK FOR STATIC LINKAGE HERE, SINCE FUNCTIONS WITH MATCHING
			// POINTER NAMES ARE VALID CALL CANDIDATES EVEN IF THEY HAVE STATIC LINKAGE.
			GlobalFunctionVector* pGlobalFunctions = m_pGlobalFunctionRepository->GetGlobalFunctionVector(szSignature);
			if(pGlobalFunctions != NULL)
				callCandidates.insert(callCandidates.begin(), pGlobalFunctions->begin(), pGlobalFunctions->end());
		}


		/*
		// All functions that have a matching pointer signature are a call
		// candidate.
		GlobalFunctionVector* pGlobalFunctions = m_pGlobalFunctionRepository->GetGlobalFunctionVector(szSignature);
		if(pGlobalFunctions != NULL)
			callCandidates.insert(callCandidates.begin(), pGlobalFunctions->begin(), pGlobalFunctions->end());
		*/
	}
}

void CCC_CallGraph::GetPreferredLocalFunctions(CCC_GlobalFunction* pGlobalFunction, LocalFunctionVector & localFunctions)
{
	// Make sure the vector is empty.
	localFunctions.clear();

	// Retrieve the vector of local functions from the global function.
	LocalFunctionVector* pLocalFunctions = pGlobalFunction->GetLocalFunctions();

	// Collect all global functions that have a definition.
	LocalFunctionVector::iterator itLocalFunction = pLocalFunctions->begin();
	while(itLocalFunction != pLocalFunctions->end())
	{
		// Check whether this CCI_LocalFunction has a function definition
		// attached to it.
		if((*itLocalFunction)->HasFunctionDefinition())
		{
			localFunctions.push_back(*itLocalFunction);
		}

		itLocalFunction++;
	}

	// If (a) local function(s) with a definition has/have been found, that/those
	// is/are the preferred local function(s).
	if(localFunctions.size() > 0)
		return;

	// If no local function with a definition has been found, add the first
	// local function we can find and return.
	itLocalFunction = pLocalFunctions->begin();
	if(itLocalFunction != pLocalFunctions->end())
	{
		localFunctions.push_back(*itLocalFunction);
	}
}
/*
void CCC_CallGraph::GetPreferredLocalFunctionsForFunctionCall(CCC_GlobalFunction* pGlobalFunction, CCI_FunctionCall* pFunctionCall, LocalFunctionVector & localFunctions)
{
	// Make sure the vector is empty.
	localFunctions.clear();

	// Retrieve the vector of local functions from the global function.
	LocalFunctionVector* pLocalFunctions = pGlobalFunction->GetLocalFunctions();

	// Collect all global functions that have a definition.
	LocalFunctionVector::iterator itLocalFunction = pLocalFunctions->begin();
	while(itLocalFunction != pLocalFunctions->end())
	{
		CCI_LocalFunction* pLocalFunction = *itLocalFunction;

		// Check whether this CCI_LocalFunction has a function definition
		// attached to it.
		if(pLocalFunction->HasFunctionDefinition())
		{
			// Now check whether the function call is of type 'DirectFunction'.
			// If so and the local function has static linkage, then the local
			// function must have been defined in the same translation unit in
			// which the function call is made.
			if (pFunctionCall->IsTypeDirect() && pLocalFunction->IsFlagStaticLinkage())
			{
				std::string szFunctionTranslationUnit = pLocalFunction->GetFunctionDefinition()->GetSourceLoc()->GetTranslationUnit();
				std::string szFunctionCallTranslationUnit = pFunctionCall->GetSourceLoc()->GetTranslationUnit();

				// This local function is a call candidate only if it is defined
				// in the same translation unit in which the function call is made.
				if (szFunctionTranslationUnit != szFunctionCallTranslationUnit)
				{
					itLocalFunction++;
					continue;
				}
			}

			localFunctions.push_back(*itLocalFunction);
		}

		itLocalFunction++;
	}

	// If (a) local function(s) with a definition has/have been found, that/those
	// is/are the preferred local function(s).
	if(localFunctions.size() > 0)
		return;

	// If no suitable local function with a definition has been found, add the
	// first local function without a definition we can find and return.
	itLocalFunction = pLocalFunctions->begin();
	if(itLocalFunction != pLocalFunctions->end())
	{
		localFunctions.push_back(*itLocalFunction);
	}
}
*/

bool CCC_CallGraph::FindMainFunction(std::string & szSignature, CCC_GlobalFunction* & pGlobalFunction)
{
	// Create an array containing any possible signature of the 'main' function.
	const int MAIN_SIGNATURE_COUNT = 4;
	const char* mainSignatures[MAIN_SIGNATURE_COUNT] =
	{
		"int (main)(int, char**)",
		"int (main)()",
		"void (main)(int, char**)",
		"void (main)()"
	};

	// Now iterate over the array of signatures until a function matching the
	// signature is found, or until there are no more signatures.
	int i = 0;
	pGlobalFunction = NULL;
	while(i < MAIN_SIGNATURE_COUNT && pGlobalFunction == NULL)
	{
		szSignature = mainSignatures[i++];
		pGlobalFunction = m_pGlobalFunctionRepository->GetGlobalFunction(szSignature);
	}

	// Return whether a matching main function was found.
	return pGlobalFunction != NULL;
}


// ----------------------------------------------------------------------------
// Graph validation
// ----------------------------------------------------------------------------
void CCC_CallGraph::Validate()
{
	WarnAboutConservativeness();
}

void CCC_CallGraph::WarnAboutConservativeness()
{
	// Check whether we can skip validation.
	if(CCC_ConstructorSettings::GetConservativenessDisableWarnings())
		return;
	
	/*
	// Validate all CCI_FunctionCalls.
	std::multimap<CCI_FunctionCall*, int>::iterator itFunctionCall = m_edges.begin();
	while(itFunctionCall != m_edges.end())
	{
		CCI_FunctionCall* pFunctionCall = itFunctionCall->first;

		if(pFunctionCall->IsTypeObject())
		{
			// This is a call on an object reference or an object pointer. Check
			// whether the first call candidate is virtual. If so, then all call
			// candidates in the set are virtual.
			// (This is because the set consists of the 'root' called function,
			//  plus the closure of all methods that override it. So, all elements
			//  originate from the same base method, and are thus all virtual).
			int iFirstEdgeId = itFunctionCall->second;
			CCI_LocalFunction* pLocalFunction = m_edgeDestinationNodes[iFirstEdgeId];
			if(pLocalFunction->IsFlagVirtual())
			{
				// Retrieve the set of call candidates of this function call. We
				// need it to determine whether a warning needs to be printed and,
				// possibly, to print the warning.
				GlobalFunctionVector callCandidates;
				GetCallCandidates(pFunctionCall, callCandidates);

				// Find out whether we need to print a warning.
				if(!CCC_ConstructorSettings::GetConservativenessDisableWarningsOneCandidate() || callCandidates.size() != 1)
				{
					// The method that is called is virtual. Issue a warning that
					// call candidates have potentially been deleted during filtering
					// and print the set of call candidates.
					std::cout << "WARNING: Set of (" << callCandidates.size() << ") call candidates (of call to virtual) might not be conservative." << std::endl;
					std::cout << pFunctionCall->ToString("");
					if(CCC_ConstructorSettings::GetConservativenessWarningsListCandidates())
						std::cout << CallCandidatesToString(callCandidates);
				}
			}
		}
		else if(pFunctionCall->IsTypePointerToFunction())
		{
			// Retrieve the set of call candidates of this function call. We
			// need it to determine whether a warning needs to be printed and,
			// possibly, to print the warning.
			GlobalFunctionVector callCandidates;
			GetCallCandidates(pFunctionCall, callCandidates);

			// Find out whether we need to print a warning.
			if(!CCC_ConstructorSettings::GetConservativenessDisableWarningsOneCandidate() || callCandidates.size() != 1)
			{
				// A function is called via a pointer-to-function. Issue a warning
				// that call candidates have potentially been deleted during filtering
				// and print the set of call candidates.
				std::cout << "WARNING: Set of (" << callCandidates.size() << ") call candidates (of call via pointer-to-function) might not be conservative." << std::endl;
				std::cout << pFunctionCall->ToString("");
				if(CCC_ConstructorSettings::GetConservativenessWarningsListCandidates())
					std::cout << CallCandidatesToString(callCandidates);
			}
		}

		// Move on to the next key which is different from the current key.
		std::multimap<CCI_FunctionCall*, int>::iterator itOld = itFunctionCall;
		while(itFunctionCall != m_edges.end() && itOld->first == itFunctionCall->first)
			itFunctionCall++;
	}
	*/
}

std::string CCC_CallGraph::CallCandidatesToString(GlobalFunctionVector & globalFunctions)
{
	std::stringstream ss;

	ss << "Has the following call candidates:" << std::endl;
	ss << "{" << std::endl;
	GlobalFunctionVector::iterator itGlobalFunction = globalFunctions.begin();
	while(itGlobalFunction != globalFunctions.end())
	{
		CCC_GlobalFunction* pGlobalFunction = *itGlobalFunction;

		// Retrieve all preferred local functions of this global function.
		LocalFunctionVector localFunctions;
		GetPreferredLocalFunctions(pGlobalFunction, localFunctions);

		// Print each preferred local function.
		LocalFunctionVector::iterator itLocalFunction = localFunctions.begin();
		while(itLocalFunction != localFunctions.end())
		{
			CCI_LocalFunction* pLocalFunction = *itLocalFunction;
			ss << "\t" << pLocalFunction->GetSignature() << " ";

			// If this function has a definition, then we print that. Otherwise, we
			// print the declaration.
			if(pLocalFunction->HasFunctionDefinition())
			{
				CCI_FunctionDefinition* pFunctionDefinition = pLocalFunction->GetFunctionDefinition();
				ss << pFunctionDefinition->GetSourceLoc()->ToString("") << std::endl;
			}
			else
			{
				CCI_FunctionDeclaration* pFunctionDeclaration = pLocalFunction->GetFunctionDeclaration();
				ss << pFunctionDeclaration->GetSourceLoc()->ToString("") << std::endl;
			}

			itLocalFunction++;
		}

		itGlobalFunction++;
	}
	ss << "}" << std::endl;

	return ss.str();
}
