/***************************************************************************
 *   Copyright (C) 2009 by Hessel Hoogendorp                               *
 *   bugs.ccc@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#ifndef CCC_CALL_GRAPH_H
#define CCC_CALL_GRAPH_H


// ----------------------------------------------------------------------------
// Includes
// ----------------------------------------------------------------------------
#include "ccc_typedefs.h"
#include <libcci/cci.h>


// ----------------------------------------------------------------------------
// Class declarations
// ----------------------------------------------------------------------------
class CCC_GlobalFunctionRepository;

class GraphNode;
class DirectoryGraphNode;
class FileGraphNode;
class ClassGraphNode;
class FunctionGraphNode;

class GraphEdge;
class ContainsGraphEdge;
class CallsGraphEdge;


// ----------------------------------------------------------------------------
// Class CCC_CallGraph
// ----------------------------------------------------------------------------
class CCC_CallGraph
{
	// ------------------------------------------------------------------------
	// Construction & destruction
	// ------------------------------------------------------------------------
protected:
	// This object must be constructed by calling 'Create'.
	CCC_CallGraph();

public:
	virtual ~CCC_CallGraph();

public:
	// Creates a new CCC_CallGraph object and builds a call graph from the supplied
	// global functions and initializing and finalizing calls. The resulting graph
	// will be a partial or a full call graph, based on the supplied command line
	// parameters.
	static CCC_CallGraph* Create(CCC_GlobalFunctionRepository* pGlobalFunctionRepository, FunctionCallVector* pInitializingCalls, FunctionCallVector* pFinalizingCalls);

private:
	void Build(CCC_GlobalFunctionRepository* pGlobalFunctionRepository, FunctionCallVector* pInitializingCalls, FunctionCallVector* pFinalizingCalls);


	// ------------------------------------------------------------------------
	// Node insertion
	// ------------------------------------------------------------------------
protected:
	virtual void InsertNodes() = 0;

	void InsertFunctionGraphNode(CCI_LocalFunction* pLocalFunction);
	GraphNode* GetContainerNode(CCI_LocalFunction* pLocalFunction);
	ClassGraphNode* GetClassNode(std::string szClassName);
	FileGraphNode* GetFileNode(CCI_SourceLoc* pSourceLoc);

	std::string UnmangleSignature(std::string szMangledSignature);


	// ------------------------------------------------------------------------
	// Edge insertion
	// ------------------------------------------------------------------------
private:
	void InsertCallEdges();
	void InsertCallEdges(CCI_LocalFunction* pLocalFunction);
	void InsertCallEdges(CCI_LocalFunction* pSourceNode, CCI_FunctionCall* pFunctionCall);

	bool InsertRootFunctionNodeAndCallEdgeToEntryPoint();
	void ProcessInitializingFunctionCalls();
	void ProcessFinalizingFunctionCalls();
	
protected:
	virtual CCC_GlobalFunction* GetMainEntryPoint() = 0;

private:
	int m_iNextFreeCallCandidateSetId;


	// ------------------------------------------------------------------------
	// Helpers
	// ------------------------------------------------------------------------
protected:
	// Retrieves a vector of global functions containing all call candidates of
	// the supplied function call.
	void GetCallCandidates(CCI_FunctionCall* pFunctionCall, GlobalFunctionVector & callCandidates);
	// Retrieves a list of the relevant local functions of a global function. The
	// returned list consists of all functions that have a definition, or, if there
	// are no functions with a definition, the first function (declaration) associated
	// with the global function.
	void GetPreferredLocalFunctions(CCC_GlobalFunction* pGlobalFunction, LocalFunctionVector & localFunctions);
//	void GetPreferredLocalFunctionsForFunctionCall(CCC_GlobalFunction* pGlobalFunction, CCI_FunctionCall* pFunctionCall, LocalFunctionVector & localFunctions);
	//
	bool FindMainFunction(std::string & szSignature, CCC_GlobalFunction* & pGlobalFunction);


	// ------------------------------------------------------------------------
	// Graph validation
	// ------------------------------------------------------------------------
public:
	void Validate();

private:
	// Prints warnings when it cannot be guaranteed that the set of call
	// candidates is conservative. The verbosity of the warnings can be controlled
	// with the --Wxxx switches.
	void WarnAboutConservativeness();
	std::string CallCandidatesToString(GlobalFunctionVector & globalFunctions);


	// ------------------------------------------------------------------------
	// Attributes which are the input for the call graph
	// ------------------------------------------------------------------------
protected:
	CCC_GlobalFunctionRepository* m_pGlobalFunctionRepository;
	FunctionCallVector* m_pInitializingCalls;
	FunctionCallVector* m_pFinalizingCalls;


	// ------------------------------------------------------------------------
	// Attributes which together form the call graph
	// ------------------------------------------------------------------------
protected:
	// The vector of GraphNodes.
	std::vector<GraphNode*> m_nodes;
	// The vector of GraphEdges.
	std::vector<GraphEdge*> m_edges;

	std::map<CCI_LocalFunction*, FunctionGraphNode* > m_nodeFromFunction;
	std::map<std::string       , DirectoryGraphNode*> m_nodeFromDirectory;
	std::map<std::string       , FileGraphNode*     > m_nodeFromFile;
	std::map<std::string       , ClassGraphNode*    > m_nodeFromClass;

	std::map<std::string       , ContainsGraphEdge* > m_edgeFromClass;
	std::map<CCI_FunctionCall* , CallsGraphEdge*    > m_edgeFromCall;

	// NEW !!
	std::map<std::string       , FunctionGraphNode* > m_nodeFromUnmangledFQN;


	// Allow the graph printers to access the call graph members.
	friend class CCC_CallGraphPrinterDot;
	friend class CCC_CallGraphPrinterTlp;
	friend class CCC_CallGraphPrinterSql;
	  
	CCC_GlobalFunction* m_pMainEntryPoint;


	// ------------------------------------------------------------------------
	// Attributes required to insert the fictional 'Root' node into the graph
	// ------------------------------------------------------------------------
protected:

	// The local function representing the 'Root' node.
	CCI_LocalFunction* m_plfRootFunctionNode;
	// The function call representing the edge from the 'Root' node to the 'main' function.
	CCI_FunctionCall* m_pfcCallToMainEntryPoint;
};


#endif
