/***************************************************************************
 *   Copyright (C) 2009 by Hessel Hoogendorp                               *
 *   bugs.ccc@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "ccc_call_graph_edge.h"
#include "ccc_call_graph_node.h"


// ----------------------------------------------------------------------------
// Class GraphEdge
// ----------------------------------------------------------------------------
GraphEdge::GraphEdge(GraphNode* pSrcNode, GraphNode* pDstNode)
{
	cci_assert(pSrcNode != NULL);
	cci_assert(pDstNode != NULL);
	m_iId      = m_iNextFreeId++;
	m_pSrcNode = pSrcNode;
	m_pDstNode = pDstNode;		
}

GraphEdge::~GraphEdge()
{
}

int GraphEdge::GetId()
{
	return m_iId;
}

GraphNode* GraphEdge::GetSourceNode()
{
	return m_pSrcNode;
}

GraphNode* GraphEdge::GetDestinationNode()
{
	return m_pDstNode;
}


// ----------------------------------------------------------------------------
// Class ContainsGraphEdge
// ----------------------------------------------------------------------------
ContainsGraphEdge::ContainsGraphEdge(GraphNode* pSrcNode, GraphNode* pDstNode) : GraphEdge(pSrcNode, pDstNode)
{
}

GraphEdgeType ContainsGraphEdge::GetType()
{
	return GET_Contains;
}


// ----------------------------------------------------------------------------
// Class CallsGraphEdge
// ----------------------------------------------------------------------------
CallsGraphEdge::CallsGraphEdge(FunctionGraphNode* pSrcNode, FunctionGraphNode* pDstNode, CCI_FunctionCall* pFunctionCall, int iCallCandidateSetSize, int iCallCandidateSetId) : GraphEdge(pSrcNode, pDstNode)
{
	cci_assert(pFunctionCall != NULL);
	m_pFunctionCall         = pFunctionCall;
	m_iCallCandidateSetSize = iCallCandidateSetSize;
	m_iCallCandidateSetId   = iCallCandidateSetId;
}

GraphEdgeType CallsGraphEdge::GetType()
{
	return GET_Calls;
}

std::string CallsGraphEdge::GetCallFile()
{
	return m_pFunctionCall->GetSourceLoc()->GetFileName();
}

std::string CallsGraphEdge::GetCallPosition()
{
	CCI_SourceLoc* pSourceLoc = m_pFunctionCall->GetSourceLoc();
	int iLine = pSourceLoc->GetLine();
	int iColumn = pSourceLoc->GetColumn();
	std::stringstream ss;
	ss << "(Line: " << iLine << ", Column: " << iColumn << ")";
	return ss.str();
}

bool CallsGraphEdge::IsCallVirtual()
{
	return ((FunctionGraphNode*)GetDestinationNode())->IsVirtual();
}

bool CallsGraphEdge::IsCallAccurate()
{
	// The destination node of this edge is guaranteed to be the actual call target
	// when:
	// 1. This is a plain C-function call, or plain C++-method call (not on an
	//    object reference or object pointer).
	// 2. This is a call on an object reference or object pointer and the
	//    destination node is not virtual.
	// 3. This is a call via pointer-to-member, the destination node is not virtual
	//    and is the only candidate in its call candidate set.
	return (m_pFunctionCall->IsTypeDirect()) ||
	       (m_pFunctionCall->IsTypeObject() && !IsCallVirtual()) ||
	       (m_pFunctionCall->IsTypePointerToMember() && !IsCallVirtual() && m_iCallCandidateSetSize == 1);
}

int CallsGraphEdge::GetCallCandidateSetId()
{
	return m_iCallCandidateSetId;
}

CallViaPointer CallsGraphEdge::GetCallViaPointer()
{
	if(m_pFunctionCall->IsTypePointerToFunction())
		return CVP_PointerToFunction;
	if(m_pFunctionCall->IsTypePointerToMember())
		return CVP_PointerToMember;

	return CVP_No;
}

CallOnObject CallsGraphEdge::GetCallOnObject()
{
	if(m_pFunctionCall->IsTypeObjectReference())
		return COO_ObjectReference;
	if(m_pFunctionCall->IsTypeObjectPointer())
		return COO_ObjectPointer;
	if(m_pFunctionCall->IsTypeDirectMethod() || m_pFunctionCall->IsTypeConstructor() || m_pFunctionCall->IsTypeDestructor())
		return COO_ObjectInstance;

	return COO_No;
}
