/***************************************************************************
 *   Copyright (C) 2009 by Hessel Hoogendorp                               *
 *   bugs.ccc@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "ccc_call_graph_node.h"


// ----------------------------------------------------------------------------
// Class GraphNode
// ----------------------------------------------------------------------------
GraphNode::GraphNode(std::string szName)
{
	m_iId    = m_iNextFreeId++;
	m_szName = szName;
}

GraphNode::~GraphNode()
{
}

int GraphNode::GetId()
{
	return m_iId;
}

std::string GraphNode::GetName()
{
	return m_szName;
}


// ----------------------------------------------------------------------------
// Class DirectoryGraphNode
// ----------------------------------------------------------------------------
DirectoryGraphNode::DirectoryGraphNode(std::string szName) : GraphNode(szName)
{
}

GraphNodeType DirectoryGraphNode::GetType()
{
	return GNT_Directory;
}


// ----------------------------------------------------------------------------
// Class FileGraphNode
// ----------------------------------------------------------------------------
FileGraphNode::FileGraphNode(CCI_SourceLoc* pSourceLoc, std::string szName) : GraphNode(szName)
{
	m_pSourceLoc = pSourceLoc;
}

GraphNodeType FileGraphNode::GetType()
{
	return GNT_File;
}

std::string FileGraphNode::GetFileName()
{
	return m_pSourceLoc->GetFileName();
}

std::string FileGraphNode::GetFilePath()
{
	return m_pSourceLoc->GetFilePath();
}


// ----------------------------------------------------------------------------
// Class ClassGraphNode
// ----------------------------------------------------------------------------
ClassGraphNode::ClassGraphNode(std::string szName) : GraphNode(szName)
{
}

GraphNodeType ClassGraphNode::GetType()
{
	return GNT_Class;
}


// ----------------------------------------------------------------------------
// Class FunctionGraphNode
// ----------------------------------------------------------------------------
FunctionGraphNode::FunctionGraphNode(CCI_LocalFunction* pLocalFunction) : GraphNode(pLocalFunction->GetName())
{
	cci_assert(pLocalFunction != NULL);
	m_pLocalFunction = pLocalFunction;
}

GraphNodeType FunctionGraphNode::GetType()
{
	return GNT_Function;
}

std::string FunctionGraphNode::GetSignature()
{
	return m_pLocalFunction->GetSignature();
}

std::string FunctionGraphNode::GetClassName()
{
	return m_pLocalFunction->GetClassName();
}

std::string FunctionGraphNode::GetDeclarationFileName()
{
	return m_pLocalFunction->GetFunctionDeclaration()->GetSourceLoc()->GetFileName();
}

std::string FunctionGraphNode::GetDeclarationPosition()
{
	CCI_SourceLoc* pSourceLoc = m_pLocalFunction->GetFunctionDeclaration()->GetSourceLoc();
	int iLine = pSourceLoc->GetLine();
	int iColumn = pSourceLoc->GetColumn();
	std::stringstream ss;
	ss << "(Line: " << iLine << ", Column: " << iColumn << ")";
	return ss.str();
}

std::string FunctionGraphNode::GetDefinitionFileName()
{
	return m_pLocalFunction->HasFunctionDefinition() ? m_pLocalFunction->GetFunctionDefinition()->GetSourceLoc()->GetFileName() : "";
}

std::string FunctionGraphNode::GetDefinitionPosition()
{
	if(m_pLocalFunction->HasFunctionDefinition())
	{
		CCI_SourceLoc* pSourceLoc = m_pLocalFunction->GetFunctionDefinition()->GetSourceLoc();
		int iLine = pSourceLoc->GetLine();
		int iColumn = pSourceLoc->GetColumn();
		std::stringstream ss;
		ss << "(Line: " << iLine << ", Column: " << iColumn << ")";
		return ss.str();
	}
	else
	{
		return "";
	}
}

bool FunctionGraphNode::HasClassName()
{
	return m_pLocalFunction->HasClassName();
}

bool FunctionGraphNode::HasDefinitionFileName()
{
	return m_pLocalFunction->HasFunctionDefinition();
}

bool FunctionGraphNode::HasDefinitionPosition()
{
	return m_pLocalFunction->HasFunctionDefinition();
}

bool FunctionGraphNode::IsMethod()
{
	return m_pLocalFunction->IsFlagMethod();
}

bool FunctionGraphNode::IsVirtual()
{
	return m_pLocalFunction->IsFlagVirtual();
}

bool FunctionGraphNode::IsStatic()
{
	return m_pLocalFunction->IsFlagStatic();
}

bool FunctionGraphNode::IsStaticLinkage()
{
	return m_pLocalFunction->IsFlagStaticLinkage();
}

bool FunctionGraphNode::IsInline()
{
	return m_pLocalFunction->IsFlagInline();
}

AccessSpecifier FunctionGraphNode::GetAccessSpecifier()
{
	if(m_pLocalFunction->IsFlagPublic())
		return AS_Public;
	else if(m_pLocalFunction->IsFlagProtected())
		return AS_Protected;
	else if(m_pLocalFunction->IsFlagPrivate())
		return AS_Private;

	return AS_None;
}
