/***************************************************************************
 *   Copyright (C) 2009 by Hessel Hoogendorp                               *
 *   bugs.ccc@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


// ----------------------------------------------------------------------------
// Includes
// ----------------------------------------------------------------------------
#include "ccc_call_graph_printer_sql.h"
#include "ccc_call_graph.h"
#include "ccc_call_graph_node.h"
#include "ccc_call_graph_edge.h"
#include "ccc_settings.h"
#include <libcci/cci.h>
#include <sqlite3.h>
//#include <sstream>


// ----------------------------------------------------------------------------
// Callback
// ----------------------------------------------------------------------------
static int callback(void *NotUsed, int argc, char **argv, char **azColName)
{
	return 0;
}



// ----------------------------------------------------------------------------
// Construction
// ----------------------------------------------------------------------------
CCC_CallGraphPrinterSql::CCC_CallGraphPrinterSql(std::string szFileName) :
	CCC_CallGraphPrinter("")
{
	// Don't let the base constructor set the file name, because that will
	// automatically open the file for writing. We are going to let sqlite
	// write the file for us.
	m_szFileName = szFileName;
}


// ----------------------------------------------------------------------------
// Graph printing
// ----------------------------------------------------------------------------
void CCC_CallGraphPrinterSql::PrintGraph()
{
	char* pszErrMsg = NULL;
	int   result;

	// Open the database.

	Open() &&
	ApplySettings() &&
	ClearTables() &&
	CreateTables() &&
	CreateIndices() &&
	PrepareQueries() &&

	CreateEdges() &&
	AnnotateEdges() &&
	AnnotateNodes();

	/*
	if(!Open()          ) return;
	if(!ApplySettings() ) return;
	if(!ClearTables()   ) return;
	if(!CreateTables()  ) return;
	if(!CreateIndices() ) return;
	if(!PrepareQueries()) return;

	if(!CreateEdges()   ) return;
	if(!AnnotateEdges() ) return;
	if(!AnnotateNodes() ) return;
	*/

	DestroyQueries();
	Close();
}


// ----------------------------------------------------------------------------
// Querying primitives
// ----------------------------------------------------------------------------
bool CCC_CallGraphPrinterSql::Open()
{
	if(sqlite3_open(m_szFileName.c_str(), &m_pSqlite) != SQLITE_OK)
	{
		std::cerr << "ERROR: Failed to open database '" << m_szFileName << "' for writing." << std::endl;
		//Close();
		return false;
	}
	
	return true;
}

bool CCC_CallGraphPrinterSql::Close()
{
	if(sqlite3_close(m_pSqlite) != SQLITE_OK)
	{
		std::cerr << "ERROR: Failed to close database '" << m_szFileName << "'." << std::endl;
		return false;
	}
	
	return true;
}

bool CCC_CallGraphPrinterSql::Query(std::string szQuery)
{
	char* pszErrorMessage = NULL;
	if(sqlite3_exec(m_pSqlite, szQuery.c_str(), NULL, 0, &pszErrorMessage) != SQLITE_OK)
	{
		std::cerr << "ERROR: Query '" << szQuery << "' returned error: '" << pszErrorMessage << "'." << std::endl;
		sqlite3_free(pszErrorMessage);
		//Close();
		return false;
	}

	return true;
}

bool CCC_CallGraphPrinterSql::Prepare(sqlite3_stmt** pppq, std::string szQuery)
{
	if (sqlite3_prepare_v2(m_pSqlite, szQuery.c_str(), -1, pppq, NULL) != SQLITE_OK)
	{
		std::cerr << "ERROR: Error preparing query '" << szQuery << "'." << std::endl;
		//Close();
		return false;
	}

	return true;
}

bool CCC_CallGraphPrinterSql::BindAndQuery(sqlite3_stmt* ppq, int i1)
{
	// Bind the parameter.
	if(sqlite3_bind_int(ppq, 1, i1) != SQLITE_OK)
	{
		std::cerr << "ERROR: An error occurred while binding an single integer argument to a precompiled query." << std::endl;
		//Close();
		return false;
	}

	return QueryFromStatement(ppq);
}

bool CCC_CallGraphPrinterSql::BindAndQuery(sqlite3_stmt* ppq, int i1, int i2, int i3)
{
	//std::cout << "BAQ: (" << i1 << ", " << i2 << ", " << i3 << ")\n";
	// Bind the parameter.
	if(sqlite3_bind_int(ppq, 1, i1) != SQLITE_OK)
	{
		std::cerr << "ERROR: An error occurred while binding the first of three integer arguments to a precompiled query: " << std::endl;
		//Close();
		return false;
	}
	if(sqlite3_bind_int(ppq, 2, i2) != SQLITE_OK)
	{
		std::cerr << "ERROR: An error occurred while binding the second of three integer arguments to a precompiled query." << std::endl;
		//Close();
		return false;
	}
	if(sqlite3_bind_int(ppq, 3, i3) != SQLITE_OK)
	{
		std::cerr << "ERROR: An error occurred while binding the thrid of three integer arguments to a precompiled query." << std::endl;
		//Close();
		return false;
	}

	return QueryFromStatement(ppq);
}

bool CCC_CallGraphPrinterSql::BindAndQuery(sqlite3_stmt* ppq, int i1, std::string sz2)
{
	// Bind the parameter.
	int error;
	if((error = sqlite3_bind_int(ppq, 1, i1)) != SQLITE_OK)
	{
		std::cerr << "ERROR: An error occurred while binding the first integer of two arguments to a precompiled query." << std::endl;
		//Close();
		return false;
	}
	if(sqlite3_bind_text(ppq, 2, sz2.c_str(), sz2.size(), SQLITE_STATIC) != SQLITE_OK)
	{
		std::cerr << "ERROR: An error occurred while binding the second string of two arguments to a precompiled query." << std::endl;
		//Close();
		return false;
	}

	return QueryFromStatement(ppq);
}

bool CCC_CallGraphPrinterSql::QueryFromStatement(sqlite3_stmt* ppq)
{
	// Execute the query.
	if(sqlite3_step(ppq) != SQLITE_DONE)
	{
		std::cerr << "ERROR: An error occurred while executing a precompiled query." << std::endl;
		//Close();
		return false;
	}

	// Clear the parameter bindings.
	//sqlite3_clear_bindings(ppq);

	// Reset the prepared statement.
	sqlite3_reset(ppq);

	return true;
}


// ----------------------------------------------------------------------------
// Queries
// ----------------------------------------------------------------------------
bool CCC_CallGraphPrinterSql::ApplySettings()
{
	return Query("pragma synchronous=off;");
}

bool CCC_CallGraphPrinterSql::ClearTables()
{
	if(!Query("DROP TABLE IF EXISTS edges;"    )) return false;
	if(!Query("DROP TABLE IF EXISTS nodeattrs;")) return false;
	if(!Query("DROP TABLE IF EXISTS edgeattrs;")) return false;
	return true;
}

bool CCC_CallGraphPrinterSql::CreateTables()
{
	if(!Query("CREATE TABLE edges (eid INTEGER PRIMARY KEY, fromnid INTEGER, tonid INTEGER);")) return false;
	if(!Query("CREATE TABLE nodeattrs (nid INTEGER, key STRING, value STRING);"              )) return false;
	if(!Query("CREATE TABLE edgeattrs (eid INTEGER, key STRING, value STRING);"              )) return false;
	return true;
}

bool CCC_CallGraphPrinterSql::CreateIndices()
{
	if(!Query("CREATE INDEX index_edge_id ON edges (eid);"          )) return false;
	if(!Query("CREATE INDEX index_edge_fromnid ON edges (fromnid);" )) return false;
	if(!Query("CREATE INDEX index_edge_tonid ON edges (tonid);"     )) return false;
	if(!Query("CREATE INDEX index_nodeattrs_nid ON nodeattrs (nid);")) return false;
	if(!Query("CREATE INDEX index_edgeattrs_eid ON edgeattrs (eid);")) return false;

	return true;
}

bool CCC_CallGraphPrinterSql::PrepareQueries()
{
	// Edge insertion queries.
	if(!Prepare(&ppqInsertEdge, "INSERT INTO edges (eid, fromnid, tonid) VALUES (?, ?, ?);")) return false;

	// Edge attribute insertion queries.
	if(!Prepare(&ppqInsertEdgeAttrTypeInclusion                  , "INSERT INTO edgeattrs (eid, key, value) VALUES (?, 'type'                 , 'inclusion');"     )) return false;
	if(!Prepare(&ppqInsertEdgeAttrTypeCall                       , "INSERT INTO edgeattrs (eid, key, value) VALUES (?, 'type'                 , 'call');"          )) return false;
	if(!Prepare(&ppqInsertEdgeAttrCallFile                       , "INSERT INTO edgeattrs (eid, key, value) VALUES (?, 'call_file'            , ?     );"          )) return false;
	if(!Prepare(&ppqInsertEdgeAttrCallPosition                   , "INSERT INTO edgeattrs (eid, key, value) VALUES (?, 'call_position'        , ?     );"          )) return false;
	if(!Prepare(&ppqInsertEdgeAttrCallIsVirtual                  , "INSERT INTO edgeattrs (eid, key, value) VALUES (?, 'call_is_virtual'      , ?     );"          )) return false;
	if(!Prepare(&ppqInsertEdgeAttrCallIsAccurate                 , "INSERT INTO edgeattrs (eid, key, value) VALUES (?, 'call_is_accurate'     , ?     );"          )) return false;
	if(!Prepare(&ppqInsertEdgeAttrCallCandidateSetId             , "INSERT INTO edgeattrs (eid, key, value) VALUES (?, 'call_candidate_set_id', ?     );"          )) return false;
    if(!Prepare(&ppqInsertEdgeAttrCallViaPointerNo               , "INSERT INTO edgeattrs (eid, key, value) VALUES (?, 'call_via_pointer', 'no'                 );")) return false;
	if(!Prepare(&ppqInsertEdgeAttrCallViaPointerPointerToFunction, "INSERT INTO edgeattrs (eid, key, value) VALUES (?, 'call_via_pointer', 'pointer-to-function');")) return false;
	if(!Prepare(&ppqInsertEdgeAttrCallViaPointerPointerToMember  , "INSERT INTO edgeattrs (eid, key, value) VALUES (?, 'call_via_pointer', 'pointer-to-member'  );")) return false;
	if(!Prepare(&ppqInsertEdgeAttrCallOnObjectNo                 , "INSERT INTO edgeattrs (eid, key, value) VALUES (?, 'call_on_object'  , 'no'                 );")) return false;
	if(!Prepare(&ppqInsertEdgeAttrCallOnObjectObjectInstance     , "INSERT INTO edgeattrs (eid, key, value) VALUES (?, 'call_on_object'  , 'object-instance'    );")) return false;
	if(!Prepare(&ppqInsertEdgeAttrCallOnObjectObjectReference    , "INSERT INTO edgeattrs (eid, key, value) VALUES (?, 'call_on_object'  , 'object-reference'   );")) return false;
	if(!Prepare(&ppqInsertEdgeAttrCallOnObjectObjectPointer      , "INSERT INTO edgeattrs (eid, key, value) VALUES (?, 'call_on_object'  , 'object-pointer'     );")) return false;

	// Node attribute insertion queries.
	if(!Prepare(&ppqInsertNodeAttrName                    , "INSERT INTO nodeattrs (nid, key, value) VALUES (?, 'name'                , ?);"          )) return false;
	if(!Prepare(&ppqInsertNodeAttrTypeDirectory           , "INSERT INTO nodeattrs (nid, key, value) VALUES (?, 'type'                , 'Directory');")) return false;
	if(!Prepare(&ppqInsertNodeAttrTypeFile                , "INSERT INTO nodeattrs (nid, key, value) VALUES (?, 'type'                , 'File');"     )) return false;
	if(!Prepare(&ppqInsertNodeAttrTypeClass               , "INSERT INTO nodeattrs (nid, key, value) VALUES (?, 'type'                , 'Class');"    )) return false;
	if(!Prepare(&ppqInsertNodeAttrTypeFunction            , "INSERT INTO nodeattrs (nid, key, value) VALUES (?, 'type'                , 'Function');" )) return false;
	if(!Prepare(&ppqInsertNodeAttrSignature               , "INSERT INTO nodeattrs (nid, key, value) VALUES (?, 'signature'           , ?);"          )) return false;
	if(!Prepare(&ppqInsertNodeAttrClass                   , "INSERT INTO nodeattrs (nid, key, value) VALUES (?, 'class'               , ?);"          )) return false;
	if(!Prepare(&ppqInsertNodeAttrIsMethod                , "INSERT INTO nodeattrs (nid, key, value) VALUES (?, 'is_method'           , ?);"          )) return false;
	if(!Prepare(&ppqInsertNodeAttrIsVirtual               , "INSERT INTO nodeattrs (nid, key, value) VALUES (?, 'is_virtual'          , ?);"          )) return false;
	if(!Prepare(&ppqInsertNodeAttrIsStatic                , "INSERT INTO nodeattrs (nid, key, value) VALUES (?, 'is_static'           , ?);"          )) return false;
	if(!Prepare(&ppqInsertNodeAttrIsStaticLinkage         , "INSERT INTO nodeattrs (nid, key, value) VALUES (?, 'is_static_linkage'   , ?);"          )) return false;
	if(!Prepare(&ppqInsertNodeAttrIsInline                , "INSERT INTO nodeattrs (nid, key, value) VALUES (?, 'is_inline'           , ?);"          )) return false;
	if(!Prepare(&ppqInsertNodeAttrDeclarationFile         , "INSERT INTO nodeattrs (nid, key, value) VALUES (?, 'declaration_file'    , ?);"          )) return false;
	if(!Prepare(&ppqInsertNodeAttrDeclarationPosition     , "INSERT INTO nodeattrs (nid, key, value) VALUES (?, 'declaration_position', ?);"          )) return false;
	if(!Prepare(&ppqInsertNodeAttrDefinitionFile          , "INSERT INTO nodeattrs (nid, key, value) VALUES (?, 'definition_file'     , ?);"          )) return false;
	if(!Prepare(&ppqInsertNodeAttrDefinitionPosition      , "INSERT INTO nodeattrs (nid, key, value) VALUES (?, 'definition_position' , ?);"          )) return false;
	if(!Prepare(&ppqInsertNodeAttrAccessSpecifierPublic   , "INSERT INTO nodeattrs (nid, key, value) VALUES (?, 'access_specifier'    , 'public');"   )) return false;
	if(!Prepare(&ppqInsertNodeAttrAccessSpecifierProtected, "INSERT INTO nodeattrs (nid, key, value) VALUES (?, 'access_specifier'    , 'protected');")) return false;
	if(!Prepare(&ppqInsertNodeAttrAccessSpecifierPrivate  , "INSERT INTO nodeattrs (nid, key, value) VALUES (?, 'access_specifier'    , 'private');"  )) return false;
	
	return true;
}

void CCC_CallGraphPrinterSql::DestroyQueries()
{
	// Edge insertion queries.
	sqlite3_finalize(ppqInsertEdge);

	// Edge attribute insertion queries.
	sqlite3_finalize(ppqInsertEdgeAttrTypeInclusion                  );
	sqlite3_finalize(ppqInsertEdgeAttrTypeCall                       );
	sqlite3_finalize(ppqInsertEdgeAttrCallFile                       );
	sqlite3_finalize(ppqInsertEdgeAttrCallPosition                   );
	sqlite3_finalize(ppqInsertEdgeAttrCallIsVirtual                  );
	sqlite3_finalize(ppqInsertEdgeAttrCallIsAccurate                 );
	sqlite3_finalize(ppqInsertEdgeAttrCallCandidateSetId             );
    sqlite3_finalize(ppqInsertEdgeAttrCallViaPointerNo               );
	sqlite3_finalize(ppqInsertEdgeAttrCallViaPointerPointerToFunction);
	sqlite3_finalize(ppqInsertEdgeAttrCallViaPointerPointerToMember  );
	sqlite3_finalize(ppqInsertEdgeAttrCallOnObjectNo                 );
	sqlite3_finalize(ppqInsertEdgeAttrCallOnObjectObjectInstance     );
	sqlite3_finalize(ppqInsertEdgeAttrCallOnObjectObjectReference    );
	sqlite3_finalize(ppqInsertEdgeAttrCallOnObjectObjectPointer      );

	// Node attribute insertion queries.
	sqlite3_finalize(ppqInsertNodeAttrName                    );
	sqlite3_finalize(ppqInsertNodeAttrTypeDirectory           );
	sqlite3_finalize(ppqInsertNodeAttrTypeFile                );
	sqlite3_finalize(ppqInsertNodeAttrTypeClass               );
	sqlite3_finalize(ppqInsertNodeAttrTypeFunction            );
	sqlite3_finalize(ppqInsertNodeAttrSignature               );
	sqlite3_finalize(ppqInsertNodeAttrClass                   );
	sqlite3_finalize(ppqInsertNodeAttrIsMethod                );
	sqlite3_finalize(ppqInsertNodeAttrIsVirtual               );
	sqlite3_finalize(ppqInsertNodeAttrIsStatic                );
	sqlite3_finalize(ppqInsertNodeAttrIsStaticLinkage         );
	sqlite3_finalize(ppqInsertNodeAttrIsInline                );
	sqlite3_finalize(ppqInsertNodeAttrDeclarationFile         );
	sqlite3_finalize(ppqInsertNodeAttrDeclarationPosition     );
	sqlite3_finalize(ppqInsertNodeAttrDefinitionFile          );
	sqlite3_finalize(ppqInsertNodeAttrDefinitionPosition      );
	sqlite3_finalize(ppqInsertNodeAttrAccessSpecifierPublic   );
	sqlite3_finalize(ppqInsertNodeAttrAccessSpecifierProtected);
	sqlite3_finalize(ppqInsertNodeAttrAccessSpecifierPrivate  );
}

bool CCC_CallGraphPrinterSql::CreateEdges()
{
	if(!Query("BEGIN TRANSACTION;")) return false;
	
	std::vector<GraphEdge*>::iterator itGraphEdge = m_pCallGraph->m_edges.begin();
	while(itGraphEdge != m_pCallGraph->m_edges.end())
	{
		GraphEdge* pGraphEdge = *itGraphEdge;
		/*
		std::stringstream ss;
		ss << "INSERT INTO edges (eid, fromnid, tonid) VALUES (";
		ss << pGraphEdge->GetId()                  << ", ";
		ss << pGraphEdge->GetSourceNode()->GetId() << ", ";
		ss << pGraphEdge->GetDestinationNode()->GetId();
		ss << ");";

		if(!Query(ss.str()))
			return false;
		/*/
		if(!BindAndQuery(ppqInsertEdge, pGraphEdge->GetId(), pGraphEdge->GetSourceNode()->GetId(), pGraphEdge->GetDestinationNode()->GetId()))
			return false;
		/**/

		itGraphEdge++;
	}

	if(!Query("COMMIT TRANSACTION;")) return false;

	return true;
}

bool CCC_CallGraphPrinterSql::AnnotateEdges()
{
	if(!Query("BEGIN TRANSACTION;")) return false;
	
	std::vector<GraphEdge*>::iterator itGraphEdge = m_pCallGraph->m_edges.begin();
	while(itGraphEdge != m_pCallGraph->m_edges.end())
	{
		if(!AnnotateEdge(*itGraphEdge)) return false;
		itGraphEdge++;
	}

	if(!Query("COMMIT TRANSACTION;")) return false;
	
	return true;
}

bool CCC_CallGraphPrinterSql::AnnotateEdge(GraphEdge* pGraphEdge)
{
	//std::stringstream ss;
	switch(pGraphEdge->GetType())
	{
		case GET_Calls:
			{
				CallsGraphEdge* pCallsGraphEdge = (CallsGraphEdge*)pGraphEdge;
				/*
				ss << "INSERT INTO edgeattrs (eid, key, value) VALUES (" << pGraphEdge->GetId() << ", 'type', 'call');";
				ss << "INSERT INTO edgeattrs (eid, key, value) VALUES (" << pGraphEdge->GetId() << ", 'call_file', '" << pCallsGraphEdge->GetCallFile() << "');";
				ss << "INSERT INTO edgeattrs (eid, key, value) VALUES (" << pGraphEdge->GetId() << ", 'call_position', '" << pCallsGraphEdge->GetCallPosition() << "');";
				ss << "INSERT INTO edgeattrs (eid, key, value) VALUES (" << pGraphEdge->GetId() << ", 'call_is_virtual', '" << BoolToString(pCallsGraphEdge->IsCallVirtual()) << "');";
				ss << "INSERT INTO edgeattrs (eid, key, value) VALUES (" << pGraphEdge->GetId() << ", 'call_is_accurate', '" << BoolToString(pCallsGraphEdge->IsCallAccurate()) << "');";
				ss << "INSERT INTO edgeattrs (eid, key, value) VALUES (" << pGraphEdge->GetId() << ", 'call_candidate_set_id', '" << IntToString(pCallsGraphEdge->GetCallCandidateSetId()) << "');";
				/*/
				if(!BindAndQuery(ppqInsertEdgeAttrTypeCall, pGraphEdge->GetId())) return false;
				if(!BindAndQuery(ppqInsertEdgeAttrCallFile, pGraphEdge->GetId(), pCallsGraphEdge->GetCallFile())) return false;
				if(!BindAndQuery(ppqInsertEdgeAttrCallPosition, pGraphEdge->GetId(), pCallsGraphEdge->GetCallPosition())) return false;
				if(!BindAndQuery(ppqInsertEdgeAttrCallIsVirtual, pGraphEdge->GetId(), BoolToString(pCallsGraphEdge->IsCallVirtual()))) return false;
				if(!BindAndQuery(ppqInsertEdgeAttrCallIsAccurate, pGraphEdge->GetId(), BoolToString(pCallsGraphEdge->IsCallAccurate()))) return false;
				if(!BindAndQuery(ppqInsertEdgeAttrCallCandidateSetId, pGraphEdge->GetId(), IntToString(pCallsGraphEdge->GetCallCandidateSetId()))) return false;
				/**/
				switch(pCallsGraphEdge->GetCallViaPointer())
				{
					case CVP_No:
						//ss << "INSERT INTO edgeattrs (eid, key, value) VALUES (" << pGraphEdge->GetId() << ", 'call_via_pointer', 'no');";
						if(!BindAndQuery(ppqInsertEdgeAttrCallViaPointerNo, pGraphEdge->GetId())) return false;
						break;
					case CVP_PointerToFunction:
						//ss << "INSERT INTO edgeattrs (eid, key, value) VALUES (" << pGraphEdge->GetId() << ", 'call_via_pointer', 'pointer-to-function');";
						if(!BindAndQuery(ppqInsertEdgeAttrCallViaPointerPointerToFunction, pGraphEdge->GetId())) return false;
						break;
					case CVP_PointerToMember:
						//ss << "INSERT INTO edgeattrs (eid, key, value) VALUES (" << pGraphEdge->GetId() << ", 'call_via_pointer', 'pointer-to-member');";
						if(!BindAndQuery(ppqInsertEdgeAttrCallViaPointerPointerToMember, pGraphEdge->GetId())) return false;
						break;
				}
				switch(pCallsGraphEdge->GetCallOnObject())
				{
					case COO_No:
						//ss << "INSERT INTO edgeattrs (eid, key, value) VALUES (" << pGraphEdge->GetId() << ", 'call_on_object', 'no');";
						if(!BindAndQuery(ppqInsertEdgeAttrCallOnObjectNo, pGraphEdge->GetId())) return false;
						break;
					case COO_ObjectInstance:
						//ss << "INSERT INTO edgeattrs (eid, key, value) VALUES (" << pGraphEdge->GetId() << ", 'call_on_object', 'object-instance');";
						if(!BindAndQuery(ppqInsertEdgeAttrCallOnObjectObjectInstance, pGraphEdge->GetId())) return false;
						break;
					case COO_ObjectReference:
						//ss << "INSERT INTO edgeattrs (eid, key, value) VALUES (" << pGraphEdge->GetId() << ", 'call_on_object', 'object-reference');";
						if(!BindAndQuery(ppqInsertEdgeAttrCallOnObjectObjectReference, pGraphEdge->GetId())) return false;
						break;
					case COO_ObjectPointer:
						//ss << "INSERT INTO edgeattrs (eid, key, value) VALUES (" << pGraphEdge->GetId() << ", 'call_on_object', 'object-pointer');";
						if(!BindAndQuery(ppqInsertEdgeAttrCallOnObjectObjectPointer, pGraphEdge->GetId())) return false;
						break;
				}
			}
			break;
		case GET_Contains:
			//ss << "INSERT INTO edgeattrs (eid, key, value) VALUES (" << pGraphEdge->GetId() << ", 'type', 'inclusion');";
			if(!BindAndQuery(ppqInsertEdgeAttrTypeInclusion, pGraphEdge->GetId())) return false;
			break;
	}
	//return Query(ss.str());
	return true;
}

bool CCC_CallGraphPrinterSql::AnnotateNodes()
{
	if(!Query("BEGIN TRANSACTION;")) return false;
	
	std::vector<GraphNode*>::iterator itGraphNode = m_pCallGraph->m_nodes.begin();
	while(itGraphNode != m_pCallGraph->m_nodes.end())
	{
		AnnotateNode(*itGraphNode);
		itGraphNode++;
	}

	if(!Query("COMMIT TRANSACTION;")) return false;

	return true;
}

bool CCC_CallGraphPrinterSql::AnnotateNode(GraphNode* pGraphNode)
{
	//std::stringstream ss;
	
	//ss << "INSERT INTO nodeattrs (nid, key, value) VALUES (" << pGraphNode->GetId() << ", 'name', '" << pGraphNode->GetName() << "');";
	if(!BindAndQuery(ppqInsertNodeAttrName, pGraphNode->GetId(), pGraphNode->GetName())) return false;
	
	switch(pGraphNode->GetType())
	{
		case GNT_Directory:
			{
				//ss << "INSERT INTO nodeattrs (nid, key, value) VALUES (" << pGraphNode->GetId() << ", 'type', 'Directory');";
				if(!BindAndQuery(ppqInsertNodeAttrTypeDirectory, pGraphNode->GetId())) return false;
			}
			break;
		case GNT_File:
			{
				//ss << "INSERT INTO nodeattrs (nid, key, value) VALUES (" << pGraphNode->GetId() << ", 'type', 'File');";
				if(!BindAndQuery(ppqInsertNodeAttrTypeFile, pGraphNode->GetId())) return false;
			}
			break;
		case GNT_Class:
			{
				//ss << "INSERT INTO nodeattrs (nid, key, value) VALUES (" << pGraphNode->GetId() << ", 'type', 'Class');";
				if(!BindAndQuery(ppqInsertNodeAttrTypeClass, pGraphNode->GetId())) return false;
			}
			break;
		case GNT_Function:
			{
				FunctionGraphNode* pgnFunction = (FunctionGraphNode*)pGraphNode;

				/*
				ss << "INSERT INTO nodeattrs (nid, key, value) VALUES (" << pGraphNode->GetId() << ", 'type', 'Function');";
				ss << "INSERT INTO nodeattrs (nid, key, value) VALUES (" << pGraphNode->GetId() << ", 'signature', '" << pgnFunction->GetSignature() << "');";
				/*/
				if(!BindAndQuery(ppqInsertNodeAttrTypeFunction, pGraphNode->GetId())) return false;
				if(!BindAndQuery(ppqInsertNodeAttrSignature, pGraphNode->GetId(), pgnFunction->GetSignature())) return false;
				/**/

				if(pgnFunction->HasClassName())
					//ss << "INSERT INTO nodeattrs (nid, key, value) VALUES (" << pGraphNode->GetId() << ", 'class', '" << pgnFunction->GetClassName() << "');";
					if(!BindAndQuery(ppqInsertNodeAttrClass, pGraphNode->GetId(), pgnFunction->GetClassName())) return false;

				/*
				ss << "INSERT INTO nodeattrs (nid, key, value) VALUES (" << pGraphNode->GetId() << ", 'is_method', '" << (pgnFunction->IsMethod() ? "true" : "false") << "');";
				ss << "INSERT INTO nodeattrs (nid, key, value) VALUES (" << pGraphNode->GetId() << ", 'is_virtual', '" << (pgnFunction->IsVirtual() ? "true" : "false") << "');";
				ss << "INSERT INTO nodeattrs (nid, key, value) VALUES (" << pGraphNode->GetId() << ", 'is_static', '" << (pgnFunction->IsStatic() ? "true" : "false") << "');";
				ss << "INSERT INTO nodeattrs (nid, key, value) VALUES (" << pGraphNode->GetId() << ", 'is_static_linkage', '" << (pgnFunction->IsStaticLinkage() ? "true" : "false") << "');";
				ss << "INSERT INTO nodeattrs (nid, key, value) VALUES (" << pGraphNode->GetId() << ", 'declaration_file', '" << pgnFunction->GetDeclarationFileName() << "');";
				ss << "INSERT INTO nodeattrs (nid, key, value) VALUES (" << pGraphNode->GetId() << ", 'declaration_position', '" << pgnFunction->GetDeclarationPosition() << "');";
				/*/
				if(!BindAndQuery(ppqInsertNodeAttrIsMethod, pGraphNode->GetId(), BoolToString(pgnFunction->IsMethod()))) return false;
				if(!BindAndQuery(ppqInsertNodeAttrIsVirtual, pGraphNode->GetId(), BoolToString(pgnFunction->IsVirtual()))) return false;
				if(!BindAndQuery(ppqInsertNodeAttrIsStatic, pGraphNode->GetId(), BoolToString(pgnFunction->IsStatic()))) return false;
				if(!BindAndQuery(ppqInsertNodeAttrIsStaticLinkage, pGraphNode->GetId(), BoolToString(pgnFunction->IsStaticLinkage()))) return false;
				if(!BindAndQuery(ppqInsertNodeAttrIsInline, pGraphNode->GetId(), BoolToString(pgnFunction->IsInline()))) return false;
				if(!BindAndQuery(ppqInsertNodeAttrDeclarationFile, pGraphNode->GetId(), pgnFunction->GetDeclarationFileName())) return false;
				if(!BindAndQuery(ppqInsertNodeAttrDeclarationPosition, pGraphNode->GetId(), pgnFunction->GetDeclarationPosition())) return false;
				/**/

				if(pgnFunction->HasDefinitionFileName())
					//ss << "INSERT INTO nodeattrs (nid, key, value) VALUES (" << pGraphNode->GetId() << ", 'definition_file', '" << pgnFunction->GetDefinitionFileName() << "');";
					if(!BindAndQuery(ppqInsertNodeAttrDefinitionFile, pGraphNode->GetId(), pgnFunction->GetDefinitionFileName())) return false;
				if(pgnFunction->HasDefinitionPosition())
					//ss << "INSERT INTO nodeattrs (nid, key, value) VALUES (" << pGraphNode->GetId() << ", 'definition_position', '" << pgnFunction->GetDefinitionPosition() << "');";
					if(!BindAndQuery(ppqInsertNodeAttrDefinitionPosition, pGraphNode->GetId(), pgnFunction->GetDefinitionPosition())) return false;
				
				switch(pgnFunction->GetAccessSpecifier())
				{
				case AS_Public:
					//ss << "INSERT INTO nodeattrs (nid, key, value) VALUES (" << pGraphNode->GetId() << ", 'access_specifier', 'public');";
					if(!BindAndQuery(ppqInsertNodeAttrAccessSpecifierPublic, pGraphNode->GetId())) return false;
					break;
				case AS_Protected:
					//ss << "INSERT INTO nodeattrs (nid, key, value) VALUES (" << pGraphNode->GetId() << ", 'access_specifier', 'protected');";
					if(!BindAndQuery(ppqInsertNodeAttrAccessSpecifierProtected, pGraphNode->GetId())) return false;
					break;
				case AS_Private:
					//ss << "INSERT INTO nodeattrs (nid, key, value) VALUES (" << pGraphNode->GetId() << ", 'access_specifier', 'private');";
					if(!BindAndQuery(ppqInsertNodeAttrAccessSpecifierPrivate, pGraphNode->GetId())) return false;
					break;
				}
			}
			break;
	}
	//return Query(ss.str());
	return true;
}
