/***************************************************************************
 *   Copyright (C) 2009 by Hessel Hoogendorp                               *
 *   bugs.ccc@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


// ----------------------------------------------------------------------------
// Includes
// ----------------------------------------------------------------------------
#include "ccc_call_graph_printer_tlp.h"
#include "ccc_call_graph.h"
#include "ccc_call_graph_node.h"
#include "ccc_call_graph_edge.h"
#include "ccc_settings.h"
#include <libcci/cci.h>


// ----------------------------------------------------------------------------
// Construction
// ----------------------------------------------------------------------------
CCC_CallGraphPrinterTlp::CCC_CallGraphPrinterTlp(std::string szFileName) :
	CCC_CallGraphPrinter(szFileName)
{
}


// ----------------------------------------------------------------------------
// Graph printing
// ----------------------------------------------------------------------------
void CCC_CallGraphPrinterTlp::PrintGraph()
{
	// Print graph header.
	m_os << "(tlp \"2.0\"" << std::endl;

	// Print the nodes and edges.
	PrintNodes();
	PrintEdges();

	// Print generic properties (for both nodes and edges).
	PrintPropertyType();

	// Print node specific properties.
	PrintNodePropertyViewLabel();
	PrintNodePropertySignature();
	PrintNodePropertyClassName();
	PrintNodePropertyIsMethod();
	PrintNodePropertyIsVirtual();
	PrintNodePropertyIsStatic();
	PrintNodePropertyIsStaticLinkage();
	PrintNodePropertyIsInline();
	PrintNodePropertyDeclarationFile();
	PrintNodePropertyDeclarationPosition();
	PrintNodePropertyDefinitionFile();
	PrintNodePropertyDefinitionPosition();
	PrintNodePropertyAccessSpecifier();

	// Print edge specific properties.
	PrintEdgePropertyCallFile();
	PrintEdgePropertyCallPosition();
	PrintEdgePropertyCallIsVirtual();
	PrintEdgePropertyCallIsAccurate();
	PrintEdgePropertyCallCandidateSetId();
	PrintEdgePropertyCallViaPointer();
	PrintEdgePropertyCallOnObject();

	// Print graph footer.
	m_os << ")" << std::endl;
}


// ----------------------------------------------------------------------------
// Printing nodes and edges
// ----------------------------------------------------------------------------
void CCC_CallGraphPrinterTlp::PrintNodes()
{
	// Print graph nodes.
	m_os << "  (nodes";

	std::vector<GraphNode*>::iterator itGraphNode = m_pCallGraph->m_nodes.begin();
	while(itGraphNode != m_pCallGraph->m_nodes.end())
	{
		GraphNode* pGraphNode = *itGraphNode;
		m_os << " " << pGraphNode->GetId();
		itGraphNode++;
	}

	m_os << ")" << std::endl;
}

void CCC_CallGraphPrinterTlp::PrintEdges()
{
	// Print graph edges.
	std::vector<GraphEdge*>::iterator itGraphEdge = m_pCallGraph->m_edges.begin();
	while(itGraphEdge != m_pCallGraph->m_edges.end())
	{
		GraphEdge* pGraphEdge = *itGraphEdge;

		m_os
		<< "  (edge " << pGraphEdge->GetId()         << " "
		<< pGraphEdge->GetSourceNode()->GetId()      << " "
		<< pGraphEdge->GetDestinationNode()->GetId() << ")"
		<< std::endl;

		itGraphEdge++;
	}
}


// ----------------------------------------------------------------------------
// Printing generic properties
// ----------------------------------------------------------------------------
void CCC_CallGraphPrinterTlp::PrintPropertyType()
{
	// Print 'viewLabel' property.
	m_os << "  (property 0 string \"type\"" << std::endl;
	m_os << "    (default \"Function\" \"Calls\")" << std::endl;

	// Print the type for the nodes.
	std::vector<GraphNode*>::iterator itGraphNode = m_pCallGraph->m_nodes.begin();
	while(itGraphNode != m_pCallGraph->m_nodes.end())
	{
		GraphNode* pGraphNode = *itGraphNode;
		switch(pGraphNode->GetType())
		{
			case GNT_Directory:
				m_os << "    (node " << pGraphNode->GetId() << " " << "\"Directory\")" << std::endl;
				break;
			case GNT_File:
				m_os << "    (node " << pGraphNode->GetId() << " " << "\"File\")" << std::endl;
				break;
			case GNT_Class:
				m_os << "    (node " << pGraphNode->GetId() << " " << "\"Class\")" << std::endl;
				break;
		}
		itGraphNode++;
	}

	// Print the type for the edges.
	std::vector<GraphEdge*>::iterator itGraphEdge = m_pCallGraph->m_edges.begin();
	while(itGraphEdge != m_pCallGraph->m_edges.end())
	{
		GraphEdge* pGraphEdge = *itGraphEdge;
		switch(pGraphEdge->GetType())
		{
			case GET_Contains:
				m_os << "    (edge " << pGraphEdge->GetId() << " " << "\"Contains\")" << std::endl;
				break;
		}

		itGraphEdge++;
	}

	m_os << "  )" << std::endl;
}


// ----------------------------------------------------------------------------
// Printing node properties
// ----------------------------------------------------------------------------
void CCC_CallGraphPrinterTlp::PrintNodePropertyViewLabel()
{
	// Print 'viewLabel' property.
	m_os << "  (property 0 string \"viewLabel\"" << std::endl;
	m_os << "    (default \"\" \"\")" << std::endl;

	std::vector<GraphNode*>::iterator itGraphNode = m_pCallGraph->m_nodes.begin();
	while(itGraphNode != m_pCallGraph->m_nodes.end())
	{
		GraphNode* pGraphNode = *itGraphNode;
		m_os << "    (node " << pGraphNode->GetId() << " " << "\"" << pGraphNode->GetName() << "\")" << std::endl;
		itGraphNode++;
	}

	m_os << "  )" << std::endl;
}

void CCC_CallGraphPrinterTlp::PrintNodePropertySignature()
{
	// Print 'viewLabel' property.
	m_os << "  (property 0 string \"signature\"" << std::endl;
	m_os << "    (default \"\" \"\")" << std::endl;

	std::vector<GraphNode*>::iterator itGraphNode = m_pCallGraph->m_nodes.begin();
	while(itGraphNode != m_pCallGraph->m_nodes.end())
	{
		if((*itGraphNode)->GetType() == GNT_Function)
		{
			FunctionGraphNode* pGraphNode = (FunctionGraphNode*)*itGraphNode;
			m_os << "    (node " << pGraphNode->GetId() << " " << "\"" << pGraphNode->GetSignature() << "\")" << std::endl;
		}
		itGraphNode++;
	}

	m_os << "  )" << std::endl;
}

void CCC_CallGraphPrinterTlp::PrintNodePropertyClassName()
{
	// Print 'viewLabel' property.
	m_os << "  (property 0 string \"class\"" << std::endl;
	m_os << "    (default \"\" \"\")" << std::endl;

	std::vector<GraphNode*>::iterator itGraphNode = m_pCallGraph->m_nodes.begin();
	while(itGraphNode != m_pCallGraph->m_nodes.end())
	{
		if((*itGraphNode)->GetType() == GNT_Function)
		{
			FunctionGraphNode* pGraphNode = (FunctionGraphNode*)*itGraphNode;
			if(pGraphNode->HasClassName())
			{
				m_os << "    (node " << pGraphNode->GetId() << " " << "\"" << pGraphNode->GetClassName() << "\")" << std::endl;
			}
		}
		itGraphNode++;
	}

	m_os << "  )" << std::endl;
}

void CCC_CallGraphPrinterTlp::PrintNodePropertyIsMethod()
{
	// Print 'isMethod' property.
	m_os << "  (property 0 bool \"is_method\"" << std::endl;
	m_os << "    (default \"false\" \"false\")" << std::endl;

	std::vector<GraphNode*>::iterator itGraphNode = m_pCallGraph->m_nodes.begin();
	while(itGraphNode != m_pCallGraph->m_nodes.end())
	{
		if((*itGraphNode)->GetType() == GNT_Function)
		{
			FunctionGraphNode* pGraphNode = (FunctionGraphNode*)*itGraphNode;
			if(pGraphNode->IsMethod())
			{
				m_os << "    (node " << pGraphNode->GetId() << " " << "\"true\")" << std::endl;
			}
		}
		itGraphNode++;
	}

	m_os << "  )" << std::endl;
}

void CCC_CallGraphPrinterTlp::PrintNodePropertyIsVirtual()
{
	// Print the 'isVirtual' property.
	m_os << "  (property 0 bool \"is_virtual\"" << std::endl;
	m_os << "    (default \"false\" \"false\")" << std::endl;

	std::vector<GraphNode*>::iterator itGraphNode = m_pCallGraph->m_nodes.begin();
	while(itGraphNode != m_pCallGraph->m_nodes.end())
	{
		if((*itGraphNode)->GetType() == GNT_Function)
		{
			FunctionGraphNode* pGraphNode = (FunctionGraphNode*)*itGraphNode;
			if(pGraphNode->IsVirtual())
			{
				m_os << "    (node " << pGraphNode->GetId() << " " << "\"true\")" << std::endl;
			}
		}
		itGraphNode++;
	}

	m_os << "  )" << std::endl;
}

void CCC_CallGraphPrinterTlp::PrintNodePropertyIsStatic()
{
	// Print the 'isVirtual' property.
	m_os << "  (property 0 bool \"is_static\"" << std::endl;
	m_os << "    (default \"false\" \"false\")" << std::endl;

	std::vector<GraphNode*>::iterator itGraphNode = m_pCallGraph->m_nodes.begin();
	while(itGraphNode != m_pCallGraph->m_nodes.end())
	{
		if((*itGraphNode)->GetType() == GNT_Function)
		{
			FunctionGraphNode* pGraphNode = (FunctionGraphNode*)*itGraphNode;
			if(pGraphNode->IsStatic())
			{
				m_os << "    (node " << pGraphNode->GetId() << " " << "\"true\")" << std::endl;
			}
		}
		itGraphNode++;
	}

	m_os << "  )" << std::endl;
}

void CCC_CallGraphPrinterTlp::PrintNodePropertyIsStaticLinkage()
{
	// Print the 'isStaticLinkage' property.
	m_os << "  (property 0 bool \"is_static_linkage\"" << std::endl;
	m_os << "    (default \"false\" \"false\")" << std::endl;

	std::vector<GraphNode*>::iterator itGraphNode = m_pCallGraph->m_nodes.begin();
	while(itGraphNode != m_pCallGraph->m_nodes.end())
	{
		if((*itGraphNode)->GetType() == GNT_Function)
		{
			FunctionGraphNode* pGraphNode = (FunctionGraphNode*)*itGraphNode;
			if(pGraphNode->IsStaticLinkage())
			{
				m_os << "    (node " << pGraphNode->GetId() << " " << "\"true\")" << std::endl;
			}
		}
		itGraphNode++;
	}

	m_os << "  )" << std::endl;
}

void CCC_CallGraphPrinterTlp::PrintNodePropertyIsInline()
{
	// Print the 'isInline' property.
	m_os << "  (property 0 bool \"is_inline\"" << std::endl;
	m_os << "    (default \"false\" \"false\")" << std::endl;

	std::vector<GraphNode*>::iterator itGraphNode = m_pCallGraph->m_nodes.begin();
	while(itGraphNode != m_pCallGraph->m_nodes.end())
	{
		if((*itGraphNode)->GetType() == GNT_Function)
		{
			FunctionGraphNode* pGraphNode = (FunctionGraphNode*)*itGraphNode;
			if(pGraphNode->IsInline())
			{
				m_os << "    (node " << pGraphNode->GetId() << " " << "\"true\")" << std::endl;
			}
		}
		itGraphNode++;
	}

	m_os << "  )" << std::endl;
}

void CCC_CallGraphPrinterTlp::PrintNodePropertyDeclarationFile()
{
	// Print the 'declarationTranslationUnit' property.
	m_os << "  (property 0 string \"declaration_file\"" << std::endl;
	m_os << "    (default \"\" \"\")" << std::endl;

	std::vector<GraphNode*>::iterator itGraphNode = m_pCallGraph->m_nodes.begin();
	while(itGraphNode != m_pCallGraph->m_nodes.end())
	{
		if((*itGraphNode)->GetType() == GNT_Function)
		{
			FunctionGraphNode* pGraphNode = (FunctionGraphNode*)*itGraphNode;
			m_os << "    (node " << pGraphNode->GetId() << " " << "\"" << pGraphNode->GetDeclarationFileName() << "\")" << std::endl;
		}
		itGraphNode++;
	}

	m_os << "  )" << std::endl;
}

void CCC_CallGraphPrinterTlp::PrintNodePropertyDeclarationPosition()
{
	// Print the 'declarationTranslationUnit' property.
	m_os << "  (property 0 string \"declaration_position\"" << std::endl;
	m_os << "    (default \"\" \"\")" << std::endl;

	std::vector<GraphNode*>::iterator itGraphNode = m_pCallGraph->m_nodes.begin();
	while(itGraphNode != m_pCallGraph->m_nodes.end())
	{
		if((*itGraphNode)->GetType() == GNT_Function)
		{
			FunctionGraphNode* pGraphNode = (FunctionGraphNode*)*itGraphNode;
			m_os << "    (node " << pGraphNode->GetId() << " " << "\"" << pGraphNode->GetDeclarationPosition() << "\")" << std::endl;
		}
		itGraphNode++;
	}

	m_os << "  )" << std::endl;
}

void CCC_CallGraphPrinterTlp::PrintNodePropertyDefinitionFile()
{
	// Print the 'definitionTranslationUnit' property.
	m_os << "  (property 0 string \"definition_file\"" << std::endl;
	m_os << "    (default \"\" \"\")" << std::endl;

	std::vector<GraphNode*>::iterator itGraphNode = m_pCallGraph->m_nodes.begin();
	while(itGraphNode != m_pCallGraph->m_nodes.end())
	{
		if((*itGraphNode)->GetType() == GNT_Function)
		{
			FunctionGraphNode* pGraphNode = (FunctionGraphNode*)*itGraphNode;
			if(pGraphNode->HasDefinitionFileName())
			{
				m_os << "    (node " << pGraphNode->GetId() << " " << "\"" << pGraphNode->GetDefinitionFileName() << "\")" << std::endl;
			}
		}
		itGraphNode++;
	}

	m_os << "  )" << std::endl;
}

void CCC_CallGraphPrinterTlp::PrintNodePropertyDefinitionPosition()
{
	// Print the 'declarationTranslationUnit' property.
	m_os << "  (property 0 string \"definition_position\"" << std::endl;
	m_os << "    (default \"\" \"\")" << std::endl;

	std::vector<GraphNode*>::iterator itGraphNode = m_pCallGraph->m_nodes.begin();
	while(itGraphNode != m_pCallGraph->m_nodes.end())
	{
		if((*itGraphNode)->GetType() == GNT_Function)
		{
			FunctionGraphNode* pGraphNode = (FunctionGraphNode*)*itGraphNode;
			m_os << "    (node " << pGraphNode->GetId() << " " << "\"" << pGraphNode->GetDefinitionPosition() << "\")" << std::endl;
		}
		itGraphNode++;
	}

	m_os << "  )" << std::endl;
}

void CCC_CallGraphPrinterTlp::PrintNodePropertyAccessSpecifier()
{
	// Print the 'definitionTranslationUnit' property.
	m_os << "  (property 0 string \"access_specifier\"" << std::endl;
	m_os << "    (default \"\" \"\")" << std::endl;

	std::vector<GraphNode*>::iterator itGraphNode = m_pCallGraph->m_nodes.begin();
	while(itGraphNode != m_pCallGraph->m_nodes.end())
	{
		if((*itGraphNode)->GetType() == GNT_Function)
		{
			FunctionGraphNode* pGraphNode = (FunctionGraphNode*)*itGraphNode;
			switch(pGraphNode->GetAccessSpecifier())
			{
				case AS_Public:
					m_os << "    (node " << pGraphNode->GetId() << " " << "\"public\")" << std::endl;
					break;
				case AS_Protected:
					m_os << "    (node " << pGraphNode->GetId() << " " << "\"protected\")" << std::endl;
					break;
				case AS_Private:
					m_os << "    (node " << pGraphNode->GetId() << " " << "\"private\")" << std::endl;
					break;
			}
		}
		itGraphNode++;
	}

	m_os << "  )" << std::endl;
}


// ----------------------------------------------------------------------------
// Printing edge properties
// ----------------------------------------------------------------------------
void CCC_CallGraphPrinterTlp::PrintEdgePropertyCallFile()
{
	m_os << "  (property 0 string \"call_file\"" << std::endl;
	m_os << "    (default \"\" \"\")" << std::endl;

	std::vector<GraphEdge*>::iterator itGraphEdge = m_pCallGraph->m_edges.begin();
	while(itGraphEdge != m_pCallGraph->m_edges.end())
	{
		if((*itGraphEdge)->GetType() == GET_Calls)
		{
			CallsGraphEdge* pGraphEdge = (CallsGraphEdge*)*itGraphEdge;
			m_os << "    (edge " << pGraphEdge->GetId() << " " << "\"" << pGraphEdge->GetCallFile() << "\")" << std::endl;
		}
		itGraphEdge++;
	}

	m_os << "  )" << std::endl;
}

void CCC_CallGraphPrinterTlp::PrintEdgePropertyCallPosition()
{
	m_os << "  (property 0 string \"call_position\"" << std::endl;
	m_os << "    (default \"\" \"\")" << std::endl;

	std::vector<GraphEdge*>::iterator itGraphEdge = m_pCallGraph->m_edges.begin();
	while(itGraphEdge != m_pCallGraph->m_edges.end())
	{
		if((*itGraphEdge)->GetType() == GET_Calls)
		{
			CallsGraphEdge* pGraphEdge = (CallsGraphEdge*)*itGraphEdge;
			m_os << "    (edge " << pGraphEdge->GetId() << " " << "\"" << pGraphEdge->GetCallPosition() << "\")" << std::endl;
		}
		itGraphEdge++;
	}

	m_os << "  )" << std::endl;
}

void CCC_CallGraphPrinterTlp::PrintEdgePropertyCallIsVirtual()
{
	m_os << "  (property 0 bool \"call_is_virtual\"" << std::endl;
	m_os << "    (default \"false\" \"false\")" << std::endl;

	std::vector<GraphEdge*>::iterator itGraphEdge = m_pCallGraph->m_edges.begin();
	while(itGraphEdge != m_pCallGraph->m_edges.end())
	{
		if((*itGraphEdge)->GetType() == GET_Calls)
		{
			CallsGraphEdge* pGraphEdge = (CallsGraphEdge*)*itGraphEdge;
			if(pGraphEdge->IsCallVirtual())
			{
				m_os << "    (edge " << pGraphEdge->GetId() << " " << "\"true\")" << std::endl;
			}
		}
		itGraphEdge++;
	}

	m_os << "  )" << std::endl;
}

void CCC_CallGraphPrinterTlp::PrintEdgePropertyCallIsAccurate()
{
	m_os << "  (property 0 bool \"call_is_accurate\"" << std::endl;
	m_os << "    (default \"false\" \"false\")" << std::endl;

	std::vector<GraphEdge*>::iterator itGraphEdge = m_pCallGraph->m_edges.begin();
	while(itGraphEdge != m_pCallGraph->m_edges.end())
	{
		if((*itGraphEdge)->GetType() == GET_Calls)
		{
			CallsGraphEdge* pGraphEdge = (CallsGraphEdge*)*itGraphEdge;
			if(pGraphEdge->IsCallAccurate())
			{
				m_os << "    (edge " << pGraphEdge->GetId() << " " << "\"true\")" << std::endl;
			}
		}
		itGraphEdge++;
	}

	m_os << "  )" << std::endl;
}

void CCC_CallGraphPrinterTlp::PrintEdgePropertyCallCandidateSetId()
{
	m_os << "  (property 0 int \"call_candidate_set_id\"" << std::endl;
	m_os << "    (default \"0\" \"0\")" << std::endl;

	std::vector<GraphEdge*>::iterator itGraphEdge = m_pCallGraph->m_edges.begin();
	while(itGraphEdge != m_pCallGraph->m_edges.end())
	{
		if((*itGraphEdge)->GetType() == GET_Calls)
		{
			CallsGraphEdge* pGraphEdge = (CallsGraphEdge*)*itGraphEdge;
			m_os << "    (edge " << pGraphEdge->GetId() << " " << "\"" << pGraphEdge->GetCallCandidateSetId() << "\")" << std::endl;
		}
		itGraphEdge++;
	}

	m_os << "  )" << std::endl;
}

void CCC_CallGraphPrinterTlp::PrintEdgePropertyCallViaPointer()
{
	m_os << "  (property 0 string \"call_via_pointer\"" << std::endl;
	m_os << "    (default \"\" \"\")" << std::endl;

	std::vector<GraphEdge*>::iterator itGraphEdge = m_pCallGraph->m_edges.begin();
	while(itGraphEdge != m_pCallGraph->m_edges.end())
	{
		if((*itGraphEdge)->GetType() == GET_Calls)
		{
			CallsGraphEdge* pGraphEdge = (CallsGraphEdge*)*itGraphEdge;
			switch(pGraphEdge->GetCallViaPointer())
			{
				case CVP_No:
					m_os << "    (edge " << pGraphEdge->GetId() << " " << "\"no\")" << std::endl;
					break;
				case CVP_PointerToFunction:
					m_os << "    (edge " << pGraphEdge->GetId() << " " << "\"pointer-to-function\")" << std::endl;
					break;
				case CVP_PointerToMember:
					m_os << "    (edge " << pGraphEdge->GetId() << " " << "\"pointer-to-member\")" << std::endl;
					break;
			}
		}
		itGraphEdge++;
	}

	m_os << "  )" << std::endl;
}

void CCC_CallGraphPrinterTlp::PrintEdgePropertyCallOnObject()
{
	m_os << "  (property 0 string \"call_on_object\"" << std::endl;
	m_os << "    (default \"\" \"\")" << std::endl;

	std::vector<GraphEdge*>::iterator itGraphEdge = m_pCallGraph->m_edges.begin();
	while(itGraphEdge != m_pCallGraph->m_edges.end())
	{
		if((*itGraphEdge)->GetType() == GET_Calls)
		{
			CallsGraphEdge* pGraphEdge = (CallsGraphEdge*)*itGraphEdge;
			switch(pGraphEdge->GetCallOnObject())
			{
				case COO_No:
					m_os << "    (edge " << pGraphEdge->GetId() << " " << "\"no\")" << std::endl;
					break;
				case COO_ObjectInstance:
					m_os << "    (edge " << pGraphEdge->GetId() << " " << "\"object-instance\")" << std::endl;
					break;
				case COO_ObjectReference:
					m_os << "    (edge " << pGraphEdge->GetId() << " " << "\"object-reference\")" << std::endl;
					break;
				case COO_ObjectPointer:
					m_os << "    (edge " << pGraphEdge->GetId() << " " << "\"object-pointer\")" << std::endl;
					break;
			}
		}
		itGraphEdge++;
	}

	m_os << "  )" << std::endl;
}
