/***************************************************************************
 *   Copyright (C) 2009 by Hessel Hoogendorp                               *
 *   bugs.ccc@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


// ----------------------------------------------------------------------------
// Includes
// ----------------------------------------------------------------------------
#include "ccc_global_function_repository.h"


// ----------------------------------------------------------------------------
// Construction & destruction
// ----------------------------------------------------------------------------
void CCC_GlobalFunctionRepository::Finalize()
{
	// Delete all GlobalFunctionVectors. The GlobalFunctions they reference
	// will be deleted below.
	GlobalFunctionVectorMap::iterator itGlobalFunctionVector = m_pointerSignatureToGlobalFunctionVector.begin();
	while(itGlobalFunctionVector != m_pointerSignatureToGlobalFunctionVector.end())
	{
		delete itGlobalFunctionVector->second;
		itGlobalFunctionVector++;
	}
	m_pointerSignatureToGlobalFunctionVector.clear();

	// Delete all global functions.
	GlobalFunctionMap::iterator itGlobalFunction = m_signatureToGlobalFunction.begin();
	while(itGlobalFunction != m_signatureToGlobalFunction.end())
	{
		delete itGlobalFunction->second;
		itGlobalFunction++;
	}
	m_signatureToGlobalFunction.clear();
}


// ----------------------------------------------------------------------------
// Repository management
// ----------------------------------------------------------------------------
bool CCC_GlobalFunctionRepository::ContainsGlobalFunctionWithSignature(std::string signature)
{
	return m_signatureToGlobalFunction.find(signature) != m_signatureToGlobalFunction.end();
}

bool CCC_GlobalFunctionRepository::ContainsGlobalFunctionWithPointerSignature(std::string szPointerSignature)
{
	return m_pointerSignatureToGlobalFunctionVector.find(szPointerSignature) != m_pointerSignatureToGlobalFunctionVector.end();
}

CCC_GlobalFunction* CCC_GlobalFunctionRepository::GetGlobalFunction(std::string signature)
{
	if(!ContainsGlobalFunctionWithSignature(signature)) return NULL;
	return m_signatureToGlobalFunction[signature];
}

GlobalFunctionMap* CCC_GlobalFunctionRepository::GetGlobalFunctions()
{
	return &m_signatureToGlobalFunction;
}

GlobalFunctionVector* CCC_GlobalFunctionRepository::GetGlobalFunctionVector(std::string szPointerSignature)
{
	if(!ContainsGlobalFunctionWithPointerSignature(szPointerSignature)) return NULL;
	return m_pointerSignatureToGlobalFunctionVector[szPointerSignature];
}

GlobalFunctionVectorMap* CCC_GlobalFunctionRepository::GetGlobalFunctionVectors()
{
	return &m_pointerSignatureToGlobalFunctionVector;
}

void CCC_GlobalFunctionRepository::AddLocalFunction(CCI_LocalFunction* ptuFunction)
{
	cci_assert(ptuFunction != NULL);

	// Retrieve the signature of the supplied method and create a new
	// GlobalFunction if that does not exist yet.
	std::string signature = ptuFunction->GetSignature();
	if(!ContainsGlobalFunctionWithSignature(signature))
	{
		// Create a new GlobalFunction.
		CCC_GlobalFunction* pGlobalFunction = new CCC_GlobalFunction();
		m_signatureToGlobalFunction[signature] = pGlobalFunction;

		// Add the new GlobalFunction to the m_pointerSignatureToGlobalFunctionVector
		// map, associating the function's call signature with the newly created
		// GlobalFunction.
		std::string szPointerSignature = ptuFunction->GetPointerSignature();
		if(!ContainsGlobalFunctionWithPointerSignature(szPointerSignature))
			m_pointerSignatureToGlobalFunctionVector[szPointerSignature] = new GlobalFunctionVector();

		m_pointerSignatureToGlobalFunctionVector[szPointerSignature]->push_back(pGlobalFunction);
	}

	// Add this CCI_LocalFunction to the GlobalFunction.
	CCC_GlobalFunction* pGlobFunc = m_signatureToGlobalFunction[signature];

	pGlobFunc->AddLocalFunction(ptuFunction);
}

void CCC_GlobalFunctionRepository::AddLocalFunctionMap(LocalFunctionMap* pLocalFunctions)
{
	LocalFunctionMap::iterator itLocalFunction = pLocalFunctions->begin();
	while(itLocalFunction != pLocalFunctions->end())
	{
		AddLocalFunction(itLocalFunction->second);
		itLocalFunction++;
	}
}


// ----------------------------------------------------------------------------
// Debugging
// ----------------------------------------------------------------------------
void CCC_GlobalFunctionRepository::Print(std::string prefix)
{
	PrintSignatureMapping(prefix);
	PrintCallSignatureMapping(prefix);
}

void CCC_GlobalFunctionRepository::PrintSignatureMapping(std::string prefix)
{
	std::cout << std::endl << prefix << "Signature to GlobalFunction mapping";
	std::cout << std::endl << prefix << "{";

	// Print the m_signatureToGlobalFunction mapping.
	GlobalFunctionMap::iterator itGlobalFunction = m_signatureToGlobalFunction.begin();
	while(itGlobalFunction != m_signatureToGlobalFunction.end())
	{
		if(itGlobalFunction != m_signatureToGlobalFunction.begin())
			std::cout << std::endl;

		itGlobalFunction->second->Print(prefix + "\t");
		itGlobalFunction++;
	}

	std::cout << std::endl << prefix << "}";
}

void CCC_GlobalFunctionRepository::PrintCallSignatureMapping(std::string prefix)
{
	// Print the m_pointerSignatureToGlobalFunctionVector mapping.
	std::cout << std::endl << prefix << "Call signature to GlobalFunction mapping";
	std::cout << std::endl << prefix << "{";

	GlobalFunctionVectorMap::iterator itGlobalFunctionVector = m_pointerSignatureToGlobalFunctionVector.begin();
	while(itGlobalFunctionVector != m_pointerSignatureToGlobalFunctionVector.end())
	{
		if(itGlobalFunctionVector != m_pointerSignatureToGlobalFunctionVector.begin())
			std::cout << std::endl;

		std::cout << std::endl << prefix << "\t" << itGlobalFunctionVector->first;
		std::cout << std::endl << prefix << "\t{";

		GlobalFunctionVector* pGlobalFunctionVector = itGlobalFunctionVector->second;
		GlobalFunctionVector::iterator itGlobalFunction = pGlobalFunctionVector->begin();
		while(itGlobalFunction != pGlobalFunctionVector->end())
		{
			std::cout << std::endl << prefix << "\t\tGlobal function: " << (*itGlobalFunction)->GetSignature();
			itGlobalFunction++;
		}

		std::cout << std::endl << prefix << "\t}";
		itGlobalFunctionVector++;
	}

	std::cout << std::endl << prefix << "}";
}
