/***************************************************************************
 *   Copyright (C) 2009 by Hessel Hoogendorp                               *
 *   bugs.ccc@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


// ----------------------------------------------------------------------------
// Includes
// ----------------------------------------------------------------------------
#include "ccc_partial_call_graph.h"
#include "ccc_global_function_repository.h"
#include "ccc_settings.h"
#include "ccc_verbose_print.h"


// ----------------------------------------------------------------------------
// Construction
// ----------------------------------------------------------------------------
CCC_PartialCallGraph::CCC_PartialCallGraph(std::string szEntryPoint)
{
	m_szEntryPoint = szEntryPoint;

}


// ----------------------------------------------------------------------------
// Graph calculation
// ----------------------------------------------------------------------------
CCC_GlobalFunction* CCC_PartialCallGraph::GetMainEntryPoint()
{
	// Find and return the global function corresponding to the requested entry point.
	CCC_GlobalFunction* pGlobalFunction = NULL;

	if(m_szEntryPoint == "main")
	{
		if(!FindMainFunction(m_szEntryPoint, pGlobalFunction))
		{
			cci_assert_msg_allow
			(
				false,
				std::endl <<
				"\t{"                                                                                     << std::endl <<
				"\t\tProblem  : A 'main' function could not be found in the global function repository."  << std::endl <<
				"\t\tCause    : No 'main' function was found during extraction. Maybe this is a library?" << std::endl <<
				"\t\tSignature: main (shorthand specified on the command line)"                           << std::endl <<
				"\t}"
			);
		}
	}
	else
	{
		pGlobalFunction = m_pGlobalFunctionRepository->GetGlobalFunction(m_szEntryPoint);
		if(pGlobalFunction == NULL)
		{
			cci_assert_msg_allow
			(
				false,
				std::endl <<
				"\t{"                                                                                                     << std::endl <<
				"\t\tProblem  : Specified call graph entry point not found in the global function repository."            << std::endl <<
				"\t\tCause    : The entry point was not found during extraction. Did you specify an invalid entry point?" << std::endl <<
				"\t\tSignature: " << m_szEntryPoint                                                                       << std::endl <<
				"\t}"
			);
		}
	}

	return pGlobalFunction;
}

void CCC_PartialCallGraph::InsertNodes()
{
	// Retrieve the global function that represents the entry point.
	if(m_pMainEntryPoint == NULL)
		return;

	VERBOSE_PRINT("Using '" << m_pMainEntryPoint->GetSignature() << "' as the entry point of the call-graph.");

	InsertNode(m_pMainEntryPoint);
	
	if(CCC_ConstructorSettings::GetIncludeInitFinit())
	{
		InsertNodesForInitializingFunctionCalls();
		InsertNodesForFinalizingFunctionCalls();
	}
}

void CCC_PartialCallGraph::InsertNode(CCC_GlobalFunction* pGlobalFunction)
{
	// Retrieve all preferred local functions from the entry point.
	LocalFunctionVector localFunctions;
	GetPreferredLocalFunctions(pGlobalFunction, localFunctions);
	
	// Iterate over all the local functions.
	LocalFunctionVector::iterator itLocalFunction = localFunctions.begin();
	while(itLocalFunction != localFunctions.end())
	{
		CCI_LocalFunction* pLocalFunction = *itLocalFunction;
		InsertNode(pLocalFunction);
		itLocalFunction++;
	}
}

void CCC_PartialCallGraph::InsertNode(CCI_LocalFunction* pLocalFunction)
{
	// If this local function already has a node, then it has already been processed.
	if(m_nodeFromFunction.find(pLocalFunction) != m_nodeFromFunction.end())
		return;

	// Create and insert a Function node for this local function.
	InsertFunctionGraphNode(pLocalFunction);

	// Recursively insert nodes for all the functions that this function calls.
	if(pLocalFunction->HasFunctionDefinition())
	{
		FunctionCallVector* pFunctionCalls = pLocalFunction->GetFunctionDefinition()->GetFunctionCalls();
		FunctionCallVector::iterator itFunctionCall = pFunctionCalls->begin();
		while(itFunctionCall != pFunctionCalls->end())
		{
			GlobalFunctionVector callCandidates;
			GetCallCandidates(*itFunctionCall, callCandidates);
			GlobalFunctionVector::iterator itCallCandidate = callCandidates.begin();
			while(itCallCandidate != callCandidates.end())
			{
				InsertNode(*itCallCandidate);
				itCallCandidate++;
			}
			itFunctionCall++;
		}
	}
}

void CCC_PartialCallGraph::InsertNodesForInitializingFunctionCalls()
{
	FunctionCallVector::iterator itFunctionCall = m_pInitializingCalls->begin();
	while(itFunctionCall != m_pInitializingCalls->end())
	{
		if(m_pGlobalFunctionRepository->GetGlobalFunction((*itFunctionCall)->GetSignature()) == NULL)
			std::cout << "Call not found" << std::endl;
		std::cout << "I:" << (*itFunctionCall)->ToString("");
		std::cout << std::endl;
		
		GlobalFunctionVector callCandidates;
		GetCallCandidates(*itFunctionCall, callCandidates);
		GlobalFunctionVector::iterator itCallCandidate = callCandidates.begin();
		while(itCallCandidate != callCandidates.end())
		{
			std::cout << "Initializer: " << (*itCallCandidate)->GetSignature() << std::endl;
			InsertNode(*itCallCandidate);
			itCallCandidate++;
		}
		itFunctionCall++;
	}
}

void CCC_PartialCallGraph::InsertNodesForFinalizingFunctionCalls()
{
	FunctionCallVector::iterator itFunctionCall = m_pFinalizingCalls->begin();
	while(itFunctionCall != m_pFinalizingCalls->end())
	{
		std::cout << "F:" << (*itFunctionCall)->ToString("") << std::endl;
		
		GlobalFunctionVector callCandidates;
		GetCallCandidates(*itFunctionCall, callCandidates);
		GlobalFunctionVector::iterator itCallCandidate = callCandidates.begin();
		while(itCallCandidate != callCandidates.end())
		{
			std::cout << "Finalizer: " << (*itCallCandidate)->GetSignature() << std::endl;
			InsertNode(*itCallCandidate);
			itCallCandidate++;
		}
		itFunctionCall++;
	}
}
