/***************************************************************************
 *   Copyright (C) 2009 by Hessel Hoogendorp                               *
 *   bugs.ccc@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


// ----------------------------------------------------------------------------
// Includes
// ----------------------------------------------------------------------------
#include "ccc_settings.h"
#include <iostream>
#include <string.h>


// ----------------------------------------------------------------------------
// Defines
// ----------------------------------------------------------------------------
#define PRINT_ERROR(msg)																	\
{																							\
	std::cerr << msg << std::endl;															\
	std::cerr << std::endl;																	\
	std::cerr << "Try '" << m_programName << " --help' for more information." << std::endl;	\
}


// ----------------------------------------------------------------------------
// Static member initialization
// ----------------------------------------------------------------------------
StringVector     CCC_ConstructorSettings::m_inputFileNames;
CCC_InputType    CCC_ConstructorSettings::m_inputType                   = CCC_IT_Binary;
std::string      CCC_ConstructorSettings::m_entryPointName              = "";
CCC_OutputFormat CCC_ConstructorSettings::m_outputFormat                = CCC_OF_Sqlite;
std::string      CCC_ConstructorSettings::m_szOutputFileName            = "";
bool             CCC_ConstructorSettings::m_bValidateInput              = false;
bool             CCC_ConstructorSettings::m_bIncludeInitFinit           = false;
bool             CCC_ConstructorSettings::m_bVerbose                    = false;
CCC_Action       CCC_ConstructorSettings::m_action                      = CCC_A_ConstructCallGraph;
std::string      CCC_ConstructorSettings::m_programName                 = "ccc";
bool             CCC_ConstructorSettings::m_bConservativenessWarningsListCandidates      = false;
bool             CCC_ConstructorSettings::m_bConservativenessDisableWarningsOneCandidate = false;
bool             CCC_ConstructorSettings::m_bConservativenessDisableWarnings             = false;


// ----------------------------------------------------------------------------
// Parsing
// ----------------------------------------------------------------------------
bool CCC_ConstructorSettings::Parse(int argc, char** argv)
{
	m_inputFileNames.clear();

	int iCurrentArg = 0;

	// Parse the program name.
	char* pszProgramName = argv[iCurrentArg++];
	if(StartsWith(pszProgramName, "./"))
		pszProgramName += 2;
	m_programName = pszProgramName;

	// If there are no arguments, print the usage description.
	if(argc == 1)
	{
		m_action = CCC_A_PrintUsage;
		return true;
	}

	// Parse the program arguments.
	while(iCurrentArg < argc)
	{
		char* pSwitch   = argv[iCurrentArg++];
		char* pArgument = iCurrentArg < argc ? argv[iCurrentArg] : NULL;

		if(IsSwitch(pSwitch, "-t", "--input-type"))
		{
			iCurrentArg++;
			if(!CheckArgument(pSwitch, pArgument, "'binary' or 'xml'"))
				return false;

			if(strcmp(pArgument, "xml") == 0)
			{
				m_inputType = CCC_IT_Xml;
			}
			else if(strcmp(pArgument, "binary") == 0)
			{
				m_inputType = CCC_IT_Binary;
			}
			else
			{
				PRINT_ERROR("Invalid argument to switch '" << pSwitch << "': " << pArgument << ". Expected: 'binary' or 'xml'.");
				return false;
			}
		}
		else if(IsSwitch(pSwitch, "-e", "--entry-point"))
		{
			iCurrentArg++;
			if(!CheckArgument(pSwitch, pArgument, "<program-entry-point>"))
				return false;

			m_entryPointName = pArgument;
		}
		else if(IsSwitch(pSwitch, "-f", "--output-format"))
		{
			iCurrentArg++;
			if(!CheckArgument(pSwitch, pArgument, "<'dot', 'tulip' or 'sqlite'>"))
				return false;

			if(strcmp(pArgument, "dot") == 0)
			{
				m_outputFormat = CCC_OF_Dot;
			}
			else if(strcmp(pArgument, "tulip") == 0)
			{
				m_outputFormat = CCC_OF_Tulip;
			}
			else if(strcmp(pArgument, "sqlite") == 0)
			{
				m_outputFormat = CCC_OF_Sqlite;
			}
			else
			{
				PRINT_ERROR("Invalid argument to switch '" << pSwitch << "': " << pArgument << ". Expected: 'dot', 'tulip' or 'sqlite'.");
				return false;
			}
		}
		else if(IsSwitch(pSwitch, "-o", "--output-file"))
		{
			iCurrentArg++;
			if(!CheckArgument(pSwitch, pArgument, "<output-file-name>"))
				return false;

			m_szOutputFileName = pArgument;
		}
		else if(IsSwitch(pSwitch, "-d", "--validate-input"))
		{
			m_bValidateInput = true;
		}
		else if(IsSwitch(pSwitch, "-Wl", "--Wcnsv-list-cands"))
		{
			m_bConservativenessWarningsListCandidates = true;
		}
		else if(IsSwitch(pSwitch, "-Wo", "--Wcnsv-disable-one"))
		{
			m_bConservativenessDisableWarningsOneCandidate = true;
		}
		else if(IsSwitch(pSwitch, "-Wd", "--Wcnsv-disable"))
		{
			m_bConservativenessDisableWarnings = true;
		}
		else if(IsSwitch(pSwitch, "-i", "--include-init-finit"))
		{
			m_bIncludeInitFinit = true;
		}
		else if(IsSwitch(pSwitch, "-p", "--print-signatures"))
		{
			m_action = CCC_A_PrintSignatures;
		}
		else if(IsSwitch(pSwitch, "-b", "--verbose"))
		{
			m_bVerbose = true;
		}
		else if(IsSwitch(pSwitch, "-v", "--version"))
		{
			m_action = CCC_A_PrintVersion;
			return true;
		}
		else if(IsSwitch(pSwitch, "-h", "--help"))
		{
			m_action = CCC_A_PrintUsage;
			return true;
		}
		else if(StartsWith(pSwitch, "-"))
		{
			PRINT_ERROR("Unknown switch: " << pSwitch);
			return false;
		}
		else // This is an input filename
		{
			m_inputFileNames.push_back(pSwitch);
		}
	}

	return true;
}

bool CCC_ConstructorSettings::IsSwitch(char* pSwitch, const char* pszShortSwitch, const char* pszLongSwitch)
{
	return strcmp(pSwitch, pszShortSwitch) == 0 || strcmp(pSwitch, pszLongSwitch) == 0;
}

bool CCC_ConstructorSettings::StartsWith(char* pszSwitch, const char* pszStartsWith)
{
	return strstr(pszSwitch, pszStartsWith) == pszSwitch;
}

bool CCC_ConstructorSettings::CheckArgument(char* pszSwitch, char* pszArgument, const char* pszExpected)
{
	if(pszArgument == NULL)
	{
		PRINT_ERROR("Too few arguments to switch '" << pszSwitch << "'. Expected: " << pszExpected << ".");
		return false;
	}

	return true;
}


// ----------------------------------------------------------------------------
// Settings
// ----------------------------------------------------------------------------
StringVector* CCC_ConstructorSettings::GetInputFileNames()
{
	return &m_inputFileNames;
}

void CCC_ConstructorSettings::SetInputType(CCC_InputType inputType)
{
	m_inputType = inputType;
}

CCC_InputType CCC_ConstructorSettings::GetInputType()
{
	return m_inputType;
}

void CCC_ConstructorSettings::SetEntryPointName(std::string entryPointName)
{
	m_entryPointName = entryPointName;
}

std::string CCC_ConstructorSettings::GetEntryPointName()
{
	return m_entryPointName;
}

void CCC_ConstructorSettings::SetOutputFormat(CCC_OutputFormat outputFormat)
{
	m_outputFormat = outputFormat;
}

CCC_OutputFormat CCC_ConstructorSettings::GetOutputFormat()
{
	return m_outputFormat;
}

void CCC_ConstructorSettings::SetOutputFileName(std::string szOutputFileName)
{
	m_szOutputFileName = szOutputFileName;
}

std::string CCC_ConstructorSettings::GetOutputFileName()
{
	return m_szOutputFileName;
}

void CCC_ConstructorSettings::SetValidateInput(bool bValidateInput)
{
	m_bValidateInput = bValidateInput;
}

bool CCC_ConstructorSettings::GetValidateInput()
{
	return m_bValidateInput;
}

void CCC_ConstructorSettings::SetConservativenessWarningsListCandidates(bool bConservativenessWarningsListCandidates)
{
	m_bConservativenessWarningsListCandidates = bConservativenessWarningsListCandidates;
}

bool CCC_ConstructorSettings::GetConservativenessWarningsListCandidates()
{
	return m_bConservativenessWarningsListCandidates;
}

void CCC_ConstructorSettings::SetConservativenessDisableWarningsOneCandidate(bool bConservativenessDisableWarningsOneCandidate)
{
	m_bConservativenessDisableWarningsOneCandidate = bConservativenessDisableWarningsOneCandidate;
}

bool CCC_ConstructorSettings::GetConservativenessDisableWarningsOneCandidate()
{
	return m_bConservativenessDisableWarningsOneCandidate;
}

void CCC_ConstructorSettings::SetConservativenessDisableWarnings(bool bConservativenessDisableWarnings)
{
	m_bConservativenessDisableWarnings = bConservativenessDisableWarnings;
}

bool CCC_ConstructorSettings::GetConservativenessDisableWarnings()
{
	return m_bConservativenessDisableWarnings;
}

void CCC_ConstructorSettings::SetIncludeInitFinit(bool bIncludeInitFinit)
{
	m_bIncludeInitFinit = bIncludeInitFinit;
}

bool CCC_ConstructorSettings::GetIncludeInitFinit()
{
	return m_bIncludeInitFinit;
}

void CCC_ConstructorSettings::SetVerbose(bool bVerbose)
{
	m_bVerbose = bVerbose;
}

bool CCC_ConstructorSettings::GetVerbose()
{
	return m_bVerbose;
}

void CCC_ConstructorSettings::SetAction(CCC_Action action)
{
	m_action = action;
}

CCC_Action CCC_ConstructorSettings::GetAction()
{
	return m_action;
}

std::string CCC_ConstructorSettings::GetProgramName()
{
	return m_programName;
}


// ----------------------------------------------------------------------------
// Debugging
// ----------------------------------------------------------------------------
void CCC_ConstructorSettings::Print(std::string prefix)
{
	StringVector::iterator itInputFileName = m_inputFileNames.begin();
	while(itInputFileName != m_inputFileNames.end())
	{
		std::cout << prefix << "--input  : " << *itInputFileName << std::endl;
		itInputFileName++;
	}
	std::cout << prefix << "--type   : " << (m_inputType == CCC_IT_Xml ? "xml" : "binary") << std::endl;
	std::cout << prefix << "--entry  : " << m_entryPointName << std::endl;
	std::cout << prefix << "--verbose: " << (m_bVerbose ? "yes" : "no") << std::endl;
	std::cout << prefix << "Action   : " << (m_action == CCC_A_ConstructCallGraph ? "ConstructCallGraph" : "PrintVersion") << std::endl;
}
